// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "hw_ei.h"
#include "stringCommons.h"

void HwEI::ww_sr(Hardware *h, WORD offset, DWORD data) {
	EIDEGUB("Hardware ww: 0x%04X, 0x%08X\n", offset, data);
	h->do_exi_select(EXI_CHANNEL_FROM_OFFSET(offset), data);
}

void HwEI::ww_cr(Hardware *h, WORD offset, DWORD data) {
	EIDEGUB("Hardware ww: 0x%04X, 0x%08X\n", offset, data);
	if(getbitr(data, 0)) {
		h->do_exi_rw(EXI_CHANNEL_FROM_OFFSET(offset), data);
	}
}

DWORD HwEI::rw_sr(Hardware *h, WORD offset) {
	DWORD status = h->hrw(offset);
	status |= getDCBit(h, EXI_CHANNEL_FROM_OFFSET(offset));
	if(EXI_CHANNEL_FROM_OFFSET(offset) == 0)
		status |= 0x2000; //ROMDIS
	EIDEGUB("Hardware rw: 0x%04X, 0x%08X\n", offset, status);
	return status;
}
WORD HwEI::rh_sr_low(Hardware *h, WORD offset) {
	WORD status = h->hrh(offset);
	if(EXI_CHANNEL_FROM_OFFSET(offset) == 0)
		status |= 0x2000; //ROMDIS
	status |= getDCBit(h, 0);
	EIDEGUB("Hardware rh: 0x%04X, 0x%04X\n", offset, status);
	return status;
}

// External Insertion Status (1: External EXI device present)
WORD HwEI::getDCBit(Hardware *h, int channel) {
	return (h->exi_devices[channel][0] != NULL) ? 0x1000 : 0;
}

int getEXISelection(DWORD status) {
	int device = -1;
	if(status & makeflag(22)) {
		//DEGUB("EXI %i:2 selected\n", channel);
		if(device != -1) {
			//DEGUB("Multiple EXI devices selected\n");
			//if(g::bouehr)
			throw bouehr_exception("Multiple EXI devices selected");
		}
		device = 2;
	}
	if(status & makeflag(23)) {
		//DEGUB("EXI %i:1 selected\n", channel);
		if(device != -1) {
			//DEGUB("Multiple EXI devices selected\n");
			//if(g::bouehr)
			throw bouehr_exception("Multiple EXI devices selected");
		}
		device = 1;
	}
	if(status & makeflag(24)) {
		//DEGUB("EXI %i:0 selected\n", channel);
		if(device != -1) {
			//DEGUB("Multiple EXI devices selected\n");
			//if(g::bouehr)
			throw bouehr_exception("Multiple EXI devices selected");
		}
		device = 0;
	}
	return device;
}

void Hardware::do_exi_rw(BYTE channel, DWORD controlreg) {
	DWORD status = hrw(EXI_STATUS(channel));

	int device = getEXISelection(status);
	if(device == -1) {
		//DEGUB("No EXI device selected\n");
		//if(g::bouehr)
		throw bouehr_exception("No EXI device selected");
	}

	int len = getbitsw(controlreg, 26, 27) + 1;
	EXITransferType ett = (EXITransferType)getbitsw(controlreg, 28, 29);
	EXIResultValue erv = EXI_UNHANDLED;

	EXI_DEGUB("EXI %i:%i ", channel, device);
	switch(ett) {
	case EXI_WRITE:
		EXI_DEGUB("write ");
		if(controlreg & EICR_DMA) {
			DWORD address = hrw(EXI_DMABUF(channel)), length = hrw(EXI_DMALEN(channel));
			EXI_DEGUB("DMA from [0x%08X], 0x%X bytes\n", address, length);
			if(!exi_devices[channel][device]) {
				DEGUB("No device installed!\n");
			} else if((length & (32-1)) != 0) {
				throw hardware_fatal_exception("EXI DMA length 32 byte unaligned");
			} else {
				erv = exi_devices[channel][device]->writeDMA(length, address, m);
			}
		} else {
			DWORD data = hrw(EXI_IMMDAT(channel));
			EXI_DEGUB("immediate %i bytes, 0x%08X\n", len, data);
			if(!exi_devices[channel][device]) {
				DEGUB("No device installed!\n");
			} else {
				erv = exi_devices[channel][device]->writeImm(len,
					getbitsw(data, 0, len * 8 - 1));
			}
		}
		break;
	case EXI_READ:
		EXI_DEGUB("read ");
		if(controlreg & EICR_DMA) {
			DWORD address = hrw(EXI_DMABUF(channel)), length = hrw(EXI_DMALEN(channel));
			EXI_DEGUB("DMA to [0x%08X], 0x%X bytes\n", address, length);
			if(!exi_devices[channel][device]) {
				EIDEGUB("No device installed!\n");
			} else if((length & (32-1)) != 0) {
				throw hardware_fatal_exception("EXI DMA length 32 byte unaligned");
			} else {
				erv = exi_devices[channel][device]->readDMA(length, address, m);
			}
		} else {
			DWORD data;
			EXI_DEGUB("immediate %i bytes, ", len);
			if(!exi_devices[channel][device]) {
				EXI_DEGUB("No device installed!\n");
			} else {
				erv = exi_devices[channel][device]->readImm(len, data);
				EXI_DEGUB("0x%0*X\n", 1 << getbitsw(controlreg, 26, 27), data);
				hww(EXI_IMMDAT(channel), data);
			}
		}
		break;
	default:
		DEGUB("Unknown EXI transfer type: %i\n", ett);
		throw bouehr_exception("Unknown EXI transfer type");
	}
	if(erv != EXI_UNHANDLED) {
		hww(EXI_CONTROL(channel), controlreg & ~EICR_TSTART);  //operation complete
		if(controlreg & EICR_DMA) {
			hww(EXI_STATUS(channel), status | EICSR_TCINT);
			if(status & EICSR_TCINTMSK)
				interrupt.raise(INTEX_EXI, "EXI Transfer Complete");
		} //else 
		//hww(EXI_STATUS(channel), status & ~EICSR_TCINT);	//hack?
	}
}

void EXIInterruptRaiser::raiseEXI(const std::string& desc) {
	DWORD status = h->hrw(EXI_STATUS(mChannel));
	if(!(status & EICSR_EXIINTMSK)) {
		if(h->exi.exi_interrupt_waiting[mChannel])
			return;	//interrupt already raised
		h->exi.exi_interrupt_waiting[mChannel] = true;
		if(g::verbose && g::log_interrupts) {
			DEGUB("EXI %i EXI interrupt \"%s\" waiting.\n", mChannel, desc.c_str());
		}
	} else {
		h->hww(EXI_STATUS(mChannel), status | EICSR_EXIINT);
		h->interrupt.raise(INTEX_EXI, desc);
	}
}

void EXIInterruptRaiser::raiseEXT(const std::string& desc) {
	DWORD status = h->hrw(EXI_STATUS(mChannel));
	if(!(status & EICSR_EXTINTMSK)) {
		h->exi.ext_interrupt_waiting[mChannel] = true;
		if(g::verbose && g::log_interrupts) {
			DEGUB("EXI %i EXT interrupt \"%s\" waiting.\n", mChannel, desc.c_str());
		}
	} else {
		h->hww(EXI_STATUS(mChannel), status | EICSR_EXTINT);
		h->interrupt.raise(INTEX_EXI, desc);
	}
}

void Hardware::do_exi_select(const BYTE channel, const DWORD statusreg) {
	bool endline = false;
	DWORD prev = hrw(EXI_STATUS(channel));

#define FALL(bit) (!(statusreg & makeflag(bit)) && (prev & makeflag(bit)))

	if(FALL(22)) {
		if(g::exi_log || channel != 0) {	//0:2 is BBA, we've dealt with it.
			DEGUB("EXI %i:2 deselected\n", channel);
			endline = true;
		}
		if(!exi_devices[channel][2]) {
			if(endline) {
				DEGUB("No device installed!\n");
			}
		} else {
			exi_devices[channel][2]->notify_deselected();
		}
	}
	if(FALL(23)) {
		if(g::exi_log || !(channel == 0 && !EXI_01_DEGUB_CONDITION)) {
			DEGUB("EXI %i:1 deselected\n", channel);
			endline = true;
		}
		if(!exi_devices[channel][1]) {
			if(endline) {
				DEGUB("No device installed!\n");
			}
		} else {
			exi_devices[channel][1]->notify_deselected();
		}
	}
	if(FALL(24)) {
		if(g::exi_log || g::memcard_log) {
			DEGUB("EXI %i:0 deselected\n", channel);
			endline = true;
		}
		if(!exi_devices[channel][0]) {
			if(endline) {
				DEGUB("No device installed!\n");
			}
		} else {
			exi_devices[channel][0]->notify_deselected();
		}
	}

#define RISE(bit) ((statusreg & makeflag(bit)) && !(prev & makeflag(bit)))
#define ES_DEGUB_CONDITION (g::exi_log || \
	!(channel == 0 && (RISE(22) || (RISE(23) && !EXI_01_DEGUB_CONDITION))))

	if(RISE(22)) {
		if(g::exi_log || channel != 0) {	//0:2 is BBA, we've dealt with it.
			DEGUB("EXI %i:2 selected", channel);
			endline = true;
		}
	}
	if(RISE(23)) {
		if(g::exi_log || !(channel == 0 && !EXI_01_DEGUB_CONDITION)) {
			DEGUB("EXI %i:1 selected", channel);
			endline = true;
		}
	}
	if(RISE(24)) {
		if(g::exi_log || g::memcard_log) {
			DEGUB("EXI %i:0 selected", channel);
			endline = true;
		}
	}
	if(endline && getbitsw(statusreg, 22, 24)) {
		DEGUB(" frequency: %i", getbitsw(statusreg, 25, 27));
		endline = true;
	}
	if(endline) {
		//we've dealt with 0:1 and 0:2
		//if(!(channel != 0 && (RISE(22) || (RISE(23) && (!g::advanced_mode || g::verbose))) {
		DEGUB("\n");
		//}
	}
	if(endline && (statusreg & EICSR_TCINTMSK)) {
		DEGUB("TC interrupt enabled\n");
	}
	if(endline && (statusreg & EICSR_EXIINTMSK)) {
		DEGUB("EXI interrupt enabled\n");
	}
	if(endline && (statusreg & EICSR_EXTINTMSK)) {
		DEGUB("EXT interrupt enabled\n");
	}
	/*DEGUB("EXI channel %i status:\n", channel);
	if(statusreg & makeflag(28)) {
	DEGUB("TC cleared\n");
	}
	if(statusreg & makeflag(30)) {
	DEGUB("EXI cleared\n");
	}
	if(statusreg & makeflag(20)) {
	DEGUB("EXT cleared\n");
	}*/

	DWORD newstatus = CLEARINTFLAGS(prev, statusreg,
		EICSR_EXTINT | EICSR_TCINT | EICSR_EXIINT);

	if(exi.exi_interrupt_waiting[channel] &&
		(newstatus & (EICSR_EXIINT | EICSR_EXIINTMSK)) == EICSR_EXIINTMSK)
	{
		newstatus |= EICSR_EXIINT;
		exi.exi_interrupt_waiting[channel] = false;
		interrupt.raise(INTEX_EXI, STRING_PLUS_DIGIT("Waiting EXI ", channel)+" EXI");
	}

	if(exi.ext_interrupt_waiting[channel] &&
		(newstatus & (EICSR_EXTINT | EICSR_EXTINTMSK)) == EICSR_EXTINTMSK)
	{
		newstatus |= EICSR_EXTINT;
		exi.ext_interrupt_waiting[channel] = false;
		interrupt.raise(INTEX_EXI, STRING_PLUS_DIGIT("Waiting EXI ", channel)+" EXT");
	}

	hww(EXI_STATUS(channel), newstatus);
}
