// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef INTERRUPT_H
#define INTERRUPT_H

#include "emutime.h"
#include "common.h"
#include <list>

class CriticalSectionWithSpin {
public:
	CriticalSectionWithSpin() : spin(0) {
		InitializeCriticalSection(&cs);
	}
	~CriticalSectionWithSpin() {
		if(spin != 0)
			throw generic_fatal_exception("CriticalSectionWithSpin deleted while spun!");
		DeleteCriticalSection(&cs);
	}
	void spinUp() {
		EnterCriticalSection(&cs);
		spin++;
		//VDEGUB("spinUp of 0x%08X to %i\n", &cs, spin);
		if(spin > 10)
			throw generic_fatal_exception("CriticalSectionWithSpin spin overflow!");
	}
	void spinDown() {
		if(spin == 0)
			throw generic_fatal_exception("CriticalSectionWithSpin spin underflow!");
		spin--;
		//VDEGUB("spinDown of 0x%08X to %i\n", &cs, spin);
		//if(spin == 0) {
		//VDEGUB("  Leaving Critical Section\n");
		//EnterCriticalSection(&cs);  //hack to avoid loss of wakeup  //useless
		LeaveCriticalSection(&cs);
		//}
	}
private:
	CRITICAL_SECTION cs;
	BYTE spin;
};

class CSSHandler {
public:
	CSSHandler(CriticalSectionWithSpin& c) : css(c) {
		css.spinUp();
	}
	~CSSHandler() {
		css.spinDown();
	}
private:
	CSSHandler(const CSSHandler&);
	CSSHandler& operator=(const CSSHandler&);
	CriticalSectionWithSpin& css;
};

class Hardware;
class CoreBase;

typedef void (*hw_callback)(Hardware *h);
struct PENDING_INTERRUPT {
	WORD vector;
	hw_callback callback;
	std::string desc;
};

typedef void (*event_callback)(void *);
struct TIMED_EVENT {
	event_callback function;
	void *arg;
	ULI time;
	std::string desc;
};

template<event_callback f>
class event_test : public std::unary_function<TIMED_EVENT, bool> {
public:
	bool operator() (const TIMED_EVENT &val) {
		return val.function == f;
	}
};

extern CriticalSectionWithSpin css_interrupts;

class InterruptRaiser {
public:
	InterruptRaiser(std::list<PENDING_INTERRUPT> &pi, std::list<TIMED_EVENT> &ti, ULI &et,
		CycleCounterBase &ccb) : pending_interrupts(pi), timed_events(ti), evt(et), m_cc(ccb)
	{}
	bool raise(WORD vector) {
		return raiseAsync(vector, NULL, "Standard interrupt"); }
	bool raise_unless_pending(WORD vector, const std::string &desc) {
		return raiseAsync_unless_pending(vector, NULL, desc); }
	bool raiseAsync(WORD vector, hw_callback callback, const std::string &desc);
	bool raiseAsync_unless_pending(WORD vector, hw_callback callback,
		const std::string &desc);
	void add_event(const TIMED_EVENT &event);
	template<class Pr1> void remove_events(Pr1 pred) {
		CSSHandler csshandler(css_interrupts);
		timed_events.remove_if(pred);
	}
	ULI event_time() { CSSHandler csshandler(css_interrupts); return evt; }
	void dump_events(const char *intro);
private:
	InterruptRaiser& operator=(const InterruptRaiser&);

	std::list<PENDING_INTERRUPT> &pending_interrupts;
	std::list<TIMED_EVENT> &timed_events;
	ULI &evt;
	CycleCounterBase &m_cc;
};

#define INTERRUPT_SYSTEMRESET	0x0100
#define INTERRUPT_MACHINECHECK	0x0200
#define INTERRUPT_DSI		0x0300
#define INTERRUPT_ISI		0x0400
#define INTERRUPT_EXTERNAL	0x0500
#define INTERRUPT_ALIGNMENT	0x0600
#define INTERRUPT_PROGRAM	0x0700
#define INTERRUPT_FPUNAVAIL	0x0800
#define INTERRUPT_DEC		0x0900
#define INTERRUPT_SC		0x0C00
#define INTERRUPT_TRACE		0x0D00
#define INTERRUPT_PERFMON	0x0F00
#define INTERRUPT_IABR		0x1300
#define INTERRUPT_THERMAL	0x1700

#endif	//INTERRUPT_H
