// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "exi_memcard.h"
#include "memcard.h"
#include "degub.h"
#include "ini.h"

#define MCDEGUB if(g::memcard_log) DEGUB

#define CARD_SECTORS memcard_sizes[MCT2MSC(m_type)]
#define CARD_SIZE_BYTES (CARD_SECTORS * MEMCARD_SECTOR_SIZE)

static const DWORD mc_exi_devtypes[] = { EXI_DEVTYPE_MEMORY_CARD_59,
EXI_DEVTYPE_MEMORY_CARD_123, EXI_DEVTYPE_MEMORY_CARD_251, EXI_DEVTYPE_MEMORY_CARD_507,
EXI_DEVTYPE_MEMORY_CARD_1019 | 0x500, EXI_DEVTYPE_MEMORY_CARD_2043 };
static const DWORD mc_latencies[] = { 4, 4, 4, 4, 0x80, 4 };

EXIDevice_Memcard::EXIDevice_Memcard(EXIInterruptRaiser& eir, MemcardType type,
																		 const std::string& filename, bool caching) :
state(Ready), write_buffer(1024), cardmemory(0), interrupt(eir), m_type(type),
m_filename(filename), m_caching(caching) {
	if(!construct()) {
		DumpGLE();
		throw generic_fatal_exception("Memcard file error!");
	}
}

bool EXIDevice_Memcard::construct() {
	if(m_file.open(m_filename.c_str(), File::OpenExisting)) {
		int temp;
		TGLE(GetMCFileContent(m_file, temp));
		if(MSC2MCT(temp) != m_type) {
			FAIL(UE_INVALID_FILE_SIZE);
		}
		if(m_caching) {
			Container<BYTE> tempcon(CARD_SECTORS * MEMCARD_SECTOR_SIZE);
			TGLE(m_file.read(tempcon, MEMCARD_SECTOR_SIZE * CARD_SECTORS));
			cardmemory.steal(tempcon);
		}
	} else {  //file doesn't exist, we're authorized to create a new one.
		TGLE(m_file.open(m_filename.c_str(), File::CreateNew));
		if(m_caching) {
			cardmemory.resize(CARD_SECTORS * MEMCARD_SECTOR_SIZE);
			ZeroMemory(cardmemory, CARD_SECTORS * MEMCARD_SECTOR_SIZE);
		} else {
			Container<BYTE> tempcon(CARD_SECTORS * MEMCARD_SECTOR_SIZE);
			ZeroMemory(tempcon, CARD_SECTORS * MEMCARD_SECTOR_SIZE);
			TGLE(m_file.write(tempcon, MEMCARD_SECTOR_SIZE * CARD_SECTORS));
		}
	}
	return true;
}

EXIDevice_Memcard::~EXIDevice_Memcard() {
	if(!destruct()) {
		DumpGLE();
		throw generic_fatal_exception("Memcard file error!");
	}
}

bool EXIDevice_Memcard::destruct() {
	TGLE(flush());
	TGLE(m_file.close());
	return true;
}

bool EXIDevice_Memcard::flush() {
	if(m_caching) {
		TGLE(m_file.write(cardmemory, MEMCARD_SECTOR_SIZE * CARD_SECTORS));
	}
	return true;
}

bool EXIDevice_Memcard::setCaching(bool caching) {
	TGLE(flush());
	m_caching = caching;
	return true;
}

void EXIDevice_Memcard::readCard(DWORD offset, DWORD size, void *dst) {
	if(offset + size < CARD_SIZE_BYTES) {
		if(m_caching) {
			memcpy(dst, cardmemory + offset, size);
		} else {
			HWGLE(m_file.seek(offset));
			HWGLE(m_file.read(dst, size));
		}
	} else {
		DEGUB("Memcard Read is out of bounds!\n");
		ZeroMemory(dst, size);
		if(g::bouehr)
			throw bouehr_exception("Memcard Read is out of bounds!");
	}
}
void EXIDevice_Memcard::writeCard(DWORD offset, DWORD size, void *src) {
	if(offset + size < CARD_SIZE_BYTES) {
		if(m_caching) {
			memcpy(cardmemory + m_offset, src, size);
		} else {
			HWGLE(m_file.seek(offset));
			HWGLE(m_file.write(src, size));
		}
	} else {
		MCDEGUB("Memcard Write is out of bounds!\n");
		if(g::bouehr)
			throw bouehr_exception("Memcard Write is out of bounds!");
	}
}

void EXIDevice_Memcard::notify_deselected() {
	MCDEGUB("Memcard deselected\n");
	if(write_buffer.size() != 0) {
		DEGUB("Unemulated Memcard command. ");
		PrintPacket(write_buffer.p(), write_buffer.size());
		throw hardware_fatal_exception("Unemulated Memcard command");
	}
	state = Ready;
}

EXIResultValue EXIDevice_Memcard::writeImm(DWORD size, DWORD data) {
	if(state == Ready) {
		EXIResultValue erv = EXI_HANDLED;
		MCDEGUB("Memcard Imm write: %i, 0x%08X\n", size, data);
		data = swapw(data);
		write_buffer.write(size, (&MAKE(BYTE, data) + (4-size)));

		if(write_buffer.size() == 1) {
			BYTE input = *write_buffer.p();
			if(input == 0x89) {
				MCDEGUB("Memcard Status clear (ignored)\n");
				write_buffer.clear();
			}
		} else if(write_buffer.size() == 2) {
			WORD input = swaph(*MAKEP(WORD, write_buffer.p()));
			switch(input) {
			case 0x0000:
				MCDEGUB("Memcard Device ID request\n");
				state = ExpectingSingleImmRead;
				imm_output = mc_exi_devtypes[m_type];
				write_buffer.clear();
				break;
			case 0x8300:
				MCDEGUB("Memcard Status request\n");
				state = ExpectingSingleImmRead;
				imm_output = swapw(0xC3); //t'is what my 3rd-party 1019 gives
				//0x81); //uninitialized  //unmounted. mounting doesn't work.
				//0x41);  //signal ready?  //0x01
				write_buffer.clear();
				break;
			case 0x8101:
				MCDEGUB("Memcard 0x8101 (__CARDEnableInterrupt, ignored)\n");
				state = Ready;
				write_buffer.clear();
				break;
			case 0x8500:
				MCDEGUB("Memcard get ID\n");
				state = ExpectingSingleImmRead;
				imm_output = 0;	//I don't know what should be here
				write_buffer.clear();
				break;
			default:
				erv = EXI_UNHANDLED;
			}
		} else if(write_buffer.size() == 3 && write_buffer.p()[0] == 0xF1) {
			DWORD address = (write_buffer.p()[1] << 17) | (write_buffer.p()[2] << 9);
			MCDEGUB("Memcard Erase Sector: address=0x%X\n", address);
			interrupt.raiseEXI("Memcard Erase Sector");
			state = Ready;
			write_buffer.clear();
		} else if(write_buffer.size() == 5 && write_buffer.p()[0] == 0xF2) {
			state = ExpectingWrite;
			BYTE *offset = write_buffer.p() + 1;
			m_offset = (offset[0] << 17) | (offset[1] << 9) | (offset[2] << 7) | offset[3];
			MCDEGUB("Memcard Write request: offset 0x%X\n", m_offset);
			interrupt.raiseEXI("Memcard Write");
			write_buffer.clear();
		} else if(write_buffer.size() == 5 && write_buffer.p()[0] == 0x52) {
			state = ExpectingLatency;
			BYTE *offset = write_buffer.p() + 1;
			m_offset = (offset[0] << 17) | (offset[1] << 9) | (offset[2] << 7) | offset[3];
			MCDEGUB("Memcard Read request: offset 0x%X\n", m_offset);
			/*if(*(DWORD*)(write_buffer.p() + 5) != 0) {
			//wtf is this anyway?
			MCDEGUB("Memcard Read request padding data = 0x%08X\n",
			*MAKEP(DWORD, write_buffer.p() + 5));
			//throw hardware_fatal_exception("Memcard strange error no.6");
			}*/
			write_buffer.clear();
		}
		if(erv == EXI_HANDLED)
			return erv;
	} else if(state == ExpectingWrite) {
		MCDEGUB("Memcard write: offset 0x%08X | %i bytes\n", m_offset, size);
		writeCard(m_offset, size, &data);
		m_offset += size;
		return EXI_HANDLED;
	} else if(state == ExpectingLatency) {
		MCDEGUB("Memcard Imm write: %i, 0x%08X\n", size, data);
		data = swapw(data);
		write_buffer.write(size, (&MAKE(BYTE, data) + (4-size)));
		if(mc_latencies[m_type] == write_buffer.size()) {
			state = ExpectingRead;
			MCDEGUB("Memcard latency: %i\n", write_buffer.size());
			write_buffer.clear();
		}
		return EXI_HANDLED;
	}

	DEGUB("Unhandled Memcard Imm write: %i, 0x%08X\n", size, data);
	//if(g::bouehr)
	throw bouehr_exception("Unhandled Memcard Imm write");
	//return EXI_UNHANDLED;
}

EXIResultValue EXIDevice_Memcard::readImm(DWORD size, DWORD &data) {
	if(state == ExpectingSingleImmRead) {
		data = imm_output;
		state = Ready;
		MCDEGUB("Memcard SImm read: %i, 0x%08X\n", size, data);
		return EXI_HANDLED;
	} else if(state == ExpectingRead) {
		MCDEGUB("Memcard read: offset 0x%08X | %i bytes\n", m_offset, size);
		readCard(m_offset, size, &data);
		m_offset += size;
		return EXI_HANDLED;
	} else {
		//if((g::degub_run || g::bouehr) && !g::recompiler) {
		DEGUB("Unhandled Memcard Imm read: %i, 0x%08X\n", size, data);
		//}
		//if(g::bouehr)
		throw bouehr_exception("Unhandled Memcard Imm read");
		//data = UNDEFINED_PPCWORD;
		//return EXI_UNHANDLED;
	}
}

EXIResultValue EXIDevice_Memcard::writeDMA(DWORD size, DWORD address,
																					 MemInterface &mem)
{
	if(state == ExpectingWrite) {
		MCDEGUB("Memcard write: offset 0x%08X | %i bytes\n", m_offset, size);
		writeCard(m_offset, size, mem.getp_physical(address, size));
		m_offset += size;
		return EXI_HANDLED;
	} else {
		DEGUB("Unhandled Memcard DMA write: %i, 0x%08X\n", size, address);
		//if(g::bouehr)
		throw bouehr_exception("Unhandled Memcard DMA write");
		//return EXI_UNHANDLED;
	}
}

EXIResultValue EXIDevice_Memcard::readDMA(DWORD size, DWORD address,
																					MemInterface &mem)
{
	if(state == ExpectingRead) {
		MCDEGUB("Memcard read: offset 0x%08X | %i bytes\n", m_offset, size);
		readCard(m_offset, size, mem.getp_physical(address, size));
		m_offset += size;
		return EXI_HANDLED;
	} else {
		DEGUB("Unhandled Memcard DMA read: %i, 0x%08X\n", size, address);
		//if(g::bouehr)
		throw bouehr_exception("Unhandled Memcard DMA read");
		//return EXI_UNHANDLED;
	}
}

/*void EXIDevice_Memcard::do_exiinthack() {
//interrupt.raiseDelay(EICSR_EXIINT, EICSR_EXIINTMSK, 16000); //arbitrary delay
event.add_delay(16000, "EXI int hack"); //arbitrary delay
}*/

bool GetMCFileContent(const char* filename, int& content) {
	ReadOnlyFile file;
	TGLE(file.open(filename));
	TGLE(GetMCFileContent(file, content));
	return true;
}

bool GetMCFileContent(ReadOnlyFile& file, int& content) {
	content = 0;

	uint file_size;
	TGLE(file.size(&file_size));

	for(int i=0; i<sizeof(memcard_sizes)/sizeof(int); i++) {
		if(file_size == memcard_sizes[i]*MEMCARD_SECTOR_SIZE) {
			content = i;
			break;
		}
	}
	if(content == 0) {
		FAIL(UE_INVALID_FILE_SIZE);
	}
	return true;
}

