// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef MEMINTERFACE_H
#define MEMINTERFACE_H

#include "common.h"
#include "container.h"
#include "hardware.h"
#include "defineCommons.h"
#include <vector>

class RegistersBase;
class InterruptRaiser;
class Hardware;

#define BAD_ADDRESS MAX_DWORD

#define LOCKED_CACHE_BASE 0xE0000000
#define LOCKED_CACHE_SIZE (16*K)
#define LOCKED_CACHE_END (LOCKED_CACHE_BASE + LOCKED_CACHE_SIZE)
#define IS_LOCKED_CACHE_ADDRESS(address) ((address) >= LOCKED_CACHE_BASE &&\
	(address) < LOCKED_CACHE_END)

#define EFB_BASE 0xC8000000
#define EFB_PBASE 0x08000000
#define EFB_SIZE (2*M)
#define IS_EFB_ADDRESS(address) ((address) >= EFB_BASE &&\
	(address) < (EFB_BASE + EFB_SIZE))
#define IS_EFB_PADDRESS(address) ((address) >= EFB_PBASE &&\
	(address) < (EFB_PBASE + EFB_SIZE))

#define HARDWARE_WRITE (PHYSICALIZE(address) >= 0x0C000000 &&\
	PHYSICALIZE(address) <= 0x0C008000)
#define HARDWARE_READ (PHYSICALIZE(address) >= 0x0C000000 &&\
	PHYSICALIZE(address) < 0x0C008000)
#define TRANSLATE_D_READ(address) if(g::advanced_mode) {\
	address = translate_d_read(address); }
#define TRANSLATE_D_WRITE(address) if(g::advanced_mode) {\
	address = translate_d_write(address); }

template<class T> inline char tname();
#define DEFINE_TCHAR(name, type) template<> inline char tname<type>() { return #@name; }
MEM_ACCESS_TYPES(DEFINE_TCHAR);

class MemInterface {  //Only suitable for rare operations, because of virtual overhead
public:
	MemInterface(RegistersBase &_r, InterruptRaiser &i, Hardware &_h, bool rec);
	virtual ~MemInterface();

#define DECLARE_MEM_READ(name, type) type r##name(DWORD address) {\
	return tr<type>(address); }
	MEM_ACCESS_TYPES(DECLARE_MEM_READ);
	//Write
#define DECLARE_MEM_WRITE(name, type) void w##name(DWORD address, type data) {\
	tw(address, data); }
	MEM_ACCESS_TYPES(DECLARE_MEM_WRITE);

	//Read using physical address
#define DECLARE_MEM_PREAD(name, type) type pr##name(DWORD address) {\
	return tpr<type>(address); }
	MEM_ACCESS_TYPES(DECLARE_MEM_PREAD);
	//Write using physical address
#define DECLARE_MEM_PWRITE(name, type) void pw##name(DWORD address, type data) {\
	tpw(address, data); }
	MEM_ACCESS_TYPES(DECLARE_MEM_PWRITE);

	void read(DWORD address, DWORD size, void *dest);
	void write(DWORD address, DWORD size, const void *src);
	void read_cached(DWORD address, DWORD size, void *dest);
	void write_cached(DWORD address, DWORD size, const void *src);
	void read_physical(DWORD address, DWORD size, void *dest);
	void write_physical(DWORD address, DWORD size, const void *src);

	BYTE *getp_translated(DWORD address, DWORD size) { TRANSLATE_D_READ(address);
	return getp_physical(PHYSICALIZE(address), size); }
	BYTE *getp_physical(DWORD address, DWORD size);

	void dumpmem() const;

	void bat_notify(DWORD spr);  //Call whenever a BAT changes.
	void msr_notify();  //Call when MSR_PR, _IR or _DR changes.
	void bat_notify_all();  //Call when multiple BATs changes.

	DWORD translate_i(DWORD ea);
	DWORD translate_d_read(DWORD ea);
	DWORD translate_d_write(DWORD ea);
	DWORD translate_d_read_rec(DWORD ea);
	DWORD translate_d_write_rec(DWORD ea);

	void raiseISI();

	BYTE *pmem() { return mainmem; }
	BYTE *pcmem() { return cachemem; }
	const BYTE *pmem() const { return mainmem; }
	const BYTE *pcmem() const { return cachemem; }

	void load_cache() { memcpy(cachemem, mainmem, CACHE_SIZE); }

	void degub() const { if(memdegub[0]) { DEGUB("%s\n", memdegub); } clear_degub(); }
	void clear_degub() const { memdegub[0] = 0; p_memdegub = memdegub; }

	//Kinda unrelated functionality
	const bool recompiler;  //used to be protected. needed for GP operation in dual mode.
	DWORD getmsr() const;
	const RegistersBase &reg;
protected:
	void *getphc(DWORD address);
	void *getph(DWORD address);
	void *getpph(DWORD paddress);

	Container<BYTE> mainmem, cachemem, locked_cache, efb;

	mutable char memdegub[8*K], *p_memdegub;
	bool mdegub_on;
#define MDEGUB(...)// if(false) p_memdegub += sprintf
private:
	RegistersBase &r;
	InterruptRaiser &interrupt;
	Hardware &h;

	struct BAT {
		DWORD mask, src, dst;
		bool supervisor, user, read, write;
	};
	std::vector<BAT> ibatv, dbatv;
	void rebuild_batv(bool data);
	bool ibats_valid, dbats_valid;
	DWORD translate_d(DWORD ea, bool write, bool recompiler);

	void dump_bat(DWORD spr);
	bool bat_enabled[8];
	bool bat_valid[8];
	DWORD msrbits;

	DWORD dpage_faults, ipage_faults;
	DWORD page_table_search(DWORD ea, bool instruction, bool write, bool recompiler);
	DWORD page_group_search(DWORD ea, bool instruction, bool write,
		bool secondary, DWORD pta, DWORD sr);

	template<class T> T tr(DWORD address);
	template<class T> T tpr(DWORD address);
	template<class T> void tw(DWORD address, T data);
	template<class T> void tpw(DWORD address, T data);
};
/*
#define DEFINE_MEM_TSPECIALIZATIONS(name, type)\
template<> type MemInterface::tr<type>(DWORD address);\
template<> type MemInterface::tpr<type>(DWORD address);\
template<> void MemInterface::tw<type>(DWORD address, type data);\
template<> void MemInterface::tpw<type>(DWORD address, type data);\

MEM_ACCESS_TYPES(DEFINE_MEM_TSPECIALIZATIONS);*/

template<class T> T MemInterface::tr(DWORD address) {
	TRANSLATE_D_READ(address);
	if(HARDWARE_READ) {
		return h.tr<T>((WORD)address);
	} else {
		T data = tswap(*(T*)getphc(address));
		MDEGUB(p_memdegub, "r%c[%08X]%0*I64X ", tname<T>(), address, sizeof(T)*2,
			(QWORD)data);
		return data;
	}
}
template<class T> T MemInterface::tpr(DWORD address) {
	T data = tswap(*(T*)getpph(address));
	MDEGUB(p_memdegub, "pr%c[%08X]%0*I64X ", tname<T>(), address, sizeof(T)*2,
		(QWORD)data);
	return data;
}

template<class T> void MemInterface::tw(DWORD address, T data) {
	TRANSLATE_D_WRITE(address);
	MDEGUB(p_memdegub, "w%c[%08X]%0*I64X ", tname<T>(), address, sizeof(T)*2, (QWORD)data);
	if(HARDWARE_WRITE) {
		h.tw<T>((WORD)address, data);
	} else {
		*(T*)getphc(address) = tswap(data);
	}
}
template<class T> void MemInterface::tpw(DWORD address, T data) {
	MDEGUB(p_memdegub, "pw%c[%08X]%0*I64X ", tname<T>(), address, sizeof(T)*2, (QWORD)data);
	*(T*)getpph(address) = tswap(data);
}

#endif	//MEMINTERFACE_H
