// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef RECBASE_H
#define RECBASE_H

#include "container.h"
#include "common.h"

#define RECBUF_MIN_SIZE 1*K
struct POSOFF {
	DWORD pos, offset;
};

class RecBlockInterface {
public:
	virtual const BYTE* rcb() const = 0;
	virtual size_t recbuf_size() const = 0;
};
void dumpRecBlock(const RecBlockInterface& aBuf, const string& aBasename);

class RecBufBase : public RecBlockInterface {
public:
	RecBufBase() : container(RECBUF_MIN_SIZE), _size(0) {}

	void add_byte(BYTE data) {
		if(container.size() - _size < 1) {
			recapacitate(container.size() * 2);
		}
		container[_size] = data;
		_size++;
	}
	void add_dword(DWORD data) {
		if(container.size() - _size < 4) {
			recapacitate(container.size() * 2);
		}
		MAKE(DWORD, container[_size]) = data;
		_size += 4;
	}
	//void finalize() { recapacitate(_size); }
	//void clear() { _size = 0; container.resize(0); }
	size_t capacity() const { return container.size(); }
	size_t recbuf_size() const { return _size; }
	BYTE *rcb() { return container.p(); }
	const BYTE *rcb() const { return container.p(); }
	void step_back(size_t newsize) { MYASSERT(newsize <= _size); _size = newsize; }
	void reset() { _size = 0; container.resize(RECBUF_MIN_SIZE); }

	void request_call(DWORD function_pointer) {
		POSOFF po = { _size, function_pointer };
		r_call_v.push_back(po);
		add_dword(0);
	}
	const vector<POSOFF> &rcv() const { return r_call_v; }
private:
	Container<BYTE> container;
	size_t _size;
	vector<POSOFF> r_call_v;  //Request Call Vector
	void recapacitate(size_t newcap) {
		//DEGUB("Recapacitating: %i->%i\n", capacity(), newcap);
		if(newcap < _size || newcap < RECBUF_MIN_SIZE || !is_power_of_2(newcap)) {
			DEGUB("Bad size: %i\n", newcap);
			throw generic_fatal_exception("Bad size in RecBuf resize()!");
		}
		if(newcap != _size) {
			Container<BYTE> new_container(newcap);
			memcpy(new_container, container, _size);
			container.steal(new_container);
		}
	}
};

template<class T> class SharedContainer {
private:
	void *a;

	int &refCount() { MYASSERT(a != NULL); return *(int*)a; }
	void *getDataPtr() const { return ((int*)a) + 1; }

	void allocate(size_t size) {
		MYASSERT(a == NULL);
		//DEGUB("SharedContainer allocate %i\n", size);
		if(size > 100*M)
			throw generic_fatal_exception("SharedContainer size: " + size);

		if(size == 0)
			a = NULL;
		else {
			a = malloc(size+sizeof(int));
			if(!_CrtIsValidHeapPointer(a))
				throw generic_fatal_exception("malloc failed in SharedContainer");
			refCount() = 1;
			//DEGUB("SharedContainer allocated 0x%08X\n", a);
		}
	}
	void release() {
		//DEGUB("SharedContainer release 0x%08X @ count %i... ", a, refCount());
		if(a) {
			if(--refCount() == 0) {
				free(a);
				//DEGUB("Freed.\n");
				a = NULL;
			} else {
				//DEGUB("Not freed.\n");
			}
		} else {
			//DEGUB("NULL.\n");
		}
	}
public:
	SharedContainer(size_t size) : a(NULL) {
		allocate(size);
	}
	SharedContainer(const SharedContainer &other) : a(other.a) {
		refCount()++;
		//DEGUB("SharedContainer created with 0x%08X. New count: %i\n", a, refCount());
	}
	SharedContainer &operator=(const SharedContainer &other) {
		release();
		a = other.a;
		refCount()++;
		//DEGUB("SharedContainer assigned 0x%08X. New count: %i\n", a, refCount());
		return *this;
	}
	~SharedContainer() {
		release();
	}

	operator T*() { return (T *)getDataPtr(); }
	operator const T*() const { return (T *)getDataPtr(); }
	T* p() { return (T *)getDataPtr(); }
	const T* p() const { return (T *)getDataPtr(); }
	T *operator->() { return (T *)getDataPtr(); }
	size_t size() const { return _msize(a) - sizeof(int); }
	int refCount() const { MYASSERT(a != NULL); return *(const int*)a; }
};

class DynaRecBlockBase : public RecBlockInterface {
public:
	DynaRecBlockBase(const DynaRecBlockBase &other) : CCI(_rcb), CCI(rcb_size) {}
	DynaRecBlockBase &operator=(const DynaRecBlockBase &other) {
		//MYASSERT(_rcb == NULL);
		CCO(_rcb);
		CCO(rcb_size);
		return *this;
	}

	BYTE *rcb() { return _rcb; }
	const BYTE *rcb() const { return _rcb; }
	size_t recbuf_size() const { return rcb_size; }
protected:
	DynaRecBlockBase(const RecBufBase &recbuf, size_t additional_data_size=0);
	int refCount() const { return _rcb.refCount(); }
private:
	SharedContainer<BYTE> _rcb;
	size_t rcb_size;

	void copyRCB(const RecBufBase &recbuf);
};

//This class is designed to set a pointer to null on destruction.
class PointerKeeper {
public:
	PointerKeeper(void **pp) : p(pp) {}
	~PointerKeeper() { *p = NULL; }
private:
	void **p;
};
#define KEEP_POINTER(pointer, local_variable) PointerKeeper\
	keeper_of_##pointer((void**)&pointer); pointer = &local_variable;

#endif	//RECBASE_H
