// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef RECOMPILER_H
#define RECOMPILER_H

#include "oparray.h"
#include "container.h"
#include "registers.h"
#include "hardware.h"
#include "corebase.h"
#include "ini.h"
#include "recops.h"
#include "recmem.h"
#include "recbase.h"

using namespace std;

typedef DWORD (*RecFunc)();

class Recompiler : public CoreBase {
public:
	Recompiler(CORE_ARGUMENTS a);
	~Recompiler();
	bool disassemble() { FAIL(UE_REC_INTERNAL_ERROR); }
	string getdasma(DWORD address);
	string getdasmo(DWORD cia, DWORD opcode);

	DWORD getNIA() { return reverse_rcbt(nia_pos); }
private:
	bool handle_load();
	bool _run(DWORD runstop, bool degub, QWORD degub_skip, bool dumpmem);
	HWND hWnd;

	bool recompile(DWORD start_eia, DWORD end_eia);	//Stop before recompiling end_eia
	void function_container(RecFunc function);
	bool take_interrupt(DWORD requested_nia);
	bool next_instruction_is_static_branch();

	bool clearDRB(size_t index);
	void makeStaticBranchUnknown(DWORD ref_pos);
	void makeStaticBranchKnown(DWORD ref_pos, DWORD target_pos);
	void addReference(size_t from, size_t to, DWORD pos);

	RecMemory recmem;
	BYTE *const memory;
	BYTE *const cache_memory;

	//Stores pos for later filling by add_label(). For use with relative jump instructions.
	//All labels must be satisfied at end of opcode.
	struct LABEL {
		DWORD pos;
		const char *label;
	};
	void request_label(const char *label);
	void add_label(const char *label);
	void request_internal_label(const char *label);
	void add_internal_label(const char *label);
	void clear_internal_labels();
	void _add_label(vector<LABEL> &_lv, vector<LABEL> &_lrv, const char *label);
	void _request_label(vector<LABEL> &_lv, vector<LABEL> &_lrv, const char *label);
	//[Internal|Long] Label [Request] Vector
	vector<LABEL> lv, lrv, ilv, ilrv;//, llv, llrv;

	//DWORD: function_pointer - DWORD(recbuf.p()) - (pos + 4)
	//void request_call(DWORD function_pointer);

	struct IAPOS {  //Instruction address and recompiled Position
		DWORD ia, pos;
	};

	struct RefBlock {
		size_t drbv_index;
		vector<DWORD> refs;
	};

	class RecBuf : public RecBufBase {
	public:
		RecBuf(DWORD start_eia, DWORD start_pia) : CI(start_eia), CI(start_pia) {}

		const DWORD start_eia, start_pia;
		vector<DWORD> rcbtb;
		vector<DWORD> ppc_code;
		vector<IAPOS> trv;  //Translation Request Vector
	};
	//RecBufBase *cur_recbufbase;  //Must be null out of recompile()
	RecBuf *cur_recbuf;  //Must be null out of recompile()

	class DynaRecBlock : public DynaRecBlockBase {
	public:
		DynaRecBlock(const RecBuf &recbuf);
		DynaRecBlock(const DynaRecBlock &other) : DynaRecBlockBase(other),
			CCI(_start_eia), CCI(_start_pia), CCI(_rcbtb_size), CCI(their_refs), CCI(my_refs) {}
		DynaRecBlock &operator=(const DynaRecBlock &other) {
			DynaRecBlockBase::operator=(other);
			CCO(_start_eia); CCO(_start_pia); CCO(_rcbtb_size);
			CCO(their_refs); CCO(my_refs);
			return *this;
		}
		friend bool Recompiler::clearDRB(size_t index); //DynaRecBlock
		friend void Recompiler::addReference(size_t from, size_t to, DWORD pos);

		//ReCBuf Translation Buffer
		const DWORD *rcbtb() const { return (DWORD*)(rcb() + recbuf_size()); }
		DWORD *rcbtb() { return (DWORD*)(rcb() + recbuf_size()); }
		//same size as rcbtb
		const DWORD *ppc_code() const { return rcbtb() + _rcbtb_size; }
		DWORD *ppc_code() { return rcbtb() + _rcbtb_size; }
		DWORD start_eia() const { return _start_eia; }
		DWORD end_eia() const { return _start_eia + _rcbtb_size * 4; }
		DWORD start_pia() const { return _start_pia; }
		DWORD end_pia() const { return _start_pia + _rcbtb_size * 4; }
		size_t rcbtb_size() const { return _rcbtb_size; }
		DWORD eia2pos(DWORD eia) const { return rcbtb()[(eia - start_eia()) / 4]; }
	private:
		DWORD _start_eia, _start_pia;
		size_t _rcbtb_size;
		list<RefBlock> their_refs; //this is the references to this block from others
		list<size_t> my_refs; //this is the drbv indices of blocks that this block refers to

		void clearReferencesFromBlock(size_t index);
		void clearReferencesToBlock(size_t index);
	};
	vector<const DynaRecBlock> drbv;

	enum StaticFuncs { SF_MEM1WRITE, SF_STATICBRANCHHANDLER, SF_MEM1READ, _NSTATICFUNCS };

	class StaticFuncBlock : private DynaRecBlockBase {
	public:
		StaticFuncBlock(const RecBufBase &recbuf, const size_t *offs);
		StaticFuncBlock(const StaticFuncBlock &other) : DynaRecBlockBase(other) {
			memcpy(offsets, other.offsets, sizeof(offsets));
		}
		StaticFuncBlock &operator=(const StaticFuncBlock &other) {
			DynaRecBlockBase::operator =(other);
			memcpy(offsets, other.offsets, sizeof(offsets));
			return *this;
		}

		DWORD getFuncPos(StaticFuncs func) const {
			MYASSERT(func < _NSTATICFUNCS && func >= 0);
			return DWORD(rcb() + ((func == SF_MEM1READ) ? 0 : offsets[func]));
		}
	private:
		size_t offsets[_NSTATICFUNCS];
	};
	StaticFuncBlock m_staticblock;
	StaticFuncBlock createStaticBlock();
	void _add_static_branch_handler();

	//(Effective Instruction Address to ((Recompilation Buffer) Position)) Translation
	DWORD eia2pos(DWORD eia);
	bool addcia2pos();
	DWORD reverse_rcbt(DWORD rcbpos) const;
	size_t findDRBIndexByEIA(DWORD eia) const;
	size_t findDRBIndexByPIA(DWORD pia) const;
	size_t findDRBIndexByPos(DWORD pos) const;
	size_t getDRBIndexRaw(DWORD eia);
	size_t getDRBIndexRaw_base(DWORD eia, DWORD temp_opcode);
	size_t getDRBIndexRawPIA_base(DWORD pia, DWORD temp_opcode);
	DWORD getPPCCode(DWORD pia);
	DWORD raw_eia2pos(DWORD eia);
	DWORD drbv_eia2pos(DWORD eia);

	void add_drb2overwrite(DWORD pia, size_t new_index);
	void remove_drb_from_overwrite(DWORD pia, size_t index);
	vector<vector<size_t> > m_type2;

	bool do_nia_pos(DWORD current_ia, DWORD requested_nia, bool allow_overwrite);
	bool running_rec(DWORD eia);
	DWORD get_end_eia(DWORD start_eia);
	static size_t get_rcbtb_index(const DynaRecBlock &drb, DWORD h4xx0r_pos);
	bool doICBI();

	//These are for use in the recompile() context only
	DWORD cia, nia, opcode;
	enum InstructionType { IT_NORMAL, IT_BRANCH_UNCONDITIONAL,
		IT_BRANCH_CONDITIONAL, IT_ERROR } last_instruction_type;

	//this should contain the PC address of the next recompiled instruction to execute
	DWORD nia_pos;

	DWORD rec_cycles; //For use in recompile(), to be safekept afterwards.
	DWORD total_rec_cycles; //dito
	mutable RegistersBase old_registers;
	DWORD ipage_faults;

	//stack should be used instead, where possible

	enum Rec_Errorcodes { REC_NO_ERROR, REC_INFINITE_LOOP,
		REC_UNDEFINED_OPCODE, REC_UNEMULATED_INSTRUCTION, REC_EXCEPTION,
		REC_INVALID_INSTRUCTION_FORM, REC_INVALID_MEMORY_ADDRESS,
		REC_UNEMULATED_INSTRUCTION_FORM,
		REC_INTERNAL_ERROR, REC_IPAGE_FAULT, REC_ONESTEP_STATIC_BRANCH, REC_OOVPA,
		REC_UNIMPLEMENTED_OPCODE, REC_STATIC_BRANCH, REC_BCLR, REC_BCCTR, REC_RFI,
		REC_ICBI };
	BYTE errorcode;

	//puts a cr field by eflags(SF and CF) and XER_SO into high nibble of al, trashes eflags
	void add_makecr2al();  //treats the result as signed
	void add_makecr2al_cmp(); //special handling for cmp(i)
	void add_makecr2al_unsigned(); //guess what? ;)
	//helper function to add_makecr2al, uses al
	void add_or_cr1al_so();
	//by eflags(SF and CF) and XER_SO, trashes al, trashes and restores eflags except OF
	void add_setcr0();
	//by eflags(OF), trashes ah, trashes and restores eflags except OF
	void add_setoverflow();
	void add_setoverflow_nosave();  //doesn't restore eflags
	//uses add_makecr2al and add_setal2cr, trashes eflags
	void add_setcr(DWORD crfD);
	//copies high nibble of al into r.cr(crfD), low nibble must be 0x0, trashes eflags
	void add_setal2cr(DWORD crfD);
	//bc ctr/cond code, modifies al, trashes eflags
	void add_bc_ctr_cond(DWORD BO, DWORD BI);
	void add_bc_ctr(DWORD BO);
	void add_bc_cond(DWORD BO, DWORD BI);
	//sets XER_CA by eflag CF/NZF, trashes eflags
	void add_setxerca_by_cf();
	void add_setxerca_by_nzf();
	//sets the 5 bits of fprf by floating register ST(0), leaves C3-C0 in low eax, trashes ecx, trashes eflags
	void add_set_fpscr_fprf();
	//sets r.cr(crfD) by index in eax left by add_set_fpscr_fprf, trashes al, trashes eflags
	void add_set_fpcc_cr(DWORD crfD);
	//integral part of b and bc
	void add_static_branch(DWORD target_address, bool LK);

	//major common parts of load/store instructions. uses labels. trashes ecx and eflags. trashes everything in advanced mode.
	void add_mem1_rA0d(DWORD rA, short d, bool write);
	void add_mem1_x(DWORD rA, DWORD rB, bool write);
	void add_mem2();

	void _add_mem1_base(bool write);
	void _add_mem1_call(bool write);

	//sets intel fp precision to single, uses unused stack space, trashes eflags
	void add_set_fpcw_precision_float();
	//sets intel fp precision to extended, assumes control word is already in stack, trashes eflags
	void add_set_fpcw_precision_extd();
	//sets error status and exits rec code. make sure eax is properly loaded.
	void add_fatal_error(BYTE code);
	//loads eax, sets error status and exits rec code.
	void add_pos_fatal_error(BYTE code);
	//copies high nibble of al (low nibble must be 0) into bits 16-19 of the fpscr, trashes eflags
	void add_set_fpscr_fpcc_by_al();
	//updates FPSCR_FEX and VX, trashes eflags, eax and ebx
	void add_update_fpscr_fex_vx();
	//sets/clears bit in target according to eflags and jcc, trashes eflags
	void add_setflag(DWORD *target, int bit, BYTE jcc);
	//operates on a ps/d operand. use only with loading instructions. modifies ST(0).
	void add_fxx_ps_d(BYTE op1s, BYTE digit, DWORD fr, bool disp8);
	//stores ST(0) in FPR[fr]
	void add_fstx_d(bool pop, DWORD fr, bool disp8);
	//stores ST(0) in PS0(fr) and PS1(fr) and pops the stack
	void add_set_fpr_single(DWORD fr, bool disp8);

	OpArray<void (Recompiler::*)()> rcarr;

	OPCODES(DECLARE_OPFUNC);
	void _addic_();
	void _stwcx_();
	void __undefined();
	void _unimplemented_opcode();
	//UNIMPLEMENTED_OPCODES(DECLARE_OPFUNC);
};

#endif	//RECOMPILER_H
