// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "recompiler.h"
#include "recops.h"

//FPSR[C3-C0] ~= FPSCR[15-19]. 1F-1A is for undefined results; it should never be used.
static const DWORD fxam2fprf[16] = {
	0x0001F000,	//Unsupported
	0x00011000,	//We assume the NaN is quiet
	0x0001E000,	//Unsupported
	0x00011000,	//We assume the NaN is quiet
	0x00004000,	//Positive normalized
	0x00005000,	//Positive infinity
	0x00008000,	//Negative normalized
	0x00009000,	//Negative infinity
	0x00002000,	//Positive zero
	0x0001D000,	//Empty
	0x00012000,	//Negative zero
	0x0001C000,	//Empty
	0x00014000,	//Positive denormalized
	0x0001B000,	//not defined by Intel
	0x00018000,	//Negative denormalized
	0x0001A000
};	//not defined by Intel

//FXAM result (C3-C0) to FPCC for high nibble of al
static const BYTE fxam2fpcc4hal[16] = {
	0xF0,	//Unsupported
	0x10,	//We assume the NaN is quiet
	0xF0,	//Unsupported
	0x10,	//We assume the NaN is quiet
	0x40,	//Positive normalized
	0x90,	//Positive infinity
	0x80,	//Negative normalized
	0x50,	//Negative infinity
	0x20,	//Positive zero
	0xF0,	//Empty
	0x20,	//Negative zero
	0xF0,	//Empty
	0x40,	//Positive denormalized
	0xF0,	//not defined by Intel
	0x80,	//Negative denormalized
	0xF0
};  //not defined by Intel

//------------------------------HELPER FUNCTIONS--------------------------------
//These could be made into functions, called by the recompiled code, instead of
//inline, as they are now, to save memory space.

//14 + 12 = 26 bytes
void Recompiler::add_makecr2al() {
	//jnz check_sign
	AB(0x75);
	request_internal_label("check_sign");
	{ //zero:
		//c = CR0_EQ;
		//mov al, CR0_EQ << 4
		AB(0xB0);
		AB(CR_EQ << 4);
		//jmp check_so
		AB(0xEB);
		request_internal_label("check_so");
	}
	add_internal_label("check_sign");
	//js less
	AB(0x78);
	request_internal_label("less");
	{ //greater:
		//c = CR_GT;
		//mov al, CR_GT << 4
		AB(0xB0);
		AB(CR_GT << 4);
		//jmp check_so
		AB(0xEB);
		request_internal_label("check_so");
	}
	//else
	{
		add_internal_label("less");
		//c = CR_LT;
		//mov al, CR_LT << 4
		AB(0xB0);
		AB(CR_LT << 4);
	}
	add_internal_label("check_so");
	clear_internal_labels();
	add_or_cr1al_so();
}

//14 + 12 = 26 bytes
void Recompiler::add_makecr2al_cmp() {
	//jnl check_greater
	AB(0x7D);
	request_internal_label("check_greater");
	{ //less:
		//c = CR_LT;
		//mov al, CR_LT << 4
		AB(0xB0);
		AB(CR_LT << 4);
		//jmp check_so
		AB(0xEB);
		request_internal_label("check_so");
	}
	add_internal_label("check_greater");
	//jng equal
	AB(0x7E);
	request_internal_label("equal");
	{ //greater:
		//c = CR_GT;
		//mov al, CR_GT << 4
		AB(0xB0);
		AB(CR_GT << 4);
		//jmp check_so
		AB(0xEB);
		request_internal_label("check_so");
	}
	//else
	{
		add_internal_label("equal");
		//c = CR_EQ;
		//mov al, CR_EQ << 4
		AB(0xB0);
		AB(CR_EQ << 4);
	}
	add_internal_label("check_so");
	clear_internal_labels();
	add_or_cr1al_so();
}

void Recompiler::add_makecr2al_unsigned() {
	//jnb check_above
	AB(0x73);
	request_internal_label("check_above");
	{ //below:
		//c = CR_LT;
		//mov al, CR_LT << 4
		AB(0xB0);
		AB(CR_LT << 4);
		//jmp check_so
		AB(0xEB);
		request_internal_label("check_so");
	}
	add_internal_label("check_above");
	//jna equal
	AB(0x76);
	request_internal_label("equal");
	{ //above:
		//c = CR_GT;
		//mov al, CR_GT << 4
		AB(0xB0);
		AB(CR_GT << 4);
		//jmp check_so
		AB(0xEB);
		request_internal_label("check_so");
	}
	//else
	{
		add_internal_label("equal");
		//c = CR_EQ;
		//mov al, CR_EQ << 4
		AB(0xB0);
		AB(CR_EQ << 4);
	}
	add_internal_label("check_so");
	clear_internal_labels();
	add_or_cr1al_so();
}

//12 bytes
void Recompiler::add_or_cr1al_so() {
	//bt r.xer, 31  //is bit 0 in PPC
	AB(0x0F);
	AB(0xBA);
	AB(0x25);
	AD(&r.xer);
	AB(31);
	//jnc end
	AB(0x73);
	request_internal_label("end");
	{ //so:
		//or al, CR_SO << 4
		AB(0x0C);
		AB(CR_SO << 4);
	}
	add_internal_label("end");
	clear_internal_labels();
}

//15 + 26 = 41 bytes
void Recompiler::add_setcr0() {
	//lahf
	ADD_BYTE(0x9F);

	add_makecr2al();

	//store:
	//and r.cr(byte 3/0), 0x0F
	AB(0x80);
	AB(0x25);
	AD(DWORD(&r.cr) + 3);
	AB(0x0F);
	//or r.cr(byte 3/0), al
	AB(0x08);
	AB(0x05);
	AD(DWORD(&r.cr) + 3);
	//sahf
	ADD_BYTE(0x9E);
}

void Recompiler::add_setoverflow() {
	//lahf
	ADD_BYTE(0x9F);
	add_setoverflow_nosave();
	//sahf
	ADD_BYTE(0x9E);
}

void Recompiler::add_setoverflow_nosave() {
	//if(ov)
	//jno notoverflow
	ADD_BYTE(0x71);
	request_internal_label("notoverflow");
	{ //overflow:
		//setflags(xer, XER_OV | XER_SO, true);
		//or r.xer(byte 0/3), (XER_OV | XER_SO) >> 24
		ADD_BYTE(0x80);
		ADD_BYTE(0x0D);
		ADD_DWORD(DWORD(&r.xer) + 3);
		ADD_BYTE((XER_OV | XER_SO) >> 24);
		//jmp end
		ADD_BYTE(0xEB);
		request_internal_label("end");
	}
	//else
	{
		add_internal_label("notoverflow");
		//setflags(xer, XER_OV, false);
		//and r.xer(byte 0/3), (~XER_OV) >> 24
		ADD_BYTE(0x80);
		ADD_BYTE(0x25);
		ADD_DWORD(DWORD(&r.xer) + 3);
		ADD_BYTE((~XER_OV) >> 24);
	}
	add_internal_label("end");
	clear_internal_labels();
}

//18 bytes
void Recompiler::add_setxerca_by_cf() {
	//setflags(r.xer, XER_CA, CARRY?)
	//jc setxerca
	AB(0x72);
	request_internal_label("setxerca");
	{
		//and byte ptr ((&r.xer) + 3), ~0x20
		AB(0x80);
		AB(0x25);
		AD(DWORD(&r.xer) + 3);
		AB(~0x20);
		//jmp end
		AB(0xEB);
		request_internal_label("end");
	}
	{
		add_internal_label("setxerca");
		//or byte ptr ((&r.xer) + 3), 0x20
		AB(0x80);
		AB(0x0D);
		AD(DWORD(&r.xer) + 3);
		AB(0x20);
	}
	add_internal_label("end");
	clear_internal_labels();
}
void Recompiler::add_setxerca_by_nzf() {
	//setflags(r.xer, XER_CA, NOT_ZERO?)
	//jnz setxerca
	AB(0x75);
	request_internal_label("setxerca");
	{ //clearxerca
		//and byte ptr ((&r.xer) + 3), ~0x20
		AB(0x80);
		AB(0x25);
		AD(DWORD(&r.xer) + 3);
		AB(~0x20);
		//jmp end
		AB(0xEB);
		request_internal_label("end");
	}
	{
		add_internal_label("setxerca");
		//or byte ptr ((&r.xer) + 3), 0x20
		AB(0x80);
		AB(0x0D);
		AD(DWORD(&r.xer) + 3);
		AB(0x20);
	}
	add_internal_label("end");
	clear_internal_labels();
}

void Recompiler::add_set_fpscr_fprf() {
	//xor eax, eax
	AB(0x31);
	AB(0xC0);

	//fxam
	AB(0xD9);
	AB(0xE5);
	//fnstsw ax
	AB(0xDF);
	AB(0xE0);

	//move C3-C0 into eax (low nibble of al)  //0100 0111 0000 0000b
	//mov ecx, eax
	AB(0x89);
	AB(0xC1);
	//shr eax, 8
	AB(0xC1);
	AB(0xE8);
	AB(8);
	//and al, 0x07
	AB(0x80);
	AB(0xE0);
	AB(0x07);
	//shr ecx, 11
	AB(0xC1);
	AB(0xE9);
	AB(11);
	//and cl, 0x08
	AB(0x80);
	AB(0xE1);
	AB(0x08);
	//or eax, ecx
	AB(0x09);
	AB(0xC8);

	//mov ecx, [fxam2fprf + eax*4]  //fxam2fprf[C3-C0]
	AB(0x8B);
	AB(0x0C);
	AB(0x85);
	AD(fxam2fprf);
	//and r.fpscr, ~0x0001F000
	AB(0x81);
	AB(0x25);
	AD(&r.fpscr);
	AD(~0x0001F000);
	//or r.fpscr, ecx
	AB(0x09);
	AB(0x0D);
	AD(&r.fpscr);
}

void Recompiler::add_set_fpcc_cr(DWORD crfD) {
	//mov al, [fxam2fpcc4hal + eax]
	AB(0x8A);
	AB(0x80);
	AD(fxam2fpcc4hal);

	add_setal2cr(crfD);
}

//Uses stack space for temp storage
void Recompiler::add_set_fpcw_precision_float() {
	//fnstcw [esp - 2]
	AB(0xD9);
	AB(0x7C);
	AB(0x24);
	AB(-2);
	//and byte [esp - 1], ~0x03 //set PC to 00b
	AB(0x80);
	AB(0x7C);
	AB(0x24);
	AB(-1);
	AB(~0x03);
	//fldcw [esp - 2]
	AB(0xD9);
	AB(0x6C);
	AB(0x24);
	AB(-2);
}
void Recompiler::add_set_fpcw_precision_extd() {
	//or byte [esp - 1], 0x03 //set PC to 11b
	AB(0x80);
	AB(0x4C);
	AB(0x24);
	AB(-1);
	AB(0x03);
	//fldcw [esp - 2]
	AB(0xD9);
	AB(0x6C);
	AB(0x24);
	AB(-2);
}

void Recompiler::add_set_fpscr_fpcc_by_al() {
	//and r.fpscr(byte 1 (or 2), 0x0F
	AB(0x80);
	AB(0x25);
	AD(DWORD(&r.fpscr) + 1);
	AB(0x0F);
	//or r.fpscr(byte 1 (or 2), al
	AB(0x08);
	AB(0x05);
	AD(DWORD(&r.fpscr) + 1);
}

void Recompiler::add_update_fpscr_fex_vx() {
	/*setflags(fpscr, FPSCR_VX, getflag(fpscr, FPSCR_VXSNAN) ||
	getflag(fpscr, FPSCR_VXISI) || getflag(fpscr, FPSCR_VXIDI) ||
	getflag(fpscr, FPSCR_VXZDZ) || getflag(fpscr, FPSCR_VXIMZ) ||
	getflag(fpscr, FPSCR_VXVC) || getflag(fpscr, FPSCR_VXSOFT) ||
	getflag(fpscr, FPSCR_VXSQRT) || getflag(fpscr, FPSCR_VXCVI));*/
	//mov eax, r.fpscr
	AB(0xA1);
	AD(&r.fpscr);
	//test eax, {vx's}
	AB(0xA9);
	AD(FPSCR_VXSNAN | FPSCR_VXISI | FPSCR_VXIDI | FPSCR_VXZDZ | FPSCR_VXIMZ |
		FPSCR_VXVC | FPSCR_VXSOFT | FPSCR_VXSQRT | FPSCR_VXCVI);
	add_setflag(&r.fpscr, 2, JCC_NZ); //set if not zero, clear otherwise

	/*setflags(fpscr, FPSCR_FEX,
	(getflag(fpscr, FPSCR_VX) && getflag(fpscr, FPSCR_VE)) ||
	(getflag(fpscr, FPSCR_OX) && getflag(fpscr, FPSCR_OE)) ||
	(getflag(fpscr, FPSCR_UX) && getflag(fpscr, FPSCR_UE)) ||
	(getflag(fpscr, FPSCR_ZX) && getflag(fpscr, FPSCR_ZE)) ||
	(getflag(fpscr, FPSCR_XX) && getflag(fpscr, FPSCR_XE)));*/
	//fpscr is already in eax
	//mov ebx, eax
	AB(0x89);
	AB(0xC3);
	//shr ebx, 22
	AB(0xC1);
	AB(0xEB);
	AB(22);
	//and eax, ebx
	AB(0x21);
	AB(0xD8);
	//test al, 0xF8 //bitmask
	AB(0xA8);
	AB(0xF8);
	add_setflag(&r.fpscr, 1, JCC_NZ); //set if not zero, clear otherwise
}


//--------------------------------HELPER MACROS---------------------------------
//These can not be made into functions, because they depend on their arguments.

void Recompiler::add_setcr(DWORD crfD) {
	add_makecr2al();
	add_setal2cr(crfD);
}

void Recompiler::add_setal2cr(DWORD crfD) {
	//r.setcr(crfD, c);
	if((crfD % 2) != 0) {
		//shr al, 4
		AB(0xC0);
		AB(0xE8);
		AB(4);
	}
	//and [byte ptr] r.cr(crfD/2), (crfD % 2) != 0 ? 0xF0 : 0x0F
	AB(0x80);
	AB(0x25);
	AD(DWORD(&r.cr) + 3 - crfD/2);
	AB((crfD % 2) != 0 ? 0xF0 : 0x0F);
	//or [byte ptr] r.cr(crfD/2), al
	AB(0x08);
	AB(0x05);
	AD(DWORD(&r.cr) + 3 - crfD/2);
}

//getbitr => 0=>4, 1=>3, 2=2, 3=>1, 4=>0. Fix someday by BO << 27(?).
//the ctr/cond stuff can be optimized further by checking special cases like
//getbitr(BO, 2) && getbitr(BO, 4)

void Recompiler::add_bc_ctr_cond(DWORD BO, DWORD BI) {
	add_bc_ctr(BO);
	add_bc_cond(BO, BI);
}

void Recompiler::add_bc_ctr(DWORD BO) {
	//if(!getbitr(BO, 2))
	//r.ctr--;
	//mov al, 0
	ADD_BYTE(0xB0);
	ADD_BYTE(0x00);
	//bool ctr_ok = getbitr(BO, 2) || ((r.ctr != 0) != getbitr(BO, 1));
	if(getbitr(BO, 2)) {
		//ctr_ok = true;	//or al, 0x01
		ADD_BYTE(0x0C);
		ADD_BYTE(0x01);
	} else {
		//r.ctr--;
		//dec r.ctr
		ADD_BYTE(0xFF);
		ADD_BYTE(0x0D);
		ADD_DWORD(&r.ctr);
		//Thanks to the DEC instruction, ZF is now set.
		//(getbitr(BO, 1) ? jnz : jz) end
		ADD_BYTE(getbitr(BO, 1) ? 0x75 : 0x74);
		request_internal_label("end");
		//ctr_ok = true;	//or al, 0x01
		ADD_BYTE(0x0C);
		ADD_BYTE(0x01);
	}
	add_internal_label("end");
	clear_internal_labels();
}

void Recompiler::add_bc_cond(DWORD BO, DWORD BI) {
	//bool cond_ok = getbitr(BO, 4) || (getbit(r.cr, BI) == getbitr(BO, 3));
	if(getbitr(BO, 4)) {
		//cond_ok = true;	//or al, 0x02
		ADD_BYTE(0x0C);
		ADD_BYTE(0x02);
	} else {
		//bt r.cr, BI
		ADD_BYTE(0x0F);
		ADD_BYTE(0xBA);
		ADD_BYTE(0x25);
		ADD_DWORD(&r.cr);
		ADD_BYTE(31 - (BYTE)BI);
		//(getbitr(BO, 3) ? jnc : jc) end
		ADD_BYTE(getbitr(BO, 3) ? 0x73 : 0x72);
		request_internal_label("end");
		//cond_ok = true;	//or al, 0x02
		ADD_BYTE(0x0C);
		ADD_BYTE(0x02);
	}
	add_internal_label("end");
	clear_internal_labels();
}

void Recompiler::add_mem1_rA0d(DWORD rA, short d, bool write) {
	if(rA == 0) {
		if(d == 0) {
			//xor eax, eax
			AB(0x31);
			AB(0xC0);
		} else {
			//mov eax, d
			AB(0xB8);
			AD(d);
		}
	} else {
		//mov eax, r.gpr[rA]
		AB(0xA1);
		AD(&r.gpr[rA]);
		if(d != 0) {
			//add eax, d
			AB(0x05);
			AD(d);
		}
	}
	_add_mem1_call(write);
}

void Recompiler::add_mem1_x(DWORD rA, DWORD rB, bool write) {
	if(rA == 0) {
		//mov eax, r.gpr[rB]
		ADD_BYTE(0xA1);
		ADD_DWORD(&r.gpr[rB]);
	} else {
		//mov eax, r.gpr[rA]
		ADD_BYTE(0xA1);
		ADD_DWORD(&r.gpr[rA]);
		//add eax, r.gpr[rB]
		ADD_BYTE(0x03);
		ADD_BYTE(0x05);
		ADD_DWORD(&r.gpr[rB]);
	}
	_add_mem1_call(write);
}

Recompiler::StaticFuncBlock Recompiler::createStaticBlock() {
	RecBuf recbuf(BAD_ADDRESS, BAD_ADDRESS);
	PointerKeeper keeper((void**)&cur_recbuf); //this should ensure cur_tdrb's null-ness
	cur_recbuf = &recbuf;

	size_t offsets[_NSTATICFUNCS];

	_add_mem1_base(false);
	offsets[SF_MEM1WRITE] = POS;
	_add_mem1_base(true);
	offsets[SF_STATICBRANCHHANDLER] = POS;
	_add_static_branch_handler();

#define UNUSED_DRB_STUFF(macro) \
	macro(rcbtb)\
	macro(trv)\
	macro(ppc_code)\

#define ASSERT_STUFF(name) MYASSERT(recbuf.name.size() == 0);

	UNUSED_DRB_STUFF(ASSERT_STUFF);

	StaticFuncBlock sfb(recbuf, offsets);

	if(g::verbose) {
		DEGUB("StaticBlock pos:\n");
		DEGUB("mem1read %08X | mem1write %08X | static_branch_handler %08X\n",
			sfb.getFuncPos(SF_MEM1READ), sfb.getFuncPos(SF_MEM1WRITE),
			sfb.getFuncPos(SF_STATICBRANCHHANDLER));
	}
	return sfb;
}

Recompiler::StaticFuncBlock::StaticFuncBlock(const RecBufBase &recbuf, const size_t *offs)
: DynaRecBlockBase(recbuf) {
	memcpy(offsets, offs, sizeof(offsets));
}

void Recompiler::_add_mem1_call(bool write) {
	//call (write ? mem1_base_write : mem1_base_read)
	AB(0xE8);
	REQUEST_CALL(m_staticblock.getFuncPos(write ? SF_MEM1WRITE : SF_MEM1READ));
	//jc no_hardware
	AB(0x72);
	request_label("no_hardware");
	if(g::advanced_mode) {
		//cmp eax, -1
		AB(0x83);
		AB(0xF8);
		AB(-1);
		//jne no_page_fault
		AB(0x75);
		request_internal_label("no_page_fault");
		{ //page_fault:
			//mov eax, eia2pos(cia)
			AB(0xB8);
			AD(eia2pos(cia));
			//return
			AB(0xC3);
		}
		add_internal_label("no_page_fault");
	}
	{ //yes_hardware:
		//push edi
		AB(0x57);
	}
	clear_internal_labels();
}

//on return:
//if CF is set, no_hardware.
//else yes_hardware.
//Errors return immediately :D
void Recompiler::_add_mem1_base(bool write) {
	MYASSERT(!(g::advanced_mode && g::cache_enabled));
	if(g::cache_enabled) {  //test for cache hit
		//cmp eax, CACHE_BASE
		AB(0x3D);
		AD(CACHE_BASE);
		//jb cache_miss
		AB(0x72);
		request_internal_label("cache_miss");
		//cmp eax, CACHE_BASE + CACHE_SIZE
		AB(0x3D);
		AD(CACHE_BASE + CACHE_SIZE);
		//jnb cache_miss
		AB(0x73);
		request_internal_label("cache_miss");
		{ //cache_hit:
			//add eax, cache_memory - CACHE_BASE
			AB(0x05);
			AD(cache_memory - CACHE_BASE);
			//jmp no_hardware
			//stc
			AB(0xF9);
			//return
			AB(0xC3);
		}
		add_internal_label("cache_miss");
	} else if(g::advanced_mode) {
		//eax = recmem.translate_d(eax);
		//push edi
		AB(0x57);
		//push eax
		AB(0x50);
		//mov ecx, &recmem //the this pointer
		AB(0xB9);
		AD(&recmem);
		//call translate_d  //thiscall assumed
		AB(0xE8);
		DWORD (RecMemory::*temp)(DWORD) = write ?
			&RecMemory::translate_d_write_rec : &RecMemory::translate_d_read_rec;
		REQUEST_CALL(MAKE(DWORD, temp));
		//pop edi
		AB(0x5F);
		//cmp eax, -1
		AB(0x83);
		AB(0xF8);
		AB(-1);
		//jne no_page_fault
		AB(0x75);
		request_internal_label("no_page_fault");
		{ //page_fault:
			//clc
			AB(0xF8);
			//return
			AB(0xC3);
		}
		add_internal_label("no_page_fault");
	}
	//cmp eax, (0xE0000000)
	AB(0x3D);
	AD(memory + 0xE0000000);
	//jb not_locked_cache  //eax < 0xE0000000
	AB(0x72);
	request_internal_label("not_locked_cache");
	//cmp eax, (memory + 0xE0004000)
	AB(0x3D);
	AD(memory + 0xE0004000); 
	//jae not_locked_cache  //!(0xE0000000 <= eax < 0xE0004000)
	AB(0x73);
	request_internal_label("not_locked_cache");
	//locked cache access, unemulated for now
	/*AB(0x58); //pop eax
	add_fatal_error(REC_INVALID_MEMORY_ADDRESS);*/
	//jmp error
	AB(0xEB);
	request_internal_label("error");

	add_internal_label("not_locked_cache");
	//and eax, ~0xC0000000  //0xC0000000 == (0xC0000000 | 0x80000000)
	ADD_BYTE(0x25);
	ADD_DWORD(~0xC0000000);

	//add eax, memory
	ADD_BYTE(0x05);
	ADD_DWORD(memory);

	//cmp eax, (memory + MAIN_MEMORY_SIZE)
	AB(0x3D);
	AD(memory + MAIN_MEMORY_SIZE);
	//jae not_main_memory
	AB(0x73);
	request_internal_label("not_main_memory");
	//stc	//not needed, since jae == jnc
	//return
	AB(0xC3);

	add_internal_label("not_main_memory");
	//cmp eax, (memory + 0x0C000000)
	AB(0x3D);
	AD(memory + 0x0C000000);
	/*//jb between  //MAIN_MEMORY_SIZE <= eax < 0x0C000000
	AB(0x72);
	request_internal_label("between");*/
	//jb error
	AB(0x72);
	request_internal_label("error");

	//cmp eax, (memory + 0x0C008000)
	AB(0x3D);
	AD(memory + 0x0C008000); 
	//jbe yes_hardware  //0x0C000000 <= eax <= 0x0C008000
	AB(0x76);
	request_internal_label("yes_hardware");
	{
		add_internal_label("error");
		AB(0x58); //pop eax
		add_fatal_error(REC_INVALID_MEMORY_ADDRESS);
	}
	/*{
	add_internal_label("between");
	//cmp eax, (memory + 0x08000000)
	AB(0x3D);
	AD(memory + 0x08000000);
	//jb error  //MAIN_MEMORY_SIZE <= eax < 0x08000000
	AB(0x72);
	request_internal_label("error");
	//cmp eax, (memory + 0x08200000)
	AB(0x3D);
	AD(memory + 0x08200000); 
	//jae error  //!(0x08000000 <= eax < 0x08200000)
	AB(0x73);
	request_internal_label("error");
	//efb access, unemulated for now
	AB(0x58); //pop eax
	add_fatal_error(REC_INVALID_MEMORY_ADDRESS);
	}*/
	{
		add_internal_label("yes_hardware");
		//translate back to Hardware memory offset
		//add eax, (0x0C000000 - memory)
		AB(0x05);
		AD(0x0C000000 - (DWORD)memory);
		//clc
		AB(0xF8);
		//return
		AB(0xC3);
	}
	clear_internal_labels();
}

void Recompiler::add_mem2() {
	//pop edi
	AB(0x5F);
	//jmp end
	AB(0xEB);
	request_label("end");
}

void Recompiler::add_pos_fatal_error(BYTE code) {
	//mov eax, eia2pos(cia)
	AB(0xB8);
	AD(eia2pos(cia));
	add_fatal_error(code);
}

//8 bytes
void Recompiler::add_fatal_error(BYTE code) {
	//mov errorcode, code
	AB(0xC6);
	AB(0x05);
	AD(&errorcode);
	AB(code);
	//return
	AB(0xC3);
}

//0 <= bit < 32
void Recompiler::add_setflag(DWORD *target, int bit, BYTE jcc) {
	if(jcc < 0x70 || jcc > 0x7F)
		throw rec_fatal_exception("Invalid jcc in Recompiler::add_setflag()");
	if(bit < 0 || bit > 32)
		throw rec_fatal_exception("Invalid bit in Recompiler::add_setflag()");

	//setflags(*target, makeflag(bit), NOT_ZERO?)
	//jcc set
	AB(jcc);
	request_internal_label("set");
	{ //clear:
		//(31 - bit) / 8 works thanks to integer division
		//and byte ptr (target + (31 - bit) / 8), ~(0x80 >> (bit % 8)))
		AB(0x80);
		AB(0x25);
		AD(DWORD(target) + (31 - bit) / 8);
		AB(~(0x80 >> (bit % 8)));
		//jmp end
		AB(0xEB);
		request_internal_label("end");
	}
	{
		add_internal_label("set");
		//or byte ptr (target + (31 - bit) / 8), (0x80 >> (bit % 8))
		AB(0x80);
		AB(0x0D);
		AD(DWORD(target) + (31 - bit) / 8);
		AB(0x80 >> (bit % 8));
	}
	add_internal_label("end");
	clear_internal_labels();
}

void Recompiler::add_fxx_ps_d(BYTE op1s, BYTE digit, DWORD fr, bool disp8) {
	MYASSERT(digit < 8);
	MYASSERT(fr < 32);
	MYASSERT(op1s == 0xD8 || op1s == 0xD9);
	MYASSERT(!(op1s == 0xD9 && (digit == 2 || digit == 3)));  //no stores!

	//fxx PS0(fr)
	AB(op1s | 0x04);
	AB((disp8 ? 0x43 : 0x05) | (digit << 3));
	if(disp8) {
		ADD_DISP8(&PS0(fr));
	} else {
		AD(&PS0(fr));
	}
}

void Recompiler::add_fstx_d(bool pop, DWORD fr, bool disp8) {
	//r.fpr[fr].d = ST(0);
	MYASSERT(fr < 32);

	//fst/fstp r.fpr[fr].d
	AB(0xDD);
	AB((disp8 ? 0x43 : 0x05) | ((pop ? 3 : 2) << 3));
	if(disp8) {
		ADD_DISP8(&r.fpr[fr].d);
	} else {
		AD(&r.fpr[fr].d);
	}
}

void Recompiler::add_set_fpr_single(DWORD fr, bool disp8) {
	//fst PS1(fr)	//can't disp8 it, out of range.
	AB(0xDD);
	AB(0x15);
	AD(&PS1(fr));
	//fstp PS0(fr)
	AB(0xDD);
	AB(disp8 ? 0x5B : 0x1D);
	if(disp8) {
		ADD_DISP8(&PS0(fr));
	} else {
		AD(&PS0(fr));
	}
}

void Recompiler::add_static_branch(DWORD target_address, bool LK) {
	if(LK) {
		//r.lr = cia + 4;
		//mov r.lr, cia + 4;
		ADD_BYTE(0xC7);
		ADD_BYTE(0x05);
		ADD_DWORD(&r.lr);
		ADD_DWORD(cia + 4);
	}
	if(g::rec_onestep) {
		//mov eax, target_address
		ADD_BYTE(0xB8);
		ADD_DWORD(target_address);

		add_fatal_error(REC_ONESTEP_STATIC_BRANCH);
	} else {
		//nia = target_address;
		//mov eax, eia2pos(target_address)
		ADD_BYTE(0xB8);
		ADD_DWORD(eia2pos(target_address));

		if(target_address != cia) {
			//jmp eax
			ADD_BYTE(0xFF);
			ADD_BYTE(0xE0);
		} else {
			if(g::iloop)
				add_fatal_error(REC_INFINITE_LOOP);
			else {
				//An infinite loop is usually waiting for an interrupt, hw or dec.
				//Having it return here is good if we're in Realtime mode,
				//but in exact mode we should have it loop a little more...
				//As long as !g::rec_onestep, of course.
				//We should do it by setting the check bit in the cycle counter (edi, isn't it?).
				//That would indicate that BLOCK_SIZE_EXPONENT^2 cycles have been run here,
				//which should be fast and good.
				//Not so much. gc-linux is actually faster with this off.
				if(g::timing_mode != g::TM_REALTIME) {
					//bts edi, BLOCK_SIZE_EXPONENT
					AB(0x0F);
					AB(0xBA);
					AB(0xEF);
					AB(BLOCK_SIZE_EXPONENT);
				}
				//return
				AB(0xC3);
			}
		}
	}
}

void Recompiler::_add_static_branch_handler() {
	//pop eax
	AB(0x58);
	add_fatal_error(REC_STATIC_BRANCH);
}
