// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "registers.h"
#include "ini.h"
#include "meminterface.h"
#include <crtdbg.h>
#include <float.h>
#include <sstream>

#define USER_REG_INIT(reg) ,u##reg(reg)

RegistersBase::RegistersBase() : pvr(0x00083214)//pvr(0x70000100)
USER_REG(USER_REG_INIT) {
#define ZERO_REG_ARRAY(rega, size) memset(rega, 0, sizeof(rega));
#define ZERO_REG(reg) reg = 0;

	REG_ARRAY(ZERO_REG_ARRAY);
	REG(ZERO_REG);

	ZeroMemory(gpr, sizeof(gpr));
	ZeroMemory(fpr, sizeof(fpr));
	ZeroMemory(ps1, sizeof(ps1));

	hid2.word = 0;
	sda = 0;
	msr = 0x00000040;
	dec = 0xFFFFFFFF;

	DWORD i;
#define SET_SPR_F(spr, reg) { mfspr[spr] = &reg; sprasm[spr] = #reg; }
#define SET_SPR_T(spr, reg) { mtspr[spr] = &reg; sprasm[spr] = #reg; }
#define SET_SPR_BOTH(spr, reg) { mfspr[spr] = &reg; mtspr[spr] = &reg;\
	sprasm[spr] = #reg; }
#define SET_SPRASM_I(spr, reg, add, i) { str.str(""); str << #reg << add << i;\
	sprasm[spr] = str.str(); }
#define SET_SPRASM_2(spr, reg, add, i) { str.str(""); str << #reg << i << add;\
	sprasm[spr] = str.str(); }
#define SET_PSPR_ARRAY_BOTH(spr, reg, size) for(i=0; i<size; i++) {\
	mfspr[spr+i] = &reg[i]; mtspr[spr+i] = &reg[i]; SET_SPRASM_I(spr+i, reg, "", i); }
#define SET_PSPR_INTERLEAVED_ARRAY_BOTH(spr, reg, size) for(i=0; i<size*2; i++) {\
	mfspr[spr+i] = &reg##u[i/2]; mtspr[spr+i] = &reg##u[i/2];\
	SET_SPRASM_2(spr+i, reg, "u", i/2);\
	i++; mfspr[spr+i] = &reg##l[i/2]; mtspr[spr+i] = &reg##l[i/2];\
	SET_SPRASM_2(spr+i, reg, "l", i/2); }

	memset(mfspr, 0, sizeof(mfspr));
	memset(mtspr, 0, sizeof(mtspr));
	ostringstream str;
	//PowerPC SPRs
	SET_SPR_BOTH(9, ctr);
	SET_SPR_BOTH(1013, dabr);
	SET_SPR_BOTH(19, dar);
	SET_PSPR_INTERLEAVED_ARRAY_BOTH(536, dbat, 4);
	SET_SPR_BOTH(22, dec);
	SET_SPR_BOTH(18, dsisr);
	SET_SPR_BOTH(282, ear);
	SET_PSPR_INTERLEAVED_ARRAY_BOTH(528, ibat, 4);
	SET_SPR_BOTH(8, lr);
	SET_SPR_F(287, pvr);
	SET_SPR_BOTH(25, sdr1);
	SET_PSPR_ARRAY_BOTH(272, sprg, 4);
	SET_SPR_BOTH(26, srr0);
	SET_SPR_BOTH(27, srr1);
	SET_SPR_F(268, tbl);
	SET_SPR_T(284, tbl);
	SET_SPR_F(269, tbu);
	SET_SPR_T(285, tbu);
	SET_SPR_BOTH(1, xer);

	//Gekko SPRs
	SET_SPR_BOTH(922, dmau);
	SET_SPR_BOTH(923, dmal);
	SET_PSPR_ARRAY_BOTH(912, gqr, 8);
	SET_SPR_BOTH(1008, hid0);
	SET_SPR_BOTH(1009, hid1);
	mfspr[920] = &hid2.word; mtspr[920] = &hid2.word; sprasm[920] = "hid2";
	SET_SPR_BOTH(1010, iabr);
	SET_SPR_BOTH(1019, ictc);
	SET_SPR_BOTH(1017, l2cr);
	SET_SPR_BOTH(952, mmcr0);
	SET_SPR_BOTH(956, mmcr1);
	SET_SPR_BOTH(953, pmc1);
	SET_SPR_BOTH(954, pmc2);
	SET_SPR_BOTH(957, pmc3);
	SET_SPR_BOTH(958, pmc4);
	SET_SPR_BOTH(955, sia);
	SET_SPR_BOTH(1020, thrm1);
	SET_SPR_BOTH(1021, thrm2);
	SET_SPR_BOTH(1022, thrm3);
	SET_SPR_F(936, ummcr0);
	SET_SPR_F(940, ummcr1);
	SET_SPR_F(937, upmc1);
	SET_SPR_F(938, upmc2);
	SET_SPR_F(941, upmc3);
	SET_SPR_F(942, upmc4);
	SET_SPR_F(939, usia);
	SET_SPR_BOTH(921, wpar);

	//Broadway-specific SPRs
	if(g::wii) {
		SET_SPR_BOTH(1011, hid4);
	}

	//Unimplemented SPRs
	SET_SPR_BOTH(959, sda);
	SET_SPR_F(943, usda);

	//FPSCR flags
#define FPSCR_FLAGS(macro) \
	macro(FX, 0)\
	macro(FEX, 1)\
	macro(VX, 2)\
	macro(OX, 3)\
	macro(UX, 4)\
	macro(ZX, 5)\
	macro(XX, 6)\
	macro(VXSNAN, 7)\
	macro(VXISI, 8)\
	macro(VXIDI, 9)\
	macro(VXZDZ, 10)\
	macro(VXIMZ, 11)\
	macro(VXVC, 12)\
	macro(FR, 13)\
	macro(FI, 14)\
	macro(C, 15)\
	macro(FL, 16)\
	macro(FG, 17)\
	macro(FE, 18)\
	macro(FU, 19)\
	macro(VXSOFT, 21)\
	macro(VXSQRT, 22)\
	macro(VXCVI, 23)\
	macro(VE, 24)\
	macro(OE, 25)\
	macro(UE, 26)\
	macro(ZE, 27)\
	macro(XE, 28)\
	macro(NI, 29)\
	macro(RN0, 30)\
	macro(RN1, 31)

	memset(fpscrasm, 0, sizeof(fpscrasm));

#define SET_FPSCR_FLAG_ASM(asm, bit) fpscrasm[bit] = #asm;

	FPSCR_FLAGS(SET_FPSCR_FLAG_ASM);
}

bool RegistersBase::degub(bool all, RegistersBase &old) const {
	bool change = false;

	//DC == Degub Change
#define DC_REG_ARRAY(reg, size) for(int i=0; i<size; i++) {\
	if(all || reg[i] != old.reg[i]) { old.reg[i] = reg[i]; change = true;\
	DEGUB("%s%i=%08X\n", #reg, i, reg[i]); } }
#define DC_REG(reg) if(all || reg != old.reg) { change = true;\
	DEGUB("%s=%08X\n", #reg, reg); old.reg = reg; }

	if(all) {
		DC_REG_ARRAY(gpr, 32);
	}

	REG_ARRAY(DC_REG_ARRAY);
	REG(DC_REG);
	if(all || hid2.word != old.hid2.word) { change = true;
	DEGUB("%s=%08X\n", "hid2", hid2.word); old.hid2.word = hid2.word; }

	if(!all) {
		DC_REG_ARRAY(gpr, 32);
	}

	for(int i=0; i<32; i++) {
		if(all || fpr[i].dword != old.fpr[i].dword ||
			MAKE(QWORD, ps1[i]) != MAKE(QWORD, old.ps1[i]))
		{
			old.fpr[i] = fpr[i];
			old.ps1[i] = ps1[i];
			change = true;
			DEGUB("fpr%i=%016I64X\t", i, fpr[i].dword);
			DEGUB("%.10g %.10g\n", fpr[i].d, ps1[i]);
		}
	}

	if(all || msr != old.msr) {
		old.msr = msr;
		change = true;
		DEGUB("msr=%08X\n", msr);
	}
	if(all || pvr != old.pvr) { //A little strange, considering that pvr is const
		//old.pvr = pvr;  //Yeah.
		change = true;
		DEGUB("pvr=%08X\n", pvr);
	}

	return change;
}

void RegistersBase::degub_all() const {
	//DA == Degub All
#define DA_REG_ARRAY(reg, size) for(int i=0; i<size; i++) {\
	DEGUB("%s%i=%08X\n", #reg, i, reg[i]); }
#define DA_REG(reg) { DEGUB("%s=%08X\n", #reg, reg); }

	DA_REG_ARRAY(gpr, 32);

	REG_ARRAY(DA_REG_ARRAY);
	REG(DA_REG);
	DEGUB("%s=%08X\n", "hid2", hid2.word);

	for(int i=0; i<32; i++) {
		DEGUB("fpr%i=%016I64X\t", i, fpr[i].dword);
		DEGUB("%.10g %.10g\n", fpr[i].d, ps1[i]);
	}

	DEGUB("pvr=%08X\n", pvr);
}

void RegistersBase::degub_tb_dec(RegistersBase &old) const {
	old.tbu = tbu;
	old.tbl = tbl;
	old.dec = dec;
}

bool RegistersBase::compare_and_degub(const RegistersBase &other) const {
	bool change = false;

	//CD == Compare with Degub
#define CD_REG_ARRAY(reg, size) for(int i=0; i<size; i++) {\
	if(reg[i] != other.reg[i]) { change = true;\
	DEGUB("this->%s%i=%08X\nthat->%s%i=%08X\n", #reg, i, reg[i], #reg, i, other.reg[i]); } }
#define CD_REG(reg) if(reg != other.reg) { change = true;\
	DEGUB("this->%s=%08X\nthat->%s=%08X\n", #reg, reg, #reg, other.reg); }

	REG_ARRAY(CD_REG_ARRAY);
	REG(CD_REG);
	if(hid2.word != other.hid2.word) { change = true;
	DEGUB("this->%s=%08X\nthat->%s=%08X\n", "hid2", hid2.word, "hid2", other.hid2.word); }

	//Custom registers
	CD_REG_ARRAY(gpr, 32);

	for(int i=0; i<32; i++) {
		if(fpr[i].dword != other.fpr[i].dword ||
			MAKE(QWORD, ps1[i]) != MAKE(QWORD, other.ps1[i]))
		{
			change = true;
			DEGUB("this->fpr%i=%016I64X\t", i, fpr[i].dword);
			DEGUB("%.10g %.10g\n", fpr[i].d, ps1[i]);
			DEGUB("that->fpr%i=%016I64X\t", i, other.fpr[i].dword);
			DEGUB("%.10g %.10g\n", other.fpr[i].d, other.ps1[i]);
		}
	}

	if(pvr != other.pvr) { //A little strange, considering that pvr is const
		change = true;
		DEGUB("DUDE!!! pvr=%08X\n", pvr);
	}
	return change;
}

void RegistersBase::copy(const RegistersBase &other) {
#define COPY_REG_ARRAY(reg, size) for(int i=0; i<size; i++) {\
	reg[i] = other.reg[i]; }
#define COPY_REG(reg) { reg = other.reg; }

	REG_ARRAY(COPY_REG_ARRAY);
	REG(COPY_REG);
	hid2.word = other.hid2.word;

	COPY_REG_ARRAY(gpr, 32);

	for(int i=0; i<32; i++) {
		fpr[i] = other.fpr[i];
		ps1[i] = other.ps1[i];
	}

	msr = other.msr;
}


const DWORD *Registers::getmfspr(DWORD spr) const {
	mfspr_is_valid(spr);
	return mfspr[spr];
}
DWORD *Registers::getmtspr(DWORD spr) const {
	mtspr_is_valid(spr);
	return mtspr[spr];
}

bool RegistersBase::mfspr_is_valid(DWORD spr) const {
	MYASSERT(spr < 1024);
	if(mfspr[spr] == NULL) {
		//DEGUB("Invalid mfSPR: %i\n", spr);
		return false;
	} else
		return true;
}
bool RegistersBase::mtspr_is_valid(DWORD spr) const {
	MYASSERT(spr < 1024);
	if(mtspr[spr] == NULL) {
		//DEGUB("Invalid mtSPR: %i\n", spr);
		return false;
	} else
		return true;
}

const string& RegistersBase::getsprasm(DWORD spr) const {
	MYASSERT(spr < 1024);
	return sprasm[spr];
}

const char *RegistersBase::getfpscrasm(DWORD bit) const {
	MYASSERT(bit < 32);
	return fpscrasm[bit];
}

void RegistersBase::setcr(DWORD n, DWORD c) {
	MYASSERT(n < 8);
	setbits(cr, n*4, n*4+3, c);
}

void RegistersBase::setcr0(DWORD result) {
	setflags(cr, CR0_LT, signw(result));
	setflags(cr, CR0_GT, !signw(result) && result != 0);
	setflags(cr, CR0_EQ, result == 0);
	setflags(cr, CR0_SO, getflag(xer, XER_SO));
}

void RegistersBase::set_overflow(bool ov) {
	if(ov)
		setflags(xer, XER_OV | XER_SO, true);
	else
		setflags(xer, XER_OV, false);
}

void RegistersBase::update_fpscr_fex_vx() {
	//Must be done in this order because FEX depends on VX
	setflags(fpscr, FPSCR_VX, getflag(fpscr, FPSCR_VXSNAN | FPSCR_VXISI | FPSCR_VXIDI |
		FPSCR_VXZDZ | FPSCR_VXIMZ | FPSCR_VXVC | FPSCR_VXSOFT | FPSCR_VXSQRT | FPSCR_VXCVI));

	setflags(fpscr, FPSCR_FEX,
		(getflag(fpscr, FPSCR_VX) && getflag(fpscr, FPSCR_VE)) ||
		(getflag(fpscr, FPSCR_OX) && getflag(fpscr, FPSCR_OE)) ||
		(getflag(fpscr, FPSCR_UX) && getflag(fpscr, FPSCR_UE)) ||
		(getflag(fpscr, FPSCR_ZX) && getflag(fpscr, FPSCR_ZE)) ||
		(getflag(fpscr, FPSCR_XX) && getflag(fpscr, FPSCR_XE)));
}

//Hmm. This might actually work. Thanks god(?) for the fp support functions.
void RegistersBase::set_fpscr_fprf(double result) {
	//setflags(fpscr, FPSCR_C | FPSCR_FL | FPSCR_FG | FPSCR_FE | FPSCR_FU, false);
	fpscr &= ~0x0001F000;

	switch(_fpclass(result)) {
		//case _FPCLASS_SNAN:
	case _FPCLASS_QNAN:
		setflags(fpscr, FPSCR_C | FPSCR_FU, true);
		break;
	case _FPCLASS_NINF:
		setflags(fpscr, FPSCR_FL | FPSCR_FU, true);
		break;
	case _FPCLASS_NN:
		setflags(fpscr, FPSCR_FL, true);
		break;
	case _FPCLASS_ND:
		setflags(fpscr, FPSCR_FL | FPSCR_C, true);
		break;
	case _FPCLASS_NZ:
		setflags(fpscr, FPSCR_C | FPSCR_FE, true);
		break;
	case _FPCLASS_PZ:
		setflags(fpscr, FPSCR_FE, true);
		break;
	case _FPCLASS_PD:
		setflags(fpscr, FPSCR_FG | FPSCR_C, true);
		break;
	case _FPCLASS_PN:
		setflags(fpscr, FPSCR_FG, true);
		break;
	case _FPCLASS_PINF:
		setflags(fpscr, FPSCR_FG | FPSCR_FU, true);
		break;
	default:
		throw interp_fatal_exception("Unknown floating-point class!");
	}
}

void Registers::setmsr(DWORD data) {
#define UNSUPPORTED_BIT(bit) if(data & MSR_##bit)\
	throw generic_fatal_exception("MSR[" #bit "] unemulated!")

	if(data & (MSR_ILE | MSR_LE | MSR_PM | MSR_SE | MSR_BE | MSR_IP)) {
		//UNSUPPORTED_BIT(PR);
		UNSUPPORTED_BIT(ILE);
		UNSUPPORTED_BIT(LE);
		//UNSUPPORTED_BIT(POW);
		//UNSUPPORTED_BIT(PM);
		UNSUPPORTED_BIT(SE);
		UNSUPPORTED_BIT(BE);
		UNSUPPORTED_BIT(IP);
	}

	if(g::log_interrupts) {// && MASKED_NEQUAL(msr, data, ~(MSR_EE))) {
		//DEGUB("MSR <- 0x%08X\n", data);
	}
	msr = data;
	m.msr_notify();
}
