// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef REGISTERS_H
#define REGISTERS_H

#include "common.h"

class MemInterface;

union FPR {
	QWORD dword;
	double d;
	struct { DWORD loword, hiword; }; //reversed word order
};
union HID2 {
	DWORD word;
	struct {
DWORD : 16;
		DWORD dqoee : 1;
		DWORD dcmee : 1;
		DWORD dchee : 1;
		DWORD dncee : 1;
		DWORD dcherr : 1;
		DWORD dncerr : 1;
		DWORD dcmerr : 1;
		DWORD dqoerr : 1;
		DWORD dmaql : 4;
		DWORD lce : 1;
		DWORD pse : 1;
		DWORD wpe : 1;
		DWORD lsqe : 1;
	};
};

class RegistersBase {
public:
	RegistersBase();

	//Registers &operator=(const Registers &other);
	bool degub(bool all, RegistersBase &old) const;
	void degub_tb_dec(RegistersBase &old) const;
	bool compare_and_degub(const RegistersBase &other) const;
	void degub_all() const;
	void copy(const RegistersBase &other);

	//Flagzorz
#define CR_LT		0x8
#define CR_GT		0x4
#define CR_EQ		0x2
#define CR_SO		0x1
#define CR0_LT		0x80000000
#define CR0_GT		0x40000000
#define CR0_EQ		0x20000000
#define CR0_SO		0x10000000

#define MSR_LE		0x00000001
#define MSR_RI		0x00000002
#define MSR_PM		0x00000004
#define MSR_DR		0x00000010
#define MSR_IR		0x00000020
#define MSR_IP		0x00000040
#define MSR_FE1		0x00000100
#define MSR_BE		0x00000200
#define MSR_SE		0x00000400
#define MSR_FE0		0x00000800
#define MSR_ME		0x00001000
#define MSR_FP		0x00002000
#define MSR_PR		0x00004000
#define MSR_EE		0x00008000
#define MSR_ILE		0x00010000
#define MSR_POW		0x00040000

#define XER_SO		0x80000000
#define XER_OV		0x40000000
#define XER_CA		0x20000000

#define HID0_NOOPTI	0x00000001

#define FPSCR_FX	0x80000000
#define FPSCR_FEX	0x40000000
#define FPSCR_VX	0x20000000
#define FPSCR_OX	0x10000000
#define FPSCR_UX	0x08000000
#define FPSCR_ZX	0x04000000
#define FPSCR_XX	0x02000000
#define FPSCR_VXSNAN	0x01000000
#define FPSCR_VXISI	0x00800000
#define FPSCR_VXIDI	0x00400000
#define FPSCR_VXZDZ	0x00200000
#define FPSCR_VXIMZ	0x00100000
#define FPSCR_VXVC	0x00080000
#define FPSCR_FR	0x00040000
#define FPSCR_FI	0x00020000
#define FPSCR_C		0x00010000
#define FPSCR_FL	0x00008000
#define FPSCR_FG	0x00004000
#define FPSCR_FE	0x00002000
#define FPSCR_FU	0x00001000
	//bit 20 is reserved
#define FPSCR_VXSOFT	0x00000400
#define FPSCR_VXSQRT	0x00000200
#define FPSCR_VXCVI	0x00000100
#define FPSCR_VE	0x00000080
#define FPSCR_OE	0x00000040
#define FPSCR_UE	0x00000020
#define FPSCR_ZE	0x00000010
#define FPSCR_XE	0x00000008
#define FPSCR_NI	0x00000004
#define FPSCR_RN0	0x00000002
#define FPSCR_RN1	0x00000001

#define REG_ARRAY(macro) \
	macro(ibatu, 4)\
	macro(ibatl, 4)\
	macro(dbatu, 4)\
	macro(dbatl, 4)\
	macro(sr, 16)\
	macro(sprg, 4)\
	macro(gqr, 8)

#define REG(macro) \
	macro(cr) macro(fpscr)\
	macro(xer) macro(lr) macro(ctr)\
	macro(sdr1)\
	macro(dar)\
	macro(dsisr)\
	macro(srr0) macro(srr1)\
	macro(tbu) macro(tbl)\
	macro(dec)\
	macro(dabr)\
	macro(ear)\
	macro(iabr)\
	macro(hid0) macro(hid1) macro(hid4)\
	macro(dmau) macro(dmal)\
	macro(wpar)\
	macro(l2cr)\
	macro(pmc1) macro(pmc2) macro(pmc3) macro(pmc4)\
	macro(mmcr0) macro(mmcr1)\
	macro(sia)\
	macro(ictc)\
	macro(thrm1) macro(thrm2) macro(thrm3)\

#define DECLARE_REG(reg) DWORD reg;
#define DECLARE_REG_ARRAY(reg, size) DWORD reg[size];

	REG_ARRAY(DECLARE_REG_ARRAY);
	REG(DECLARE_REG);

	//Custom registers
	DWORD gpr[32];
	const DWORD pvr;
	DWORD getmsr() const { return msr; }

	FPR fpr[32];
	double ps1[32];

	HID2 hid2;

	//Unimplemented registers
	DWORD sda;

	//User-level read access only
#define USER_REG(macro) \
	macro(pmc1) macro(pmc2) macro(pmc3) macro(pmc4)\
	macro(mmcr0) macro(mmcr1)\
	macro(sia)\
	macro(sda)

#define DECLARE_USER_REG(reg) const DWORD &u##reg;

	USER_REG(DECLARE_USER_REG);

	bool mfspr_is_valid(DWORD spr) const;
	bool mtspr_is_valid(DWORD spr) const;
	const string& getsprasm(DWORD spr) const;
	const char *getfpscrasm(DWORD bit) const;

	//Opcode helpers
	void setcr(DWORD n, DWORD c);
	void setcr0(DWORD result);
	void set_overflow(bool ov);
	void update_fpscr_fex_vx();
	void set_fpscr_fprf(double result);
protected:
	RegistersBase& operator=(const RegistersBase&);

	const DWORD *mfspr[1024];
	DWORD *mtspr[1024];
	string sprasm[1024];
	const char *fpscrasm[32];

	DWORD msr;
};

class Registers : public RegistersBase {
public:
	Registers(MemInterface &_m) : m(_m) {}

	void setmsr(DWORD data);
	const DWORD *getmfspr(DWORD spr) const;
	DWORD *getmtspr(DWORD spr) const;
private:
	Registers& operator=(const Registers&);
	MemInterface &m;
};

#endif	//REGISTERS_H
