/*
    Space Invaders Emulator
    Copyright (c) 1997-2002,2003 Alessandro Scotti
    http://www.walkofmind.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <ddraw.h>
#include <dsound.h>
#include <stdio.h>

#include "arcade.h"
#include "dib24.h"
#include "resource.h"

static char     szAppName[] = "Space Invaders";

static char     szHelp[] =
    "1 = one player\n"
    "2 = two players\n"
    "3 = insert coin\n"
    "Space = fire\n"
    "Left = move left\n"
    "Right = move right\n"
    "Escape = quit";

static char     szAbout[] = 
    "Space Invaders Didactic Emulator\n"
    "Copyright (c) 1997-2003 Alessandro Scotti\n"
    "\n"
    "http://www.walkofmind.com\n"
    "\n"
    "This program is free software; you can redistribute it and/or modify\n"
    "it under the terms of the GNU General Public License as published by\n"
    "the Free Software Foundation; either version 2 of the License, or\n"
    "(at your option) any later version.\n"
    "\n"
    "This program is distributed in the hope that it will be useful,\n"
    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
    "GNU General Public License for more details.\n"
    "\n"
    "You should have received a copy of the GNU General Public License\n"
    "along with this program; if not, write to the Free Software\n"
    "Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n"
    ;

static int      nShips  = 3;
static int      nEasy   = 0;

static char *   szSound[9] = {
    "ufo", "walk1", "walk2", "walk3", "walk4", "shot", "ufohit", "basehit", "invhit"
};

HWND     hWindow = NULL;         // Handle of main window
Dib24 *  pCanvas = NULL;
BOOL     boIsActive = FALSE;
DWORD    scanLineColor[InvadersMachine::ScreenHeight];

InvadersMachine     arcade;         // Arcade emulator

LPDIRECTSOUND       directSound;    // Main direct sound object
IDirectSoundBuffer *lpSound[9];     // Sound buffers for game sounds

/*
    Creates a (DirectX) sound buffer from a wave file.
*/
IDirectSoundBuffer * loadSoundBuffer( IDirectSound * pds, char * filename )
{
    DSBUFFERDESC    bufdesc = { 0 };

    BYTE *          data_;

    bufdesc.lpwfxFormat = (WAVEFORMATEX *)malloc( sizeof(WAVEFORMATEX) );
    data_ = 0;

    FILE *  f = fopen( filename, "rb" );
    if( f != NULL ) {
        DWORD   dwRiff, dwLen, dwType;

        fread( &dwRiff, 4, 1, f );
        fread( &dwLen , 4, 1, f );
        fread( &dwType, 4, 1, f );

        DWORD   dwOffset = ftell( f );

        if( (dwRiff == mmioFOURCC('R','I','F','F')) && (dwType == mmioFOURCC('W','A','V','E')) ) {
            while( 1 ) {
                if( fread( &dwType, 4, 1, f ) != 1 )
                    break;
                if( fread( &dwLen , 4, 1, f ) != 1 )
                    break;

                if( dwType == mmioFOURCC('f','m','t',' ') ) {
                    if( dwLen < sizeof(WAVEFORMAT) )
                        break;
                    DWORD n = dwLen > sizeof(WAVEFORMATEX) ? sizeof(WAVEFORMATEX) : dwLen;
                    if( fread( bufdesc.lpwfxFormat, n, 1, f ) != 1 )
                        break;
                }
                else if( dwType == mmioFOURCC('d','a','t','a') ) {
                    bufdesc.dwBufferBytes = dwLen;
                    data_ = new BYTE [ bufdesc.dwBufferBytes ];
                    if( fread( data_, 1, bufdesc.dwBufferBytes, f ) != bufdesc.dwBufferBytes ) {
                        delete [] data_;
                        data_ = 0;
                        break;
                    }
                }

                dwOffset += 8 + (dwLen+1)& ~1;
                fseek( f, dwOffset, SEEK_SET );
            }
        }
        fclose( f );
    }

    IDirectSoundBuffer *    result = 0;

    if( data_ ) {
        bufdesc.dwSize = sizeof(bufdesc);
        bufdesc.dwFlags = DSBCAPS_STATIC | DSBCAPS_CTRLDEFAULT;

        if( SUCCEEDED(pds->CreateSoundBuffer( &bufdesc, &result, NULL)) ) {
            LPVOID  p1, p2;
            DWORD   size1, size2;

            if( SUCCEEDED(result->Lock(0, bufdesc.dwBufferBytes, &p1, &size1, &p2, &size2, 0)) ) {
                CopyMemory( p1, data_, size1 );

                if( size2 != 0 ) 
                    CopyMemory( p2, data_+size1, size2 );

                result->Unlock( p1, size1, p2, size2 );
            }
            else {
                result->Release();
                result = 0;
            }
        }

        delete [] data_;
    }

    return result;
}

/*
    Displays an error message.
*/
BOOL showError( char * msg )
{
    MessageBox( 0, msg, "Error", MB_OK | MB_ICONERROR );
    return FALSE;
}

/*
    Resets the game.
*/
void restartGame()
{
    // Stop sounds
    for( int i=0; i<9; i++ ) {
        if( lpSound[i] )
            lpSound[i]->Stop();
    }

    // Reset machine
    arcade.reset( nShips, nEasy );
}

/*
    Initializes the game engine.
*/
BOOL initGame( void )
{
    CHAR    path[MAX_PATH];

    GetModuleFileName( NULL, path, sizeof(path) );

    CHAR *  psz = strrchr( path, '\\' );

    if( psz == 0 )
        return FALSE;

    // Load game ROM
    strcpy( psz+1, "invaders.rom" );

    CHAR    buffer[ 0x2000 ];
    FILE *  f = fopen( path, "rb" );

    if( f == NULL )
        return showError( "Cannot open ROM file." );
    
    int n = fread( buffer, 0x2000, 1, f );
    fclose( f );

    if( n != 1 )
        return showError( "Error reading ROM file." );

    arcade.setROM( buffer );

    // Load sounds
    if( DS_OK == DirectSoundCreate(NULL, &directSound, NULL) ) {
        directSound->SetCooperativeLevel( hWindow, DSSCL_NORMAL );

        for( int i=0; i<9; i++ ) {
            strcpy( psz+1, szSound[i] );
            strcat( psz+1, ".wav" );
            lpSound[i] = loadSoundBuffer( directSound, path );
        }
    }
    else {
        showError( "Warning: cannot create DirectSound object." );
        directSound = 0;
    }

    // Setup scanlines color
    for( int i=0; i<InvadersMachine::ScreenHeight; i++ ) {
        DWORD   c = 0xFFFFFF;

        if( i >=  32 && i <  64 ) c = 0x0000FF; // Red
        if( i >= 184 && i < 240 ) c = 0x00FF00; // Green

        scanLineColor[i] = c;
    }

    restartGame();

    return TRUE;
}

/*
    Executes and displays one frame.
*/
void CALLBACK stepGame( void )
{
    static DWORD    lasttime = 0;

    if( !boIsActive )
        return;

    arcade.step();

    // Clear background
    pCanvas->clear();

    // Draw video
    const unsigned char *   video = arcade.getVideo();

    for( int y=0; y<InvadersMachine::ScreenHeight; y++ ) {
        DWORD   c = scanLineColor[y];

        for( int x=0; x<InvadersMachine::ScreenWidth; x++ ) {
            if( *video++ ) pCanvas->setPixelFast( x, y, c );
        }
    }

    // Sleep if needed to keep frame rate constant
    DWORD   time = timeGetTime();

    unsigned    frameDuration   = 1000 / arcade.getFrameRate();

    while( (time-lasttime) < frameDuration ) {
        Sleep( 0 );
        time = timeGetTime();
    }

    lasttime = time;

    // Play sounds
    static unsigned mask[9] = {
        InvadersMachine::SoundUfo, 
        InvadersMachine::SoundWalk1,
        InvadersMachine::SoundWalk2,
        InvadersMachine::SoundWalk3,
        InvadersMachine::SoundWalk4,
        InvadersMachine::SoundShot,
        InvadersMachine::SoundUfoHit,
        InvadersMachine::SoundBaseHit, 
        InvadersMachine::SoundInvaderHit
    };

    unsigned sounds = arcade.getSounds();

    for( int i=1; i<9; i++ ) {
        if( lpSound[i] && (sounds & mask[i]) ) 
            lpSound[i]->Play( 0, 0, 0 );
    }
    if( lpSound[0] ) {
        if( sounds & InvadersMachine::SoundUfo ) 
            lpSound[0]->Play( 0, 0, DSBPLAY_LOOPING );
        else
            lpSound[0]->Stop();
    }

    // Update the window
    HDC dc = GetDC( hWindow );
    pCanvas->copy( dc, 0, 0 );
    ReleaseDC( hWindow, dc );
}

/*
    Terminates the game engine.
*/
void termGame( void )
{
    // Free sound resources
    if( directSound ) {
        for( int i=0; i<9; i++ ) {
            if( lpSound[i] ) {
                lpSound[i]->Stop();
                lpSound[i]->Release();
            }
        }
        directSound->Release();
    }
}

/*
    Window procedure.
*/
LRESULT PASCAL WndProc( HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam )
{
    PAINTSTRUCT stPS;
    HDC         hDC;

    switch( wMsg ) {
        // No need to erase the background
        case WM_ERASEBKGND:
            return TRUE;
        // Paint the window by just blitting the game video
        case WM_PAINT:
            hDC = BeginPaint( hWnd, &stPS );
            pCanvas->copy( hDC, 0, 0 );
            EndPaint( hWnd, &stPS );
            return TRUE;
        // Application is activating/deactivating
        case WM_ACTIVATEAPP:
            boIsActive = (BOOL)wParam;
            break;
        // Menu commands
        case WM_COMMAND:
            switch( LOWORD(wParam) ) {
            case IDM_NEW:
                restartGame();
                break;
            case IDM_EXIT:
                PostMessage( hWnd, WM_CLOSE, 0, 0 );
                break;
            case IDM_HELP_KEYS:
                MessageBox( 0, szHelp, "Keys", MB_OK | MB_ICONINFORMATION );
                break;
            case IDM_ABOUT:
                MessageBox( 0, szAbout, szAppName, MB_OK | MB_ICONINFORMATION );
                break;
            case IDM_3_SHIPS:
            case IDM_4_SHIPS:
            case IDM_5_SHIPS:
            case IDM_6_SHIPS:
                {
                    int n = 3 + LOWORD(wParam) - IDM_3_SHIPS;
                    if( n != nShips ) {
                        CheckMenuItem( GetMenu(hWnd), nShips-3+IDM_3_SHIPS, MF_UNCHECKED );
                        CheckMenuItem( GetMenu(hWnd), LOWORD(wParam), MF_CHECKED );
                        nShips = n;
                        restartGame();
                    }
                }
                break;
            case IDM_EASY:
            case IDM_NORMAL:
                nEasy = LOWORD(wParam) == IDM_EASY ? 1 : 0;
                CheckMenuItem( GetMenu(hWnd), nEasy ? IDM_NORMAL : IDM_EASY, MF_UNCHECKED );
                CheckMenuItem( GetMenu(hWnd), LOWORD(wParam), MF_CHECKED );
                restartGame();
                break;
            }
            break;
        // Keyboard events
        case WM_KEYDOWN:
            switch( wParam ) {
            case '1':
                arcade.fireEvent( InvadersMachine::KeyOnePlayerDown );
                break;
            case '2':
                arcade.fireEvent( InvadersMachine::KeyTwoPlayersDown );
                break;
            case '3':
                arcade.fireEvent( InvadersMachine::CoinInserted );
                break;
            case VK_LEFT: 
                arcade.fireEvent( InvadersMachine::KeyLeftDown );
                break;
            case VK_RIGHT:
                arcade.fireEvent( InvadersMachine::KeyRightDown );
                break;
            case VK_CONTROL:
            case VK_SPACE:
                arcade.fireEvent( InvadersMachine::KeyFireDown );
                break;
            case VK_ESCAPE:
                PostMessage( hWnd, WM_CLOSE, 0, 0 );
                return 0;
            }
            break;
        case WM_KEYUP:
            switch( wParam ) {
            case '1':
                arcade.fireEvent( InvadersMachine::KeyOnePlayerUp );
                break;
            case '2':
                arcade.fireEvent( InvadersMachine::KeyTwoPlayersUp );
                break;
            case VK_LEFT: 
                arcade.fireEvent( InvadersMachine::KeyLeftUp );
                break;
            case VK_RIGHT:
                arcade.fireEvent( InvadersMachine::KeyRightUp );
                break;
            case VK_CONTROL:
            case VK_SPACE:
                arcade.fireEvent( InvadersMachine::KeyFireUp );
                break;
            }
            break;
        // Destroy window
        case WM_DESTROY:
            PostQuitMessage( 0 );
            break;
    }
    return DefWindowProc( hWnd, wMsg, wParam, lParam );
}

int PASCAL WinMain( HANDLE hInstance, HANDLE hPrevInst, LPSTR lpCmdLine, int nCmdShow )
{
    MSG         stMsg;
    WNDCLASS    stWndClass;
    HWND        hWnd;

    /* Setup window class structure */
    stWndClass.style = CS_DBLCLKS;
    stWndClass.lpfnWndProc = WndProc;
    stWndClass.cbClsExtra = 0;
    stWndClass.cbWndExtra = 0;
    stWndClass.hInstance = hInstance;
    stWndClass.hIcon = LoadIcon( hInstance, MAKEINTRESOURCE(IDI_MAIN) );
    stWndClass.hCursor = LoadCursor( NULL, IDC_ARROW );
    stWndClass.hbrBackground = (HBRUSH)GetStockObject( BLACK_BRUSH );
    stWndClass.lpszMenuName = MAKEINTRESOURCE( IDM_MAIN );
    stWndClass.lpszClassName = "InvadersDemoProgram";

    /* Register window class */
    if( ! RegisterClass( &stWndClass ) )
        return 0;

    DWORD   wndStyle    = WS_VISIBLE | WS_OVERLAPPEDWINDOW;
    RECT    rc;

    rc.left = 0; 
    rc.top = 0; 
    rc.right = InvadersMachine::ScreenWidth; 
    rc.bottom = InvadersMachine::ScreenHeight;
    AdjustWindowRect( &rc, wndStyle, TRUE );

    /* Create window */
    hWnd = CreateWindowEx( 0,               // Extended style
        stWndClass.lpszClassName,           // Class
        szAppName,                          // Title
        wndStyle,                           // Style
        CW_USEDEFAULT,                      // X
        CW_USEDEFAULT,                      // Y
        rc.right-rc.left, 
        rc.bottom-rc.top,
        NULL,                               // Parent or owner window
        NULL,                               // Menu or child window id
        hInstance,                          // Instance handle
        NULL );                             // Pointer to window creation data

    if( hWnd == NULL )
        return 0;

    hWindow = hWnd;

    /* */
    pCanvas = new Dib24( InvadersMachine::ScreenWidth, InvadersMachine::ScreenHeight );

    /* Main loop */
    if( initGame() ) {
        ShowWindow( hWnd, SW_SHOWNORMAL );
        UpdateWindow( hWnd );

        boIsActive = TRUE;

        while( TRUE ) {
            if( PeekMessage( &stMsg, NULL, 0, 0, PM_NOREMOVE ) ) {
                if( !GetMessage( &stMsg, NULL, 0, 0 ) )
                    break;
                TranslateMessage( &stMsg );
                DispatchMessage( &stMsg );
            }
            else if( boIsActive ) {
                stepGame();
            }
            else {
                WaitMessage();
            }
        }

        termGame();
    }

    /* */
    delete pCanvas;

    return 0;
}
