/*
    24 bit Device Independent Bitmap class
    Copyright (c) 2002,2003 Alessandro Scotti
    http://www.walkofmind.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef DIB24_
#define DIB24_

#include <windows.h>

/**
    Device independent 24 bit bitmap.

    This class provides a simplified interface to a Windows 24-bit Device Independent
    Bitmap (DIB). Since the bitmap is kept in memory it can be manipulated directly,
    without having to pass thru GDI for the most common functions, such as <i>getPixel()</i>
    or <i>setPixel()</i>. 
    Even if only a very limited interface is provided, it is still possible to obtain a 
    handle to the bitmap, via the <i>getHandle()</i> function, and use standard GDI calls to
    perform unsupported functions.

    @author Alessandro Scotti
*/
class Dib24
{
public:
    /**
        Creates a bitmap of the specified size.

        @param  width   width of bitmap in pixels
        @param  height  height of bitmap in pixels
    */
    Dib24( int width, int height );

    /** Copy constructor. */
    Dib24( const Dib24 & );

    /** Destructor. */
    virtual ~Dib24();

    /** 
        Clears the bitmap, filling it with black.
    */
    void clear();

    /** 
        Stretch-copies the bitmap into a device context. 

        @param  hdc     handle of device context where bitmap will be copied
        @param  x       x coordinate of upper left corner of destination rectangle
        @param  y       y coordinate of upper left corner of destination rectangle
        @param  width   width of destination rectangle
        @param  height  height of destination rectangle

        @see #copy
    */
    void stretch( HDC hdc, int x, int y, int width, int height );

    /** 
        Copies the bitmap into a device context.

        The destination rectangle has the same size as the bitmap. If a different
        size is required the <i>stretch()</i> functions should be used instead.

        @param  hdc     handle of device context where bitmap will be copied
        @param  x       x coordinate of upper left corner of destination rectangle
        @param  y       y coordinate of upper left corner of destination rectangle

        @see #stretch
    */
    void copy( HDC hdc, int x, int y );

    /**
        Returns the color of the specified pixel.

        <b>Note:</b> this is a faster but <b>unsafe</b> version of <i>getPixel()</i>. 
        If there is a chance that <i>x</i> or <i>y</i> are out of range then <i>getPixel()</i> 
        should be used.

        @param  x   x coordinate of pixel
        @param  y   y coordinate of pixel

        @return color of specified pixel

        @see #getPixel    
    */
    DWORD getPixelFast( int x, int y ) const {
        LPBYTE  pb = bitmapbits_ + y*bitmaplinesize_ + x*3;
        return pb[0] | ((DWORD)pb[1] << 8) | ((DWORD)pb[2] << 16);
    }

    /**
        Sets the color of the specified pixel.

        <b>Note:</b> this is a faster but <b>unsafe</b> version of <i>setPixel()</i>. 
        If there is a chance that <i>x</i> or <i>y</i> are out of range then <i>setPixel()</i> 
        should be used.

        @param  x       x coordinate of pixel
        @param  y       y coordinate of pixel
        @param  color   color to set for specified pixel

        @see #setPixel
    */
    void setPixelFast( int x, int y, DWORD value ) {
        LPBYTE  pb = bitmapbits_ + y*bitmaplinesize_ + x*3;
        *pb++ = (BYTE)(value >> 16);
        *pb++ = (BYTE)(value >> 8);
        *pb   = (BYTE)(value);
    }

    /**
        Returns the color of the specified pixel.

        @param  x   x coordinate of pixel
        @param  y   y coordinate of pixel

        @return color of specified pixel or zero (black) if pixel coordinates are invalid
        
        @see #getPixelFast
    */
    DWORD getPixel( int x, int y ) const;

    /**
        Sets the color of the specified pixel.

        @param  x       x coordinate of pixel
        @param  y       y coordinate of pixel
        @param  color   color to set for specified pixel

        @see #setPixelFast
    */
    void setPixel( int x, int y, DWORD value );

    /**
        Returns the handle of the bitmap device context.

        This handle may be used in all the GDI calls that require a device context.

        @return handle of bitmap device context
    */
    HDC getHandle() {
        return hdc_;
    }

private:
    void createBitmap( int width, int height );

    BITMAPINFO  bitmapinfo_;
    LPBYTE      bitmapbits_;        // Pointer to the (device independent) bitmap data
    int         bitmaplinesize_;    // Size in bytes of a bitmap scanline
    int         bitmapwidth_;       // Width of bitmap in pixels
    int         bitmapheight_;      // Height of bitmap in pixels
    HANDLE      hbm_;               // Bitmap handle
    HDC         hdc_;               // Device context handle
};

#endif // DIB24_