; FAT clusters

	module	fat_cluster
	
include	"#idedos.def"
include	"#p3dos.def"
include	"#packages.def"
include	"fatfs.def"

	; In fat_fat
	xref	fat_readfat
	xref	fat_writefat
	
	; In fat_buffers
	xref	buf_findbuf
	xref	buf_findbuf_noread
	xref	buf_writebuf
	
	; Our exports
	xdef	clus_readtobuf
	xdef	clus_readtomem
	xdef	clus_writefrommem
	xdef	clus_tobuf_noread
	xdef	clus_getsector
	xdef	clus_nextsector
	xdef	clus_allocate
	xdef	clus_extendchain
	xdef	clus_freechain
	xdef	clus_erase
	xdef	clus_valid
	

; TBD: Maximum cluster size is limited to 64K-512bytes by file_getvalidaddr.
;      Is this okay?


; ***************************************************************************
; * Subroutine to calculate sector number of cluster portion                *
; ***************************************************************************
; On entry, IX=partition handle, BC=cluster number and A=sector offset
; On exit, Fc=1 (success) and BCDE=sector number
; Or, Fc=0 and A=error
; AHL corrupted.

.clus_getsector
	cp	(ix+fph_clussize)	; ensure sector offset in range
	jr	nc,clus_getsector_seekerr
	ld	h,0
	ld	l,a
	ld	a,h			; AHL=sector offset
	dec	bc
	dec	bc			; BC=cluster-2
	ld	e,(ix+fph_clussize)
.clus_getsector_loop
	add	hl,bc
	adc	a,0			; form AHL=sector offset+clussize*(cluster-2)
	dec	e
	jr	nz,clus_getsector_loop
	ld	c,(ix+fph_data_start)
	ld	b,(ix+fph_data_start+1)
	add	hl,bc
	adc	a,(ix+fph_data_start+2)	; AHL=logical sector number in partition
	ex	de,hl
	ld	c,a
	ld	b,0			; BCDE=logical sector number
	scf				; success!
	ret
.clus_getsector_seekerr
	ld	a,rc_seek		; Fc=0 already; seek error
	ret


; ***************************************************************************
; * Read sector from cluster                                                *
; ***************************************************************************
; On entry, IX=partition handle, BC=cluster number and A=sector offset
; On exit, HL=sector address, Fc=1 (success)
; Or, Fc=0, failure (A=error)
; ABCDEHL corrupted.

.clus_readtobuf
	call	clus_getsector		; BCDE=logical sector
	ret	nc			; exit if error
	jp	buf_findbuf		; read to buffer


; ***************************************************************************
; * Get buffer for cluster sector, without reading disk                     *
; ***************************************************************************
; On entry, IX=partition handle, BC=cluster number and A=sector offset
; On exit, HL=sector address, Fc=1 (success)
; Or, Fc=0, failure (A=error)
; ABCDEHL corrupted.

.clus_tobuf_noread
	call	clus_getsector		; BCDE=logical sector
	ret	nc			; exit if error
	call	buf_findbuf_noread	; get buffer without reading
	scf				; success!
	ret

; ***************************************************************************
; * Read sector from cluster to destination                                 *
; ***************************************************************************
; On entry, IX=partition handle, BC=cluster number, A=sector offset,
; and HL=address
; On exit, Fc=1 (success)
; Or, Fc=0, failure (A=error)
; ABCDEHL corrupted.
	
.clus_readtomem
	push	hl			; save destination
	call	clus_readtobuf		; read to buffer
	pop	de
	ret	nc			; exit if error
	ld	bc,buf_secsize
	ldir				; move to destination
	scf				; success!
	ret


; ***************************************************************************
; * Write sector to cluster                                                 *
; ***************************************************************************
; On entry, IX=partition handle, BC=cluster number, A=sector offset
; and HL=sector address
; On exit, Fc=1 (success)
; Or, Fc=0, failure (A=error)
; ABCDEHL corrupted.

.clus_writefrommem
	push	hl			; save source
	call	clus_getsector		; BCDE=logical sector
	pop	hl
	ret	nc			; exit if error
	push	hl			; save source
	call	buf_findbuf_noread	; find buffer, but don't read from disk
	ex	de,hl
	pop	hl
	ld	bc,buf_secsize
	ldir				; move to buffer
	jp	buf_writebuf		; write the MRU buffer (this one)


; ***************************************************************************
; * Get next cluster portion                                                *
; ***************************************************************************
; On entry, IX=partition handle, BC=cluster number, A=sector offset
; On exit, Fc=1 (success) with BC & A updated to next cluster portion
; Or, Fc=0 (fail) and A=error if end of cluster chain reached (or FAT read error).
; DEHL corrupted.

.clus_nextsector
	inc	a
	cp	(ix+fph_clussize)
	ret	c			; OK if < number of sectors per cluster
	call	fat_readfat		; get next cluster in chain
	ret	nc			; exit if error reading FAT
	call	clus_valid
	ld	a,rc_eof
	ret	nc			; exit if end of cluster chain
.clus_nextsector_use
	xor	a			; sector offset=0
	scf				; success!
	ret


; ***************************************************************************
; * Check cluster validity                                                  *
; ***************************************************************************
; On entry, IX=partition handle, BC=cluster number
; On exit, Fc=1 (success) if cluster is valid
; Or, Fc=0 (fail) if not
; A corrupted.

.clus_valid
	push	hl
	ld	hl,$0001
	and	a
	sbc	hl,bc
	pop	hl
	ret	nc			; invalid if $0000 or $0001
	push	hl
	ld	l,(ix+fph_maxclust)
	ld	h,(ix+fph_maxclust+1)
	and	a
	sbc	hl,bc			; Fc=1 if BC > max cluster
	ccf				; Fc=1 if BC <= max cluster
	pop	hl
	ret


; ***************************************************************************
; * Allocate new cluster                                                    *
; ***************************************************************************
; On entry, IX=partition handle
; On exit, Fc=1 (success), with BC=cluster number
; Or, Fc=1 (fail) and A=error
; ADEHL corrupted

.clus_allocate
	ld	c,(ix+fph_lastalloc)
	ld	b,(ix+fph_lastalloc+1)	; BC=last cluster allocated
	push	bc
.clus_allocate_search
	inc	bc			; try next cluster
	pop	hl			; HL=starting cluster number
	push	hl
	and	a
	sbc	hl,bc
	jr	z,clus_allocate_failed	; failed if searched entire FAT
	call	clus_valid
	jr	c,clus_allocate_try
	ld	bc,$0001		; loop to first valid cluster
	jr	clus_allocate_search
.clus_allocate_try
	push	bc			; save cluster number
	call	fat_readfat		; read its entry
	jr	c,clus_allocate_readok
	pop	hl			; discard stack items
	pop	hl
	ret				; exit with error
.clus_allocate_readok
	ld	a,b
	or	c			; is it free? (0)
	pop	bc			; restore cluster number
	jr	nz,clus_allocate_search	; keep going if in use
	pop	hl			; discard search start point
	ld	(ix+fph_lastalloc),c	; update last allocation
	ld	(ix+fph_lastalloc+1),b
	push	bc
	ld	e,(ix+fph_freeclusts)
	ld	d,(ix+fph_freeclusts+1)
	dec	de			; reduce free clusters
	ld	(ix+fph_freeclusts),e
	ld	(ix+fph_freeclusts+1),d
	ld	de,$ffff
	call	fat_writefat		; mark as allocated (last in chain)
	pop	bc			; restore cluster number
	ret				; exit with any error from writing FAT
.clus_allocate_failed
	pop	hl			; discard starting point
	ld	a,rc_diskfull
	and	a			; Fc=0
	ret


; ***************************************************************************
; * Add cluster to chain                                                    *
; ***************************************************************************
; On entry, IX=partition handle, BC=last cluster in current chain
; On exit, Fc=1 (success), with BC=new cluster number (added to chain)
; Or, Fc=1 (fail) and A=error
; ADEHL corrupted

.clus_extendchain
	push	bc
	call	clus_allocate		; allocate a new cluster
	ld	d,b
	ld	e,c			; DE=new cluster
	pop	bc			; BC=current cluster
	ret	nc			; exit if allocation error
	push	de			; save new cluster
	call	fat_writefat		; store at end of chain
	pop	bc			; BC=cluster
	ret				; exit with any error from writing FAT
	

; ***************************************************************************
; * Free entire cluster chain                                               *
; ***************************************************************************
; On entry, IX=partition handle, BC=starting cluster
; On exit, Fc=1 (success)
; Or, Fc=1 (fail) and A=error
; ABCDEHL corrupted

.clus_freechain
	call	clus_valid
	ccf
	ret	c			; end of chain reached
	push	bc
	call	fat_readfat		; get next cluster in chain
	jr	nc,clus_freechain_error
	ld	d,b
	ld	e,c
	pop	bc			; BC=cluster to free
	push	de			; save next cluster in chain
	inc	(ix+fph_freeclusts)	; increment free clusters
	jr	nz,clus_fc_nocarry
	inc	(ix+fph_freeclusts+1)
.clus_fc_nocarry
	ld	de,$0000
	call	fat_writefat		; write 0 to FAT entry, to free up
.clus_freechain_error
	pop	bc			; BC=next cluster
	jr	c,clus_freechain	; go to free remainder of chain
	ret				; exit with error

	
; ***************************************************************************
; * Erase cluster                                                           *
; ***************************************************************************
; On entry, IX=partition handle, BC=cluster
; On exit, Fc=1 (success), with HL=buffer address of first sector
; Or, Fc=1 (fail) and A=error
; ABCDE corrupted.
	
.clus_erase
	push	bc
	xor	a
	call	clus_tobuf_noread	; get a buffer for the first cluster sector
	pop	bc
	ret	nc			; exit if error
	push	hl			; save buffer address
	push	bc			; and cluster
	ld	d,h
	ld	e,l
	inc	de
	ld	bc,buf_secsize-1
	ld	(hl),0
	ldir				; erase the buffer
	pop	bc
	xor	a			; BC,A=cluster,start sector
	call	clus_getsector		; BCDE=logical sector
	pop	hl
	ret	nc			; exit if error
	ld	a,(ix+fph_clussize)	; A=sectors per cluster
.clus_erase_loop
	push	af
	push	bc
	push	de
	push	hl
	call	PACKAGE_FS_SECTOR_WRITE	; write the sector
	pop	hl
	pop	de
	pop	bc
	jr	nc,baderase		; exit if error
	inc	de			; increment sector low word
	ld	a,d
	or	e
	jr	nz,clus_erase_sectorok
	inc	bc			; increment sector high word
.clus_erase_sectorok
	pop	af
	dec	a
	jr	nz,clus_erase_loop	; back to erase remaining sectors
	scf				; success!
	ret				; exit with HL=buffer address
.baderase
	pop	de			; discard remaining sectors
	ret
