; FAT drives

	module	fat_drives
	
include	"#idedos.def"
include	"#p3dos.def"
include	"#packages.def"
include	"fatfs.def"
include	"fatfspkg.def"

	; In fat_buffers.asm
	xref	buf_findbuf_noread
	xref	buf_findbuf
	xref	buf_flushbuffers
	
	; In fat_dirs.asm
	xref	dir_root
	xref	dir_findvolumelabel
	
	; In fat_files.asm
	xref	file_getunopenhandles
	xref	file_commit
	
	; Our exports
	xdef	drive_free_space
	xdef	drive_drive
	xdef	drive_identify
	xdef	drive_partitions
	xdef	drive_partitions_physical
	xdef	drive_partition_read
	xdef	drive_partition_find
	xdef	drive_dos_map
	xdef	drive_dos_unmap
	xdef	drive_dos_mapping
	xdef	drive_getdirhandle
	xdef	drive_getdircopy
	xdef	drive_setupunit
	xdef	drive_flush
	

; ***************************************************************************
; * FAT_FREE_SPACE                                                          *
; ***************************************************************************
; Entry: A=drive, 'A'-'P'
; Exit: Fc=1 (success), HL=free space[limited to 65535] (K),
;			BCDE=free space (K)
;       Fc=0 (failure), A=error
; BCDEIXIY corrupt

.drive_free_space
	call	drive_getdirhandle	; IY=directory handle, IX=partition handle for drive A
	ret	nc			; exit if error
	ld	e,(ix+fph_freeclusts)	; get DE=free clusters
	ld	d,(ix+fph_freeclusts+1)
	ld	a,d
	and	e
	inc	a			; if DE=$ffff, not yet calculated
	scf				; force Fc=1 (success)
	call	z,drive_calcfree	; so calculate it
	ret	nc			; exit if error
	ld	hl,0			; BCHL=0
	ld	b,h
	ld	c,l
	ld	a,(ix+fph_clussize)	; A=number of sectors per cluster
.drive_fs_loop
	add	hl,de			; calculate BCHL=free sectors
	jr	nc,drive_fs_nocarry
	inc	bc
.drive_fs_nocarry
	dec	a
	jr	nz,drive_fs_loop
	srl	b			; calculate BCHL=free K
	rr	c
	rr	h
	rr	l
	ld	d,h
	ld	e,l			; BCDE=free K, HL=free K (to 65535)
	ld	a,b
	or	c			; is high word non-zero?
	jr	z,drive_fs_small
	ld	hl,$ffff		; if so, use 65535K
.drive_fs_small
	scf				; success!
	ret


; ***************************************************************************
; * FAT_DRIVE                                                               *
; ***************************************************************************
; Entry: A=logical unit
; Exit: Fc=1 (success), IX=0 (notional address of unit information)
;       Fc=0 (failure), A=error
; AFIXIY corrupt

.drive_drive
	call	drive_translateunit	; doesn't return if invalid unit
	ld	ix,0
	scf				; success!
	ret


; ***************************************************************************
; * FAT_IDENTIFY                                                            *
; ***************************************************************************
; Entry: C=logical unit, B=page for transfer, HL=address
; Exit: Fc=1 (success), HL=end address+1
;       Fc=0 (failure), A=error
; AFHLIY corrupt

.drive_identify
	ld	a,c
	call	drive_translateunit	; doesn't return if invalid unit
	ld	c,a
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_IDENTIFY
	jp	PACKAGE_CALL_PKG	; run IDEDOS call on physical unit


; ***************************************************************************
; * FAT_PARTITIONS                                                          *
; ***************************************************************************
; Entry: A=logical unit
; Exit: Fc=1 (success), A=open partitions, HL=0 (no free partition handle)
;
; AFBCDEHLIXIY corrupt
;
; For FAT, just count the number of mapped drives to this unit.

.drive_partitions
	call	drive_translateunit	; doesn't return if invalid unit
.drive_partitions_physical
	ld	b,0			; partitions found so far
	ld	c,a			; save unit
	ld	a,'A'
.drive_partitions_nextdrive
	push	af
	call	drive_refdir		; reference next drive
	jr	z,drive_partitions_skip	; ignore if pointer free
	push	de
	pop	iy			; IY=dir handle
	ld	e,(iy+fdh_phandle)
	ld	d,(iy+fdh_phandle+1)
	push	de
	pop	iy			; IY=partition handle
	ld	a,(iy+fph_parthandle+ph_unit)
	cp	c			; is it the one we're looking for?
	jr	nz,drive_partitions_skip
	inc	b			; increment partitions found
.drive_partitions_skip
	pop	af
	inc	a
	cp	'P'+1
	jr	c,drive_partitions_nextdrive
	ld	a,b			; A=partitions found
	scf				; success!
	ret


; ***************************************************************************
; * Subroutine to translate logical unit number to physical (0/1)           *
; ***************************************************************************
; Entry: A=logical unit
; Exit: A=physical unit
; Does not exit if invalid translation; instead returns to outer caller
; with error.

.drive_translateunit
	push	hl
	ld	hl,unit_translate
	cp	(hl)
	jr	z,physunit0
	inc	hl
	cp	(hl)
	pop	hl
	jr	z,physunit1
	pop	af		; discard return address
	ld	a,rc_nodrive
	and	a
	ret			; return to outer caller with error
.physunit0
	pop	hl
	xor	a
	ret
.physunit1
	ld	a,1
	ret


; ***************************************************************************
; * Subroutine to get directory handle pointer for drive                    *
; ***************************************************************************
; Entry: A=drive, 'A'-'P'
; Exit: Fc=1 (success), HL=address of pointer, DE=pointer, Fz=1 if DE=0
;       Fc=0 (failure), A=error

.drive_refdir
	sub	'A'
	jr	c,drive_refdir_bad
	cp	drive_numdrives
	jr	nc,drive_refdir_bad
	ld	l,a
	ld	h,0
	add	hl,hl
	ld	de,drive_ptrs+1
	add	hl,de			; HL=address of pointer+1
	ld	d,(hl)
	dec	hl
	ld	e,(hl)			; HL=address of pointer, DE=pointer
	ld	a,d
	or	e			; Fz=1 if DE=0
	scf				; success
	ret
.drive_refdir_bad
	and	a
	ld	a,rc_badparam		; bad parameter error
	ret


; ***************************************************************************
; * Subroutine to get directory and partition handles for drive             *
; ***************************************************************************
; Entry: A=drive, 'A'-'P'
; Exit: Fc=1 (success), IY=directory handle, IX=partition handle
;       Fc=0 (failure), A=error
; DEHL corrupt

.drive_getdirhandle
	call	drive_refdir		; DE=directory handle if Fz=0
	ret	nc			; exit if error
	ld	a,rc_nodrive
	ccf
	ret	z
	push	de
	pop	iy			; IY=directory handle
	ld	e,(iy+fdh_phandle)
	ld	d,(iy+fdh_phandle+1)
	push	de
	pop	ix			; IX=partition handle
	scf				; success!
	ret


; ***************************************************************************
; * Subroutine to get directory handle copy for drive                       *
; ***************************************************************************
; Entry: A=drive, 'A'-'P'
; Exit: Fc=1 (success), IY=copy of directory handle, IX=partition handle
;       Fc=0 (failure), A=error
; BCDEHL corrupt

.drive_getdircopy
	call	drive_getdirhandle	; IY=directory handle, IX=partition handle for drive A
	push	iy
	pop	hl
	ld	de,sys_directory
	ld	bc,fdh_size
	ldir				; copy to system handle
	ld	iy,sys_directory
	ret


; ***************************************************************************
; * Subroutine to set partition handle                                      *
; ***************************************************************************
; Entry: IX=partition handle, IY=partition entry, DEHL=current sector LBA
; Exit: -
; BCDEHL corrupt

.drive_setpartition
	ld	c,(iy+partentry_start_lba)
	ld	b,(iy+partentry_start_lba+1)
	add	hl,bc
	ld	(ix+ph_new_lba),l	; calculate and store LBA
	ld	(ix+ph_new_lba+1),h	; start in partition handle
	ex	de,hl
	ld	c,(iy+partentry_start_lba+2)
	ld	b,(iy+partentry_start_lba+3)
	adc	hl,bc
	ld	(ix+ph_new_lba+2),l
	ld	(ix+ph_new_lba+3),h
	ld	c,(iy+partentry_size_lba)
	ld	b,(iy+partentry_size_lba+1)
	ld	(ix+ph_new_secs),c	; store LBA size
	ld	(ix+ph_new_secs+1),b	; in partition handle
	ld	c,(iy+partentry_size_lba+2)
	ld	b,(iy+partentry_size_lba+3)
	ld	(ix+ph_new_secs+2),c
	ld	(ix+ph_new_secs+3),b
	ret


; ***************************************************************************
; * Subroutine to setup partition handle for whole drive                    *
; ***************************************************************************
; Entry: IX=partition handle, A=unit (0/1)
; Exit: -
; BCDEHL corrupt

.drive_setdrive
	and	1			; ensure 0 or 1
	ld	hl,drive_unit0
	jr	z,drive_sd_got0
	ld	hl,drive_unit1
.drive_sd_got0
	push	ix
	pop	de
	ld	bc,ph_length
	ldir				; copy from one provided by IDEDOS
	ret
	

; ***************************************************************************
; * Subroutine to initialise partition handle for unit                      *
; ***************************************************************************
; Entry: IX=partition handle, B=physical unit (0/1),
;        DE=address for logical unit translation,
;        HL=512-byte buffer for scratch usage, in main RAM
; Exit: -
; AFC corrupt

.drive_setupunit
	push	de
	push	hl
	push	bc
	ld	c,b			; C=physical unit
	ld	b,0			; B=page (0)
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_IDENTIFY		; get drive identity information
	call	PACKAGE_CALL_PKG
	pop	bc
	jr	nc,nophysunit		; exit if not found
	pop	hl
	push	hl
	ld	de,49*2+1		; word 49 = caps (want high byte)
	add	hl,de
	bit	1,(hl)			; bit 1 must be set for LBA mode
	jr	z,nophysunit		; exit if not
	ld	c,2			; start with logical unit 0
.findfreeunit
	push	bc			; save logical and physical units
	ld	b,FATFSPKG_ID
	ld	a,fs_request
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_FS_UNIT		; request a unit
	call	PACKAGE_CALL_PKG
	pop	bc
	jr	c,gotfreeunit
	inc	c
	ld	a,c
	cp	16
	jr	c,findfreeunit		; search through all 16 logical units
.nophysunit
	pop	hl
	pop	de
	ret				; exit if not found
.gotfreeunit
	ld	(ix+ph_unit),b		; store physical unit in partition handle
	pop	hl			; get HL=address of identity info
	push	hl
	ld	de,60*2			; words 60-61 = LBA capacity
	add	hl,de
	ld	e,(hl)
	inc	hl
	ld	d,(hl)			; word 60
	inc	hl
	dec	de			; decrement, to give highest sector
	ld	(ix+ph_new_secs),e
	ld	(ix+ph_new_secs+1),d
	ld	a,d
	and	e
	inc	a			; Fz=1 if high word needs decrementing
	ld	e,(hl)
	inc	hl
	ld	d,(hl)			; word 61
	jr	nz,capacityok
	dec	de			; decrement high word
.capacityok
	ld	(ix+ph_new_secs+2),e
	ld	(ix+ph_new_secs+3),d
	pop	hl
	pop	de
	ld	a,c
	ld	(de),a			; save reserved logical unit
	ret


; ***************************************************************************
; * Subroutine to get partition info                                        *
; ***************************************************************************
; Entry: A=physical unit, BC=partition number, IX=destination partition handle
; Exit: Fc=1 (success), IX=filled in, DE=name address, B=name length (8/11)
;       Fc=0 (failure), A=error
; AFBCDEHLIY corrupt

.drive_getpartition
	push	bc			; save partition number
	push	ix			; save IX
	call	drive_setdrive		; setup partition handle for whole drive
	ld	ix,0
	call	buf_findbuf_noread	; get a buffer not associated with a partition - TBD!
	ld	(drive_sectorbuf),hl	; save buffer address for later
	pop	ix
	pop	hl			; HL=partition number
	ld	(ix+fph_secondary),0	; secondary partition zero (not extended)
	ld	(ix+ph_new_num),l	; store partition number in handle
	ld	(ix+ph_new_num+1),h
	ld	b,0			; MBR on sector zero
	ld	c,b
	ld	d,b
	ld	e,b
.drive_gp_recurse
	push	bc			; save LBA(high)
	push	de			; save LBA(low)
	push	hl			; save partition number
	ld	hl,(drive_sectorbuf)
	call	PACKAGE_FS_SECTOR_READ	; read MBR
	jp	nc,drive_gp_error
	ld	iy,(drive_sectorbuf)	; IY=buffer address
	inc	iyh
	inc	iyh
	ld	a,(iy+mbr_signature-512) ; check for valid MBR signature
	cp	mbr_signature_byte0
	jp	nz,drive_gp_error
	ld	a,(iy+mbr_signature+1-512)
	cp	mbr_signature_byte1
	jp	nz,drive_gp_error
	dec	iyh
	dec	iyh
	ld	de,mbr_partentry0
	ld	(ix+fph_primary),1	; primary partition 1 (1-4)
.drive_gp_checkentry
	add	iy,de			; IX=next partition entry
	ld	a,(iy+partentry_type)
	cp	parttype_extended
	jr	z,drive_gp_extended
	cp	parttype_extended_lba
	jr	nz,drive_gp_normal
.drive_gp_extended
	pop	bc			; BC=partition number
	pop	hl
	pop	de			; D'E'H'L'=current LBA
	push	de
	push	hl			; resave
	push	bc			; resave partition number
	ld	c,(iy+partentry_start_lba)
	ld	b,(iy+partentry_start_lba+1)
	add	hl,bc
	ex	de,hl
	ld	c,(iy+partentry_start_lba+2)
	ld	b,(iy+partentry_start_lba+3)
	adc	hl,bc
	ld	b,h
	ld	c,l			; 'B'C'D'E'=LBA of extended
	pop	hl			; HL=partition number
	push	iy			; save partition entry address
	ld	a,(ix+fph_primary)
	inc	(ix+fph_secondary)	; doing an extended partition
	push	af
	call	drive_gp_recurse	; do the extended partition
	pop	hl
	ld	(ix+fph_primary),h	; restory primary ID
	jr	nc,drive_gp_continue	; not finished yet, continue
	pop	hl			; discard info
	pop	hl
	pop	hl
	ret				; exit with success
.drive_gp_continue
	pop	iy			; IY=partition entry address
	pop	de
	pop	hl			; HLDE=LBA of current
	push	hl
	push	de			; resave
	push	bc			; save reduced partition number
	ld	hl,(drive_sectorbuf)
	call	PACKAGE_FS_SECTOR_READ	; re-read sector
	jr	nc,drive_gp_error
.drive_gp_nextentry
	ld	de,mbr_partentry_size
	inc	(ix+fph_primary)
	ld	a,(ix+fph_primary)
	cp	5
	jr	c,drive_gp_checkentry	; back for more
	ld	a,(ix+ph_new_num)
	or	(ix+ph_new_num+1)	; are we looking for partition 0?
	jr	nz,drive_gp_error
	ld	a,(ix+ph_unit)
	call	drive_setdrive		; reset partition to entire drive
	ld	(ix+fph_primary),0	; 0s0>
	ld	(ix+fph_secondary),0
	pop	bc			; discard stacked data
	pop	bc
	pop	bc
	jr	drive_gp_wholedrive	; attempt to read boot sector from LBA 0
.drive_gp_error
	pop	bc			; BC=reduced partition number
	pop	de			; discard LBA
	pop	de
	ld	a,rc_invpartition	; didn't find it
	and	a			; Fc=0, fail
	ret
.drive_gp_normal
;	cp	parttype_fat12		; TBD
;	jr	z,drive_gp_fat12
;	cp	parttype_fat32		; TBD
;	jr	z,drive_gp_fat32
;	cp	parttype_fat32_lba	; TBD
;	jr	z,drive_gp_fat32
;	cp	parttype_altos		; TBD
;	jr	z,drive_gp_altos
	cp	parttype_fat16_small
	jr	z,drive_gp_fat16
	cp	parttype_fat16_large
	jr	z,drive_gp_fat16
	cp	parttype_fat16_lba
	jr	nz,drive_gp_nextentry
.drive_gp_fat16
	pop	bc			; BC=partition #
	ld	a,b
	or	c			; Fz=1 if this is the one
	jr	z,drive_gp_gotit
	dec	bc			; reduce, since found one
	push	bc
	jr	drive_gp_nextentry
.drive_gp_gotit
	pop	hl
	pop	de			; DEHL=current LBA
	call	drive_setpartition	; setup partition handle IX
.drive_gp_wholedrive
	ld	b,0
	ld	c,b
	ld	d,b
	ld	e,b
	ld	hl,(drive_sectorbuf)
	call	PACKAGE_FS_SECTOR_READ	; read the boot sector
	ret	nc			; exit with error
	ld	iy,(drive_sectorbuf)
	ld	(ix+fph_fattype),fattype_fat16
	ld	(ix+fph_maxclust),$ef	; max FAT16 cluster is $ffef
	ld	(ix+fph_maxclust+1),$ff
	ld	a,(iy+bootrec_secsize)	; must be 512bytes/sec
	and	a
	jp	nz,drive_gp_badfat
	ld	a,(iy+bootrec_secsize+1)
	cp	2
	jp	nz,drive_gp_badfat
	ld	a,(iy+bootrec_clussize)
	and	a
	jp	z,drive_gp_badfat
	ld	(ix+fph_clussize),a	; set cluster size
	ld	l,(iy+bootrec_resvd)
	ld	h,(iy+bootrec_resvd+1)
	ld	(ix+fph_fat1_start),l	; set FAT start
	ld	(ix+fph_fat1_start+1),h	; 0HL=FAT1 start
	ld	(ix+fph_fat1_start+2),0
	ld	a,(iy+bootrec_fatcopies)
	and	a
	jp	z,drive_gp_badfat
	ld	(ix+fph_fat_copies),a	; set # FATs
	ld	b,a			; B=# FATs
	ld	e,(iy+bootrec_fatsize)
	ld	d,(iy+bootrec_fatsize+1) ; DE=FATsize
	ld	a,d
	or	e
	jp	z,drive_gp_badfat
	ld	(ix+fph_fatsize),e
	ld	(ix+fph_fatsize+1),d
	xor	a			; AHL=FAT1 start
.drive_gp_fat16_calcrootstart
	add	hl,de
	adc	a,0			; next FAT start
	djnz	drive_gp_fat16_calcrootstart
	ld	(ix+fph_root_start),l	; set root start
	ld	(ix+fph_root_start+1),h
	ld	(ix+fph_root_start+2),a
	ld	e,(iy+bootrec_rootents)
	ld	d,(iy+bootrec_rootents+1)
	ld	a,e
	and	15
	jr	nz,drive_gp_badfat	; should be multiple of 16
	srl	d
	rr	e
	srl	d
	rr	e
	srl	d
	rr	e
	srl	d
	rr	e			; DE=root sectors
	ld	a,d
	or	e
	jp	z,drive_gp_badfat
	ld	(ix+fph_rootsize),e	; set root size
	ld	(ix+fph_rootsize+1),d
	add	hl,de
	ld	a,(ix+fph_root_start+2)
	adc	a,0			; AHL=data start
	ld	(ix+fph_data_start),l	; set data start
	ld	(ix+fph_data_start+1),h
	ld	(ix+fph_data_start+2),a
	ld	(ix+fph_lastalloc),$01	; initialise lastalloc
	ld	(ix+fph_lastalloc+1),$00
	ld	(ix+fph_freeclusts),$ff	; set free clusters=$ffff (not yet calculated)
	ld	(ix+fph_freeclusts+1),$ff
	ld	a,(iy+bootrec_extsig)
	cp	bootrec_extsig_byte
	ld	de,novolname		; use "NO NAME" as default if no ext sig
	jr	nz,drive_gp_noextsig
	ld	de,bootrec_volname	; else use boot record volume name
	add	iy,de
	push	iy			; stack address of volumename
	pop	hl
	ld	de,dir_filespec1
	ld	bc,11
	ldir				; copy to filespec1
	ld	de,dir_filespec1
.drive_gp_noextsig
	push	de
	call	dir_findvolumelabel	; HL=volume label, if any
	pop	de			; DE=default volume name
	jr	nc,drive_gp_badfat	; exit with any error; not valid
	ld	b,11			; volume name is 11 chars
	scf				; success
	ret	z			; exit if didn't find volume label
	ex	de,hl			; else use found name
	ret
.drive_gp_badfat
	ld	a,rc_invpartition
	and	a			; Fc=0
	ret
	
.novolname
	defm	"NO NAME    "


; ***************************************************************************
; * Subroutine to calculate drive free space                                *
; ***************************************************************************
; Entry: IX=partition handle
; Exit: Fc=1 (success), IX has free space updated, DE=free clusters
;       Fc=0 (failure), A=error
; AFBCHL corrupt

; Doesn't use FAT_READFAT, in order to speed things up a bit.
; TBD: Only works for FAT16

.drive_calcfree
	ld	e,(ix+fph_fat1_start)
	ld	d,(ix+fph_fat1_start+1)
	ld	c,(ix+fph_fat1_start+2)	; CDE=start of FAT1
	ld	b,0			; B=current sector in FAT
	ld	hl,0			; HL=free clusters so far
.drive_cf_secloop
	push	de
	push	bc
	ld	b,0
	push	hl
	call	buf_findbuf		; get sector to HL
	pop	bc			; BC=free clusters so far
	jr	nc,drive_cf_error
	ld	e,0			; 256 entries to check
.drive_cf_clusloop
	ld	a,(hl)			; test next entry
	inc	hl
	or	(hl)
	inc	hl
	jr	nz,drive_cf_notfree
	inc	bc			; increment free clusters
.drive_cf_notfree
	dec	e
	jp	nz,drive_cf_clusloop
	ld	h,b			; HL=free clusters
	ld	l,c
	pop	bc			; restore sector details
	pop	de
	inc	de			; step to next FAT sector
	ld	a,d
	or	e
	jr	nz,drive_cf_noinc
	inc	c
.drive_cf_noinc
	inc	b
	ld	a,b
	cp	(ix+fph_fatsize)
	jr	c,drive_cf_secloop	; back for more sectors
	ex	de,hl			; DE=free clusters
	ld	(ix+fph_freeclusts),e	; store in partition handle
	ld	(ix+fph_freeclusts+1),d
	scf				; success!
	ret
.drive_cf_error
	pop	bc
	pop	de
	ret


; ***************************************************************************
; * FAT_PARTITION_READ                                                      *
; ***************************************************************************
; Entry: A=logical unit, BC=partition#, HL=address for 64-byte entry
; Exit: Fc=1 (success)
;       Fc=0 (fail), A=error
;
; AFDEHL corrupt

.drive_partition_read
	call	drive_translateunit	; A=physical unit (0/1)
	push	ix			; save registers
	push	bc
	push	hl
	ld	ix,sys_partition
	call	drive_getpartition
	pop	hl			; HL=partentry address
	jr	nc,drive_pr_end
	ld	a,(ix+fph_primary)	; get primary number
	add	a,'0'
	ld	(hl),a			; insert to name
	inc	hl
	ld	(hl),'s'
	inc	hl
	ld	a,(ix+fph_secondary)	; get secondary number
	add	a,'0'
	cp	'9'+1
	jr	c,secondok
	add	a,7			; if >9, use letters
.secondok
	ld	(hl),a			; insert to name
	inc	hl
	ld	(hl),'>'
	inc	hl
	ld	a,12
	sub	b			; A=spaces at end
	ld	c,b
	ld	b,0
	ex	de,hl
	ldir				; copy name
	ex	de,hl
.blankend
	ld	(hl),' '		; blank remainder
	inc	hl
	dec	a
	jr	nz,blankend
	ld	(hl),ptype_fat16
	inc	hl
	ld	b,6			; head/cyl info - use $ff
.setcylhead
	ld	(hl),$ff
	inc	hl
	djnz	setcylhead
	ex	de,hl
	push	ix
	pop	hl
	ld	bc,ph_new_secs
	add	hl,bc
	ld	bc,4
	ldir				; copy sector size
	ex	de,hl
	ld	b,37			; zero the rest of the partition entry
.zerorest
	ld	(hl),0
	inc	hl
	djnz	zerorest
	scf				; success!
.drive_pr_end
	pop	bc			; restore registers
	pop	ix
	ret
	
	
; ***************************************************************************
; * FAT_PARTITION_FIND                                                      *
; ***************************************************************************
; Entry: A=logical unit, HL=address of 16-byte name (space-padded)
; Exit: Fc=1 (success), BC=partition number
;       Fc=0 (fail), A=error
;
; AFDEHL corrupt

.drive_partition_find
	ld	bc,0			; start with partition 0
.drive_pf_trynext
	push	af			; save logical unit
	push	bc			; save partition
	push	hl			; save name address
	ld	hl,sys_partentry
	call	drive_partition_read	; read one
	pop	hl			; restore name
	jr	nc,drive_pf_failed	; exit if error
	inc	hl
	inc	hl
	inc	hl
	ld	a,(hl)
	cp	'>'			; is partition ID specified?
	jr	nz,drive_pf_nameonly
	dec	hl
	dec	hl
	dec	hl
	ld	bc,16			; if so, match entire name
	ld	de,sys_partentry
	scf				; Fc=1; may finish after '>'
	jr	drive_pf_trymatch
.drive_pf_nameonly
	ld	bc,12			; match only name
	ld	de,sys_partentry+4
	inc	hl
	and	a			; Fc=0; must match all
.drive_pf_trymatch
	ld	a,(de)
	inc	de
	cpi				; compare next byte with HL
	jr	nz,drive_pf_mismatch
	jp	pe,drive_pf_trymatch
.drive_pf_matched
	pop	bc			; BC=matched partition number
	pop	hl			; discard logical unit
	scf				; success!
	ret
.drive_pf_mismatch
	jr	nc,drive_pf_nextpart	; definitely no match
	ld	a,c
	cp	11			; was mismatch directly after '>'?
	jr	nz,drive_pf_nextpart	; definitely no match
	ld	a,' '
	dec	hl			; re-test current char
	ld	b,12			; must have 12 spaces after '>'
.drive_pf_blankname
	cp	(hl)
	jr	nz,drive_pf_nextpart	; definitely no match
	inc	hl
	djnz	drive_pf_blankname
	jr	drive_pf_matched	; matched
.drive_pf_nextpart
	pop	bc			; restore partition & unit
	pop	af
	inc	bc			; next partition
	jr	drive_pf_trynext
.drive_pf_failed
	pop	bc			; discard partition & unit
	pop	bc
	ret


; ***************************************************************************
; * FAT_DOS_MAP                                                             *
; ***************************************************************************
; Entry: A=logical unit, BC=partition#, L=drive letter
; Exit: Fc=1 (success)
;       Fc=0 (fail), A=error
;
; AFBCDEHL corrupt

.drive_dos_map
	push	ix			; save IX
	push	af			; save unit
	push	bc			; save partition
	ld	a,l
	and	$df			; make upper-case
	push	af			; save A=drive letter A..P
	call	drive_refdir		; check if drive already mapped
	jp	nc,drive_dm_error	; exit if error
	ld	a,rc_mapped
	ccf				; Fc=0
	jp	nz,drive_dm_error	; or already mapped
	pop	af
	sub	'A'-1
	pop	bc			; BC=partition#
	pop	de			; D=logical unit
	ld	e,a			; E=drive 1..16
	push	hl			; save address of drive pointer
	push	de			; save logical unit, drive 1..16
	ld	a,'A'			; check from drive A
.map_checkopen_loop
	push	af
	call	drive_refdir
	jr	z,map_notopen
	ex	de,hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)			; DE=partition handle pointer
	push	de
	pop	iy			; IY=partition handle
	ld	a,c			; check if partition number matches
	cp	(iy+ph_new_num)
	jr	nz,map_notopen
	ld	a,b
	cp	(iy+ph_new_num+1)
	jr	nz,map_notopen
	pop	hl
	pop	de			; D=logical unit
	push	de
	push	hl
	ld	a,d			; check if unit matches
	call	drive_translateunit
	cp	(iy+ph_new_unit)
	ld	a,rc_partopen
	jr	z,drive_dm_error	; if so, Fz=1, Fc=0 so exit with error
.map_notopen
	pop	af
	inc	a
	cp	'P'+1
	jr	c,map_checkopen_loop
	pop	hl			; H=logical unit, L=drive 1..16
	push	hl
	ld	ix,part_handles-fph_size
	ld	de,fph_size
.drive_dm_calcph
	add	ix,de			; calculate IX=partition handle
	dec	l
	jr	nz,drive_dm_calcph
	ld	a,h
	call	drive_translateunit	; A=physical unit
	call	drive_getpartition	; fill in the partition handle
	jr	nc,drive_dm_error2	; exit if error
	pop	de			; E=drive 1..16
	push	de
	ld	iy,dir_handles-fdh_size
	ld	bc,fdh_size
.drive_dm_calcdh
	add	iy,bc			; calculate IY=dir handle
	dec	e
	jr	nz,drive_dm_calcdh
	call	dir_root		; initialise directory handle to root
	pop	de			; E=drive 1..16
	ld	c,e
	dec	c			; C=drive 0..15
	ld	b,FATFSPKG_ID
	ld	a,fs_request
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_FS_DRIVE		; request the drive
	call	PACKAGE_CALL_PKG
	pop	hl			; HL=address of drive pointer
	jr	nc,drive_dm_exit	; exit if error
	ld	a,iyl
	ld	(hl),a			; store directory handle in pointer
	inc	hl
	ld	a,iyh
	ld	(hl),a
	scf				; success!
.drive_dm_exit
	pop	ix			; restore IX
	ret
.drive_dm_error
	pop	bc			; discard stacked data
.drive_dm_error2
	pop	bc
	pop	bc
	jr	drive_dm_exit
	

; ***************************************************************************
; * FAT_DOS_UNMAP                                                           *
; ***************************************************************************
; Entry: L=drive letter
; Exit: Fc=1 (success)
;       Fc=0 (fail), A=error
;
; AFBCDEHL corrupt

; TBD: Should fail if any files are open on this drive

.drive_dos_unmap
	ld	a,l
	and	$df			; make upper-case
	push	af			; save A=drive letter A..P
	call	drive_refdir		; check if drive already mapped
	pop	de			; D=drive letter A..P
	ret	nc			; exit if error
	ld	a,rc_nodrive
	ccf
	ret	z			; or if not mapped
	; TBD: Here, check for open files.
	ld	(hl),0			; clear pointer to directory handle
	inc	hl
	ld	(hl),0
	ld	a,d			; A=drive letter A..P
	sub	'A'
	ld	c,a			; C=drive 0..15
	ld	b,FATFSPKG_ID
	ld	a,fs_release
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_FS_DRIVE		; release the drive
	call	PACKAGE_CALL_PKG
	scf				; success!
	ret


; ***************************************************************************
; * FAT_DOS_MAPPING                                                         *
; ***************************************************************************
; Entry: L=drive letter, BC=18-byte buffer
; Exit: Fc=1 (success), Fz=0 if mapped, Fz=1 if not mapped
;       Fc=0 (fail), A=error
;
; AFBCDEHLIY corrupt

.drive_dos_mapping
	ld	a,l
	call	drive_refdir		; check if drive already mapped
	ret	nc			; exit if error (Fc=0)
	ret	z			; or if not mapped (Fc=1, Fz=1)
	ex	de,hl
	ld	e,(hl)
	inc	hl
	ld	d,(hl)			; DE=partition handle pointer
	push	de
	pop	iy			; IY=partition handle
	ld	hl,unit_translate
	bit	0,(iy+ph_new_unit)	; phys unit 0 or 1?
	jr	z,drive_dmi_phys0
	inc	hl
.drive_dmi_phys0
	ld	a,(hl)			; get logical unit
	add	a,'0'
	ld	(bc),a			; store in buffer
	inc	bc
	ld	a,'>'
	ld	(bc),a
	inc	bc
	push	bc			; save buffer address
	ld	a,(hl)			; re-fetch logical unit
	ld	(file_number),a		; save for later
	ld	c,(iy+ph_new_num)	; get partition number
	ld	b,(iy+ph_new_num+1)
	ld	hl,sys_partentry
	push	hl
	call	drive_partition_read	; read partition entry
	pop	hl
	pop	de
	ret	nc			; exit if error
	push	bc			; save partition number
	xor	a
	sbc	a,1			; Fc=1, Fz=0; mapped, success!
	ld	bc,16
	ldir				; copy name to buffer
	pop	bc			; restore partition number
	ld	a,(file_number)		; and unit
	ret


; ***************************************************************************
; * FAT_FLUSH / Flush any updated buffers                                   *
; ***************************************************************************
; On exit, Fc=1 (success)
;      or, Fc=0 (failure) and A=error
; ABCDEHL corrupted.
; This call actually flushes everything on all FAT drives, not just the one
; requested.

.drive_flush
	push	ix
	push	iy
	ld	b,16			; 16 files
.drive_flush_fileloop
	push	bc
	dec	b
	call	file_getunopenhandles	; handles to IY & IX
	ld	a,(iy+ffh_mode)
	and	a			; check for open files
	call	nz,file_commit		; update header & dir, commit changes
	pop	bc
	djnz	drive_flush_fileloop
	pop	iy
	pop	ix
	jp	buf_flushbuffers		; flush buffers and exit
