; FAT file handling

	module	fat_files
	
include	"#idedos.def"
include	"#p3dos.def"
include	"#packages.def"
include	"fatfs.def"
include	"fatfspkg.def"
	
	; In fat_fat.asm
	xref	fat_readfat
	
	; In fat_cluster.asm
	xref	clus_getsector
	xref	clus_readtobuf
	xref	clus_nextsector
	xref	clus_extendchain
	xref	clus_valid
	xref	clus_freechain
	xref	clus_allocate

	; In fat_buffers.asm
	xref	buf_writebuf
	xref	buf_update
	xref	buf_flushbuffers
	
	; In fat_dirs.asm
	xref	dir_parsefilename
	xref	dir_getfirstmatch
	xref	dir_getnextmatch
	xref	dir_getentry
	xref	dir_entrydetails
	xref	dir_getfree
	
	; Our exports
	xdef	file_byte_read
	xdef	file_byte_write
	xdef	file_read
	xdef	file_write
	xdef	file_get_position
	xdef	file_set_position
	xdef	file_get_eof
	xdef	file_close
	xdef	file_abandon
	xdef	file_isopen
	xdef	file_rename
	xdef	file_delete
	xdef	file_set_access
	xdef	file_set_attributes
	xdef	file_open
	xdef	file_ref_head
	xdef	file_snapdata
	xdef	file_delete_currententry
	xdef	file_create_entry
	xdef	file_commit
	xdef	file_getunopenhandles


; ***************************************************************************
; * Subroutine to get filehandle                                            *
; ***************************************************************************
; Entry: B=filenumber
; Enter at file_getunopenhandles if handle not needed to be open
; Exit: IY=filehandle, IX=partition handle
;       If invalid filenumber, returns to outer caller
; B corrupt.

.file_gethandles
	push	af
	ld	a,b
	cp	file_numhandles
	jr	nc,file_gethandle_bad2	; max filenumber exceeded
	call	file_getunopenhandles	; (no error can occur now)
	ld	a,(iy+ffh_mode)		; ensure file is open
	and	a
	jr	z,file_gethandle_bad2
	pop	af
	ret
.file_getunopenhandles
	push	af			; save registers
	push	de
	ld	a,b
	cp	file_numhandles
	jr	nc,file_gethandle_bad	; max filenumber exceeded
	ld	(file_number),a		; save current file number
	ld	iy,file_handles-ffh_size
	ld	de,ffh_size
	inc	b
.file_gethandle_formaddr
	add	iy,de			; calculate handle address
	djnz	file_gethandle_formaddr
	ld	a,(iy+ffh_dhandle+fdh_phandle)
	ld	ixl,a
	ld	a,(iy+ffh_dhandle+fdh_phandle+1)
	ld	ixh,a			; IX=partition handle
	pop	de
	pop	af
	ret
.file_gethandle_bad
	pop	de			; restore registers
.file_gethandle_bad2
	pop	af
	pop	af			; discard return address
	ld	a,rc_number
	and	a			; Fc=0, error
	ret				; return to outer caller
	

; ***************************************************************************
; * Subroutine to check if a file is open                                   *
; ***************************************************************************
; Entry: IY=directory handle, IX=partition handle, current entry is file
;        to be checked. Works by comparing directory handle contents.
; Exit: Fc=1 if file not open, Fc=0 if file open with A=mode flags
; ABCDEHL corrupt.

.file_isopen
	ld	hl,file_handles
	ld	b,file_numhandles
.file_isopen_checknext
	push	hl			; save registers
	push	bc
	ld	d,iyh			; DE=dir handle to compare with
	ld	e,iyl
	ld	b,fdh_size
.file_isopen_checkloop
	ld	a,(de)
	cp	(hl)			; compare dir handle contents
	inc	de
	inc	hl
	jr	nz,file_isopen_notthisone
	djnz	file_isopen_checkloop
	ld	a,(hl)			; A=mode of matching filehandle
	and	a
	jr	z,file_isopen_notthisone ; not actually open
	pop	bc
	pop	hl
	ret				; with Fc=0, A=mode
.file_isopen_notthisone
	pop	bc
	pop	hl
	ld	de,ffh_size
	add	hl,de			; next filehandle
	djnz	file_isopen_checknext
	scf				; Fc=1, not open
	ret
	

; ***************************************************************************
; * Subroutine to check fileposition                                        *
; ***************************************************************************
; Entry: IY=filehandle
; Exit: Fc=1 if fileposition within current file size, Fc=0 if not
; DEHL corrupt.

.file_checkfilepos
	ld	l,(iy+ffh_filepos)
	ld	h,(iy+ffh_filepos+1)
	ld	e,(iy+ffh_filesize)
	ld	d,(iy+ffh_filesize+1)
	and	a
	sbc	hl,de
	ld	l,(iy+ffh_filepos+2)
	ld	h,(iy+ffh_filepos+3)
	ld	e,(iy+ffh_filesize+2)
	ld	d,(iy+ffh_filesize+3)
	sbc	hl,de			; Fc=1 if size > filepos (okay)
	ret


; ***************************************************************************
; * Subroutine to ensure filesize includes current fileposition             *
; ***************************************************************************
; Entry: IY=filehandle
; Exit: -
; DEHL corrupt.

.file_maximisefilesize
	call	file_checkfilepos	; is filepos within size?
	ret	c			; exit if so
	call	file_getfilepos		; get DEHL=filepos
	inc	l			; increment it, as size must be 1 larger
	jr	nz,file_maximise_nowrap
	inc	h
	jr	nz,file_maximise_nowrap
	inc	e
	jr	nz,file_maximise_nowrap
	inc	d
.file_maximise_nowrap
	ld	(iy+ffh_filesize),l	; store in filesize
	ld	(iy+ffh_filesize+1),h
	ld	(iy+ffh_filesize+2),e
	ld	(iy+ffh_filesize+3),d
	set	fm_entry,(iy+ffh_mode)	; file-entry needs rewriting
	ret
	

; ***************************************************************************
; * Subroutine to increment fileposition                                    *
; ***************************************************************************
; Entry: IY=filehandle, IX=partition handle
; Exit: Fc=1 (success)
;   or: Fc=0 (failure) and A=error
; ABCDEHL corrupt.

.file_incfpos
	inc	(iy+ffh_filepos)	; increment low byte
	jr	nz,file_incfpos_offset	; on if no wrap
	inc	(iy+ffh_filepos+1)	; increment 2nd byte
	jr	nz,file_incfpos_offset	; on if no wrap
	inc	(iy+ffh_filepos+2)	; increment 3rd byte
	jr	nz,file_incfpos_offset	; on if no wrap
	inc	(iy+ffh_filepos+3)	; increment high byte
.file_incfpos_offset
	bit	fm_valid,(iy+ffh_mode)	; is location valid?
	scf				; success
	ret	z			; don't bother updating if not
	inc	(iy+ffh_offset)		; increment low byte of offset
	ret	nz			; done if no wrap
	inc	(iy+ffh_offset+1)	; increment high byte of offset
	bit	1,(iy+ffh_offset+1)
	ret	z			; okay if still < 512
	ld	(iy+ffh_offset+1),0	; reset offset to zero
	ld	c,(iy+ffh_cluster)
	ld	b,(iy+ffh_cluster+1)
	ld	a,(iy+ffh_sector)	; BC,A=cluster,sector
	call	clus_nextsector		; update to next
	jr	c,file_incfpos_okay	; on if okay
	res	fm_valid,(iy+ffh_mode)	; location no longer valid
	cp	rc_eof			; is error EOF?
	scf
	ccf				; ensure Fc=0
	ret	nz			; exit with real error if not
	scf				; success
	ret
.file_incfpos_okay
	ld	(iy+ffh_cluster),c	; store updated cluster,sector
	ld	(iy+ffh_cluster+1),b
	ld	(iy+ffh_sector),a
	ret				; exit with Fc=1


; ***************************************************************************
; * Subroutine to add offset to fileposition                                *
; ***************************************************************************
; TBD: May not work with offsets >64K-512. Does this matter?
; Entry: IY=filehandle, IX=partition handle, BC=bytes
; Exit: Fc=1 (success)
;   or: Fc=0 (failure) and A=error
; ABCDEHL corrupt.

.file_addfpos
	ld	l,(iy+ffh_filepos)
	ld	h,(iy+ffh_filepos+1)
	add	hl,bc			; add to low word
	ld	(iy+ffh_filepos),l
	ld	(iy+ffh_filepos+1),h
	jr	nc,file_addfpos_offset	; on if no carry
	inc	(iy+ffh_filepos+2)	; increment 3rd byte
	jr	nz,file_addfpos_offset	; on if no wrap
	inc	(iy+ffh_filepos+3)	; increment high byte
.file_addfpos_offset
	bit	fm_valid,(iy+ffh_mode)	; is location valid?
	scf				; success
	ret	z			; don't bother updating if not
	ld	l,(iy+ffh_offset)
	ld	h,(iy+ffh_offset+1)
	add	hl,bc			; add to offset
	ld	(iy+ffh_offset),l
	ld	(iy+ffh_offset+1),h
.file_addfpos_nextsector
	ld	bc,512
	and	a
	sbc	hl,bc
	ret	c			; okay if still < 512
	ld	(iy+ffh_offset),l	; reduce offset by 512
	ld	(iy+ffh_offset+1),h
	ld	c,(iy+ffh_cluster)
	ld	b,(iy+ffh_cluster+1)
	ld	a,(iy+ffh_sector)	; BC,A=cluster,sector
	call	clus_nextsector		; update to next
	jr	nc,file_addfpos_failed
	ld	(iy+ffh_cluster),c	; store updated cluster,sector
	ld	(iy+ffh_cluster+1),b
	ld	(iy+ffh_sector),a
	ld	l,(iy+ffh_offset)
	ld	h,(iy+ffh_offset+1)
	jr	file_addfpos_nextsector	; go back to re-test offset
.file_addfpos_failed
	res	fm_valid,(iy+ffh_mode)	; location no longer valid
	cp	rc_eof			; is error EOF?
	scf
	ccf				; ensure Fc=0
	ret	nz			; exit with real error if not
	scf				; success
	ret


; ***************************************************************************
; * Subroutine to get valid address for current filepos                     *
; ***************************************************************************
; Entry: IY=filehandle, IX=partition handle
; Exit: Fc=1 (success), HL=address, DE=offset (ie HL-DE=start of sector buffer)
;   or: Fc=0 (failure) and A=error
; ABCDE corrupt.

.file_getvalidaddr
	bit	fm_valid,(iy+ffh_mode)	; is location valid?
	jr	nz,file_getvalidaddr_valid	; almost done, then

	; Now we have to go through from the beginning, until get to right position
	
	ld	c,(iy+ffh_clusstart)
	ld	b,(iy+ffh_clusstart+1)
	call	clus_valid		; is this a valid cluster?
	jr	c,file_getvalidaddr_startok
	call	clus_allocate		; allocate a new cluster
	ret	nc			; exit if error
	ld	(iy+ffh_clusstart),c	; store in file handle
	ld	(iy+ffh_clusstart+1),b
	set	fm_entry,(iy+ffh_mode)	; directory entry needs updating
.file_getvalidaddr_startok
	push	bc			; TOS=initial cluster
	call	file_getfilepos		; DEHL=fileposition required
	ld	a,(ix+fph_clussize)	; A=cluster size in sectors
	add	a,a
	ld	b,a
	ld	c,0			; BC=cluster size in bytes	
.file_getvalidaddr_clusterloop
	and	a
	sbc	hl,bc			; subtract cluster size from filepos
	jr	nc,file_getvalidaddr_nextcluster
	ld	a,d
	or	e			; if high word zero, within cluster
	jr	z,file_getvalidaddr_thiscluster
	dec	de			; decrement high word
.file_getvalidaddr_nextcluster
	ex	(sp),hl			; save HL, HL=cluster
	push	de			; save registers
	push	bc
	ld	b,h
	ld	c,l
	push	bc			; save source cluster
	call	fat_readfat		; BC=next cluster
	jr	nc,file_getvalidaddr_badcluster
	call	clus_valid		; is this a valid cluster?
	jr	nc,file_getvalidaddr_extend	; extend file if not
	pop	hl			; discard source cluster
.file_getvalidaddr_extended
	ld	h,b
	ld	l,c
	pop	bc			; restore registers
	pop	de
	ex	(sp),hl
	jr	file_getvalidaddr_clusterloop	; loop back
.file_getvalidaddr_extend
	pop	bc			; get previous cluster again
	call	clus_extendchain	; extend the chain
	jr	file_getvalidaddr_extended
.file_getvalidaddr_badcluster
	pop	bc			; discard registers
	pop	bc
	pop	bc
	pop	bc
	ret				; exit with error
.file_getvalidaddr_thiscluster
	add	hl,bc			; HL=offset within cluster
	pop	bc
	ld	(iy+ffh_cluster),c	; store cluster
	ld	(iy+ffh_cluster+1),b
	ld	bc,buf_secsize
	xor	a			; A=sector offset, Fc=0
.file_getvalidaddr_sectorloop
	sbc	hl,bc
	inc	a
	jr	nc,file_getvalidaddr_sectorloop
	add	hl,bc			; HL=offset within sector
	dec	a			; A=sector offset
	ld	(iy+ffh_sector),a	; store sector
	ld	(iy+ffh_offset),l	; store offset
	ld	(iy+ffh_offset+1),h
	set	fm_valid,(iy+ffh_mode)	; now valid!
.file_getvalidaddr_valid
	ld	c,(iy+ffh_cluster)
	ld	b,(iy+ffh_cluster+1)
	ld	a,(iy+ffh_sector)
	call	clus_readtobuf		; get to buffer
	ret	nc			; exit if error
	ld	e,(iy+ffh_offset)
	ld	d,(iy+ffh_offset+1)
	add	hl,de			; HL=address of byte
	scf				; success!
	ret


; ***************************************************************************
; * FAT_BYTE_READ                                                           *
; ***************************************************************************
; Entry: B=filenumber
; Exit: Fc=1 (success), C=byte, Fz=1 if C=$1a
;       Fc=0 (failure), A=error
; ABDEHLIXIY corrupt

.file_byte_read
	call	file_gethandles		; handles to IY & IX
	bit	fm_read,(iy+ffh_mode)	; is access mode okay?
	jr	z,file_error_access
	call	file_checkfilepos	; Fc=1 if size > filepos (okay)
	jr	nc,file_error_eof
	call	file_getvalidaddr	; get HL=address in buffer
	jr	nc,file_error		; exit if error
	ld	c,(hl)			; C=byte
	push	bc			; save
	call	file_incfpos		; add 1 to filepos and cluster etc
	pop	bc
	ld	a,c
	cp	$1a			; check for soft-EOF
	scf				; success!
	ret

.file_error_access
	ld	a,rc_number
	jr	file_error
.file_error_eof
	ld	a,rc_eof
.file_error
	and	a			; Fc=0, error
	ret


; ***************************************************************************
; * FAT_BYTE_WRITE                                                          *
; ***************************************************************************
; Entry: B=filenumber, C=byte
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIXIY corrupt

.file_byte_write
	call	file_gethandles		; handles to IY & IX
	bit	fm_write,(iy+ffh_mode)	; is access mode okay?
	jr	z,file_error_access
	push	bc
	call	file_maximisefilesize	; ensure filesize updated if necessary
	call	file_getvalidaddr	; get HL=address in buffer
	pop	bc
	jr	nc,file_error		; exit if error
	ld	(hl),c			; store byte
	call	buf_update		; mark buffer as updated
	call	file_incfpos		; add 1 to filepos and cluster etc
	scf				; success!
	ret


; ***************************************************************************
; * FAT_READ                                                                *
; ***************************************************************************
; Entry: B=filenumber, C=page (**IGNORED**), HL=address, DE=size (0==64K)
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error, DE=bytes remaining unread
; Soft-EOF ($1a) is not considered.
; Attempting to read over the ROM area is an error. TBD: Should we trap this?
; ABCDEHLIXIY corrupt

.file_read
	call	file_gethandles		; handles to IY & IX
	ld	(tmp_rwaddr),hl		; save address
	ld	(tmp_rwsize),de		; save size
	bit	fm_read,(iy+ffh_mode)	; is access mode okay?
	jp	z,file_read_access
	call	file_checkfilepos	; Fc=1 if filesize > filepos (okay)
	jr	nc,file_read_eof
	ld	c,(iy+ffh_filepos)
	ld	b,(iy+ffh_filepos+1)
	ld	l,(iy+ffh_filesize)
	ld	h,(iy+ffh_filesize+1)
	and	a
	sbc	hl,bc
	ex	de,hl			; DE=low word of bytes remaining in file
	ld	c,(iy+ffh_filepos+2)
	ld	b,(iy+ffh_filepos+3)
	ld	l,(iy+ffh_filesize+2)
	ld	h,(iy+ffh_filesize+3)
	sbc	hl,bc			; HLDE=bytes remaining in file
	ld	hl,(tmp_rwsize)		; HL=bytes to read
	jr	nz,file_read_readmax	; if bytes remaining in file >= 64K, we can read the whole lot
	ld	a,h
	or	l
	jr	z,file_read_readremaining ; if want to read 64K, just read all that's left
	and	a
	sbc	hl,de
.file_read_readremaining
	ex	de,hl
	jr	nc,file_read_readmax	; if bytes remaining < what we want, read all that's left
	add	hl,de			; else, read what we want
.file_read_readmax
	; At this point, HL=number of bytes we can read from the file (0==64K)
	ld	(tmp_rwpossible),hl
	call	file_getvalidaddr	; get HL=address, DE=512-bytes
	jr	nc,file_read_error	; exit if error
	push	hl
	ld	hl,512
	and	a
	sbc	hl,de			; HL=bytes in sector
	pop	de			; DE=buffer address
	ld	bc,(tmp_rwpossible)	; BC=bytes readable: May be zero==64K
	ld	a,b
	or	c
	jr	z,file_read_readallbuf
	sbc	hl,bc
	jr	nc,file_read_readable	; buffer contains all rest of file
	add	hl,bc
.file_read_readallbuf
	ld	b,h
	ld	c,l			; BC=bytes to read from this buffer
.file_read_readable
	push	bc
	ld	hl,(tmp_rwaddr)
	ex	de,hl
	ldir				; copy bytes from buffer
	ld	(tmp_rwaddr),de		; update address
	pop	bc
	ld	hl,(tmp_rwsize)
	and	a
	sbc	hl,bc
	ld	(tmp_rwsize),hl		; update size left to read
	push	af			; save flag
	push	bc
	call	file_addfpos		; update filepointer
	pop	bc
	pop	af			; restore finished flag
	jr	z,file_read_done	; finished if size=0
	ld	hl,(tmp_rwpossible)
	and	a
	sbc	hl,bc
	jr	z,file_read_eof		; also finished if can't read any more
	jr	file_read_readmax	; back for next sector

.file_read_done
	scf				; success!
	ret
.file_read_eof
	ld	a,rc_eof		; EOF
	jr	file_read_error
.file_read_access
	ld	a,rc_number		; access mode error
.file_read_error
	ld	de,(tmp_rwsize)		; get bytes remaining
	and	a			; Fc=0, failure
	ret


; ***************************************************************************
; * FAT_WRITE                                                               *
; ***************************************************************************
; Entry: B=filenumber, C=page (**IGNORED**), HL=address, DE=size (0==64K)
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error, DE=bytes remaining unread
; Soft-EOF ($1a) is not considered.
; Attempting to read over the ROM area is an error. TBD: Should we trap this?
; ABCDEHLIXIY corrupt

.file_write
	call	file_gethandles		; handles to IY & IX
	ld	(tmp_rwaddr),hl		; save address
	ld	(tmp_rwsize),de		; save size
	bit	fm_write,(iy+ffh_mode)	; is access mode okay?
	jr	z,file_read_access
.file_write_writemax	
	call	file_getvalidaddr	; get HL=address, DE=512-bytes
	jr	nc,file_read_error	; exit if error
	push	hl
	ld	hl,512
	and	a
	sbc	hl,de			; HL=bytes in sector
	pop	de			; DE=buffer address
	ld	bc,(tmp_rwsize)		; BC=bytes writable: May be zero==64K
	ld	a,b
	or	c
	jr	z,file_write_writeallbuf
	sbc	hl,bc
	jr	nc,file_write_writable	; buffer contains all rest of file
	add	hl,bc
.file_write_writeallbuf
	ld	b,h
	ld	c,l			; BC=bytes to write to this buffer
.file_write_writable
	push	bc
	ld	hl,(tmp_rwaddr)
	ldir				; copy bytes to buffer
	ld	(tmp_rwaddr),hl		; update address
	call	buf_update
	pop	bc
	ld	hl,(tmp_rwsize)
	and	a
	sbc	hl,bc
	ld	(tmp_rwsize),hl		; update size left to write
	push	af			; save flag
	dec	bc
	call	file_addfpos		; update filepointer to last written byte
	call	file_maximisefilesize	; ensure filesize updated if necessary
	call	file_incfpos		; update filepointer to next byte
	pop	af			; restore finished flag
	jr	z,file_read_done	; finished if size=0
	jr	file_write_writemax	; back for next sector


; ***************************************************************************
; * FAT_GET_POSITION                                                        *
; ***************************************************************************
; Entry: B=filenumber
; Exit: Fc=1 (success), DEHL=filepos (NB: This exceeds +3DOS specification!)
;       Fc=0 (failure), A=error
; ABCIXIY corrupt
; Can enter at file_getfilepos if IY=filehandle already

.file_get_position
	call	file_gethandles		; handles to IY & IX
.file_getfilepos
	ld	l,(iy+ffh_filepos)
	ld	h,(iy+ffh_filepos+1)
	ld	e,(iy+ffh_filepos+2)
	ld	d,(iy+ffh_filepos+3)	; DEHL=filepos
	scf				; success!
	ret


; ***************************************************************************
; * FAT_SET_POSITION                                                        *
; ***************************************************************************
; Entry: B=filenumber, DEHL=filepos (NB: This exceeds +3DOS specification!)
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIXIY corrupt

.file_set_position
	call	file_gethandles		; handles to IY & IX
.file_setpos_withhandles
	ld	(iy+ffh_filepos),l
	ld	(iy+ffh_filepos+1),h
	ld	(iy+ffh_filepos+2),e
	ld	(iy+ffh_filepos+3),d	; set filepos
	res	fm_valid,(iy+ffh_mode)	; no longer valid location
	scf				; success!
	ret


; ***************************************************************************
; * FAT_GET_EOF                                                             *
; ***************************************************************************
; Entry: B=filenumber
; Exit: Fc=1 (success), DEHL=filesize (NB: This exceeds +3DOS specification!)
;       Fc=0 (failure), A=error
; ABCIXIY corrupt
; Can enter at file_getfilesize if IY=filehandle already

.file_get_eof
	call	file_gethandles		; handles to IY & IX
.file_getfilesize
	ld	l,(iy+ffh_filesize)
	ld	h,(iy+ffh_filesize+1)
	ld	e,(iy+ffh_filesize+2)
	ld	d,(iy+ffh_filesize+3)	; DEHL=filesize
	scf				; success!
	ret


; ***************************************************************************
; * FAT_REF_HEAD                                                            *
; ***************************************************************************
; Entry: B=filenumber
; Exit: Fc=1 (success), IX=address in IDEDOS bank
;                       Fz=0 if header present, Fz=1 if not
;       Fc=0 (failure), A=error
; ABCIXIY corrupt

.file_ref_head
	call	file_gethandles		; handles to IY & IX
	ld	c,(iy+ffh_idedos_hdr)
	ld	b,(iy+ffh_idedos_hdr+1)
	push	bc
	pop	ix			; IX=address in IDEDOS bank
	bit	fm_header,(iy+ffh_mode) ; is there a header?
	scf				; success!
	ret


; ***************************************************************************
; * FAT_CLOSE                                                               *
; ***************************************************************************
; Entry: B=filenumber
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIXIY corrupt

.file_close
	call	file_gethandles		; handles to IY & IX
	call	file_commit		; update header, dir & commit
.file_close_entryok
	ld	(iy+ffh_mode),0		; free the filenumber
.file_release
	push	af			; save error condition
	ld	a,(file_number)
	ld	c,a			; C=filenumber
	ld	b,FATFSPKG_ID
	ld	a,fs_release
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_FS_FILE
	call	PACKAGE_CALL_PKG	; release the filenumber to IDEDOS
	pop	af
	ret
	
	
; ***************************************************************************
; * Write file header, update directory & FAT and commit changes            *
; ***************************************************************************
; Entry: IY=file handle, IX=directory handle
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHL corrupt

; TBD: Would be nice to update date & time if we had a RTC.
	
.file_commit
	bit	fm_write,(iy+ffh_mode)	; write access?
	jr	z,file_commit_readonly
	bit	fm_header,(iy+ffh_mode) ; is there a header?
	jr	z,file_commit_noheader	; on if not
	call	file_header_write	; write the header
	ret	nc			; exit if error
.file_commit_noheader
	call	buf_flushbuffers	; ensure buffers flushed if writeable
	ret	nc			; exit if error
.file_commit_readonly
	bit	fm_entry,(iy+ffh_mode)	; update dir entry?
	jr	z,file_commit_entryok	; on if not
	call	dir_getentry		; get the directory entry
	ret	nc			; exit if error
	ld	bc,direntry_cluster
	add	hl,bc
	ex	de,hl
	push	iy
	pop	hl
	ld	bc,ffh_clusstart
	add	hl,bc
	ld	c,2+4
	ldir				; copy cluster + size from handle
	call	buf_writebuf		; write the buffer back
.file_commit_entryok
	scf				; success!
	ret


; ***************************************************************************
; * FAT_ABANDON                                                             *
; ***************************************************************************
; Entry: B=filenumber
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIXIY corrupt

.file_abandon
	call	buf_flushbuffers	; flush buffers to be on the safe side
	call	file_gethandles		; handles to IY & IX
	jr	file_close_entryok
	

; ***************************************************************************
; * FAT_RENAME                                                              *
; ***************************************************************************
; Entry: DE=address of new filename (no wildcards)
;        HL=address of old filename (no wildcards)
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; BCDEHLIXIY corrupt

.file_rename
	push	hl
	ld	hl,dir_filespec2
	call	dir_parsefilename	; parse the new name
	pop	de
	ret	nc			; exit if error
	jr	nz,file_badname		; or if wildcard
	jp	p,file_badname		; or if no filename part
	; TBD: Also ought to fail if path specified in new name
	push	af			; save drive
	ld	hl,dir_filespec1
	call	dir_parsefilename	; parse the original name, and get handles
	pop	bc
	ret	nc			; exit if error
	jr	nz,file_badname		; or if wildcard
	jp	p,file_badname		; or if no filename part
	cp	b
	jr	nz,file_rename_drives	; exit if renaming between drives
	ld	de,dir_filespec2
	call	dir_getfirstmatch	; check if new name already exists
	ret	nc			; exit if error
	ld	a,rc_exists
	ccf
	ret	z			; exit if already exists
.file_rename_one
	ld	de,dir_filespec1
	call	dir_getfirstmatch	; find the original file
	ret	nc			; exit if error
	ld	a,rc_nofile
	ccf
	ret	nz			; exit if not found
	call	file_isopen
	ld	a,rc_denied
	ret	nc			; can't rename if file is open
	call	dir_entrydetails	; HL=entry
	ret	nc
	and	dirattr_syshid_mask	; don't find system/hidden files
	ld	a,rc_nofile
	ret	nz
	ld	de,dir_filespec2
	ex	de,hl
	ld	bc,11
	ldir				; replace old name with new
	jp	buf_writebuf		; update entry and exit with any error
.file_badname	
	ld	a,rc_badname
	and	a			; Fc=0, failure
	ret
.file_rename_drives
	ld	a,rc_norename
	and	a			; Fc=0, failure
	ret


; ***************************************************************************
; * FAT_DELETE                                                              *
; ***************************************************************************
; Entry: HL=address of filename (wildcards allowed)
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; BCDEHLIXIY corrupt

.file_delete
	ex	de,hl
	ld	hl,dir_filespec1
	call	dir_parsefilename	; parse filename and get handles
	ret	nc			; exit if error
	jp	p,file_badname		; or if no filename part
	jr	nz,file_delete_wild	; move on if deleting with wildcards
.file_delete_1st
	ld	de,dir_filespec1
.file_delete_de
	call	dir_getfirstmatch	; find file
	ret	nc			; exit if error
	ld	a,rc_nofile
	ccf
	ret	nz			; exit if not found
.file_delete_next
	call	file_isopen
	ld	a,rc_denied
	ret	nc			; can't delete if file is open
	call	dir_entrydetails	; HL=entry, A=attribs
	ret	nc
	and	dirattr_nodeletemask
	ld	a,rc_nofile
	ret	nz			; can't delete R/O, hidden, sys, vol or dir
.file_delete_currententry
	ld	(hl),direntry_deleted	; mark entry as deleted
	ld	de,direntry_cluster
	add	hl,de
	ld	c,(hl)
	inc	hl
	ld	b,(hl)			; BC=starting cluster
	push	bc
	call	buf_writebuf		; update directory entry
	pop	bc
	ret	nc
	jp	clus_freechain		; free the chain and exit with any error
.file_delete_wild
	call	file_delete_1st		; delete at least 1 file
	ret	nc			; exit if error
.file_delete_loop
	ld	de,dir_filespec1
	call	dir_getnextmatch	; find next file
	ret	nc
	ret	nz			; exit with success if no more matches
	call	file_delete_next	; delete it
	ret	nc
	jr	file_delete_loop


; ***************************************************************************
; * FAT_SET_ACCESS                                                          *
; ***************************************************************************
; Entry: B=filenumber
;	 C=access mode: bit 0=read, bit 1=write, bit 2=shared
;                       (shared only allowed with read)
; Or enter at file_validate access with: IY=filehandle (filled in except mode)
;					 C=access mode, B=0
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIXIY corrupt

.file_set_access
	call	file_gethandles		; handles to IY & IX
	ld	b,(iy+ffh_mode)
	push	bc			; save access mode and new
	call	file_commit		; update header & dir, commit changes
	pop	bc
	ret	nc			; exit if error
	ld	(iy+ffh_mode),0		; temporarily close file
.file_validate_access
	push	bc			; save current (B) and new (C) modes
	call	file_isopen		; check if other handle open to it
	pop	bc			; restore access mode and new
	ld	(iy+ffh_mode),b		; change mode back
	jr	c,file_set_unshared	; okay if not open to other handle
	bit	fm_shared,c		; if open, must stay shared
	jr	z,file_set_bad
	cp	fm_shared_read		; and other file must be shared read
	jr	nz,file_set_bad
.file_set_unshared
	ld	a,c			; check desired mode
	cp	fm_exc_read		; read is always okay
	jr	z,file_set_okay
	cp	fm_shared_read		; read is always okay
	jr	z,file_set_okay
	cp	fm_exc_write		; write needs checking
	jr	z,file_set_write
	cp	fm_exc_rw		; write needs checking
	jr	nz,file_set_bad		; any other combination is bogus
.file_set_write
	push	bc
	call	dir_entrydetails	; get A=attributes
	pop	bc
	ret	nc			; exit if error
	bit	dirattr_ro,a		; is it read-only?
	jr	nz,file_set_bad		; error if so
.file_set_okay
	ld	a,b
	and	$ff-fm_access_mask	; get other existing mode bits
	or	c			; combine with new access mode
	ld	(iy+ffh_mode),a
	scf				; success!
	ret
.file_set_bad
	ld	a,rc_denied
	and	a			; Fc=0
	ret


; ***************************************************************************
; * FAT_SET_ATTRIBUTES                                                      *
; ***************************************************************************
; Entry: D=attribs to set, E=attribs to clear
;        HL=filespec (wildcards allowed)
; Exit: Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIXIY corrupt

; TBD: Currently ignoring the "file attribs" f1-f4.

.file_set_attributes
	ex	de,hl
	ld	(file_attrmasks),hl	; save masks
	ld	hl,dir_filespec1
	call	dir_parsefilename	; parse filename and get handles
	ret	nc			; exit if error
	jp	p,file_badname		; or if no filename part
	jr	nz,file_setattr_wild	; move on if setting with wildcards
.file_setattr_1st
	ld	de,dir_filespec1
	call	dir_getfirstmatch	; find file
	ret	nc			; exit if error
	ld	a,rc_nofile
	ccf
	ret	nz			; exit if not found
.file_setattr_next
	call	file_isopen
	ld	a,rc_denied
	ret	nc			; can't change if file is open
	call	dir_entrydetails	; HL=entry, A=attribs
	ret	nc
	ld	bc,(file_attrmasks)	; get set/clear masks
	bit	0,b			; set archive?
	jr	z,file_setattr_noset_archive
	set	dirattr_arc,a
.file_setattr_noset_archive
	bit	0,c			; reset archive?
	jr	z,file_setattr_nores_archive
	res	dirattr_arc,a
.file_setattr_nores_archive
	bit	1,b			; set system?
	jr	z,file_setattr_noset_system
	set	dirattr_sys,a
	set	dirattr_hid,a
.file_setattr_noset_system
	bit	1,c			; reset system?
	jr	z,file_setattr_nores_system
	res	dirattr_sys,a
	res	dirattr_hid,a
.file_setattr_nores_system
	bit	2,b			; set r/o?
	jr	z,file_setattr_noset_ro
	set	dirattr_ro,a
.file_setattr_noset_ro
	bit	2,c			; reset r/o?
	jr	z,file_setattr_nores_ro
	res	dirattr_ro,a
.file_setattr_nores_ro
	ld	bc,direntry_attr
	add	hl,bc
	ld	(hl),a			; store new attribs
	jp	buf_writebuf		; update buffer and exit with any error
.file_setattr_wild
	call	file_setattr_1st	; change at least 1 file
	ret	nc			; exit if error
.file_setattr_loop
	ld	de,dir_filespec1
	call	dir_getnextmatch	; find next file
	ret	nc
	ret	nz			; exit with success if no more matches
	call	file_setattr_next	; change it
	ret	nc
	jr	file_setattr_loop


; ***************************************************************************
; * FAT_OPEN                                                                *
; ***************************************************************************
; Entry: B=file number
;        C=access mode
;        D=create action: 0=error, 1=create with header, 2=create without header
;        E=open action: 0=error, 1=open with header, 2=open, ignore header,
;                       3=rename to BAK & create, 4=erase & create
;        HL=filename (no wildcards)
; Exit: Fc=1 (success)
;       Fz=1 if file created, Fz=0 if existing file opened
;       Fc=0 (failure), A=error
; ABCDEHLIXIY corrupt

.file_open
	call	file_getunopenhandles	; file handle to IY
	ld	a,(iy+ffh_mode)
	and	a
	ld	a,rc_number
	ret	nz			; exit if file number already open
	push	bc			; save access mode
	push	iy			; save new file handle
	push	de			; save open and create actions
	ld	a,(file_number)
	ld	c,a			; C=filenumber
	ld	b,FATFSPKG_ID
	ld	a,fs_request
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_FS_FILE
	call	PACKAGE_CALL_PKG	; request the filenumber from IDEDOS
	ld	a,rc_number
	jp	nc,file_open_error	; exit if error
	ex	de,hl
	ld	hl,dir_filespec1
	call	dir_parsefilename	; parse filename and get handles
	jp	nc,file_open_error	; exit if error
	jp	nz,file_open_errwild	; or if wildcard present
	jp	p,file_open_errwild	; or if no filename part
	ld	de,dir_filespec1
	call	dir_getfirstmatch	; find file
	jp	nc,file_open_error	; exit if error
	pop	de			; restore open and create actions
	jr	nz,file_open_create	; if not found, follow create action	
.file_open_open
	inc	e
	dec	e
	ld	a,rc_exists
	jp	z,file_open_error2	; file exists for open action 0
	dec	e
	jr	z,file_open_open_hdr
	dec	e
	jr	z,file_open_open_nohdr
	dec	e
	jr	z,file_open_open_rename
	dec	e
	jr	nz,file_open_errparam	; bad parameter if action > 4
.file_open_open_delete
	push	de			; save create action
	call	file_delete_next	; delete the existing file
	jr	nc,file_open_error	; exit if error
	pop	de			; restore create action
	jr	file_open_create
.file_open_open_hdr
	pop	de			; DE=file handle
	call	file_fillin_handle	; fill in the file handle
	pop	bc			; C=access mode
	jr	nc,file_open_error3	; exit if error
	ld	b,0			; B="original" access mode
	call	file_validate_access	; check the access mode is okay
	jr	nc,file_open_error3	; exit if error
	jp	file_header_read	; read the header, if there is one
.file_open_open_nohdr
	pop	de			; DE=file handle
	call	file_fillin_handle	; fill in the file handle
	pop	bc			; C=access mode
	jr	nc,file_open_error3	; exit if error
	ld	b,0			; B="original" access mode
	call	file_validate_access	; check the access mode is okay
	jr	nc,file_open_error3	; exit if error
	ret
.file_open_open_rename
	push	de			; save create action
	ld	hl,dir_filespec1
	ld	de,dir_filespec2
	ld	bc,8
	ldir				; copy file name
	ld	hl,file_open_bak
	ld	bc,3
	ldir				; set extension to "BAK"
	ld	de,dir_filespec2
	call	file_delete_de		; delete any existing BAK file
	jr	c,file_open_open_rename2 ; okay if deleted
	cp	rc_nofile
	jr	nz,file_open_error	; or if no BAK file
.file_open_open_rename2
	call	file_rename_one		; rename file to BAK
	jr	nc,file_open_error	; exit if error
	pop	de			; restore create action
	; drop through to create action
.file_open_create
	inc	d
	dec	d
	ld	a,rc_nofile
	jr	z,file_open_error2	; file not found error for create action 0
	dec	d
	jr	z,file_open_create_hdr	; create action=1, create with header
	dec	d
	jr	nz,file_open_errparam	; bad parameter if action > 2
.file_open_create_nohdr
	call	file_create_entry	; create the file entry
	jr	nc,file_open_error2	; exit if error
	jr	file_open_open_nohdr	; rest is common with open
.file_open_create_hdr
	call	file_create_entry	; create the file entry
	jr	nc,file_open_error2	; exit if error
	pop	de			; DE=file handle
	call	file_fillin_handle	; fill in the file handle
	pop	bc			; C=access mode
	jr	nc,file_open_error3	; exit if error
	ld	b,0			; B="original" access mode
	call	file_validate_access	; check the access mode is okay
	jr	nc,file_open_error3	; exit if error
	jp	file_header_write	; write the header
.file_open_errparam
	ld	a,rc_badparam
	jr	file_open_error2
.file_open_errwild
	ld	a,rc_badname
.file_open_error
	pop	de			; restore registers
.file_open_error2
	pop	iy
	pop	bc
.file_open_error3
	and	a			; Fc=0, error
	jp	file_release		; release the file and exit

.file_open_bak
	defm	"BAK"


; ***************************************************************************
; * FAT_EXP_FS_SNAPDATA                                                     *
; ***************************************************************************
; Entry: B=filenumber, CDE=size required from current filepos
;        IY=sectorid buffer
; Exit: Fc=1 (success), IY=end of sectorid buffer+1
;       Fc=0 (failure), A=error
; ABCIXIY corrupt

.file_snapdata
	ld	(tmp_rwaddr),iy		; save buffer address
	call	file_gethandles		; handles to IY & IX
	push	bc			; save size needed
	push	de
	call	file_getvalidaddr	; get DE=offset into first sector
	pop	hl
	pop	bc			; CHL=size needed
	ld	b,(iy+ffh_sector)	; B=sector offset
	exx
	ld	c,(iy+ffh_cluster)
	ld	b,(iy+ffh_cluster+1)	; B'C'=cluster
	ld	e,(iy+ffh_sector)	; E'='sector offset
	exx
	ld	iy,(tmp_rwaddr)		; get buffer address
	call	PACKAGE_FS_SECBUF_INIT	; initialise the sector buffer
	push	hl
	ld	hl,512
	and	a
	sbc	hl,de
	ex	de,hl			; DE=bytes available in this sector
	pop	hl
	and	a
	sbc	hl,de
	ld	a,c
	sbc	a,0
	ld	c,a			; CHL=bytes needed after this sector, Fc=1 or Fz=1 if no more needed
	jr	c,file_sd_1sector
	or	h
	or	l
	jr	z,file_sd_1sector
	ld	de,511
	add	hl,de
	ld	a,c
	adc	a,0			; AHL=bytes needed+511
	srl	a
	rr	h
	ld	l,h
	ld	h,a			; HL=(bytes needed+511)/512=sectors needed
	ld	a,(ix+fph_clussize)
	sub	b
	dec	a
	ld	d,1			; sectors to read in first cluster
	jr	z,file_sd_firstcount
	ld	e,a			; E=number of sectors left in first cluster
.file_sd_findfirst
	ld	a,h
	or	l
	jr	z,file_sd_firstcount
	dec	hl
	inc	d
	dec	e
	jr	nz,file_sd_findfirst
.file_sd_firstcount			; ATP, D=sectors from first cluster, HL=sectors left
	ld	a,d
	exx
	push	bc			; save cluster
	push	af			; save sectors
	ld	a,e			; offset in 1st sector
	call	clus_getsector		; get BCDE=sector number
	pop	af			; A=sector count
	call	PACKAGE_FS_SECBUF_ADD	; add to sectorid buffer
	exx				; now HL=sectors left
.file_sd_clusloop
	pop	bc			; restore cluster
	ld	a,h
	or	l
	jr	z,file_sd_finish	; finish up if no more
	push	hl
	call	fat_readfat		; next cluster in chain
	pop	hl
	ld	a,(ix+fph_clussize)
	ld	e,a
	ld	d,0
	and	a
	sbc	hl,de			; subtract available sectors
	jr	nc,file_sd_notlast
	add	hl,de
	ld	a,l			; use sectors needed
	ld	l,0
.file_sd_notlast
	push	bc
	push	hl
	push	af			; save sectors
	xor	a			; offset zero
	call	clus_getsector		; BCDE=sector number
	pop	af			; restore sectors
	call	PACKAGE_FS_SECBUF_ADD	; add to sectorid buffer
	pop	hl
	jr	file_sd_clusloop
.file_sd_1sector
	ld	d,1			; everything in 1st sector
	ld	hl,0
	jr	file_sd_firstcount
.file_sd_finish
	call	PACKAGE_FS_SECBUF_FINISH	; terminate buffer
	scf				; success!
	ret


; ***************************************************************************
; * Subroutine to fill in filehandle for current directory entry            *
; ***************************************************************************
; Entry: IY=directory handle, pointing to desired entry
;	 IX=partition handle
;	 DE=file handle address
; Exit: Fc=1 (success), IY=file handle, filled in (no mode set)
;       Fc=0 (failure), A=error
; ABCDEHL corrupt

.file_fillin_handle
	push	iy
	pop	hl			; HL=dir handle
	push	de
	pop	iy			; IY=file handle
	ld	bc,fdh_size
	ldir				; copy dir handle to file handle
	ld	h,d
	ld	l,e
	inc	de
	ld	(hl),0
	ld	bc,ffh_size-fdh_size-1
	ldir				; zeroise rest of file handle
	call	dir_getentry		; HL=directory entry
	ret	nc			; exit if error
	ld	bc,direntry_cluster
	add	hl,bc			; HL=add of start cluster, filesize
	ex	de,hl
	push	iy
	pop	hl
	ld	bc,ffh_clusstart
	add	hl,bc
	ex	de,hl			; DE=address in file handle
	ld	bc,2+4
	ldir				; copy start cluster, filesize
	; drop through into next subroutine

; ***************************************************************************
; * Subroutine to copy header data to IDEDOS bank                           *
; ***************************************************************************
; Also sets the address of the header data in ffh_idedos_hdr.
; Entry: IY=filehandle
; Exit: Fc=1 (success)
; ABCDEHL corrupt

.file_set_idedos_header
	push	ix
	push	iy			; save filehandle
	ld	e,(iy+ffh_header+0)	; fetch to DE/HL/IX/IY
	ld	d,(iy+ffh_header+1)
	ld	l,(iy+ffh_header+2)
	ld	h,(iy+ffh_header+3)
	ld	c,(iy+ffh_header+4)
	ld	b,(iy+ffh_header+5)
	push	bc
	pop	ix
	ld	c,(iy+ffh_header+6)
	ld	b,(iy+ffh_header+7)
	push	bc
	pop	iy
	ld	b,FATFSPKG_ID
	ld	a,(file_number)
	ld	c,a			; C=file number
	ld	a,fs_puthead
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_FS_FILE		; set the header data, get address
	call	PACKAGE_CALL_PKG
	pop	iy			; restore file handle
	pop	ix
	ld	(iy+ffh_idedos_hdr),l	; save address in IDEDOS
	ld	(iy+ffh_idedos_hdr+1),h
	scf				; success!
	ret


; ***************************************************************************
; * Subroutine to create directory entry for new file                       *
; ***************************************************************************
; Entry: IY=directory handle, IX=partition handle
;	 dir_filespec1 contains 11-byte filename (must not exist in directory)
; Exit: Fc=1 (success), current entry is now filled in
;       Fc=0 (failure), A=error
; ABCDEHLIX corrupt

.file_create_entry
	call	dir_getfree		; HL=free directory entry
	ret	nc			; exit if error
	ex	de,hl
	ld	hl,dir_filespec1
	ld	bc,11
	ldir				; copy in name and extension
	ld	h,d
	ld	l,e
	xor	a
	ld	(hl),a			; zero attributes
	ld	c,10+2+2
	inc	de
	ldir				; zero unused, time and date
	ex	de,hl
	ld	(hl),$ff		; start cluster=$ffff
	inc	hl
	ld	(hl),$ff
	inc	hl
	ld	(hl),a			; zero filesize
	inc	hl
	ld	(hl),a
	inc	hl
	ld	(hl),a
	inc	hl
	ld	(hl),a
	jp	buf_writebuf		; write directory entry


; ***************************************************************************
; * Subroutine to get +3DOS file header sector for current file             *
; ***************************************************************************
; Entry: IY=file handle
; Exit: HL=buffer address,Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIX corrupt

.file_header_sector
	ld	h,0
	ld	l,h
	ld	d,h
	ld	e,h
	call	file_setpos_withhandles	; home to start of file
	jp	file_getvalidaddr	; HL=address of buffer, DE=0


; ***************************************************************************
; * Subroutine to calculate +3DOS file header checksum                      *
; ***************************************************************************
; Entry: HL=address of 128-byte header
; Exit: HL=address of header checksum
;	A=calculated checksum
; B corrupt

.file_header_checksum
	xor	a
	ld	b,$7f
.file_header_checksum_loop
	add	a,(hl)			; form checksum
	inc	hl
	djnz	file_header_checksum_loop
	ret

	
; ***************************************************************************
; * Subroutine to write +3DOS file header for current file                  *
; ***************************************************************************
; Entry: IY=file handle
; Exit: Filepointer is positioned after 128-byte header
;       Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIX corrupt

.file_header_write
	ld	d,0
	ld	e,d
	ld	h,d
	ld	l,127
	call	file_setpos_withhandles
	call	file_maximisefilesize	; ensure at least 128 bytes in file
	call	file_header_sector	; HL=header address
	ret	nc
	set	fm_header,(iy+ffh_mode) ; flag header present
	push	hl			; save address
	ex	de,hl
	ld	hl,file_header_signature
	ld	bc,file_header_signature_end-file_header_signature
	ldir				; copy signature
	push	de			; save destination
	push	iy			; save filehandle
	ld	b,FATFSPKG_ID
	ld	a,(file_number)
	ld	c,a			; C=file
	ld	a,fs_gethead
	exx
	ld	b,PKG_IDEDOS
	ld	hl,IDE_FS_FILE		; get header from IDEDOS to DE/HL/IX/IY
	call	PACKAGE_CALL_PKG
	push	iy
	pop	bc			; now DE/HL/IX/BC
	pop	iy			; restore filehandle
	ld	(iy+ffh_header+0),e
	ld	(iy+ffh_header+1),d
	ld	(iy+ffh_header+2),l
	ld	(iy+ffh_header+3),h
	push	ix
	pop	de
	ld	(iy+ffh_header+4),e
	ld	(iy+ffh_header+5),d
	ld	(iy+ffh_header+6),c
	ld	(iy+ffh_header+7),b
	pop	de			; restore destination
	push	iy
	pop	hl			; HL=filehandle
	ld	bc,ffh_filesize
	add	hl,bc
	ld	bc,4+8
	ldir				; copy file size & 8-byte header
	ld	h,d
	ld	l,e
	inc	de
	ld	(hl),0
	ld	bc,103
	ldir				; zeroise rest of header to checksum
	pop	hl
	call	file_header_checksum	; calculate checksum
	ld	(hl),a			; store checksum
	call	buf_writebuf		; write header
	ld	d,0
	ld	e,d
	ld	h,d
	ld	l,128
	jp	file_setpos_withhandles	; set position after header
	
.file_header_signature
	defm	"PLUS3DOS",$1a,$01,$00
.file_header_signature_end


; ***************************************************************************
; * Subroutine to read +3DOS file header for current file                   *
; ***************************************************************************
; Entry: IY=file handle
; Exit: Filepointer is positioned after 128-byte header, if there was one
;       Fc=1 (success)
;       Fc=0 (failure), A=error
; ABCDEHLIX corrupt

.file_header_read
	res	fm_header,(iy+ffh_mode) ; flag no header
	ld	a,(iy+ffh_filesize)
	and	$80
	or	(iy+ffh_filesize+1)
	or	(iy+ffh_filesize+2)
	or	(iy+ffh_filesize+3)	; A=non-zero if size >= 128 bytes
	scf
	ret	z			; file not big enough for header
	call	file_header_sector	; HL=header address
	ret	nc
	push	hl
	call	file_header_checksum	; calculate checksum
	cp	(hl)			; compare with stored checksum
	pop	de			; DE=header address
	scf
	ret	nz			; invalid checksum
	ld	hl,file_header_signature
	ld	b,file_header_signature_end-file_header_signature
.file_header_sigcheck
	ld	a,(de)
	cp	(hl)
	scf
	ret	nz			; bad signature
	inc	de
	inc	hl
	djnz	file_header_sigcheck
	set	fm_header,(iy+ffh_mode) ; flag header present
	push	iy
	pop	hl
	ld	bc,ffh_filesize
	add	hl,bc			; HL=address of filehandle size
	ld	bc,4+8
	ex	de,hl
	ldir				; copy file size & 8-byte header
	call	file_set_idedos_header	; copy to IDEDOS
	ld	d,0
	ld	e,d
	ld	h,d
	ld	l,128
	jp	file_setpos_withhandles	; move filepointer past header
