/*
	decode/encode CPS-2 roms into/from raw 32x32 4bpp bitmaps
	iq_132 - 6/11/20
*/

#include <stdio.h>

struct filenames
{
	char name[128];
	unsigned int offset;
	unsigned int length;
};

static struct filenames filetable[32];
static char bitmap_name[128];
static int load_mode = 0;
static unsigned int palette[16];
static int user_palette = 0;

static void dump_bitmap(unsigned char *tmp, int len, int tilesize)
{
#define SET_FILE_SIZE(x)	\
	bmp_data[2] = (x);	\
	bmp_data[3] = (x)>>8;	\
	bmp_data[4] = (x)>>16;	\
	bmp_data[5] = (x)>>24

#define SET_BITMAP_SIZE(x)	\
	bmp_data[0x22] = (x);	\
	bmp_data[0x23] = (x)>>8;	\
	bmp_data[0x24] = (x)>>16;	\
	bmp_data[0x25] = (x)>>24

#define SET_BITMAP_WIDTH(x)	\
	bmp_data[0x12] = (x);	\
	bmp_data[0x13] = (x)>>8;	\
	bmp_data[0x14] = (x)>>16;	\
	bmp_data[0x15] = (x)>>24

#define SET_BITMAP_HEIGHT(x)	\
	bmp_data[0x16] = (x);	\
	bmp_data[0x17] = (x)>>8;	\
	bmp_data[0x18] = (x)>>16;	\
	bmp_data[0x19] = (x)>>24

	unsigned char bmp_data[0x36] = {
		0x42, 0x4D, 					// 'BM' (leave alone)
		0x00, 0x00, 0x00, 0x00, 		// file size
		0x00, 0x00, 0x00, 0x00, 		// padding
		0x36+(16*4), 0x00, 0x00, 0x00,	// offset to data (leave alone)
		0x28, 0x00, 0x00, 0x00,			// windows mode (leave alone)
		0x00, 0x00, 0x00, 0x00,			// bitmap width
		0x00, 0x00, 0x00, 0x00,			// bitmap height
		0x01, 0x00,						// planes (1) always!
		0x04, 0x00, 					// bits per pixel (let's do 4 so no conversion!)
		0x00, 0x00, 0x00, 0x00,			// compression (none)
		0x00, 0x00, 0x00, 0x00,			// size of bitmap data
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00
	};

	unsigned char palette_data[16 * 4];

	// Fake palette (16 24-bit colors)
	int i;
	for (i = 0; i < 16; i++)
	{
		int r = (i & 1) ? 0xff : 0;
		int g = (i & 2) ? 0x7f : 0;
		int b = (i & 4) ? 0xff : 0;
		
		if (i & 8) {
			g |= 0x80;
		}

		palette_data[i * 4 + 0] = user_palette ? (palette[i] >>  0) : b;
		palette_data[i * 4 + 1] = user_palette ? (palette[i] >>  8) : g;
		palette_data[i * 4 + 2] = user_palette ? (palette[i] >> 16) : r;
		palette_data[i * 4 + 3] = 0; // must be 0
	}

	unsigned char *tmp2 = (unsigned char*)malloc(len);
	
	int wide = 0x1000;
	int high = len / 0x1000;

	{
		int x,y,xx,yy,c=0;
		for (y = 0; y < high; y+=tilesize)
		{
			for (x = 0; x < wide; x+=tilesize,c++)
			{
				int offs = (y/tilesize)*(wide/tilesize)+(x/tilesize);
	
				unsigned char *src = tmp + offs * tilesize * tilesize;
				unsigned char *dst = tmp2 + y * wide + x;

				for (yy = 0; yy < tilesize; yy++)
				{
					for (xx = 0; xx < tilesize; xx++)
					{
						dst[xx] = src[xx];			
					}

					src += tilesize;
					dst += wide;
				}
			}
		}
	}
	
	// pack data (2 pixels / byte)
	for (i = 0; i < len / 2; i++)
	{
		int o = (len - 2) - (i * 2);
		tmp[i] = (tmp2[o + 1] << 4) | (tmp2[o + 0] & 0xf);
	}

	SET_FILE_SIZE((len/2)+(16*4)+0x36); // pixel data + palette data + header
	SET_BITMAP_SIZE(len/2);
	SET_BITMAP_WIDTH(wide);
	SET_BITMAP_HEIGHT(high);
	
	FILE *fa = fopen(bitmap_name, "wb");
	fwrite (bmp_data, 0x36, 1, fa);
	fwrite (palette_data, 16*4, 1, fa);
	fwrite (tmp, len/2, 1, fa);
	fclose (fa);

	free (tmp2);
}

static int get_bitmap(unsigned char *tmp, int len, int tilesize)
{
	unsigned char *tmp2 = (unsigned char*)malloc(len);
	unsigned char bmp_data[0x36];

	FILE *fa = fopen(bitmap_name, "rb");
	if (fa == NULL) {
		printf ("ERROR: Failed to open %s\n", bitmap_name);
		return 1;
	}

	unsigned int tlen = 0;

#if 0 // verify file size - doesn't work?? weird
	unsigned char palette_data[16 * 4];
	fseek (fa, 0, SEEK_END);
	tlen = (ftell(fa) * 2); // - (0x36 + (16*4));
	rewind (fa);
	fseek (fa, 0x36 + (16*4), SEEK_SET);
#else
	fread (bmp_data, 0x36, 1, fa);
	tlen = (bmp_data[0x22] | (bmp_data[0x23] << 8) | (bmp_data[0x24] << 16) | (bmp_data[0x25] << 24)) * 2;
	fseek (fa, 16*4, SEEK_CUR); // skip palette data
#endif

	if (tlen != len) {
		printf ("ERROR! Bitmap size (%d) does not match expanded rom size (%d)!\n", tlen, len);
		return 1;
	}

	fread (tmp, len/2, 1, fa);
	fclose (fa);

	int wide = 0x1000;
	int high = len / 0x1000;

	int i;
	// unpack data (2 pixels / byte) -> 1 pixel / byte
	for (i = 0; i < len / 2; i++)
	{
		tmp2[(len - 2) - (i * 2) + 0] = tmp[i] & 0xf;
		tmp2[(len - 2) - (i * 2) + 1] = tmp[i] >> 4;
	}

	{
		int x,y,xx,yy,c=0;
		for (y = 0; y < high; y+=tilesize)
		{
			for (x = 0; x < wide; x+=tilesize,c++)
			{
				int offs = (y/tilesize)*(wide/tilesize)+(x/tilesize);
	
				unsigned char *src = tmp + offs * tilesize * tilesize;
				unsigned char *dst = tmp2 + y * wide + x;

				for (yy = 0; yy < tilesize; yy++)
				{
					for (xx = 0; xx < tilesize; xx++)
					{
						src[xx] = dst[xx];			
					}

					src += tilesize;
					dst += wide;
				}
			}
		}
	}

	free (tmp2);

	return 0;
}

static void gfx_decode(unsigned char *rom, int gfxlen) // unshuffle and decode
{
	int i,j,k;
	unsigned char *tmp = (unsigned char*)malloc(gfxlen);
	
	memcpy (tmp, rom, gfxlen);
	memset (rom, 0, gfxlen * 2);
	
	for (i = 0; i < gfxlen*8; i++)
	{
		j = ((((i >> 3) & 0x1ffff0) >> 1) | (((i >> 3) & 8) << 17) | ((i >> 3) & 0xffe00007)) << 3; // step 1 unshuffle

		k = (((i ^ 7) & 7) << 3) | ((j & 0x18) >> 3) | ((j & ~0x1f) << 1); // step 2 decode

		rom[k/8] |= ((tmp[i/8] >> (i & 7)) & 1) << (k & 7);	
	}
}

static void gfx_encode(unsigned char *rom, int gfxlen) // encode and shuffle
{
	int i,j,k;
	unsigned char *tmp = (unsigned char*)malloc(gfxlen);
	
	memcpy (tmp, rom, gfxlen);
	memset (rom, 0, gfxlen / 2);
	
	for (i = 0; i < gfxlen*8; i++)
	{
		k = ((i & 3) << 3) | (((i >> 3) & 7) ^ 7) | ((i & 0xffffffc0) >> 1); // step 1 encode

		j = (((k >> 3) << 1) & 0x1ffff0) | (((k >> 3) >> 17) & 8) | ((k >> 3) & 0xffe00007); // step 2 shuffle

		rom[j] |= ((tmp[i/8] >> (i & 7)) & 1) << (k & 7);	
	}
}

static int copy_until(char *ptr, char *output, char limit)
{
	char *iptr = ptr;

	int i = 0;
	while (ptr[0] != limit && ptr[0] != ' ' && ptr[0] != ';' && ptr[0] != 0)
	{
		output[i] = ptr[0];
		i++, ptr++;
	}
	output[i] = 0;
	ptr++;

	return ptr - iptr;
}

static int parse_userpalette()
{
	user_palette = 0;

	FILE *fa = fopen("user_palette.txt", "rt");
	if (fa == NULL) {
		return 1;
	}

	user_palette = 1;

	int entry = 0;
	char line[128], r[3], g[3], b[3];

	while (1)
	{
		if (fgets (line, 128, fa) == NULL) break;

		if (strncmp(line, "rgb(", strlen("rgb(")) != 0) continue; // if not rgb line continue

		int hexmode = 0;

		char *ptr = line + strlen("rgb(");

		if (ptr[0] == 'x' || ptr[1] == 'x') {
			user_palette = 0;
			printf ("Warning! User-defined palette has hexdecimal values, must be decimal!\n");
			return 1;
		}

		ptr += copy_until(ptr, r, ','); // r
		ptr += copy_until(ptr, g, ','); // g
		ptr += copy_until(ptr, b, ')'); // b

		palette[entry]  = (strtol(r, NULL, 10) & 0xff) << 16;
		palette[entry] |= (strtol(g, NULL, 10) & 0xff) <<  8;
		palette[entry] |= (strtol(b, NULL, 10) & 0xff) <<  0;

		entry++;
	}

	if (entry < 16) {
		printf ("Warning! Found user-defined palette! Has %d entries, needs 16!\n", entry);
		user_palette = 0; // not enough entries! fall back to default!
	}

	return 0;
}

static int parse_filelist(char *filename, int function)
{
	FILE *fa = fopen(filename, "rt");
	if (fa == NULL) {
		printf ("ERROR: Failed to read %s\n", filename);
		return 1;
	}

	char line[256];
	char offset[32];
	char length[32];

	struct filenames *sptr = &filetable[0];

	while (1)
	{
		if (fgets (line, 256, fa) == NULL) break;

		if (line[0] == ';') continue;
		if (strlen(line) < 3) continue;

		line[strlen(line)-1] = 0;

		char *ptr = line;

		if (strncmp(ptr, "bitmap:", strlen("bitmap:")) == 0) {
			ptr += strlen("bitmap:");
			strcpy(bitmap_name, ptr);
	//		printf ("bitmap -> %s\n", bitmap_name);
			continue;
		}

		ptr += copy_until(ptr, sptr->name, ',');
		ptr += copy_until(ptr, offset, ',');
		ptr += copy_until(ptr, length, 0);

		sptr->offset = (unsigned int)strtol(offset, NULL, 16);
		sptr->length = (unsigned int)strtol(length, NULL, 16);

		load_mode = (load_mode | sptr->offset) & 1;

	//	printf ("%s->%7.7x->%x\n", sptr->name, sptr->offset, sptr->length);
		sptr++;
	}

	if (strlen(bitmap_name) < 3) {
		printf ("ERROR bitmap name not found!\n");
		return 1;
	}

	return 0;
}

int main(int argc, char **argv)
{
	if (argc != 4) {
		printf ("Usage:\n");
		printf ("decode o s descriptor_file\n");
		printf ("o can be e (encode) or d (decode)\n");
		printf ("s is tile size (16 or 32) [16x16 or 32x32]\n");
		printf ("descriptor_file is the name of the input file\n");
		printf ("containing listing of rom names,size,offsets\n");
		printf ("and bitmap name.\n");
		printf ("Example:\n");
		printf ("\tdecode d 16 ssf2t.txt\n");
		return 1;
	}

	if (tolower(argv[1][0]) != 'e' && tolower(argv[1][0]) != 'd') {
		printf ("ERROR! Invalid processing option (%s)! Must be 'e' or 'd'!\n", argv[1]);
		return 1;
	}

	if (argv[2][0] != '1' && argv[2][0] != '3') {
		printf ("ERROR! Invalid tile size (%s)! must be '16' or '32'!\n", argv[2]);
		return 1;
	}

	if (strlen(argv[3]) < 5) {
		printf ("Warning! Possible descriptor file name (%s)!\n", argv[3]);
	}

	memset (filetable, 0, sizeof(filetable));

	int function = 0; // 0 - decode, 1 = encode
	int tilesize = 32; // 32 or 16
	int tmpsize = 0;

	if (argv[1][0] == 'e') function = 1; // encode
	if (argv[2][0] == '1') tilesize = 16; // 16x16 or 32x32

	if (parse_filelist(argv[3], function)) return 1;

	struct filenames *sptr = &filetable[0];

	while (sptr->name[0] != 0)
	{
		int tsize = (sptr->offset & ~0xf) + (sptr->length * (load_mode ? 8 : 4));
		if (tsize > tmpsize) tmpsize = tsize;
		sptr++;
	}

	printf ("%scoding %dx%d tiles - size: %d!\n", function ? "En" : "De", tilesize, tilesize, tmpsize);

	unsigned char *tmp = malloc(tmpsize * 2);
	if (tmp == NULL) {
		printf ("ERROR: Failed to allocate %d bytes!\n", tmpsize * 2);
		return 0;
	}

	if (function == 0) // decode
	{
		sptr = &filetable[0];

		while (sptr->name[0] != 0)
		{
			FILE *fa = fopen(sptr->name, "rb");
			if (fa == NULL) {
				printf ("ERROR: Unable to read %s!\n", sptr->name);
				return 0;
			}

			printf ("Reading: %s, size: %6.6x, offset: %7.7x, bytes before gap: %d\n", sptr->name, sptr->length, sptr->offset, load_mode ? 1 : 2);

			int i;
			int size = sptr->length;
			int offset = 0;
			unsigned char *rom = (unsigned char*)malloc(sptr->length);

			fread (rom, sptr->length, 1, fa);
			fclose (fa);

			for (i = 0; i < size; i += load_mode ? 1 : 2)
			{
				memcpy (tmp + sptr->offset + offset, rom + i, load_mode ? 1 : 2);
			//	fread (tmp + sptr->offset + offset, load_mode ? 1 : 2, 1, fa);
				offset += 8;
			}

			free (rom);

			sptr++;
		}

		parse_userpalette();
		gfx_decode(tmp, tmpsize);
		dump_bitmap(tmp, tmpsize * 2, tilesize); // destroys tmp data!
	}
	else // encode
	{
		if (get_bitmap(tmp, tmpsize * 2, tilesize)) return 1;
		gfx_encode(tmp, tmpsize * 2);

		sptr = &filetable[0];

		while (sptr->name[0] != 0)
		{
			char tname[128];
			sprintf (tname, "%s_modified", sptr->name);

			FILE *fa = fopen(tname, "wb");

			printf ("Writing: %s, size: %6.6x, offset: %7.7x, bytes before gap: %d\n", tname, sptr->length, sptr->offset, load_mode ? 1 : 2);

			int size = sptr->length;
			int offset = 0;
			while (size)
			{
				fwrite (tmp + sptr->offset + offset, load_mode ? 1 : 2, 1, fa);
				offset += 8;
				size -= load_mode ? 1 : 2;
			}
			fclose (fa);
			sptr++;
		}
	}

	free (tmp);

	return 1;
}

