;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;*****************************************************;;;
;;;*              ,-----------,                        *;;;
;;;*              |_______   /                         *;;;
;;;*    Z.asm            /  /                          *;;;
;;;*    Tiny Spectrum48K/128K emulator for PC (DOS)    *;;;
;;;*                   /  /                            *;;;
;;;*    Copyright (c) Vladimir Kladov, 2003            *;;;
;;;*                 /  /                              *;;;
;;;*    Build with the provided makefile: make -B      *;;;
;;;*               /  /-------,                        *;;;
;;;*              /___________|                        *;;;
;;;*                                                   *;;;
;;;*****************************************************;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; started: 15-Sep-2003
; mainly ready: 9-Oct-2003 (most 48K sna-files work good)
; 10-Oct-2003:  F2 - save, F3 - Load, F4 - reload, F1 - help
; 11-Oct-2003:  128K support, renamed to Z (previous name was 48.com)
; 12-Oct-2003:  SNA extended support, careful INT 23 tacts
; 13-Oct-2003:  F1 - show keybd in help
; 15-Oct-2003:  Tables compression - started
; 18-Oct-2003:  Compression - finished
; 20-Oct-2003:  select file from pop-up list on F3
; 26-Oct-2003:  simple DAA
; 25-Nov-2003:  fix: careful TCounter change for JR_cond, JP_cond
; todo:
;       * on OUT @dest -> int XX (AY,joystick,other)
;----------------------------------------------------------

        TITLE Z
        .model small
        .code
        .486
        org 100h
        assume ds:@code

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;               BUILD OPTIONS
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
SPECTRUM_128K   EQU 1 ; support of Spectrum128K (port 7FFD, RAM/ROM banks)
                      ; if 0, Spectrum-48K only supported.
OPTION_SILENCE  EQU 1 ; recognize silence option /s in command line
EXTRA_KEYBOARD  EQU 1 ; keys ',', '.', '/', '-', '+', '\', BackSpace, ';', "'", '*'
HELP_ON_F1      EQU 1 ; F1 shows help screen
COPYRIGHT       EQU 1 ; show copyright in help
HELP_SHOW_KBD   EQU 1 ; Spectrum keyboard is shown on F1 too
HELP_BOLD_CHARS EQU 1 ; Bold characters
SAVE_ON_F2      EQU 1 ; F2 saves to Z.SNA
LOAD_ON_F3      EQU 1 ; F3 loads from Z.SNA (if it exists)
LOAD_SELECT     EQU 1 ; F3 pop-up file list to load
RELOAD_ON_F4    EQU 1 ; F4 reloads initially loaded file (if any)
SNA_EXTENDED    EQU 1 ; recognize extended SNA-file with size > 49179 bytes
CAREFUL_INT     EQU 1 ; careful INT signal timing (23 tacts?)
COMPRESS_TABLES EQU 1 ; compress tables

SIMPLE_DAA      EQU 1 ; use simple DAA emulation
SEPARATE_INT9   EQU 1 ; Separate procedure Set_Restore_INT9
                      ; (bigger a bit, but necessary for LOAD_SELECT)

Dark    EQU     47
Lght    EQU     63

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;               DEBUG BUILD OPTIONS
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
LOGS      EQU 0 ; see log.48 created (if 0 all other log options have no effect)
LOGBUFFER EQU 1 ; using buffer (fast)
TRACE_ALL EQU 1 ; trace instructions
TRACEREGS EQU 1 ; see traced registers there
LOG_Bank_Init           EQU 0 ; log build Bank_Access_Table
LOG_MEMORY              EQU 0 ; Log memory word
LOG_MEM_ADDR            EQU 0000h ; Address of word to log
LOG_end_pixel_line      EQU 0 ; log each pixel line end
LOG_OPcode_ONint        EQU 0 ; log op code when int signalled
LOG_Reason_Int          EQU 0 ; log interrupt reason
LOG_Key_input           EQU 0 ; log scan-code received
LOG_Screen_CC           EQU 0 ; log screen cyclic check sum (very slow!)
LOG_DAA_Table           EQU 0 ; log DAA table 2 initialization results
DEBG_NOINT9             EQU 0 ; while debug, do not intercept int9 vector
LOG_DECOMPRESS_TABLES   EQU 0 ; log decompression while debugging it
LOG_F3_SELECT           EQU 0 ; log file select details

start:
        ;======================================================
        ; INITIALIZATION
        ;======================================================

IF      LOGS
        mov     dx, offset[@Log_file_name]
        push    dx
        mov     ah, 41h
        int     21h ; delete log file
        pop     dx
        mov     ah, 3Ch
        mov     cx, 20h
        int     21h ; create file
        xchg    bx, ax
        mov     ah, 3Eh
        int     21h ; close file handle

        call    LogText
        DB      'CS=',0
        mov     ax, cs
        call    LogAX
        call    LogText
        DB      13,10,0
ENDIF

        pop     ax
        mov     bx, offset[Stack_top]
        and     bx, not 15
        mov     sp, bx
        shr     bx, 4
        push    ax
        mov     ah, 4Ah
        int     21h
        jnc short Memory_relocated_OK
        ;------------------------------------------------------
        ; error: No memory
        ;------------------------------------------------------
Error_No_memory:
        call    ShowError
        DB      'no memory',13,10,'$'

Memory_relocated_OK:
        ;======================================================
        ; Setup interrupt vertors (1C = timer, 09 = keyboard)
        ;======================================================
TimerVector     EQU 1Ch
KeybdVector     EQU 9

IF DEBG_NOINT9
ELSE
IF SEPARATE_INT9 or LOAD_ON_F3 and LOAD_SELECT ; HELP_ON_F1
        ;------ read old vector 9 value AND
        ;------ set new keyboard vector value:
        mov     word ptr [OldKeybVector+2], cs
        call    Set_Restore_INT9
ELSE
        ;------ read old vector 9 value:
        mov     ax, 3500h + KeybdVector   ; ah:35H function(get int vector); al:vector
        int     21h
        push    bx
        push    es
        ;------ set new keyboard vector value:
        mov     ax, 2500h + KeybdVector   ; aj:25H function(set int vector); al:vector
        push    cs
        pop     ds
        mov     dx, offset[Keyboard_Proc]
        int     21h
ENDIF ; HELP_ON_F1
ENDIF

        ;------ read old vector 1C value:
        mov     ax, 3500h + TimerVector   ; ah:35H function(get int vector); al:vector
        int     21h
        push    bx
        mov     ax, es
        push    ax
        ;------ set new timer vector value:
        mov     ax, 2500h + TimerVector   ; ah:25H function(set int vector); al:vector
        mov     dx, offset[Timer_Tick_Proc]
        int     21h

        call    Main_Proc

        ;------ restore old timer vector value:
        pop     ds
        pop     dx
        mov     ax, 2500h + TimerVector
        int     21h

IF DEBG_NOINT9
        ret
ELSE
IF SEPARATE_INT9 or LOAD_ON_F3 and LOAD_SELECT ; HELP_ON_F1
        ;------ restore old keyboard vector value and terminate execution:

Set_Restore_INT9:
        push    bx
        push    es ;-----------------------------\
        mov     ax, 3500h + KeybdVector
        int     21h
        mov     dx, es
        pop     es ;-----------------------------/
        xchg    dx, word ptr cs:[OldKeybVector+2]
        push    ds ;-----------------------------\
        mov     ds, dx
        xchg    dx, bx
        xchg    dx, word ptr cs:[OldKeybVector]
        ;mov     ax, 2500h + KeybdVector
        mov     ah, 25h
        int     21h
        pop     ds ;-----------------------------/
        pop     bx
        ret

OldKeybVector: DW Keyboard_Proc, 0
ELSE
        ;------ restore old keyboard vector value:
        pop     ds
        pop     dx
        mov     ax, 2500h + KeybdVector
        int     21h

        ;------ terminate execution
        ;int     20h
        ret
ENDIF ; HELP_ON_F1
ENDIF

;##############################################################
;  WAIT KEY PROCEDURE
;##############################################################
WaitKey proc
        mov     ah, 0
        add     ah, ah
        jc short WaitKey
        shr     ah, 1
        cmp     ah, 60h
        jz short WaitKey
        ret
WaitReleaseKey:
        mov     byte ptr [WaitKey+1], 0
@WaitReleaseKey_1:
        mov     ah, byte ptr [WaitKey+1]
        add     ah, ah
        jnc short @WaitReleaseKey_1
@Wait_2:
        shr     ah, 1
        cmp     ah, 60h
        jz short WaitReleaseKey
        ret
WaitKey endp ;#################################################

;##############################################################
;  KEYBOARD PROCEDURE
;##############################################################
Keyboard_Proc proc
        pushf
        push    ax
        push    bx
        push    dx

        mov     dx, 60h
        in      al, dx
        movzx   bx, al ; bx = scan code received, bit 7 = 1 if release key

IF HELP_ON_F1 or LOAD_ON_F3 and LOAD_SELECT
        mov     byte ptr [WaitKey+1], al
ENDIF

        ;------ acknowledge received key:
        mov     al, 20h
        out     20h, al

IF LOGS and LOG_Key_input
        mov     word ptr cs:[log_key_load+1], bx
        mov     byte ptr cs:[log_key_skip+1], 0
ENDIF
        ;------ handle received key scan-code:
        add     bl, bl
        mov     bx, word ptr cs:[Keybd_Table+bx]
        lahf    ; ah bit 1 set if release key
        cmp     bh, 0
        jge short @key_call_bx
        ;------ change Spectrum keyboard byte [bl] state using al as a mask
IF EXTRA_KEYBOARD
        mov     dl, bl
ENDIF
        test    bl, 20h
Ext_Key_check:
        jmp short @not_after_E0
        mov     bh, 0FFh ; ignore key LSHIFT following E0 prefix
@not_after_E0:
        mov     byte ptr cs:[Ext_Key_check], 0EBh ; put "jmp" instruction there

        and     bl, 1Fh
        mov     al, bh
        mov     bh, 0
        call    Update_Keybd_State
IF EXTRA_KEYBOARD
        mov     bl, 7
        mov     al, not 2
        add     dl, dl
        jc short @caps_keys
@no_sym_shift:
        mov     bl, 0
        mov     al, not 1
        add     dl, dl
        jnc short @no_caps
@caps_keys:
        call    Update_Keybd_State
@no_caps:
ENDIF
        jmp short @key_end
@key_call_bx:
        test    ah, 1
        jz      @key_end ; call [bx] only when key is released
        call    bx
@key_end:

        pop     dx
        pop     bx
        pop     ax
        popf
        iret

Keyboard_Proc endp ;###########################################

Key_Esc:        ; exits program
        mov     byte ptr cs:[Cont_loop], 0C3h ; write 'ret' command there
IF HELP_ON_F1
ELSE
_Help:
ENDIF
IF SAVE_ON_F2
ELSE
_Save:
ENDIF
IF LOAD_ON_F3
ELSE
_Load:
ENDIF
IF RELOAD_ON_F4
ELSE
;_Reload:
ENDIF
_Reset:
Key_No: ret

_Sound: ;******************************************************
        ; Toggle sound on/off - F5 on keyboard
        ;******************************************************
        xor     byte ptr cs:[Sound_Jmp], 74h xor 0EBh ; toggle between jmp and jnz
        ret

KeyExt: ;******************************************************
        ; Extended key flag E0 received, next key is extended
        ;******************************************************
        mov     byte ptr cs:[Ext_Key_check], 74h ; put "jz" instruction there
        ret

IF SAVE_ON_F2
_Save:  ;******************************************************
        ; Save Z.SNA file when F2 pressed
        ;******************************************************
        mov     byte ptr cs:[Skip_Save_Sna+1], 0
        ret
ENDIF

IF LOAD_ON_F3
_Load:  ;******************************************************
        ; Load from Z.SNA (if it exists)
        ;******************************************************
IF LOAD_SELECT
        mov     byte ptr cs:[Skip_Load_Sel+1], 0
ELSE
        mov     byte ptr cs:[Skip_Load_Sna+1], 0
ENDIF
        ret
ENDIF

;IF RELOAD_ON_F4
_ReLoad:;******************************************************
        ; ReLoad from initially loaded file (if any)
        ;******************************************************
        mov     byte ptr cs:[Skip_ReLoad_Sna+1], 0
        ret
;ENDIF

IF HELP_ON_F1 ;F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1

IF HELP_SHOW_KBD
  YHLP EQU 15
ELSE
  YHLP EQU 8
ENDIF

        ;******************************************************
        ; Display help screen or Paused message, wait any key
        ;******************************************************
Show_Help:
        mov     dx, offset[Help_Text]
        ;--push    ds
        pusha
        push    cs
        pop     ds
        mov     si, dx
        mov     dh, YHLP-1
@help_lines_loop:
        inc     dh
        mov     al, byte ptr [si]
        cmp     al, 0
        jz short @help_2
        mov     bh, 0
        mov     dl, 10
        mov     ah, 2
        int     10h ; set cursor pos (dl=x, dh=y)
@help_chars_loop:
        lodsb
        cmp     al, 0
        jz short @help_lines_loop
        mov     ah, 0Eh
        movzx   bx, ah ; yellow color = function id "out char"
        int     10h
        jmp short @help_chars_loop
@help_2:
IF HELP_SHOW_KBD or COPYRIGHT
        push    0A000h
        pop     es
ENDIF
IF COPYRIGHT
        mov     di, 320*194+110
        mov     al, token_Copyright
        mov     bh, 0Ch
        call    Draw_Token
ENDIF
IF HELP_SHOW_KBD
        mov     dh, 0
        ;++++++ provide background for buttons
        mov     di, 320 * 7 + 35
        mov     cl, 4  ; 4 keyboard button lines
@help_kbd_btns_1:
        mov     dl, 24 ; 24 pixels height each button
@help_kbd_btns_2:
        mov     ch, 10 ; 10 keys in a line
@help_kbd_btns_3:
        mov     dh, 25 ; 25 pixels width each button
        mov     al, 17 + 24
        sub     al, dl ; 16, 17, 18, ... - starts each pixel line in a button
@help_kbd_btns_4:
        stosb
        inc     al
        dec     dh
        jnz short @help_kbd_btns_4
        dec     ch
        jnz short @help_kbd_btns_3 ; pixel line ended
        add     di, 320 - 250
        dec     dl
        jnz short @help_kbd_btns_2
        loop    @help_kbd_btns_1
        ;++++++ write text over buttons using Spectrum's rom font
        mov     di, 320 * 12 + 36
        mov     si, offset [Keyboard_keys]
        mov     cl, 4 ; 4 keyboard lines
@help_kbd_btns_5:
        mov     ch, 10 ; 10 buttons in each line
@help_kbd_btns_6:
        ;++++++ load'n'draw main character (bold, white)
        lodsb
IF HELP_BOLD_CHARS
        mov     bx, 0F01h
ELSE
        mov     bh, 0Fh ; light white color for a character
ENDIF
        cmp     al, ' '
        mov     dl, 25
        je short @help_kbd_no_text
        call    Draw_Char
        add     di, 8
        ;++++++ load'n'draw 1st symbol
        lodsb
IF HELP_BOLD_CHARS
        dec     bl
ENDIF
        dec     bh      ; 0Eh - yellow color for a symbol
        cmp     al, token_RND
        jae short @help_kbd_draw_sym_token
        cmp     al, '<'
        jz short @help_kbd_btns_2chars
        cmp     al, '>'
        jnz short @help_kbd_btns_7
@help_kbd_btns_2chars:
        call    Draw_Char ; yellow color for a symbol
        ;++++++ load'n'draw 2nd symbol (if any)
        lodsb
@help_kbd_btns_7:
        add     di, 7
        call    Draw_Char
        sub     di, 15
        jmp short @help_kbd_token1
        ;++++++ draw symbol token
@help_kbd_draw_sym_token:
        add     di, 320 + 1
        call    Draw_Token
        add     di, -320 - 1 - 8
@help_kbd_token1:
        ;++++++ load'n'draw upper token (or char)
        mov     dl, 25
        lodsb
        mov     bh, 0Ch ;
        push    di ;---------------------\
        add     di, -320*6
        cmp     al, 80h
        ja short @help_kbd_token2
        add     di, 8
        call    Draw_Char
        ;mov     dl, 17
        jmp short @help_kbd_token2_end
@help_kbd_token2:
        call    Draw_Token
@help_kbd_token2_end:
        pop     di ;---------------------/
@help_kbd_no_text:
        add     di, dx
        ;++++++ load'n'draw main token
        push    di
        add     di, -25 + 320*7
        mov     bh, 0Dh ;
        lodsb
        call    Draw_Token
@help_kbd_token4:
        ;++++++ load'n'draw below token
        lodsb
        mov     bh, 0Bh
        add     di, 5 * 320
        call    Draw_Token
@help_kbd_token_5:
        pop     di
        dec     ch
        jnz short @help_kbd_btns_6
        add     di, -25*10 + 320*24
        dec     cl
        jnz short @help_kbd_btns_5
ENDIF ; HELP_SHOW_KBD

        call    WaitKey
        call    WaitReleaseKey
        mov     byte ptr [Cont_loop], 80h ; prevent from exit on escape
        popa
        ;--pop     ds
        ; and finally:

_Help:  ; toggle jmp offset to call Show_Help on next emulation / prevent it
        xor     byte ptr cs:[Get_op_start+1], 3
        ret

IF HELP_SHOW_KBD
;********************************************************
; Draw character using font in spectrum rom
; in:   al = character
;       es:di = destination
;       bh = color
;********************************************************
Draw_Char proc
        push    ds
        pusha
IF SPECTRUM_128K ;.......................................128K
        call    LoadROM1segment2ds
ELSE             ;.......................................48K
        mov     ds, word ptr [Spectrum_Memory_Segment]
ENDIF
        mov     ah, 0
        shl     ax, 3
        add     ax, 3C00h
        xchg    si, ax
        mov     cl, 8 ; 8 pixel lines
@draw_char_1:
        lodsb
        mov     ch, 8 ; 8 pixels in line
@draw_char_2:
        add     al, al
        jnc short @draw_char_3
        mov     byte ptr es:[di], bh
IF HELP_BOLD_CHARS
        test    bl, 1
        jz short @draw_char_3
        mov     byte ptr es:[di+1], bh
ENDIF
@draw_char_3:
        inc     di
        dec     ch
        jnz short @draw_char_2
        add     di, 320 - 8
        loop    @draw_char_1
        popa
        pop     ds
draw_token_end:
        ret
Draw_Char endp ;*****************************************
ENDIF
IF HELP_SHOW_KBD or COPYRIGHT
;********************************************************
; Draw token from spectrum rom, using Small_Font
; in:   al = token
;       es:di = destination
;       bh = color
;********************************************************
Draw_Token proc
        test    al, al
        jz short draw_token_end
        push ds
        pusha
        mov     ah, al
        sub     ah, token_RND - 1
        mov     si, offset [Additional_Tokens]
        jae short @draw_token_spectrum
        add     ah, token_RND - token_EDIT
IF COPYRIGHT
        mov     bl, 0
        cmp     al, token_Copyright
        jz short @draw_token_1
ENDIF
        mov     bh, 0Ah ; RED
        add     di, 320 * 3
        cmp     al, token_CAPS
        jb short @draw_token_0
        mov     bh, 7 ; GRAY
        sub     di, 320 * (3+8)
        cmp     al, token_SHIFT
        jb short @draw_token_0
        jz short @draw_token_2lines_down
        add     di, 320 * 2
@draw_token_2lines_down:
        add     di, 320 * 2
        jmp short @draw_token_0
@draw_token_spectrum:
        mov     si, 96h
IF SPECTRUM_128K ;.......................................128K
        call    LoadROM1segment2ds
ELSE             ;.......................................48K
        mov     ds, word ptr [Spectrum_Memory_Segment]
ENDIF
@draw_token_0:
        mov     bl, 6 ; maximum 6 chars
@draw_token_1:
        dec     ah
        jz short @draw_token_3
@draw_token_2:
        lodsb
        test    al, 80h
        jz short @draw_token_2
        jmp     @draw_token_1
@draw_token_3:
        ;++++++ token found, draw it
        lodsb
        push    ax ;--------------\
        push    si ;------------\
        and     ax, 7Fh
        sub     al, 'A'
        jnb short @draw_token_char
        cmp     al, '$'-'A'
        mov     al, 'J'-'A' ; '$' -> 'J'
        jne short @draw_token_6 ; skip other non-chars ('#')
@draw_token_char:
        mov     si, ax
        add     si, si
        add     si, ax
        add     si, offset [Small_Font]
        push    ds ;----------\
        push    cs
        pop     ds ; ds = cs to access Small_Font
        mov     cl, 3 ; 3 columns in a char
@draw_token_4:
        lodsb
        mov     ch, 7 ; 7 max pixel rows in a char
@draw_token_5:
        ror     al, 1
        jnc short @draw_token_7
        mov     byte ptr es:[di], bh
@draw_token_7:
        add     di, 320
        dec     ch
        jnz short @draw_token_5
        add     di, 1 - 320 * 7
        loop    @draw_token_4
        pop     ds ;----------/
@draw_token_6:
        pop     si ;------------/
        pop     ax ;--------------/
        inc     di
        dec     bl
        jz short @draw_token_stop
        test    al, 80h
        jz short @draw_token_3 ; get next char of the token
@draw_token_stop:
        popa
        pop  ds
        ret
Draw_Token endp ;****************************************
IF SPECTRUM_128K
;********************************************************
; Load ROM 1 segment to ds - for Spectrum 128K
LoadROM1segment2ds proc ;********************************
        mov     dx, word ptr [Spectrum_Memory_Segment]
        add     dx, 2400h
        mov     ds, dx
        ret
LoadROM1segment2ds endp ;********************************
ENDIF ; SPECTRUM_128K

ENDIF ; HELP_SHOW_KBD

ENDIF ; HELP_ON_F1 ;F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1

;**************************************************************
; Update Keybd_State procedure
; in:   ah:bit0 = 1 if key is released
;       al = "AND" mask to set Spectrum keyboard keys to "pressed" state
;       bx = 0..7 keyboard line
;**************************************************************
Update_Keybd_State proc
        test    ah, 1
        jz short @key_pressed
        not     al
        or      byte ptr cs:[Keybd_State+bx], al
        ret
@key_pressed:
        and     byte ptr cs:[Keybd_State+bx], al
        ret
Update_Keybd_State endp ;**************************************

;##############################################################
;  TIMER TICK PROCEDURE
;##############################################################
Timer_Tick_Proc proc
        mov     byte ptr cs:[TimerTick], 1
        iret
Timer_Tick_Proc endp ;#########################################

;##############################################################
;  MAIN PROCEDURE, TIMER IS SET
;##############################################################
Main_Proc  proc

        ;==================================================================
        ; Allocate 64K memory (if Spectrum48K) or 176K (if Spectrum128K)
        ;==================================================================
IF SPECTRUM_128K
        mov     bx, 2C00h ; 176K / 16 segments
ELSE
        mov     bx, 1000h ; 64K / 16 segments
ENDIF
        mov     ah, 48h
        int     21h
        jc Error_No_memory

        ;------------------------------------------------------
        ; AX = segment address of memory block allocated
        ;------------------------------------------------------
        mov     word ptr [Spectrum_Memory_Segment], ax

        ;======================================================
        ; search SPECTRUM .ROM file
        ;======================================================
        mov     ax, 3D00h       ; AL=00 means "read only; deny write"
        mov     cl, 0           ; attributes mask
        mov     dx, offset [Rom_Filename]
        int     21h
        jnc short rom_file_opened

        ;------------------------------------------------------
        ; error: Spectrum.rom not found
        ;------------------------------------------------------
Error_Spectrum_rom:
        mov     byte ptr [Msg_rom_end], 20h
        call    ShowError
Rom_filename:
IF SPECTRUM_128K ;.....................128K
        DB      '128.ROM'
ELSE             ;.....................48K
        DB      '48.ROM'
ENDIF            ;.....................
Msg_rom_end:
        DB      0,'not found',13,10,'$'

rom_file_opened:       ; AX = file handle
        ;======================================================
        ; load SPECTRUM.ROM file
        ;======================================================
        push    ax     ; save file handle ;------------------------------------\

IF SPECTRUM_128K
        push    ax ;--------------------------------------------------------\
        ;******************************************************
        ; Initializy Spectrum-128K memory banks access tables
        ;******************************************************
IF LOGS and LOG_Bank_Init
        call    LogText
        DB      'Bank_Access_Table',13,10,0
ENDIF
        push    cs
        pop     es

        mov     di, offset [Bank_Access_Table]
        mov     ch, 0 ; ch = bit3 - ROM# (0-128K, 1-48K), bit2..0 - RAM# at page 2
                      ;      bit4 - 0:table for read 1:table for write
@Init_Bank_Access_Table:
IF LOGS and LOG_Bank_Init
        call    LogText
        DB      'Port_7FFD=',0
        movzx   ax, ch
        call    LogAX
        call    LogText
        DB      13,10,0
ENDIF
        mov     cl, 0
@Init_Bank_Access_Table_1:
        movzx   ax, cl
        and     al, 0C0h
        sal     ax, 4 ; 00->000,01->000,..,3F->000,40->400,41->400,..,FF->C00
        cmp     cl, 40h
        jae short @Init_Bank_RAM
        test    ch, 10h
        mov     ah, 28h ; if write to ROM - readdress it to a dummy bank
        jnz short @Init_Bank_Ready
        test    ch, 8
        mov     ah, 20h ; ROM0
        jz short @Init_Bank_Ready ;
        mov     ah, 24h ; otherwise readdress to ROM1 (S48K ROM)
        jmp short @Init_Bank_Ready
@Init_Bank_RAM:
        mov     dl, cl
        sub     dl, 80h
        jae short @Init_Bank_RAM_1
        ; address 4000h-7FFFh, always in RAM5:
        mov     ah, 14h - 4
        jmp short @Init_Bank_Ready
@Init_Bank_RAM_1:
        sub     dl, 40h
        jb short @Init_Bank_RAM_2
        ; address 0C000h-0FFFFh, in paged RAM bank (CH & 7):
        mov     ah, ch
        and     ah, 7
        shl     ah, 2
        sub     ah, 0Ch
        DB 3Dh; - use op code cmp cx,... instead jmp @Init_Bank_Ready
@Init_Bank_RAM_2:
        ; address 8000h-0BFFFh, always in RAM2:
        mov     ah, 8h - 8
@Init_Bank_Ready:
IF LOGS and LOG_Bank_Init
        call    LogAX
        call    LogText
        DB      ' ',0
ENDIF
        ; finally, add base segment value and store it as a segment to use with
        ; the address:
        add     ax, word ptr [Spectrum_Memory_Segment]
        stosw
        inc     cl
        jnz short @Init_Bank_Access_Table_1
IF LOGS and LOG_Bank_Init
        call    LogText
        DB      13,10,0
ENDIF
        inc     ch
        cmp     ch, 32
        jb short @Init_Bank_Access_Table
        pop     bx ;--------------------------------------------------------/
ENDIF
        ;++++++ continue loading Spectrum.rom
IF SPECTRUM_128K
        mov     cx, 32768
        mov     ax, word ptr [Spectrum_Memory_Segment]
        add     ax, 2000h
        mov     ds, ax
ELSE
        xchg    bx, ax
        mov     cx, 16384
        mov     ds, word ptr cs:[Spectrum_Memory_Segment]
ENDIF
        xor     dx, dx
        mov     ah, 3Fh
        int     21h
        push    cs
        pop     ds

        jc      Error_Spectrum_rom
        ;------------------------------------------------------
        ; Spectrum.rom loaded OK, close file
        ;------------------------------------------------------
        pop     bx     ; restore file handle ----------------------------------/
        mov     ah, 3Eh  ; close file
        int     21h

        ;======================================================
        ; check parameters
        ;======================================================
        mov     si, 80h
        lodsb
        movzx   cx, al
        jcxz    Switch_Video_Mode
search_param_start:
        mov     dx, si
        lodsb
        cmp     al, 13
        jz short Switch_Video_Mode
        cmp     al, 20h
        jbe short search_param_start
IF OPTION_SILENCE
        ;------ possible option /s - silence
        cmp     al, '/'
        jnz short search_param_end
        lodsb
        cmp     al, 's'
        jnz short search_param_start
        call    _Sound
        jmp short search_param_start
ENDIF
search_param_end:
        lodsb
        cmp     al, '.'
        jne short @search_param_end2
        xor     cx, cx
@search_param_end2:
        cmp     al, 13
        ja short search_param_end
        dec     si
        jcxz    @set_param_end
        mov     dword ptr [si], 'ans.'
        add     si, 4
@set_param_end:
        mov     byte ptr [si], 0 ; set finishing #0 character
;IF RELOAD_ON_F4
        mov     word ptr [ReLoad_Sna_SetName+1], dx
;ENDIF
        call    _ReLoad ; Load_Sna

        ;======================================================
        ; switch to 320x200x256 video mode
        ;======================================================
Switch_Video_Mode:
IF SPECTRUM_128K
        push    dx
        mov     cl, 7
        call    out_7FFD
        pop     dx
ENDIF

        ;------
        mov     ax, 0013h
        int     10h
        ;------ work in graphic mode 320x256 x 256 colors, A000h = video buffer
        call    Work_Graphic_Mode
        ;------
Text_Mode:
        mov     ax, 0003h
        int     10h
        ret

IF LOAD_ON_F3
IF LOAD_SELECT
;############################################################
; SELECT FILE TO LOAD
;############################################################
Load_Sel proc
        mov     byte ptr [Skip_Load_Sel+1], 5
        call    Set_Restore_INT9
        jmp short @Detect_Dir_Path
@Detect_Dir_Path:
        mov     byte ptr [@Detect_Dir_Path-1], @Popup_DirList - @Detect_Dir_Path
        ; for the 1st call, detect current disk and directory
        mov     dx, offset[DTA]
        mov     ah, 1Ah ; move DTA from ds:80h
        int     21h

        mov     ah, 19h ; get current disk -> al (0='A', 1='B',...)
        int     21h
        mov     dl, al
        inc     dl ; dl = disk (1='A', 2='B',...)
        add     al, 'A'
        mov     byte ptr [Dir_Path], al
        mov     ah, 47h ; get current path -> ds:si
        mov     si, offset [Dir_Path+3]
        int     21h

@Popup_DirList:
        mov     bp, 0

        ;++++++ draw frame
        mov     si, offset[Dir_Frame]
        mov     dh, Dir_Top ;dh = Y
        mov     bl, 05h ; bl = color (Cyan dark)
@frame_loop:
        lodsb
@frame_got_char:
        test    al, al
        jnz short @frame_out_char
        mov     dl, Dir_Left ; dl = X
        inc     dh  ; dh = next Y
        mov     bh, 0
        mov     ah, 2
        int     10h

        lodsb
        test    al, al
        jnz short @frame_got_char
        jmp     @frame_ready ; 0, 0 - signal "end of frame"
@frame_out_char:
        mov     bh, al ; bh = previous char
        call    Dir_out_char
        jmp short @frame_loop
@frame_ready:

Dir_Left        EQU 12
Dir_Top         EQU 2

Show_DirList:
        ;++++++ Show directory
        xor     bx, bx
        ;cmp     byte ptr [Dir_Path+3], bl
        ;jz      @noadd_dotdot
        ;
        ;mov     dword ptr [DTA+1Eh], '..'
        ;call    Dir_Add_File
;@noadd_dotdot:

        inc     byte ptr [Dirs_or_Files] ; STC => show directories
        call    Show_Dirs_or_Files
        dec     byte ptr [Dirs_or_Files] ; CLC => show files
        call    Show_Dirs_or_Files

IF LOGS and LOG_F3_SELECT
        call    LogText
        DB      '--------------bx=',0
        xchg    ax, bx
        call    LogAX
        xchg    ax, bx
        call    LogText
        DB      13,10,0
ENDIF

        cmp     bp, bx
        jl short @store_Dir_Count
        xor     bp, bp
        jmp short Show_DirList

@store_Dir_Count:
        mov     word ptr [Dir_Count], bx
        mov     ax, bx
        shr     ax, 4
        mov     cx, bp
        shr     cx, 4
        cmp     ax, cx
        jnz short @no_clear_rest_of_page

@loop_clear_rest:
        and     bx, 0Fh
        jz short @no_clear_rest_of_page
        mov     byte ptr [DTA+1Eh], bh
        call    Dir_Add_Line
        jmp short @loop_clear_rest

@no_clear_rest_of_page:

        ;++++++ save bp for the next F3
        mov     word ptr [@Popup_DirList+1], bp
        ;++++++ get key
Wait_Next_Key:
        mov     ah, 0
        int     16h
        ;call    WaitKey
        ;mov     byte ptr [WaitKey+1], 80h

IF LOGS and LOG_F3_SELECT
        call    LogText
        DB      'scan-code: ', 0
        call    LogAX
        call    LogText
        DB      13,10,0
ENDIF

        call    Load_Sel_Key_Cmd
        jmp short Show_DirList

DirPage EQU     16 ; number of files shown on page

;========================================================
Show_Dirs_or_Files proc
;========================================================
        mov     si, offset [Dir_Path]
@search_end_of_path:
        lodsb
        test    al, al
        jnz short @search_end_of_path
        dec     si
        cmp     byte ptr [si-1], '\'
        jz short @noadd_slash
        mov     byte ptr [si], '\'
        inc     si
@noadd_slash:

        push    si ;--------------------\
Dirs_or_Files:
        clc
        jc short @show_dirs

        mov     dword ptr [si], 'NS.*'
        mov     word ptr [si+4], 'A'
        xor     cx, cx ; attributes "any"

        mov     byte ptr [@skip_non_dir-1], @file_accept - @skip_non_dir

        jmp short @find_first

@show_dirs:
        mov     dword ptr [si], '*.*'
        mov     cx, 10h ; attributes "directory"
        mov     byte ptr [@skip_non_dir-1], 0

@find_first:

IF LOGS and LOG_F3_SELECT
        pusha
        mov     dx, offset [Dir_Path]
        call    LogText
        DB      'mask to find first: ',0
        call    LogTextDX
        call    LogText
        DB      13,10,0
        popa
ENDIF

        mov     ah, 4Eh ; find first
        mov     dx, offset [Dir_Path]
        int     21h

        pop     si ;-------------------/
        mov     byte ptr [si], 0
        jc short @Show_Dirs_End

@next_file:
        jmp short @file_accept
@skip_non_dir:
        test    byte ptr [DTA+15h],10h
        jz short @find_next_file
        cmp     word ptr [DTA+1Eh], '.'
        jz short @find_next_file
@file_accept:
        call    Dir_Add_File

@find_next_file:
        mov     ah, 4Fh
        int     21h
        jnc short @next_file
@Show_Dirs_End:
        ret
Show_Dirs_or_Files endp ;================================

;========================================================
Dir_Add_File proc
;========================================================
IF LOGS and LOG_F3_SELECT
        pusha
        call    LogText
        DB      'file to add:',0
        mov     dx, offset [DTA+1Eh]
        mov     cx, 12
        call    LogTextDX
        call    LogText
        DB      13,10,0
        popa
ENDIF
        mov     ax, bx
        shr     ax, 4
        mov     cx, bp
        shr     cx, 4
        cmp     ax, cx
        jnz short @end_Dir_Add_File

Dir_Add_Line:
        ;++++++ file is on the page, show it

        push    bx ;------------------------\
        mov     dl, Dir_Left+1 ; X-coord
        mov     dh, bl
        and     dh, 0Fh
        add     dh, Dir_Top+2 ; Y-coord
        mov     bh, 0
        mov     ah, 2
        int     10h

        mov     si, offset[DTA+1Eh]
        pop     bx ;------------------------/
        push    bx ;------------------------\
        mov     cx, 12
        mov     al, 7
        test    byte ptr [DTA+15h], 10h
        jz short @color_file
        mov     al, 5
@color_file:
        cmp     bx, bp
        jnz short @color_draw_char
        pusha
        ;++++++ copy current filename
        push    cs
        pop     es
        mov     di, offset [CurFileName]
        mov     si, offset [DTA+15h]
        mov     cx, 1Eh-15h+13
        rep movsb
        popa
        or      al, 8 ; light color for selected item
@color_draw_char:
        mov     bl, al
@get_char_to_draw:
        lodsb
        test    al, al
        jnz short @out_filename_char
        test    cx, cx
        jz short @fin_filename_out
        mov     al, 20h
        dec     si
@out_filename_char:
        dec     cx
        call    Dir_out_char
        jmp short @get_char_to_draw
@fin_filename_out:
        pop     bx ;------------------------/
@end_Dir_Add_File:
        inc     bx
        ret
Dir_Add_File endp ;======================================

;========================================================
Load_Sel_Key_Cmd proc
; in:     ah = scan-code to handle
;         [Dir_Count] = count of items >=0
; in/out: bp = index of current item < [DirCount] or 0
;========================================================
        cmp     ah, 1Ch
        jz short @Enter
        movzx   cx, ah
        dec     ah
        jz      @Esc
        sub     cl, 46h
        loop    @Sel_48
@Home:  xor     bp, bp
        ret

@Sel_48:loop    @Sel_49
@Up:    test    bp, bp
        jnz short @End_1
        ret

@Sel_49:loop    @Sel_4B
@PgUp:  sub     bp, DirPage
        jc short @Home
        ret

@Sel_4B:dec     cx
        loop    @Sel_4D
@Left:  sub     bp, DirPage
        jc short @PgDn
        ret

@Sel_4D:dec     cx
        loop    @Sel_4F
@Right: add     bp, DirPage
        cmp     bp, word ptr [Dir_Count]
        jnc short @Left
        ret

@Sel_4F:dec     cx
        loop    @Sel_50
@End:   mov     bp, word ptr [Dir_Count]
@End_1:
        dec     bp
        ret

@Sel_50:loop    @Sel_51
@Down:  inc     bp
        cmp     bp, word ptr [Dir_Count]
        jnc short @End_1
        ret

@Sel_51:loop    @EndOfDirCmd
@PgDn:  add     bp, DirPage
        cmp     bp, word ptr [Dir_Count]
        jnc short @PgUp
        ;ret

@EndOfDirCmd:
        ;pop     cx
        ;jmp     Wait_Next_Key
        ret

@Enter:
        mov     si, offset [Dir_Path]
@Enter_loop_find_eol:
IF LOGS and LOG_F3_SELECT
        pusha
        mov     dx, si
        call    LogText
        DB      'si->',0
        call    LogTextDX
        call    LogText
        DB      13,10,0
        popa
ENDIF

        lodsb
        test    al, al
        jnz short @Enter_loop_find_eol
        dec     si

        test    bp, bp
        jnz short @Enter_1
        ;++++++ level up
        dec     si
@Enter_loop_find_slash:
        dec     si

IF LOGS and LOG_F3_SELECT
        pusha
        mov     dx, si
        call    LogText
        DB      'searching\: si->',0
        call    LogTextDX
        call    LogText
        DB      13,10,0
        popa
ENDIF
        cmp     byte ptr [si], '\'
        jnz short @Enter_loop_find_slash
        ;inc     si
        mov     byte ptr [si], 0
        ret     ; bp = 0 already
@Enter_1:
        test    byte ptr [CurFileName], 10h
        jz short @Enter_Open
        ;++++++ enter subdir
        pusha
        push    cs
        pop     es
        mov     di, si
        mov     si, offset [CurFileName+1Eh-15h]
@Enter_loop_move_fname:
        lodsb
        stosb
        test    al, al
        jnz short @Enter_loop_move_fname
        popa
        xor     bp, bp
        ret
@Enter_open:
        ;++++++ finally, file is selected, open it
        call    Set_Restore_INT9
        call    WaitReleaseKey
        pop     cx
        mov     si, offset [Dir_Path]
        mov     di, offset [File2Load]
        mov     byte ptr [Skip_Load_Sna+1], 0 ; will be loaded on return
        ;mov     dx, di
@Enter_2:
        lodsb
        stosb
        test    al, al
        jnz short @Enter_2
        mov     si, offset [CurFileName+1Eh-15h]
        dec     di
        mov     al, '\'
        cmp     byte ptr [di-1], al
        jz short @Enter_3
        stosd
@Enter_3:
        lodsb
        stosb
        test    al, al
        jnz short @Enter_3
        ;jmp     Load_Sna
        ret

@Esc:
        call    Set_Restore_INT9
        call    WaitReleaseKey
Dir_End_Show:
        pop     cx
        mov     byte ptr [Cont_loop], 80h ; prevent exit from the program
        ret

Load_Sel_Key_Cmd endp ;==================================

;========================================================
Dir_out_char proc
; in:   bl = color
;       al = char
;       dh, dl = Y, X
;========================================================
        ;pusha   ;----------\
        ;mov     bh, 0
        ;mov     ah, 2
        ;int     10h
        mov     ah, 0Eh
        int     10h
        ;popa    ;----------/
        inc     dl
        ret
Dir_out_char endp ;======================================

Load_Sel endp ;##############################################
ENDIF

;IF RELOAD_ON_F4
;############################################################
; RELOAD INITIALLY LOADED FILE (if any)
;############################################################
ReLoad_Sna proc
        mov     byte ptr [Skip_ReLoad_Sna+1], 3
ReLoad_Sna_SetName:
        mov     dx, 80h
        jmp short Load_Sna
ReLoad_Sna endp ;############################################
;ENDIF

IF SAVE_ON_F2
;############################################################
; SAVE SNA-FILE
;############################################################
Save_Sna proc
        mov     byte ptr [Skip_Save_Sna+1],3

        ;++++++ correct registers to save SNA-file correctly:
        mov     al, byte ptr [R_R]
        mov     ah, byte ptr [R_R_hi]
        and     ax, 807Fh
        or      al, ah
        mov     byte ptr [R_R], al

        ;++++++ decide how to save PC and if extended sna-format should be used
        mov     cx, si
IF SPECTRUM_128K and SNA_EXTENDED
        cmp     word ptr [R_SP], 4001h ; if SP too low, use extended SNA
        jbe short @Save_prepare_PC
        test    byte ptr [Port_7FFD], 20h ; otherwise if port 7FFD locked, use 48K SNA
        jz short @Save_prepare_PC
        xor     cx, cx
@Save_prepare_PC:
        mov     word ptr [R_PC], cx
        inc     cx
        loop    @Save_PC_ready
        mov     cx, si
ENDIF

        ;++++++ simulate NMI interrupt here (just execute push PC, for 48K only):
        call    push_CX
@Save_PC_ready:

        mov     dx, offset [Z_sna]
        mov     ah, 3Ch ; create file
        xor     cx, cx
        int     21h
        jc short Z_Sna_created
        xchg    bx, ax
        mov     ah, 3Eh ; close file
        int     21h
Z_Sna_created:

        mov     ax, 3D01h ; 3D=open file, 01=write
        mov     di, -1
        jmp short Save_Sna_1
Z_Sna:  DB      'Z.SNA',0
Save_Sna endp ;##############################################
ENDIF

;############################################################
; LOAD Z.SNA FILE (if it exists)
;############################################################
Load_Z_Sna proc
        mov     byte ptr [Skip_Load_Sna+1],3
IF LOAD_SELECT
        mov     dx, offset [File2Load]
ELSE
        mov     dx, offset [Z_Sna]
ENDIF
Load_Z_Sna endp ;############################################
ENDIF   ; immediately go to Load SNA-file
        ;       |
        ;       V
;############################################################
; LOAD SNA-FILE
; in: dx = address of filename to load
;############################################################
Load_Sna proc

IF LOGS and LOG_F3_SELECT
        call    LogText
        DB      'file to load: ',0
        call    LogTextDX
        call    LogText
        DB      13,10,0
ENDIF

        mov     ax, 3D00h ; 3D=open file, 00=read only
        xor     di, di
        mov     word ptr [R_PC], di ; reset R_PC
Save_Sna_1:
        push    ds ;-----------------------------------------------------------\
        xor     cx, cx
        int     21h
        jc  Load_end
        push    ax ;---------------------------------------------------------\

        ;++++++ load registers first
        xchg    bx, ax
        mov     ah, 3Fh
        add     di, di
        adc     ah, 0
IF SPECTRUM_128K ;.......................................128K
        push    ax ;---------------------------------------------\
ENDIF            ;.......................................
        mov     dx, offset[Sna_Regs_Start]
        mov     cx, Sna_Regs_Lengh
        int     21h

        ;++++++ load RAM now
IF SPECTRUM_128K ;.......................................128K
        pop     ax ;---------------------------------------------/
        mov     dh, 14h ; RAM bank 5
        call    Load_Save_Bank
        mov     dh, 8 ; RAM bank 2
        call    Load_Save_Bank

        add     di, di
        jc short @Load_Sna_port_7FFD_initialized
        mov     byte ptr cs:[Port_7FFD], 37h ; locked + ROM1 + RAM7 + video5
@Load_Sna_port_7FFD_initialized:
        mov     dh, 1Ch ; RAM bank 7
        jnc short @Load_Save_Page_3
        mov     dh, byte ptr cs:[Port_7FFD]
        and     dh, 7
        shl     dh, 2
@Load_Save_Page_3:
        call    Load_Save_Bank

IF SNA_EXTENDED ; Extended SNA format support: load additional data
        add     di, di
        jnc short @Ext_Sna
        mov     cx, word ptr cs:[R_PC]
        jcxz    @Not_Ext_Sna ; if Save and R_PC = 0 then 48K saved
@Ext_Sna:
        push    ds ;--------------------------\
        ;++++++ restore ds = cs
        push    cs
        pop     ds

        push    ax ;---------------------\
        mov     dx, offset [Sna_Ext_Start]
        mov     cx, Sna_Ext_Length
        int     21h
        xchg    dx, ax
        pop     ax ;---------------------/
        pop     ds ;--------------------------/ ds = RAM bank 7 segment
        jc short @Not_Ext_Sna
        and     dx, dx ; if 0 bytes read?
        jz short @Not_Ext_Sna
        ;++++++ OK, it is extended SNA
        add     di, di
        jc short @Paged_Bank_Moved
        mov     cl, byte ptr cs:[Port_7FFD]
        ;++++++ move paged RAM bank at its correct place
        mov     dh, cl
        and     dx, 700h
        shl     dx, 2
        add     dx, word ptr cs:[Spectrum_Memory_Segment]
        mov     es, dx ; es = RAM bank paged segment
        push    si ;--------------------------------\
        push    di ;-------------------------------\
        xor     si, si
        xor     di, di
        mov     cx, 4096
        rep movsd
        pop     di ;-------------------------------/
        pop     si ;--------------------------------/
@Paged_Bank_Moved:
        ;++++++ Load/Save other banks (except 5, 2 and paged)
        mov     cl, 0
@Load_Banks_loop:
        push    cx ;---------------\
        ;++++++ skip already loaded/saved banks
        cmp     cl, 2
        jz short @Next_Bank_loop
        cmp     cl, 5
        jz short @Next_Bank_loop
        mov     ch, byte ptr cs:[Port_7FFD]
        and     ch, 7
        cmp     cl, ch
        jz short @Next_Bank_loop
        ;++++++ load/save bank cl:
        mov     dh, cl
        shl     dh, 2 ; 0 -> 0, 1 -> 4, 2 -> 8, ...
        call    Load_Save_Bank

@Next_Bank_loop:
        pop     cx ;---------------/
        inc     cl
        cmp     cl, 8
        jb short @Load_Banks_loop
ENDIF
; Extended SNA format either not supported or not extended SNA is loaded
@Not_Ext_Sna:
        ;++++++ restore ds = cs
        push    cs
        pop     ds

        mov     cl, byte ptr [Port_7FFD]
        call    out_7FFD

ELSE             ;.......................................48K
        mov     ds, word ptr [Spectrum_Memory_Segment]
        mov     ah, 3Fh
        add     di, di
        adc     ah, 0
        mov     dx, 4000h
        mov     cx, 0C000h
        int     21h
ENDIF            ;.......................................

        ;++++++ restore ds = cs
        push    cs
        pop     ds

        ;++++++ initialization to allow start loaded program
        mov     al, byte ptr [R_R]
        and     al, 80h
        mov     byte ptr [R_R_hi], al
        mov     ah, byte ptr [R_F]
        and     byte ptr [IFF1IFF2], 4

        mov     ah, byte ptr [R_F]

        mov     si, word ptr [R_PC]
        and     si, si
        jnz short @Load_PC_restored
        call    Ret_N
@Load_PC_restored:

        mov     byte ptr [Reset_Flags+1], ah
        mov     byte ptr [R_F], ah
        mov     word ptr [Reset_PC+1], si

        inc     byte ptr [R_R]

        pop     bx ;---------------------------------------------------------/
        mov     ah, 3Eh  ; close file
        int     21h
Load_end:
        pop     ds ;-----------------------------------------------------------/
        ret
Load_Sna endp ;##############################################

IF SPECTRUM_128K
;************************************************************
; Load 16384 bytes to a certain bank from a file
; OR Save it to a file
; in:   ah = function: 3Fh - read 40h - write
;       dh = RAM bank hi byte offset / 16
;       bx = file descriptor
;************************************************************
Load_Save_Bank proc
        push    ax ;----------------------------------------\
        mov     dl, 0
        add     dx, word ptr cs:[Spectrum_Memory_Segment]
        mov     ds, dx
        xor     dx, dx
        mov     cx, 16384
        int     21h
        pop     ax ;----------------------------------------/
        ret
Load_Save_Bank endp ;****************************************
ENDIF

;############################################################
; SHOW ERROR MESSAGE
;############################################################
ShowError       proc
        push    cs
        pop     ds
        call    Text_Mode
        mov     ah, 9
        pop     dx
        int     21h
        int     20h ; EXIT TO DOS
ShowError       endp

IF      LOGS
;############################################################
; LOGGING WHILE DEBUGGING
;############################################################
LogText proc
        push    bp
        mov     bp, sp
        xchg    dx, word ptr ss:[bp+2]
        pop     bp

        push    ax ;-----------------------------------------\
        push    bx ;-----------------------------------------\
        push    cx ;-----------------------------------------\
        push    si ;-----------------------------------------\
        push    di ;-----------------------------------------\
        pushf      ;-----------------------------------------\
        push    ds ;-------------------------------------------\
        push    cs
        pop     ds

        ; dx = offset of text to log to log-file
        mov     si, dx
        mov     cx, -1
@calc_text_len:
        inc     cx
        lodsb
        test    al, al
        jnz short @calc_text_len
        ; si = end address of text (return address)
        ; cx = text length
        push    si ;-----------\
IF LOGBUFFER
        call    Log_Text2Buffer
ELSE
        call    Log_Text2File
ENDIF
        pop     dx ;-----------/
        pop     ds ;-------------------------------------------/
        popf       ;-----------------------------------------/
        pop     di ;-----------------------------------------/
        pop     si ;-----------------------------------------/
        pop     cx ;-----------------------------------------/
        pop     bx ;-----------------------------------------/
        pop     ax ;-----------------------------------------/

        push    bp
        mov     bp, sp
        xchg    dx, word ptr ss:[bp+2]
        pop     bp

        ret
LogText endp ;###############################################

;############################################################
;
;############################################################
LogTextDX proc
        push    ax
        push    cx
        push    si
        mov     si, dx
        mov     cx, -1
@loop_count_len:
        inc     cx
        lodsb
        test    al, al
        jnz short @loop_count_len
IF LOGBUFFER
        call    Log_Text2Buffer
ELSE
        call    Log_Text2File
ENDIF
        pop     si
        pop     cx
        pop     ax
        ret
LogTextDX endp ;#############################################

;############################################################
;       LOG TEXT TO FILE
; in:   dx = address of text
;       cx = text length
;############################################################
Log_Text2File proc
        pusha      ;-----------------------------------------\
        push    ds ;---------------------------------------\
        push    dx ;---------------------------------\
        push    cx ;-------------------------------\
        push    cs
        pop     ds
        mov     ax, 3D02h
        mov     dx, offset[@Log_file_name]
        int     21h ; open log file
        jc short log_end
        push    ax ; ax = file handle --------\
        xchg    bx, ax
        mov     ax, 4202h
        xor     cx, cx
        xor     dx, dx
        int     21h ; seek to the end of file
        pop     bx ; -------------------------/
        pop     cx ;-------------------------------/
        pop     dx ;---------------------------------/
        push    bx ;-------------\
        mov     ah, 40h
        int     21h ; write text
        pop     bx ;-------------/
        mov     ah, 3Eh
        int     21h ; close file
        pop     ds ;---------------------------------------/
        popa       ;-----------------------------------------/
log_end:
        ret
IF COMPRESS_TABLES
@Log_file_name: DB 'logC.z',0
ELSE
@Log_file_name: DB 'log_.z',0
ENDIF
Log_Text2File endp ;#########################################

IF LOGBUFFER
;############################################################
;       LOG TEXT TO BUFFER
; in:   dx = address of text
;       cx = text length
;############################################################
LogBufSiz       EQU 1024
Log_Text2Buffer proc
        mov     di, word ptr cs:[LogBufPos]
        mov     bx, di
        sub     bx, offset[Log_buff]
        mov     si, dx
@loop_store2logbuff:
        cmp     bx, LogBufSiz
        jb short @cont_store2logbuff
        push    cx ;------------------------\
        mov     cx, bx
        mov     dx, offset[Log_buff]
        mov     word ptr cs:[LogBufPos], dx
        call    Log_Text2File
        pop     cx ;------------------------/
        mov     bx, 0
        mov     di, offset[Log_buff]
@cont_store2logbuff:
        mov     al, byte ptr cs:[si]
        inc     si
        mov     byte ptr cs:[di], al
        inc     di
        inc     bx
        loop    @loop_store2logbuff
        mov     word ptr cs:[LogBufPos], di
        ret
LogBufPos: dw Log_buff
Log_Text2Buffer endp ;#######################################
ENDIF ;LOGBUFFER

;############################################################
;       LOG AX AS HEX
;############################################################
LogAX   proc
        pusha ;-----------------------------------[
        push    ax ;--------------------------\
        push    di ;------------------------\
        pushf ;---------------------------\
        mov     di, offset[@Hex_Buffer]
        call    ah2Hex
        xchg    ah, al
        call    ah2Hex
        popf ;----------------------------/
        pop     di ;------------------------/
        pop     ax ;--------------------------/
        call    LogText
@Hex_Buffer:
        DB 0, 0, 0, 0, 0
        popa ;------------------------------------]
        ret

ah2Hex: push    ax ;--------\
        mov     al, ah
        shr     al, 4
        call    tetr2Hex
        xchg    al, ah
        call    tetr2Hex
        pop     ax ;--------/
        ret

tetr2Hex:
        and     al, 0Fh
        cmp     al, 9
        jbe short @add30h
        add     al, 'A'-'0'-10
@add30h:add     al, 30h
        mov     byte ptr cs:[di], al
        inc     di
        ret

LogAX   endp ;###############################################
ENDIF   ;LOGS

IF SIMPLE_DAA
ELSE ; not SIMPLE_DAA
;############################################################
; check if CL high nibble is between AH>>4 and AH&0Fh
; return C flag set if yes. AL is draft on exit.
; This function is used in Initialize DAATable2 loop below.
check_nibble proc ;##########################################
        push    cx ;--------\
        shr     cl, 4
        mov     al, ah
        shr     al, 4
        sub     cl, al
        mov     al, ah
        and     al, 0Fh
        cmp     cl, al
        pop     cx ;--------/
        ret
check_nibble endp ;##########################################
ENDIF ; SIMPLE_DAA

IF SPECTRUM_128K

;############################################################
; Read_Mem function - for SPECTRUM_128K only
; in:   bx = Spectrum memory address
; out:  dl = byte read at that address
Read_Mem proc ;##############################################
        push    bx
        movzx   bx, bh
        add     bx, bx
        mov     es, word ptr gs:[Bank_Access_Table+bx]
        pop     bx
        mov     dl, byte ptr es:[bx]
        ret
Read_Mem endp ;##############################################

;############################################################
; Write_Mem procedure - for SPECTRUM_128K only
; in:   bx = Spectrum memory address
;       dl = byte to write at that address
Write_Mem proc ;#############################################
        push    bx
        movzx   bx, bh
        add     bx, bx
        mov     es, word ptr gs:[Bank_Access_Table+8192+bx]
        pop     bx
        mov     byte ptr es:[bx], dl
        ret
Write_Mem endp ;#############################################

ENDIF

;############################################################
; Registers usage
;############################################################
base_HL         equ bp
seg_Op          equ fs

CF              EQU 1
NF              EQU 2
PF              EQU 4
XF              EQU 8
HF              EQU 16
YF              EQU 32
ZF              EQU 64
SF              EQU 128

R_L             EQU 0 ; R_H and R_L are the offsets from base_HL,
R_H             EQU 1 ; rather then from Rmain_

R_HL            EQU -5 ; offsets from Rmain_
R_DE            EQU -3
R_E             EQU -3
R_D             EQU -2
R_BC            EQU -1
R_C             EQU -1
R_B             EQU 0


;########################################################
; VIDEO SYSTEM STATE
;########################################################
VideoCount:     DW 0
                ; counts 312 pixel lines of screen buffer
                ; 0..63 - top border (no video output, simulated therefore)
                ; 64 - first pixel line of Spectrum screen (#0)
                ; ...
                ; 255 - last pixel line of Spectrum screen (#191)
                ; 256..311 - bottom part (no video output, simulated therefore).
                ; At finish, VideoCount is reset to 0, and Int is signalled

FrameCount:     DB 0
                ; used to change flash state each 16 frames
                ; it is incremented each frame, so mask 10h set to 1 flags
                ; flash on

;########################################################
; WORKING IN GRAPHIC MODE
;########################################################
Work_Graphic_Mode proc
        push    cs
        pop     es

IF SIMPLE_DAA
ELSE ; not SIMPLE_DAA
        ;------------------------------------------------
        ; Initialize DAATable1 (256 bytes, converts flags
        ; to shorter range 0..63)
        ; S Z Y H X P N C  -> 0 0 0 0 0 H N C
        ;------------------------------------------------
        mov     di, offset[DAATable1]
        mov     cl, 0
loop_InitDAATable1:
        mov     al, cl
        mov     ah, al
        and     ax, CF or NF or (HF shl 8)
        shr     ah, 2
        or      al, ah
        stosb
        inc     cl
        jnz short loop_InitDAATable1

        ;------------------------------------------------
        ; Initialize DAATable2 (4K bytes = 2K words)
        ;------------------------------------------------
        lea     di, [DAATable2]
        xor     cx, cx
loop_InitDAATable:
        push    cx
        ;mov     al, ch
        ;mov     ah, al
        ;and     ax, (2 shl 8) or 3
        ;rol     al, 2
        ;or      al, ah
        ;mov     bh, ah ; bh = flags when nothing to do with the operand
        lea     si, [DAADefTable]
daa_init_0:
        lodsd   ; eax = next 4 bytes from DAADefTable
        xor     bx, bx
        cmp     al, bl
        jl short daa_init_1
        mov     dl, al
        and     dl, 7
        cmp     dl, ch      ; are flags HNC match?
        jnz short daa_init_0 ; no, continue search matching entry in DAADefTable
        ;------ flags HNC the same, check other conditions
        mov     bl, al      ; bl:bit7 = 1 if CF after operation
        call    check_nibble ; check if high nibble in CL is between AH>>4 and AH&0Fh
        ja short daa_init_0 ; no, out of bounds -> continue searching
        push    cx ;-----------\
        sal     cx, 4
        sar     EAX, 8
        call    check_nibble ; check if low nibble in CL is between AH>>4 and AH&0Fh
        pop     cx ;-----------/
        ja short daa_init_0 ; no, out of bounds -> continue searching
        sar     EAX, 16 ; al = what to add to cl
        mov     bh, al
daa_init_1:
        mov     al, bh
        add     al, cl
        lahf
        ror     ah, 1
        add     bl, bl
        add     bl, bl
        rcl     ah, 1 ; ah = corrected flag with CF from DAADefTable
        stosw
        pop     cx
        inc     cx
        cmp     cx, 2048
        jb short loop_InitDAATable

IF LOG_DAA_Table
        mov     si, offset [DAATable2]
        mov     cx, 256
@loop_log_daa_lines:
        call    LogText
        DB      ' DW ',0
        push    cx
        mov     cx, 8
@loop_log_daa_words:
        lodsw
        call    LogText
        DB      '0',0
        call    LogAX
        call    LogText
        DB      'h',0
        cmp     cl, 1
        jz short @nx_log_daa_words
        call    LogText
        DB      ', ',0
@nx_log_daa_words:
        loop    @loop_log_daa_words
        call    LogText
        DB      13,10,0
        pop     cx
        loop    @loop_log_daa_lines
ENDIF

ENDIF ; SIMPLE_DAA

IF COMPRESS_TABLES
;--------------------------------------------------------------
; Decompress tables
;--------------------------------------------------------------
        ;++++++ 1st, clear area where decompressed data are stored
        ;mov     si, offset [TableOpd1]
        ;mov     di, si
        ;inc     di
        ;mov     cx, 7 * 256 * 4 - 1
        ;mov     byte ptr [si], 0
        ;rep     movsb
        ;++++++ init decompression
        mov     si, offset [Decompress_Cmd]
        mov     di, offset [TableOpd1]
        xor     bp, bp ; Rows counter
        xor     bx, bx ; init "Value" to $STORE
        ;++++++ main decompression loop
@decompress_loop:
        cmp     si, offset [Decompress_End]
        jge     @decompress_end
        lodsb
        call    de_exec
        jmp short @decompress_loop

$REPEAT EQU     0BFh ; use in form $REPEAT+N (1 <= N <= 64)
$FORWARD EQU     7Fh ; use in form $FORWARD+N (1 <= N <= 64)
$LOOP   EQU     -1 ; use in form $LOOP+N (1 <= N <= 64)
$MOVE   EQU     5Fh ; use in form $MOVE+N (1 <= N <= 32)
$ROWS   EQU     3Fh ; use in form $ROWS+N, Step (1 <= N <= 16, Step=0..255)
$BYTES  EQU     51h ; work with bytes
$WORDS  EQU     52h ; work with words
$RET    EQU     53h ; end loop / sub-sequence
$CALLFAR EQU    54h ; call sub-sequence, following word is starting position in data stream
$CALL   EQU     55h ; near call sub-sequence, following byte is offset to call
$OFFSET EQU     56h ; offset by given word in the target stream
$ROWS0  EQU     57h ; reset rows and step to 0
$STORE  EQU     58h ; store Value (bx)
$INC    EQU     59h ; inc Value (bx) by a word
IF LOGS and LOG_DECOMPRESS_TABLES
$LOG    EQU     5Ah ; output current offset and next cmd
                    ; with some comment text while logging
ENDIF

;------------------------------------------------------
de_exec proc ; execute single decompress instruction
;------------------------------------------------------
;IF LOGS and LOG_DECOMPRESS_TABLES
;        pusha
;        call    LogText
;        DB      '$cmd=',0
;        movzx   ax, al
;        call    LogAX
;        call    LogText
;        DB      13, 10, 0
;        popa
;ENDIF

        movzx   cx, al
        add     al, al
        jnc short @de_0

@de_1:  ; 1xxx.xxxx
        and     cl, 63
        add     al, al
        jnc short @de_10
@de_11: ; 11xx.xxxx = $REPEAT + N, (1 <= N <= 64)
        inc     cx
@de_repeat_loop:
        push    si
        call    de_move_data
        xchg    ax, si
        pop     si
        loop    @de_repeat_loop
        xchg    si, ax
        ret

@de_10: ; 10xx.xxxx = $FORWARD + N, (1 <= N <= 64) - forward in target
        inc     cx
        add     di, cx
        call    de_data_len
        jnc short @de_10_ret
        add     di, cx
@de_10_ret:
        ret

@de_0:  ; 0xxx.xxxx
        add     al, al
        jc short @de_01

@de_00: ; 00xx.xxxx = $LOOP + N (1 <= N <= 64) start loop
        pop     dx ;----------/
        push    si ; save position to return to in on each loop
        inc     cx ; other 6 bits = n - 1, where n = count to repeat
        push    cx
        push    dx ;----------\
        ret

@de_01: ; 01xx.xxxx
        add     al, al
        jc short @de_011

@de_010:; 010x.xxxx
        and     cl, 15
        add     al, al
        jc short @de_0101

@de_0100: ; 0100.xxxx - $ROWS + N (1 <= N <= 16)
;IF LOGS and LOG_DECOMPRESS_TABLES
;        pusha
;        mov     ax, cx
;        call    LogText
;        DB      '*** $ROWS:',0
;        call    LogAX
;        call    LogText
;        DB      13,10,0
;        popa
;ENDIF
        lodsb   ; get step also
;@de_0100_0:
@de_rows_0:
        mov     byte ptr [de_data_step+1], al
        inc     cx
        mov     bp, cx
        mov     word ptr [de_Rows_Count+1], cx
        mov     ah, 0
        inc     ax
        mul     cx
        dec     ax
        mov     word ptr [de_Row_Length+1], ax
;IF LOGS and LOG_DECOMPRESS_TABLES
;        call    LogText
;        DB 'after $ROWS: de_data_step=',0
;        mov     ax, word ptr [de_data_step+1]
;        call    LogAX
;        call    LogText
;        DB      13,10,0
;ENDIF
        ret

@de_011: ; 011x.xxxx = $MOVE + N (1 <= N <= 32)
        and     cl, 31
        inc     cx
@de_011_move_loop:
        call    de_move_data
        loop    @de_011_move_loop
        ret

@de_0101: ; 0101.xxxx special decompress command
        mov     al, 0F8h
        loop    @de_cmd_2
@de_cmd_1:      ; 0101.0001 = $BYTES - work with bytes
        dec     al
        inc     cx
@de_cmd_2:
        loop    @de_cmd_3
                ; 0101.0010 = $WORDS - work with words
        inc     al ; 0F9h = STC op code
        mov     byte ptr [de_data_len], al
        ret

@de_cmd_3:
        loop    @de_cmd_4
                ; 0101.0011 = $RET - end loop / end subsequence
        pop     dx ;-----------/
        pop     cx
        pop     ax
        jcxz    @de_ret_from_call
        loop    @de_loop_cont
        ; end of loop
        xchg    si, ax
        jmp short @de_ret_from_call
@de_loop_cont:
        push    ax
@de_loop_cont1:
        push    cx
@de_ret_from_call:
        xchg    si, ax
        push    dx ;-----------\
        ret

@de_cmd_4:
        loop    @de_cmd_5
                ; 0101.0100 = $CALLFAR - call sub-sequence
        pop     dx
        lodsw
@de_call_1:
        xor     cx, cx
        push    si
        jmp short @de_loop_cont1

@de_cmd_5:
        loop    @de_cmd_6
                ; 0101.0101 - $CALL - call sub-sequence by short offset
        pop     dx
        lodsb
        movsx   ax, al
        add     ax, si
        push    si
        xor     cx, cx
        jmp short @de_loop_cont1

@de_cmd_6:
        loop    @de_cmd_7
                ; 0101.0110 = $OFFSET - offset by word
        lodsw
        add     di, ax
        ret

@de_cmd_7:
        loop    @de_cmd_8
                ; 0101.0111 = $ROWS0 - reset rows and step
        xor     ax, ax
        jmp short @de_rows_0

@de_cmd_8:
        loop    @de_cmd_9
                ; 0101.1000 = $STORE - store value from bx
        mov     ax, bx
        stosw
        ret

@de_cmd_9:
        loop    @de_cmd_A
                ; 0101.1001 = $INC - add a word to Value in bx
        lodsw
        add     bx, ax

IF LOGS and LOG_DECOMPRESS_TABLES
        ret
@de_cmd_A: ; $LOG, 'some text',0
        pusha ;--------------------------\
        xchg    ax, di
        sub     ax, offset [TableOpd1]
        call    LogAX
        call    LogText
        DB      ' ',0
        push    si
@de_log_search_nx_cmd:
        lodsb
        test    al, al
        jnz short @de_log_search_nx_cmd
        mov     ax, word ptr [si]
        call    LogAX
        call    LogText
        DB      ' ',0
        mov     ax, bx
        call    LogAX
        pop     si
        call    LogText
        DB      ' ',0
        popa ;---------------------------/
        mov     dx, si
        xor     cx, cx
        dec     cx
@de_log_search_text_end:
        inc     cx
        lodsb
        test    al, al
        jnz short @de_log_search_text_end
        pusha ;-------------------------\
IF LOGBUFFER
        call    Log_Text2Buffer
ELSE
        call    Log_Text2File
ENDIF
        call    LogText
        DB      13,10,0
        popa ;--------------------------/
        ret
ELSE
@de_cmd_A:
        ret
ENDIF
de_exec endp ;-----------------------------------------

;------------------------------------------------------
de_move_data proc
;------------------------------------------------------
        push    cx ;---------------------------------\
        call    de_data_step ; cx = step
        call    de_data_len  ; CF = data len (set, if words, otherwise bytes)
        pushf   ;-------------------------\
        jnc short @de_move_byte
@de_move_word:
        movsw
        add     cx, cx
        DB      3Ch ; op code "cmp al,byte" to skip "movsb"
@de_move_byte:
        movsb
        add     di, cx
        jcxz    @de_move_after_step
        ;test    bp, bp
        ;jz      @de_move_after_step
        dec     bp
        jnz short @de_move_after_step
de_Row_Length:
        mov     cx, 0
        sub     di, cx
        popf    ;-------------------------/
        pushf   ;-------------------------\
        jnc short de_Rows_Count
        sub     di, cx
de_Rows_Count:
        mov     bp, 0
@de_move_after_step:
        popf    ;-------------------------/
        pop     cx ;---------------------------------/
        ret
de_move_data endp ;------------------------------------

;------------------------------------------------------
de_data_len proc
;------------------------------------------------------
        clc     ; replaced with stc when "work with words"
        ret
de_data_len endp ;-------------------------------------

;------------------------------------------------------
de_data_step proc
        mov     cx, 0 ; count bytes/words containing 0 to skip before put next data
        ret
de_data_step endp ;------------------------------------

@decompress_end:

ELSE  ; not COMPRESS_TABLES

IF LOGS and LOG_DECOMPRESS_TABLES
        DB      12Ch dup (90h) ; to synchronize jump addresses
ENDIF

ENDIF ; COMPRESS_TABLES

IF LOGS and LOG_DECOMPRESS_TABLES
        pusha
        call    LogText
        DB      '--------------------------------- Tables dump:',13,10,0
        mov     si, offset[TableOpd1]
        mov     dx, 230h+75h+1Dh
@log_tables_dump_loop0:
        mov     ax, si
        sub     ax, offset[TableOpd1]
        call    LogAX
        call    LogText
        DB      ':',0
        mov     cx, 8
@log_tables_dump_loop1:
        lodsw
        xchg    ah, al
        call    LogText
        DB      ' ',0
        call    LogAX
        loop    @log_tables_dump_loop1
        call    LogText
        DB      13,10,0
        dec     dx
        jnz short @log_tables_dump_loop0
        popa
        ;RET ;!!!! while debugging
ENDIF

        ;------ set palette
        ;push    ds
        ;pop     es ; es = ds here
        mov     ax, 1012h
        xor     bx, bx
IF HELP_ON_F1 and HELP_SHOW_KBD
        mov     cx, 16+49
ELSE
        mov     cx, 16
ENDIF
        mov     dx, offset[Palette]
        int     10h
;################################################
;************************************************
;       RESET & START EMULATION
;************************************************
;################################################
;------------------------------------------------
; cs = ds = code segment, with all the variables.
;------------------------------------------------
Reset_spectrum:
Reset_PC:
        mov     si, 0000h        ; PC = 0 when reset
Reset_Flags:
        mov     ah, 0
        ;
        ; ah (flags register) initialization (to 0 when reset)
        ;
Get_op_start:

IF HELP_ON_F1 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
        jmp short @get_op_code
        call    Show_Help
@get_op_code:
ENDIF ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

IF LOGS and LOG_Key_input
log_key_skip:
        jmp short log_key_skip_1
        jmp short log_key_output
log_key_skip_1:
        jmp short log_key_after
log_key_output:
        pusha ;--------------------------------------[
        mov     byte ptr cs:[log_key_skip+1], 2
log_key_load:
        mov     ax, 01234h
        call    LogText
        DB      '///////// key input scan-code: ', 0
        call    LogAX
        call    LogText
        DB      13,10,0
        popa ;---------------------------------------]
log_key_after:
ENDIF

IF SAVE_ON_F2 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Skip_Save_Sna:
        jmp short Skip_Save_Sna_1
        call    Save_Sna
Skip_Save_Sna_1:
ENDIF ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

IF LOAD_ON_F3 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
IF LOAD_SELECT
Skip_Load_Sel:
        jmp short Skip_Load_Sel_1
        pusha
        call    Load_Sel
        popa
Skip_Load_Sel_1:
ENDIF

Skip_Load_Sna:
        jmp short Skip_Load_Sna_1
        call    Load_Z_Sna
Skip_Load_Sna_1:
ENDIF ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;IF RELOAD_ON_F4 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Skip_ReLoad_Sna:
        jmp short Skip_ReLoad_Sna_1
        call    ReLoad_Sna
Skip_ReLoad_Sna_1:
;ENDIF ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

        mov     ecx, dword ptr [DelayTime]
        sar     ecx, 4
        inc     ecx
Short_Delay:
        dec     ecx
        jnz short Short_Delay
IF SPECTRUM_128K
ELSE
        mov     es, word ptr [Spectrum_Memory_Segment]
                ; es initially assigned to Spectrum memory segment to allow
                ; loading instruction
ENDIF

        push    cs
        pop     seg_Op 
                ; initially seg_Op is equal to ds=cs and can be used to access
                ; operation tables (JumpTable, TableOpd1, TableOpd2 and TactTable)
                ; But for commands with prefixes it is shifted to access
                ; correspondent operation tables.

        lea     base_HL, [Rmain_+R_HL]
                ; usually base_HL (bp) points to R_HL in the current registers
                ; bank, but prefixes DD and FD switch it to point to
                ; R_IX and R_IY correspondently.
Get_op:
        mov     al, byte ptr [After_EI]
        shr     al, 1
        mov     byte ptr [After_EI], al

IF SPECTRUM_128K ;......................128K
        xchg    bx, si
        call    Read_Mem
        xchg    bx, si
        movzx   bx, dl
ELSE             ;......................48K
        movzx   bx, byte ptr es:[si]   ; load code operation
ENDIF            ;......................

IF      LOGS
IF      TRACE_ALL
        xchg    ax, si
        call    LogAX
        xchg    ax, si
IF      TRACEREGS

        call    LogText
        DB      ': ',0
        xchg    ax, bx
        call    LogAX
        xchg    ax, bx

        call    LogText
        DB      ' AF=',0
        push    ax ;---------------------------------\
        push    dx ;----------------------------\
        mov     ax, word ptr cs:[R_AF]
        call    LogAX
        call    LogText
        DB      ' BC=',0
        mov     ax, word ptr cs:[Rmain_+R_BC]
        call    LogAX
        call    LogText
        DB      ' DE=',0
        mov     ax, word ptr cs:[Rmain_+R_DE]
        call    LogAX
        call    LogText
        DB      ' HL=',0
        mov     ax, word ptr cs:[Rmain_+R_HL]
        call    LogAX
        call    LogText
        DB      ' SP=',0
        mov     ax, word ptr cs:[R_SP]
        call    LogAX

        call    LogText
        DB      ' IX=',0
        mov     ax, word ptr cs:[R_IX]
        call    LogAX

        call    LogText
        DB      ' IY=',0
        mov     ax, word ptr cs:[R_IY]
        call    LogAX

        ;call    LogText
        ;DB      ' IR=',0
        ;mov     ah, byte ptr cs:[R_I]
        ;mov     al, byte ptr cs:[R_R]
        ;and     al, 7Fh
        ;or      al, byte ptr cs:[R_R_hi]
        ;call    LogAX

IF LOG_MEMORY
        call    LogText
        DB      ' Mem=',0
IF SPECTRUM_128K
        push    bx
        push    dx
        mov     bx, LOG_MEM_ADDR+1
        call    Read_Mem
        mov     ah, dl
        dec     bx
        call    Read_Mem
        mov     al, dl
        pop     dx
        pop     bx
ELSE
        push    es
        mov     es, word ptr cs:[Spectrum_Memory_Segment]
        mov     ax, word ptr es:[LOG_MEM_ADDR]
        push    es
ENDIF
        CALL    LogAX
ENDIF

        ;push    bx
        ;call    LogText
        ;DB      ' (SP)=',0
        ;mov     bx, word ptr cs:[R_SP]
        ;mov     al, byte ptr es:[bx]
        ;inc     bx
        ;mov     ah, byte ptr es:[bx]
        ;call    LogAX
        ;pop     bx

        ;push    bx ;------------------------------\
        ;call    LogText
        ;DB      ' (DEHL)=',0
        ;mov     bx, word ptr cs:[Rmain_+R_DE]
        ;mov     ah, byte ptr es:[bx]
        ;mov     bx, word ptr cs:[Rmain_+R_HL]
        ;mov     al, byte ptr es:[bx]
        ;pop     bx ;------------------------------/
        ;call    LogAX

        ;mov     ah, byte ptr cs:[R_I]
        ;mov     al, byte ptr cs:[ImMode]
        ;call    LogText
        ;DB      ' I,IM=',0
        ;call    LogAX

IF SPECTRUM_128K
;        push    bx
;        push    dx
;        mov     bx, 0C000h
;        call    Read_Mem
;        mov     al, dl
;        mov     ah, byte ptr [Port_7FFD]
;        call    LogText
;        DB      ' Port_7FFD,[C000]=',0
;        call    LogAX
;        pop     dx
;        pop     bx
ENDIF

IF LOG_Screen_CC
        push    bx ;------------------------\
        mov     bx, 4000h
        xor     ax, ax
@loop_calc_screen_CC:
        rol     ax, 1
        xor     al, byte ptr es:[bx]
        inc     bx
        cmp     bx, 5B00h
        jb short @loop_calc_screen_CC
        call    LogText
        DB      ' screen CC=', 0
        call    LogAX
        pop     bx ;------------------------/
ENDIF

        pop     dx ;----------------------------/
        pop     ax ;---------------------------------/
ELSE
        call    LogText
        DB      ': ',0
        xchg    ax, bx
        call    LogAX
        xchg    ax, bx

ENDIF   ; TRACEREGS
        call    LogText
        DB      13,10,0
ENDIF   ; TRACE_ALL
ENDIF   ; LOGS

        movzx   cx, byte ptr seg_Op:[TactTable+bx]
        add     word ptr [TCounter], cx
        jge     End_pixel_line
        ;---------------------------------------------------------
        ; continue interpreting instruction after handling
        ; video output (16 pixel lines at a time,
        ; 600 times per second)
        ;---------------------------------------------------------
Cont_loop:
IF CAREFUL_INT
        cmp     bl, 76h
        jz short Cont_loop1
        cmp     byte ptr [IFF1IFF2], 0
        jz short Cont_loop1
        cmp     byte ptr [After_EI], 0
        jnz short Cont_loop1
        cmp     byte ptr [IntSignal], 0
        jz short Cont_loop1
        push    dx
        mov     dx, word ptr [TCntInt]
        cmp     dx, word ptr [TCounter]
        pop     dx
        jl short Cont_loop1
IF LOGS and LOG_Reason_Int
        call    LogText
        DB      '*** INT since INT signal ***',13,10,0
ENDIF
        call    Interrupt_
        jmp short Get_op_start
ENDIF
Cont_loop1:
        inc     si            ; IP = IP + 1
        call    Execute_instruction ; execute single instruction
        jmp     Get_op_start  ; continue main loop

;---------------------------------------------------------
; EXECUTING INSTRUCTION
; bh = 0
; bl = op code
; fs = cs for normal instruction
;    = cs+(512+256+256+256)/16=cs+80 for instructions with *CB* prefix
;    = cs+160 for instructions with *IX* or *IY* prefixes
;    = cs+240 for instructions with *IXCB* or *IYCB* prefixes
;    = cs+320 for instructions with *ED* prefix
; bp = address of R_HL for all instructions except *IX* or *IY*
;    = or address of R_IX or R_IY
; si = address of the next instruction (for 1-byte length instruction)
;      or address of the next byte in the instruction
; di = address of the current registers bank
; ah = flags register
;---------------------------------------------------------
Execute_instruction:
        inc     byte ptr [R_R]
        push    bx ;-----------------------------------\

        mov     al, byte ptr seg_Op:[TableOpd2+bx]
        call    Get_Operand
        ;
        ;       dx = opd2
        ;       cx = ???
        ;       es = its segment
        ;

        mov     al, byte ptr seg_Op:[TableOpd1+bx]
        test    al, ST_
        jnz short Store_Result_Needed
        push    dx ;-------------\
        xor     dx, dx ; opd1 for opdNo should be = 0
        call    Get_Operand
        pop     cx ;-------------/
        ;
        ;       dx = opd1; cx = opd2
        ;
        add     bx, bx
        ;-----------------------------------------;
        ; call certain instruction procedure      ;
        ;-----------------------------------------;
        call    word ptr seg_Op:[JumpTable+bx]
Update_Flags:
        ;-----------------------------------------;
        ;                                         ;
        ;       ah = flags, al = YF, XF flags     ;
        ;                                         ;
        ;-----------------------------------------;
        pop     bx ;-----------------------------------/
        add     bx, bx
        mov     cx, word ptr seg_Op:[TableUpdFlgs+bx]
        test    ch, XF or YF
        ;jz      Update_Flags_1
        DB      74h, Update_Flags_1 - 1 - $
        and     ax, ((not(XF or YF)) shl 8) or (XF or YF)
        or      ah, al
Update_Flags_1:
        and     ah, ch
        not     cx
        and     ch, byte ptr [R_F]
        or      ah, ch
        and     ah, cl
        mov     byte ptr [R_F], ah
        ret

Store_Result_Needed:
        ;-----------------------------------;
        ;       al = opd1 type              ;
        ;       bh = 0; bl = op code        ;
        ;       dx = opd2                   ;
        ;-----------------------------------;
        and     al, 7Fh
        push    ax ;-----------------------\
        push    dx ;--------------------\
        call    Get_OperandAddr

        ;++++++ get opd1 to DX by its address DX:CX (or Spectrum_Mem[ CX ], if 128K)
IF SPECTRUM_128K
ELSE
        mov     es, dx
ENDIF
        push    bx ;-----------------\
        mov     bx, cx

        ;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ;       self-modifying code
IF SPECTRUM_128K ; store also flag/segment of the opd1
        mov     word ptr [Store_Result_S128+1], dx
ENDIF
        mov     word ptr [Store_Result_+1], bx
        ;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
IF SPECTRUM_128K ;........................128K
        test    dx, dx
        jz short @get_opd1_from_mem
        mov     dl, byte ptr [bx]
        inc     bx
        mov     dh, byte ptr [bx]
        jmp short @opd1_loaded
@get_opd1_from_mem:
        inc     bx
        call    Read_Mem
        mov     dh, dl
        dec     bx
        call    Read_Mem
@opd1_loaded:
ELSE             ;........................48K
        mov     dl, byte ptr es:[bx]
        inc     bx
        mov     dh, byte ptr es:[bx]
ENDIF            ;........................
        pop     bx ;-----------------/

        ;++++++ get opd2 from stack to CX
        pop     cx ;--------------------/

        ;++++++ execute instruction
        add     bx, bx
        ;-----------------------------------------;
        ; call certain instruction procedure      ;
        ;-----------------------------------------;
        call    word ptr seg_Op:[JumpTable+bx]

Store_Result_:
        mov     bx, 1234h ; actual value is stored before executing instruction

        ;++++++ restore opd1 type
        pop     cx ;-----------------------/
        cmp     cl, opdBC
IF SPECTRUM_128K
Store_Result_S128:
        mov     cx, 1234h ; segment (ds) for regs OR 0 for Speccy memory
ENDIF
        jb short One_Byte_Result

        ;++++++ store low byte of the result
IF SPECTRUM_128K ;....................................128K
        jcxz    @S128_store_result_lo_2mem
        mov     byte ptr [bx], dl
        jmp short store_result_hi
@S128_store_result_lo_2mem:
        call    Write_Mem ; Spectrum_Mem[ bx ] = dl
ELSE             ;....................................48K
        cmp     bh, 40h
        jb short store_result_hi
        mov     byte ptr es:[bx], dl
ENDIF            ;....................................
store_result_hi:
        inc     bx

        ;++++++ store high byte of the result
        mov     dl, dh
One_Byte_Result:
IF SPECTRUM_128K ;....................................128K
        jcxz    @S128_store_result_hi_2mem
        mov     byte ptr [bx], dl
        jmp short Update_Flags
@S128_store_result_hi_2mem:
        call    Write_Mem ; Spectrum_Mem[ bx ] = dl
ELSE             ;....................................48K
        cmp     bh, 40h
        jb short Update_Flags
        mov     byte ptr es:[bx], dl
ENDIF            ;....................................
        jmp short Update_Flags

;*************************************************************
; Output next pixel line to the video buffer
; 1950 times a second (this is equivalent of 50Hz refresh rate).
; Also, after each 312 calls (entire screen output) check
; here if to signal interrupt.
;*************************************************************
End_pixel_line:
IF LOGS and LOG_end_pixel_line
        call    LogText
        DB      '--------------End of pixel line',13,10,0
ENDIF
        pusha
        ;------ 1st, decrement TCounter back on 224 TStates
        sub     word ptr [TCounter], 224
        ;------ 2nd, get current VideoCount counter and handle video output
        mov     bx, word ptr [VideoCount]
        mov     ax, bx
        add     bx, bx
        mov     di, word ptr [VideoLines+bx]
        mov     si, word ptr [PixelLines+bx]
        mov     bx, word ptr [AttrLines+bx]
        ;------ now save es and ds and set to target (A000) and source (Spectrum_Memory_Segment) segments
        push    es ;----------------------------------------------------\
        mov     dx, 0A000h
        mov     es, dx ; target segment in video memory
        push    ds ;-----------------------------------------------\
IF SPECTRUM_128K ;.......................................128K
        mov     dx, 1400h - 400h ; segment corresponding to RAM bank 5
        test    byte ptr [Port_7FFD], 8 ; if video page 1 active ?
        jz short Vbank_selected
        mov     dx, 1C00h - 400h ; segment corresponding to RAM bank 7
Vbank_selected:
        add     dx, word ptr [Spectrum_Memory_Segment]
ELSE             ;.......................................48K
        mov     dx, word ptr [Spectrum_Memory_Segment]
ENDIF            ;.......................................
        sub     ax, 64
        cmp     al, 191
        jbe short Vout_start
        ;------ when out of screen, output from dummy pixel lines which is locating
        ;       in command segment rather then in spectrum memory
        mov     dx, cs
Vout_start:
        mov     byte ptr [IntSignal], 0
        mov     ds, dx ; source segment
        ;------ output 1 pixel line from Spectrum video buffer to video page
        ; ds = Spectrum memory segment OR = cs for dummy output out of screen
        ; si = source pixel line (8 bit per byte, 32 bytes in a line)
        ; bx = source attribute line (1 byte for each 8 pixels)
        ; es = Video buffer segment (A000h)
        ; di = target pixel line (128 bits per line)
        ;-------------------------------------------------------------
Vout_loop_lines:
        mov     cl, 32 ; 32 bytes to output
Vout_loop_chars:
        mov     dl, byte ptr [bx]; attribute for next 8 bits
        mov     byte ptr cs:[CurAttrByte], dl
        inc     bx               ; FBXXXYYY
                                 ;      YYY - fore color 0..7 - black..white
                                 ;   XXX - back color 0..7 - black..white
                                 ;  B - brightness
                                 ; F - flash fore and back color FlashState=1
        mov     dh, dl
        sar     dl, 3
        mov     al, dh
        and     dx, 0707h
        add     al, al
        jnc short no_flash
        test    byte ptr cs:[FrameCount], 10h
        jz short no_flash
        xchg    dl, dh
no_flash:
        add     al, al
        jnc short no_bright
        or      dx, 0808h
no_bright:
        ; dl = back color
        ; dh = fore color
        lodsb
        mov     ah, al           ; ah = next byte (8 bits to output from hi to lo)
        mov     ch, 8
Vout_loop_bits:
        mov     al, dl
        add     ah, ah
        jnc short Vout_pixel
        mov     al, dh
Vout_pixel:
        stosb
        dec     ch
        jnz short Vout_loop_bits
        loop    Vout_loop_chars
        ;------ restore es = Spectrum_Memory_Segment, ds = cs
        pop     ds ;-----------------------------------------------/
        pop     es ;----------------------------------------------------/
after_video_out:
        ;------ now increment VideoCount
        lea     bx, [VideoCount]
        inc     word ptr [bx]
        cmp     word ptr [bx], 312
        jb short video_out_end
        ;------ 312 lines output finished, end of frame
        mov     word ptr [bx], cx ; reset pixel lines counter to 0
        ;------ now increment FrameCount
        inc     bx
        inc     bx ; bx = address of FrameCount
        inc     byte ptr [bx]
video_out_frame_end:
video_out_end:
        ;------ delay after each pixel line
        mov     ecx, dword ptr [DelayTime]
delay:  dec     ecx
        jnz short delay
        inc     word ptr [WaitCounter] ; count how much lines output between two timer ticks
        ;------ wait for tick which occur 18.2 times a second
        mov     bx, offset[TimerTick]
        cmp     byte ptr [bx], cl
        jz short delay_end
        mov     byte ptr [bx], cl ; reset TimerTick flag
        inc     bx ; bx = address of TwoTicksOrMore flag
        cmp     byte ptr [bx], cl ; check TwoTicksOrMore flag
        mov     byte ptr [bx], 1  ; set TwoTicksOrMore flags
        jz short clear_wait_counter
        inc     bx ; bx = @ DelayTime
        cmp     word ptr [WaitCounter], 857 ; 312 lines x 50 fr/sec in Speccy / 18.2 fr/sec in PC
        mov     eax, dword ptr [bx]
        je short clear_wait_counter
        ja short inc_delay
        sar     eax, 1
        inc     eax
        mov     dword ptr [bx], eax ; DelayTime := DelayTime / 2 + 1
        jmp short clear_wait_counter
inc_delay:
        sar     eax, 1
        inc     eax
        add     dword ptr [bx], eax ; DelayTime := DelayTime * 1.5;
clear_wait_counter:
        mov     word ptr [WaitCounter], cx ; 0
delay_end:
        ;------ all works finished, continue instruction execution
        popa
        cmp     word ptr [VideoCount], 56 ; signal int only 16 pixels before 1st pixel line
        jnz short @2_Cont_loop
        ;------ it is a time to signal Int:
IF CAREFUL_INT
        push    ax
        mov     ax, word ptr [TCounter]
        add     ax, 23
        mov     word ptr [TCntInt], ax
        pop     ax
ENDIF
        mov     byte ptr [IntSignal], 1

IF LOGS and LOG_OPCODE_onINT
        pusha
        mov     ax, word ptr es:[si]
        call    LogText
        DB      'opcode: ',0
        call    LogAX
        call    LogText
        DB      ' at addr: ',0
        mov     ax, si
        call    LogAX
        call    LogText
        DB      13,10,0
        popa
ENDIF
        cmp     byte ptr [After_EI], 0
@2_Cont_loop:
        jnz     Cont_loop1
        cmp     byte ptr [IntSignal], 0
        jz short no_int_signal
        cmp     byte ptr [IFF1IFF2], 0
no_int_signal:
        jz      Cont_loop1

IF LOGS and LOG_Reason_Int
        call    LogText
        DB      '*** possible int signal ***',13,10,0
ENDIF

IF SPECTRUM_128K ;.............................128K
        xchg    bx, si
        call    Read_Mem
        xchg    bx, si
        cmp     dl, 76h
ELSE             ;.............................48K
        cmp     byte ptr es:[si], 76h
ENDIF            ;.............................
        jz short  @no_int
IF LOGS and LOG_Reason_Int
        call    LogText
        DB      '*** interrupt on frame start ***',13,10,0
ENDIF

        call    Interrupt_
@no_int:
        jmp     Get_op_start


;*************************************************************
; IMPLEMENTATION OF PREFIXES _DD_ (IX) AND _FD_ (IY)
; dh = 0
; dl = next op code
;*************************************************************
_IX_:
        mov     base_HL, offset[R_IX]
        jmp short _IX_IY_
_IY_:
        mov     base_HL, offset[R_IY]
_IX_IY_:
        mov     bx, dx
        mov     dx, cs
        add     dx, TablesSet_SegmentsSize * 2
        mov     seg_Op, dx
        jmp     Execute_instruction

XY_IX_:                  ; another IX / IY prefix following this
XY_IY_: dec     si
        ret

;*************************************************************
; IMPLEMENTATION OF PREFIX _CB_
;*************************************************************
XY_CB_: ; *CB* prefix following *IX* or *IY* prefix
; ch = 0
; cl = next op code
; dx = memory address calculated as IX+offset or IY+offset depending on
;      previous DD or CB prefix.
        dec     byte ptr [R_R] ; DDCB / FDCB increases R only by two
        xchg    cx, dx
        push    cx ;------\
        push    dx ;---\
        call    _CB_
        pop     bx ;---/
        pop     cx ;------/
        sub     cl, 40h
        cmp     cl, 40h
        jb short xy_cb_end ; except BIT n, (IX/IY+offset), r instructions
IF SPECTRUM_128K ;...................................128K
        call    Write_Mem ; Spectrum_Mem[ bx ] = dl
ELSE             ;...................................48K
        mov     byte ptr es:[bx], dl ; store also result in memory
ENDIF            ;...................................
xy_cb_end:
        ret
_CB_:
; ch = 0
; cl = next op code
; for DDCB and FDCB instructions, dx = address IX+offset or IY+offset
        mov     bx, cx
        mov     cx, seg_Op
        add     cx, TablesSet_SegmentsSize
        mov     seg_Op, cx
        jmp     Execute_instruction

;*************************************************************
; IMPLEMENTATION OF PREFIX _ED_
; dh = 0
; dl = next op code
;*************************************************************
_ED_:
        mov     bx, dx
        mov     dx, seg_Op
        add     dx, TablesSet_SegmentsSize * 4
        mov     seg_Op, dx
        jmp     Execute_instruction

;-------------------------------------------------------------------------
; operand encoding constants - used to define operand tables
;-------------------------------------------------------------------------
opdNo   equ     0
opdC    equ     1
opdB    equ     2
opdE    equ     3
opdD    equ     4
opdLx   equ     5 ; L/IXL/IYL
opdHx   equ     6 ; H/IXH/IYH
opdL    equ     7
opdH    equ     8
opdA    equ     9
opdM    equ     0Ah ; (HL)/(IX+offset)/(IY+offset)
opd@BC  equ     0Bh ; (BC)
opd@DE  equ     0Ch ; (DE)
opdByte equ     0Dh ; 1 byte in the instruction immediately
opdI    equ     0Eh ; I register
opdR    equ     0Fh ; R register
opdS    equ     10h ; special M when IXCB or IYCB instruction: bp = memory address
opdAddr1 equ    11h ; 2 bytes address in the instruction of 1 byte in memory
opdBC   equ     12h
opdDE   equ     13h
opdHL   equ     14h ; HL/IX/IY
opdAF   equ     15h
opdSP   equ     16h
opdPC   equ     17h
opdWord equ     18h ; 2 bytes word in the operation immediately
opdAddr2 equ    19h ; 2 bytes address in the instruction of 2 bytes in memory
opdJR   equ     1Ah ; 1 byte in the instruction as an offset from PC
opdCondNZ equ   1Bh ; if ZF then 0 else 1
opdCondZ  equ   1Ch ; if ZF then 1 else 0
opdCondNC equ   1Dh ; if CF then 0 else 1
opdCondC  equ   1Eh ; if CF then 1 else 0
opdCondPO equ   1Fh ; if PF then 0 else 1
opdCondPE equ   20h ; if PF then 1 else 0
opdCondP  equ   21h ; if SF then 0 else 1
opdCondM  equ   22h ; if SF then 1 else 0
opdCondDJNZ equ 23h ; special djnz condition: B = B - 1; if B = 0 then 0 else 1
opdbbb  equ     24h ; 3 lowest bits from op code (for rst commands)
opdNbit equ     25h ; = 1 shl ( bits d5.d4.d3 from the op code )
opdMbit equ     26h ; low byte the same as for opdNbit,
                    ; high byte is loaded from es:[bp]
opd@M   equ     27h ; address of (HL)/(IX+offset)/(IY+offset)
opdM_XYCB equ   28h ; operand address already loaded (DX), load it into ch

ST_     equ     80h ; flag "store result in 1st operand" - only for the 1st operand

;-------------------------------------------------------------------------
; in : AL = operand type
;      BL = op code * 2
; out: DX = operand
;-------------------------------------------------------------------------
Get_Operand:
        test    al, al
        jz short Get_Operand_End ; for opdNo, DX is returned as it is passed, w/o changes
        and     al, 7Fh
        push    bx ;------------------------------\
        push    ax ;---------------\
        call    Get_OperandAddr
        pop     bx ;---------------/
        mov     bh, 0
IF SPECTRUM_128K
ELSE
        mov     es, dx
ENDIF
        push    di ;---------------------------\
        mov     di, cx
        add     bx, bx
        call    word ptr [GetOpd_Table+bx]

IF SPECTRUM_128K
ELSE
        mov     es, word ptr [Spectrum_Memory_Segment]
ENDIF

        pop     di ;---------------------------/
        pop     bx ;------------------------------/
Get_Operand_End:
        ret

GetOpd_Table:
;        C     B     E     D     Lx    Hx    L
dw go_0, go_1, go_1, go_1, go_1, go_1, go_1, go_1

;  H     A     M     @BC  @DE   Byte   I     R
dw go_1, go_1, go_1, go_1, go_1, go_0, go_1, go_0

;  S     Addr1 BC    DE    HL    AF    SP    PC
dw go_1, go_1 ,go_2, go_2, go_2, go_2, go_2, go_2

;  Word  Addr2 JR    ?NZ   ?Z    ?NC   ?C    ?PO
dw go_0, go_11,go_0, go_01,go_01,go_01,go_01,go_01

;  ?PE   ?P    ?M    DJNZ  bbb   Nbit  Mbit  @M
dw go_01,go_01,go_01,go_0, go_0, go_0, go_0, go_0

;  M_XYCB
dw go_0

;------------------------------------------------
; for all go_... entries:
; di = address of operand obtained in Get_OperandAddr (CX)
; es = segment obtained there (DX)
;------------------------------------------------

go_0:   ;- immediate word operand - already loaded (di), just return it
        mov     dx, di
        ret

go_01:  ;- immediate byte operand from di, clear high byte
        mov     dx, di
        mov     dh, 0
        ret


go_1:   ;- 1 byte register or memory
IF SPECTRUM_128K ;.......................128K
        test    dx, dx
        jz short @go_1_from_Mem
        movzx   dx, byte ptr [di] ; from a register
        ret
@go_1_from_Mem:
        xchg    bx, di
        call    Read_Mem
        mov     dh, 0
        xchg    bx, di
ELSE             ;.......................48K
        movzx   dx, byte ptr es:[di]
ENDIF            ;.......................
        ret

go_2:   ;- 2 bytes register
        mov     dx, word ptr [di]
        ret

go_11:  ;- 2 bytes from memory, never from a register pair
IF SPECTRUM_128K ;.......................128K
        xchg    bx, di
        inc     bx
        call    Read_Mem
        mov     dh, dl
        dec     bx
        call    Read_Mem
        xchg    bx, di
ELSE             ;.......................48K
        mov     dl, byte ptr es:[di]
        inc     di
        mov     dh, byte ptr es:[di]
ENDIF            ;.......................
        ret

;-------------------------------------------------------------------------
; in : AL = operand type (bit7 already reset)
;      BL = op code * 2
; out: DX:CX = segment:address of the register/memory (48K)
IF SPECTRUM_128K
;      CX = address in Spectrum memory, DX = 0
;       or
;      DX = DS; CX = address (for registers)
ENDIF
;-------------------------------------------------------------------------
Get_OperandAddr:
        mov     cx, dx ; initially result = dx
        mov     dx, cs ; for most operands, those are registers
        push    bx ;----------------------------\
        xchg    al, bl ; al = op code
        xchg    al, ah ; al = flags, ah = op code
        mov     bh, 0
        add     bx, bx
        call    word ptr [GetOpdAddr_Table+bx]
        mov     ah, al
        pop     bx ;----------------------------/
        ret

GetOpdAddr_Table:
dw go_No, go_C, go_B, go_E, go_D, go_Lx, go_Hx, go_L
dw go_H, go_A, go_M, go_@BC, go_@DE, go_byte, go_I, go_R
dw go_S, go_Addr1, go_BC, go_DE, go_HL, go_AF, go_SP, go_PC
dw go_Word, go_Addr2, go_JR, go_CondNZ, go_CondZ, go_CondNC, go_CondC
dw go_CondPO, go_CondPE, go_CondP, go_CondM, go_CondDJNZ, go_bbb, go_Nbit, go_Mbit
dw go_@M, go_M_XYCB

;------------------------------------------------------------------------
; for all following go_XX functions:
; base_HL    = bp - points to selected HL / IX / IY
;       ah = op code
;       al = flags
;       cx = 0
;------------------------------------------------------------------------

go_BC:
go_C:   mov     cx, offset [Rmain_+R_C]
go_No:
        ret

go_B:   lea     cx, [Rmain_+R_B]
        ret

go_DE:
go_E:   lea     cx, [Rmain_+R_E]
        ret

go_D:   lea     cx, [Rmain_+R_D]
        ret

go_HL:
go_Lx:  ;lea     cx, [base_HL+R_L]
        mov     cx, base_HL
        ret

go_Hx:  lea     cx, [base_HL+R_H]
        ret

go_L:   lea     cx, [Rmain_+R_HL]
        ret

go_H:   lea     cx, [Rmain_+R_HL+1]
        ret

go_AF:
go_F:
        mov     cx, offset[R_F]
        ret

go_A:   mov     cx, offset[R_A]
        ret

go_@M:  ; this entry used for DDCB / FDCB - prefixed instructions.
        ; Like for opdM, this operand type returns address of memory
        ; calculated as IX+offset or IY+offset.
go_M:
IF SPECTRUM_128K ;............................................128K
        xor     dx, dx ; dx = 0 indicates that Read_Mem / Write_Mem should be used
ELSE             ;............................................48K
        mov     dx, word ptr [Spectrum_Memory_Segment]
                ; use Spectrum memory rather then registers
ENDIF            ;............................................
        cmp     base_HL, offset[R_IY]
        jb short go_M_HL
IF SPECTRUM_128K ;............................................128K
        push    dx ;----------------\
        xchg    bx, si
        call    Read_Mem
        movsx   cx, dl
        xchg    bx, si
        pop     dx ;----------------/
ELSE             ;............................................48K
        movsx   cx, byte ptr es:[si] ; get offset byte from the instruction
ENDIF            ;............................................
        inc     si
        add     cx, word ptr [base_HL]
        mov     byte ptr [RHidden], ch
        ret

go_M_HL:mov     cx, word ptr [base_HL]
        ret

go_@BC:
IF SPECTRUM_128K ;...........................................128K
        xor     dx, dx
ELSE             ;...........................................48K
        mov     dx, word ptr [Spectrum_Memory_Segment]
                ; use Spectrum memory rather then registers
ENDIF            ;...........................................
        mov     cx, word ptr [Rmain_+R_BC]
        ret

go_@DE:
IF SPECTRUM_128K ;...........................................128K
        xor     dx, dx
ELSE             ;...........................................48K
        mov     dx, word ptr [Spectrum_Memory_Segment]
                ; use Spectrum memory rather then registers
ENDIF            ;...........................................
        mov     cx, word ptr [Rmain_+R_DE]
        ret

go_byte:
IF SPECTRUM_128K ;...........................................128K
        push    dx ;------------\
        xchg    bx, si
        call    Read_Mem
        movzx   cx, dl
        xchg    bx, si
        pop     dx ;------------/
ELSE             ;...........................................48K
        movzx   cx, byte ptr es:[si]
ENDIF            ;...........................................
        inc     si
        ret

go_I:   mov     cx, offset[R_I]
        ret

go_R:
        movzx   cx, byte ptr [R_R]
        and     cl, 7Fh
        or      cl, byte ptr [R_R_hi]
        ret

go_S:   mov     cx, bp ; special access to (IX+offset)/(IY+offset)
        ret            ; in CB-prefixed instructions: bp = IX+offset or IY+offset

go_SP:  mov     cx, offset[R_SP]
        ret

go_Addr1:
go_Addr2:
IF SPECTRUM_128K ;...........................................128K
        xor     dx, dx
ELSE             ;...........................................48K
        mov     dx, word ptr [Spectrum_Memory_Segment]
                ; use Spectrum memory rather then registers
ENDIF            ;...........................................
        ;
go_Word:
IF SPECTRUM_128K ;...........................................128K
        push    dx ;----------------\
        xchg    bx, si
        call    Read_Mem
        mov     cl, dl
        inc     bx
        call    Read_Mem
        mov     ch, dl
        inc     bx
        xchg    bx, si
        pop     dx ;----------------/
ELSE             ;...........................................48K
        mov     cl, byte ptr es:[si]
        inc     si
        mov     ch, byte ptr es:[si]
        inc     si
ENDIF            ;...........................................
        ret

go_PC:  mov     word ptr [R_PC], si ; first provide R_PC to hold current PC
        mov     cx, offset[R_PC]
        ret

go_JR:
IF SPECTRUM_128K ;...........................................128K
        push    dx ;-------------\
        xchg    bx, si
        call    Read_Mem
        movsx   cx, dl
        xchg    bx, si
        pop     dx ;-------------/
ELSE             ;...........................................48K
        movsx   cx, byte ptr es:[si]
ENDIF            ;...........................................
        inc     si
        add     cx, si
        mov     byte ptr [RHidden], ch
        ret

go_CondNZ:
        test    al, ZF
        setz    cl
        ret

go_CondZ:
        test    al, ZF
        setnz   cl
        ret

go_condNC:
        test    al, CF
        setz    cl
        ret

go_condC:
        test    al, CF
        setnz   cl
        ret

go_condPO:
        test    al, PF
        setz    cl
        ret

go_condPE:
        test    al, PF
        setnz   cl
        ret

go_condP:
        test    al, SF
        setz    cl
        ret

go_condM:
        test    al, SF
        setnz   cl
        ret

go_CondDJNZ:
        dec     byte ptr [Rmain_+R_B]
        setnz   cl
        mov     ch, 0
        ret

go_bbb:
        movzx   cx, ah
        and     cl, 38h
        ret

go_Nbit:
        mov     cl, ah
        shr     cl, 3
        and     cl, 7
        mov     ch, 1
        shl     ch, cl
        movzx   cx, ch
        ret

go_M_XYCB:
go_Mbit:
        push    cx
        call    go_Nbit
        pop     bx
IF SPECTRUM_128K ;..............................128K
        push    dx ;--------\
        call    Read_Mem
        mov     ch, dl
        pop     dx ;--------/
ELSE             ;..............................48K
        mov     ch, byte ptr es:[bx]
ENDIF            ;..............................
        ret


;*************************************************************
; IMPLEMENTATION OF INSTRUCTIONS  ///////////////////////////*
;-----------------------------------------------------;//////*
;       ds = cs                                       ;//////*
;       es = segment of target (cs if a register,     ;//////*
;                               es if data in memory) ;//////*
; in:   dx or dl = first operand                      ;//////*
;       cx or cl = second operand                     ;//////*
;       ah = flags                                    ;//////*
; out:  dx or dl = result to store (if necessary)     ;//////*
;       ah = flags, al = XF, YF                       ;//////*
;*************************************************************
; if opd1 is opdNo, dx = 0
; but if opd2 is opdNo, cx is undefined

Ld_A_I: ; opd1 = ST_+opdA, opd2 = opdI
Ld_A_R: ; opd1 = ST_+opdA, opd2 = opdR
        mov     dl, cl
        cmp     cl, 0
        lahf
        and     ah, not PF
        or      ah, byte ptr[IFF1IFF2]
        mov     al, dl
        ret

Ld_R_A: ; opd1 = ST_+opdR, opd2 = opdA
        mov     dl, cl
        and     cl, 80h
        mov     byte ptr [R_R_hi], cl
        jmp short Ld_

Ex_de_hl: ; opd1 = ST_+opdDE, opd2 = opdHL
        mov     word ptr [base_HL], dx
Ld_:
        mov     dx, cx
Nop_:   ret

Ld_@XY_byte:
IF SPECTRUM_128K ;......................128K
        xchg    bx, si
        call    Read_Mem
        xchg    bx, si
ELSE             ;......................48K
        mov     dl, byte ptr es:[si]
ENDIF            ;......................
        inc     si
        ret

Ldd_:   ; opd1 = ST_+opd@DE, opd2 = opdM
        dec     word ptr [base_HL]
        dec     word ptr [Rmain_+R_DE]
        jmp short Ldi_Ldd_
Ldi_:   ; opd1 = ST_+opd@DE, opd2 = opdM
        inc     word ptr [base_HL]
        inc     word ptr [Rmain_+R_DE]
Ldi_Ldd_:
        mov     dl, cl
        ; handling flags X and Y:
        add     cl, byte ptr [R_A]
        mov     al, cl
        shl     al, 4
        and     cl, XF
        or      al, cl
        ; handling other flags:
        and     ah, not PF
        dec     word ptr [Rmain_+R_BC]
        jz short ldi_ldd_end
        or      ah, PF
ldi_ldd_end:
        ret

Lddr_:
        call    Ldd_
        jmp short Ldir_Lddr_
Ldir_:
        call    Ldi_
Ldir_Lddr_:
        jz short Ldir_end
        add     word ptr [TCounter], 5
        dec     si
        dec     si
Ldir_end:
        ret

Inc_rp: ; opd1 = ST_+opdBC / opdDE / opdHL / opdSP
        inc     dx
        ret

Dec_rp: ; opd1 = ST_+opdBC / opdDE / opdHL / opdSP
        dec     dx
        ret

Inc_r:  ; opd1 = ST_+opdB / opdC / opdD / opdE / opdHx / opdLx / opdM / opdA
        sahf ; to restore CF
        inc     dl
        lahf
        cmp     dl, 80h
inc_dec_r_chk_overflow:
        mov     al, dl ; YF, XF are from result
        jnz short inc_r_notPF
        or      ah, PF
        ret
inc_r_notPF:
        and     ah, not PF
        ret

Dec_r:  ; opd1 = ST_+opdB / opdC / opdD / opdE / opdHx / opdLx / opdM / opdA
        sahf
        dec     dl
        lahf
        or      ah, NF
        cmp     dl, 7Fh
        jmp short inc_dec_r_chk_overflow

Rlca_:  ; opd1 = ST_+opdA
        rol     dl, 1
rla_rra:
        mov     al, dl  ; YF, XF are from result
        lahf            ; AH - new flags (only CF interested)
        ret

Rrca_:  ; opd1 = ST_+opdA
        ror     dl, 1
        jmp short rla_rra

Rla_:   ; opd1 = ST_+opdA
        sahf
        rcl     dl, 1
        jmp short rla_rra

Rra_:   ; opd1 = ST_+opdA
        sahf
        rcr     dl, 1
        jmp short rla_rra

Rlc_2:  mov     dl, ch
Rlc_:   ; opd1 = ST_+opdB / opdC / ... / opdM / opdA
        rol     dl, 1
rl_rr:
        lahf
        mov     al, ah  ; AL - new flags (CF only interested)
        and     dl, dl
        lahf            ; AH = SF, ZF, PF
        sar     ax, 1
        rcl     ah, 1
        mov     al, dl
        ret

Rrc_2:  mov     dl, ch
Rrc_:   ; opd1 = ST_+opdB / opdC / ... / opdM / opdA
        ror     dl, 1
        jmp short rl_rr

Rl_2:   mov     dl, ch
Rl_:    ; opd1 = ST_+opdB / opdC / ... / opdM / opdA
        sahf
        rcl     dl, 1
        jmp short rl_rr

Rr_2:   mov     dl, ch
Rr_:    ; opd1 = ST_+opdB / opdC / ... / opdM / opdA
        sahf
        rcr     dl, 1
        jmp short rl_rr

Sla_2:  mov     dl, ch
Sla_:   ; opd1 = ST_+opdB / opdC / ... / opdM / opdA
        shl     dl, 1
Sla_flags:
        mov     al, dl
        lahf
        ret

Sra_2:  mov     dl, ch
Sra_:   ; opd1 = ST_+opdB / opdC / ... / opdM / opdA
        sar     dl, 1
        jmp short Sla_flags

Srl_2:  mov     dl, ch
Srl_:   ; opd1 = ST_+opdB / opdC / ... / opdM / opdA
        shr     dl, 1
        jmp short Sla_flags

Sll_2:  mov     dl, ch
Sll_:   ; opd1 = ST_+opdB / opdC / ... / opdM / opdA
        sal     dl, 1
        lahf
        inc     dl
        mov     al, dl
        ret

Rld_:   ; opd1 = ST_+opdM, opd2 = opdA
        ; dl = (HL), cl = A given by operands
        mov     dh, cl
        ror     dh, 4
        rol     dx, 4 ; dl = new value (HL)
        mov     byte ptr [R_A], dh
        and     dh, dh
        lahf
        mov     al, ah
        cmp     dh, 0
        lahf
        and     ax, ((not PF) shl 8) or PF
        or      ah, al
        mov     al, dh
        ret

Rrd_:   ; opd1 = ST_+opdM, opd2 = opdA
        ; dl = (HL), cl = A given by operands
        mov     ch, dl
        push    cx
        and     cx, 0F00Fh
        or      ch, cl
        ror     ch, 4
        mov     dl, ch ; new value for (HL)
        pop     cx
        and     cx, 0FF0h
        or      cl, ch
        mov     byte ptr [R_A], cl
        lahf
        mov     al, ah
        cmp     cl, 0
        lahf
        and     ax, ( (not PF) shl 8 ) or PF
        or      ah, al
        mov     al, cl
        ret

Daa_:   ; opd1 = ST_+opdA
IF SIMPLE_DAA
        mov     al, dl
        test    ah, 2
        jnz short @das
        sahf
        daa
        jmp short @daa_das
        ret
@das:   sahf
        das
@daa_das:
        lahf
        mov     dl, al
        ret
ELSE ; not SIMPLE_DAA
        movzx   bx, ah
        mov     bh, byte ptr [DAATable1+bx]
        mov     bl, dl
        add     bx, bx
        mov     dx, word ptr [DAATable2+bx]
        mov     ax, dx
        ret
ENDIF ; SIMPLE_DAA

Cpl_:   ; opd1 = ST_+opdA
        xor     dl, 0FFh
        lahf
        or      ah, HF or NF
        mov     al, dl
        ret

Scf_:   ; opd1 = opdNo, opd2 = opdA
        or      ah, CF
        mov     al, cl ; XF,YF are from [R_A]
        ret

Ccf_:   ; opd1 = opdNo, opd2 = opdA
        shl     al, 4 ; previous CF -> HF
        xor     ah, CF
        mov     al, cl ; XF, YF from [R_A]
        ret

Ex_af_af_alt:   ; dh = F, dl = A
        xchg    dx, word ptr [R_AFalt]
        ret

Exx_:
        mov     edx, dword ptr [RmainHL]
        xchg    edx, dword ptr [R_HLalt]
        mov     dword ptr [RmainHL], edx
        mov     dx, word ptr [RmainBC]
        xchg    dx, word ptr [R_BCalt]
        mov     word ptr [RmainBC], dx
        ret

Add_rp: ; opd1 = ST_+opdHL, opd2 = opdBC / opdDE / opdHL / opdSP
        add     dl, cl
        adc     dh, ch
        mov     byte ptr [RHidden], dh
        lahf
        mov     al, dh
        ret

Adc_rp: ; opd1 = ST_+opdHL, opd2 = opdBC / opdDE / opdHL / opdSP
        push    dx ;-----------------------\
        sahf
        pushf      ;--------------------\
        adc     dl, cl
        adc     dh, ch
        lahf       ; ah = flags of high-byte addition (HF only interested)
        mov     byte ptr [R_F], ah
        popf       ;--------------------/
        pop     dx ;-----------------------/
        push    ax ;-----------------------\
        mov     al, dh
        adc     dx, cx
        lahf       ; ah = flags of two-byte addition
Adc_Sbc_rp_flags:
        and     ah, not PF
        mov     cl, al ; cl = high byte of opd1
        and     cx, 8080h
        cmp     cl, ch
        jnz short Adc_Sbc_rp_end
        mov     al, dh
        and     al, 80h
        cmp     al, cl
        jz short Adc_Sbc_rp_end
        or      ah, PF
Adc_Sbc_rp_end:
        pop     cx ;-----------------------/
        mov     al, dh ; YF, XF from high-byte addition
        mov     byte ptr [RHidden], al
        ret

Sbc_rp: ; opd1 = ST_+opdHL, opd2 = opdBC / opdDE / opdHL / opdSP
        push    dx ;-----------------------\
        sahf
        pushf      ;--------------------\
        sbb     dl, cl
        sbb     dh, ch
        lahf       ; ah = flags of high-byte addition (HF only interested)
        mov     byte ptr [R_F], ah
        popf       ;--------------------/
        pop     dx ;-----------------------/
        push    ax ;-----------------------\
        mov     al, dh
        sbb     dx, cx
        lahf       ; ah = flags of two-byte addition
        neg     ch
        jmp short Adc_Sbc_rp_flags

Add_r:  ; opd1 = ST_+opdA, opd2 = opdB / opdC /... / opdA
        mov ch, dl
        add dl, cl
add_calc_PF:
        lahf
add_sub_flag_PF:
        and ah, not PF
        and cx, 8080h
        cmp cl, ch
        jnz short add_end
        mov al, dl
        and al, 80h
        cmp al, ch
        jz short add_end
        or  ah, PF
add_end:
        mov al, dl ; YF, XF are from result
        ret

Adc_r:  ; opd1 = ST_+opdA, opd2 = opdB / opdC /... / opdA
        mov ch, dl
        sahf
        adc dl, cl
        jmp short add_calc_PF

Neg_:   ; opd1 = ST_+opdA, opd2 = opdA
        xor dx, dx ; implemented as 0 - A
Sub_r:  ; opd1 = ST_+opdA, opd2 = opdB / opdC / ... / opdA
        mov ch, dl
        sub dl, cl
sub_calc_PF:
        lahf
        or  ah, NF
        neg cl
        jmp short add_sub_flag_PF

Sbc_r:  ; opd1 = ST_+opdA, opd2 = opdB / opdC / ... / opdA
        mov ch, dl
        sahf
        sbb dl, cl
        jmp short sub_calc_PF

Cp_r:   ; opd1 = opdA, opd2 = opdB / opdC / ... / opdA
        push cx ;------\
        call Sub_r
        pop  cx ;------/
        mov  al, cl ; YF, XF are from opd2 rather then from result
        ret

And_r:  ; opd1 = ST_+opdA, opd2 = opdB / opdC / ... / opdA
        and     dl, cl
        lahf
        or      ah, HF
        mov     al, dl ; YF, XF are from result
        ret

Xor_r:  ; opd1 = ST_+opdA, opd2 = opdB / opdC / ... / opdA
        xor dl, cl
        DB  0B9h ; op code mov cx,XXXX - to skip next 2-bytes instruction
        ;--lahf
        ;--mov     al, dl
        ;--ret

Or_r:   ; opd1 = ST_+opdA, opd2 = opdB / opdC / ... / opdA
        or dl, cl
        lahf
        mov     al, dl
        ret

BitN_r2_hidden:
        mov     dl, ch
BitN_r_hidden:
        call    BitN_r
        mov     al, byte ptr [RHidden]
        and     al, cl ; (is this necessary???)
        ret

BitN_r2:mov     dl, ch
BitN_r: ; opd1 = opdB / opdC / ... / opdA, opd2 = opdNbit
        and     dl, cl
        setz    al
        or      cl, ZF or PF
        sar     ax, 1
        sbb     ax, ax
        mov     al, dl
        and     ax, ((ZF or PF) shl 8) or (YF or XF)
        and     ah, cl
        and     al, cl
        or      ah, HF
        ret

ResN_r2:mov     dl, ch
ResN_r: ; opd1 = ST_+opdB / opdC / ... / opdA, opd2 = opdN
        not     cl
        and     dl, cl
        ret

SetN_r2:mov     dl, ch
SetN_r:
        or      dl, cl
        ret

JR_cond: ; opd1 = opdJR / opdWord, opd2 = opdCondxxx
        jcxz    j_end
        add     word ptr [TCounter], 5
JP_cond:
        jcxz    j_end
J_:     ; opd1 = opdJR / opdWord
        mov     si, dx
j_end:  ret

Ret_cond:
        jcxz    ret_end
        add     word ptr [TCounter], 7
Ret_I:
Ret_N:
Ret_:
        call    pop_DX
        mov     si, dx
ret_end:ret

Pop_rp: ; opd1 = ST_+opdBC / opdDE / opdHL / opdAF
pop_DX: ; this entry used in RET also
        push    es ;-----------------------------------\
        mov     bx, word ptr [R_SP]
IF SPECTRUM_128K ;......................................128K
        call    Read_Mem ; dl = Spectrum_Mem[ bx ]
        mov     dh, dl
        inc     bx
        call    Read_Mem ; dl = Spectrum_Mem[ bx ]
        xchg    dl, dh
        inc     bx
ELSE             ;......................................48K
        mov     es, word ptr [Spectrum_Memory_Segment]
        mov     dl, byte ptr es:[bx]
        inc     bx
        mov     dh, byte ptr es:[bx]
        inc     bx
ENDIF            ;......................................
        mov     word ptr [R_SP], bx
        pop     es ;-----------------------------------/
        ret

Ex_@sp_hl:
        ; opd1 = ST_+opdHL, opd2 = opdHL
        ; cx = HL previous value (opd2)
        call    pop_DX ; dx = HL new value (stored as opd1)
        ;--jmp     push_CX

Push_rp: ; opd1 = opdNo, opd2 = opdBC / opdDE / opdHL, opdAF
push_CX: ; this entry used in CALL also
        push    es ;-----------------------------------\ ??????
        mov     bx, word ptr [R_SP]
IF SPECTRUM_128K ;......................................128K
        xchg    dx, cx
        xchg    dh, dl
        dec     bx
        call    Write_Mem ; Spectrum_Mem[ bx ] = dl
        xchg    dl, dh
        dec     bx
        call    Write_Mem ; Spectrum_Mem[ bx ] = dl
        xchg    dx, cx
ELSE             ;......................................48K
        mov     es, word ptr [Spectrum_Memory_Segment]
        dec     bx
        cmp     bh, 40h
        jb short no_push_hi
        mov     byte ptr es:[bx], ch
no_push_hi:
        dec     bx
        cmp     bh, 40h
        jb short no_push_lo
        mov     byte ptr es:[bx], cl
no_push_lo:
ENDIF            ;......................................
        mov     word ptr [R_SP], bx
        pop     es ;-----------------------------------/ ??????
        ret

Cpd_:   ; opd1 = opdA, opd2 = opdM
        dec     word ptr [base_HL]
        jmp short Cpi_Cpd_
Cpi_:   ; opd1 = opdA, opd2 = opdM
        inc     word ptr [base_HL]
Cpi_Cpd_:
        push    ax ;-----\
        call    Sub_r
        pop     cx ;-----/
        sar     cx, 4
        and     ch, 1
        sub     dl, ch ; dl = opd1 - opd2 - old_HF = A - (HL) - old_HF
        mov     al, dl
        shl     al, 4
        and     ax, YF or ((not PF) shl 8)
        and     dl, XF
        or      al, dl ; al = YF, XF from A - (HL) - old_HF
        or      ah, NF
        dec     word ptr [Rmain_+R_BC]
        jz short cpi_cpd_end
        or      ah, PF
cpi_cpd_end:
        ret

Cpdr_:  ; opd1 = opdA, opd2 = opdM
        call    Cpd_
        jmp short cpir_cpdr_
Cpir_:  ; opd1 = opdA, opd2 = opdM
        call    Cpi_
cpir_cpdr_:
        setnz   cl
        test    ah, ZF
        jnz short cpir_cpdr_end
        test    cl, 1
        ;jz      cpir_cpdr_end
        DB      74h, cpir_cpdr_end - cpir_cpdr_cont
cpir_cpdr_cont:
        add     word ptr cs:[TCounter], 5
        dec     si
halt1:  dec     si
cpir_cpdr_end:
call_end:
        ret

Halt_:  ; opd1 = opdNo, opd2 = opdNo
        ;cmp     byte ptr [IntEnabled], dl ; =0 ?
        cmp     byte ptr [IFF1IFF2], dl
        jz short halt1
        cmp     byte ptr [IntSignal], dl  ; =0 ?
        jz short halt1
IF CAREFUL_INT
        push    dx
        mov     dx, word ptr [TCntInt]
        cmp     dx, word ptr [TCounter]
        pop     dx
        jl short halt1
ENDIF
        cmp     byte ptr [After_EI], dl   ; =0 ?
        jnz short halt1
IF LOGS and LOG_Reason_Int
        call    LogText
        DB      '*** interrupt follow HALT ***',13,10,0
ENDIF

Interrupt_:
        mov     byte ptr [IntSignal], 0
        add     word ptr [TCounter], 2 ; int 7 requires 11 tacts
        call    Di_
        mov     dx, 38h
        cmp     byte ptr [IMmode], 2
        jz short Int_IM2
        ;jmp     Call_ ; crazy TASM5 faults compile this instruction
        DB      0E9h
        DW      Call_ - Int_Im2

Int_Im2:
        ;
        ; interrupt in IM2 mode
        mov     bl, 0FFh
        mov     bh, byte ptr [R_I]
        add     word ptr [TCounter], 6
IF SPECTRUM_128K ;........................................128K
        inc     bx
        call    Read_Mem
        mov     dh, dl
        dec     bx
        call    Read_Mem
ELSE             ;........................................48K
        mov     es, word ptr [Spectrum_Memory_Segment]
        mov     dl, byte ptr es:[bx]
        inc     bx
        mov     dh, byte ptr es:[bx]
ENDIF            ;........................................
        DB 0B9h; mov cx,... instead of jmp     Call_ ;%%%%%%%%%

Call_cond: ; opd1 = opdWord / opdbbb, opd2 = opdCondxxx
        jcxz    call_end
Call_:
        mov     cx, si
        call    push_CX
        jmp     J_

        ; opd1 = opdNo => dx = 0
Im_2:   inc     dx
Im_1:   inc     dx
Im_0:   mov     byte ptr [IMmode], dl
        ret

Ei_:
        mov     byte ptr [After_EI], 2
                ; when an instruction is processed, [After_EI] is shifted
                ; 1 bit right, so it is still not =0 one instruction after EI
        mov     al, 4
        ;jmp     set_ei_di
        db      3Dh ; op code of "cmp cx,..." to skip 2 following bytes

Di_:
        mov     al, 0
set_ei_di:
        ;mov     byte ptr [IntEnabled], al ; IntEnabled=4 or 0
        mov     byte ptr [IFF1IFF2], al   ; IFF1IFF2=4 or 0
        ret

;----------------------------------------------
; output value CL to port DX
;----------------------------------------------
Out_:
        test    dl, 1
IF SPECTRUM_128K
        jnz short out_FD
ELSE
        jnz short out_end
ENDIF
        ;------ sound output
        mov     dx, 61h
        push    ax ;-------------------\
        mov     al, cl
        xchg    byte ptr [BorderColor], al
        xor     ax, cx
        test    al, 10h
Sound_Jmp:
        jz short no_sound
        in      al, dx
        xor     al, 2
        and     al, not 1
        out     dx, al
no_sound:
        pop     ax ;-------------------/
out_end:        ret

;MIC:    DB      0

IF SPECTRUM_128K
out_FD: ;**************************************************
        ; OUT 7FFD ?
        ;**************************************************
        test    dx, 8002h
        jnz short out_end ; possibly out FFFD or BFFD (AY output)
        ; OUT xxxx xxxx xxxx xxxx 01 = OUT 7FFD
        test    byte ptr [Port_7FFD], 20h
        jnz short out_end ; port 7FFD locked, ignore output to it
out_7FFD:
        ;------ write to port 7FFD
        mov     byte ptr cs:[Port_7FFD], cl ; prefix "cs" needed because
                ; out_7FFD can be called from other place (Load_Sna) when ds!=cs
        ;------ provide access to memory
        mov     ch, cl
        and     cx, 1007h
        add     cl, cl
        or      cl, ch
        shl     cx, 4
        mov     dx, cs
        add     dx, cx
        mov     gs, dx ; gs = base offset to access correspondent part of
                       ; Bank_Access_Table (when gs = cs, initial configuration
                       ; is accessed: RAM Page 2 = Bank RAM 0, ROM Page = ROM Bank 0)
        ret
ENDIF

In_@byte:
        mov     ch, byte ptr [R_A]
;----------------------------------------------------------
; input port CX, result -> DX
;----------------------------------------------------------
in_CX:
        mov     dl, 0FFh
        test    cl, 1
        jnz short in_1F
        ;------- keyboard input
        mov     bx, offset[Keybd_State]
        mov     dh, 8; BFh
loop_key_scanlines:                ;<------------;
        ror     ch, 1                            ;
        jc short nx_key_scanline                 ;
        and     dl, byte ptr cs:[bx]             ;
nx_key_scanline:                                 ;
        inc     bx                               ;
        dec     dh                               ;
        jnz short loop_key_scanlines ;-----------;
        ret
in_1F:  inc     cl
        jz short in_FF
        ;dec     cl
        ;test    cl, 2
        ;jz      in_Other
        ;------ kempston input
        mov     dl, byte ptr [Keybd_State+8]
        not     dl
in_Other: ;---- possibly in FD
        ret
in_FF:
        mov     dl, 0FFh ; byte ptr [CurAttrByte]
IF LOGS
;        pusha
;        call    LogText
;        DB      'IN FF: BorderColor=',0
;        movzx   ax, dl
;        call    LogAX
;        call    LogText
;        DB      13,10,0
;        popa
ENDIF
in_FF_end:
        ret

In_@C:
        push    ax ;-----------------
        call    In_CX               ;
        cmp     dl, 0               ;
        lahf                        ;
        pop     cx ;<----------------
        mov     al, dl ; YF, XF are from byte received
        ret

Outd_:
        dec     word ptr [base_HL]
        jmp short Outi_Outd_
Outi_:
        inc     word ptr [base_HL]
Outi_Outd_:
        push    cx
        call    Out_
        pop     cx
        sub     byte ptr [Rmain_+R_B], 1
        lahf           ; SF, ZF from (B-1)
        mov     al, cl
        add     al, al
        sbb     al, al
        and     ax, ((not(NF or HF or CF)) shl 8) or NF
        or      ah, al ; NF = copy of bit 7 of byte transferred
        add     cl, byte ptr [base_HL+R_L] ; CF, HF are set if (L+n)>255
Outid_Inid_HF_CF:
        sbb     al, al
        and     al, CF or HF
        or      ah, al
        mov     al, byte ptr [Rmain_+R_B] ; YF, XF from (B-1)
        ret

Ind_:
        dec     word ptr [base_HL]
        mov     al, byte ptr [Rmain_+R_C]
        dec     al ; C' = C-1 for IND / INDR
        jmp short Ini_Ind_
Ini_:
        inc     word ptr [base_HL]
        mov     al, byte ptr [Rmain_+R_C]
        inc     al ; C' = C+1 for INI / INIR
Ini_Ind_:
        push    ax ;------------------------------\
        call    In_CX
        ; result in dl
        sub     byte ptr [Rmain_+R_B], 1
        lahf    ; SF, ZF are result of (B-1)
        mov     al, dl
        add     al, al
        sbb     al, al
        and     ax, NF or ((not(CF or HF)) shl 8)
        or      ah, al ; NF = bit 7 of byte read
        pop     cx ;------------------------------/
        add     cl, dl ; CF, HF are set if (C'+n)>255
        jmp short Outid_Inid_HF_CF

Otdr_:  call    Outd_
        ;jmp     Inir_Indr_Otir_Otdr_
        DB      0EBh, Inir_Indr_Otir_Otdr_ - Otir_
Otir_:  call    Outi_
        ;jmp     Inir_Indr_Otir_Otdr_
        DB      0EBh, Inir_Indr_Otir_Otdr_ - Indr_
Indr_:  call    Ind_
        ;jmp     Inir_Indr_Otir_Otdr_
        DB      0EBh, Inir_Indr_Otir_Otdr_ - Inir_
Inir_:  call    Ini_
Inir_Indr_Otir_Otdr_:
        cmp     byte ptr [Rmain_+R_B], 0
        jz short Inir_Indr_end
        dec     si
        dec     si
Inir_Indr_end:
        ret

;******************************************************;
;                       TABLES                         ;
;******************************************************;

_SZYHXPNC EQU SF or ZF or YF or HF or XF or PF or NF or CF
_SZYHXPN  EQU SF or ZF or YF or HF or XF or PF or NF
_SZYHXPC  EQU SF or ZF or YF or HF or XF or PF or       CF
_SZYHXP   EQU SF or ZF or YF or HF or XF or PF
_SZYXPC   EQU SF or ZF or YF or       XF or PF or       CF
_SZYXP    EQU SF or ZF or YF or       XF or PF
_HNC      EQU                   HF or             NF or CF
_HN       EQU                   HF or             NF
_NC       EQU                                     NF or CF
_YXP      EQU             YF or       XF or PF
_YXC      EQU             YF or       XF or             CF
_YHXN     EQU             YF or HF or XF or       NF
_YHXC     EQU             YF or HF or XF or             CF

TablesSet_SegmentsSize EQU 112
IF COMPRESS_TABLES

Decompress_Cmd:
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'Decompression started',0
ENDIF
  DB $LOOP+1    ; to start the first set of tables as a sub-routine
                ; to use it later with IX/IY tables
de_Table_Opd1: ;work with bytes (initially it is so)
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '----------------',0
ENDIF
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'TableOpd1',0
ENDIF
; 00
  DB $ROWS+8, 7
  DB $MOVE+2, opdNo, ST_+opdAF
  DB $REPEAT+6, opdJR
  DB $ROWS+4, 15
  DB $LOOP+1
de_st_all_rp:
  DB $MOVE+4, ST_+opdBC, ST_+opdDE, ST_+opdHL, ST_+opdSP
  DB $RET
  DB $MOVE+4, ST_+opd@BC, ST_+opd@DE, ST_+opdAddr2, ST_+opdAddr1
  DB $CALL, de_st_all_rp - 1 - $
  DB $FORWARD+5
  DB $REPEAT+4, ST_+opdHL
  DB $MOVE+4, ST_+opdA, ST_+opdA, ST_+opdHL, ST_+opdA
  DB $CALL, de_st_all_rp - 1 - $
  DB $OFFSET
  DW -8
  DB $ROWS+8, 7
  DB $LOOP+1
de_st_all_r:
  DB $MOVE+8, ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE
  DB ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB $RET
  DB $CALL, de_st_all_r - 1 - $
  DB $CALL, de_st_all_r - 1 - $
  DB $REPEAT+6, ST_+opdA, $REPEAT+2, opdNo
  DB $FORWARD+38h
; 40
  DB $ROWS0
  DB $REPEAT+8, ST_+opdB
  DB $REPEAT+8, ST_+opdC
  DB $REPEAT+8, ST_+opdD
  DB $REPEAT+8, ST_+opdE
  DB $REPEAT+6, ST_+opdHx, $MOVE+2, ST_+opdH, ST_+opdHx
  DB $REPEAT+6, ST_+opdLx, $MOVE+2, ST_+opdL, ST_+opdLx
  DB $REPEAT+6, ST_+opdM, $MOVE+2, opdNo, ST_+opdM
  DB $REPEAT+64, ST_+opdA
  DB $REPEAT+8, opdA
; C0
  DB $ROWS+4, 15
  DB $REPEAT+4, opdNo
  DB $LOOP+1
de_all_rp_af:
  DB $MOVE+4, ST_+opdBC, ST_+opdDE, ST_+opdHL, ST_+opdAF
  DB $RET
  DB $REPEAT+5, opdWord, $MOVE+3, opdByte, ST_+opdHL, opdNo
  DB $REPEAT+4, opdWord, $REPEAT+4, opdNo
  DB $REPEAT+4, ST_+opdA, $REPEAT+4, opdbbb
  DB $REPEAT+6, opdNo, $MOVE+2, opdHL, ST_+opdSP, $REPEAT+4, opdWord
IF LOGS and LOG_DECOMPRESS_TABLES
  DB $MOVE+2, opdNo, ST_+opdA
  DB $LOG ,'opd1: EB',0
  DB $MOVE+2, ST_+opdDE, opdNo
ELSE
  DB $MOVE+4, opdNo, ST_+opdA, ST_+opdDE, opdNo
ENDIF
  DB $REPEAT+5, opdWord, $REPEAT+3, opdByte
  DB $REPEAT+3, ST_+opdA, $MOVE+1, opdA
  DB $REPEAT+4, opdbbb
  DB $FORWARD+48
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'TableOpd2',0
ENDIF
de_TableOpd2: ; step = 15, rows 4
  DB $MOVE+4, opdNo, opdCondDJNZ, opdCondNZ, opdCondNC
  DB $REPEAT+4, opdWord
  DB $MOVE+4, opdA, opdA, opdHL, opdA
  DB $REPEAT+12, opdNo
  DB $REPEAT+4, opdByte
  DB $REPEAT+3, opdNo
  DB $MOVE+13, opdA, opdNo, opdNo, opdCondZ, opdCondC
  DB opdBC, opdDE, opdHL, opdSP, opd@BC, opd@DE, opdAddr2, opdAddr1
  DB $REPEAT+12, opdNo
  DB $REPEAT+4, opdByte
  DB $REPEAT+3, opdNo
  DB $ROWS0
  DB $MOVE+1, opdA
  ; 40
  DB $LOOP+6
de_all_r:
  DB $LOOP+1
de_B_C_D_E:
  DB $MOVE+4, opdB, opdC, opdD, opdE
  DB $RET
  DB $MOVE+4, opdHx, opdLx, opdM, opdA
  DB $RET
  DB $CALL, de_B_C_D_E - 1 - $
  DB $MOVE+4, opdH, opdL, opdNo, opdA
  DB $LOOP+9
  DB $CALL, de_all_r - 1 - $
  DB $RET
  ; C0
  DB $ROWS+8, 7
  DB $LOOP+1
de_all_conds:
  DB $MOVE+8, opdCondNZ, opdCondZ, opdCondNC, opdCondC
  DB opdCondPO, opdCondPE, opdCondP, opdCondM
  DB $RET
  DB $REPEAT+5, opdNo, $MOVE+3, opdHL, opdNo, opdHL
  DB $CALL, de_all_conds - 1 - $
  DB $MOVE+8, opdNo, opdByte, opdA, opdByte
  DB opdHL, opdHL, opdNo, opdNo
  DB $CALL, de_all_conds - 1 - $
  DB $MOVE+8, opdBC, opdNo, opdDE, opdNo
  DB opdHL, opdNo, opdAF, opdNo
  DB $REPEAT+8, opdByte, $REPEAT+8, opdNo
  DB $FORWARD+56
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'JumpTable',0
ENDIF
de_JumpTable: ; words, step 7, rows 8
; 00
  DB $WORDS
  DB $MOVE+4
  DW Nop_, Ex_af_af_alt, JR_cond, J_
  DB $REPEAT+4
  DW JR_cond
  DB $LOOP+4, $MOVE+2
  DW Ld_, Add_rp
  DB $RET
  DB $REPEAT+8
  DW Ld_
  DB $LOOP+4, $MOVE+2
  DW Inc_rp, Dec_rp
  DB $RET
  DB $REPEAT+8
  DW Inc_r
  DB $REPEAT+8
  DW Dec_r
  DB $REPEAT+8
  DW Ld_
  DB $MOVE+8
  DW Rlca_, Rrca_, Rla_, Rra_, Daa_, Cpl_, Scf_, Ccf_
  DB $FORWARD+56
; 40
  DB $ROWS0
  DB $LOOP+6, $MOVE+1
  DW Nop_
  DB $REPEAT+8
  DW Ld_
  DB $RET
  DB $MOVE+1
  DW Halt_
  DB $REPEAT+8
  DW Ld_
  DB $MOVE+1
  DW Nop_
; 80
  DB $REPEAT+8
  DW Add_r
  DB $REPEAT+8
  DW Adc_r
  DB $REPEAT+8
  DW Sub_r
  DB $REPEAT+8
  DW Sbc_r
  DB $REPEAT+8
  DW And_r
  DB $REPEAT+8
  DW Xor_r
  DB $REPEAT+8
  DW Or_r
  DB $REPEAT+8
  DW Cp_r
; C0
  DB $ROWS+4, 15
  DB $REPEAT+4
  DW Ret_cond
  DB $REPEAT+4
  DW Pop_rp
  DB $REPEAT+4
  DW JP_cond
  DB $MOVE+4
  DW J_, Out_, Ex_@sp_hl, Di_
  DB $REPEAT+4
  DW Call_cond
  DB $REPEAT+4
  DW Push_rp
  DB $MOVE+4
  DW Add_r, Sub_r, And_r, Or_r
  DB $REPEAT+4
  DW Call_
  DB $REPEAT+4
  DW Ret_cond
  DB $MOVE+4
  DW Ret_, Exx_, J_, Ld_
  DB $REPEAT+4
  DW JP_cond
  DB $MOVE+4
  DW _CB_, In_@byte, Ex_de_hl, Ei_
  DB $REPEAT+4
  DW Call_cond
  DB $MOVE+8
  DW Call_, _IX_, _ED_, _IY_
  DW Adc_r, Sbc_r, Xor_r, Cp_r
  DB $REPEAT+4
  DW Call_
  DB $FORWARD+48
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'TableUpdFlgs',0
ENDIF
de_TableUpdFlgs:
; 00
  DB $ROWS+8, 7
  DB $REPEAT+9, 0,0
  DB $LOOP+4, $MOVE+2, NF, _YHXC, 0,0, $RET
  DB $REPEAT+15, 0,0
  DB $REPEAT+8, NF, _SZYHXP
  DB $REPEAT+8, 0, _SZYHXPN
  DB $REPEAT+8, 0,0
  DB $REPEAT+4, _HN,_YXC
  DB $MOVE+4, 0,_SZYHXPC, 0,_YHXN, _HN,_YXC, NF,_YHXC
  DB $FORWARD+56
; 40
  DB $ROWS0
  DB $REPEAT+64, 0,0
  DB $REPEAT+16, NF,_SZYHXPC
  DB $REPEAT+16, 0,_SZYHXPNC
  DB $REPEAT+8, _NC,_SZYHXP
  DB $REPEAT+16, _HNC,_SZYXP
  DB $REPEAT+8, 0,_SZYHXPNC
; C0
  DB $LOOP+2
  DB $REPEAT+6, 0,0
  DB $MOVE+2, NF,_SZYHXPC, 0,0
  DB $RET
  DB $LOOP+2
  DB $REPEAT+6, 0,0
  DB $MOVE+2, 0,_SZYHXPNC, 0,0
  DB $RET
  DB $REPEAT+6, 0,0
  DB $MOVE+1, _NC,_SZYHXP
  DB $REPEAT+7, 0,0
  DB $LOOP+2
  DB $MOVE+1, _HNC,_SZYXP
  DB $REPEAT+7, 0,0
  DB $RET
  DB $MOVE+2, 0,_SZYHXPNC, 0,0
  DB $RET ; - finish the 1st set of tables - to repeat it later for IX/IY

IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'TactTable',0
ENDIF
de_TactTable:
  DB $BYTES
  DB $MOVE+2, 4,0Ah
  DB $LOOP+1
t764474:
  DB $MOVE+6, 7,6,4,4,7,4
  DB $RET
  DB $MOVE+2, 4,0Bh
  DB $CALL, t764474 - 1 - $
  DB $MOVE+2, 8,0Ah
  DB $CALL, t764474 - 1 - $
  DB $MOVE+2, 0Ch,0Bh
  DB $CALL, t764474 - 1 - $
  DB $MOVE+3, 7,0Ah,10h
  DB $LOOP+1
t64474:
  DB $MOVE+5, 6,4,4,7,4
  DB $RET
  DB $MOVE+3, 7,0Bh,10h
  DB $CALL, t64474 - 1 - $
  DB $MOVE+11, 7,0Ah,0Dh,6,0Bh,0Bh,0Ah,4,7,0Bh,0Dh
  DB $CALL, t64474 - 1 - $
  DB $LOOP+6
t44444474:
  DB $REPEAT+6, 4, $MOVE+2, 7,4
  DB $RET
  DB $REPEAT+6, 7, $MOVE+2, 4,7
  DB $LOOP+9
  DB $CALL, t44444474 - 1 - $
  DB $RET
  DB $ROWS+8, 7
  DB $REPEAT+8, 5
  DB $REPEAT+3, 0Ah
  DB $MOVE+5, 4,0Ah,4,0Ah,6
  DB $REPEAT+9, 0Ah
  DB $MOVE+4, 0,0Bh,0Bh,13h
  DB $REPEAT+3, 4
  DB $REPEAT+8, 0Ah
  DB $MOVE+8, 0Bh,11h,0Bh,04h,0Bh,0,0Bh,4
  DB $REPEAT+8, 7
  DB $REPEAT+8, 0Bh
  DB $FORWARD+56

IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'CB_TableOpd1',0
ENDIF
  DB $LOOP+1 ; following de_CB_TableOpd1 is a sub-routine to use in XYCB_TableOpd1
de_CB_TableOpd1:
; 00
  DB $ROWS0
  DB $LOOP+8
de_st_all_r_1:
  DB $CALLFAR
  DW de_st_all_r
  DB $RET
; 40
  DB $LOOP+8
  DB $CALLFAR
  DW de_all_r
  DB $RET
; 80
  DB $LOOP+16, $CALL, de_st_all_r_1 - 1 - $
  DB $RET
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'CB_TableOpd2',0
ENDIF
  DB $RET ; end of de_CB_TableOpd1 subroutine
de_CB_TableOpd2:
  ; 00
  DB $REPEAT+64, opdNo
  ; 40
  DB $LOOP+3, $REPEAT+64, opdNbit, $RET
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'CB_JumpTable',0
ENDIF
de_CB_JumpTable:
  DB $WORDS
  DB $ROWS+8, 7
  DB $LOOP+8
  DB $MOVE+8
  DW Rlc_, Rrc_, Rl_, Rr_, Sla_, Sra_, Sll_, Srl_
  DB $RET, $FORWARD+38h
  DB $REPEAT+48
  DW BitN_r
  DB $REPEAT+8
  DW BitN_r_hidden
  DB $REPEAT+8
  DW BitN_r
  DB $FORWARD+38h
  DB $ROWS0
  DB $REPEAT+64
  DW ResN_r
  DB $REPEAT+64
  DW SetN_r
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'CB_TableUpdFlgs',0
ENDIF
  DB $LOOP+1 ; de_CB_TableUpdFlgs is the same as de_XYCB_TableUpdFlags
de_CB_TableUpdFlgs:
  DB $REPEAT+64, _HN,_SZYXPC
  DB $REPEAT+64, NF,_SZYHXP
  DB $BYTES
  DB $LOOP+4
  DB $REPEAT+64, 0
  DB $RET
  DB $RET ; end of subroutine
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'CB_TactTable',0
ENDIF
de_CB_TactTable:
  DB $LOOP+8
t888888F8:
  DB $REPEAT+6, 8, $MOVE+2, 0Fh, 8
  DB $RET
  DB $LOOP+8
  DB $REPEAT+6, 8, $MOVE+2, 0Ch, 8
  DB $RET
  DB $LOOP+16, $CALL, t888888F8 - 1 - $, $RET
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'XY_Tables',0
ENDIF
de_XY_Tables:   ; the same as main Tables except some certain entries
                ; and tacts table
  DB $CALLFAR
  DW de_Table_Opd1
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'XY_Tables - end of main set (except tacts table)',0
ENDIF
  ; now some corrections:
  DB $BYTES
  DB $OFFSET
;  DW -600h+76h
;IF LOGS and LOG_DECOMPRESS_TABLES
;        DB $LOG, 'opd1: 76',0
;ENDIF
;  DB $MOVE+1, opdNo
;  DB $OFFSET
;  DW 0CBh-77h
  DW -600h+0CBh
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'opd1: CB',0
ENDIF
  DB $MOVE+1, opdByte
  DB $FORWARD+0DDh-0CCh
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'opd1: DD',0
ENDIF
  DB $MOVE+1, opdNo
;IF LOGS and LOG_DECOMPRESS_TABLES
;  DB $FORWARD+0EBh-0DEh
;        DB $LOG, 'opd1: EB',0
;  DB $MOVE+1, ST_+opdDE
;  DB $FORWARD+1
;ELSE
  DB $FORWARD+0EDh-0DEh
;ENDIF
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'opd1: ED',0
ENDIF
  DB $MOVE+1, opdNo
  DB $FORWARD+0FDh-0EEh
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'opd1: FD',0
ENDIF
  DB $MOVE+1, opdNo
  DB $FORWARD+100h-0FEh + 36h
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'opd2: 36',0
ENDIF
  DB $MOVE+1, opdNo
  DB $OFFSET
  DW 0CBh - 37h
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'opd2: CB',0
ENDIF
  DB $MOVE+1, opd@M
  DB $OFFSET
  DW 100h + 36h*2 - 0CCh
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'jmp: 36',0
ENDIF
  DB $MOVE+2
  DW Ld_@XY_byte
  DB $OFFSET
  DW (0CBh - 37h)*2
  DB $MOVE+2
  DW XY_CB_
  DB $FORWARD+(0DDh-0CCh)*2
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'jmp: DD',0
ENDIF
  DB $MOVE+2
  DW XY_IX_
  DB $FORWARD+(0EDh-0DEh)*2
  DB $MOVE+2
  DW Nop_
  DB $FORWARD+(0FDh-0EEh)*2
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'jmp: FD',0
ENDIF
  DB $MOVE+2
  DW XY_IY_
  DB $OFFSET
  DW (100h-0FEh)*2 + 200h
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'XY_TactTable',0
ENDIF

de_XY_TactTable:
  DB $MOVE+3, 4,0Ah,7
  DB $LOOP+1
@t64474:
  DB $CALLFAR
  DW t64474
  DB $RET
  DB $MOVE+3, 4,0Bh,7
  DB $CALL, @t64474 - 1 - $
  DB $MOVE+3, 8,0Ah,10h
  DB $CALL, @t64474 - 1 - $
  DB $MOVE+3, 8,0Bh,10h
  DB $CALL, @t64474 - 1 - $
  DB $MOVE+3, 7,0Ah,10h
  DB $CALL, @t64474 - 1 - $
  DB $MOVE+3, 7,0Bh,10h
  DB $CALL, @t64474 - 1 - $
  DB $MOVE+11, 7,0Ah,0Dh, 6,13h,13h,0Ah,4, 7,0Bh,0Dh
  DB $CALL, @t64474 - 1 - $

  DB $LOOP+6
t444444F4:
  DB $REPEAT+6, 4
  DB $MOVE+2, 0Fh,4
  DB $RET
  DB $REPEAT+6,0Fh
  DB $MOVE+2, 4,0Fh
  DB $LOOP+9, $CALL, t444444F4 - 1 - $, $RET
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '@@@@@@@@@@',0
ENDIF
  DB $ROWS+8, 7
  DB $REPEAT+8, 5
  DB $REPEAT+3, 0Ah
  DB $MOVE+5, 4,0Ah,4,0Ah,6
  DB $REPEAT+9, 0Ah
  DB $MOVE+4, 4,0Bh,0Bh,13h
  DB $REPEAT+3, 4
  DB $REPEAT+9, 0Ah
  DB $MOVE+1, 11h
  DB $LOOP+3, $MOVE+2, 0Bh,4, $RET
  DB $REPEAT+8, 7
  DB $REPEAT+8, 0Bh
  DB $FORWARD+56
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'XYCB_TableOpd1',0
ENDIF
de_XYCB_TableOpd1: ; similar to CB_TableOpd1 but...
  DB $CALLFAR
  DW de_CB_TableOpd1
  ; now corrections:
  DB $OFFSET
  DW -256+6
  DB $ROWS+8, 7
  DB $LOOP+4
  DB $REPEAT+8, opdNo
  DB $FORWARD+63
  DB $RET
  DB $OFFSET
  DW -6
  DB $ROWS0
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'XYCB_TableOpd2',0
ENDIF
  DB $REPEAT+64, opdM_XYCB
  DB $LOOP+3, $REPEAT+64, opdMbit, $RET
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'XYCB_JumpTable',0
ENDIF
de_XYCB_JumpTable:
  DB $WORDS
  DB $ROWS+8, 7
  DB $LOOP+8
  DB $MOVE+8
  DW Rlc_2, Rrc_2, Rl_2, Rr_2, Sla_2, Sra_2, Sll_2, Srl_2
  DB $RET
  DB $FORWARD+56
  DB $REPEAT+48
  DW BitN_r2
  DB $REPEAT+8
  DW BitN_r2_hidden
  DB $REPEAT+8
  DW BitN_r2
  DB $FORWARD+56, $ROWS0
  DB $REPEAT+64
  DW ResN_r2
  DB $REPEAT+64
  DW SetN_r2
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'XYCB_TableUpdFlgs',0
ENDIF
de_XYCB_TableUpdFlgs: ; the same as CB_TableUpdFlgs
  DB $CALLFAR
  DW de_CB_TableUpdFlgs
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'XYCB_TactTable',0
ENDIF
de_XYCB_TactTable:
  DB $BYTES
  DB $REPEAT+64, 0Fh
  DB $REPEAT+64, 0Ch
  DB $REPEAT+64, 0Fh
  DB $REPEAT+64, 0Fh
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'ED_TableOpd1',0
ENDIF
de_ED_TableOpd1:
  DB $REPEAT+64, opdNo
  DB $ROWS+8, 7
  DB $MOVE+8, ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE
  DB ST_+opdH, ST_+opdL, opdNo, ST_+opdA
  DB $REPEAT+8, opdBC
  DB $REPEAT+8, ST_+opdHL
  DB $FORWARD+1
  DB $REPEAT+8, ST_+opdA
  DB $REPEAT+16, opdNo
  DB $MOVE+8, ST_+opdI, opdR, ST_+opdA, ST_+opdA
  DB ST_+opdM, ST_+opdM, opdNo, opdNo
  DB $OFFSET
  DW -5
  DB $ROWS+4, 15
  DB $REPEAT+4, ST_+opdAddr2
  DB $FORWARD+7
  DB $CALLFAR
  DW de_st_all_rp
  DB $FORWARD+52
; 80
  DB $ROWS0
  DB $REPEAT+32, opdNo
  DB $LOOP+4
  DB $MOVE+4, ST_+opd@DE, opdA, ST_+opdM, opdBC
  DB $REPEAT+4, opdNo
  DB $RET
; C0
  DB $REPEAT+64, opdNo
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'ED_TableOpd2',0
ENDIF
de_ED_TableOpd2:
  DB $REPEAT+64, opdNo
  DB $ROWS+8, 7
  DB $REPEAT+8, opdBC
  DB $MOVE+8, opdB, opdC, opdD, opdE, opdH, opdL, opdNo, opdA
  DB $FORWARD+2
  DB $REPEAT+8, opdA
  DB $REPEAT+16, opdNo
  DB $OFFSET
  DW -5
  DB $ROWS+4, 15
  DB $LOOP+1
de_all_rp:
  DB $MOVE+4, opdBC, opdDE, opdHL, opdSP
  DB $RET
  DB $CALL, de_all_rp - 1 - $
  DB $FORWARD+3
  DB $LOOP+1
de_ARAno:
  DB $MOVE+4, opdA, opdR, opdA, opdNo
  DB $RET
  DB $FORWARD+2
  DB $CALL, de_all_rp - 1 - $
  DB $REPEAT+4, opdAddr2
  DB $FORWARD+3
  DB $CALL, de_ARAno - 1 - $
  DB $FORWARD+48
; 80
  DB $ROWS0
  DB $REPEAT+32, opdNo
  DB $LOOP+4
  DB $MOVE+4, opdM, opdM, opdBC, opdM
  DB $REPEAT+4, opdNo
  DB $RET
  DB $REPEAT+64, opdNo
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'ED_JumpTable',0
ENDIF
de_ED_JumpTable:
  DB $WORDS
  DB $REPEAT+64
  DW Nop_
  DB $ROWS+8, 7
  DB $REPEAT+8
  DW In_@C
  DB $REPEAT+8
  DW Out_
  DB $LOOP+4, $MOVE+2
  DW Sbc_rp, Adc_rp
  DB $RET
  DB $REPEAT+8
  DW Ld_
  DB $REPEAT+8
  DW Neg_
  DB $REPEAT+8
  DW Ret_N ; Ret_I = Ret_N
  DB $MOVE+15
  DW Im_0, Im_0, Im_1, Im_2, Ld_, Im_0, Im_1, Im_2
  DW Ld_, Ld_R_A, Ld_A_I, Ld_A_R, Rrd_, Rld_, Nop_
  DB $ROWS0
  DB $REPEAT+1+32
  DW Nop_
  DB $MOVE+4
  DW Ldi_, Cpi_, Ini_, Outi_
  DB $LOOP+1
de_4xNop:
  DB $REPEAT+4
  DW Nop_
  DB $RET
  DB $MOVE+4
  DW Ldd_, Cpd_, Ind_, Outd_
  DB $CALL, de_4xNop - 1 - $
  DB $MOVE+4
  DW Ldir_, Cpir_, Inir_, Otir_
  DB $CALL, de_4xNop - 1 - $
  DB $MOVE+4
  DW Lddr_, Cpdr_, Indr_, Otdr_
  DB $LOOP+2
  DB $REPEAT+34
  DW Nop_
  DB $RET
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'ED_TableUpdFlgs',0
ENDIF
de_ED_TableUpdFlgs:
  DB $REPEAT+64, 0,0
  DB $LOOP+1
de_ED_UpdFlags_40:
  DB $MOVE+5, _HN,_SZYXP, 0,0, NF,_SZYXPC, 0,0, 0,_SZYHXPNC
  DB $RET
  DB $LOOP+1
de_ED_UpdFlags_3x00:
  DB $REPEAT+3, 0,0
  DB $RET
  DB $LOOP+1
de_ED_UpdFlags_48:
  DB $MOVE+5, _HN,_SZYXP, 0,0, NF,_SZYHXPC, 0,0, 0,_SZYHXPNC
  DB $RET
  DB $CALL, de_ED_UpdFlags_3x00 - 1 - $

  DB $LOOP+2
  DB $CALL, de_ED_UpdFlags_40 - 1 - $
  DB $CALL, de_ED_UpdFlags_2x00_HN_SZYXP - 1 - $
  DB $CALL, de_ED_UpdFlags_48 - 1 - $
de_ED_UpdFlags_2x00_HN_SZYXP:
  DB $REPEAT+2, 0,0
  DB $MOVE+1, _HN,_SZYXP
  DB $RET

  DB $CALL, de_ED_UpdFlags_40 - 1 - $
  DB $CALL, de_ED_UpdFlags_3x00 - 1 - $
  DB $CALL, de_ED_UpdFlags_48 - 1 - $

  DB $REPEAT+32+3, 0,0
  DB $LOOP+4
  DB $MOVE+4, _HN,_YXP, 0,_SZYHXPN, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB $REPEAT+4, 0,0
  DB $RET
  DB $REPEAT+64, 0,0
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, 'ED_TactTable',0
ENDIF
de_ED_TactTable:
  DB $BYTES
  DB $REPEAT+64, 4
  DB $LOOP+4
  DB $LOOP+1
tCCF148E8:
  DB $MOVE+7, 0ch, 0ch, 0fh, 14h, 8, 0eh, 8
  DB $RET
  DB $MOVE+1, 9
  DB $RET
  DB $LOOP+2
  DB $CALL, tCCF148E8 - 1 - $
  DB $MOVE+1, 18h
  DB $RET
  DB $LOOP+2
  DB $CALL, tCCF148E8 - 1 - $
  DB $MOVE+1, 4
  DB $RET
  DB $REPEAT+32, 4
  DB $LOOP+4
  DB $REPEAT+4, 10h, $REPEAT+4, 4
  DB $RET
  DB $REPEAT+64, 4
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- End of emulation tables, start video tables',0
ENDIF
; VIDEO OUTPUT TABLES
de_PixelLines:
  DB $WORDS
  DB $REPEAT+64, 0, 0
  DB $INC
  DW 4000h
  DB $LOOP+3
  DB $LOOP+8
  DB $LOOP+8
  DB $STORE
  DB $INC
  DW 100h
  DB $RET
  DB $INC
  DW -800h+20h
  DB $RET
  DB $INC
  DW 700h
  DB $RET
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- ',0
ENDIF
de_NoAttr:
  DB $REPEAT+56, 0, 0
de_AttrLines:
  DB $REPEAT+64
  DW NoAttr
  DB $LOOP+24
  DB $LOOP+8, $STORE, $RET
  DB $INC
  DW 20h
  DB $RET ; bx = 5B00h
  DB $REPEAT+56
  DW NoAttr
de_VideoLines:
  DB $REPEAT+64
  DW 3E0h
  DB $INC
  DW 520h-5B00h
  DB $LOOP+28
  DB $LOOP+7
  DB $STORE, $INC
  DW 140h
  DB $RET
  DB $RET
  DB $REPEAT+52
  DW 0F920h
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- End of video tables, start of keyboard',0
ENDIF
de_Keybd_Table:
IF EXTRA_KEYBOARD
  DB $MOVE+32
  DW Key_No, Key_Esc
  DW 0FE03h, 0FD03h, 0FB03h, 0F703h, 0EF03h
  DW 0EF04h, 0F704h, 0FB04h, 0FD04h, 0FE04h
  DW 0F786h, 0FB86h, 0FE44h, 0EF08h
  DW 0FE02h, 0FD02h, 0FB02h, 0F702h, 0EF02h
  DW 0EF05h, 0F705h, 0FB05h, 0FD05h, 0FE05h
  DW Key_No, Key_No, 0FE06h, 0FE00h
  DW 0FE01h, 0FD01h
  DB $MOVE+32
  DW 0FB01h, 0F701h, 0EF01h
  DW 0EF06h, 0F706h, 0FB06h, 0FD06h
  DW 0FD85h, 0FE85h, 0F784h, 0FD27h, Key_No
  DW 0FD00h, 0FB00h, 0F700h, 0EF00h
  DW 0EF07h, 0F707h, 0FB07h
  DW 0F787h, 0FB87h, 0EF80h
  DW 0FD07h, Key_No, Key_No, 0FE07h, Key_no
  DW _Help,  _Save,  _Load,  _Reload,_Sound
  DB $REPEAT+4
  DW Key_No
  DB $MOVE+16
  DW _Reset
  DW Key_No, Key_No, 0F508h, 0F708h, 0F608h
  DW Key_No, 0FD08h, 0EF08h, 0FE08h
  DW Key_No, 0F908h, 0FB08h, 0FA08h, 0EF08h
  DW 0EF08h
  DB $REPEAT+12
  DW Key_No
  DB $MOVE+1
  DW KeyExt
  DB $REPEAT+31
  DW Key_No
ELSE ; not EXTRA_KEYBOARD
  DB $MOVE+12
        DW      Key_No, Key_Esc
        DW      0FE03h, 0FD03h, 0FB03h, 0F703h, 0EF03h
        DW      0EF04h, 0F704h, 0FB04h, 0FD04h, 0FE04h
  DB $REPEAT+3
        DW      Key_No
  DB $MOVE+24
        DW      0EF08h
        DW      0FE02h, 0FD02h, 0FB02h, 0F702h, 0EF02h
        DW      0EF05h, 0F705h, 0FB05h, 0FD05h, 0FE05h
        DW      Key_No, Key_No, 0FE06h, 0FE00h
        DW      0FE01h, 0FD01h, 0FB01h, 0F701h, 0EF01h
        DW      0EF06h, 0F706h, 0FB06h, 0FD06h
  DB $REPEAT+3
        DW      Key_No
  DB $MOVE+9
        DW 0FD27h, Key_No
        DW      0FD00h, 0FB00h, 0F700h, 0EF00h
        DW      0EF07h, 0F707h, 0FB07h
  DB $REPEAT+3
        DW      Key_No
  DB $MOVE+10
        DW      0FD07h, Key_No, Key_No, 0FE07h, Key_no
        DW      _Help,  _Save,  _Load,  _Reload,_Sound
  DB $REPEAT+4
        DW      Key_No
  DB $MOVE+16
        DW _Reset
        DW      Key_No, Key_No, 0F508h, 0F708h, 0F608h
        DW      Key_No, 0FD08h, 0EF08h, 0FE08h
        DW      Key_No, 0F908h, 0FB08h, 0FA08h, 0EF08h
        DW      0EF08h
  DB $REPEAT+12
        DW Key_No
  DB $MOVE+1
        DW KeyExt
  DB $REPEAT+31
        DW Key_No
ENDIF ; EXTRA_KEYBOARD
de_Keybd_State:
  DB $BYTES
  DB $REPEAT+9, 0FFh
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- End of keyboard tables, start of Palette',0
ENDIF
de_Palette:
  DB $REPEAT+5, 0
  DB $MOVE+11
        DB      Dark
        DB      Dark,0, 0
        DB      Dark,0, Dark
        DB      0, Dark,0
        DB      0
  DB $REPEAT+4
        DB Dark
  DB $MOVE+1, 0
  DB $REPEAT+3
        DB      Dark
  DB $REPEAT+5, 0
  DB $MOVE+11
        DB      Lght
        DB      Lght,0, 0
        DB      Lght,0, Lght
        DB      0, Lght,0
        DB      0
  DB $REPEAT+4
        DB Lght
  DB $MOVE+1
        DB      0
  DB $REPEAT+3
        DB      Lght
IF HELP_ON_F1 and HELP_SHOW_KBD
  DB $WORDS
  DB $INC
  DW 3030h-0F920h
  DB $LOOP+49
  DB $STORE
  DB $OFFSET
  DW -1
  DB $STORE
  DB $INC
  DW 0000h-0101h
  DB $RET ; value = FEFF
ENDIF
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- End of Palette',0
ENDIF

IF HELP_ON_F1
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- Start of Help_Text',0
ENDIF
de_Help_Text:
  DB $BYTES
  DB $MOVE+1, 0C9h
  DB $REPEAT+19, 0CDh
  DB $MOVE+18, 0BBh,0,0BAh,' F1 - this Help'
  DB $REPEAT+4, ' '
IF SAVE_ON_F2
  DB $MOVE+19, 0BAh,0,0BAh,' F2 - save Z.SNA'
  DB $REPEAT+3, ' '
ENDIF
IF LOAD_ON_F3
IF LOAD_SELECT
  DB $MOVE+19, 0BAh,0,0BAh,' F3 - load .SNA '
ELSE
  DB $MOVE+19, 0BAh,0,0BAh,' F3 - load Z.SNA'
ENDIF
  DB $REPEAT+3, ' '
ENDIF
IF RELOAD_ON_F4
  DB $MOVE+22, 0BAh,0,0BAh,' F4 - reload .SNA  '
ENDIF
  DB $MOVE+22, 0BAh,0,0BAh,' F5 - Sound toggle '
  DB $MOVE+13, 0BAh,0,0BAh,' ESC- Exit'
  DB $REPEAT+9, ' '
  DB $MOVE+3, 0BAh,0,0C8h
  DB $REPEAT+19, 0CDh
  DB $MOVE+3, 0BCh,0,0
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- End of Help_Text',0
ENDIF
ENDIF ; HELP_ON_F1
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- Start of Processor_State',0
ENDIF
de_Processor_State:
  DB $BYTES
  DB $LOOP+64
  DB $REPEAT+64, 0
  DB $RET
de_TCounter:
  DB $MOVE+2
  DW 0FFFFh - 224 + 1
de_Sna_Regs_Start:
  DB $REPEAT+33, 0
IF CAREFUL_INT
de_TCntInt:
  DB $MOVE+2
  DW -1000
ENDIF
IF SPECTRUM_128K
  DB $MOVE+4, 0, 0, 7, 9
ELSE
  DB $MOVE+2, 0, 0
ENDIF
IF LOGS and LOG_DECOMPRESS_TABLES
        DB $LOG, '--- End of Processor_State',0
ENDIF

IF LOAD_ON_F3 and LOAD_SELECT

  DB $MOVE+3, ' :\'
  DB $REPEAT+38, 0
  DB $LOOP+5
  DB $REPEAT+64, 0
  DB $RET
  DB $MOVE+15, 0DAh, 'Select',0C4h,'file:',0BFh,0
  DB $LOOP+16
  DB $MOVE+1, 0B3h
  DB $REPEAT+12, ' '
  DB $MOVE+2, 0B3h,0
  DB $RET
  DB $MOVE+1, 0C0h
  DB $REPEAT+12, 0C4h
  DB $MOVE+3, 0D9h,0,0
ENDIF ; LOAD_ON_F3 and LOAD_SELECT

Decompress_End:

ELSE ; not COMPRESS_TABLES
;*************************************************************
; MAIN JUMP TABLES
;*************************************************************
TableOpd1:
  ; 00
  DB opdNo, ST_+opdBC, ST_+opd@BC, ST_+opdBC
  DB ST_+opdB, ST_+opdB, ST_+opdB, ST_+opdA
  DB ST_+opdAF, ST_+opdHL, ST_+opdA, ST_+opdBC
  DB ST_+opdC, ST_+opdC, ST_+opdC, ST_+opdA
  ; 10
  DB opdJR, ST_+opdDE, ST_+opd@DE, ST_+opdDE
  DB ST_+opdD, ST_+opdD, ST_+opdD, ST_+opdA
  DB opdJR, ST_+opdHL, ST_+opdA, ST_+opdDE
  DB ST_+opdE, ST_+opdE, ST_+opdE, ST_+opdA
  ; 20
  DB opdJR, ST_+opdHL, ST_+opdAddr2, ST_+opdHL
  DB ST_+opdHx, ST_+opdHx, ST_+opdHx, ST_+opdA
  DB opdJR, ST_+opdHL, ST_+opdHL, ST_+opdHL
  DB ST_+opdLx, ST_+opdLx, ST_+opdLx, ST_+opdA
  ; 30
  DB opdJR, ST_+opdSP, ST_+opdAddr1, ST_+opdSP
  DB ST_+opdM, ST_+opdM, ST_+opdM, opdNo
  DB opdJR, ST_+opdHL, ST_+opdA, ST_+opdSP
  DB ST_+opdA, ST_+opdA, ST_+opdA, opdNo
  ; 40
  DB ST_+opdB, ST_+opdB, ST_+opdB, ST_+opdB
  DB ST_+opdB, ST_+opdB, ST_+opdB, ST_+opdB
  DB ST_+opdC, ST_+opdC, ST_+opdC, ST_+opdC
  DB ST_+opdC, ST_+opdC, ST_+opdC, ST_+opdC
  ; 50
  DB ST_+opdD, ST_+opdD, ST_+opdD, ST_+opdD
  DB ST_+opdD, ST_+opdD, ST_+opdD, ST_+opdD
  DB ST_+opdE, ST_+opdE, ST_+opdE, ST_+opdE
  DB ST_+opdE, ST_+opdE, ST_+opdE, ST_+opdE
  ; 60
  DB ST_+opdHx, ST_+opdHx, ST_+opdHx, ST_+opdHx
  DB ST_+opdHx, ST_+opdHx, ST_+opdH, ST_+opdHx  ; 66: was ST_+opdHx
  DB ST_+opdLx, ST_+opdLx, ST_+opdLx, ST_+opdLx
  DB ST_+opdLx, ST_+opdLx, ST_+opdL, ST_+opdLx
  ; 70
  DB ST_+opdM, ST_+opdM, ST_+opdM, ST_+opdM
  DB ST_+opdM, ST_+opdM, opdNo, ST_+opdM
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  ; 80
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  ; 90
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  ; A0
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  ; B0
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB opdA, opdA, opdA, opdA
  DB opdA, opdA, opdA, opdA
  ; C0
  DB opdNo, ST_+opdBC, opdWord, opdWord
  DB opdWord, opdNo, ST_+opdA, opdbbb
  DB opdNo, opdNo, opdWord, opdNo
  DB opdWord, opdWord, ST_+opdA, opdbbb
  ; D0
  DB opdNo, ST_+opdDE, opdWord, opdByte
  DB opdWord, opdNo, ST_+opdA, opdbbb
  DB opdNo, opdNo, opdWord, ST_+opdA
  DB opdWord, opdByte, ST_+opdA, opdbbb
  ; E0
  DB opdNo, ST_+opdHL, opdWord, ST_+opdHL
  DB opdWord, opdNo, ST_+opdA, opdbbb
  DB opdNo, opdHL, opdWord, ST_+opdDE
  DB opdWord, opdByte, ST_+opdA, opdbbb
  ; F0
  DB opdNo, ST_+opdAF, opdWord, opdNo
  DB opdWord, opdNo, ST_+opdA, opdbbb
  DB opdNo, ST_+opdSP, opdWord, opdNo
  DB opdWord, opdByte, opdA, opdbbb

TableOpd2:
  ; 00
  DB opdNo, opdWord, opdA, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  DB opdNo, opdBC, opd@BC, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  ; 10
  DB opdCondDJNZ, opdWord, opdA, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  DB opdNo, opdDE, opd@DE, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  ; 20
  DB opdCondNZ, opdWord, opdHL, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  DB opdCondZ, opdHL, opdAddr2, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  ; 30
  DB opdCondNC, opdWord, opdA, opdNo
  DB opdNo, opdNo, opdByte, opdA
  DB opdCondC, opdSP, opdAddr1, opdNo
  DB opdNo, opdNo, opdByte, opdA
  ; 40
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 50
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 60
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 70
  DB opdB, opdC, opdD, opdE
  DB opdH, opdL, opdNo, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 80
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 90
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; A0
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; B0
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; C0
  DB opdCondNZ, opdNo, opdCondNZ, opdNo
  DB opdCondNZ, opdBC, opdByte, opdNo
  DB opdCondZ, opdNo, opdCondZ, opdByte
  DB opdCondZ, opdNo, opdByte, opdNo
  ; D0
  DB opdCondNC, opdNo, opdCondNC, opdA
  DB opdCondNC, opdDE, opdByte, opdNo
  DB opdCondC, opdNo, opdCondC, opdByte
  DB opdCondC, opdNo, opdByte, opdNo
  ; E0
  DB opdCondPO, opdNo, opdCondPO, opdHL
  DB opdCondPO, opdHL, opdByte, opdNo
  DB opdCondPE, opdHL, opdCondPE, opdHL
  DB opdCondPE, opdNo, opdByte, opdNo
  ; F0
  DB opdCondP, opdNo, opdCondP, opdNo
  DB opdCondP, opdAF, opdByte, opdNo
  DB opdCondM, opdHL, opdCondM, opdNo
  DB opdCondM, opdNo, opdByte, opdNo

JumpTable:
; 00
  DW Nop_, Ld_, Ld_, Inc_rp
  DW Inc_r, Dec_r, Ld_, Rlca_
  DW Ex_af_af_alt, Add_rp, Ld_, Dec_rp
  DW Inc_r, Dec_r, Ld_, Rrca_
; 10
  DW JR_cond, Ld_, Ld_, Inc_rp
  DW Inc_r, Dec_r, Ld_, Rla_
  DW J_, Add_rp, Ld_, Dec_rp
  DW Inc_r, Dec_r, Ld_, Rra_
; 20
  DW JR_cond, Ld_, Ld_, Inc_rp
  DW Inc_r, Dec_r, Ld_, Daa_
  DW JR_cond, Add_rp, Ld_, Dec_rp
  DW Inc_r, Dec_r, Ld_, Cpl_
; 30
  DW JR_cond, Ld_, Ld_, Inc_rp
  DW Inc_r, Dec_r, Ld_, Scf_
  DW JR_cond, Add_rp, Ld_, Dec_rp
  DW Inc_r, Dec_r, Ld_, Ccf_
; 40
  DW Nop_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Nop_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
; 50
  DW Ld_, Ld_, Nop_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Nop_
  DW Ld_, Ld_, Ld_, Ld_
; 60
  DW Ld_, Ld_, Ld_, Ld_
  DW Nop_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Nop_, Ld_, Ld_
; 70
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Halt_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Nop_
; 80
  DW Add_r, Add_r, Add_r, Add_r
  DW Add_r, Add_r, Add_r, Add_r
  DW Adc_r, Adc_r, Adc_r, Adc_r
  DW Adc_r, Adc_r, Adc_r, Adc_r
; 90
  DW Sub_r, Sub_r, Sub_r, Sub_r
  DW Sub_r, Sub_r, Sub_r, Sub_r
  DW Sbc_r, Sbc_r, Sbc_r, Sbc_r
  DW Sbc_r, Sbc_r, Sbc_r, Sbc_r
; A0
  DW And_r, And_r, And_r, And_r
  DW And_r, And_r, And_r, And_r
  DW Xor_r, Xor_r, Xor_r, Xor_r
  DW Xor_r, Xor_r, Xor_r, Xor_r
; B0
  DW Or_r, Or_r, Or_r, Or_r
  DW Or_r, Or_r, Or_r, Or_r
  DW Cp_r, Cp_r, Cp_r, Cp_r
  DW Cp_r, Cp_r, Cp_r, Cp_r
; C0
  DW Ret_cond, Pop_rp, JP_cond, J_
  DW Call_cond, Push_rp, Add_r, Call_
  DW Ret_cond, Ret_, JP_cond, _CB_
  DW Call_cond, Call_, Adc_r, Call_
; D0
  DW Ret_cond, Pop_rp, JP_cond, Out_
  DW Call_cond, Push_rp, Sub_r, Call_
  DW Ret_cond, Exx_, JP_cond, In_@byte
  DW Call_cond, _IX_, Sbc_r, Call_
; E0
  DW Ret_cond, Pop_rp, JP_cond, Ex_@sp_hl
  DW Call_cond, Push_rp, And_r, Call_
  DW Ret_cond, J_, JP_cond, Ex_de_hl
  DW Call_cond, _ED_, Xor_r, Call_
; F0
  DW Ret_cond, Pop_rp, JP_cond, Di_
  DW Call_cond, Push_rp, Or_r, Call_
  DW Ret_cond, Ld_, JP_cond, Ei_
  DW Call_cond, _IY_, Cp_r, Call_

TableUpdFlgs:
  ; 00
  DB 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXP         ; INC B
  DB 0,_SZYHXPN         ; DEC B
  DB 0,0, _HN,_YXC      ; RLCA
  ; 08
  DB 0,0                ; EX AF,AF'
  DB NF, _YHXC          ; ADD HL, BC
  DB 0,0, 0,0, NF,_SZYHXP; INC C
  DB 0,_SZYHXPN         ; DEC C
  DB 0,0, _HN,_YXC      ; RRCA
  ; 10
  DB 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXP ; INC D
  DB 0,_SZYHXPN ; DEC D
  DB 0,0, _HN,_YXC      ; RLA
  ; 18
  DB 0,0, NF, _YHXC     ; ADD HL, DE
  DB 0,0, 0,0, NF,_SZYHXP; INC E
  DB 0,_SZYHXPN         ; DEC E
  DB 0,0, _HN,_YXC      ; RRA
  ; 20
  DB 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXP         ; INC H
  DB 0,_SZYHXPN         ; DEC H
  DB 0,0, 0,_SZYHXPC    ; DAA
  ; 28
  DB 0,0, NF, _YHXC     ; ADD HL, HL
  DB 0,0, 0,0
  DB NF,_SZYHXP         ; INC L
  DB 0,_SZYHXPN         ; DEC L
  DB 0,0, 0,_YHXN       ; CPL
  ; 30
  DB 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXP         ; INC (HL)
  DB 0,_SZYHXPN         ; DEC (HL)
  DB 0,0, _HN,_YXC      ; SCF
  ; 38
  DB 0,0, NF, _YHXC     ; ADD HL, SP
  DB 0,0, 0,0
  DB NF,_SZYHXP         ; INC A
  DB 0,_SZYHXPN         ; DEC A
  DB 0,0, NF,_YHXC      ; CCF
  ; 40
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  ; 80
  DB NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC
  DB NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC
  DB NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC
  DB NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC
  ; 90
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  ; A0
  DB _NC,_SZYHXP, _NC,_SZYHXP, _NC,_SZYHXP, _NC,_SZYHXP
  DB _NC,_SZYHXP, _NC,_SZYHXP, _NC,_SZYHXP, _NC,_SZYHXP
  DB _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP
  DB _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP
  ; B0
  DB _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP
  DB _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  ; C0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXPC, 0,0 ; <- ADD byte
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXPC, 0,0 ; <- ADC byte
  ; D0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,_SZYHXPNC, 0,0 ; <- SUB byte
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,_SZYHXPNC, 0,0 ; <- SBC byte
  ; E0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB _NC,_SZYHXP, 0,0 ; <- AND byte
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB _HNC,_SZYXP, 0,0 ; <- XOR byte
  ; F0
  DB 0,0, 0,0, 0,0, 0,0 ; <- POP AF
  DB 0,0, 0,0, _HNC,_SZYXP ; OR byte
  DB 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,_SZYHXPNC, 0,0

TactTable:
db 04h, 0Ah, 07h, 06h, 04h, 04h, 07h, 04h, 04h, 0Bh, 07h, 06h, 04h, 04h, 07h, 04h
db 08h, 0Ah, 07h, 06h, 04h, 04h, 07h, 04h, 0Ch, 0Bh, 07h, 06h, 04h, 04h, 07h, 04h
db 07h, 0Ah, 10h, 06h, 04h, 04h, 07h, 04h, 07h, 0Bh, 10h, 06h, 04h, 04h, 07h, 04h
db 07h, 0Ah, 0Dh, 06h, 0Bh, 0Bh, 0Ah, 04h, 07h, 0Bh, 0Dh, 06h, 04h, 04h, 07h, 04h
db 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h, 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h
db 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h, 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h
db 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h, 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h
db 07h, 07h, 07h, 07h, 07h, 07h, 04h, 07h, 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h
db 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h, 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h
db 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h, 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h
db 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h, 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h
db 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h, 04h, 04h, 04h, 04h, 04h, 04h, 07h, 04h
db 05h, 0Ah, 0Ah, 0Ah, 0Ah, 0Bh, 07h, 0Bh, 05h, 0Ah, 0Ah, 00h, 0Ah, 11h, 07h, 0Bh
db 05h, 0Ah, 0Ah, 0Bh, 0Ah, 0Bh, 07h, 0Bh, 05h, 04h, 0Ah, 0Bh, 0Ah, 04h, 07h, 0Bh
db 05h, 0Ah, 0Ah, 13h, 0Ah, 0Bh, 07h, 0Bh, 05h, 04h, 0Ah, 04h, 0Ah, 00h, 07h, 0Bh
db 05h, 0Ah, 0Ah, 04h, 0Ah, 0Bh, 07h, 0Bh, 05h, 06h, 0Ah, 04h, 0Ah, 04h, 07h, 0Bh

;*************************************************************
; _CB_ TABLES
;*************************************************************
CB_TableOpd1:
  ; 00
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; 10
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; 20
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; 30
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; 40
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdM, opdA
  ; 50
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdM, opdA
  ; 60
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdM, opdA
  ; 70
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdM, opdA
  ; 80
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; 90
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; A0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; B0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; C0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; D0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; E0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  ; F0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, ST_+opdM, ST_+opdA

CB_TableOpd2:
  ; 00
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  ; 10
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  ; 20
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  ; 30
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  ; 40
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; 50
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; 60
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; 70
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; 80
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; 90
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; A0
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; B0
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; C0
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; D0
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; E0
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  ; F0
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit
  DB opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit, opdNbit

CB_JumpTable:
  ; 00
  DW Rlc_, Rlc_, Rlc_, Rlc_, Rlc_, Rlc_, Rlc_, Rlc_
  DW Rrc_, Rrc_, Rrc_, Rrc_, Rrc_, Rrc_, Rrc_, Rrc_
  ; 10
  DW Rl_, Rl_, Rl_, Rl_, Rl_, Rl_, Rl_, Rl_
  DW Rr_, Rr_, Rr_, Rr_, Rr_, Rr_, Rr_, Rr_
  ; 20
  DW Sla_, Sla_, Sla_, Sla_, Sla_, Sla_, Sla_, Sla_
  DW Sra_, Sra_, Sra_, Sra_, Sra_, Sra_, Sra_, Sra_
  ; 30
  DW Sll_, Sll_, Sll_, Sll_, Sll_, Sll_, Sll_, Sll_
  DW Srl_, Srl_, Srl_, Srl_, Srl_, Srl_, Srl_, Srl_
  ; 40
  DW BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r_hidden, BitN_r
  DW BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r_hidden, BitN_r
  DW BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r_hidden, BitN_r
  DW BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r_hidden, BitN_r
  DW BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r_hidden, BitN_r
  DW BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r_hidden, BitN_r
  DW BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r_hidden, BitN_r
  DW BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r, BitN_r_hidden, BitN_r
  ; 80
  DW ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r
  DW ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r
  DW ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r
  DW ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r
  DW ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r
  DW ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r
  DW ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r
  DW ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r, ResN_r
  ; C0
  DW SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r
  DW SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r
  DW SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r
  DW SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r
  DW SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r
  DW SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r
  DW SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r
  DW SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r, SetN_r

CB_TableUpdFlgs:
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  ; 10
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  ; 20
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  ; 30
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  ; 40
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  ; 50
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  ; 60
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  ; 70
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  ; 80
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  ; C0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0

CB_TactTable:
db 8, 8, 8, 8, 8, 8, 0Fh, 8 ; 00
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8 ; 20
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Ch, 8 ; 40
db 8, 8, 8, 8, 8, 8, 0Ch, 8
db 8, 8, 8, 8, 8, 8, 0Ch, 8
db 8, 8, 8, 8, 8, 8, 0Ch, 8
db 8, 8, 8, 8, 8, 8, 0Ch, 8 ; 60
db 8, 8, 8, 8, 8, 8, 0Ch, 8
db 8, 8, 8, 8, 8, 8, 0Ch, 8
db 8, 8, 8, 8, 8, 8, 0Ch, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8 ; 80
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8 ; A0
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8 ; C0
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8 ; E0
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8
db 8, 8, 8, 8, 8, 8, 0Fh, 8

;*************************************************************
; _DD_ (IX) AND _FD_ (IY) TABLES
;*************************************************************
XY_TableOpd1: ; the same as main TableOpd1
  ; 00
  DB opdNo, ST_+opdBC, ST_+opd@BC, ST_+opdBC
  DB ST_+opdB, ST_+opdB, ST_+opdB, ST_+opdA
  DB ST_+opdAF, ST_+opdHL, ST_+opdA, ST_+opdBC
  DB ST_+opdC, ST_+opdC, ST_+opdC, ST_+opdA
  ; 10
  DB opdJR, ST_+opdDE, ST_+opd@DE, ST_+opdDE
  DB ST_+opdD, ST_+opdD, ST_+opdD, ST_+opdA
  DB opdJR, ST_+opdHL, ST_+opdA, ST_+opdDE
  DB ST_+opdE, ST_+opdE, ST_+opdE, ST_+opdA
  ; 20
  DB opdJR, ST_+opdHL, ST_+opdAddr2, ST_+opdHL
  DB ST_+opdHx, ST_+opdHx, ST_+opdHx, ST_+opdA
  DB opdJR, ST_+opdHL, ST_+opdHL, ST_+opdHL
  DB ST_+opdLx, ST_+opdLx, ST_+opdLx, ST_+opdA
  ; 30
  DB opdJR, ST_+opdSP, ST_+opdAddr1, ST_+opdSP
  DB ST_+opdM, ST_+opdM, ST_+opdM, opdNo
  DB opdJR, ST_+opdHL, ST_+opdA, ST_+opdSP
  DB ST_+opdA, ST_+opdA, ST_+opdA, opdNo
  ; 40
  DB ST_+opdB, ST_+opdB, ST_+opdB, ST_+opdB
  DB ST_+opdB, ST_+opdB, ST_+opdB, ST_+opdB
  DB ST_+opdC, ST_+opdC, ST_+opdC, ST_+opdC
  DB ST_+opdC, ST_+opdC, ST_+opdC, ST_+opdC
  ; 50
  DB ST_+opdD, ST_+opdD, ST_+opdD, ST_+opdD
  DB ST_+opdD, ST_+opdD, ST_+opdD, ST_+opdD
  DB ST_+opdE, ST_+opdE, ST_+opdE, ST_+opdE
  DB ST_+opdE, ST_+opdE, ST_+opdE, ST_+opdE
  ; 60
  DB ST_+opdHx, ST_+opdHx, ST_+opdHx, ST_+opdHx
  DB ST_+opdHx, ST_+opdHx, ST_+opdH, ST_+opdHx  ; 66: was ST_+opdHx
  DB ST_+opdLx, ST_+opdLx, ST_+opdLx, ST_+opdLx
  DB ST_+opdLx, ST_+opdLx, ST_+opdL, ST_+opdLx
  ; 70
  DB ST_+opdM, ST_+opdM, ST_+opdM, ST_+opdM
  DB ST_+opdM, ST_+opdM, opdNo, ST_+opdM
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  ; 80
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  ; 90
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  ; A0
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  ; B0
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB ST_+opdA, ST_+opdA, ST_+opdA, ST_+opdA
  DB opdA, opdA, opdA, opdA
  DB opdA, opdA, opdA, opdA
  ; C0
  DB opdNo, ST_+opdBC, opdWord, opdWord
  DB opdWord, opdNo, ST_+opdA, opdbbb
  DB opdNo, opdNo, opdWord, opdByte ; for CB, byte opd1 is loaded from the second byte
                                    ; it is used as additional op code: DD/FD CB opd2 opd1
  DB opdWord, opdWord, ST_+opdA, opdbbb
  ; D0
  DB opdNo, ST_+opdDE, opdWord, opdByte
  DB opdWord, opdNo, ST_+opdA, opdbbb
  DB opdNo, opdNo, opdWord, ST_+opdA  ; DD: opdNo replaces opdByte
  DB opdWord, opdNo, ST_+opdA, opdbbb
  ; E0
  DB opdNo, ST_+opdHL, opdWord, ST_+opdHL
  DB opdWord, opdNo, ST_+opdA, opdbbb ; ED: opdNo replaces opdByte
  DB opdNo, opdHL, opdWord, ST_+opdDE
  DB opdWord, opdNo, ST_+opdA, opdbbb
  ; F0
  DB opdNo, ST_+opdAF, opdWord, opdNo
  DB opdWord, opdNo, ST_+opdA, opdbbb ; FD: opdNo replaces opdByte
  DB opdNo, ST_+opdSP, opdWord, opdNo
  DB opdWord, opdNo, opdA, opdbbb

XY_TableOpd2: ; the same as main TableOpd2
  ; 00
  DB opdNo, opdWord, opdA, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  DB opdNo, opdBC, opd@BC, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  ; 10
  DB opdCondDJNZ, opdWord, opdA, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  DB opdNo, opdDE, opd@DE, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  ; 20
  DB opdCondNZ, opdWord, opdHL, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  DB opdCondZ, opdHL, opdAddr2, opdNo
  DB opdNo, opdNo, opdByte, opdNo
  ; 30
  DB opdCondNC, opdWord, opdA, opdNo
  DB opdNo, opdNo, opdNo, opdA ; opdNo for 36h opcode is correct! do not modify
  DB opdCondC, opdSP, opdAddr1, opdNo
  DB opdNo, opdNo, opdByte, opdA
  ; 40
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 50
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 60
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 70
  DB opdB, opdC, opdD, opdE
  DB opdH, opdL, opdNo, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 80
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; 90
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; A0
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; B0
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  DB opdB, opdC, opdD, opdE
  DB opdHx, opdLx, opdM, opdA
  ; C0
  DB opdCondNZ, opdNo, opdCondNZ, opdNo
  DB opdCondNZ, opdBC, opdByte, opdNo
  DB opdCondZ, opdNo, opdCondZ, opd@M ; for _CB_ changed in compare to main TableOpd2
  DB opdCondZ, opdNo, opdByte, opdNo
  ; D0
  DB opdCondNC, opdNo, opdCondNC, opdA
  DB opdCondNC, opdDE, opdByte, opdNo
  DB opdCondC, opdNo, opdCondC, opdByte
  DB opdCondC, opdNo, opdByte, opdNo
  ; E0
  DB opdCondPO, opdNo, opdCondPO, opdHL
  DB opdCondPO, opdHL, opdByte, opdNo
  DB opdCondPE, opdHL, opdCondPE, opdHL
  DB opdCondPE, opdNo, opdByte, opdNo
  ; F0
  DB opdCondP, opdNo, opdCondP, opdNo
  DB opdCondP, opdAF, opdByte, opdNo
  DB opdCondM, opdHL, opdCondM, opdNo
  DB opdCondM, opdNo, opdByte, opdNo

XY_JumpTable: ; the same as main JumpTable except prefixes interpreting
; 00
  DW Nop_, Ld_, Ld_, Inc_rp
  DW Inc_r, Dec_r, Ld_, Rlca_
  DW Ex_af_af_alt, Add_rp, Ld_, Dec_rp
  DW Inc_r, Dec_r, Ld_, Rrca_
; 10
  DW JR_cond, Ld_, Ld_, Inc_rp
  DW Inc_r, Dec_r, Ld_, Rla_
  DW J_, Add_rp, Ld_, Dec_rp
  DW Inc_r, Dec_r, Ld_, Rra_
; 20
  DW JR_cond, Ld_, Ld_, Inc_rp
  DW Inc_r, Dec_r, Ld_, Daa_
  DW JR_cond, Add_rp, Ld_, Dec_rp
  DW Inc_r, Dec_r, Ld_, Cpl_
; 30
  DW JR_cond, Ld_, Ld_, Inc_rp
  DW Inc_r, Dec_r, Ld_@XY_byte, Scf_ ; 36h: a change to main jump table!
  DW JR_cond, Add_rp, Ld_, Dec_rp
  DW Inc_r, Dec_r, Ld_, Ccf_
; 40
  DW Nop_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Nop_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
; 50
  DW Ld_, Ld_, Nop_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Nop_
  DW Ld_, Ld_, Ld_, Ld_
; 60
  DW Ld_, Ld_, Ld_, Ld_
  DW Nop_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Nop_, Ld_, Ld_
; 70
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Halt_, Ld_
  DW Ld_, Ld_, Ld_, Ld_
  DW Ld_, Ld_, Ld_, Nop_
; 80
  DW Add_r, Add_r, Add_r, Add_r
  DW Add_r, Add_r, Add_r, Add_r
  DW Adc_r, Adc_r, Adc_r, Adc_r
  DW Adc_r, Adc_r, Adc_r, Adc_r
; 90
  DW Sub_r, Sub_r, Sub_r, Sub_r
  DW Sub_r, Sub_r, Sub_r, Sub_r
  DW Sbc_r, Sbc_r, Sbc_r, Sbc_r
  DW Sbc_r, Sbc_r, Sbc_r, Sbc_r
; A0
  DW And_r, And_r, And_r, And_r
  DW And_r, And_r, And_r, And_r
  DW Xor_r, Xor_r, Xor_r, Xor_r
  DW Xor_r, Xor_r, Xor_r, Xor_r
; B0
  DW Or_r, Or_r, Or_r, Or_r
  DW Or_r, Or_r, Or_r, Or_r
  DW Cp_r, Cp_r, Cp_r, Cp_r
  DW Cp_r, Cp_r, Cp_r, Cp_r
; C0
  DW Ret_cond, Pop_rp, JP_cond, J_
  DW Call_cond, Push_rp, Add_r, Call_
  DW Ret_cond, Ret_, JP_cond, XY_CB_
  DW Call_cond, Call_, Adc_r, Call_
; D0
  DW Ret_cond, Pop_rp, JP_cond, Out_
  DW Call_cond, Push_rp, Sub_r, Call_
  DW Ret_cond, Exx_, JP_cond, In_@byte
  DW Call_cond, XY_IX_, Sbc_r, Call_  ; XY_IX_ replaces _IX_
; E0
  DW Ret_cond, Pop_rp, JP_cond, Ex_@sp_hl
  DW Call_cond, Push_rp, And_r, Call_
  DW Ret_cond, J_, JP_cond, Ex_de_hl
  DW Call_cond, Nop_, Xor_r, Call_      ; _XY_ED_ replaced with Nop_
; F0
  DW Ret_cond, Pop_rp, JP_cond, Di_
  DW Call_cond, Push_rp, Or_r, Call_
  DW Ret_cond, Ld_, JP_cond, Ei_
  DW Call_cond, XY_IY_, Cp_r, Call_ ; XY_IY_ replaces _IY_

XY_TableUpdFlgs: ; - the same as for main table
  ; 00
  DB 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXP         ; INC B
  DB 0,_SZYHXPN         ; DEC B
  DB 0,0, _HN,_YXC      ; RLCA
  ; 08
  DB 0,0        ; EX AF,AF'
  DB NF, _YHXC          ; ADD HL, BC
  DB 0,0, 0,0, NF,_SZYHXP; INC C
  DB 0,_SZYHXPN         ; DEC C
  DB 0,0, _HN,_YXC      ; RRCA
  ; 10
  DB 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXP ; INC D
  DB 0,_SZYHXPN ; DEC D
  DB 0,0, _HN,_YXC      ; RLA
  ; 18
  DB 0,0, NF, _YHXC     ; ADD HL, DE
  DB 0,0, 0,0, NF,_SZYHXP; INC E
  DB 0,_SZYHXPN         ; DEC E
  DB 0,0, _HN,_YXC      ; RRA
  ; 20
  DB 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXP         ; INC H
  DB 0,_SZYHXPN         ; DEC H
  DB 0,0, 0,_SZYHXPC    ; DAA
  ; 28
  DB 0,0, NF, _YHXC     ; ADD HL, HL
  DB 0,0, 0,0
  DB NF,_SZYHXP         ; INC L
  DB 0,_SZYHXPN         ; DEC L
  DB 0,0, 0,_YHXN       ; CPL
  ; 30
  DB 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXP         ; INC (HL)
  DB 0,_SZYHXPN         ; DEC (HL)
  DB 0,0, _HN,_YXC      ; SCF
  ; 38
  DB 0,0, NF, _YHXC     ; ADD HL, SP
  DB 0,0, 0,0
  DB NF,_SZYHXP         ; INC A
  DB 0,_SZYHXPN         ; DEC A
  DB 0,0, NF,_YHXC      ; CCF
  ; 40
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  ; 80
  DB NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC
  DB NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC
  DB NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC
  DB NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC, NF,_SZYHXPC
  ; 90
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  ; A0
  DB _NC,_SZYHXP, _NC,_SZYHXP, _NC,_SZYHXP, _NC,_SZYHXP
  DB _NC,_SZYHXP, _NC,_SZYHXP, _NC,_SZYHXP, _NC,_SZYHXP
  DB _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP
  DB _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP
  ; B0
  DB _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP
  DB _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP, _HNC,_SZYXP
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  DB 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,_SZYHXPNC
  ; C0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXPC, 0,0 ; <- ADD byte
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB NF,_SZYHXPC, 0,0 ; <- ADC byte
  ; D0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,_SZYHXPNC, 0,0 ; <- SUB byte
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,_SZYHXPNC, 0,0 ; <- SBC byte
  ; E0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB _NC,_SZYHXP, 0,0 ; <- AND byte
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB _HNC,_SZYXP, 0,0 ; <- XOR byte
  ; F0
  DB 0,0, 0,0, 0,0, 0,0 ; <- POP AF
  DB 0,0, 0,0, _HNC,_SZYXP ; OR byte
  DB 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,_SZYHXPNC, 0,0

XY_TactTable:
db   4, 0Ah,   7,   6,   4,   4,   7,   4 ; 00
db   4, 0Bh,   7,   6,   4,   4,   7,   4
db   8, 0Ah, 10h,   6,   4,   4,   7,   4 ; 10
db   8, 0Bh, 10h,   6,   4,   4,   7,   4
db   7, 0Ah, 10h,   6,   4,   4,   7,   4 ; 20
db   7, 0Bh, 10h,   6,   4,   4,   7,   4
db   7, 0Ah, 0Dh,   6, 13h, 13h, 0Ah,   4 ; 30
db   7, 0Bh, 0Dh,   6,   4,   4,   7,   4
db   4,   4,   4,   4,   4,   4, 0Fh,   4 ; 40
db   4,   4,   4,   4,   4,   4, 0Fh,   4
db   4,   4,   4,   4,   4,   4, 0Fh,   4 ; 50
db   4,   4,   4,   4,   4,   4, 0Fh,   4
db   4,   4,   4,   4,   4,   4, 0Fh,   4 ; 60
db   4,   4,   4,   4,   4,   4, 0Fh,   4
db 0Fh, 0Fh, 0Fh, 0Fh, 0Fh, 0Fh,   4, 0Fh ; 70
db   4,   4,   4,   4,   4,   4, 0Fh,   4
db   4,   4,   4,   4,   4,   4, 0Fh,   4 ; 80
db   4,   4,   4,   4,   4,   4, 0Fh,   4
db   4,   4,   4,   4,   4,   4, 0Fh,   4 ; 90
db   4,   4,   4,   4,   4,   4, 0Fh,   4
db   4,   4,   4,   4,   4,   4, 0Fh,   4 ; A0
db   4,   4,   4,   4,   4,   4, 0Fh,   4
db   4,   4,   4,   4,   4,   4, 0Fh,   4 ; B0
db   4,   4,   4,   4,   4,   4, 0Fh,   4
db   5, 0Ah, 0Ah, 0Ah, 0Ah, 0Ah,   7, 0Bh ; C0
db   5, 0Ah, 0Ah,   4, 0Ah, 11h,   7, 0Bh
db   5, 0Ah, 0Ah, 0Bh, 0Ah, 0Bh,   7, 0Bh ; D0
db   5,   4, 0Ah, 0Bh, 0Ah,   4,   7, 0Bh
db   5, 0Ah, 0Ah, 13h, 0Ah, 0Bh,   7, 0Bh ; E0
db   5,   4, 0Ah,   4, 0Ah,   4,   7, 0Bh
db   5, 0Ah, 0Ah,   4, 0Ah, 0Bh,   7, 0Bh ; F0
db   5,   6, 0Ah,   4, 0Ah,   4,   7, 0Bh

;*************************************************************
; _DD_CB_ (IX*CB) AND _FD_CB_ (IY*CB) TABLES
;*************************************************************
XYCB_TableOpd1: ; similar to CB_TableOpd1
  ; 00
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; 10
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; 20
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; 30
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; 40
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdNo, opdA
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdNo, opdA
  ; 50
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdNo, opdA
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdNo, opdA
  ; 60
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdNo, opdA
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdNo, opdA
  ; 70
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdNo, opdA
  DB opdB, opdC, opdD, opdE, opdHx, opdLx, opdNo, opdA
  ; 80
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; 90
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; A0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; B0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; C0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; D0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; E0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  ; F0
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA
  DB ST_+opdB, ST_+opdC, ST_+opdD, ST_+opdE, ST_+opdHx, ST_+opdLx, opdNo, ST_+opdA

XYCB_TableOpd2: ; the same as main XY_TableOpd2
  ; 00
  DB opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB
  DB opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB
  ; 10
  DB opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB
  DB opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB
  ; 20
  DB opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB
  DB opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB
  ; 30
  DB opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB
  DB opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB, opdM_XYCB
  ; 40
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; 50
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; 60
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; 70
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; 80
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; 90
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; A0
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; B0
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; C0
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; D0
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; E0
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  ; F0
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit
  DB opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit, opdMbit

XYCB_JumpTable: ; the same as CB_JumpTable except prefixes interpreting
  ; 00
  DW Rlc_2, Rlc_2, Rlc_2, Rlc_2, Rlc_2, Rlc_2, Rlc_2, Rlc_2
  DW Rrc_2, Rrc_2, Rrc_2, Rrc_2, Rrc_2, Rrc_2, Rrc_2, Rrc_2
  ; 10
  DW Rl_2, Rl_2, Rl_2, Rl_2, Rl_2, Rl_2, Rl_2, Rl_2
  DW Rr_2, Rr_2, Rr_2, Rr_2, Rr_2, Rr_2, Rr_2, Rr_2
  ; 20
  DW Sla_2, Sla_2, Sla_2, Sla_2, Sla_2, Sla_2, Sla_2, Sla_2
  DW Sra_2, Sra_2, Sra_2, Sra_2, Sra_2, Sra_2, Sra_2, Sra_2
  ; 30
  DW Sll_2, Sll_2, Sll_2, Sll_2, Sll_2, Sll_2, Sll_2, Sll_2
  DW Srl_2, Srl_2, Srl_2, Srl_2, Srl_2, Srl_2, Srl_2, Srl_2
  ; 40
  DW BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2_hidden, BitN_r2
  DW BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2_hidden, BitN_r2
  DW BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2_hidden, BitN_r2
  DW BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2_hidden, BitN_r2
  DW BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2_hidden, BitN_r2
  DW BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2_hidden, BitN_r2
  DW BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2_hidden, BitN_r2
  DW BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2, BitN_r2_hidden, BitN_r2
  ; 80
  DW ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2
  DW ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2
  DW ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2
  DW ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2
  DW ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2
  DW ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2
  DW ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2
  DW ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2, ResN_r2
  ; C0
  DW SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2
  DW SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2
  DW SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2
  DW SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2
  DW SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2
  DW SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2
  DW SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2
  DW SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2, SetN_r2

XYCB_TableUpdFlgs: ; the same as CB_TableUpdFlgs
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  ; 10
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  ; 20
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  ; 30
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  DB _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC, _HN,_SZYXPC
  ; 40
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  ; 50
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  ; 60
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  ; 70
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  DB NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP, NF,_SZYHXP
  ; 80
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  ; C0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0

XYCB_TactTable:
; 00
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
; 40
db 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch
db 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch
db 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch
db 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch, 0ch
; 80
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
; C0
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh
db 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh, 0fh

;*************************************************************
; PREFIX _ED_ TABLES
;*************************************************************
ED_TableOpd1:
  ; 00
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  ; 40
  DB ST_+opdB, opdBC, ST_+opdHL, ST_+opdAddr2
  DB ST_+opdA, opdNo, opdNo, ST_+opdI
  DB ST_+opdC, opdBC, ST_+opdHL, ST_+opdBC
  DB ST_+opdA, opdNo, opdNo, opdR
  ; 50
  DB ST_+opdD, opdBC, ST_+opdHL, ST_+opdAddr2
  DB ST_+opdA, opdNo, opdNo, ST_+opdA
  DB ST_+opdE, opdBC, ST_+opdHL, ST_+opdDE
  DB ST_+opdA, opdNo, opdNo, ST_+opdA
  ; 60
  DB ST_+opdH, opdBC, ST_+opdHL, ST_+opdAddr2
  DB ST_+opdA, opdNo, opdNo, ST_+opdM
  DB ST_+opdL, opdBC, ST_+opdHL, ST_+opdHL
  DB ST_+opdA, opdNo, opdNo, ST_+opdM
  ; 70
  DB opdNo, opdBC, ST_+opdHL, ST_+opdAddr2
  DB ST_+opdA, opdNo, opdNo, opdNo
  DB ST_+opdA, opdBC, ST_+opdHL, ST_+opdSP
  DB ST_+opdA, opdNo, opdNo, opdNo
  ; 80
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  ; A0
  DB ST_+opd@DE, opdA, ST_+opdM, opdBC
  DB opdNo, opdNo, opdNo, opdNo
  DB ST_+opd@DE, opdA, ST_+opdM, opdBC
  DB opdNo, opdNo, opdNo, opdNo
  ; B0
  DB ST_+opd@DE, opdA, ST_+opdM, opdBC
  DB opdNo, opdNo, opdNo, opdNo
  DB ST_+opd@DE, opdA, ST_+opdM, opdBC
  DB opdNo, opdNo, opdNo, opdNo
  ; C0
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo

ED_TableOpd2:
  ; 00
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  ; 40
  DB opdBC, opdB, opdBC, opdBC
  DB opdA, opdNo, opdNo, opdA
  DB opdBC, opdC, opdBC, opdAddr2
  DB opdA, opdNo, opdNo, opdA
  ; 50
  DB opdBC, opdD, opdDE, opdDE
  DB opdA, opdNo, opdNo, opdR
  DB opdBC, opdE, opdDE, opdAddr2
  DB opdA, opdNo, opdNo, opdR
  ; 60
  DB opdBC, opdH, opdHL, opdHL
  DB opdA, opdNo, opdNo, opdA
  DB opdBC, opdL, opdHL, opdAddr2
  DB opdA, opdNo, opdNo, opdA
  ; 70
  DB opdBC, opdNo, opdSP, opdSP
  DB opdA, opdNo, opdNo, opdNo
  DB opdBC, opdA, opdSP, opdAddr2
  DB opdA, opdNo, opdNo, opdNo
  ; 80
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  ; A0
  DB opdM, opdM, opdBC, opdM
  DB opdNo, opdNo, opdNo, opdNo
  DB opdM, opdM, opdBC, opdM
  DB opdNo, opdNo, opdNo, opdNo
  ; B0
  DB opdM, opdM, opdBC, opdM
  DB opdNo, opdNo, opdNo, opdNo
  DB opdM, opdM, opdBC, opdM
  DB opdNo, opdNo, opdNo, opdNo
  ; C0
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo
  DB opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo, opdNo

ED_JumpTable:
  ; 00
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  ; 40
  DW In_@C, Out_, Sbc_rp, Ld_
  DW Neg_, Ret_N, Im_0, Ld_
  DW In_@C, Out_, Adc_rp, Ld_
  DW Neg_, Ret_I, Im_0, Ld_R_A
  ; 50
  DW In_@C, Out_, Sbc_rp, Ld_
  DW Neg_, Ret_N, Im_1, Ld_A_I
  DW In_@C, Out_, Adc_rp, Ld_
  DW Neg_, Ret_N, Im_2, Ld_A_R
  ; 60
  DW In_@C, Out_, Sbc_rp, Ld_
  DW Neg_, Ret_N, Ld_, Rrd_
  DW In_@C, Out_, Adc_rp, Ld_
  DW Neg_, Ret_N, Im_0, Rld_
  ; 70
  DW In_@C, Out_, Sbc_rp, Ld_
  DW Neg_, Ret_N, Im_1, Nop_
  DW In_@C, Out_, Adc_rp, Ld_
  DW Neg_, Ret_N, Im_2, Nop_
  ; 80
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  ; A0
  DW Ldi_, Cpi_, Ini_, Outi_, Nop_, Nop_, Nop_, Nop_
  DW Ldd_, Cpd_, Ind_, Outd_, Nop_, Nop_, Nop_, Nop_
  DW Ldir_, Cpir_, Inir_, Otir_, Nop_, Nop_, Nop_, Nop_
  DW Lddr_, Cpdr_, Indr_, Otdr_, Nop_, Nop_, Nop_, Nop_
  ; C0
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_
  DW Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_, Nop_

ED_TableUpdFlgs:
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  ; 20
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  ; 40
  DB _HN,_SZYXP, 0,0, NF,_SZYXPC, 0,0
  DB 0,_SZYHXPNC, 0,0, 0,0, 0,0
  DB _HN,_SZYXP, 0,0, NF,_SZYHXPC, 0,0
  DB 0,_SZYHXPNC, 0,0, 0,0, 0,0
  ; 50
  DB _HN,_SZYXP, 0,0, NF,_SZYXPC, 0,0
  DB 0,_SZYHXPNC, 0,0, 0,0, _HN,_SZYXP
  DB _HN,_SZYXP, 0,0, NF,_SZYHXPC, 0,0
  DB 0,_SZYHXPNC, 0,0, 0,0, _HN,_SZYXP
  ; 60
  DB _HN,_SZYXP, 0,0, NF,_SZYXPC, 0,0
  DB 0,_SZYHXPNC, 0,0, 0,0, _HN,_SZYXP
  DB _HN,_SZYXP, 0,0, NF,_SZYHXPC, 0,0
  DB 0,_SZYHXPNC, 0,0, 0,0, _HN,_SZYXP
  ; 70
  DB _HN,_SZYXP, 0,0, NF,_SZYXPC, 0,0
  DB 0,_SZYHXPNC, 0,0, 0,0, 0,0
  DB _HN,_SZYXP, 0,0, NF,_SZYHXPC, 0,0
  DB 0,_SZYHXPNC, 0,0, 0,0, 0,0
  ; 80
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  ; A0
  DB _HN,_YXP, 0,_SZYHXPN, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,0, 0,0, 0,0, 0,0
  DB _HN,_YXP, 0,_SZYHXPN, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,0, 0,0, 0,0, 0,0
  ; B0
  DB _HN,_YXP, 0,_SZYHXPN, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,0, 0,0, 0,0, 0,0
  DB _HN,_YXP, 0,_SZYHXPN, 0,_SZYHXPNC, 0,_SZYHXPNC, 0,0, 0,0, 0,0, 0,0
  ; C0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0
  DB 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0, 0,0

ED_TactTable:
  ; 00
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  ; 40
  db 0ch, 0ch, 0fh, 14h, 8, 0eh, 8, 9
  db 0ch, 0ch, 0fh, 14h, 8, 0eh, 8, 9
  db 0ch, 0ch, 0fh, 14h, 8, 0eh, 8, 9
  db 0ch, 0ch, 0fh, 14h, 8, 0eh, 8, 9
  ; 60
  db 0ch, 0ch, 0fh, 14h, 8, 0eh, 8, 18h
  db 0ch, 0ch, 0fh, 14h, 8, 0eh, 8, 18h
  db 0ch, 0ch, 0fh, 14h, 8, 0eh, 8, 4
  db 0ch, 0ch, 0fh, 14h, 8, 0eh, 8, 4
  ; 80
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  ; A0
  db 10h, 10h, 10h, 10h, 4, 4, 4, 4
  db 10h, 10h, 10h, 10h, 4, 4, 4, 4
  db 10h, 10h, 10h, 10h, 4, 4, 4, 4
  db 10h, 10h, 10h, 10h, 4, 4, 4, 4
  ; C0
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4
  db 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4

;*************************************************************
; VIDEO OUTPUT TABLES
;*************************************************************
PixelLines:
  DW 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  DW 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  DW 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  DW 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  DW 4000h, 4100h, 4200h, 4300h, 4400h, 4500h, 4600h, 4700h
  DW 4020h, 4120h, 4220h, 4320h, 4420h, 4520h, 4620h, 4720h
  DW 4040h, 4140h, 4240h, 4340h, 4440h, 4540h, 4640h, 4740h
  DW 4060h, 4160h, 4260h, 4360h, 4460h, 4560h, 4660h, 4760h
  DW 4080h, 4180h, 4280h, 4380h, 4480h, 4580h, 4680h, 4780h
  DW 40A0h, 41A0h, 42A0h, 43A0h, 44A0h, 45A0h, 46A0h, 47A0h
  DW 40C0h, 41C0h, 42C0h, 43C0h, 44C0h, 45C0h, 46C0h, 47C0h
  DW 40E0h, 41E0h, 42E0h, 43E0h, 44E0h, 45E0h, 46E0h, 47E0h
  DW 4800h, 4900h, 4A00h, 4B00h, 4C00h, 4D00h, 4E00h, 4F00h
  DW 4820h, 4920h, 4A20h, 4B20h, 4C20h, 4D20h, 4E20h, 4F20h
  DW 4840h, 4940h, 4A40h, 4B40h, 4C40h, 4D40h, 4E40h, 4F40h
  DW 4860h, 4960h, 4A60h, 4B60h, 4C60h, 4D60h, 4E60h, 4F60h
  DW 4880h, 4980h, 4A80h, 4B80h, 4C80h, 4D80h, 4E80h, 4F80h
  DW 48A0h, 49A0h, 4AA0h, 4BA0h, 4CA0h, 4DA0h, 4EA0h, 4FA0h
  DW 48C0h, 49C0h, 4AC0h, 4BC0h, 4CC0h, 4DC0h, 4EC0h, 4FC0h
  DW 48E0h, 49E0h, 4AE0h, 4BE0h, 4CE0h, 4DE0h, 4EE0h, 4FE0h
  DW 5000h, 5100h, 5200h, 5300h, 5400h, 5500h, 5600h, 5700h
  DW 5020h, 5120h, 5220h, 5320h, 5420h, 5520h, 5620h, 5720h
  DW 5040h, 5140h, 5240h, 5340h, 5440h, 5540h, 5640h, 5740h
  DW 5060h, 5160h, 5260h, 5360h, 5460h, 5560h, 5660h, 5760h
  DW 5080h, 5180h, 5280h, 5380h, 5480h, 5580h, 5680h, 5780h
  DW 50A0h, 51A0h, 52A0h, 53A0h, 54A0h, 55A0h, 56A0h, 57A0h
  DW 50C0h, 51C0h, 52C0h, 53C0h, 54C0h, 55C0h, 56C0h, 57C0h
  DW 50E0h, 51E0h, 52E0h, 53E0h, 54E0h, 55E0h, 56E0h, 57E0h
NoAttr:
  DW 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  DW 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  DW 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  DW 0, 0, 0, 0, 0, 0, 0, 0

AttrLines:
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW 5800h, 5800h, 5800h, 5800h, 5800h, 5800h, 5800h, 5800h
  DW 5820h, 5820h, 5820h, 5820h, 5820h, 5820h, 5820h, 5820h
  DW 5840h, 5840h, 5840h, 5840h, 5840h, 5840h, 5840h, 5840h
  DW 5860h, 5860h, 5860h, 5860h, 5860h, 5860h, 5860h, 5860h
  DW 5880h, 5880h, 5880h, 5880h, 5880h, 5880h, 5880h, 5880h
  DW 58A0h, 58A0h, 58A0h, 58A0h, 58A0h, 58A0h, 58A0h, 58A0h
  DW 58C0h, 58C0h, 58C0h, 58C0h, 58C0h, 58C0h, 58C0h, 58C0h
  DW 58E0h, 58E0h, 58E0h, 58E0h, 58E0h, 58E0h, 58E0h, 58E0h
  DW 5900h, 5900h, 5900h, 5900h, 5900h, 5900h, 5900h, 5900h
  DW 5920h, 5920h, 5920h, 5920h, 5920h, 5920h, 5920h, 5920h
  DW 5940h, 5940h, 5940h, 5940h, 5940h, 5940h, 5940h, 5940h
  DW 5960h, 5960h, 5960h, 5960h, 5960h, 5960h, 5960h, 5960h
  DW 5980h, 5980h, 5980h, 5980h, 5980h, 5980h, 5980h, 5980h
  DW 59A0h, 59A0h, 59A0h, 59A0h, 59A0h, 59A0h, 59A0h, 59A0h
  DW 59C0h, 59C0h, 59C0h, 59C0h, 59C0h, 59C0h, 59C0h, 59C0h
  DW 59E0h, 59E0h, 59E0h, 59E0h, 59E0h, 59E0h, 59E0h, 59E0h
  DW 5A00h, 5A00h, 5A00h, 5A00h, 5A00h, 5A00h, 5A00h, 5A00h
  DW 5A20h, 5A20h, 5A20h, 5A20h, 5A20h, 5A20h, 5A20h, 5A20h
  DW 5A40h, 5A40h, 5A40h, 5A40h, 5A40h, 5A40h, 5A40h, 5A40h
  DW 5A60h, 5A60h, 5A60h, 5A60h, 5A60h, 5A60h, 5A60h, 5A60h
  DW 5A80h, 5A80h, 5A80h, 5A80h, 5A80h, 5A80h, 5A80h, 5A80h
  DW 5AA0h, 5AA0h, 5AA0h, 5AA0h, 5AA0h, 5AA0h, 5AA0h, 5AA0h
  DW 5AC0h, 5AC0h, 5AC0h, 5AC0h, 5AC0h, 5AC0h, 5AC0h, 5AC0h
  DW 5AE0h, 5AE0h, 5AE0h, 5AE0h, 5AE0h, 5AE0h, 5AE0h, 5AE0h
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr
  DW NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr,NoAttr

VideoLines:
 DW 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h
 DW 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h
 DW 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h
 DW 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h
 DW 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h
 DW 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h
 DW 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h
 DW 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h, 3E0h
 DW 00520h, 00660h, 007A0h, 008E0h, 00A20h, 00B60h, 00CA0h, 00DE0h
 DW 00F20h, 01060h, 011A0h, 012E0h, 01420h, 01560h, 016A0h, 017E0h
 DW 01920h, 01A60h, 01BA0h, 01CE0h, 01E20h, 01F60h, 020A0h, 021E0h
 DW 02320h, 02460h, 025A0h, 026E0h, 02820h, 02960h, 02AA0h, 02BE0h
 DW 02D20h, 02E60h, 02FA0h, 030E0h, 03220h, 03360h, 034A0h, 035E0h
 DW 03720h, 03860h, 039A0h, 03AE0h, 03C20h, 03D60h, 03EA0h, 03FE0h
 DW 04120h, 04260h, 043A0h, 044E0h, 04620h, 04760h, 048A0h, 049E0h
 DW 04B20h, 04C60h, 04DA0h, 04EE0h, 05020h, 05160h, 052A0h, 053E0h
 DW 05520h, 05660h, 057A0h, 058E0h, 05A20h, 05B60h, 05CA0h, 05DE0h
 DW 05F20h, 06060h, 061A0h, 062E0h, 06420h, 06560h, 066A0h, 067E0h
 DW 06920h, 06A60h, 06BA0h, 06CE0h, 06E20h, 06F60h, 070A0h, 071E0h
 DW 07320h, 07460h, 075A0h, 076E0h, 07820h, 07960h, 07AA0h, 07BE0h
 DW 07D20h, 07E60h, 07FA0h, 080E0h, 08220h, 08360h, 084A0h, 085E0h
 DW 08720h, 08860h, 089A0h, 08AE0h, 08C20h, 08D60h, 08EA0h, 08FE0h
 DW 09120h, 09260h, 093A0h, 094E0h, 09620h, 09760h, 098A0h, 099E0h
 DW 09B20h, 09C60h, 09DA0h, 09EE0h, 0A020h, 0A160h, 0A2A0h, 0A3E0h
 DW 0A520h, 0A660h, 0A7A0h, 0A8E0h, 0AA20h, 0AB60h, 0ACA0h, 0ADE0h
 DW 0AF20h, 0B060h, 0B1A0h, 0B2E0h, 0B420h, 0B560h, 0B6A0h, 0B7E0h
 DW 0B920h, 0BA60h, 0BBA0h, 0BCE0h, 0BE20h, 0BF60h, 0C0A0h, 0C1E0h
 DW 0C320h, 0C460h, 0C5A0h, 0C6E0h, 0C820h, 0C960h, 0CAA0h, 0CBE0h
 DW 0CD20h, 0CE60h, 0CFA0h, 0D0E0h, 0D220h, 0D360h, 0D4A0h, 0D5E0h
 DW 0D720h, 0D860h, 0D9A0h, 0DAE0h, 0DC20h, 0DD60h, 0DEA0h, 0DFE0h
 DW 0E120h, 0E260h, 0E3A0h, 0E4E0h, 0E620h, 0E760h, 0E8A0h, 0E9E0h
 DW 0EB20h, 0EC60h, 0EDA0h, 0EEE0h, 0F020h, 0F160h, 0F2A0h, 0F3E0h
 DW 0F520h, 0F660h, 0F7A0h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h
 DW 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h
 DW 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h
 DW 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h
 DW 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h
 DW 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h
 DW 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h, 0F920h

;**************************************************************
; Keyboard encoding table. If word <= 7FFFh, it points to
; procedure entry. Otherwise hibyte = mask, lobyte = scan line.
;**************************************************************
; Values for keyboard make/break (scan) code:
;  01h  Esc              31h    N
;  02h  1 !              32h    M
;  03h  2 @              33h    , <              63h    F16
;  04h  3 #              34h    . >              64h    F17
;  05h  4 $              35h    / ?              65h    F18
;  06h  5 %              36h    Right Shift      66h    F19
;  07h  6 ^              37h    Grey*            67h    F20
;  08h  7 &              38h    Alt              68h    F21 (Fn) [*]
;  09h  8 *              39h    SpaceBar         69h    F22
;  0Ah  9 (              3Ah    CapsLock         6Ah    F23
;  0Bh  0 )              3Bh    F1               6Bh    F24
;  0Ch  - _              3Ch    F2               6Ch    --
;  0Dh  = +              3Dh    F3               6Dh    EraseEOF
;  0Eh  Backspace        3Eh    F4
;  0Fh  Tab              3Fh    F5               6Fh    Copy/Play
;  10h  Q                40h    F6
;  11h  W                41h    F7
;  12h  E                42h    F8               72h    CrSel
;  13h  R                43h    F9               73h    <delta> [*]
;  14h  T                44h    F10              74h    ExSel
;  15h  Y                45h    NumLock          75h    --
;  16h  U                46h    ScrollLock       76h    Clear
;  17h  I                47h    Home
;  18h  O                48h    UpArrow
;  19h  P                49h    PgUp
;  1Ah  [ {              4Ah    Grey-
;  1Bh  ] }              4Bh    LeftArrow
;  1Ch  Enter            4Ch    Keypad 5
;  1Dh  Ctrl             4Dh    RightArrow
;  1Eh  A                4Eh    Grey+
;  1Fh  S                4Fh    End
;  20h  D                50h    DownArrow
;  21h  F                51h    PgDn
;  22h  G                52h    Ins
;  23h  H                53h    Del
;  24h  J                54h    SysReq          ---non-key codes---
;  25h  K                                        00h    kbd buffer full
;  26h  L                56h    left \| (102-key)
;  27h  ; :              57h    F11              AAh    self-test complete
;  28h  ' "              58h    F12              E0h    prefix code
;  29h  ` ~                                      E1h    prefix code
;  2Ah  Left Shift       5Ah    PA1              EEh    ECHO
;  2Bh  \ |              5Bh    F13 (LWin)       F0h    prefix code (key break)
;  2Ch  Z                5Ch    F14 (RWin)       FAh    ACK
;  2Dh  X                5Dh    F15 (Menu)       FCh    diag failure (MF-kbd)
;  2Eh  C                                        FDh    diag failure (AT-kbd)
;  2Fh  V                                        FEh    RESEND
;  30h  B                                        FFh    kbd error/buffer full

Keybd_Table:
        ;00     .       ESCAPE
        DW      Key_No, Key_Esc
        ;02     1       2       3       4       5
        DW      0FE03h, 0FD03h, 0FB03h, 0F703h, 0EF03h
        ;07     6       7       8       9       0
        DW      0EF04h, 0F704h, 0FB04h, 0FD04h, 0FE04h
IF EXTRA_KEYBOARD
        ;0C     -_      +       back    tab=FIRE
        ;       $+J     $+K     ^+0     =INS
        DW      0F786h, 0FB86h, 0FE44h, 0EF08h
ELSE
        ;0C     -_      =+      back    tab=FIRE
        DW      Key_No, Key_No, Key_No, 0EF08h
ENDIF
        ;10     Q       W       E       R       T
        DW      0FE02h, 0FD02h, 0FB02h, 0F702h, 0EF02h
        ;15     Y       U       I       O       P
        DW      0EF05h, 0F705h, 0FB05h, 0FD05h, 0FE05h
        ;1A     [       ]       ENTER   CTRL
        DW      Key_No, Key_No, 0FE06h, 0FE00h
        ;1E     A       S       D       F       G
        DW      0FE01h, 0FD01h, 0FB01h, 0F701h, 0EF01h
        ;23     H       J       K       L
        DW      0EF06h, 0F706h, 0FB06h, 0FD06h
IF EXTRA_KEYBOARD
        ;27     ;       "       '       LSHIFT  \|
        ;       $+O     $+P     $+7
        DW      0FD85h, 0FE85h, 0F784h, 0FD27h, Key_No
ELSE
        ;27     ;:      '"      `       LSHIFT  \|
        DW      Key_No, Key_No, Key_No, 0FD27h, Key_No
ENDIF
        ;2C     Z       X       C       V
        DW      0FD00h, 0FB00h, 0F700h, 0EF00h
        ;30     B       N       M
        DW      0EF07h, 0F707h, 0FB07h
IF EXTRA_KEYBOARD
        ;33     ,       .       /
        ;       $+N     $+M     $+V
        DW      0F787h, 0FB87h, 0EF80h
ELSE
        ;33     ,<      .>      /?
        DW      Key_No, Key_No, Key_No
ENDIF
        ;36     RSHIFT  *(num)  Alt     SPACE   CAPSLOCK
        DW      0FD07h, Key_No, Key_No, 0FE07h, Key_no
        ;3B     F1      F2      F3      F4      F5
        DW      _Help,  _Save,  _Load,  _Reload,_Sound
        ;40     F6      F7      F8      F9      F10
        DW      Key_No, Key_No, Key_No, Key_No, _Reset
        ;45     NUMLOCK SCRLLOCK HOME   UP      PGUP
        DW      Key_No, Key_No, 0F508h, 0F708h, 0F608h
        ;4A     -(num)  LEFT    [5]     RIGHT
        DW      Key_No, 0FD08h, 0EF08h, 0FE08h
        ;4E     +(num)  END     DOWN    PGDN    INS
        DW      Key_No, 0F908h, 0FB08h, 0FA08h, 0EF08h
        ;53     DEL     SYSRQ   .       \|      F11
        DW      0EF08h, Key_No, Key_No, Key_No, Key_No
        ;58     F12     .       PA1     F13     F14
        DW      Key_No, Key_No, Key_No, Key_No, Key_No
        ;5D     F15     .       .       .       .
        DW      Key_No, Key_No, Key_No, KeyExt, Key_No
        ;62     .       F16     F17     F18
        DW      Key_No, Key_No, Key_No, Key_No
        ;66     F19     F20     F21     F22     F23
        DW      Key_No, Key_No, Key_No, Key_No, Key_No
        ;6B     F24     --      ERASE   .       COPY
        DW      Key_No, Key_No, Key_No, Key_No, Key_No
        ;70     .       .       CRSEL   .       EXSEL
        DW      Key_No, Key_No, Key_No, Key_No, Key_No
        ;75     --      CLEAR   77h     78h     79h
        DW      Key_No, Key_No, Key_No, Key_No, Key_No
        ;7A     7Ah     7Bh     7Ch     7Dh     7Eh
        DW      Key_No, Key_No, Key_No, Key_No, Key_No
        ;7F     7Fh
        DW      Key_No

Keybd_State:
        DB      0FFh, 0FFh, 0FFh, 0FFh, 0FFh, 0FFh, 0FFh, 0FFh, 0FFh
        ; (initially all keys are released)

Palette:       ;R, G, B (0..63)
        ;------ main palette 16 colors - to represent Spectrum screen
        DB      0, 0, 0         ; BLACK
        DB      0, 0, Dark      ; BLUE
        DB      Dark,0, 0       ; RED
        DB      Dark,0, Dark    ; MAGENTA
        DB      0, Dark,0       ; GREEN
        DB      0, Dark,Dark    ; CYAN
        DB      Dark,Dark,0     ; YELLOW
        DB      Dark,Dark,Dark  ; WHITE
        DB      0, 0, 0         ; BLACK (not depending on brightness)
        DB      0, 0, Lght      ; bright BLUE
        DB      Lght,0, 0       ; bright RED
        DB      Lght,0, Lght    ; bright MAGENTA
        DB      0, Lght,0       ; bright GREEN
        DB      0, Lght,Lght    ; bright CYAN
        DB      Lght,Lght,0     ; bright YELLOW
        DB      Lght,Lght,Lght  ; bright WHITE
IF HELP_ON_F1 and HELP_SHOW_KBD
        ;------ additional 49 colors to fill keyboard buttons as gradient
        DB      48, 48, 48
        DB      47, 47, 47
        DB      46, 46, 46
        DB      45, 45, 45
        DB      44, 44, 44
        DB      43, 43, 43
        DB      42, 42, 42
        DB      41, 41, 41
        DB      40, 40, 40
        DB      39, 39, 39
        DB      38, 38, 38
        DB      37, 37, 37
        DB      36, 36, 36
        DB      35, 35, 35
        DB      34, 34, 34
        DB      33, 33, 33
        DB      32, 32, 32
        DB      31, 31, 31
        DB      30, 30, 30
        DB      29, 29, 29
        DB      28, 28, 28
        DB      27, 27, 27
        DB      26, 26, 26
        DB      25, 25, 25
        DB      24, 24, 24
        DB      23, 23, 23
        DB      22, 22, 22
        DB      21, 21, 21
        DB      20, 20, 20
        DB      19, 19, 19
        DB      18, 18, 18
        DB      17, 17, 17
        DB      16, 16, 16
        DB      15, 15, 15
        DB      14, 14, 14
        DB      13, 13, 13
        DB      12, 12, 12
        DB      11, 11, 11
        DB      10, 10, 10
        DB      9, 9, 9
        DB      8, 8, 8
        DB      7, 7, 7
        DB      6, 6, 6
        DB      5, 5, 5
        DB      4, 4, 4
        DB      3, 3, 3
        DB      2, 2, 2
        DB      1, 1, 1
        DB      0, 0, 0
ENDIF

IF HELP_ON_F1
Help_Text:
 DB 0C9h,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh
 DB     0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0BBh,0
 DB 0BAh,' F1 - this Help    ',0BAh,0
IF SAVE_ON_F2
 DB 0BAh,' F2 - save Z.SNA   ',0BAh,0
ENDIF
IF LOAD_ON_F3
IF LOAD_SELECT
 DB 0BAh,' F3 - load .SNA    ',0BAh,0
ELSE
 DB 0BAh,' F3 - load Z.SNA   ',0BAh,0
ENDIF
ENDIF
IF RELOAD_ON_F4
 DB 0BAh,' F4 - reload .SNA  ',0BAh,0
ENDIF
 DB 0BAh,' F5 - Sound toggle ',0BAh,0
 DB 0BAh,' ESC- Exit         ',0BAh,0
 DB 0C8h,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh
 DB     0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0BCh,0,0
ENDIF ; HELP_ON_F1

;###############################################################################
; ZX PROCESSOR STATE
;###############################################################################
Processor_State:

TCounter:       DW 0FFFFh - 224 + 1

;************* IMPORTANT !!! ***************
; All registers for case of 48K must be located after offset 4000h !
  IF SPECTRUM_128K
  ELSE
IF HELP_SHOW_KBD
    org 4000h
ELSE
org 4000h
ENDIF
  ENDIF

; this registers structure matches those defined for SNA file format
Sna_Regs_Start:
R_I:    DB 0
R_HLalt:DW 0
R_DEalt:DW 0
R_BCalt:DW 0
Ralt_   EQU R_BCalt+1 ; points to Balt register
R_AFalt:
R_Falt: DB 0
R_Aalt: DB 0
RmainHL:DW 0
RmainDE:DW 0
RmainBC:DW 0
Rmain_  EQU RmainBC+1 ; points to B register
R_IY:   DW 0
R_IX:   DW 0
IFF1IFF2: DB 0
R_R:    DB 0
R_AF:
R_F:    DB 0
R_A:    DB 0
R_SP:   DW 0
IMmode: DB 0
BorderColor: DB 0
Sna_Regs_End:
Sna_Regs_Lengh EQU Sna_Regs_End-Sna_Regs_Start

; additional registers and ZX-processor states
RHidden_1       DB 0
RHidden:        DB 0
R_R_hi:         DB 0
;IntEnabled:     DB 0 ; =0 (int disabled) or 4 (int enabled) ; IntEnabled = IFF1IFF2
After_EI:       DB 0
IntSignal:      DB 0

CurAttrByte     DB 0 ; to store currently output attribute byte for IN FF

IF CAREFUL_INT
TCntInt:        DW -1000 ; while this value > TCounter, INT is yet possible
ENDIF

IF SPECTRUM_128K ;..........................................................128K
Sna_Ext_Start:
R_PC:           DW 0
Port_7FFD       DB 7 ; controls memory/video banks paging:
                     ; 76543210
                     ; xxLRVMMM
                     ;   |||MMM - RAM page at address 8000h-BFFFh
                     ;   ||V - Active Video Bank (0-RAM5, 1-RAM2)
                     ;   |R - ROM pare at address 0000h-3FFFh
                     ;   L - lock port 7FFD until reset (load another state)
TRDOS_ROM       DB 0 ; not used
Sna_Ext_End:
Sna_Ext_Length  EQU Sna_Ext_End - Sna_Ext_Start
ELSE             ;...........................................................48K
R_PC:           DW 0
ENDIF            ;..............................................................

Processor_State_End:
Processor_State_Len EQU Processor_State_End - Processor_State

IF LOAD_ON_F3 and LOAD_SELECT
Dir_Path:
        DB      ' :\', 77 dup(0)
Dir_Count:
        DW      0

DTA:    DB      128 dup(0)
CurFileName: DB 1Eh-15h+13 dup(0)
File2Load:   DB 128 dup(0)

Dir_Frame:
        DB      0, 0DAh, 'Select',0C4h,'file:'
        DB 0BFh
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0B3h, '            ', 0B3h
        DB      0, 0C0h, 0C4h,0C4h,0C4h,0C4h,0C4h
        DB 0C4h,0C4h,0C4h,0C4h,0C4h,0C4h,0C4h, 0D9h
        DB      0, 0
ENDIF ; LOAD_ON_F3 and LOAD_SELECT

ENDIF ; COMPRESS_TABLES

Work_Graphic_Mode endp ;#####################################

Main_Proc endp ;#############################################

TimerTick:      DB 0 ; flag "timer ticked": set it 18.2 times a second
TwoTicksOrMore: DB 0 ; flag "tick occur more than 1 time"
DelayTime:      DD 1000 ; delay counter (initial value, later will be corrected)
WaitCounter:    DW 0 ; counts 1/39 parts of screen between two timer ticks

IF HELP_ON_F1

IF HELP_SHOW_KBD or COPYRIGHT
;++++++++++++++ additional pseudo-tokens:
token_EDIT      EQU 128
token_CAPLOC    EQU 129
token_TR_VID    EQU 130
token_INVVID    EQU 131
token_left      EQU 132
token_down      EQU 133
token_up        EQU 134
token_right     EQU 135
token_GRAPH     EQU 136
token_DELETE    EQU 137
token_CAPS      EQU 138
token_SYMBOL    EQU 139
token_SHIFT     EQU 140
token_SPACE     EQU 141
token_ENTER     EQU 142
token_Copyright EQU 143
;++++++++++++++ Speccy tokens:
token_RND       EQU 165
token_INKEY$    EQU 166
token_PI        EQU 167
token_FN        EQU 168
token_POINT     EQU 169
token_SCREEN$   EQU 170
token_ATTR      EQU 171
token_AT        EQU 172
token_TAB       EQU 173
token_VAL$      EQU 174
token_CODE      EQU 175
token_VAL       EQU 176
token_LEN       EQU 177
token_SIN       EQU 178
token_COS       EQU 179
token_TAN       EQU 180
token_ASN       EQU 181
token_ACS       EQU 182
token_ATN       EQU 183
token_LN        EQU 184
token_EXP       EQU 185
token_INT       EQU 186
token_SQR       EQU 187
token_SGN       EQU 188
token_ABS       EQU 189
token_PEEK      EQU 190
token_IN        EQU 191
token_USR       EQU 192
token_STR$      EQU 193
token_CHR$      EQU 194
token_NOT       EQU 195
token_BIN       EQU 196
token_OR        EQU 197
token_AND       EQU 198
token_LessOrEq  EQU 199
token_GreatOrEq EQU 200
token_NotEq     EQU 201
token_LINE      EQU 202
token_THEN      EQU 203
token_TO        EQU 204
token_STEP      EQU 205
token_DEF_FN    EQU 206
token_CAT       EQU 207
token_FORMAT    EQU 208
token_MOVE      EQU 209
token_ERASE     EQU 210
token_OPEN_n    EQU 211
token_CLOSE_n   EQU 212
token_MERGE     EQU 213
token_VERIFY    EQU 214
token_BEEP      EQU 215
token_CIRCLE    EQU 216
token_INK       EQU 217
token_PAPER     EQU 218
token_FLASH     EQU 219
token_BRIGHT    EQU 220
token_INVERSE   EQU 221
token_OVER      EQU 222
token_OUT       EQU 223
token_LPRINT    EQU 224
token_LLIST     EQU 225
token_STOP      EQU 226
token_READ      EQU 227
token_DATA      EQU 228
token_RESTORE   EQU 229
token_NEW       EQU 230
token_BORDER    EQU 231
token_CONTINUE  EQU 232
token_DIM       EQU 233
token_REM       EQU 234
token_FOR       EQU 235
token_GO_TO     EQU 236
token_GO_SUB    EQU 237
token_INPUT     EQU 238
token_LOAD      EQU 239
token_LIST      EQU 240
token_LET       EQU 241
token_PAUSE     EQU 242
token_NEXT      EQU 243
token_POKE      EQU 244
token_PRINT     EQU 245
token_PLOT      EQU 246
token_RUN       EQU 247
token_SAVE      EQU 248
token_RANDOMIZE EQU 249
token_IF        EQU 250
token_CLS       EQU 251
token_DRAW      EQU 252
token_CLEAR     EQU 253
token_RETURN    EQU 254
token_COPY      EQU 255
Keyboard_keys:
 DB "1!",token_DEF_FN,token_EDIT,0,"2@",token_FN,token_CAPLOC,0
 DB "3#",token_LINE,token_TR_VID,0
 DB "4$",token_OPEN_n,token_INVVID,0
 DB "5%",token_CLOSE_n,token_left,0,"6&",token_MOVE,token_down,0
 DB "7'",token_ERASE,token_up,0
 DB "8(",token_POINT,token_right,0,"9)",token_CAT,token_GRAPH,0
 DB "0_",token_FORMAT,token_DELETE,0
 DB "Q<=",token_ASN,token_PLOT,token_SIN
 DB "W<>",token_ACS,token_DRAW,token_COS
 DB "E>=",token_ATN,token_REM,token_TAN
 DB "R< ",token_VERIFY,token_RUN,token_INT
 DB "T> ",token_MERGE,token_RANDOMIZE,token_RND
 DB "Y",token_AND,'[',token_RETURN,token_STR$
 DB "U",token_OR,']',token_IF,token_CHR$
 DB "I",token_AT,token_IN,token_INPUT,token_CODE
 DB "O;",token_OUT,token_POKE,token_PEEK
 DB "P",'"',127,token_PRINT,token_TAB
 DB 'A',token_STOP,'~',token_NEW,token_READ
 DB 'S',token_NOT,'|',token_SAVE,token_RESTORE
 DB 'D',token_STEP,'\',token_DIM,token_DATA
 DB 'F',token_TO,'{',token_FOR,token_SGN
 DB 'G',token_THEN,'}',token_GO_TO,token_ABS
 DB 'H^',token_CIRCLE,token_GO_SUB,token_SQR
 DB 'J-',token_VAL$,token_LOAD,token_VAL
 DB 'K+',token_SCREEN$,token_LIST,token_LEN
 DB 'L=',token_ATTR,token_LET,token_USR
 DB ' ',token_ENTER,0,' ',token_CAPS,token_SHIFT
 DB 'Z:',token_BEEP,token_COPY,token_LN
 DB 'X',96,token_INK,token_CLEAR,token_EXP
 DB 'C?',token_PAPER,token_CONTINUE,token_LPRINT
 DB 'V/',token_FLASH,token_CLS,token_LLIST
 DB 'B*',token_BRIGHT,token_BORDER,token_BIN
 DB 'N,',token_OVER,token_NEXT,token_INKEY$
 DB 'M.',token_INVERSE,token_PAUSE,token_PI
 DB ' ',token_SYMBOL,token_SHIFT,' ',token_SPACE,0
Small_Font:
 DB 3Ch,0Ah,3Ch ;A
 DB 3Eh,2Ah,14h ;B
 DB 1Ch,22h,22h ;C
 DB 3Eh,22h,1Ch ;D
 DB 3Eh,2Ah,22h ;E
 DB 3Eh,0Ah,2   ;F
 DB 1Ch,22h,32h ;G
 DB 3Eh,8,3Eh   ;H
 DB 22h,3Eh,22h ;I
 DB 2Eh,6Bh,3Ah ;J - not needed, use it for $
 DB 3Eh,8,36h   ;K
 DB 3Eh,20h,20h ;L
 DB 3Eh,1Eh,3Ch ;M
 DB 3Eh,2,3Ch   ;N
 DB 1Eh,22h,3Ch ;O
 DB 3Eh,0Ah,4   ;P
 DB 4,3Eh,4     ;Q - not needed, redefine as arow up
 DB 3Eh,0Ah,34h ;R
 DB 24h,2Ah,12h ;S
 DB 2,3Eh,2     ;T
 DB 1Eh,20h,3Eh ;U
 DB 1Eh,20h,1Eh ;V
 DB 3Eh,3Ch,1Eh ;W
 DB 36h,8,36h   ;X
 DB 6,38h,6     ;Y
 DB 32h,2Ah,26h ;Z
 DB 10h,3Eh,10h ;  arrow down
 DB 8,1Ch,3Eh   ; arrow left (head)
 DB 3Eh,1Ch,8   ; arrow rigth (head)
 DB 8,8,8       ; '-' for arrow left/right
IF COPYRIGHT
 DB 0,3Eh,22h   ; '('
 DB 22h,3Eh,0   ; ')'
 DB 0,0,0
 DB 22h,32h,2Ch ; '2'
 DB 22h,2Ah,14h ; '3'
ENDIF
Additional_Tokens:
 DB 'EDI','T'+80h
 DB 'CAPLO','K'+80h
 DB 'TR VI','D'+80h
 DB 'INVVI','D'+80h
 DB '   ','Z'+2,'Z'+84h
 DB '    ','Z'+81h
 DB '    ','Q'+80h
 DB '   ','Z'+4,'Z'+83h
 DB 'GRAP','H'+80h
 DB 'DELET','E'+80h
 DB 'CAP','S'+80h
 DB 'SYMBO','L'+80h
 DB 'SHIF','T'+80h
 DB 'SPAC','E'+80h
 DB 'ENTE','R'+80h
IF COPYRIGHT
 DB 'Z'+5,'C','Z'+6,'Z'+7,'BY','Z'+7,'VLADIMIR','Z'+7,'KLADOV'
 DB 'Z'+7,'Z'+8,'OO','Z'+9+80h
ENDIF
ENDIF ; HELP_SHOW_KBD

ENDIF ; HELP_ON_F1

Spectrum_Memory_Segment: DW 0

;###############################################################################

align 2

IF SIMPLE_DAA
DAA_End:
ELSE ; not SIMPLE_DAA
DAADefTable: ; it is used to build DAATable2 at run-time
  DB 0,09h,09h,0
  DB 0,08h,0AFh,6
  DB 40h,0AFh,09h,60h
  DB 40h,9Fh,0AFh,66h
  DB 41h,02h,09h,60h
  DB 41h,02h,0AFh,66h
  DB 2,09h,09h,0
  DB 43h,7Fh,09h,0A0h
  DB 4,09h,03h,6
  DB 44h,0AFh,03h,66h
  DB 45h,03h,03h,66h
  DB 6,08h,6Fh,0FAh
  DB 47h,6Fh,6Fh,9Ah
  DB 0FFh ; end mark
  DB 0FFh ; just to align
        ; bytes above are coding following table:
        ;---------------------------------------
        ;             #added to A
        ;       high    |
        ;       nibble  |  CF after
        ; HF    |       |
        ; | NF  |   low |  execution
        ; | | CF|   nibble |
        ; | | | |   |   |  |
        ; 0 0 0 0-9 0-9 00 0
        ; 0 0 0 0-8 a-f 06 0
        ; 0 0 0 a-f 0-9 60 1
        ; 0 0 0 9-f a-f 66 1
        ; 0 0 1 0-2 0-9 60 1
        ; 0 0 1 0-2 a-f 66 1
        ; 0 1 0 0-9 0-9 00 0
        ; 0 1 1 7-f 0-9 a0 1
        ; 1 0 0 0-9 0-3 06 0
        ; 1 0 0 a-f 0-3 66 1
        ; 1 0 1 0-3 0-3 66 1
        ; 1 1 0 0-8 6-f fa 0
        ; 1 1 1 6-f 6-f 9a 1
        ;------------------------------

DAATable1:

DAATable2       EQU DAATable1 + 256

DAA_End         EQU DAATable2 + 4096

ENDIF ; SIMPLE_DAA

IF COMPRESS_TABLES
TableOpd1       EQU DAA_End
TableOpd2       EQU TableOpd1 + 256
JumpTable       EQU TableOpd2 + 256
TableUpdFlgs    EQU JumpTable + 512
TactTable       EQU TableUpdFlgs + 512
PixelLines      EQU TableOpd1 + 7 * 256 * 5
NoAttr          EQU PixelLines + 512
AttrLines       EQU NoAttr + 112
VideoLines      EQU AttrLines + 624
Keybd_Table     EQU VideoLines + 624
Keybd_State     EQU Keybd_Table + 256
Palette         EQU Keybd_State+9
  IF HELP_ON_F1 and HELP_SHOW_KBD
  EndOfPalette EQU Palette+48+49*3
  ELSE
  EndOfPalette EQU Palette+48
  ENDIF
Help_Text       EQU EndOfPalette
  IF HELP_ON_F1
    IF SAVE_ON_F2
    EndOfF2         EQU Help_Text+44+22
    ELSE
    EndOfF2         EQU Help_Text+44
    ENDIF
    IF LOAD_ON_F3
    EndOfF3         EQU EndOfF2+22
    ELSE
    EndOfF3         EQU EndOfF2
    ENDIF
    IF RELOAD_ON_F4
    EndOfF4         EQU EndOfF3+22
    ELSE
    EndOfF4         EQU EndOfF3
    ENDIF
   DB 0BAh,' F5 - Sound toggle ',0BAh,0
   DB 0BAh,' ESC- Exit         ',0BAh,0
   DB 0C8h,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh
   DB     0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0CDh,0BCh,0,0
  EndOf_Help_Text EQU EndOfF4+67
  ELSE
  EndOf_Help_Text EQU Help_Text
  ENDIF ; HELP_ON_F1

Processor_State EQU EndOf_Help_Text+4096
TCounter        EQU Processor_State
Sna_Regs_Start  EQU TCounter+2
R_I             EQU Sna_Regs_Start
R_HLalt         EQU R_I+1
R_DEalt         EQU R_HLalt+2
R_BCalt         EQU R_DEalt+2
Ralt_   EQU R_BCalt+1 ; points to Balt register
R_AFalt         EQU R_BCalt+2
R_Falt          EQU R_AFalt
R_Aalt          EQU R_Falt+1
RmainHL         EQU R_Aalt+1
RmainDE         EQU RmainHL+2
RmainBC         EQU RmainDE+2
Rmain_  EQU RmainBC+1 ; points to B register
R_IY            EQU RmainBC+2
R_IX            EQU R_IY+2
IFF1IFF2        EQU R_IX+2
R_R             EQU IFF1IFF2+1
R_AF            EQU R_R+1
R_F             EQU R_AF
R_A             EQU R_F+1
R_SP            EQU R_A+1
IMmode          EQU R_SP+2
BorderColor     EQU IMmode+1
Sna_Regs_End    EQU BorderColor+1
Sna_Regs_Lengh EQU Sna_Regs_End-Sna_Regs_Start

; additional registers and ZX-processor states
RHidden_1       EQU Sna_Regs_End
RHidden         EQU RHidden_1+1
R_R_hi          EQU RHidden+1
After_EI        EQU R_R_hi+1
IntSignal       EQU After_EI+1
CurAttrByte     EQU IntSignal+1

  IF CAREFUL_INT
  TCntInt         EQU CurAttrByte+1
  EndOf_48KRegs   EQU TCntInt+2
  ELSE
  EndOf_48KRegs   EQU CurAttrByte+1
  ENDIF

  IF SPECTRUM_128K ;..........................................................128K
  Sna_Ext_Start   EQU EndOf_48KRegs
  R_PC            EQU Sna_Ext_Start
  Port_7FFD       EQU R_PC+2
  TRDOS_ROM       EQU Port_7FFD+1
  Sna_Ext_End     EQU TRDOS_ROM+1
  Sna_Ext_Length  EQU Sna_Ext_End - Sna_Ext_Start

  EndOf_AllRegs   EQU Sna_Ext_End
  ELSE
  R_PC            EQU EndOf_48KRegs
  EndOf_AllRegs   EQU R_PC+2
  ENDIF            ;..............................................................

  IF LOAD_ON_F3 and LOAD_SELECT
  Dir_Path        EQU EndOf_AllRegs
  Dir_Count       EQU Dir_Path+80
  DTA             EQU Dir_Count+2
  CurFileName     EQU DTA+128
  File2Load       EQU CurFileName+22
  Dir_Frame       EQU File2Load+128
  EndOfTables     EQU Dir_Frame+272
  ELSE  ; no (LOAD_ON_F3 and LOAD_SELECT)
  EndOfTables     EQU EndOf_AllRegs
  ENDIF ; LOAD_ON_F3 and LOAD_SELECT

Stack_bottom    EQU EndOfTables + 1000h
ELSE ; not COMPRESS_TABLES
Stack_bottom    EQU DAA_End + 1000h
ENDIF ; COMPRESS_TABLES

IF LOGS and LOGBUFFER
Log_buff        EQU Stack_bottom
Stk_bot         EQU Stack_bottom + LogBufSiz
ELSE
Stk_bot         EQU Stack_bottom
ENDIF
IF SPECTRUM_128K
;-------------------------------------------------------------------------------
; Bank_Access_Table used to access Spectrum memory banks by an address in BX
; depending on which part of Bank_Access_Table is selected.
;
; Virtual Spectrum memory space contains of 4 banks, each 16K bytes:
;  0000-3FFF 4000-7FFF 8000-BFFF C000-FFFF
; |---------|---------|---------|---------|
; |  page0  |  page1  |  page2  |  page3  |
; |---------|---------|---------|---------|
;  ROM0 or   RAM5      RAM2      any of
;  ROM1 here                     RAM0-RAM7
;
; Spectrum-128K memory banks map:
;  0K     16K    32K    48K    64K    80K    96K    112K   128K   144K   160K   176K
; |------|------|------|------|------|------|------|------|------|------|------|
; | RAM0 | RAM1 | RAM2 | RAM3 | RAM4 | RAM5 | RAM6 | RAM7 | ROM0 | ROM1 |DUMMY |
; |------|------|------|------|------|------|------|------|------|------|------|
;  000    400    800    C00    1000   1400   1800   1C00   2000   2400   2800
;
; Bank_Access_Table contains of 2 x 2 x 8 = 32 maps
;                               |   |   8 different RAMn banks at page 3
;                               |   2 ROMn banks at page 0
;                               2 sets, 1st for read, 2nd for write
; (a set used for write maps page0 to DUMMY bank, so it is not necessary to
; compare an address with magic value 4000h before writing a byte to memory)
;
; Each map is a table of 256 words, which actually are segments calculated
; so that would such segment used with correspondent address, it provides
; access to a desired byte in Spectrum memory bank.
; So, finally size of Bank_Access_Table is calculated as 32 x 512 = 16384 bytes
;-------------------------------------------------------------------------------
Bank_Access_Table EQU Stk_bot
Stack_top       EQU Bank_Access_Table + 16384 + 800h
ELSE
Stack_top       EQU Stk_bot + 800h
ENDIF

        end     start
;<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
;<<< EOF >>>

