unit _GridScroll;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, Grids, StdCtrls, Menus,INIFiles,_CSkin;

{$WARN UNIT_PLATFORM OFF}
{$WARN SYMBOL_PLATFORM OFF}

const GS_PartsSize=14;

procedure GridScroll_LoadSkin(var CSkin:TCSkin);

type
  TGridScroll=class
  private
    { Private 錾 }
    LinkGrid:TStringGrid;
    LinkImg:TImage;
    PopupMenu:TPopupMenu;
    BGDrawedBM:TBitmap;
    BtnTimer,BarTimer:TTimer;
    SBHeight:integer;
    SBFixed,SBPageMax,SBPageSize,SBPagePos:integer;
    mx,my:integer;
    DragFlag:boolean;
    DragAdjustY:integer;
    crBodyHeight,crHeight,crY:integer; // CalcCurrent=True only.
    LinkGridOnTopLeftChanged:TNotifyEvent;
    procedure onPopupClick(Sender: TObject);
    procedure onBtnTimer(Sender: TObject);
    procedure onBarTimer(Sender: TObject);
    procedure onMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure onMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure onMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure onTopLeftChanged(Sender: TObject);
    procedure SetTopRow(row:integer);
    function CalcCurrent:boolean;
    function BarAtMouse(y:integer):integer;
    procedure ShowPopupMenu;
    procedure RedrawBar;
  public
    { Public 錾 }
    constructor Create;
    destructor Destroy; override;
    procedure LinkComponent(_LinkGrid:TStringGrid;_LinkImg:TImage);
    procedure UnlinkComponent;
    procedure ResetPosition;
    procedure RedrawAll;
  end;

implementation

uses _PicTools;

const GS_DCT_nil=0;
const GS_DCT_MoveTop=1;
const GS_DCT_MoveBottom=2;
const GS_DCT_MovePageUp=3;
const GS_DCT_MovePageDown=4;
const GS_DCT_MoveScrollUp=5;
const GS_DCT_MoveScrollDown=6;

var
  MemSync:TMultiReadExclusiveWriteSynchronizer;
  BGBM,GripTopBM,GripBGBM,GripBottomBM,UpBtnBM,DownBtnBM:TBitmap;
  BarAlpha:byte;

procedure GridScroll_LoadSkin(var CSkin:TCSkin);
  procedure LoadBM(ID:string;var bm:TBitmap;y:integer);
  begin
    if bm<>nil then bm.Free;
    bm:=TBitmap.Create;
    CSkin.LoadBitmap(ID,bm,GS_PartsSize,y,pf24bit);
  end;
begin
  MemSync.BeginWrite;

  LoadBM('SB_BG.bmp',BGBM,-1);
  LoadBM('SB_GripTop.bmp',GripTopBM,-1);
  LoadBM('SB_GripBG.bmp',GripBGBM,-1);
  LoadBM('SB_GripBottom.bmp',GripBottomBM,-1);
  LoadBM('SB_UpBtn.bmp',UpBtnBM,GS_PartsSize);
  LoadBM('SB_DownBtn.bmp',DownBtnBM,GS_PartsSize);

  BarAlpha:=CSkin.ScrollBarGripAlpha;

  MemSync.EndWrite;
end;

constructor TGridScroll.Create;
begin
  inherited Create;

  MemSync.BeginWrite;

  LinkGrid:=nil;
  LinkImg:=nil;

  PopupMenu:=nil;

  BGDrawedBM:=nil;

  BtnTimer:=TTimer.Create(nil);
  BtnTimer.Enabled:=False;
  BtnTimer.OnTimer:=onBtnTimer;

  BarTimer:=TTimer.Create(nil);
  BarTimer.Enabled:=False;
  BarTimer.OnTimer:=onBarTimer;

  SBHeight:=0;
  SBFixed:=0;
  SBPageMax:=0;
  SBPageSize:=0;
  SBPagePos:=0;

  mx:=0;
  my:=0;

  DragFlag:=False;
  DragAdjustY:=0;

  crBodyHeight:=0;
  crHeight:=0;
  crY:=0;

  MemSync.EndWrite;
end;

destructor TGridScroll.Destroy;
begin
  MemSync.BeginWrite;

  UnlinkComponent;

  if PopupMenu<>nil then PopupMenu.Free;

  if BGDrawedBM<>nil then BGDrawedBM.Free;

  BtnTimer.Free;
  BarTimer.Free;

  MemSync.EndWrite;

  inherited Destroy;
end;

// -------------------------
// ---- private
// -------------------------

procedure TGridScroll.onPopupClick(Sender: TObject);
begin
  case TMenuItem(Sender).Tag of
    GS_DCT_nil: begin end;
    GS_DCT_MoveTop: SetTopRow(0);
    GS_DCT_MoveBottom: SetTopRow(SBPageMax);
    GS_DCT_MovePageUp: SetTopRow(SBPagePos-SBPageSize);
    GS_DCT_MovePageDown: SetTopRow(SBPagePos+SBPageSize);
    GS_DCT_MoveScrollUp: SetTopRow(SBPagePos-1);
    GS_DCT_MoveScrollDown: SetTopRow(SBPagePos+1);
  end;
end;

procedure TGridScroll.onBtnTimer(Sender: TObject);
begin
  MemSync.BeginWrite;
  BtnTimer.Interval:=32;
  if my<(SBHeight div 2) then SetTopRow(LinkGrid.TopRow-1);
  if my>=(SBHeight div 2) then SetTopRow(LinkGrid.TopRow+1);
  MemSync.EndWrite;
end;

procedure TGridScroll.onBarTimer(Sender: TObject);
var
  v:integer;
begin
  MemSync.BeginWrite;
  BarTimer.Interval:=32;
  MemSync.EndWrite;

  if CalcCurrent=False then exit;

  MemSync.BeginWrite;
  v:=BarAtMouse(my-GS_PartsSize);
  case v of
     0: begin end;
    -1: SetTopRow(LinkGrid.TopRow-SBPageSize);
     1: SetTopRow(LinkGrid.TopRow+SBPageSize);
  end;
  MemSync.EndWrite;
end;

procedure TGridScroll.onMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  v:integer;
begin
  if Button=mbRight then begin
    ShowPopupMenu;
    exit;
  end;

  MemSync.BeginWrite;
  mx:=x;
  my:=y;
  MemSync.EndWrite;

  if SBHeight=0 then exit;

  MemSync.BeginWrite;
  try
    if (y<GS_PartsSize) or (y>=(SBHeight-GS_PartsSize)) then begin
      if y<GS_PartsSize then begin
        SetTopRow(LinkGrid.TopRow-1);
        end else begin
        SetTopRow(LinkGrid.TopRow+1);
      end;
      BtnTimer.Interval:=320;
      BtnTimer.Enabled:=True;
      exit;
    end;
    finally begin
      MemSync.EndWrite;
    end;
  end;

  if CalcCurrent=False then exit;

  MemSync.BeginWrite;

  v:=BarAtMouse(y-GS_PartsSize);
  case v of
     0: begin
      DragFlag:=True;
      DragAdjustY:=(y-GS_PartsSize)-crY-(1*crHeight div (SBPageMax-SBFixed));
    end;
    -1: SetTopRow(LinkGrid.TopRow-SBPageSize);
     1: SetTopRow(LinkGrid.TopRow+SBPageSize);
  end;

  if v<>0 then begin
    BarTimer.Interval:=320;
    BarTimer.Enabled:=True;
  end;

  MemSync.EndWrite;
end;

procedure TGridScroll.onMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
begin
  MemSync.BeginWrite;
  mx:=x;
  my:=y;
  MemSync.EndWrite;

  if DragFlag=False then exit;

  if CalcCurrent=False then exit;

  MemSync.BeginWrite;
  y:=y-GS_PartsSize;
  SetTopRow((y-DragAdjustY)*(SBPageMax-SBFixed) div crBodyHeight);
  MemSync.EndWrite;
end;

procedure TGridScroll.onMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  MemSync.BeginWrite;
  DragFlag:=False;
  BtnTimer.Enabled:=False;
  BarTimer.Enabled:=False;
  MemSync.EndWrite;
end;

procedure TGridScroll.onTopLeftChanged(Sender: TObject);
begin
  if LinkGrid=nil then exit;

  MemSync.BeginWrite;
  SBPagePos:=LinkGrid.TopRow;
  MemSync.EndWrite;

  RedrawBar;

  if addr(LinkGridOnTopLeftChanged)<>nil then LinkGridOnTopLeftChanged(Sender);
end;

procedure TGridScroll.SetTopRow(row:integer);
begin
  if LinkGrid=nil then exit;

  MemSync.BeginWrite;
  if row<SBFixed then row:=SBFixed;
  if (LinkGrid.RowCount-LinkGrid.VisibleRowCount)<row then row:=LinkGrid.RowCount-LinkGrid.VisibleRowCount;

  if LinkGrid.TopRow<>row then LinkGrid.TopRow:=row;

  MemSync.EndWrite;
end;

function TGridScroll.CalcCurrent:boolean;
begin
  Result:=True;
  MemSync.BeginRead;
  if (SBHeight=0) or (SBPageMax<=SBFixed) or (SBPageSize=0) then Result:=False;
  if (SBPagePos<SBFixed) or (SBPageMax<=SBPagePos) then Result:=False;
  MemSync.EndRead;
  if Result=False then exit;

  MemSync.BeginWrite;

  crBodyHeight:=SBHeight-(GS_PartsSize*2);
  if crBodyHeight<16 then begin
    Result:=False;
    crY:=0;
    crHeight:=0;
    end else begin
    Result:=True;
    crY:=(SBPagePos-SBFixed)*crBodyHeight div (SBPageMax-SBFixed);
    crHeight:=SBPageSize*crBodyHeight div (SBPageMax-SBFixed);
    if crHeight<8 then crHeight:=8;
    if (crY+crHeight)>crBodyHeight then crY:=crBodyHeight-crHeight;
    if crY<0 then crY:=0;
  end;

  MemSync.EndWrite;
end;

function TGridScroll.BarAtMouse(y:integer):integer;
begin
  if crHeight=0 then begin
    Result:=0;
    exit;
  end;

  MemSync.BeginRead;
  Result:=0;
  if y<crY then Result:=-1;
  if (crY+crHeight+1)<y then Result:=1;
  MemSync.EndRead;
end;

procedure TGridScroll.ShowPopupMenu;
var
  pos:TPoint;
  procedure AddItem(_Caption:string;_DCT:integer);
  var
    Item:TMenuItem;
  begin
    Item:=TMenuItem.Create(PopupMenu);
    Item.Enabled:=True;
    Item.Caption:=_Caption;
    Item.Tag:=_DCT;
    if _DCT<>GS_DCT_nil then begin
      Item.OnClick:=onPopupClick;
    end;
    PopupMenu.Items.Add(Item);
  end;
begin
  if PopupMenu<>nil then begin
    PopupMenu.Free;
    PopupMenu:=nil;
  end;
  PopupMenu:=TPopupMenu.Create(nil);
  PopupMenu.AutoHotkeys:=maManual;

  AddItem('Top',GS_DCT_MoveTop);
  AddItem('Bottom',GS_DCT_MoveBottom);
  AddItem('-',GS_DCT_nil);

  AddItem('Page Up',GS_DCT_MovePageUp);
  AddItem('Page Down',GS_DCT_MovePageDown);
  AddItem('-',GS_DCT_nil);

  AddItem('Scroll Up',GS_DCT_MoveScrollUp);
  AddItem('Scroll Down',GS_DCT_MoveScrollDown);
  AddItem('-',GS_DCT_nil);

  GetCursorPos(pos);
  PopupMenu.Popup(pos.X,pos.Y);
end;

procedure TGridScroll.RedrawBar;
var
  y:integer;
  ajy:integer;
  AlphaArray,InvAlphaArray:array[$00..$ff] of byte;
  AlphaCnt:integer;
  procedure DrawLine(var BM:TBitmap;DstY,SrcY:integer);
  var
    DstPB,SrcPB:PByte;
    w,x:integer;
  begin
    if (DstY<0) or (LinkImg.Picture.Bitmap.Height<=DstY) then exit;
    if (SrcY<0) or (BM.Height<=SrcY) then exit;

    DstPB:=LinkImg.Picture.Bitmap.ScanLine[DstY];
    SrcPB:=BM.ScanLine[SrcY];

    w:=GS_PartsSize*3;
    if BarAlpha=$ff then begin
      MoveMemory(DstPB,SrcPB,w);
      end else begin
      for x:=0 to w-1 do begin
        DstPB^:=byte(InvAlphaArray[DstPB^]+AlphaArray[SrcPB^]);
        inc(DstPB);
        inc(SrcPB);
      end;
    end;
  end;
begin
  if LinkImg=nil then exit;
  if BGDrawedBM=nil then exit;

  if BGBM=nil then exit;
  if GripTopBM=nil then exit;
  if GripBGBM=nil then exit;
  if GripBottomBM=nil then exit;
  if UpBtnBM=nil then exit;
  if DownBtnBM=nil then exit;

  MemSync.BeginWrite;

  for AlphaCnt:=$00 to $ff do begin
    AlphaArray[AlphaCnt]:=byte(AlphaCnt*integer(BarAlpha) div $100);
    InvAlphaArray[AlphaCnt]:=byte(AlphaCnt*integer($ff-BarAlpha) div $100);
  end;

  BitBlt(LinkImg.Canvas.Handle,0,0,BGDrawedBM.Width,BGDrawedBM.Height,BGDrawedBM.Canvas.Handle,0,0,SRCCOPY);

  if CalcCurrent=True then begin
    for y:=0 to crHeight-1 do begin
      ajy:=GS_PartsSize+crY+y;
      if y<=(GripTopBM.Height-1) then begin
        DrawLine(GripTopBM,ajy,y);
        end else begin
        if (crHeight-GripBottomBM.Height)<=y then begin
          DrawLine(GripBottomBM,ajy,y-(crHeight-GripBottomBM.Height));
          end else begin
          DrawLine(GripBGBM,ajy,(y-GripTopBM.Height) mod GripBGBM.Height);
        end;
      end;
    end;
  end;

  MemSync.EndWrite;

  LinkImg.Refresh;
end;

// -----------------------------
// ----- public
// -----------------------------

procedure TGridScroll.LinkComponent(_LinkGrid:TStringGrid;_LinkImg:TImage);
begin
  UnlinkComponent;

  MemSync.BeginWrite;

  LinkGrid:=_LinkGrid;
  if LinkGrid<>nil then begin
    LinkGridOnTopLeftChanged:=LinkGrid.OnTopLeftChanged;
    LinkGrid.OnTopLeftChanged:=onTopLeftChanged;
  end;

  LinkImg:=_LinkImg;
  if LinkImg<>nil then begin
//    LinkImg.OnPaint:=onPaint;
    LinkImg.OnMouseDown:=onMouseDown;
    LinkImg.OnMouseMove:=onMouseMove;
    LinkImg.OnMouseUp:=onMouseUp;
  end;

  ResetPosition;

  MemSync.EndWrite;
end;

procedure TGridScroll.UnlinkComponent;
begin
  MemSync.BeginWrite;

  if LinkGrid<>nil then begin
    LinkGrid.OnTopLeftChanged:=LinkGridOnTopLeftChanged;
    LinkGridOnTopLeftChanged:=nil;
  end;
  LinkGrid:=nil;

  if LinkImg<>nil then begin
//    LinkImg.OnPaint:=nil;
    LinkImg.OnMouseDown:=nil;
    LinkImg.OnMouseMove:=nil;
    LinkImg.OnMouseUp:=nil;
  end;
  LinkImg:=nil;

  MemSync.EndWrite;
end;

procedure TGridScroll.ResetPosition;
begin
  if BGBM=nil then exit;
  if GripTopBM=nil then exit;
  if GripBGBM=nil then exit;
  if GripBottomBM=nil then exit;
  if UpBtnBM=nil then exit;
  if DownBtnBM=nil then exit;
  if LinkGrid=nil then exit;
  if LinkImg=nil then exit;

  MemSync.BeginWrite;

  if LinkGrid.ScrollBars=ssBoth then LinkGrid.ScrollBars:=ssHorizontal;
  if LinkGrid.ScrollBars=ssVertical then LinkGrid.ScrollBars:=ssNone;

  SBHeight:=LinkGrid.Height;
  SBFixed:=LinkGrid.FixedRows;
  SBPageMax:=LinkGrid.RowCount;
  SBPageSize:=LinkGrid.VisibleRowCount;
  SBPagePos:=LinkGrid.TopRow;

  LinkImg.Top:=LinkGrid.Top;
  LinkImg.Left:=LinkGrid.Left+LinkGrid.Width;
  LinkImg.Width:=GS_PartsSize;
  LinkImg.Height:=SBHeight;
  MakeBlankImg(LinkImg,pf24bit);

  MemSync.EndWrite;

  RedrawAll;
end;

procedure TGridScroll.RedrawAll;
var
  y:integer;
begin
  MemSync.BeginWrite;

  if BGDrawedBM<>nil then BGDrawedBM.Free;
  BGDrawedBM:=TBitmap.Create;
  MakeBlankBM(BGDrawedBM,LinkImg.Width,LinkImg.Height,pf24bit);

  for y:=0 to (SBHeight div BGBM.Height) do begin
    BitBlt(BGDrawedBM.Canvas.Handle,0,y*BGBM.Height,GS_PartsSize,BGBM.Height,BGBM.Canvas.Handle,0,0,SRCCOPY);
  end;
  if SBHeight>(GS_PartsSize*2) then begin
    BitBlt(BGDrawedBM.Canvas.Handle,0,0,GS_PartsSize,GS_PartsSize,UpBtnBM.Canvas.Handle,0,0,SRCCOPY);
    BitBlt(BGDrawedBM.Canvas.Handle,0,SBHeight-GS_PartsSize,GS_PartsSize,GS_PartsSize,DownBtnBM.Canvas.Handle,0,0,SRCCOPY);
  end;

  MemSync.EndWrite;

  RedrawBar;
end;

initialization
  MemSync:=TMultiReadExclusiveWriteSynchronizer.Create;

  MemSync.BeginWrite;
  BGBM:=nil;
  GripTopBM:=nil;
  GripBGBM:=nil;
  GripBottomBM:=nil;
  UpBtnBM:=nil;
  DownBtnBM:=nil;
  BarAlpha:=$ff;
  MemSync.EndWrite;

finalization
  MemSync.BeginWrite;
  if BGBM<>nil then BGBM.Free;
  if GripTopBM<>nil then GripTopBM.Free;
  if GripBGBM<>nil then GripBGBM.Free;
  if GripBottomBM<>nil then GripBottomBM.Free;
  if UpBtnBM<>nil then UpBtnBM.Free;
  if DownBtnBM<>nil then DownBtnBM.Free;
  BarAlpha:=$ff;
  MemSync.EndWrite;

  MemSync.Free;

end.
