/*
    ARM9Core
    Copyright (C) 2007 Alberto Huerta Aranda,
	Sergio Hidalgo Serrano, Daniel Saudo Vacas

    This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 3 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
	License along with this library. If not, see <http://www.gnu.org/licenses/>

*/


/**************************************************************************************

  Archivo: ARM9RutinasARM.cpp

  Descripcion: Codigo de las rutinas del modo ARM

**************************************************************************************/

#include "ARM9RutinasARM.h"
#include "ARM9Core.h"
#include <stdio.h>

ARM9RutinasARM::ARM9RutinasARM () {

	tablaGenerada = 0;

	int aux[] = {0x0F0F, 0xF0F0, 0x3333, 0xCCCC, 0x00FF, 0xFF00,
				0x5555, 0xAAAA, 0x3030, 0xCFCF, 0xAA55, 0x55AA,
				0xA050, 0x5FAF, 0xFFFF, 0xFFFF};


	//Genera la tabla precalculada de condiciones
	for (int i = 0; i < 16; i++) {

		tablaCond[i][0] = (aux[i] & 0x8000) != 0;
		tablaCond[i][1] = (aux[i] & 0x4000) != 0;
		tablaCond[i][2] = (aux[i] & 0x2000) != 0;
		tablaCond[i][3] = (aux[i] & 0x1000) != 0;

		tablaCond[i][4] = (aux[i] & 0x0800) != 0;
		tablaCond[i][5] = (aux[i] & 0x0400) != 0;
		tablaCond[i][6] = (aux[i] & 0x0200) != 0;
		tablaCond[i][7] = (aux[i] & 0x0100) != 0;

		tablaCond[i][8] = (aux[i] & 0x0080) != 0;
		tablaCond[i][9] = (aux[i] & 0x0040) != 0;
		tablaCond[i][10] = (aux[i] & 0x0020) != 0;
		tablaCond[i][11] = (aux[i] & 0x0010) != 0;

		tablaCond[i][12] = (aux[i] & 0x0008) != 0;
		tablaCond[i][13] = (aux[i] & 0x0004) != 0;
		tablaCond[i][14] = (aux[i] & 0x0002) != 0;
		tablaCond[i][15] = (aux[i] & 0x0001) != 0;
		
	}

}


int ARM9RutinasARM::run (ARM9Core *core, ARM9Contexto *contexto, int nCiclos) {

	ARM9Instruccion32 instruccion;
	void *dir;

	this->core = core;
	this->contexto = contexto;

	int rn, rd, rs, rm, rot, operando1, operando2, acumulador, acumulador2, resultado;
	int signo1, signo2;
	int shift, tipoShift;
	int l, s, c, shifterCarryOut, a, v, esDPI, esLSRO, dirOffset;
	uint32 dirEfectiva, dirEfectiva2;
	int u, b, w, i, j, esSignedHalfword, listaRegistros, lsmUserMode, lsmCPSR;
	int resultado2, esMISR, r, mascara;
	int64 var1_64, var2_64; uint64 uvar1_64, uvar2_64;

/*-------------------------------------------------------------------------*/
/*  Tabla de saltos
/*-------------------------------------------------------------------------*/
#include "ARM9TablaARM.h"

/*-------------------------------------------------------------------------*/
/*  Lectura de la siguiente instruccin
/*-------------------------------------------------------------------------*/

LecturaInstruccion:

	//Antes de leer, comprobamos el numero de ciclos restantes
	if ((nCiclos <= 0) || (core->getExcepcionActiva() != EXCEP_NONE))
		return nCiclos;

	//Lee la siguiente instruccion
	instruccion.u_word = core->readInstruccion (contexto->PC->u_word);
	contexto->PC->u_word += 0x4; //+4 bytes

	if (core->getExcepcionActiva() != EXCEP_NONE)
		return nCiclos;

	//Obtenemos la direccion de la subrutina a partir del codigo de la
	//tabla de saltos (bits 27-20)
	dir = tablaSaltos[instruccion.comunes.codigoTabla][instruccion.comunes.codigoTabla2];
	
	//Salta a la subrutina
	__asm jmp dir	
	


/*-------------------------------------------------------------------------*/
/*	Rutina ADD (DP)
/*-------------------------------------------------------------------------*/
RUT_add:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT
	}
	
	//ADD	
	__asm {
		add		eax, ebx		;suma
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	GETFLAGS

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina ADC (DP)
/*-------------------------------------------------------------------------*/
RUT_adc:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	c = contexto->flags.c;

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT
	}
	
	//ADC
	__asm {
		add		eax, ebx		;suma
		add		eax, c
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	GETFLAGS

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina SUB (DP)
/*-------------------------------------------------------------------------*/
RUT_sub:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT
	}

	//SUB	
	__asm {
		mov		operando2, ebx	;preserva para el flag de carry
		neg		ebx
		add		eax, ebx		;resta
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	signo2 = signo2 ^ 0x80000000;
	GETFLAGS_SUB

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina SBC (DP)
/*-------------------------------------------------------------------------*/
RUT_sbc:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	c = contexto->flags.c;

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT
	}
	
	//SBC
	__asm {
		add		eax, c
		sub		eax, 1
		mov		operando1, eax	;preserva para el flag de carry
		mov		operando2, ebx
		neg		ebx
		add		eax, ebx		;resta
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	signo2 = signo2 ^ 0x80000000;
	GETFLAGS_SUB

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina RSB (DP)
/*-------------------------------------------------------------------------*/
RUT_rsb:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT
	}

	//RSB	
	__asm {
		mov		operando1, ebx	;preserva para el flag de carry
		mov		operando2, eax
		neg		eax
		add		ebx, eax		;resta
		pushf					;guarda los flags en la pila
		mov		resultado, ebx	;guarda el resultado
	}

	//Guarda los flags
	signo1 = signo1 ^ 0x80000000;
	GETFLAGS_SUB

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;




/*-------------------------------------------------------------------------*/
/*	Rutina RSC (DP)
/*-------------------------------------------------------------------------*/
RUT_rsc:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	c = contexto->flags.c;

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT
	}
	
	//RSC
	__asm {
		add		ebx, c
		sub		ebx, 1
		mov		operando2, eax	;preserva para el flag de carry
		mov		operando1, ebx
		neg		eax
		add		ebx, eax		;resta
		pushf					;guarda los flags en la pila
		mov		resultado, ebx	;guarda el resultado
	}

	//Guarda los flags
	signo1 = signo1 ^ 0x80000000;
	GETFLAGS_SUB

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina MOV (DP)
/*-------------------------------------------------------------------------*/
RUT_mov:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT_CARRY
	}
	
	//MOV	
	__asm {
		or		ebx, 0			;OR 0 para obtener los flags (el x86 no lo hace con move)
		pushf					;guarda los flags en la pila
		mov		resultado, ebx	;guarda el resultado
	}

	//Guarda los flags
	GETFLAGS_SHIFTER

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina MVN (DP)
/*-------------------------------------------------------------------------*/
RUT_mvn:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT_CARRY
	}
	
	//MVN	
	__asm {
		mov		eax, 0FFFFFFFFh
		xor		eax, ebx		;XOR con eax para obtener los flags (el x86 no lo hace con move)
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	GETFLAGS_SHIFTER

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina AND (DP)
/*-------------------------------------------------------------------------*/
RUT_and:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT_CARRY
	}
	
	//AND	
	__asm {
		and		eax, ebx		;and
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	GETFLAGS_SHIFTER

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina EOR (DP)
/*-------------------------------------------------------------------------*/
RUT_eor:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT_CARRY
	}
	
	//EOR	
	__asm {
		xor		eax, ebx		;or exclusiva
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	GETFLAGS_SHIFTER

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina ORR (DP)
/*-------------------------------------------------------------------------*/
RUT_orr:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT_CARRY
	}
	
	//ORR	
	__asm {
		or		eax, ebx		;or
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	GETFLAGS_SHIFTER

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina BIC (DP)
/*-------------------------------------------------------------------------*/
RUT_bic:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT_CARRY
	}
	
	//BIC	
	__asm {
		not		ebx				;NOT Op2
		and		eax, ebx		;Rn and NOT Op2
		pushf					;guarda los flags en la pila
		mov		resultado, eax	;guarda el resultado
	}

	//Guarda los flags
	GETFLAGS_SHIFTER

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina TST (DP)
/*-------------------------------------------------------------------------*/
RUT_tst:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT_CARRY
	}
	
	//TST	
	__asm {
		and		eax, ebx		;and
		pushf					;guarda los flags en la pila
	}

	//Guarda los flags
	GETFLAGS_SHIFTER

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina TEQ (DP)
/*-------------------------------------------------------------------------*/
RUT_teq:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT_CARRY
	}
	
	//TEQ	
	__asm {
		xor		eax, ebx		;or exlusiva
		pushf					;guarda los flags en la pila
	}

	//Guarda los flags
	GETFLAGS_SHIFTER

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina CMP (DP)
/*-------------------------------------------------------------------------*/
RUT_cmp:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT
	}

	//CMP	
	__asm {
		mov		operando2, ebx	;preserva para el flag de carry
		neg		ebx		
		add		eax, ebx		;resta
		pushf					;guarda los flags en la pila
	}

	//Guarda los flags
	signo2 = signo2 ^ 0x80000000;
	GETFLAGS_SUB

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina CMN (DP)
/*-------------------------------------------------------------------------*/
RUT_cmn:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_DP

	if (esDPI) {
		//Rota (expande) el operando inmediato
		ROTATE_DPI
	}else{
		//Desplaza el operando 2
		SHIFT
	}
	
	//CMN	
	__asm {
		add		eax, ebx		;suma
		pushf					;guarda los flags en la pila
	}

	//Guarda los flags
	GETFLAGS

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutinas Branch y Branch with link (BBL)
/*-------------------------------------------------------------------------*/
RUT_bbl:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	//Link
	l = instruccion.BBL.L;

	//Link: Guarda en el registro link la direccion de la instruccion siguiente
	if (l)
		contexto->bancoRegistros[14].u_word = contexto->PC->u_word;

	//El valor del PC al ejecutarse la instruccion de salto es la direccion
	//de la instruccion siguiente a la siguiente (por el pipeline segmentado)
	//Para emularlo sumamos 4 al calculo de direccion final del salto
	contexto->PC->u_word = contexto->PC->u_word + 4 + (instruccion.BBL.offset<<2);

	nCiclos-=3;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina Branch and Exchange (BX)
/*-------------------------------------------------------------------------*/
RUT_bx:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	rm = instruccion.BX.Rm;

	contexto->PC->u_word = contexto->bancoRegistros[rm].u_word;

	nCiclos-=3;
	goto LecturaInstruccion;




/*-------------------------------------------------------------------------*/
/*	Rutina LDR pre-index (Load single word and unsigned byte)
/*-------------------------------------------------------------------------*/
RUT_ldr_pre:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS_SHIFT

	//Calcular la dir. efectiva (y guardarla, si eso)
	PREINDEX
	
	if (b) {
		resultado = unsigned int (core->readMemoriaByte(dirEfectiva));
	} else {
		dirEfectiva2 = dirEfectiva & 0xFFFFFFFC;
		resultado = core->readMemoriaWord(dirEfectiva2);

		//Ajusta la palabra cargada
		rot = ((dirEfectiva2>>2) - (dirEfectiva>>2));
		__asm {
			mov	eax, resultado
			mov	ecx, rot
			ror	eax, cl
			mov	resultado, eax
		}
	} 

	CHK_ABORT
	contexto->bancoRegistros[rd].u_word = resultado;
	
	
	nCiclos-=2;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina STR pre-index (Store single word and unsigned byte)
/*-------------------------------------------------------------------------*/
RUT_str_pre:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS_SHIFT

	//Calcular la dir. efectiva (y guardarla, si eso)
	PREINDEX
	
	if (b) {
		core->writeMemoriaByte(contexto->bancoRegistros[rd].u_byte, dirEfectiva);
	} else {
		dirEfectiva2 = dirEfectiva & 0xFFFFFFFC;
		core->writeMemoriaWord(contexto->bancoRegistros[rd].u_word, dirEfectiva2);
	} 
	
	
	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina LDR post-index (Load single word and unsigned byte)
/*-------------------------------------------------------------------------*/
RUT_ldr_post:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS_SHIFT

	
	if (b) {
		resultado = unsigned int (core->readMemoriaByte(operando1));
	} else {
		dirEfectiva = operando1 & 0xFFFFFFFC;
		resultado = core->readMemoriaWord(dirEfectiva);

		//Ajusta la palabra cargada
		rot = ((dirEfectiva>>2) - (operando1>>2));
		__asm {
			mov	eax, resultado
			mov	ecx, rot
			ror	eax, cl
			mov	resultado, eax
		}
	} 

	
	//Calcular la dir. efectiva (y guardarla)
	POSTINDEX

	CHK_ABORT
	contexto->bancoRegistros[rd].u_word = resultado;
	
	nCiclos-=2;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina STR post-index (Store single word and unsigned byte)
/*-------------------------------------------------------------------------*/
RUT_str_post:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS_SHIFT

	
	if (b) {
		core->writeMemoriaByte(contexto->bancoRegistros[rd].u_byte, operando1);
	} else {
		dirEfectiva = operando1 & 0xFFFFFFFC;
		core->writeMemoriaWord(contexto->bancoRegistros[rd].u_word, dirEfectiva);
	} 

	//Calcular la dir. efectiva (y guardarla)
	POSTINDEX
	
	
	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina LDR HW/B pre-index (Load half-word / signed byte)
/*-------------------------------------------------------------------------*/
RUT_ldr_hw_pre:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS_EXTRAS

	//Calcular la dir. efectiva (y guardarla, si eso)
	PREINDEX
	
	switch (esSignedHalfword) {
	case 2:
		resultado = (int8)(core->readMemoriaByte(dirEfectiva));
		break;
	case 3:
		resultado = (int16)(core->readMemoriaHalfword(dirEfectiva));
		break;
	default:
		resultado = unsigned int (core->readMemoriaHalfword(dirEfectiva));
	}

	CHK_ABORT
	contexto->bancoRegistros[rd].u_word = resultado;
	
	nCiclos-=2;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina STR HW pre-index (Store half-word / signed byte)
/*-------------------------------------------------------------------------*/
RUT_str_hw_pre:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS_EXTRAS

	//Calcular la dir. efectiva (y guardarla, si eso)
	PREINDEX
	
	core->writeMemoriaHalfword(contexto->bancoRegistros[rd].u_halfword, dirEfectiva);
	
	
	nCiclos--;
	goto LecturaInstruccion;




/*-------------------------------------------------------------------------*/
/*	Rutina LDR HW/B post-index (Load half-word / signed byte)
/*-------------------------------------------------------------------------*/
RUT_ldr_hw_post:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS_EXTRAS
	
	switch (esSignedHalfword) {
	case 2:
		resultado = (int8)(core->readMemoriaByte(operando1));
		break;
	case 3:
		resultado = (int16)(core->readMemoriaHalfword(operando1));
		break;
	default:
		resultado = unsigned int (core->readMemoriaHalfword(operando1));
	}

	//Calcular la dir. efectiva (y guardarla)
	POSTINDEX

	CHK_ABORT
	contexto->bancoRegistros[rd].u_word = resultado;
	
	nCiclos-=2;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina STR HW post-index (Store half-word / signed byte)
/*-------------------------------------------------------------------------*/
RUT_str_hw_post:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS_EXTRAS

	core->writeMemoriaHalfword(contexto->bancoRegistros[rd].u_halfword, operando1);

	//Calcular la dir. efectiva (y guardarla)
	POSTINDEX
	
	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina LDM pre-index (Load multiple)
/*-------------------------------------------------------------------------*/
RUT_ldm_pre:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LSM

	//Para cada registro, lo transfiere si su bit esta a 1
	for ((u==1)? i = 0 : i=15;
		 (u==1)? (i<16): (i>=0);
		 (u==1)?  i++  : i--) {

		if (listaRegistros & (0x1<<i)) {
			nCiclos--;
			//Pre-indexada
			dirEfectiva += dirOffset;
			//Lectura de memoria
			resultado = core->readMemoriaWord(dirEfectiva);

			//Escribe el resultado en el registro correspondiente
			LSM_WRITEREGS
		}

	}

	//W=1 (Write-back)
	if (w) {
		contexto->bancoRegistros[rn].u_word = dirEfectiva;
	}

	CHK_ABORT

	//lsmCPSR (Se restaura el CPSR con el SPSR)
	if (lsmCPSR) {
		contexto->escribirCPSR(contexto->SPSR[contexto->modo-1].u_word);
	}
	
	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina LDM post-index (Load multiple)
/*-------------------------------------------------------------------------*/
RUT_ldm_post:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LSM

	//Para cada registro, lo transfiere si su bit esta a 1
	for ((u==1)? i = 0 : i=15;
		 (u==1)? (i<16): (i>=0);
		 (u==1)?  i++  : i--) {

		if (listaRegistros & (0x1<<i)) {
			nCiclos--;
			//Lectura de memoria
			resultado = core->readMemoriaWord(dirEfectiva);
			
			//Escribe el resultado en el registro correspondiente
			LSM_WRITEREGS

			//Post-indexada
			dirEfectiva += dirOffset;
		}

	}

	//W=1 (Write-back)
	if (w) {
		contexto->bancoRegistros[rn].u_word = dirEfectiva;
	}

	CHK_ABORT

	//lsmCPSR (Se restaura el CPSR con el SPSR)
	if (lsmCPSR) {
		contexto->escribirCPSR(contexto->SPSR[contexto->modo-1].u_word);
	}
	
	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina STM pre-index (Store multiple)
/*-------------------------------------------------------------------------*/
RUT_stm_pre:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LSM

	//Para cada registro, lo transfiere si su bit esta a 1
	for ((u==1)? i = 0 : i=15;
		 (u==1)? (i<16): (i>=0);
		 (u==1)?  i++  : i--) {

		if (listaRegistros & (0x1<<i)) {
			nCiclos--;
			//Pre-indexada
			dirEfectiva += dirOffset;

			//Carga el valor del registro correspondiente
			LSM_READREGS

			//Escritura en memoria
			core->writeMemoriaWord(resultado, dirEfectiva);
					
		}

	}

	//W=1 (Write-back)
	if (w) {
		contexto->bancoRegistros[rn].u_word = dirEfectiva;
	}
	
	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina STM post-index (Store multiple)
/*-------------------------------------------------------------------------*/
RUT_stm_post:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LSM

	//Para cada registro, lo transfiere si su bit esta a 1
	for ((u==1)? i = 0 : i=15;
		 (u==1)? (i<16): (i>=0);
		 (u==1)?  i++  : i--) {

		if (listaRegistros & (0x1<<i)) {
			nCiclos--;

			//Carga el valor del registro correspondiente
			LSM_READREGS

			//Escritura en memoria
			core->writeMemoriaWord(resultado, dirEfectiva);
			
			//Post-indexada
			dirEfectiva += dirOffset;
		}

	}

	//W=1 (Write-back)
	if (w) {
		contexto->bancoRegistros[rn].u_word = dirEfectiva;
	}
	
	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina SWP (Swap Memory and Register)
/*-------------------------------------------------------------------------*/
RUT_swap:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_LS

		
	if (b) {
		resultado = unsigned int (core->readMemoriaByte(operando1));
		CHK_ABORT
		core->writeMemoriaByte((uint8)operando2, operando1);

	} else {
		dirEfectiva = operando1 & 0xFFFFFFFC;
		resultado = core->readMemoriaWord(dirEfectiva);
		CHK_ABORT
		core->writeMemoriaWord(operando2, dirEfectiva);

	} 

	contexto->bancoRegistros[rd].u_word = resultado;
	
	
	nCiclos-=2;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina MUL (Multiply and accumulate)
/*-------------------------------------------------------------------------*/
RUT_mul:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_MUL

	v = contexto->flags.v;

	//MUL
	if (a) {
		//Acumula
		__asm {
			mov eax, operando1
			mov ebx, operando2
			mov edx, 0
			mul ebx				;edx:eax = eax * ebx
			add eax, acumulador	;eax = eax + acc
			pushf				;guarda los flags en la fila
			mov resultado, eax
		}
		
	}else{
		//No acumula
		__asm {
			mov eax, operando1
			mov ebx, operando2
			mov edx, 0
			mul ebx				;edx:eax = eax * ebx
			pushf				;guarda los flags en la fila
			mov resultado, eax
		}
	}
	
	//Guarda los flags
	GETFLAGS
	contexto->flags.v = v;

	contexto->bancoRegistros[rd].u_word = resultado;

	nCiclos-=5;
	goto LecturaInstruccion;




/*-------------------------------------------------------------------------*/
/*	Rutina MULL (Multiply and accumulate long)
/*-------------------------------------------------------------------------*/
RUT_mull:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_MUL

	v = contexto->flags.v;

	//MUL
	if (a && !u) {
		//Acumula y sin signo
		__asm {

			mov eax, operando1
			mov ebx, operando2
			mov edx, 0
			mul ebx				;edx:eax = eax * ebx

			movd mm0, eax
			movd mm1, edx
			punpckldq mm0, mm1	;Monta los dos registros de 32 bits en uno de 64

			movd mm1, acumulador
			movd mm2, acumulador2
			punpckldq mm1, mm2	;Monta el registro acumulador

			movq uvar1_64, mm0
			movq uvar2_64, mm1
		}

		uvar1_64 += uvar2_64;
		
		__asm {
			movq mm0, uvar1_64
			movd resultado, mm0	;Parte baja del resultado
			punpckhdq mm0, mm1	;Pone la parte alta de mm0 en la parte baja
			movd resultado2, mm0 ;Parte alta del resultado
			emms
		}

		contexto->flags.n = (uvar1_64 < 0)?1:0;
		contexto->flags.z = (uvar1_64 == 0)?1:0;

	}else if (a && u) {
		//Acumula y con signo
		
		__asm {

			mov eax, operando1
			mov ebx, operando2
			mov edx, 0
			imul ebx				;edx:eax = eax * ebx

			movd mm0, eax
			movd mm1, edx
			punpckldq mm0, mm1	;Monta los dos registros de 32 bits en uno de 64

			movd mm1, acumulador
			movd mm2, acumulador2
			punpckldq mm1, mm2	;Monta el registro acumulador

			movq var1_64, mm0
			movq var2_64, mm1
		}

		var1_64 += var2_64;
		
		__asm {
			movq mm0, var1_64
			movd resultado, mm0	;Parte baja del resultado
			punpckhdq mm0, mm1	;Pone la parte alta de mm0 en la parte baja
			movd resultado2, mm0 ;Parte alta del resultado
			emms
		}

		contexto->flags.n = (var1_64 < 0)?1:0;
		contexto->flags.z = (var1_64 == 0)?1:0;
		
	}else if(!a && !u){
		//No acumula y sin signo
		__asm {
			mov eax, operando1
			mov ebx, operando2
			mov edx, 0
			mul ebx				;edx:eax = eax * ebx
			pushf				;guarda los flags en la fila
			mov resultado, eax
			mov resultado2, edx
		}

		//Guarda los flags
		GETFLAGS
	}else if(!a && u){
		//No acumula y con signo
		__asm {
			mov eax, operando1
			mov ebx, operando2
			mov edx, 0
			imul ebx			;edx:eax = eax * ebx
			pushf				;guarda los flags en la fila
			mov resultado, eax
			mov resultado2, edx
		}

		//Guarda los flags
		GETFLAGS
	}
	
	contexto->flags.v = v;

	contexto->bancoRegistros[rn].u_word = resultado;
	contexto->bancoRegistros[rd].u_word = resultado2;

	nCiclos-=6;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina MSR (Move Status Register to Register)
/*-------------------------------------------------------------------------*/
RUT_msr:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	GETOPS_MSR

	//CPSR
	if (!r) {
		operando2 = (contexto->leerCPSR());
		//f
		if (mascara & 0x8) {
			operando2 = (operando1 & 0xFF000000) | (operando2 & 0x00FFFFFF);
		}
		if ((contexto->modo != USER_MODE) && (!esMISR)) {
			//s
			if (mascara & 0x4) { 
				operando2 = (operando1 & 0x00FF0000) | (operando2 & 0xFF00FFFF);
			}
			//x
			if (mascara & 0x2) { 
				operando2 = (operando1 & 0x0000FF00) | (operando2 & 0xFFFF00FF);
			}
			//c
			if (mascara & 0x1) { 
				operando2 = (operando1 & 0x000000FF) | (operando2 & 0xFFFFFF00);
				nCiclos-=2;
			}
		}

		contexto->escribirCPSR(operando2);

	//SPSR
	} else if ((contexto->modo != USER_MODE) && (contexto->modo != SYSTEM_MODE)) {
		operando2 = (contexto->SPSR[contexto->modo - 1].u_word);
		//f
		if (mascara & 0x8) {
			operando2 = (operando1 & 0xFF000000) | (operando2 & 0x00FFFFFF);
		}
		//s
		if (!esMISR) {
			if (mascara & 0x4) { 
				operando2 = (operando1 & 0x00FF0000) | (operando2 & 0xFF00FFFF);
			}
			//x
			if (mascara & 0x2) { 
				operando2 = (operando1 & 0x0000FF00) | (operando2 & 0xFFFF00FF);
			}
			//c
			if (mascara & 0x1) { 
				operando2 = (operando1 & 0x000000FF) | (operando2 & 0xFFFFFF00);
				nCiclos-=2;
			}
		}

		contexto->SPSR[contexto->modo - 1].u_word = operando2;

	}

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina MRS (Move Register to Status Register)
/*-------------------------------------------------------------------------*/
RUT_mrs:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	rd = instruccion.DPIS.Rd;
	r = instruccion.MISR.R;

	//CPSR
	if (!r) {
		contexto->bancoRegistros[rd].u_word = contexto->leerCPSR();
		
	//SPSR
	} else if ((contexto->modo != USER_MODE) && (contexto->modo != SYSTEM_MODE)) {
		contexto->bancoRegistros[rd].u_word = (contexto->SPSR[contexto->modo - 1].u_word);
		
	}

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina SWI (Software Interrupt)
/*-------------------------------------------------------------------------*/
RUT_swi:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	//Obtiene los operandos
	operando1 = instruccion.SWI.swiNumber;
	

	core->marcarExcepcion (EXCEP_SWI, operando1);

	nCiclos-=3;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina CODP (Coprocessor Data Processing)
/*-------------------------------------------------------------------------*/
RUT_codp:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	operando1 = instruccion.CODP.cp_num;
	coproDP fdp = core->getCoproDP(operando1);
	
	if (fdp != NULL) {
		nCiclos -= fdp(instruccion.u_word);
	} else {
		core->marcarExcepcion (EXCEP_UNDEFINED, operando1);
	}

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina COLS (Coprocessor Load Store)
/*-------------------------------------------------------------------------*/
RUT_cols:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	operando1 = instruccion.COLS.cp_num;
	coproLS fls = core->getCoproLS(operando1);
	
	if (fls != NULL) {
		nCiclos -= fls(instruccion.u_word);
	} else {
		core->marcarExcepcion (EXCEP_UNDEFINED, operando1);
	}

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina CORT (Coprocessor Register Transfer)
/*-------------------------------------------------------------------------*/
RUT_cort:

	//Comprueba el codigo de condicion
	COMPROBAR_COND

	operando1 = instruccion.CORT.cp_num;
	coproRT frt = core->getCoproRT(operando1);
	
	if (frt != NULL) {
		nCiclos -= frt(instruccion.u_word);
	} else {
		core->marcarExcepcion (EXCEP_UNDEFINED, operando1);
	}

	nCiclos--;
	goto LecturaInstruccion;


/*-------------------------------------------------------------------------*/
/*	Rutina INDEFINIDA
/*-------------------------------------------------------------------------*/
RUT_undefined:

	core->marcarExcepcion (EXCEP_UNDEFINED);

	nCiclos--;
	goto LecturaInstruccion;
}



