/*
    ARM9Core
    Copyright (C) 2007 Alberto Huerta Aranda,
	Sergio Hidalgo Serrano, Daniel Saudo Vacas

    This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 3 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
	License along with this library. If not, see <http://www.gnu.org/licenses/>

*/


#ifndef UtilsH
#define UtilsH

/**************************************************************************************

  Archivo: Utils.h

  Descripcion: Tipos, definiciones, y mtodos comunes

**************************************************************************************/


//Tipos de datos
typedef unsigned char  uint8; 
typedef signed char  int8; 
typedef unsigned short  uint16;
typedef short  int16;
typedef unsigned int  uint32;
typedef int  int32;
typedef unsigned __int64 uint64;
typedef __int64 int64;

//Definicion de NULL
#ifndef NULL
#ifdef __cplusplus
#define NULL	0
#else
#define NULL	((void *)0)
#endif
#endif


/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/


//Registro de 32 bits
typedef union{

	//Acceso a la palabra (32 bits)
	uint32 u_word;
	int32 s_word;

	//Acceso a la media palabra (16 bits)
	uint16 u_halfword;
	int16 s_halfword;

	//Acceso a los bytes de la media palabra
	struct {
		uint8 lByte;	//Byte bajo
		uint8 hByte;	//Byte alto
	} u_hw;

	struct {
		int8 lByte;	//Byte bajo
		int8 hByte;	//Byte alto
	} s_hw;

	//Acceso al byte
	uint8 u_byte;
	int8 s_byte;
	

	//Acceso a los bits del CPSR
	struct{
		unsigned int mode:5;			//Mode bits
		unsigned int state:1;		//State bit
		unsigned int fiqDis:1;		//FIQ disable
		unsigned int irqDis:1;		//IRQ disable
		unsigned int reservados:20;	
		unsigned int v:1;			//Overflow
		unsigned int c:1;			//Carry / Borrow / Extend
		unsigned int z:1;			//Zero
		unsigned int n:1;			//Negative / Less Than
		
		
	} CPSR;

} ARM9Reg32;


/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/


//Modos del procesador

typedef enum {
	USER_MODE = 0,	//User mode
	SVC_MODE,		//Supervisor mode
	ABORT_MODE,		//Abort mode
	UNDEF_MODE,		//Undefined mode
	IRQ_MODE,		//Interrupt mode
	FIQ_MODE,		//Fast Interrupt mode
	SYSTEM_MODE		//System mode
} ARM9Mode;


/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/


//Excepciones

typedef enum {
	EXCEP_NONE = 0,			//No hay excepcion
	EXCEP_UNDEFINED,		//Instruccion indefinida
	EXCEP_SWI,				//Software interrupt
	EXCEP_PREFETCH_ABORT,	//Error de memoria al leer instruccion
	EXCEP_IRQ,				//Interrupcion estandar
	EXCEP_FIQ,				//Interrupcion rapida
	EXCEP_DATA_ABORT,		//Error de memoria
	EXCEP_RESET				//Reinicio del procesador
} ARM9Exception;



/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/


//Regiones de memoria

typedef struct {

	uint32 inferior; //Direccin inferior del bloque de memoria
	uint32 superior; //Direccin superior

	uint8 *buffer;	 //Buffer de memoria (si se usa handler, poner a NULL)
	void *handlerRead8;	 //Punteros a funciones manejadora (si se usa buffer, poner a NULL)
	void *handlerRead16;
	void *handlerRead32;
	void *handlerWrite8;
	void *handlerWrite16;
	void *handlerWrite32;

	//Permisos
	int permisos;	//(wp, rp, wu, ru) (Write/Read Privilegiado/User)

} ARM9MemRegion;


//Tipos de punteros a funciones manejadoras

typedef uint32 (*readHandler32)(uint32 dir);
typedef uint32 (*readHandler16)(uint32 dir);
typedef uint32 (*readHandler8)(uint32 dir);
typedef void (*writeHandler32)(uint32 datos, uint32 dir);
typedef void (*writeHandler16)(uint16 datos, uint32 dir);
typedef void (*writeHandler8)(uint8 datos, uint32 dir);


//Permisos de acceso a memoria

typedef enum {
	USR_READ_PERM = 0x1,	//Permiso de acceso a lectura para usuario
	USR_WRITE_PERM = 0x2,	//Permiso de acceso a escritura para usuario
	PRIV_READ_PERM = 0x4,	//Permiso de acceso a lectura para modo privilegiado
	PRIV_WRITE_PERM = 0x8	//Permiso de acceso a escritura para modo privilegiado
} ARM9MemPermission;



/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/

//Interfaz del coprocesador

//Tipos de punteros a funciones del coprocesador
//(Devuelven el numero de ciclos "extra" consumidos por el cp (siempre se consume al menos 1))

typedef int (*coproDP)(uint32 instruccion);
typedef int (*coproLS)(uint32 instruccion);
typedef int (*coproRT)(uint32 instruccion);


/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/


//Instruccion ARM de 32 bits

typedef union {

	//Acceso a la palabra (32 bits)
	uint32 u_word;
	int32 s_word;

	//Acceso a los campos comunes
	struct {
		unsigned int relleno:4;		//No usar
		unsigned int codigoTabla2:4;//Segunda parte del codigo en la tabla de saltos
		unsigned int relleno2:12;	//No usar
		unsigned int codigoTabla:8;	//Codigo en la tabla de saltos
		unsigned int cond:4;		//Codigo de condicion
	} comunes;

	//Acceso a los campos por tipo de instruccion

	//Data processing inmediate shift
	struct {
		unsigned int Rm:4;			//2do operando
		unsigned int bit4:1;		//Distingue entre DPIS y DPRS
		unsigned int shift:2;		//Shift
		unsigned int shifta:5;		//Shift amount
		unsigned int Rd:4;			//Registro destino
		unsigned int Rn:4;			//1er operando
		unsigned int S:1;			//Set condition codes
		unsigned int opcode:4;		//Opcode
		unsigned int relleno:7;		//No usar
	} DPIS;

	//Data processing register shift
	struct {
		unsigned int Rm:4;			//2do operando
		unsigned int bit4:1;		//Distingue entre DPIS y DPRS
		unsigned int shift:2;		//Shift
		unsigned int relleno2:1;	//No usar
		unsigned int Rs:4;			//Registro de longitud de shift
		unsigned int Rd:4;			//Registro destino
		unsigned int Rn:4;			//1er operando
		unsigned int S:1;			//Set condition codes
		unsigned int opcode:4;		//Opcode
		unsigned int relleno:7;		//No usar
	} DPRS;

	//Multiplies
	struct {
		unsigned int Rm:4;		//2do operando / LoOffset
		unsigned int relleno2:4;//No usar
		unsigned int Rs:4;		//1er operando / SBZ / HiOffset
		unsigned int Rn:4;		//Reg. acumulador / RdLo / Rd(Reg. destino)
		unsigned int Rd:4;		//Registro destino / RdHi / Rn(Reg. base)
		unsigned int S:1;		//Set condition code / L(load/store) / Bit 20
		unsigned int A:1;		//Acumulador / W(rite-back) 
		unsigned int B:1;		//Unsigned / B(Byte/word) / Bit 22
		unsigned int U:1;		//Up/Down
		unsigned int P:1;		//Pre/post indexado
		unsigned int relleno:7;	//No usar
	} MUL;

	//Data processing inmediate
	struct {
		unsigned int inmediate:8;	//Operando inmediato
		unsigned int rotate:4;		//Alineamiento inmediato
		unsigned int Rd:4;			//Registro destino
		unsigned int Rn:4;			//1er operando
		unsigned int S:1;			//Set condition codes
		unsigned int opcode:4;		//Opcode
		unsigned int relleno:7;		//No usar
	} DPI;

	//Move inmediate to status register
	struct {
		unsigned int inmediate:8;	//Operando inmediato
		unsigned int rotate:4;		//Alineamiento inmediato
		unsigned int SBO:4;			//Should be One
		unsigned int Mask:4;		//Mascara
		unsigned int relleno2:2;	//No usar
		unsigned int R:1;			//SPSR/CPSR
		unsigned int relleno:9;		//No usar
	} MISR;
	
	//Load/Store inmediate offset
	struct {
		unsigned int inmediate:12;	//Operando inmediato
		unsigned int Rd:4;			//Registro destino / origen
		unsigned int Rn:4;			//Registro base
		unsigned int L:1;			//Load/Store
		unsigned int W:1;			//Write-back (autoindex)
		unsigned int B:1;			//Unsigned byte/word
		unsigned int U:1;			//Up/down
		unsigned int P:1;			//Pre/post indexado
		unsigned int relleno:7;		//No usar
	} LSIO;

	//Load/Store register offset
	struct {
		unsigned int Rm:4;			//Offset register
		unsigned int relleno2:1;	//No usar
		unsigned int shift:2;		//Shift type
		unsigned int shifta:5;		//Shift amount
		unsigned int Rd:4;			//Registro destino / origen
		unsigned int Rn:4;			//Registro base
		unsigned int L:1;			//Load/Store
		unsigned int W:1;			//Write-back (autoindex)
		unsigned int B:1;			//Unsigned byte/word
		unsigned int U:1;			//Up/down
		unsigned int P:1;			//Pre/post indexado
		unsigned int bit25:1;		//Bit 25 para diferenciar LSIO y LSRO
		unsigned int relleno:6;		//No usar
	} LSRO;

	//Load/Store multiple
	struct {
		unsigned int Rlist:16;		//Register list
		unsigned int Rn:4;			//Registro base
		unsigned int L:1;			//Load/Store
		unsigned int W:1;			//Write-back (autoindex)
		unsigned int S:1;			//Restore PSR and force user bit
		unsigned int U:1;			//Up/down
		unsigned int P:1;			//Pre/post indexado
		unsigned int relleno:7;		//No usar
	} LSM;

	//Branch and branch with link
	struct {
		signed int offset:24;		//Signed word offset
		unsigned int L:1;			//Link
		unsigned int relleno:7;		//No usar
	} BBL;

	//Branch and change
	struct {
		unsigned int Rm:4;			//Registro para cambiar el modo
		unsigned int bxid2:4;		//Debe ser 0x1
		unsigned int SBO:12;		//Should be one	
		unsigned int bxid:8;		//Debe ser 0x12
		unsigned int relleno:4;		//No usar
	} BX;

	//Coprocessor load/store and double register transfer
	struct {
		unsigned int offset:8;		//8-bit offset
		unsigned int cp_num:4;		//Numero del coprocesador
		unsigned int CRd:4;			//Registro destino / fuente del coprocesador
		unsigned int Rn:4;			//Registro base
		unsigned int L:1;			//Load/Store
		unsigned int W:1;			//Write-back (autoindex)
		unsigned int N:1;			//Data size (dependiente del coprocesador)
		unsigned int U:1;			//Up/down
		unsigned int P:1;			//Pre/post indexado
		unsigned int relleno:7;		//No usar
	} COLS;

	//Coprocessor data processing
	struct {
		unsigned int CRm:4;			//Dependiente del coprocesador
		unsigned int relleno2:1;	//No usar
		unsigned int opcode2:3;		//Dependiente del coprocesador
		unsigned int cp_num:4;		//Numero del coprocesador
		unsigned int CRd:4;			//Dependiente del coprocesador
		unsigned int CRn:4;			//Dependiente del coprocesador
		unsigned int opcode1:4;		//Dependiente del coprocesador
		unsigned int relleno:8;		//No usar
	} CODP;

	//Coprocessor register transfer
	struct {
		unsigned int CRm:4;			//Dependiente del coprocesador
		unsigned int relleno2:1;	//No usar
		unsigned int opcode2:3;		//Dependiente del coprocesador
		unsigned int cp_num:4;		//Numero del coprocesador
		unsigned int Rd:4;			//Registro fuente o destino del ARM
		unsigned int CRn:4;			//Dependiente del coprocesador
		unsigned int L:1;			//Load / store del coprocesador
		unsigned int opcode1:3;		//Dependiente del coprocesador
		unsigned int relleno:8;		//No usar
	} CORT;

	//Software interrupt 
	struct {
		unsigned int swiNumber:24;	//Numero de la interrupcion
		unsigned int relleno:8;		//No usar
	} SWI;


} ARM9Instruccion32;


/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/


#endif