/*
    ARM9Core
    Copyright (C) 2007 Alberto Huerta Aranda,
	Sergio Hidalgo Serrano, Daniel Saudo Vacas

    This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 3 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
	License along with this library. If not, see <http://www.gnu.org/licenses/>

*/


/**************************************************************************************

  Archivo: ARM9Contexto.cpp

  Descripcion: Contexto del procesador

**************************************************************************************/

#include "ARM9Contexto.h"

ARM9Contexto::ARM9Contexto () {

	PC = &bancoRegistros[15];
	reset ();
}


void ARM9Contexto::reset () {

	//Inicializa el modo (SVC_MODE)
	modo = SVC_MODE;

	//PC = 0
	PC->u_word = 0;
	
	//Inicializa el CPSR
	state = fiqDis = irqDis = condFlags = 0;

}


void ARM9Contexto::cambiarModo (ARM9Mode nuevoModo) {
	ARM9Reg32 t;
	ARM9Mode m;
	
	m = nuevoModo;
	if (nuevoModo == SYSTEM_MODE) {
		m = USER_MODE;
	}
	if (modo == SYSTEM_MODE) {
		modo = USER_MODE;
	}
	
	//Guardamos los registros del modo actual
	tablaPunteros[modo][0] = bancoRegistros[13];
	tablaPunteros[modo][1] = bancoRegistros[14];

	//Restaura los registros banqueados del nuevo modo
	bancoRegistros[13] = tablaPunteros[m][0];
	bancoRegistros[14] = tablaPunteros[m][1];

	//Guardamos los registros FIQ
	if (modo == FIQ_MODE) {
		t = tablaFIQ[0];
		tablaFIQ[0] = bancoRegistros[8];
		bancoRegistros[8] = t;
		t = tablaFIQ[1];
		tablaFIQ[1] = bancoRegistros[9];
		bancoRegistros[9] = t;
		t = tablaFIQ[2];
		tablaFIQ[2] = bancoRegistros[10];
		bancoRegistros[10] = t;
		t = tablaFIQ[3];
		tablaFIQ[3] = bancoRegistros[11];
		bancoRegistros[11] = t;
		t = tablaFIQ[4];
		tablaFIQ[4] = bancoRegistros[12];
		bancoRegistros[12] = t;
	}

	//Restauramos los registros FIQ
	if (m == FIQ_MODE) {
		t = bancoRegistros[8];
		bancoRegistros[8] = tablaFIQ[0];
		tablaFIQ[0] = t;
		t = bancoRegistros[9];
		bancoRegistros[9] = tablaFIQ[1];
		tablaFIQ[1] = t;
		t = bancoRegistros[10];
		bancoRegistros[10] = tablaFIQ[2];
		tablaFIQ[2] = t;
		t = bancoRegistros[11];
		bancoRegistros[11] = tablaFIQ[3];
		tablaFIQ[3] = t;
		t = bancoRegistros[12];
		bancoRegistros[12] = tablaFIQ[4];
		tablaFIQ[4] = t;
	}

	modo = nuevoModo;
}	


void ARM9Contexto::escribirCPSR(uint32 nuevoCPSR) {
	ARM9Reg32 r;
	r.u_word = nuevoCPSR;

	state = r.CPSR.state;
	fiqDis = r.CPSR.fiqDis;
	irqDis = r.CPSR.irqDis;
	flags.v = r.CPSR.v;
	flags.c = r.CPSR.c;
	flags.z = r.CPSR.z;
	flags.n = r.CPSR.n;

	cambiarModo (bitsToMode(r.CPSR.mode));

}


uint32 ARM9Contexto::leerCPSR() {
	ARM9Reg32 r;
	r.u_word = 0;

	r.CPSR.mode = modeToBits (modo);
	r.CPSR.state = state;
	r.CPSR.fiqDis = fiqDis;
	r.CPSR.irqDis = irqDis;
	r.CPSR.v = flags.v;
	r.CPSR.c = flags.c;
	r.CPSR.z = flags.z;
	r.CPSR.n = flags.n;

	return r.u_word;
}


ARM9Mode ARM9Contexto::bitsToMode (unsigned int bits) {

	switch (bits) {
	case 0x11:
		return FIQ_MODE;
	case 0x12:
		return IRQ_MODE;
	case 0x13:
		return SVC_MODE;
	case 0x17:
		return ABORT_MODE;
	case 0x1B:
		return UNDEF_MODE;
	case 0x1F:
		return SYSTEM_MODE;
	case 0x10:
	default:
		return USER_MODE;
	}
}


unsigned int ARM9Contexto::modeToBits (ARM9Mode m) {

	switch (m) {
	case FIQ_MODE:
		return 0x11;
	case IRQ_MODE:
		return 0x12;
	case SVC_MODE:
		return 0x13;
	case ABORT_MODE:
		return 0x17;
	case UNDEF_MODE:
		return 0x1B;
	case SYSTEM_MODE:
		return 0x1F;
	case USER_MODE:
	default:
		return 0x10;
	}
}