#include "tinydir.h"
#include "menu.h"
#include "osdepend.h"
#include <SDL.H>

#define bool _Bool
#define true 1
#define false 0

#define Uint32 unsigned int

#define		VZ_GREEN	0
#define		VZ_YELLOW	1
#define		VZ_BLUE		2
#define		VZ_RED		3
#define		VZ_BUFF		4
#define		VZ_CYAN		5
#define		VZ_MAGENTA	6
#define		VZ_ORANGE	7
#define		VZ_BLACK	8
#define		VZ_DKGREEN	9
#define		VZ_BRGREEN	10
#define		VZ_DKORANGE 11
#define		VZ_BRORANGE 12


typedef struct
{
	char *selection;
	void *nextMenu;
	void (*pFunc)();
} MENU_ACTIONS;

typedef struct
{
    char    filename[32];
    bool    isDirectory;
} DIRECTORY_ENTRY;


const char *MAIN_MENU[] = {"FILE","OPTIONS","UTILS","HELP","RESET","QUIT",NULL};

	const char *FILE_MENU[] = {"LOAD VZ","DISK","CASSETTE","PRINTER","LOAD ROM","LOAD CARTRIDGE","LOAD FONT",NULL};
		const char *DISK_MENU[] = {"MOUNT","UNMOUNT",NULL};
		const char *CASSETTE_MENU[] = {"LOAD","EJECT","STOP",NULL};


	const char *OPTIONS_MENU[] = {"MODEL","MEMORY","6847 EFFECTS","FRAMERATE","CARTRIDGE","JOYSTICKS","SPEAKER","CASSETTE AUDIO","DISK AUDIO","EXTENDED GFX","WRITE TO ROM","WRITE TO CARTRIDGE","DISPLAY CONFIG","SAVE CONFIG",NULL};
		const char *MODEL_MENU[] = {"VZ200 (3.5795 MHZ)","VZ300 (3.5469 MHZ)","PAL","NTSC",NULL};
		const char *MEMORY_MENU[] = {"8K","18K","24K","34K","4MB",NULL};
		const char *M6847_MENU[] = {"EFFECTS ON","EFFECTS OFF",NULL};
		const char *FRAMERATE_MENU[] = {"NORMAL","MAXIMUM",NULL};
		const char *JOYSTICKS_MENU[] = {"JOYSTICKS ENABLED","JOYSTICKS DISABLED","CURSOR KEYS",NULL};
		const char *SPEAKER_MENU[] = {"SOUND ON","SOUND OFF",NULL};
		const char *CASSETTE_AUDIO_MENU[] = {"CASSETTE AUDIO ON","CASSETTE AUDIO OFF",NULL};
		const char *DISK_AUDIO_MENU[] = {"DISK AUDIO ON","DISK AUDIO OFF",NULL};
		const char *EXTENDED_GFX_MENU[] = {"NONE","AUSTRALIAN","GERMAN",NULL};
		const char *WRITE_TO_ROM_MENU[] = {"WRITE TO ROM ENABLED","WRITE TO ROM DISABLED",NULL};
		const char *WRITE_TO_CARTRIDGE_MENU[] = {"WRITE TO CARTRIDGE ENABLED","WRITE TO CARTRIDGE DISABLED",NULL};

	const char *UTILS_MENU[] = {"LOAD BITMAP",NULL};
	const char *HELP_MENU[] = {"ABOUT","INSTRUCTIONS",NULL};


MENU_ACTIONS actions[] =
{
	{"FILE",FILE_MENU,NULL},
			{"LOAD VZ",NULL,&LOAD_VZ},
			{"DISK",DISK_MENU,NULL},
					{"MOUNT",NULL,&MOUNT_DISK},
					{"UNMOUNT",NULL,&DoNothing},
			{"CASSETTE",CASSETTE_MENU,NULL},
					{"LOAD",NULL,&LOAD_CASSETTE},
					{"EJECT",NULL,&DoNothing},
			{"PRINTER",NULL,&MAP_PRINTER},
			{"LOAD ROM",NULL,&LOAD_ROM},
			{"LOAD CARTRIDGE",NULL,&LOAD_CARTRIDGE},
			{"LOAD FONT",NULL,&LOAD_FONT},
			{"RESET",NULL,&RESET},
			{"QUIT",NULL,&QUIT},

	{"OPTIONS",OPTIONS_MENU,NULL},
			{"MODEL",MODEL_MENU,NULL},
					{"VZ200 (3.5795 MHZ)",NULL,&SET_PREFS_VZ200},
					{"VZ300 (3.5469 MHZ)",NULL,&SET_PREFS_VZ300},
					{"PAL",NULL,&SET_PREFS_PAL},
					{"NTSC",NULL,&SET_PREFS_NTSC},
			{"MEMORY",MEMORY_MENU,NULL},
					{"8K",NULL,&SET_PREFS_8K},
					{"18K",NULL,&SET_PREFS_18K},
					{"24K",NULL,&SET_PREFS_24K},
					{"34K",NULL,&SET_PREFS_34K},
					{"4MB",NULL,&SET_PREFS_4MB},
			{"6847 EFFECTS",M6847_MENU,NULL},
					{"EFFECTS ON",NULL,&SET_PREFS_6847_EFFECTS_ON},
					{"EFFECTS OFF",NULL,&SET_PREFS_6847_EFFECTS_OFF},
			{"FRAMERATE",FRAMERATE_MENU,NULL},
					{"NORMAL",NULL,&SET_PREFS_FRAMERATE_NORMAL},
					{"MAXIMUM",NULL,&SET_PREFS_FRAMERATE_MAXIMUM},
			{"JOYSTICKS",JOYSTICKS_MENU,NULL},
					{"JOYSTICKS ENABLED",NULL,&DoNothing},
					{"JOYSTICKS DISABLED",NULL,&DoNothing},
					{"CURSOR KEYS",NULL,&DoNothing},
			{"SPEAKER",SPEAKER_MENU,NULL},
					{"SOUND ON",NULL,&SET_PREFS_SOUND_ON},
					{"SOUND OFF",NULL,&SET_PREFS_SOUND_OFF},
			{"CASSETTE AUDIO",CASSETTE_AUDIO_MENU,NULL},
					{"CASSETTE AUDIO ON",NULL,&SET_PREFS_CASSETTE_AUDIO_ON},
					{"CASSETTE AUDIO OFF",NULL,&SET_PREFS_CASSETTE_AUDIO_OFF},
			{"DISK AUDIO",DISK_AUDIO_MENU,NULL},
					{"DISK AUDIO ON",NULL,&SET_PREFS_DISK_AUDIO_ON},
					{"DISK AUDIO OFF",NULL,&SET_PREFS_DISK_AUDIO_OFF},
			{"EXTENDED GFX",EXTENDED_GFX_MENU,NULL},
					{"NONE",NULL,&SET_PREFS_EXTENDED_GFX_NONE},
					{"AUSTRALIAN",NULL,&SET_PREFS_EXTENDED_GFX_AUS},
					{"GERMAN",NULL,&SET_PREFS_EXTENDED_GFX_GERMAN},
			{"WRITE TO ROM",WRITE_TO_ROM_MENU,NULL},
					{"WRITE TO ROM ENABLED",NULL,&SET_PREFS_WRITE_TO_ROM_ENABLED},
					{"WRITE TO ROM DISABLED",NULL,&SET_PREFS_WRITE_TO_ROM_DISABLED},
			{"WRITE TO CARTRIDGE",WRITE_TO_CARTRIDGE_MENU,NULL},
					{"WRITE TO CARTRIDGE ENABLED",NULL,&SET_PREFS_WRITE_TO_CARTRIDGE_ENABLED},
					{"WRITE TO CARTRIDGE DISABLED",NULL,&SET_PREFS_WRITE_TO_CARTRIDGE_DISABLED},
            {"DISPLAY CONFIG",NULL,&DISPLAY_CONFIG},
			{"SAVE CONFIG",NULL,&SAVE_CONFIG},

	{"UTILS",UTILS_MENU,NULL},
			{"LOAD BITMAP",NULL,&LOAD_BITMAP},

	{"HELP",HELP_MENU,NULL},
			{"ABOUT",NULL,&ABOUT},
			{"INSTRUCTIONS",NULL,&INSTRUCTIONS},

	{NULL,NULL,NULL}

};


byte	vzkbrd[8];								// VZ keyboard array
byte Display_Buffer[77760];

extern bool    RUNNING;
extern bool    MENUACTIVE;

extern byte pal_square_fontdata8x12[];
extern Uint32 palette[];


void DoNothing()
{
}


void putpixel(int x, int y, int color)
/**************************************************************************
*
* Name:			putpixel
* Purpose:		draw to device independent display buffer
* Arguments: 	x,y coordinates and color reference
*               uses global array variable Display_Buffer
*
***************************************************************************/
{
  Display_Buffer[320*y+x] = color;
}

void    DrawChar(int x, int y, unsigned char c, int forecolor, int backcolor)
/**************************************************************************
*
* Name:			DrawChar
* Purpose:		draw a character in the M6847 8x12 font
* Arguments: 	x - 0 to 31
*               y - 0 to 16
*               c - 0 to 255 VZ ascii value
*               forecolor - from the M6847 color palette
*               backcolor - from the M6847 color palette
*
***************************************************************************/
{
    int     offs,px,py;
    unsigned char mask, fb;

   offs = c * 12;  // start of character font
   for (py = 0; py < 12; py++)
   {
       fb = pal_square_fontdata8x12[offs+py];
       mask = 0x80;
       for (px = 0; px < 8; px++)
       {
           if (fb & mask)
           {
               putpixel(x*8+px,y*12+py,forecolor);
           }
           else
           {
              putpixel(x*8+px,y*12+py,backcolor);
           }
          mask = mask >> 1;
       }
   }
}

void DrawString(int x, int y, char *ptr, int forecolor, int backcolor)
/**************************************************************************
*
* Name:			DrawString
* Purpose:		draw a string in the M6847 8x12 font
* Arguments: 	x - 0 to 31
*               y - 0 to 16
*               ptr - pointer to null terminated character string
*               forecolor - from the M6847 color palette
*               backcolor - from the M6847 color palette
*
***************************************************************************/
{
    int     sx = x;
    int     len = strlen(ptr);
    char    *ch = ptr;
    char    chr;
    for (int n=0;n<len;n++)
    {
        chr = toupper(*ch);

        if (chr > 64) chr -= 64;
        DrawChar(sx+n,y,chr,forecolor,backcolor);
        (char*) ch++ ;
    }
}


void ClearMenuScreen()
/**************************************************************************
*
* Name:			ClearMenuScreen()
* Purpose:		clear the device independent display buffer
* Arguments: 	none
*
***************************************************************************/
{
    for (int x=0;x<320;x++)
    {
        for (int y=0;y<240;y++)
        {
            Display_Buffer[320*y+x] = VZ_DKGREEN;
        }
    }
}


void SelectFile(TCHAR *filePath, TCHAR *fileName, char *caption)
/**************************************************************************
*
* Name:			SelectFile()
* Purpose:		Navigate the file system and return the path and filename
*               of the selected file
* Arguments: 	*filePath - directory name of selected file
*               *fileName - name of selected file
*               *caption - display title, eg LOAD SNAPSHOT
*
***************************************************************************/
{
    int forecolor, backcolor;
    _Bool esc = false;
    SDL_Event event;
    static int lastKey;

    tinydir_dir dir;
    tinydir_file file;

    int i,ln, offset;
    offset = 0;
    int selection = 0;
    tinydir_open_sorted(&dir, ".");
    int numFiles = dir.n_files;
    char *path = dir.path;


    while (!esc)
    {

        ClearMenuScreen();
        DrawString(0,0,"                                        ",VZ_DKGREEN,VZ_GREEN);
        DrawString(16,0,caption,VZ_DKGREEN,VZ_GREEN);
        DrawString(0,19,"                                        ",VZ_DKGREEN,VZ_GREEN);
        DrawString(0,19,path,VZ_DKGREEN,VZ_GREEN);

        ln = 3;
        for (i = offset; i < numFiles; i++)
        {
            tinydir_readfile_n(&dir, &file, i);
            if (ln < 19)
            {
                forecolor = VZ_GREEN;
                backcolor = VZ_DKGREEN;

                if ((ln-3) == selection)
                {
                    forecolor = VZ_DKGREEN;
                    backcolor = VZ_GREEN;
                    DrawString(0,ln,"                    ",forecolor,backcolor);
                }

                if (file.is_dir)
                {
                    DrawString(0,ln,file.name,forecolor,backcolor);
                    DrawString(30,ln,"<DIR>",VZ_GREEN,VZ_DKGREEN);
                    ln++;
                }
                else
                {
                    DrawString(0,ln,file.name,forecolor,backcolor);
                    ln++;
                }
            }
        }

        const Uint8* g_keys = SDL_GetKeyboardState( NULL );
        SDL_PollEvent(&event);

        if ((g_keys[ SDL_SCANCODE_ESCAPE]) ||
            (g_keys[ SDL_SCANCODE_DOWN])  ||
            (g_keys[ SDL_SCANCODE_UP])    ||
            (g_keys[ SDL_SCANCODE_RETURN]))
        {
            // recognized key
        }
        else
        {
            lastKey = 0;
        }

        if (g_keys[ SDL_SCANCODE_ESCAPE])
        {
            if (lastKey != SDL_SCANCODE_ESCAPE)
            {
                lastKey = SDL_SCANCODE_ESCAPE;
                esc = true;
                ClearMenuScreen();
            }
        }
        if (g_keys[ SDL_SCANCODE_DOWN])
        {
            if (lastKey != SDL_SCANCODE_DOWN)
            {
                lastKey = SDL_SCANCODE_DOWN;

                if ((offset+selection) < (numFiles-1))
                {
                    selection++;
                    if (selection > 15)
                    {
                        selection = 15;
                        offset++;
                    }
                }
            }
        }
        if (g_keys[ SDL_SCANCODE_UP])
        {
            if (lastKey != SDL_SCANCODE_UP)
            {
                lastKey = SDL_SCANCODE_UP;

                if (selection > 0)
                {
                    selection--;
                }
                else
                {
                    if (offset > 0)
                    {
                        offset--;
                    }
                }
            }
        }
        if (g_keys[ SDL_SCANCODE_RETURN])
        {
            if (lastKey != SDL_SCANCODE_RETURN)
            {
                lastKey = SDL_SCANCODE_RETURN;
                int filen = offset+selection;
                tinydir_file file;
                tinydir_readfile_n(&dir, &file, filen);
                if (file.is_dir)
                {
                    tinydir_open_subdir_n(&dir, offset+selection);
                    numFiles = dir.n_files;
                    path = dir.path;
                    ln = 3;
                    offset = 0;
                    selection = 0;
                }
                else
                {
                    // selected a file
                    esc = true;
                    strcpy(filePath,dir.path);
                    strcpy(fileName,file.name);
                }
            }
        }

        osd_BlitBuffer(Display_Buffer);
    }
    tinydir_close(&dir);
}

void MOUNT_DISK()
/**************************************************************************
*
* Name:			MOUNT_DISK()
* Purpose:		Virtually insert a floppy disk into the drive
* Arguments: 	None. Calls SelectFile to navigate the file system
*               Returns to emulation when a file is selected
*
***************************************************************************/
{
    TCHAR   filePath[256];
    TCHAR   fileName[32];

   SelectFile(filePath,fileName,"MOUNT DISK");
   char *filespec = malloc(strlen(filePath) + strlen(fileName) + 2);
   strcpy(filespec,filePath);
   strcat(filespec,"/");
   strcat(filespec,fileName);
   vtech1_floppy_init(0,filespec);
   free(filespec);
   MENUACTIVE = false;

}

void LOAD_VZ()
/**************************************************************************
*
* Name:			LOAD_VZ()
* Purpose:		Load a .VZ snapshot file
* Arguments: 	None. Calls SelectFile to navigate the file system
*               Returns to emulation when a file is selected
*
***************************************************************************/
{
    TCHAR   filePath[256];
    TCHAR   fileName[32];

   SelectFile(filePath,fileName,"LOAD VZ");
   char *filespec = malloc(strlen(filePath) + strlen(fileName) + 2);
   strcpy(filespec,filePath);
   strcat(filespec,"/");
   strcat(filespec,fileName);
   LoadVZFile(filespec);
   free(filespec);
   MENUACTIVE = false;

}

void LOAD_CASSETTE()
/**************************************************************************
*
* Name:			LOAD_CASSETTE()
* Purpose:		Virtually insert a cassette (wav file) into the player.
* Arguments: 	None. Calls SelectFile to navigate the file system
*               Returns to emulation when a file is selected
*
***************************************************************************/
{
    TCHAR   filePath[256];
    TCHAR   fileName[32];

   SelectFile(filePath,fileName,"LOAD CASSETTE");
   char *filespec = malloc(strlen(filePath) + strlen(fileName) + 2);
   strcpy(filespec,filePath);
   strcat(filespec,"/");
   strcat(filespec,fileName);
   MapTape(filespec);
   free(filespec);
   MENUACTIVE = false;
}

void MAP_PRINTER()
/**************************************************************************
*
* Name:			MAP_PRINTER()
* Purpose:		Redirect printer output to a text file
* Arguments: 	None. Calls SelectFile to navigate the file system
*               Returns to emulation when a file is selected
*
***************************************************************************/
{
    TCHAR   filePath[256];
    TCHAR   fileName[32];

   SelectFile(filePath,fileName,"MAP PRINTER");
   char *filespec = malloc(strlen(filePath) + strlen(fileName) + 2);
   strcpy(filespec,filePath);
   strcat(filespec,"/");
   strcat(filespec,fileName);
   MapPrinter(filespec);
   free(filespec);
   MENUACTIVE = false;
}

void LOAD_ROM()
/**************************************************************************
*
* Name:			LOAD_ROM()
* Purpose:		Allow the user to load a custom ROM
* Arguments: 	None. Calls SelectFile to navigate the file system
*               Returns to emulation when a file is selected
*
***************************************************************************/
{
    TCHAR   filePath[256];
    TCHAR   fileName[32];

   SelectFile(filePath,fileName,"LOAD ROM");
   char *filespec = malloc(strlen(filePath) + strlen(fileName) + 2);
   strcpy(filespec,filePath);
   strcat(filespec,"/");
   strcat(filespec,fileName);
   LoadMem(0x0000,filespec);
   free(filespec);
   MENUACTIVE = false;
}

void LOAD_CARTRIDGE()
/**************************************************************************
*
* Name:			LOAD_CARTRIDGE()
* Purpose:		Allow the user to load a custom cartridge
* Arguments: 	None. Calls SelectFile to navigate the file system
*               Returns to emulation when a file is selected
*
***************************************************************************/
{
    TCHAR   filePath[256];
    TCHAR   fileName[32];

   SelectFile(filePath,fileName,"LOAD CARTRIDGE");
   char *filespec = malloc(strlen(filePath) + strlen(fileName) + 2);
   strcpy(filespec,filePath);
   strcat(filespec,"/");
   strcat(filespec,fileName);
   LoadMem(0x4000,filespec);
   free(filespec);
   MENUACTIVE = false;
}

void LOAD_FONT()
/**************************************************************************
*
* Name:			LOAD_FONT()
* Purpose:		Allow the user to load a custom font
* Arguments: 	None. Calls SelectFile to navigate the file system
*               Returns to emulation when a file is selected
*
***************************************************************************/
{
    TCHAR   filePath[256];
    TCHAR   fileName[32];

   SelectFile(filePath,fileName,"LOAD FONT");
   char *filespec = malloc(strlen(filePath) + strlen(fileName) + 2);
   strcpy(filespec,filePath);
   strcat(filespec,"/");
   strcat(filespec,fileName);
   LoadVZFont(filespec);
   free(filespec);
   MENUACTIVE = false;
}


void RESET()
/**************************************************************************
*
* Name:			RESET()
* Purpose:		Warm boot. Restart the Z80
* Arguments: 	None.
*
***************************************************************************/
{
    InitVZ();
    MENUACTIVE = false;
}

void QUIT()
/**************************************************************************
*
* Name:			QUIT()
* Purpose:		Quit the emulation. Will return to whatever process
*               called it, eg command line
* Arguments: 	None.
*
***************************************************************************/
{
    RUNNING = false;
    MENUACTIVE = false;
}

/**************************************************************************
*
* Name:			SET_PREFS
* Purpose:		Set options for sound, video, memory etc
* Arguments: 	Pref label
*
***************************************************************************/
void SET_PREFS_VZ200() { SetPrefs("VZ200"); }
void SET_PREFS_VZ300() { SetPrefs("VZ300"); }
void SET_PREFS_PAL()   { SetPrefs("PAL"); }
void SET_PREFS_NTSC() { SetPrefs("NTSC"); }
void SET_PREFS_8K() { SetPrefs("8K"); }
void SET_PREFS_18K() { SetPrefs("18K"); }
void SET_PREFS_24K() { SetPrefs("24K"); }
void SET_PREFS_34K() { SetPrefs("34K"); }
void SET_PREFS_4MB() { SetPrefs("4MB"); }
void SET_PREFS_6847_EFFECTS_ON() { SetPrefs("EFFECTS_ON"); }
void SET_PREFS_6847_EFFECTS_OFF() { SetPrefs("EFFECTS_OFF"); }
void SET_PREFS_FRAMERATE_NORMAL() { SetPrefs("NORMAL"); }
void SET_PREFS_FRAMERATE_MAXIMUM() { SetPrefs("MAXIMUM"); }
void SET_PREFS_SOUND_ON() { SetPrefs("SOUND_ON"); }
void SET_PREFS_SOUND_OFF() { SetPrefs("SOUND_OFF"); }
void SET_PREFS_CASSETTE_AUDIO_ON() { SetPrefs("CASSETTE_AUDIO_ON"); }
void SET_PREFS_CASSETTE_AUDIO_OFF() { SetPrefs("CASSETTE_AUDIO_OFF"); }
void SET_PREFS_DISK_AUDIO_ON() { SetPrefs("DISK_AUDIO_ON");}
void SET_PREFS_DISK_AUDIO_OFF() { SetPrefs("DISK_AUDIO_OFF"); }
void SET_PREFS_EXTENDED_GFX_NONE() { SetPrefs("NONE"); }
void SET_PREFS_EXTENDED_GFX_AUS() { SetPrefs("AUSTRALIAN"); }
void SET_PREFS_EXTENDED_GFX_GERMAN() { SetPrefs("GERMAN"); }
void SET_PREFS_WRITE_TO_ROM_ENABLED() { SetPrefs("WRITE_TO_ROM_ENABLED"); }
void SET_PREFS_WRITE_TO_ROM_DISABLED() { SetPrefs("WRITE_TO_ROM_DISABLED"); }
void SET_PREFS_WRITE_TO_CARTRIDGE_ENABLED() { SetPrefs("WRITE_TO_CARTRIDGE_ENABLED"); }
void SET_PREFS_WRITE_TO_CARTRIDGE_DISABLED() { SetPrefs("WRITE_TO_CARTRIDGE_ENABLED"); }


void DISPLAY_CONFIG()
/**************************************************************************
*
* Name:			DISPLAY_CONFIG()
* Purpose:		Display the current configs
* Arguments: 	none
*
***************************************************************************/
{
    typedef struct
{
	int vzmodel;				// 0 = VZ200, 1 = VZ300
	float cpu_speed;			// 3.58 or 3.54
	int display_type;			// pal=0 ntsc=1
	int gfx;					// normal=0 1=Leon 2 = Sorell 3=German
	int snow;					// no=0 yes=1
	int disk_drives;			// no=0 yes=1
	int joystick;				// no=0 1=arrows 2=directx
	int rom_writes;				// no=0 yes=1
	int cartridge_writes;		// no=0 yes=1
	int top_of_memory;			// 36863=8k 47103=18k 53247=24k 63487=34k
								// 65535=64k 5=4meg
	bool synchVZ;				// synchronise at normal 50fps
	bool cassetteAudio;			// play cassette tones
	bool diskAudio;				// play disk drive noises
	char romFile[MAX_PATH];
	char cartridgeFile[MAX_PATH];
	bool fast_load;
} Fprefs;

    extern Fprefs prefs;

    ClearMenuScreen();
    DrawString(0,0,"                                        ",VZ_DKGREEN,VZ_GREEN);
    DrawString(16,0,"CONFIG",VZ_DKGREEN,VZ_GREEN);

    DrawString(5,3,"MODEL        : ",VZ_GREEN,VZ_DKGREEN);
    if (prefs.vzmodel == 0) DrawString(25,3,"VZ200",VZ_GREEN,VZ_DKGREEN);
    if (prefs.vzmodel == 1) DrawString(25,3,"VZ300",VZ_GREEN,VZ_DKGREEN);

    DrawString(5,4,"MEMORY       : ",VZ_GREEN,VZ_DKGREEN);
    if (prefs.top_of_memory == 36863) DrawString(25,4,"8K",VZ_GREEN,VZ_DKGREEN);
    if (prefs.top_of_memory == 47103) DrawString(25,4,"18K",VZ_GREEN,VZ_DKGREEN);
    if (prefs.top_of_memory == 53247) DrawString(25,4,"24K",VZ_GREEN,VZ_DKGREEN);
    if (prefs.top_of_memory == 65535) DrawString(25,4,"34K",VZ_GREEN,VZ_DKGREEN);

    DrawString(5,5,"6847 EFFECTS : ",VZ_GREEN,VZ_DKGREEN);
    if (prefs.snow == 0) DrawString(25,5,"OFF",VZ_GREEN,VZ_DKGREEN);
    if (prefs.snow == 1) DrawString(25,5,"ON",VZ_GREEN,VZ_DKGREEN);

    DrawString(5,6,"FRAME RATE   : ",VZ_GREEN,VZ_DKGREEN);
    if (prefs.synchVZ == true) DrawString(25,6,"NORMAL",VZ_GREEN,VZ_DKGREEN);
    if (prefs.synchVZ == false) DrawString(25,6,"MAXIMUM",VZ_GREEN,VZ_DKGREEN);

    DrawString(5,7,"CASSETTE AUDIO : ",VZ_GREEN,VZ_DKGREEN);
    if (prefs.cassetteAudio == true) DrawString(25,7,"ON",VZ_GREEN,VZ_DKGREEN);
    if (prefs.cassetteAudio == false) DrawString(25,7,"OFF",VZ_GREEN,VZ_DKGREEN);

    DrawString(5,8,"DISK AUDIO : ",VZ_GREEN,VZ_DKGREEN);
    if (prefs.diskAudio == true) DrawString(25,8,"ON",VZ_GREEN,VZ_DKGREEN);
    if (prefs.diskAudio == false) DrawString(25,8,"OFF",VZ_GREEN,VZ_DKGREEN);

    DrawString(5,9,"EXTENDED GFX : ",VZ_GREEN,VZ_DKGREEN);
    if (prefs.gfx == 0) DrawString(25,9,"NONE",VZ_GREEN,VZ_DKGREEN);
    if (prefs.gfx == 1) DrawString(25,9,"AUSTRALIAN",VZ_GREEN,VZ_DKGREEN);
    if (prefs.gfx == 2) DrawString(25,9,"GERMAN",VZ_GREEN,VZ_DKGREEN);


    osd_BlitBuffer(Display_Buffer);
    SDL_Event event;
    bool esc = false;
    while (esc == false)
    {
        const Uint8* g_keys = SDL_GetKeyboardState( NULL );
        SDL_PollEvent(&event);
        if (g_keys[ SDL_SCANCODE_ESCAPE]) esc = true;
    }
    bool keydown = true;
    while (keydown)
    {
        const Uint8* g_keys = SDL_GetKeyboardState( NULL );
        SDL_PollEvent(&event);
        if (event.type == SDL_KEYUP) keydown = false;
    }
}


void SAVE_CONFIG()
/**************************************************************************
*
* Name:			SAVE_CONFIG()
* Purpose:		Save the current settings to vzem.cfg
* Arguments: 	none
*
***************************************************************************/
 {
    SavePrefs();
    MENUACTIVE = false;
 }



void LOAD_BITMAP()
/**************************************************************************
*
* Name:			LOAD_BITMAP()
* Purpose:		Load a mono 256x192 bitmap image to RAM at 0xC000
* Arguments: 	none
*
***************************************************************************/
 {
    // todo
   MENUACTIVE = false;
 }

void ABOUT()
/**************************************************************************
*
* Name:			ABOUT()
* Purpose:		splash screen
* Arguments: 	none
*
***************************************************************************/
{
    ClearMenuScreen();
    DrawString(0,0,"                                        ",VZ_DKGREEN,VZ_GREEN);
    DrawString(16,0,"ABOUT",VZ_DKGREEN,VZ_GREEN);
    DrawString(10,1,"    *** VZEM ***    ",VZ_GREEN,VZ_DKGREEN);
    DrawString(10,3,"VZ200/VZ300 EMULATOR",VZ_GREEN,VZ_DKGREEN);
    DrawString(10,4,"  BY GUY THOMASON   ",VZ_GREEN,VZ_DKGREEN);
    DrawString(10,6,"     CREDITS        ",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,8, "Z80  EMULATION  : MARAT FAYZULLIN ",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,9, "DISK EMULATION  : JUERGEN BUCHMUELLER ",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,10,"VIDEO TIMINGS   : RICHARD WILSON ",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,11,"VZ FILE FORMAT  : BRIAN MURRAY   ",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,13,"SPECIAL THANKS TO -   ",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,14,"  DAVID MAUNDER  ",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,15,"  JASON OAKLEY  ",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,17,"FOR THEIR CONTRIBUTIONS",VZ_GREEN,VZ_DKGREEN);
    DrawString(2,18,"SPANNING 4 DECADES",VZ_GREEN,VZ_DKGREEN);



    osd_BlitBuffer(Display_Buffer);
    SDL_Event event;
    bool esc = false;
    while (esc == false)
    {
        const Uint8* g_keys = SDL_GetKeyboardState( NULL );
        SDL_PollEvent(&event);
        if (g_keys[ SDL_SCANCODE_ESCAPE]) esc = true;
    }
    bool keydown = true;
    while (keydown)
    {
        const Uint8* g_keys = SDL_GetKeyboardState( NULL );
        SDL_PollEvent(&event);
        if (event.type == SDL_KEYUP) keydown = false;
    }


}

void INSTRUCTIONS()
/**************************************************************************
*
* Name:			INSTRUCTIONS()
* Purpose:		display usage/hotkeys
* Arguments: 	none
*
***************************************************************************/
 {
    ClearMenuScreen();
    DrawString(0,0,"                                        ",VZ_DKGREEN,VZ_GREEN);
    DrawString(13,0,"INSTRUCTIONS",VZ_DKGREEN,VZ_GREEN);
    DrawString(5,3,"USAGE: VZEM -F SNAPSHOT.VZ",VZ_GREEN,VZ_DKGREEN);
    DrawString(5,4,"            -D DISK.DSK",VZ_GREEN,VZ_DKGREEN);
    DrawString(5,5,"            -C CASSETTE.WAV",VZ_GREEN,VZ_DKGREEN);
    DrawString(5,7,"<TAB>     MENU",VZ_GREEN,VZ_DKGREEN);
    DrawString(5,8,"<F1>      PLAY CASSETTE",VZ_GREEN,VZ_DKGREEN);
    DrawString(5,9,"<F2>      RECORD CASSETTE",VZ_GREEN,VZ_DKGREEN);
    DrawString(5,10,"<F3>      STOP CASSETTE",VZ_GREEN,VZ_DKGREEN);


    osd_BlitBuffer(Display_Buffer);
    SDL_Event event;
    bool esc = false;
    while (esc == false)
    {
        const Uint8* g_keys = SDL_GetKeyboardState( NULL );
        SDL_PollEvent(&event);
        if (g_keys[ SDL_SCANCODE_ESCAPE]) esc = true;
    }
    bool keydown = true;
    while (keydown)
    {
        const Uint8* g_keys = SDL_GetKeyboardState( NULL );
        SDL_PollEvent(&event);
        if (event.type == SDL_KEYUP) keydown = false;
    }

 }


void menu()
/**************************************************************************
*
* Name:			menu()
* Purpose:		Render the main menu. Listens to SDL keys for navigation
* Arguments: 	none
*
***************************************************************************/
{
        SDL_Event event;
        _Bool bMainMenu;

       	char **activeMenu = (char**) MAIN_MENU;
       	bMainMenu = true;


        //osd_CreateScreen();
        ClearMenuScreen();

        int selection = 0;
        int numItems = 0;
        static int lastKey;

        lastKey = SDL_SCANCODE_ESCAPE;

        while (MENUACTIVE)
        {
       		int li = 0;
            while (activeMenu[li] != NULL)
            {
                if (li == selection)
                {
                    DrawString(10,3+li,"            ",VZ_DKGREEN,VZ_GREEN);
                    DrawString(13,3+li,activeMenu[li],VZ_DKGREEN,VZ_GREEN);
                }
                else
                {
                    DrawString(10,3+li,"            ",VZ_GREEN,VZ_DKGREEN);
                    DrawString(13,3+li,activeMenu[li],VZ_GREEN,VZ_DKGREEN);
                }
                li++;
            }
            numItems = li-1;

            const Uint8* g_keys = SDL_GetKeyboardState( NULL );
            SDL_PollEvent(&event);

            if ((g_keys[ SDL_SCANCODE_ESCAPE]) ||
                (g_keys[ SDL_SCANCODE_DOWN])  ||
                (g_keys[ SDL_SCANCODE_UP])    ||
                (g_keys[ SDL_SCANCODE_RETURN]))
            {
                // recognized key
            }
            else
            {
                lastKey = 0;
            }

            if (g_keys[ SDL_SCANCODE_ESCAPE])
            {
                if (lastKey != SDL_SCANCODE_ESCAPE)
                {
                    lastKey = SDL_SCANCODE_ESCAPE;
                    ClearMenuScreen();
                    if (bMainMenu)
                    {
                        MENUACTIVE = false;
                    }
                    else
                    {
                        activeMenu = (char**) MAIN_MENU;
                        bMainMenu = true;
                    }
                }
            }

            if (g_keys[ SDL_SCANCODE_DOWN])
            {
                if (lastKey != SDL_SCANCODE_DOWN)
                {
                    lastKey = SDL_SCANCODE_DOWN;
                    selection++;
                    if (selection > numItems) selection = numItems;
                }
            }

            if (g_keys[ SDL_SCANCODE_UP])
            {
                if (lastKey != SDL_SCANCODE_UP)
                {
                    lastKey = SDL_SCANCODE_UP;
                    selection--;
                    if (selection < 0) selection = 0;
                }
            }

            if (g_keys[ SDL_SCANCODE_RETURN])
            {
                if (lastKey != SDL_SCANCODE_RETURN)
                {
                    lastKey = SDL_SCANCODE_RETURN;
                    char *ch = activeMenu[selection];
                    int i = 0;
                    while (actions[i].selection != NULL)
                    {
                        if (strcmp(actions[i].selection,ch) == 0)
                        {
                            if (actions[i].nextMenu != NULL)
                            {
                                activeMenu = (char**) actions[i].nextMenu;
                                ClearMenuScreen();
                                break;
                            }
                            else
                            {
                                actions[i].pFunc();					// call the function
                                activeMenu = (char**) MAIN_MENU;	// return to main menu
                                ClearMenuScreen();
                                break;
                            }
                        }
                        i++;
                    }
                }
                selection = 0;
            }

            if(event.type == SDL_QUIT) break;
            osd_BlitBuffer(Display_Buffer);
        }
}
