#include "osdepend.h"
#include "SDL.h"
#include "SDL_audio.h"
#include <stdio.h>

#define AUDIO_BUFFER_SIZE       4410     // 441 * 10

extern byte AUDIO_BUFFER[AUDIO_BUFFER_SIZE];               // device independent circular sound buffer
extern int AUDIO_POS;                         // current position of device independent buffer


typedef unsigned int UInt32;

SDL_Window		*sdlWindow;
SDL_Renderer 	*sdlRenderer;
SDL_Texture  	*sdlTexture;
SDL_AudioSpec	device;
SDL_AudioDeviceID audioDeviceID;

void osd_CleanUp()
/**************************************************************************
*
* Name:			osd_CleanUp()
* Purpose:		release all resources
* Arguments: 	none
*
***************************************************************************/
{
	SDL_DestroyTexture(sdlTexture);
	SDL_DestroyRenderer(sdlRenderer);
	SDL_DestroyWindow(sdlWindow);
	SDL_Quit();
}

UInt32 osd_GetTicks()
/**************************************************************************
*
* Name:			osd_GetTicks()
* Purpose:		Used to lock frame rate to 20 ms
* Arguments: 	none
*
***************************************************************************/
{
	return SDL_GetTicks();
}

void fill_audio(void *udata, Uint8 *stream, int len)
/**************************************************************************
*
* Name:			fill_audio()
* Purpose:		SDL audio callback. Fill stream with VZ generated samples
* Arguments: 	internal. References global AUDIO_BUFFER and AUDIO_POS
*
***************************************************************************/
{

    for (int i=0;i<len;i++)
    {
        int pos = AUDIO_POS - len + i;
        if (pos < 0 ) pos += AUDIO_BUFFER_SIZE;
        stream[i] = AUDIO_BUFFER[pos];
    }
 }


int osd_CreateAudio()
/**************************************************************************
*
* Name:			osd_CreateAudio()
* Purpose:		Attempt to create an audio stream
* Arguments: 	returns 1 if audio failed
*
***************************************************************************/
{
    if( SDL_Init( SDL_INIT_AUDIO ) < 0 )
    {
        printf( "SDL could not create audio - SDL Error: %s\n", SDL_GetError() );
		return 1;
    }
	device.freq = SOUND_SAMPLE_RATE;
    device.format = AUDIO_U8;
    device.channels = 1;    /* 1 = mono, 2 = stereo */
    device.samples = 1024;  /* Good low-latency value for callback */
    device.callback = fill_audio; //fill_audio;
    device.userdata = NULL;

	SDL_AudioSpec obtained;
	audioDeviceID = SDL_OpenAudioDevice(NULL,0,&device,&obtained,0);
	if (audioDeviceID == 0)
	{
		printf("Failed to open audio: %s\n", SDL_GetError());
		return 1;
	}
	SDL_PauseAudioDevice(audioDeviceID,0);		// start playing!

}

int osd_CreateScreen()
/**************************************************************************
*
* Name:			osd_CreateScreen()
* Purpose:		Attempt to create the main screen
* Arguments: 	returns 1 if failed
*
***************************************************************************/
{
    //Initialize SDL
    if( SDL_Init( SDL_INIT_VIDEO ) < 0 )
    {
        printf( "SDL could not create a screen - SDL Error: %s\n", SDL_GetError() );
		return 1;
    }
	//SDL_CreateWindowAndRenderer(0, 0, SDL_WINDOW_FULLSCREEN_DESKTOP, &sdlWindow, &sdlRenderer);
	SDL_CreateWindowAndRenderer(320, 240, SDL_WINDOW_RESIZABLE, &sdlWindow, &sdlRenderer);
	SDL_SetWindowTitle(sdlWindow,"VZEM");
	SDL_SetHint(SDL_HINT_RENDER_SCALE_QUALITY, "linear");  // make the scaled rendering look smoother.
	SDL_RenderSetLogicalSize(sdlRenderer, 320, 240);
	sdlTexture = SDL_CreateTexture(sdlRenderer,
								   SDL_PIXELFORMAT_ARGB8888,
								   SDL_TEXTUREACCESS_STREAMING,
								   320, 240);
    return 0;
 }

void doNothing()
{
}

void osd_PlayTrack()
{
}

void osd_DiskStatus(int drive, char *s)
{
}

void osd_PlayStepper()
{
}

void osd_writeSoundStream(short sample)
{
}

void osd_UpdateAudioBuffer(byte *bufr, int len)
{
    SDL_QueueAudio(audioDeviceID,bufr,len);
}

void osd_ScanKbrd(byte *vzkbrd)
/**************************************************************************
*
* Name:			osd_ScanKbrd()
* Purpose:		Read the keyboard and map to VZ keys
* Arguments: 	*vzkbrd - pointer to byte arrary of keyboard matrix
*
***************************************************************************/
{
	byte k;
	extern bool RUNNING;
	extern bool MENUACTIVE;
	extern bool g_tapeplaying;

	SDL_Event e;
	const Uint8* g_keys = SDL_GetKeyboardState( NULL );
	SDL_PollEvent (&e);

	if ((e.type == SDL_WINDOWEVENT) && (e.window.event == SDL_WINDOWEVENT_CLOSE)) RUNNING = false;

    k = vzkbrd[0];
	if (g_keys[SDL_SCANCODE_R]) k = (k & 0xDF);
	if (g_keys[SDL_SCANCODE_Q] ) k = (k & 0xEF);
	if (g_keys[SDL_SCANCODE_E] ) k = (k & 0xF7);
	if (g_keys[SDL_SCANCODE_W] ) k = (k & 0xFD);
	if (g_keys[SDL_SCANCODE_T] ) k = (k & 0xFE);
	vzkbrd[0] = k;


    k = vzkbrd[1];
	if (g_keys[SDL_SCANCODE_F] ) k = (k & 0xDF);
	if (g_keys[SDL_SCANCODE_A] ) k = (k & 0xEF);
	if (g_keys[SDL_SCANCODE_D] ) k = (k & 0xF7);
	if (g_keys[SDL_SCANCODE_LCTRL]) k = (k & 0xFB);	// Left Control
	if (g_keys[SDL_SCANCODE_S] ) k = (k & 0xFD);
	if (g_keys[SDL_SCANCODE_G] ) k = (k & 0xFE);
	vzkbrd[1] = k;

    k = vzkbrd[2];
	if (g_keys[SDL_SCANCODE_V] ) k = (k & 0xDF);
	if (g_keys[SDL_SCANCODE_Z] ) k = (k & 0xEF);
	if (g_keys[SDL_SCANCODE_C] ) k = (k & 0xF7);
	if (g_keys[SDL_SCANCODE_LSHIFT]) k = (k & 0xFB);	// Left Shift
	if (g_keys[SDL_SCANCODE_X] ) k = (k & 0xFD);
	if (g_keys[SDL_SCANCODE_B] ) k = (k & 0xFE);
	vzkbrd[2] = k;

    k = vzkbrd[3];
	if (g_keys[SDL_SCANCODE_4] ) k = (k & 0xDF);
	if (g_keys[SDL_SCANCODE_1] ) k = (k & 0xEF);
	if (g_keys[SDL_SCANCODE_3] ) k = (k & 0xF7);
	if (g_keys[SDL_SCANCODE_2] ) k = (k & 0xFD);
	if (g_keys[SDL_SCANCODE_5] ) k = (k & 0xFE);
	vzkbrd[3] = k;

    k = vzkbrd[4];
	if (g_keys[SDL_SCANCODE_M] ) k = (k & 0xDF);
	if (g_keys[SDL_SCANCODE_SPACE] ) k = (k & 0xEF);
	if (g_keys[SDL_SCANCODE_COMMA] ) k = (k & 0xF7);
	if (g_keys[SDL_SCANCODE_PERIOD]  ) k = (k & 0xFD);
	if (g_keys[SDL_SCANCODE_N] ) k = (k & 0xFE);
	vzkbrd[4] = k;

    k = vzkbrd[5];
	if (g_keys[SDL_SCANCODE_7] ) k = (k & 0xDF);
	if (g_keys[SDL_SCANCODE_0] ) k = (k & 0xEF);
	if (g_keys[SDL_SCANCODE_8] ) k = (k & 0xF7);
	if (g_keys[SDL_SCANCODE_MINUS] ) k = (k & 0xFB);
	if (g_keys[SDL_SCANCODE_9] ) k = (k & 0xFD);
	if (g_keys[SDL_SCANCODE_6] ) k = (k & 0xFE);
	vzkbrd[5] = k;

    k = vzkbrd[6];
	if (g_keys[SDL_SCANCODE_U] ) k = (k & 0xDF); else k = (k | 0x20);
	if (g_keys[SDL_SCANCODE_P] ) k = (k & 0xEF); else k = (k | 0x10);
	if (g_keys[SDL_SCANCODE_I] ) k = (k & 0xF7); else k = (k | 0x08);
	if (g_keys[SDL_SCANCODE_RETURN] ) k = (k & 0xFB); else k = (k | 0x04);
	if (g_keys[SDL_SCANCODE_O] ) k = (k & 0xFD); else k = (k | 0x02);
	if (g_keys[SDL_SCANCODE_Y] ) k = (k & 0xFE); else k = (k | 0x01);
	vzkbrd[6] = k;

    k = vzkbrd[7];
	if (g_keys[SDL_SCANCODE_J] ) k = (k & 0xDF);
	if (g_keys[SDL_SCANCODE_SEMICOLON] ) k = (k & 0xEF);
	if (g_keys[SDL_SCANCODE_K] ) k = (k & 0xF7);
	if (g_keys[SDL_SCANCODE_APOSTROPHE ] ) k = (k & 0xFB);
	if (g_keys[SDL_SCANCODE_L] ) k = (k & 0xFD);
	if (g_keys[SDL_SCANCODE_H] ) k = (k & 0xFE);
	vzkbrd[7] = k;

   // extra keys

   if ((g_keys[ SDL_SCANCODE_LEFT]) || (g_keys[ SDL_SCANCODE_BACKSPACE]))
   {
	vzkbrd[4] &= 0xDF;              // press M
	vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (g_keys[ SDL_SCANCODE_RIGHT])
   {
	   vzkbrd[4] &= 0xF7;              // press comma
	   vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (g_keys[ SDL_SCANCODE_UP])
   {
	   vzkbrd[4] &= 0xFD;              // press period
	   vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (g_keys[ SDL_SCANCODE_DOWN])
   {
	   vzkbrd[4] &= 0xEF;              // press space
	   vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (g_keys[ SDL_SCANCODE_INSERT])
   {
	   vzkbrd[7] &= 0xFD;              // press L
	   vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (g_keys[ SDL_SCANCODE_DELETE])
   {
	vzkbrd[7] &= 0xEF;              // press semicolon
	vzkbrd[1] &= 0xFB;          // press ctrl
   }

   if (g_keys[ SDL_SCANCODE_END] )
   {
	vzkbrd[7] &= 0xFB;              // press apostrophe
	vzkbrd[1] &= 0xFB;          // press ctrl
   }

   //
   // tab key for debug menu
   //
   if (g_keys[ SDL_SCANCODE_TAB]) MENUACTIVE = 1;

   if (g_keys[ SDL_SCANCODE_F1]) PlayTape();
   if (g_keys[ SDL_SCANCODE_F2]) RecordTape();
   if (g_keys[ SDL_SCANCODE_F3]) StopTape();
}


void osd_BlitBuffer(byte *Buffer)
/**************************************************************************
*
* Name:			osd_BlitBuffer()
* Purpose:		Draw the device independent bitmap to the SDL surface
* Arguments: 	*Buffer - pointer to bitmap buffer
*
***************************************************************************/
{
	byte	colorIndex;
	Uint32	screen[76800];					// device independent 320*240 back buffer for 32 bit pixels
	long	offset;
	extern	Uint32 palette[];
	extern	SDL_Renderer 	*sdlRenderer;
	extern	SDL_Texture  	*sdlTexture;


	for (int y=0; y<240; y++)
	{
		for (int x=0;x<320;x++)
		{
			offset = 320*y+x;
			colorIndex =  Buffer[offset];
			screen[offset] = palette[colorIndex];
		}
	}

	SDL_UpdateTexture(sdlTexture, NULL, screen, 320 * sizeof (Uint32));
	SDL_RenderCopy(sdlRenderer, sdlTexture, NULL, NULL);
	SDL_RenderPresent(sdlRenderer);
}

void osd_synchSound()
{
}

bool osd_SynchVZ()
{
	return true;
}

void osd_InitTimer()
{
}
