#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <dirent.h>
#include "Z80.h"
#include "vzem.h"

/*****************************************

			Defines

******************************************/

#define SOUND_FREQ              22050
#define AUDIO_SAMPLES_PER_FRAME 441     // 22050 / 50
#define AUDIO_BUFFER_SIZE       4410     // 441 * 10
#define	CYCLES_PER_SCANLINE     228	    // video - draw a scanline every 228 cpu cycles
#define	CYCLES_PER_TAPEBYTE     145		// cassette - read/write a wav sample every 145 cpu cycles
#define CYCLES_PER_AUDIOSAMPLE  163		// speaker - write a audio sample every 163 cpu cycles
#define MAX_PATH                512
#define BPP                     4
#define DEPTH                   32

#define M6847_RGB(r,g,b)	((r << 16) | (g << 8) | (b << 0))

#define		VZ_GREEN	0
#define		VZ_YELLOW	1
#define		VZ_BLUE		2
#define		VZ_RED		3
#define		VZ_BUFF		4
#define		VZ_CYAN		5
#define		VZ_MAGENTA	6
#define		VZ_ORANGE	7
#define		VZ_BLACK	8
#define		VZ_DKGREEN	9
#define		VZ_BRGREEN	10
#define		VZ_DKORANGE 11
#define		VZ_BRORANGE 12

#define Uint32 unsigned int
#define byte unsigned char
#define bool _Bool
#define true 1
#define false 0


/*****************************************

			Globals

******************************************/

extern byte *RAM;									// pointer to 64K address space. Defined in Z80 lib

byte AUDIO_BUFFER[AUDIO_BUFFER_SIZE];               // device independent circular sound buffer
byte SOUND_LEVELS[] = { 0xff, 0xff, 0x00, 0x00 };	// square wave sound data
int AUDIO_SAMPLES_THIS_FRAME;						// current count of audio samples generated for the active frame
int AUDIO_POS;                                      // current position of device independent buffer

// samples for writing to cassette. Tested to work on a real vz300
static byte SHORT_HIGH[] = { 0x9D, 0xB1, 0xBC, 0xBD, 0xC1, 0xAC };
static byte SHORT_LOW[] =  { 0x70, 0x54, 0x49, 0x43, 0x43, 0x41, 0x6D };
static byte LONG_HIGH[] =  { 0x8B, 0xAA, 0xB7, 0xBC, 0xBE, 0xBE, 0xBD, 0xBC, 0xBB, 0xBB, 0xB8, 0xBA, 0x8D };
static byte LONG_LOW[] =   { 0x59, 0x48, 0x3E, 0x3C, 0x3B, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x41, 0x42, 0x76 };


Z80 REGS;						// Z80 Registers
byte VZMEM[65536];				// VZ system memory
byte BKMEM[4194304];			// Bank memory (4 meg!)
byte GXMEM[8192];				// Graphics memory - used for hardware mods
byte VZKBRD[8];					// VZ keyboard array
byte VZ_LATCH;					// copy of hardware latch
byte GFX_MOD_LATCH;				// latch for gfx hardware mod if enabled

bool RUNNING;					// emulation running
bool MENUACTIVE;                // console menu active
int SCANLINE;					// current scanline being drawn
int SCANLINEOFFSET;				// adjustment if gfx mode has changed during screen draw
int UPDATEBORDER;				// check for screen mode change during retrace
int RENDER;                     // does the screen need to be drawn
byte AUDIOSOUNDSAMPLE;			// value written to speaker
byte JOYSTICKBYTE;				// value read from joystick
int SPKRWRITES = 0;
int ACTIVE_PAGE = 0;			// bank memory page (0 - 255)
int DRAWFRAMES = 0;

FILE *tape;						// handle to file
char wavTape[MAX_PATH];			// name of wavfile
unsigned long g_tapeBytesWritten;	// size of wavfile in bytes

byte g_tapeByte;				// value read/written to tape
bool g_tapeplaying = false;		// is tape playing
bool g_taperecording = false;	// is tape recording

FILE *prt;						// handle to printer file
char prtFile[80];				// name of printer file

typedef struct
{
	int vzmodel;				// 0 = VZ200, 1 = VZ300
	float cpu_speed;			// 3.58 or 3.54
	int display_type;			// pal=0 ntsc=1
	int gfx;					// normal=0 1=Leon 2 = Sorell 3=German
	int snow;					// no=0 yes=1
	int disk_drives;			// no=0 yes=1
	int joystick;				// no=0 1=arrows 2=directx
	int rom_writes;				// no=0 yes=1
	int cartridge_writes;		// no=0 yes=1
	int top_of_memory;			// 36863=8k 47103=18k 53247=24k 63487=34k
								// 65535=64k 5=4meg
	bool synchVZ;				// synchronise at normal 50fps
	bool cassetteAudio;			// play cassette tones
	bool diskAudio;				// play disk drive noises
	char romFile[MAX_PATH];
	char cartridgeFile[MAX_PATH];
	bool fast_load;
} Fprefs;

Fprefs prefs;

typedef struct
{
	int lines_blanking;
	int lines_dummy_top;
	int lines_top_border;
	int lines_display;
	int lines_bottom_border;
	int lines_dummy_bottom;
	int lines_retrace;
	int total_lines;
	int FS;
	int OFF;
} Display;

Display display;

/*****************************************

	Screen specific definitions

******************************************/

// A "pixel" is drawn to the display buffer by assigning a color index from
// the list below.
// Eg Display_Buffer[320*y+x] = 3 will set a RED pixel

byte Display_Buffer[77760];		// 320*243 device independent display buffer.
								// Each x,y coordinate stores a VZ color, eg
								// VZ_BLUE


Uint32 palette[] =
{
	M6847_RGB(0x00, 0xff, 0x00),	/* GREEN */
	M6847_RGB(0xff, 0xff, 0x00),	/* YELLOW */
	M6847_RGB(0x00, 0x00, 0xff),	/* BLUE */
	M6847_RGB(0xff, 0x00, 0x00),	/* RED */
	M6847_RGB(0xff, 0xff, 0xff),	/* BUFF */
	M6847_RGB(0x00, 0xff, 0xff),	/* CYAN */
	M6847_RGB(0xff, 0x00, 0xff),	/* MAGENTA */
	M6847_RGB(0xff, 0x80, 0x00),	/* ORANGE */
	M6847_RGB(0x00, 0x00, 0x00),	/* BLACK */
	M6847_RGB(0x00, 0x40, 0x00),	/* ALPHANUMERIC DARK GREEN */
	M6847_RGB(0xA0, 0xA0, 0xA0),	/* ALPHANUMERIC BRIGHT GREEN */
	M6847_RGB(0x40, 0x10, 0x00),	/* ALPHANUMERIC DARK ORANGE */
	M6847_RGB(0xff, 0xc4, 0x18)		/* ALPHANUMERIC BRIGHT ORANGE */
};


byte pal_square_fontdata8x12[] =
{
	// text characters
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x1A, 0x2A, 0x2A, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x14, 0x22, 0x22, 0x3E, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x12, 0x12, 0x1C, 0x12, 0x12, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x20, 0x20, 0x20, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x12, 0x12, 0x12, 0x12, 0x12, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x20, 0x3C, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x20, 0x3C, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1E, 0x20, 0x20, 0x26, 0x22, 0x22, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x3E, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x24, 0x28, 0x30, 0x28, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x36, 0x2A, 0x2A, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x32, 0x2A, 0x26, 0x22, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x22, 0x22, 0x22, 0x22, 0x22, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x3C, 0x20, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x22, 0x2A, 0x24, 0x1A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x3C, 0x28, 0x24, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x10, 0x08, 0x04, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x14, 0x14, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x2A, 0x2A, 0x36, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x08, 0x14, 0x22, 0x22, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x22, 0x22, 0x14, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x02, 0x04, 0x08, 0x10, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x20, 0x20, 0x10, 0x08, 0x04, 0x02, 0x02, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x0E, 0x02, 0x02, 0x02, 0x02, 0x02, 0x0E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x1C, 0x2A, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x10, 0x3E, 0x10, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x14, 0x36, 0x00, 0x36, 0x14, 0x14, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x1E, 0x20, 0x1C, 0x02, 0x3C, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x32, 0x32, 0x04, 0x08, 0x10, 0x26, 0x26, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x28, 0x28, 0x10, 0x2A, 0x24, 0x1A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x10, 0x20, 0x20, 0x20, 0x10, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x04, 0x02, 0x02, 0x02, 0x04, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x1C, 0x3E, 0x1C, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x3E, 0x08, 0x08, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x10, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x02, 0x02, 0x04, 0x08, 0x10, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x24, 0x24, 0x24, 0x24, 0x24, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x08, 0x18, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x1C, 0x20, 0x20, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x02, 0x0C, 0x02, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x0C, 0x14, 0x3E, 0x04, 0x04, 0x04, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x20, 0x3C, 0x02, 0x02, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x02, 0x04, 0x08, 0x10, 0x20, 0x20, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x1C, 0x22, 0x22, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1C, 0x22, 0x22, 0x1E, 0x02, 0x02, 0x1C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x04, 0x08, 0x10, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x08, 0x04, 0x02, 0x04, 0x08, 0x10, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x24, 0x04, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,
	// semigraphics
	0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x00,0x00,0x00,0x00,0x00,0x00, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0x00,0x00,0x00,0x00,0x00,0x00, 0xff,0xff,0xff,0xff,0xff,0xff,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0x00,0x00,0x00,0x00,0x00,0x00,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f, 0xff,0xff,0xff,0xff,0xff,0xff,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0x00,0x00,0x00,0x00,0x00,0x00,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0xf0,0xf0,0xf0,0xf0,0xf0,0xf0, 0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff, 0x00,0x00,0x00,0x00,0x00,0x00,
	0xff,0xff,0xff,0xff,0xff,0xff, 0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
	0xff,0xff,0xff,0xff,0xff,0xff, 0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
	0xff,0xff,0xff,0xff,0xff,0xff, 0xff,0xff,0xff,0xff,0xff,0xff
};


/**********************************************************************

	VZDIO Disk Emulation Routines

	These routines are taken from the video technology MESS driver
	by Juergen Buchmueller

**********************************************************************/

#define TRKSIZE_VZ      0x9b0
#define TRKSIZE_FM      3172	/* size of a standard FM mode track */

FILE *vtech1_fdc_file[3] = { NULL, NULL };
byte vtech1_track_x2[2] = { 80, 80 };
byte vtech1_fdc_wrprot[2] = { 0x80, 0x80 };

byte vtech1_fdc_status = 0;
byte vtech1_fdc_data[TRKSIZE_FM];

static int vtech1_data;

static int vtech1_fdc_edge = 0;
static int vtech1_fdc_bits = 8;
static int vtech1_drive = -1;
static int vtech1_fdc_start = 0;
static int vtech1_fdc_write = 0;
static int vtech1_fdc_offs = 0;
static int vtech1_fdc_latch = 0;


void vtech1_floppy_remove(int id)
{
	if (vtech1_fdc_file[id])
		fclose(vtech1_fdc_file[id]);
	vtech1_fdc_file[id] = NULL;
}

int vtech1_floppy_init(int id, char *s)
{
	/* first try to open existing image RW */
	vtech1_fdc_wrprot[id] = 0x00;
	vtech1_fdc_file[id] = fopen(s, "rb+");
	/* failed? */
	if (!vtech1_fdc_file[id])
	{
		/* try to open existing image RO */
		vtech1_fdc_wrprot[id] = 0x80;
		vtech1_fdc_file[id] = fopen(s, "rb");
	}

	/* failed? */
	if (!vtech1_fdc_file[id])
	{
		/* create new image RW */
		vtech1_fdc_wrprot[id] = 0x00;
		vtech1_fdc_file[id] = fopen(s, "wb+");
	}
	if (vtech1_fdc_file[id])
	{
		vtech1_track_x2[id] = 0;
		return 0;
	}
	/* failed permanently */
	return 1;
}

void vtech1_floppy_exit(int id)
{
	if (vtech1_fdc_file[id])
		fclose(vtech1_fdc_file[id]);
	vtech1_fdc_file[id] = NULL;
}

static void vtech1_get_track(void)
{
	// sprintf(vtech1_frame_message, "#%d get track %02d", vtech1_drive,
	// vtech1_track_x2[vtech1_drive]/2);
	/* drive selected or and image file ok? */
	int size, offs, r;
	FILE *f;
	if (vtech1_drive >= 0 && vtech1_fdc_file[vtech1_drive] != NULL)
	{
		size = TRKSIZE_VZ;
		offs = TRKSIZE_VZ * vtech1_track_x2[vtech1_drive] / 2;
		r = fseek(vtech1_fdc_file[vtech1_drive], offs, SEEK_SET);
		f = vtech1_fdc_file[vtech1_drive];
		size = fread(&vtech1_fdc_data, size, 1, f);
		// if (prefs.diskAudio) osd_PlayTrack();

		char TS[16];
		//sprintf(TS, "  T:%02d S:%02d", VZMEM[REGS.IY.W + 18], VZMEM[REGS.IY.W + 17]);
		// osd_DiskStatus(vtech1_drive, TS);

	}
	else
	{
		for (int n = 0; n < TRKSIZE_VZ; n++)
			vtech1_fdc_data[n] = 0x00;
	}
	vtech1_fdc_offs = 0;
	vtech1_fdc_write = 0;
}

static void vtech1_put_track(void)
{
	/* drive selected and image file ok? */
	if (vtech1_drive >= 0 && vtech1_fdc_file[vtech1_drive] != NULL)
	{
		int size, offs;
		offs = TRKSIZE_VZ * vtech1_track_x2[vtech1_drive] / 2;
		fseek(vtech1_fdc_file[vtech1_drive], offs + vtech1_fdc_start, SEEK_SET);
		size =
			fwrite(&vtech1_fdc_data[vtech1_fdc_start], 1, vtech1_fdc_write,
				   vtech1_fdc_file[vtech1_drive]);
		// if (prefs.diskAudio) osd_PlayTrack();

		char TS[16];
		//sprintf(TS, "  T:%02d S:%02d", VZMEM[REGS.IY.W + 18], VZMEM[REGS.IY.W + 17]);
		// osd_DiskStatus(vtech1_drive, TS);

	}
	else
	{
		for (int n = 0; n < TRKSIZE_VZ; n++)
			vtech1_fdc_data[n] = 0x00;
	}
}

#define PHI0(n) (((n)>>0)&1)
#define PHI1(n) (((n)>>1)&1)
#define PHI2(n) (((n)>>2)&1)
#define PHI3(n) (((n)>>3)&1)

int vtech1_fdc_r(int offset)
{
	int data = 0xff;
	switch (offset)
	{
	case 1:					/* data (read-only) */
		if (vtech1_fdc_bits > 0)
		{
			if (vtech1_fdc_status & 0x80)
				vtech1_fdc_bits--;
			data = (vtech1_data >> vtech1_fdc_bits) & 0xff;
		}
		if (vtech1_fdc_bits == 0)
		{
			vtech1_data = vtech1_fdc_data[vtech1_fdc_offs];
			if (vtech1_fdc_status & 0x80)
			{
				vtech1_fdc_bits = 8;
				vtech1_fdc_offs = (vtech1_fdc_offs + 1) % TRKSIZE_FM;
			}
			vtech1_fdc_status &= ~0x80;
		}
		break;
	case 2:					/* polling (read-only) */
		/* fake */
		if (vtech1_drive >= 0)
			vtech1_fdc_status |= 0x80;
		data = vtech1_fdc_status;
		break;
	case 3:					/* write protect status (read-only) */
		if (vtech1_drive >= 0)
			data = vtech1_fdc_wrprot[vtech1_drive];
		break;
	}
	return data;
}


void vtech1_fdc_w(int offset, int data)
{
	int drive;

	switch (offset)
	{
	case 0:					/* latch (write-only) */
		drive = (data & 0x10) ? 0 : (data & 0x80) ? 1 : -1;
		if (drive != vtech1_drive)
		{
			vtech1_drive = drive;
			if (vtech1_drive >= 0)
				vtech1_get_track();
		}
		if (vtech1_drive >= 0)
		{
			if ((PHI0(data) && !(PHI1(data) || PHI2(data) || PHI3(data)) && PHI1(vtech1_fdc_latch))
				|| (PHI1(data) && !(PHI0(data) || PHI2(data) || PHI3(data))
					&& PHI2(vtech1_fdc_latch)) || (PHI2(data) && !(PHI0(data) || PHI1(data)
																   || PHI3(data))
												   && PHI3(vtech1_fdc_latch)) || (PHI3(data)
																				  && !(PHI0(data)
																					   ||
																					   PHI1(data)
																					   ||
																					   PHI2(data))
																				  &&
																				  PHI0
																				  (vtech1_fdc_latch)))
			{
				if (vtech1_track_x2[vtech1_drive] > 0)
				{
					vtech1_track_x2[vtech1_drive]--;
				}
				if ((vtech1_track_x2[vtech1_drive] & 1) == 0)
				{
					vtech1_get_track();
					// if (prefs.diskAudio) osd_PlayStepper();
				}
			}
			else if ((PHI0(data) && !(PHI1(data) || PHI2(data) || PHI3(data))
					  && PHI3(vtech1_fdc_latch)) || (PHI1(data) && !(PHI0(data) || PHI2(data)
																	 || PHI3(data))
													 && PHI0(vtech1_fdc_latch)) || (PHI2(data)
																					&& !(PHI0(data)
																						 ||
																						 PHI1(data)
																						 ||
																						 PHI3
																						 (data))
																					&&
																					PHI1
																					(vtech1_fdc_latch))
					 || (PHI3(data) && !(PHI0(data) || PHI1(data) || PHI2(data))
						 && PHI2(vtech1_fdc_latch)))
			{
				if (vtech1_track_x2[vtech1_drive] < 2 * 40)
				{
					vtech1_track_x2[vtech1_drive]++;
				}
				if ((vtech1_track_x2[vtech1_drive] & 1) == 0)
				{
					vtech1_get_track();
					// if (prefs.diskAudio) osd_PlayStepper();
				}
			}
			if ((data & 0x40) == 0)
			{
				vtech1_data <<= 1;
				if ((vtech1_fdc_latch ^ data) & 0x20)
					vtech1_data |= 1;
				if ((vtech1_fdc_edge ^= 1) == 0)
				{
					if (--vtech1_fdc_bits == 0)
					{
						byte value = 0;
						vtech1_data &= 0xffff;
						if (vtech1_data & 0x4000)
							value |= 0x80;
						if (vtech1_data & 0x1000)
							value |= 0x40;
						if (vtech1_data & 0x0400)
							value |= 0x20;
						if (vtech1_data & 0x0100)
							value |= 0x10;
						if (vtech1_data & 0x0040)
							value |= 0x08;
						if (vtech1_data & 0x0010)
							value |= 0x04;
						if (vtech1_data & 0x0004)
							value |= 0x02;
						if (vtech1_data & 0x0001)
							value |= 0x01;
						vtech1_fdc_data[vtech1_fdc_offs] = value;
						vtech1_fdc_offs = (vtech1_fdc_offs + 1) % TRKSIZE_FM;
						vtech1_fdc_write++;
						vtech1_fdc_bits = 8;
					}
				}
			}
			/* change of write signal? */
			if ((vtech1_fdc_latch ^ data) & 0x40)
			{
				/* falling edge? */
				if (vtech1_fdc_latch & 0x40)
				{
					vtech1_fdc_start = vtech1_fdc_offs;
					vtech1_fdc_edge = 0;
				}
				else
				{
					/* data written to track before? */
					if (vtech1_fdc_write)
						vtech1_put_track();
				}
				vtech1_fdc_bits = 8;
				vtech1_fdc_write = 0;
			}
		}
		vtech1_fdc_latch = data;
		break;
	}
}


byte readVZKbrd(int A)
/**************************************************************************
*
* Name:			readVZKbrd()
* Purpose:		Read a line of the keyboard matrix
* Arguments: 	A - address of the keyboard line
*
***************************************************************************/
{
	byte	k = 0xFF;

	if (!(A & 0x0001)) k &= VZKBRD[0];
	if (!(A & 0x0002)) k &= VZKBRD[1];
	if (!(A & 0x0004)) k &= VZKBRD[2];
	if (!(A & 0x0008)) k &= VZKBRD[3];
	if (!(A & 0x0010)) k &= VZKBRD[4];
	if (!(A & 0x0020)) k &= VZKBRD[5];
	if (!(A & 0x0040)) k &= VZKBRD[6];
	if (!(A & 0x0080)) k &= VZKBRD[7];

	return k;
}


byte RdZ80(word A)
/**************************************************************************
*
* Name:			RdZ80()
* Purpose:		Called by emu library to fetch a byte from memory
* Arguments: 	A - 16 bit address. Returns a byte
*
***************************************************************************/
{
	byte	b;

	b = RAM[A];

	if ((A > 0x67FF) && (A < 0x7000))		// reading keyboard
	{
		b = readVZKbrd(A);

		if (SCANLINE > display.OFF)
		{
			b = (b & 0x7f);					// retrace
		}
		if (g_tapeByte >  0xB5)				// best result for 8 bit unsigned 22050 wav file
		{
			b = (b & 0xbf);
		}
	}

    if ((A > 0x6FFF) && (A < 0x7800))		// screen memory
    {
		if (prefs.gfx > 0)
		{
			int bank = GFX_MOD_LATCH & 0x03;
			b = GXMEM[A-0x7000+bank*0x800];
		}
	}

	if (A > prefs.top_of_memory)
	{
		b = 0x7f;
	}

    return b;
}


void WrZ80(word A, byte v)
/**************************************************************************
*
* Name:			WrZ80()
* Purpose:		Called by emu library to write a byte to memory
* Arguments: 	A - 16 bit address. Returns a byte
*
***************************************************************************/
{
	// German GFX Mod writes to 30779
	if (prefs.gfx == 2)
	{
		if (A == 0x783B)
		{
			if ((v & 0x08) == 0x08)
			{
				GFX_MOD_LATCH |= 28;
				RENDER = 2;
			}
		}
	}

    //
    // writing to screen memory
    //
    if ((A > 0x6FFF) && (A < 0x7800))
    {
		RENDER = 2;
		//
		// produce snow if display active.
		// Note. could be in the middle of an LDIR, so SCANLINE
		// may not exactly match calculated scanline
		//

		if (prefs.snow == 1)
		{
			if (v != 0x00)
			{
				int display_start = display.lines_blanking +
									display.lines_dummy_top +
									display.lines_top_border;

				int elapsedCycles = REGS.IPeriod - REGS.ICount;
				int scanline = (elapsedCycles / CYCLES_PER_SCANLINE);
				int cyclesInScanline = elapsedCycles - (scanline * CYCLES_PER_SCANLINE);

				if ((scanline >= display_start) && (scanline <= display.OFF))
				{
					float pos = ((float)cyclesInScanline) / CYCLES_PER_SCANLINE;
					int xpos = (int)(pos*32);
					int offs = 320*(scanline-(display.lines_blanking + display.lines_dummy_top))+32+xpos*8;

					for (int x=0;x<8;x++)
					{
						Display_Buffer[offs+x] = VZ_BRGREEN;
					}
				}
			}
		}
		if (prefs.gfx > 0)
		{
			int bank = GFX_MOD_LATCH & 0x03;
			GXMEM[A-0x7000+bank*0x800] = v;
		}
    }

	// trap writes to ROM or cartridge memory

	if ((A < 0x4000) && (prefs.rom_writes == 1))
	{
		RAM[A] = v;
	}

	if ((A > 0x3FFF) && (A < 0x6800) && (prefs.cartridge_writes == 1))
	{
		RAM[A] = v;
	}


	if ((A > 0x67FF) && (A <= prefs.top_of_memory))
	{
		RAM[A] = v;
	}

	// writing to latch
	if ((A > 0x67FF) && (A < 0x7000))
	{
		// speaker bits toggle?
		if( (VZ_LATCH ^ v ) & 0x21 )
		{
	        AUDIOSOUNDSAMPLE = SOUND_LEVELS[(v & 0x01) | ((v >> 4) & 0x02)];
		}

		if( (VZ_LATCH ^ v ) & 0x06 )
		{

			short tapeSample = SOUND_LEVELS[((v >> 1) & 0x01) | ((v >> 1) & 0x02)];
			switch (tapeSample)
			{
				case 32767:		g_tapeByte = 0xFF;
								break;
				case -32768:	g_tapeByte = 0x00;
								break;
			}
			WriteCassette(g_tapeByte);
		}

		// If you change the screen mode during the retrace period
		// it won't come into effect until the display is turned back on.
		// So the top border will not be drawn in the new mode

		// calculate the scanline accurately, as SCANLINE will not be accurate
		// during an LDIR

		if ((VZ_LATCH & 0x08) != (v & 0x08))
		{
			RENDER = 2;

			int scanline = (REGS.IPeriod - REGS.ICount) / CYCLES_PER_SCANLINE;
			if ((scanline >= display.FS) ||
				(scanline < (display.lines_blanking + display.lines_top_border)))
			{
				UPDATEBORDER = VZ_LATCH;
			}
            // are we changing graphics modes while the display is being drawn
            // switching from text to graphics
            if ((SCANLINE > display.lines_blanking + display.lines_dummy_top + display.lines_top_border) &&
                    (SCANLINE < display.FS))
            {
                    SCANLINEOFFSET = SCANLINE - (display.lines_blanking + display.lines_dummy_top + display.lines_top_border);
                    SCANLINEOFFSET -= 12;                // adjust for 1 line of text
            }
		}
		VZ_LATCH = v;
	}
}


byte InZ80(word P)
/**************************************************************************
*
* Name:			InZ80()
* Purpose:		Called by emu library to read a hardware port
* Arguments: 	P - 16 bit port. Only lower 8 bits are used
*
***************************************************************************/
{

    byte b = 0xFF;

    //
    // Disk reads
    //

    if ((P >= 0x10) && (P <= 0x1F))
    {
		b = vtech1_fdc_r(P - 0x10);
    }

	//
	// Printer port
	//

    if ((P >= 0x00) && (P <= 0x0F))
    {
		if 	(prt != NULL)		// have we mapped to an output file
		{
			b &= ~0x01;
		}
    }

	//
	// Joystick port
	//
	bool bFIRE = false;
	if (prefs.joystick == 1)
	{
		if ((JOYSTICKBYTE & 0x80) == 0x00)		// hacked byte for FIRE signal
		{
			bFIRE = true;
			JOYSTICKBYTE = (JOYSTICKBYTE | 0x80);
		}
		if ((P >= 0x20) && (P <= 0x2F))
		{
			word offs = P - 0x20;
			b = JOYSTICKBYTE;
		}
	}

    return b;
}

void OutZ80(word P, byte B)
/**************************************************************************
*
* Name:			OutZ80()
* Purpose:		Called by emu library to write to a hardware port
* Arguments: 	P - 16 bit port. Only lower 8 bits are used
*               B - output byte
*
***************************************************************************/
{

	//
	// Is graphics mod enabled?
	//
	if (prefs.gfx == 1)
	{
		if (P == 0x20)			// output port for gfx mod
		{
			GFX_MOD_LATCH = B;
		}
	}

	if (prefs.gfx == 2)
	{
		if (P == 0xDE)			// output port for gfx mod
		{
			GFX_MOD_LATCH &= 0xFC;			// mask first 2 bits
			GFX_MOD_LATCH |= (B & 0x03);	// and select bank
		}
	}

	//
    // Disk writes
    //

    if ((P >= 0x10) && (P <= 0x1F))
    {
       vtech1_fdc_w(P - 0x10, B);
    }

	//
	// Printer
	//

    if ((P >= 0x00) && (P <= 0x0F))
    {
		if 	(prt != NULL)		// have we mapped to an output file
		{
			if ((P == 0x0E) || (P == 0x00))
			{
				fwrite(&B,1,1,prt);
			}
		}
    }

    //
    // Memory Bank switch
    //

    static int current_page = 1;
    int i;
    long cpoffs, apoffs;

    if ((P >= 0x70) && ( P <= 0x7F) && (prefs.top_of_memory == 65535))
    {
		if (B == 0) B = 1;
        ACTIVE_PAGE = B;
        cpoffs = current_page * 16384;
        apoffs = ACTIVE_PAGE * 16384;
        for (i = 0; i < 16384; i++)
        {
           // copy current page to bank memory
           BKMEM[cpoffs+i] = VZMEM[0xC000 + i];

           // copy bank memory to selected page
           VZMEM[0xC000 + i] = BKMEM[apoffs + i];
        }

        current_page = ACTIVE_PAGE;
    }
}

void PatchZ80(Z80 *R)
{
}


void GenInterrupt(Z80 * R)
/**************************************************************************
*
* Name:			GenInterrupt()
* Purpose:		Called by emu library to generate a cpu interrupt
* Arguments: 	R - Z80 register structure
*
***************************************************************************/
{
	register pair J;

	J.W = 0;

	/* If we have come after EI, get address from IRequest */
	/* Otherwise, get it from the loop handler */
	if (R->IFF & 0x20)
	{
		J.W = R->IRequest;		/* Get pending interrupt */
		R->ICount += R->IBackup - 1;	/* Restore the ICount */
		R->IFF &= 0xDF;			/* Done with AfterEI state */
	}

	if (J.W == INT_QUIT)
		return;					/* Exit if INT_QUIT */
	if (J.W != INT_NONE)
	{
		IntZ80(R, J.W);			/* Int-pt if needed */
	}
}


void DrawBorder()
/**************************************************************************
*
* Name:			DrawBorder()
* Purpose:		Draw the text/graphics background border
* Arguments: 	None. References global variable SCANLINE
*
***************************************************************************/
{
	int		mode = UPDATEBORDER & 0x08;
	int		bgcolor = UPDATEBORDER & 0x10;
	int		bdcolor;
	int		x,y;

	// adjust y coordinate for display buffer (eg no dummy lines in buffer)
	y = SCANLINE - (display.lines_blanking + display.lines_dummy_top);

	if (mode == 0)
	{
		bdcolor = VZ_BLACK;
	}
	else
	{
		// gfx mode
		if (bgcolor == 0)
		{
			bdcolor = VZ_GREEN;
		}
		else
		{
			bdcolor = VZ_BUFF;
		}
	}
	for (x=0;x<320;x++)
	{
		Display_Buffer[320*y+x] = bdcolor;
	}
}


void DrawGMScanLine(int gmMode, int y, int bdcolor)
/*************************************************************************

	Inputs:	gmMode	-	0 = 64x64 color graphics one
						1 = 128x64 resolution graphics one
						2 - 128x64 color graphics two
						3 - 128x96 resolution graphics two
						4 - 128x96 color graphics three
						5 - 128x192 resolution graphics three
						6 - 128x192 color graphics six
						7 - 256x192 resolution graphics six

			y		-	current scanline
			bdcolor	-	border color. Green or buff

*************************************************************************/
{
	 int  n,x;
	 int  pixelrow;
	 byte mask,ch,adj, r, pcol, c0, c1, c2, c3;
	 int  colorset;
	 int  xm, ym, bpp, lp, offset;

	 colorset = VZ_LATCH & 0x10;

	 //
	 //		resolution for GM0 to GM7 is from 64x64 to 256x192, so need to calc
	 //		x,y scale factor for 256x192 display
	 //

	 static int gm_mode_scaling3x8[]  =
		{3,3,2,  2,3,1,  2,3,2,  2,2,1,  2,2,2,  2,1,1,  2,1,2,  1,1,1} ;

	 offset = gmMode*3;
	 xm = gm_mode_scaling3x8[offset];
	 ym = gm_mode_scaling3x8[offset+1];
	 bpp = gm_mode_scaling3x8[offset+2];
	 lp = 8 / bpp;     // 8 or 4 pixels per byte, depending on graphics mode

	 //
	 // determine color set based on latch bit
	 //

	 if (colorset == 0)
	 {
		  c0 = VZ_GREEN;
		  c1 = VZ_YELLOW;
		  c2 = VZ_BLUE;
		  c3 = VZ_RED;
	 }
	 else
	 {
		  c0 = VZ_BUFF;
		  c1 = VZ_CYAN;
		  c2 = VZ_MAGENTA;
		  c3 = VZ_ORANGE;
	 }

	 pixelrow = (y - display.lines_top_border)/ym;  // adjust for top border
	 if (lp == 4)           // colour mode - 4 pixels per byte
	 {
		for (x=0;x<32;x++)         // 32 bytes in row
		{
			ch = GXMEM[32*pixelrow+x];      // Get the screen byte
			mask = 0xC0;
			r = 6;
			for (n=0;n<lp;n++)
			{
				adj = (ch & mask) >> r;

				if (adj == 0x00) pcol = c0;
				if (adj == 0x01) pcol = c1;
				if (adj == 0x02) pcol = c2;
				if (adj == 0x03) pcol = c3;

				offset = 320*y+32+x*8;

				for (int pm=0;pm<xm;pm++)   // scaling factor will be 1, 2 or 3
				{
					Display_Buffer[offset+n*xm+pm] = pcol;
				}
				mask = mask >> 2;
				r -= 2;
			}
		}
	}
	else   // bw mode - 8 pixels per byte
	{
		for (x=0;x<32;x++)         // 32 bytes in row
		{
			ch = GXMEM[32*pixelrow+x];      // Get the screen byte
			mask = 0x80;
			for (n=0;n<8;n++)
			{
				if (ch & mask)
				{
					Display_Buffer[320*y+32+x*8+n] = VZ_DKGREEN;
				}
				else
				{
					Display_Buffer[320*y+32+x*8+n] = bdcolor;
				}
				mask = mask >> 1;
			}
		}
	}
}


void DrawGFXScanLine(int y, int bdcolor)
/**************************************************************************
*
* Name:			DrawGFXScanLine()
* Purpose:		Draw a graphics mode scanline
* Arguments: 	y - scanline
*               bdcolor - border color
*
***************************************************************************/
{

	int	gm;

	if (prefs.gfx > 0)
	{
		gm = (GFX_MOD_LATCH & 0x1C) >> 2;
		DrawGMScanLine(gm, y, bdcolor);
	}
	else
	{
		DrawGMScanLine(2, y, bdcolor);
	}
}


void DrawTextScanLine(int y, int txtfcolor, int txtbcolor)
/**************************************************************************
*
* Name:			DrawTextScanLine()
* Purpose:		Draw a text mode scanline
* Arguments: 	y - scanline
*               txtfcolor - text fore color
*               txtbcolor - text background color
*
***************************************************************************/
{

	int		textrow;
	int		n,x;
	int		offset;
	int		forecolor;
	int		backcolor;
	byte	ch;
	byte	font;
	byte	mask;


	// y will be between 0-191. There are 16 rows of text. Each text row
	// is 12 lines. So the text row will be y/12.

	textrow = (y-display.lines_top_border)/12;						// 16 text lines: 0-15
	offset = (y-display.lines_top_border) - textrow*12;				// 12 scanlines per character
	for (x=0;x<32;x++)												// 32 text columns
	{
		if (prefs.gfx > 0)
		{
			ch = GXMEM[32*textrow+x];			// Get the screen character
		}
		else
		{
			ch = VZMEM[28672+32*textrow+x];		// Get the screen character
		}
		if ((ch > 63) && (ch < 128))			// Check for inverse characters
		{
			ch -= 64;
			forecolor = txtbcolor;
			backcolor = txtfcolor;
		}
		else
		{
			forecolor = txtfcolor;
			backcolor = txtbcolor;
		}

		if (ch > 127)											// alpha numeric characters
		{
			forecolor = (ch-128) >> 4;
			ch -= (64 + 16*forecolor);
		}

		font = pal_square_fontdata8x12[12*ch+offset];
		mask = 0x80;
		int offs = 320*y+32+x*8;
		for (n=0;n<8;n++)
		{
			if (font & mask)
			{
				Display_Buffer[offs+n] = forecolor;	// 8000 = offset for top border
			}
			else
			{
				Display_Buffer[offs+n] = backcolor;
			}
			mask = mask >> 1;
		}
	}
}



void DrawDisplay()
/**************************************************************************
*
* Name:			DrawDisplay()
* Purpose:		Draw a frame
* Arguments: 	None. References global SCANLINE
*
***************************************************************************/
{
	int		mode = VZ_LATCH & 0x08;
	int		bgcolor = VZ_LATCH & 0x10;
	int		bdcolor;
	int		txtfcolor;
	int		txtbcolor;
	int		x,y;

	// adjust y coordinate for display buffer (eg no dummy lines in buffer)
	y = SCANLINE - (display.lines_blanking + display.lines_dummy_top);

	// For text mode, the border color is always black. The text background color is
	// always dark green. Set the text foreground color to green or orange, according
	// to the latch

	if (mode == 0)
	{
		bdcolor = VZ_BLACK;
		txtbcolor = VZ_DKGREEN;
		if (bgcolor == 0)
		{
			txtfcolor = VZ_GREEN;
		}
		else
		{
			txtfcolor = VZ_BRORANGE;
		}
		// first draw side borders
		for (x=0;x<32;x++)
		{
			Display_Buffer[320*y+x] = bdcolor;
			Display_Buffer[320*y+288+x] = bdcolor;
		}
		// now draw main display scanline
		DrawTextScanLine(y,txtfcolor,txtbcolor);
	}
	else
	{
		// gfx mode
		if (bgcolor == 0)
		{
			bdcolor = VZ_GREEN;
		}
		else
		{
			bdcolor = VZ_BUFF;
		}
		// first draw side borders
		for (x=0;x<32;x++)
		{
			Display_Buffer[320*y+x] = bdcolor;
			Display_Buffer[320*y+288+x] = bdcolor;
		}
		// now draw main display scanline
		DrawGFXScanLine(y,bdcolor);
	}
}

void DrawScanLine()
/**************************************************************************
*
* Name:			DrawScanLine()
* Purpose:		Draw a text/graphics scanline
* Arguments: 	None. References global SCANLINE
*
***************************************************************************/
{
    int topborder_startline = display.lines_blanking + display.lines_dummy_top;
    int maindisplay_startline = topborder_startline + display.lines_top_border;
    int bottomborder_startline = maindisplay_startline + display.lines_display;

	//
	//	Scanlines will be 0-311 for a VZ200
	//
	//	  0-12		Blanking		 13 lines
	//   13-37		Dummy lines		 25 lines
	//	 38-62		Top border		 25 lines
	//	 63-254		Main display	192 lines
	//  255-280		Bottom border	 26 lines
	//	281-305		Dummy lines		 25 lines
	//	306-311		Retrace			  6 lines

	if ((SCANLINE >= topborder_startline) &&
		(SCANLINE < maindisplay_startline))
	{
		// Top border
		DrawBorder();
	}

	if ((SCANLINE >= maindisplay_startline) && (SCANLINE < bottomborder_startline))
	{
		// Main Display
		UPDATEBORDER = VZ_LATCH;
		DrawDisplay();
	}

	if ((SCANLINE >= bottomborder_startline) &&
		(SCANLINE < bottomborder_startline + display.lines_bottom_border))
	{
		// Bottom border
		DrawBorder();
	}


}


void MapTape(char *s)
/**************************************************************************
*
* Name:			MapTape()
* Purpose:		Map a wav file to global variable wavTape
* Arguments: 	pointer to filename string
*
***************************************************************************/
{
	char	ext[8];

	strcpy(wavTape,s);
	strcpy(ext,".wav");
	if (strchr(s,46) == NULL)	// if no extension specified, add .wav
	{
		strcat(wavTape,ext);
	}
}

void MapPrinter(char *s)
/**************************************************************************
*
* Name:			MapPrinter()
* Purpose:		Map a txt file to global variable prtFile
* Arguments: 	pointer to filename string
*
***************************************************************************/
{
	char	ext[8];

	strcpy(prtFile,s);
	strcpy(ext,".txt");
	if (strchr(s,46) == NULL)	// if no extension specified, add .wav
	{
		strcat(prtFile,ext);
	}
	prt = fopen(prtFile,"wb+");
}


int PlayTape()
/**************************************************************************
*
* Name:			PlayTape()
* Purpose:		Open the wav file and set the flag indicating the tape is playing
* Arguments: 	None. Returns -3 if the file format is incorrect
*
***************************************************************************/
{
	g_tapeplaying = false;

	if (wavTape[0] == 0)
	{
		return -1;		// no valid tape image selected
	}
	tape = fopen(wavTape,"rb");
	if (!tape)
	{
		return -2;		// cannot open selected file
	}
	HEADER wavHeader;
	fread(&wavHeader, sizeof(HEADER),1,tape);
	if ((wavHeader.sampleRate != 22050)  ||
		(wavHeader.bits_per_sample != 8) ||
		(wavHeader.channels != 1))
	{
		return -3;		// wav file is not correct type
	}
	g_tapeplaying = true;
	return 0;
}

void RecordTape()
/**************************************************************************
*
* Name:			RecordTape()
* Purpose:		Open the wav file and set the flag indicating the tape is recording
* Arguments: 	None.
*
***************************************************************************/
{
	tape = fopen(wavTape,"wb+");
	if (!tape)
	{
		g_taperecording = false;
		return;
	}
	HEADER			wavHeader = {0x52,0x49,0x46,0x46,
					0, 0x57,0x41,0x56,0x45,0x66,0x6D,0x74,0x20,
					16,1,1,22050,22050,1,8,
					0x64,0x61,0x74,0x61,0};

	fwrite(&wavHeader,sizeof(HEADER),1,tape);
	g_tapeBytesWritten = 0;
	g_taperecording = true;
}

void StopTape()
/**************************************************************************
*
* Name:			StopTape()
* Purpose:		Close the wav file and set flags to stopped
* Arguments: 	None.
*
***************************************************************************/
{
	HEADER wavHeader;

	if (g_taperecording)
	{
		g_taperecording = false;
		fseek(tape,0,0);
		fread(&wavHeader,sizeof(HEADER),1,tape);
		wavHeader.data_len = g_tapeBytesWritten;
		wavHeader.fileLength = wavHeader.data_len + 36;
		fseek(tape,0,0);
		fwrite(&wavHeader,sizeof(HEADER),1,tape);
		fclose(tape);
	}

	if (g_tapeplaying)
	{
		g_tapeplaying = false;
		g_tapeByte = 0x80;
		fclose(tape);
	}
}

void WriteCassette(byte g_tapeByte)
/**************************************************************************
*
* Name:			WriteCassette
* Purpose:		Write a byte to the cassette stream
* Arguments: 	None.
*
***************************************************************************/
{
	static int	lastCount = 0;
	int			n,wavbytes;

	int elapsedCycles;

	if (g_taperecording)
	{
		if (REGS.ICount > lastCount) // interrupt period has expired
		{
			lastCount += REGS.IPeriod;
		}
		elapsedCycles = lastCount - REGS.ICount;

		// manually tweak wavbytes to write to get correct result for 22050 file
		wavbytes = 39;	// gap for 1 bit silence - 22050 hz file
		if (elapsedCycles > 1700)
		{
			byte silence = 0x80;
			for (n=0;n<wavbytes;n++)
			{
				fwrite(&silence,1,1,tape);
				g_tapeBytesWritten++;
			}
		}

		if ((elapsedCycles > 700) && (elapsedCycles < 1700))	// long pulse
		{
			if (g_tapeByte == 0x00)
			{
				wavbytes = 13;
				for (n=0;n<wavbytes;n++)
					fwrite(&LONG_LOW[n], 1, 1, tape);
				g_tapeBytesWritten += wavbytes;
			}
			if (g_tapeByte == 0xFF)
			{
				wavbytes = 13;
				for (n=0;n<wavbytes;n++)
					fwrite(&LONG_HIGH[n], 1, 1, tape);
				g_tapeBytesWritten += wavbytes;
			}
		}
		if (elapsedCycles < 700) // short pulse
		{
			if (g_tapeByte == 0x00)
			{
				wavbytes = 7;
				for (n=0;n<wavbytes;n++)
					fwrite(&SHORT_LOW[n], 1, 1, tape);
				g_tapeBytesWritten += wavbytes;
			}
			if (g_tapeByte == 0xFF)
			{
				wavbytes = 6;
				for (n=0;n<wavbytes;n++)
					fwrite(&SHORT_HIGH[n], 1, 1, tape);
				g_tapeBytesWritten += wavbytes;
			}
		}
		lastCount = REGS.ICount;
	}
}

word LoopZ80(Z80 *R)
/**************************************************************************
*
* Name:			LoopZ80
* Purpose:		Called after each Z80 instruction to check for periodic
*               events like scanlines & cassette i/o
* Arguments: 	Z80 registers
*
***************************************************************************/
{

	static int	lastCount = 0;
	static int  cycles = 0;
	static int  audioCycles = 0;
	static int	tapeCycles = 0;

	if (R->ICount > lastCount) // interrupt period has expired
	{
		lastCount += REGS.IPeriod;
	}
	cycles += lastCount - R->ICount;
	audioCycles += lastCount - R->ICount;

	while (cycles > CYCLES_PER_SCANLINE)
	{
		DrawScanLine();
		SCANLINE++;

		if (SCANLINE == display.total_lines)
		{
			SCANLINE = 0;
			SCANLINEOFFSET = -1;
		}
		if (SCANLINE == display.FS)
		{
			GenInterrupt(R);
		}

		cycles -= CYCLES_PER_SCANLINE;
	}

	if ((g_tapeplaying || g_taperecording))
	{
		if (prefs.cassetteAudio)
		{
			if (g_tapeByte > 128)
				AUDIOSOUNDSAMPLE = 255;
			else
				AUDIOSOUNDSAMPLE = 0;
		}
	}

	while (audioCycles > CYCLES_PER_AUDIOSAMPLE)
	{
		audioCycles -= CYCLES_PER_AUDIOSAMPLE;
		AUDIO_BUFFER[AUDIO_POS] = AUDIOSOUNDSAMPLE;
		AUDIO_POS++;
		if (AUDIO_POS == AUDIO_BUFFER_SIZE) AUDIO_POS = 0;	// circular buffer of sound data
        AUDIO_SAMPLES_THIS_FRAME++;
	}
	static long bytesread = 0;
	if (g_tapeplaying)
	{
		tapeCycles += lastCount - R->ICount;
		while (tapeCycles > CYCLES_PER_TAPEBYTE)
		{
			if (!feof(tape))
			{
				fread(&g_tapeByte,1,1,tape);
				bytesread++;
			}
			else
			{
				g_tapeplaying = false;
				tapeCycles = 0;
				g_tapeByte = 0;
				bytesread = 0;
				fclose(tape);
			}
			tapeCycles -= CYCLES_PER_TAPEBYTE;
		}
	}

	lastCount = REGS.ICount;

	return R->PC.W;
}


void ScanKbrd()
/**************************************************************************
*
* Name:			ScanKbrd()
* Purpose:		Check for key presses
* Arguments: 	none. Uses global array VZKBRD
*
***************************************************************************/
{
   int  n;

	// clear the matrix to assume no keys pressed
   for (n=0;n<8;n++) VZKBRD[n] = 0xFF;

   // Call the OS dependent function to check what keys are pressed
   // and populate the vz keyboard matrix

   osd_ScanKbrd(VZKBRD);
}

void LoadMem(word startAddr, char *filespec)
/**************************************************************************
*
* Name:			LoadMem()
* Purpose:		Load a file into Z80 memory
* Arguments: 	startAddr - 16 bit address
*               *filespec - pointer to string containing filename
*
***************************************************************************/
{
    FILE    *file;
    long    n = 0;

    file = fopen(filespec ,"rb");

    if (!file)
    {
       return;
    }
    while (!feof(file))
    {
        VZMEM[startAddr+n] = fgetc(file);
        n++;
    }
    fclose(file);
}

void LoadVZFont(char *filespec)
/**************************************************************************
*
* Name:			LoadVZFont()
* Purpose:		Load a M6847 compatible font into memory
* Arguments: 	*filespec - pointer to string containing filename
*
***************************************************************************/
{
    FILE    *file;
    long    n = 0;

    file = fopen(filespec ,"rb");

    if (!file)
    {
       return;
    }
    while (!feof(file))
    {
        pal_square_fontdata8x12[n] = fgetc(file);
        n++;
    }
    fclose(file);
}

void LoadVZRom()
/**************************************************************************
*
* Name:			LoadVZRom()
* Purpose:		Load the rom file specified in the preferences config
* Arguments: 	none.
*
***************************************************************************/
{
    FILE    *vzrom;
    long    n;

    //
    // Load the VZ rom
    //
    //vzrom = fopen("vzrom.v20","rb");
    vzrom = fopen(prefs.romFile ,"rb");

    if (!vzrom)
    {
       return;
    }
    for (n=0;n<16384;n++)
    {
		VZMEM[n] = fgetc(vzrom);
    }
    fclose(vzrom);

	// clear out the cartridge memory in case disabling disk
		for (n=0;n<8192;n++)
		{
			VZMEM[16384+n] = 0x7f;
		}

	if (prefs.disk_drives == 1)
	{
		vzrom = fopen(prefs.cartridgeFile, "rb");
		if (!vzrom)
		{
		   return;
		}
		for (n=0;n<10240;n++)
		{
			VZMEM[16384+n] = fgetc(vzrom);
		}
		fclose(vzrom);
	}
}

void SetPrefs(char *vzprefs)
/**************************************************************************
*
* Name:			SetPrefs
* Purpose:		Wrapper for the internal menu
* Arguments: 	*vzprefs - pointer to preferences structure
*
***************************************************************************/
{
    if (strcmp(vzprefs,"VZ200") == 0) prefs.vzmodel = 0;
    if (strcmp(vzprefs,"VZ300") == 0) prefs.vzmodel = 1;
    if (strcmp(vzprefs,"PAL") == 0) prefs.display_type = 0;
    if (strcmp(vzprefs,"NTSC") == 0) prefs.display_type = 1;
    if (strcmp(vzprefs,"8K") == 0) prefs.top_of_memory = 36863;
    if (strcmp(vzprefs,"18K") == 0) prefs.top_of_memory = 47103;
    if (strcmp(vzprefs,"24K") == 0) prefs.top_of_memory = 53247;
    if (strcmp(vzprefs,"34K") == 0) prefs.top_of_memory = 65535;
    if (strcmp(vzprefs,"EFFECTS_ON") == 0) prefs.snow = 1;
    if (strcmp(vzprefs,"EFFECTS_OFF") == 0) prefs.snow = 0;
    if (strcmp(vzprefs,"NORMAL") == 0) prefs.synchVZ = true;
    if (strcmp(vzprefs,"MAXIMUM") == 0) prefs.synchVZ = false;
    if (strcmp(vzprefs,"MAXIMUM") == 0) prefs.synchVZ = false;
    if (strcmp(vzprefs,"SOUND_ON") == 0) {};
    if (strcmp(vzprefs,"SOUND_OFF") == 0) {};
    if (strcmp(vzprefs,"CASSETTE_AUDIO_ON") == 0) {};

}

void LoadPrefs()
/**************************************************************************
*
* Name:			LoadPrefs()
* Purpose:		Load the preferences from vzem.cfg
* Arguments: 	none
*
***************************************************************************/
{
	FILE	*fprefs;

	// open default prefs file read/write
	fprefs = fopen("vzem.cfg","rb+");
	// not found, create it
	if (!fprefs)
	{
		prefs.vzmodel = 0;								// VZ200
		prefs.display_type = 0;							// PAL
		prefs.gfx = 1;									// Leon gfx mod
		prefs.snow = 0;									// produce snow
		prefs.disk_drives = 1;
		prefs.joystick = 1;
		prefs.rom_writes = 0;
		prefs.cartridge_writes = 0;
		prefs.top_of_memory = 65535;					// 64k
		prefs.synchVZ = true;
		prefs.cassetteAudio = true;
		prefs.diskAudio = true;
		prefs.fast_load = false;
		strcpy(prefs.romFile, "vzrom.v20");
		strcpy(prefs.cartridgeFile, "vzdos.rom");

		fprefs = fopen("vzem.cfg","wb+");				// create new file
		fwrite(&prefs,sizeof(prefs),1,fprefs);			// write prefs
		fclose(fprefs);
		fprefs = fopen("vzem.cfg","rb+");
	}
	if (fprefs)
	{
		fread(&prefs,sizeof(prefs),1,fprefs);
		fclose(fprefs);
	}
}

void SavePrefs()
/**************************************************************************
*
* Name:			SavePrefs()
* Purpose:		Save the preferences to vzem.cfg
* Arguments: 	none
*
***************************************************************************/
{
	FILE	*fprefs;

	// open default prefs file read/write
	fprefs = fopen("vzem.cfg","rb+");
	// not found, create it
	if (!fprefs)
	{
		fprefs = fopen("vzem.cfg","wb+");				// create new file
		fwrite(&prefs,sizeof(prefs),1,fprefs);			// write prefs
		fclose(fprefs);
		fprefs = fopen("vzem.cfg","rb+");
	}
	if (fprefs)
	{
		fwrite(&prefs,sizeof(prefs),1,fprefs);			// write prefs
		fclose(fprefs);
	}
}


void InitVZ()
/**************************************************************************
*
* Name:			LoadPrefs()
* Purpose:		Load the ROM, reset the Z80
* Arguments: 	none
*
***************************************************************************/
{

	if (prefs.display_type == 0)
	{
		// PAL
		display.lines_dummy_top = 24;
		display.lines_dummy_bottom = 24;
	}
	else
	{
		// NTSC
		display.lines_dummy_top = 0;
		display.lines_dummy_bottom = 0;
	}

	if (prefs.vzmodel == 0)
	{
		// VZ200
		prefs.cpu_speed = 3.5795f;
		display.lines_dummy_top++;
		display.lines_dummy_bottom++;
	}
	else
	{
		// VZ300
		prefs.cpu_speed = 3.5469f;
	}

	display.lines_blanking = 13;
	display.lines_top_border = 25;
	display.lines_display = 192;
	display.lines_bottom_border = 26;
	display.lines_retrace = 6;

	display.total_lines =	display.lines_blanking +
							display.lines_dummy_top +
							display.lines_top_border +
							display.lines_display +
							display.lines_bottom_border +
							display.lines_dummy_bottom +
							display.lines_retrace;

	display.FS = display.total_lines - display.lines_retrace;

	display.OFF =	display.lines_blanking +
					display.lines_dummy_top +
					display.lines_top_border +
					display.lines_display;


    RAM = VZMEM;									// RAM pointer defined in Z80 library
    REGS.IPeriod = (int) (prefs.cpu_speed * 1000000/50);
	SCANLINE = 0;
	SCANLINEOFFSET = -1;
	GFX_MOD_LATCH = 8;					// set default mode to 128x64
	ResetZ80(&REGS);
	LoadVZRom();
	osd_InitTimer();					// Create a 20ms timer

	RENDER = 2;
    RUNNING = true;
}

void LoadVZFile(char *s)
/**************************************************************************
*
* Name:			LoadVZFile
* Purpose:		Load a .vz snapshot into memory
* Arguments: 	pointer to filename
*
***************************************************************************/
{
	FILE *vzfile;
	unsigned short n, start_addr;
	byte b;


	VZFILE vzf;

	vzfile = fopen(s, "rb");
	if (!vzfile)
	{
		exit(1);
	}


	// read the vzheader

	fread(&vzf, sizeof(vzf), 1, vzfile);
	start_addr = vzf.start_addrh;
	start_addr = start_addr * 256 + vzf.start_addrl;


	n = 0;
	fread(&b, 1, 1, vzfile);
	while (!feof(vzfile))
	{
		VZMEM[start_addr + n] = b;
		n++;
		fread(&b, 1, 1, vzfile);
	}
	fclose(vzfile);

	//
	// Basic programs cannot be loaded at startup because the basic pointers
	// are overwritten when the rom is loaded
	//
	if (vzf.ftype == 0xF0)
	{
		VZMEM[0x78A4] = vzf.start_addrl;
		VZMEM[0x78A5] = vzf.start_addrh;
		VZMEM[0x78F9] = ((start_addr + n) & 0x00FF);
		VZMEM[0x78FA] = (((start_addr + n) & 0xFF00) >> 8);
	}


	if (vzf.ftype == 0xF1)
	{
		REGS.PC.B.l = vzf.start_addrl;
		REGS.PC.B.h = vzf.start_addrh;
	}
}



void SaveVZFile(byte *fileBuffer, unsigned long *dwFileLen, VZFILE *vzf)
/**************************************************************************
*
* Name:			SaveVZFile()
* Purpose:		Load a snapshot file internal an internal memory buffer
* Arguments: 	*fileBuffer - pointer to byte array
*               *dwFileLen - length of file
*               *vzf -  file handle
*
***************************************************************************/
{
		word	startAddr;
		unsigned long		n;

		memcpy(fileBuffer,vzf,24);

		startAddr = vzf -> start_addrh * 256 + vzf -> start_addrl;
		for (n=0;n<*dwFileLen;n++)
		{
			fileBuffer[24+n] = RAM[startAddr+n];
		}
		*dwFileLen += 24;
}

int DoFrame()
/**************************************************************************
*
* Name:			DoFrame()
* Purpose:		Run the Z80, check VZ hardware, draw the screen
* Arguments: 	none
*
***************************************************************************/
{
	word PC;

	Uint32 nextFrame = osd_GetTicks() + 20;		// An emulated frame should complete in less than 20ms, so wait
	Uint32 now;

	AUDIO_SAMPLES_THIS_FRAME = 0;
	PC = RunZ80(&REGS);				// execute Z80 till next interrupt
    for (int i=AUDIO_SAMPLES_THIS_FRAME;i<AUDIO_SAMPLES_PER_FRAME;i++)
    {
        AUDIO_BUFFER[AUDIO_POS] = AUDIOSOUNDSAMPLE;
   		AUDIO_POS++;
		if (AUDIO_POS == AUDIO_BUFFER_SIZE) AUDIO_POS = 0;	// circular buffer of sound data
    }
    osd_UpdateAudioBuffer(AUDIO_BUFFER, AUDIO_SAMPLES_PER_FRAME);

	if (REGS.Trace == 0)
	{
		ScanKbrd(); 							// Get key presses/releases

		if (MENUACTIVE)
		{
			menu();
			RENDER = 2;
		}

		if (RENDER > 0)
		{
			osd_BlitBuffer(Display_Buffer);		// Paint the display
			RENDER--;
		}
		if (prefs.synchVZ)						// normal speed
		{
   			now = osd_GetTicks();
			if (now < nextFrame) SDL_Delay(nextFrame-now);
		}
	}
	return REGS.Trace;
}

void RunVZ()
/**************************************************************************
*
* Name:			RunVZ()
* Purpose:		Keep executing frames until user quits
* Arguments: 	none
*
***************************************************************************/
{
    while (RUNNING)
    {
        DoFrame();
    }
}
