/*
 * Decompiled with CFR 0.152.
 */
package jpcsp.arm;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import jpcsp.arm.ARMDecoder;
import jpcsp.arm.ARMInstruction;
import jpcsp.arm.ARMInstructions;
import jpcsp.arm.ARMInterpreter;
import jpcsp.arm.ARMMemory;
import jpcsp.arm.IARMHLECall;
import jpcsp.util.Utilities;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

public class ARMDisassembler {
    private final Logger log;
    private final Level level;
    private final ARMMemory mem;
    private final ARMInterpreter interpreter;
    private final List<Integer> pendingFunctions = new LinkedList<Integer>();
    private final Set<Integer> disassembledFunctions = new HashSet<Integer>();
    private final Map<Integer, Integer> branchingTo = new HashMap<Integer, Integer>();
    private final Set<Integer> branchingTargets = new HashSet<Integer>();

    public ARMDisassembler(Logger log, Level level, ARMMemory mem, ARMInterpreter interpreter) {
        this.log = log;
        this.level = level;
        this.mem = mem;
        this.interpreter = interpreter;
    }

    private void log(String s) {
        this.log.log((Priority)this.level, (Object)s);
    }

    private boolean isAlwaysCondition(int insn) {
        return insn >>> 28 == 14;
    }

    private int getJumpFromBranchOffset(int addr, int offset, boolean thumbMode) {
        return addr + offset + (thumbMode ? 5 : 8);
    }

    private int getRegister(int insn, int offset) {
        return insn >> offset & 0xF;
    }

    private String getRegisterName(int insn, int offset) {
        return ARMInstructions.getRegisterName(this.getRegister(insn, offset));
    }

    private int getThumbRegister(int insn, int offset) {
        return insn >> offset & 7;
    }

    private String getThumbRegisterName(int insn, int offset) {
        return ARMInstructions.getRegisterName(this.getThumbRegister(insn, offset));
    }

    private void addJump(List<Integer> list, int addr, int jumpTo) {
        this.branchingTo.put(addr, jumpTo);
        this.branchingTargets.add(Utilities.clearBit(jumpTo, 0));
        list.add(jumpTo);
    }

    private void addJumpToFunction(int addr, int jumpTo) {
        this.branchingTo.put(addr, jumpTo);
        this.branchingTargets.add(Utilities.clearBit(jumpTo, 0));
        this.pendingFunctions.add(jumpTo);
    }

    private boolean isLdrFixedValue(ARMInstruction instr, int insn, boolean allowRead8) {
        return instr == ARMInstructions.LDR && Utilities.notHasFlag(insn, 0x2000000) && (Utilities.notHasFlag(insn, 0x400000) || allowRead8) && this.getRegister(insn, 16) == 15;
    }

    private boolean isLdrFixedValue(int insn, boolean allowRead8) {
        return this.isLdrFixedValue(ARMDecoder.instruction(insn), insn, allowRead8);
    }

    private int getLdrFixedValue(int insn, int base) {
        int ptr = base;
        if (Utilities.hasFlag(insn, 0x1000000)) {
            int offset = insn & 0xFFF;
            ptr = Utilities.hasFlag(insn, 0x800000) ? (ptr += offset) : (ptr -= offset);
        }
        int value = Utilities.hasFlag(insn, 0x400000) ? this.mem.internalRead8(ptr) : this.mem.internalRead32(ptr);
        return value;
    }

    private int getThumbLdrFixedValue(int insn, int base) {
        int value = this.mem.internalRead32(Utilities.clearBit(base, 1) + ((insn & 0xFF) << 2));
        return value;
    }

    private boolean isEndOfBlock(int addr, int insn, ARMInstruction instr, boolean thumbMode) {
        int rd;
        if (instr == ARMInstructions.B_Thumb || instr == ARMInstructions.BX_Thumb) {
            return true;
        }
        if ((instr == ARMInstructions.B || instr == ARMInstructions.BX) && this.isAlwaysCondition(insn)) {
            return true;
        }
        if (instr == ARMInstructions.POP_Thumb && Utilities.hasFlag(insn, 256)) {
            return true;
        }
        if (instr == ARMInstructions.LDR && this.isAlwaysCondition(insn) && this.getRegister(insn, 12) == 15) {
            return true;
        }
        if (instr == ARMInstructions.BKPT || instr == ARMInstructions.BKPT_Thumb) {
            return true;
        }
        return instr == ARMInstructions.MOV_High_Thumb && (rd = insn & 7 | insn >> 4 & 8) == 15;
    }

    private void checkBranch(List<Integer> list, int addr, int insn, ARMInstruction instr, boolean thumbMode) {
        int previousInsn;
        ARMInstruction previousInstr;
        int rd;
        if (instr == ARMInstructions.B) {
            int jumpTo = this.getJumpFromBranchOffset(addr, insn << 8 >> 6, thumbMode);
            this.addJump(list, addr, jumpTo);
        } else if (instr == ARMInstructions.BL) {
            int jumpTo = this.getJumpFromBranchOffset(addr, insn << 8 >> 6, thumbMode);
            this.addJumpToFunction(addr, jumpTo);
        } else if (this.isLdrFixedValue(instr, insn, false) && this.getRegister(insn, 12) == 15) {
            int jumpTo = this.getLdrFixedValue(insn, addr + 8);
            this.addJump(list, addr, jumpTo);
        } else if (instr == ARMInstructions.BX) {
            if (this.getRegister(insn, 0) == 15) {
                this.addJump(list, addr, addr + 8);
            } else {
                int previousInsn2 = this.mem.internalRead32(addr - 4);
                if (this.isLdrFixedValue(previousInsn2, false) && this.getRegister(insn, 0) == this.getRegister(previousInsn2, 12)) {
                    int jumpTo = this.getLdrFixedValue(previousInsn2, addr + 4);
                    this.addJump(list, addr, jumpTo);
                }
            }
        } else if (instr == ARMInstructions.BX_Thumb) {
            if (this.getRegister(insn, 3) == 15) {
                this.addJump(list, addr, addr + 4);
            } else {
                int previousInsn3 = this.mem.internalRead16(addr - 2);
                ARMInstruction previousInstr2 = ARMDecoder.thumbInstruction(previousInsn3);
                if (previousInstr2 == ARMInstructions.ADD_Rd_Pc_Thumb && this.getThumbRegister(insn, 3) == this.getThumbRegister(previousInsn3, 8)) {
                    int jumpTo = Utilities.clearFlag(addr + 4, 3) + ((previousInsn3 & 0xFF) << 2);
                    this.addJump(list, addr, jumpTo);
                }
            }
        } else if (instr == ARMInstructions.BL_11_Thumb || instr == ARMInstructions.BLX_01_Thumb) {
            int previousInsn4 = this.mem.internalRead16(addr - 2);
            ARMInstruction previousInstr3 = ARMDecoder.thumbInstruction(previousInsn4);
            if (previousInstr3 == ARMInstructions.BL_10_Thumb) {
                int offset1 = previousInsn4 << 21 >> 9;
                int offset2 = (insn & 0x7FF) << 1;
                int jumpTo = addr + 2 + offset1 + offset2;
                if (instr == ARMInstructions.BL_11_Thumb) {
                    jumpTo = Utilities.setBit(jumpTo, 0);
                } else if (instr == ARMInstructions.BLX_01_Thumb) {
                    jumpTo = Utilities.clearFlag(jumpTo, 3);
                }
                this.addJumpToFunction(addr, jumpTo);
            }
        } else if (instr == ARMInstructions.B_Thumb) {
            int jumpTo = this.getJumpFromBranchOffset(addr, insn << 21 >> 20, thumbMode);
            this.addJump(list, addr, jumpTo);
        } else if (instr == ARMInstructions.B_Cond_Thumb) {
            int jumpTo = this.getJumpFromBranchOffset(addr, insn << 24 >> 23, thumbMode);
            this.addJump(list, addr, jumpTo);
        } else if (instr == ARMInstructions.MOV_High_Thumb && (rd = insn & 7 | insn >> 4 & 8) == 15 && (previousInstr = ARMDecoder.thumbInstruction(previousInsn = this.mem.internalRead16(addr - 2))) == ARMInstructions.LDR_Pc_Thumb && this.getThumbRegister(previousInsn, 8) == this.getRegister(insn, 3)) {
            int jumpTo = this.getThumbLdrFixedValue(previousInsn, addr + 2);
            this.addJump(list, addr, Utilities.setBit(jumpTo, 0));
        }
    }

    private String getAdditionalInfo(int addr, int insn, ARMInstruction instr, boolean thumbMode) {
        int previousInsn;
        ARMInstruction previousInstr;
        int rd;
        String additionalInfo = null;
        if (this.isLdrFixedValue(instr, insn, true)) {
            int value = this.getLdrFixedValue(insn, addr + 8);
            additionalInfo = Utilities.hasFlag(insn, 0x400000) ? String.format(" <=> mov %s, #0x%02X", this.getRegisterName(insn, 12), value) : String.format(" <=> mov %s, #0x%08X", this.getRegisterName(insn, 12), value);
        } else if (instr == ARMInstructions.LDR_Pc_Thumb) {
            int value = this.getThumbLdrFixedValue(insn, addr + 4);
            additionalInfo = String.format(" <=> mov %s, #0x%08X", this.getThumbRegisterName(insn, 8), value);
        } else if (instr == ARMInstructions.BX) {
            if (this.getRegister(insn, 0) == 15) {
                additionalInfo = String.format(" <=> bx 0x%08X", addr + 8);
            } else {
                int previousInsn2 = this.mem.internalRead32(addr - 4);
                if (this.isLdrFixedValue(previousInsn2, false) && this.getRegister(insn, 0) == this.getRegister(previousInsn2, 12)) {
                    int jumpTo = this.getLdrFixedValue(previousInsn2, addr + 4);
                    additionalInfo = String.format(" <=> bx 0x%08X", jumpTo);
                }
            }
        } else if (instr == ARMInstructions.ADD && this.getRegister(insn, 16) == 15 && Utilities.hasFlag(insn, 0x2000000)) {
            int imm8 = insn & 0xFF;
            int immRot = insn >> 8 & 0xF;
            int value = Integer.rotateRight(imm8, immRot << 1);
            additionalInfo = String.format(" <=> movs %s, #0x%08X", this.getRegisterName(insn, 12), addr + 8 + value);
        } else if (instr == ARMInstructions.BX_Thumb) {
            if (this.getRegister(insn, 3) == 15) {
                additionalInfo = String.format(" <=> bx 0x%08X", addr + 4);
            } else {
                int previousInsn3 = this.mem.internalRead16(addr - 2);
                ARMInstruction previousInstr2 = ARMDecoder.thumbInstruction(previousInsn3);
                if (previousInstr2 == ARMInstructions.ADD_Rd_Pc_Thumb && this.getThumbRegister(insn, 3) == this.getThumbRegister(previousInsn3, 8)) {
                    int jumpTo = Utilities.clearFlag(addr + 4, 3) + ((previousInsn3 & 0xFF) << 2);
                    additionalInfo = String.format(" <=> bx 0x%08X", jumpTo);
                }
            }
        } else if (instr == ARMInstructions.BL_11_Thumb || instr == ARMInstructions.BLX_01_Thumb) {
            int previousInsn4 = this.mem.internalRead16(addr - 2);
            ARMInstruction previousInstr3 = ARMDecoder.thumbInstruction(previousInsn4);
            if (previousInstr3 == ARMInstructions.BL_10_Thumb) {
                int offset1 = previousInsn4 << 21 >> 9;
                int offset2 = (insn & 0x7FF) << 1;
                int jumpTo = addr + 2 + offset1 + offset2;
                if (instr == ARMInstructions.BLX_01_Thumb) {
                    jumpTo = Utilities.clearFlag(jumpTo, 3);
                }
                additionalInfo = String.format(" <=> bl%s 0x%08X", instr == ARMInstructions.BLX_01_Thumb ? "x" : "", jumpTo);
            }
        } else if (instr == ARMInstructions.BKPT_Thumb) {
            IARMHLECall hleCall = this.interpreter.getHLECall(addr, insn & 0xFF);
            if (hleCall != null) {
                additionalInfo = String.format(" <=> %s", hleCall);
            }
        } else if (instr == ARMInstructions.BKPT) {
            IARMHLECall hleCall = this.interpreter.getHLECall(addr, insn >> 4 & 0xFFF0 | insn & 0xF);
            if (hleCall != null) {
                additionalInfo = String.format(" <=> %s", hleCall);
            }
        } else if (instr == ARMInstructions.MOV_High_Thumb && (rd = insn & 7 | insn >> 4 & 8) == 15 && (previousInstr = ARMDecoder.thumbInstruction(previousInsn = this.mem.internalRead16(addr - 2))) == ARMInstructions.LDR_Pc_Thumb && this.getThumbRegister(previousInsn, 8) == this.getRegister(insn, 3)) {
            int jumpTo = this.getThumbLdrFixedValue(previousInsn, addr + 2);
            additionalInfo = String.format(" <=> b 0x%08X", Utilities.clearBit(jumpTo, 0));
        }
        return additionalInfo;
    }

    private void disasmFunction(int startAddress) {
        if (this.disassembledFunctions.contains(startAddress)) {
            return;
        }
        this.log(String.format("Disassembling Function 0x%08X", Utilities.clearBit(startAddress, 0)));
        TreeMap<Integer, String> disassembled = new TreeMap<Integer, String>();
        LinkedList<Integer> pendingAddresses = new LinkedList<Integer>();
        pendingAddresses.add(startAddress);
        while (!pendingAddresses.isEmpty()) {
            int pc = (Integer)pendingAddresses.remove(0);
            boolean thumbMode = Utilities.hasBit(pc, 0);
            pc = Utilities.clearBit(pc, 0);
            boolean endOfBlock = false;
            while (!endOfBlock && !disassembled.containsKey(pc) && ARMMemory.isAddressGood(pc)) {
                int nextPc;
                ARMInstruction instr;
                String insnString;
                int insn;
                if (thumbMode) {
                    insn = this.mem.internalRead16(pc);
                    insnString = String.format("%04X", insn);
                    instr = ARMDecoder.thumbInstruction(insn);
                    nextPc = pc + 2;
                } else {
                    insn = this.mem.internalRead32(pc);
                    insnString = String.format("%08X", insn);
                    instr = ARMDecoder.instruction(insn);
                    nextPc = pc + 4;
                }
                String additionalInfo = this.getAdditionalInfo(pc, insn, instr, thumbMode);
                String disasm = String.format("0x%08X - [0x%s] - %s%s", pc, insnString, instr.disasm(pc, insn), additionalInfo == null ? "" : additionalInfo);
                disassembled.put(pc, disasm);
                this.checkBranch(pendingAddresses, pc, insn, instr, thumbMode);
                endOfBlock = this.isEndOfBlock(pc, insn, instr, thumbMode);
                pc = nextPc;
            }
        }
        for (Integer pc : disassembled.keySet()) {
            char branchingTarget = this.branchingTargets.contains(pc) ? (char)'>' : ' ';
            Integer branchTo = this.branchingTo.get(pc);
            int branchingFlag = branchTo == null ? 32 : (!disassembled.containsKey(branchTo) ? 60 : (branchTo <= pc ? 94 : 118));
            this.log(String.format("%c%c %s", Character.valueOf((char)branchingFlag), Character.valueOf(branchingTarget), disassembled.get(pc)));
        }
        this.disassembledFunctions.add(startAddress);
    }

    private void disasmAll() {
        while (!this.pendingFunctions.isEmpty()) {
            int addr = this.pendingFunctions.remove(0);
            this.disasmFunction(addr);
        }
    }

    public void disasm(int addr) {
        this.pendingFunctions.add(addr);
        this.branchingTargets.add(Utilities.clearBit(addr, 0));
        this.disasmAll();
    }
}

