// HDD support
#include <stdio.h>
#include <windows.h>
#include <commctrl.h>
#include "dev9.h"
#include "hdd.h"
#include "resource.h"

#define twochars(a, b) ((a << 8) | (b))

extern HINSTANCE hInst;

FILE *hddImage;

u16 hddInfo[256] = {
	0x0040,
	16383,
	0,
	16,
	0, 0,
	63,
	0, 0, 0,
	twochars('M', 'G'), // serial
	twochars('D', 'V'),
	twochars('9', 'H'),
	twochars('D', 'D'),
	twochars('1', '0'),
	twochars('0', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
/*	twochars('F', '1'),
	twochars('Y', '0'),
	twochars('7', 'V'),
	twochars('W', '7'),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),*/
	0, 0, 0,
	twochars('F', 'I'), // firmware
	twochars('R', 'M'),
	twochars('1', '0'),
	twochars('0', ' '),
/*	twochars('Z', '1'),
	twochars('1', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),*/
	twochars('M', 'E'), // model
	twochars('G', 'A'),
	twochars('D', 'E'),
	twochars('V', '9'),
	twochars(' ', 'H'),
	twochars('D', 'D'),
	twochars(' ', 'V'),
	twochars('E', 'R'),
	twochars(' ', '1'),
	twochars('.', '0'),
	twochars('.', '0'),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
/*	twochars('S', 'C'),
	twochars('P', 'H'),
	twochars('-', '2'),
	twochars('0', '4'),
	twochars('0', '1'),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars('.', ' '),
	twochars('.', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),
	twochars(' ', ' '),*/
	1,
	0,
	0x0300,
	0,
	0x0200,
	0x0200,
	0,
	0, 0, 0, 0, 0, 
	0, 
	0x0000, 0x0100		// Sector number (default : 8 GB, changed on config load)
						// After that... nothing interesting, only zeroes
};

// HDD commands
// For IRQ timing, 0x6C seems perfect,
// so we'll use it, until proven wrong 

void (*HDDcmds[256])();
char *HDDcmdNames[256];

void HDDunk()
{
	printf("DEV9 HDD error : unknown cmd %02X\n", dev9.ata_regs.command);

	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_ERR;
	dev9.ata_regs.error = ATA_ERR_ABORT;
}

/*
Command base

	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	// here do stuffs

	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

	triggerIRQ(1, 0x6C);

*/

void HDDnop()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_BUSY | ATA_STAT_DRQ | ATA_STAT_ERR);
	dev9.ata_regs.status |= ATA_STAT_READY;
}

void HDDsmart()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	switch(dev9.ata_regs.feature)
	{
	case 0xD8:
		dev9.smart_on = 1;
		break;

	default:
		printf("DEV9 : Unknown SMART command %02X\n", dev9.ata_regs.feature);
		break;
	}

	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

	triggerIRQ(1, 0x6C);
}

void HDDreadDMA()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	// here do stuffs

	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

//	triggerIRQ(1, 0x1000);
}

void HDDwriteDMA()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	// here do stuffs

	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

//	triggerIRQ(1, 0x1000);
}

void HDDidle()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	// Very simple implementation, nothing ATM :P

	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

	triggerIRQ(1, 0x6C);
}

void HDDflushCache()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	// Write cache not supported yet

	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

	triggerIRQ(1, 0x6C);
}

void HDDflushCacheExt()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	// Write cache not supported yet

	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

	triggerIRQ(1, 0x6C);
}

void HDDidentifyDevice()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	memcpy(dev9.pio_buf, hddInfo, sizeof(dev9.pio_buf));
	dev9.pio_count = 0;
	dev9.pio_size = 256;

	dev9.ata_regs.status |= (ATA_STAT_DRQ | ATA_STAT_READY);
	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

	triggerIRQ(1, 0x6C);
}

void HDDsetFeatures()
{
	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	switch(dev9.ata_regs.feature)
	{
	case 0x03:
		dev9.xfer_mode = dev9.ata_regs.nsector;
		break;
	}

	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

	triggerIRQ(1, 0x6C);
}

void HDDsceSecCtrl()
{
	printf("DEV9 : SONY-SPECIFIC SECURITY CONTROL COMMAND (%02X)\n", dev9.ata_regs.feature);

	dev9.ata_regs.error = 0;
	dev9.ata_regs.status &= ~(ATA_STAT_DRQ | ATA_STAT_ERR | ATA_STAT_READY);
	dev9.ata_regs.status |= ATA_STAT_BUSY;

	memset(dev9.pio_buf, 0, sizeof(dev9.pio_buf));
	dev9.pio_count = 0;
	dev9.pio_size = 256;

	dev9.ata_regs.status |= ATA_STAT_DRQ;
	dev9.ata_regs.status &= ~ATA_STAT_BUSY;

	triggerIRQ(1, 0x6C);
}



BOOL CALLBACK zfProgressDlgProc(HWND hw, UINT msg, WPARAM wp, LPARAM lp)
{
	static int i;
	static int lastPercent;
	static RECT rect;
	static u8 buf[1024];
	static HWND progress;
	static HDC progress_dc;

	switch(msg)
	{
		case WM_INITDIALOG:
		{
			SendDlgItemMessage(hw, IDC_ZF_PROGRESS, PBM_SETRANGE, 0, MAKELONG(0, 100));

			memset(buf, 0, 1024);

			GetWindowRect(GetDlgItem(hw, IDC_ZF_PROGRESS), &rect);
			rect.bottom -= rect.top;
			rect.top = 0;
			rect.right -= rect.left;
			rect.left = 0;

			lastPercent = 0;

			progress = GetDlgItem(hw, IDC_ZF_PROGRESS);
			progress_dc = GetDC(progress);

			SetBkMode(progress_dc, TRANSPARENT);

			SetTimer(hw, 0x80, 1, NULL);

			return TRUE;
		}

		case WM_TIMER:
		{
			u32 percent;
			char txt[256];
			int j;

			for(j = 0; j < 1024; j++)
			{
				fwrite(buf, 1024, 1, hddImage);

				percent = ((i * 100) / (config.hddSize * 1024 * 1024));

				SendDlgItemMessage(hw, IDC_ZF_PROGRESS, PBM_SETPOS, percent, 0);

				sprintf(txt, "%i %%", percent);
				DrawText(progress_dc, txt, strlen(txt), &rect, (DT_CENTER | DT_VCENTER));
				if(percent > lastPercent)
				{
					lastPercent = percent;
					InvalidateRect(progress, &rect, TRUE);
				}

				i++;
				if(i >= (config.hddSize * 1024 * 1024))
				{
					ReleaseDC(progress, progress_dc);

					KillTimer(hw, 0x80);
					EndDialog(hw, TRUE);

					return TRUE;
				}
			}

			return TRUE;
		}
	}

	return FALSE;
}

int HDDinit()
{
	int i;
	u32 nbSectors;

	// Fill the command table (inspired from DEV9linuz)
	for(i = 0; i < 256; i++)
	{
		HDDcmds[i] = HDDunk;
		HDDcmdNames[i] = "unknown";
	}

	HDDcmds[0x00] = HDDnop;					HDDcmdNames[0x00] = "nop";

	HDDcmds[0xB0] = HDDsmart;				HDDcmdNames[0xB0] = "SMART";

	HDDcmds[0xC8] = HDDreadDMA;				HDDcmdNames[0xC8] = "DMA read";
	HDDcmds[0xCA] = HDDwriteDMA;			HDDcmdNames[0xCA] = "DMA write";
/*	HDDcmds[0x25] = HDDreadDMA_ext;*/		HDDcmdNames[0x25] = "DMA read (48-bit)";		// 48-bit
/*	HDDcmds[0x35] = HDDwriteDMA_ext;*/		HDDcmdNames[0x35] = "DMA write (48-bit)";		// 48-bit

	HDDcmds[0xE3] = HDDidle;				HDDcmdNames[0xE3] = "idle";

	HDDcmds[0xE7] = HDDflushCache;			HDDcmdNames[0xE7] = "flush cache";
/*	HDDcmds[0xEA] = HDDflushCacheExt;*/		HDDcmdNames[0xEA] = "flush cache (48-bit)";		// 48-bit

	HDDcmds[0xEC] = HDDidentifyDevice;		HDDcmdNames[0xEC] = "identify device";
/*	HDDcmds[0xA1] = HDDidentifyPktDevice;*/	HDDcmdNames[0xA1] = "identify ATAPI device";	// For ATAPI devices

	HDDcmds[0xEF] = HDDsetFeatures;			HDDcmdNames[0xEF] = "set features";

/*	HDDcmds[0xF1] = HDDsecSetPassword;*/	HDDcmdNames[0xF1] = "security set password";
/*	HDDcmds[0xF2] = HDDsecUnlock;*/			HDDcmdNames[0xF2] = "security unlock";
/*	HDDcmds[0xF3] = HDDsecErasePrepare;*/	HDDcmdNames[0xF3] = "security erase prepare";
/*	HDDcmds[0xF4] = HDDsecEraseUnit;*/		HDDcmdNames[0xF4] = "security erase unit";

	/* This command is Sony-specific and isn't part of the IDE standard */
	/* The Sony HDD has a modified firmware that supports this command */
	/* Sending this command to a standard HDD will give an error */
	/* We roughly emulate it to make programs think the HDD is a Sony one */
	HDDcmds[0x8E] = HDDsceSecCtrl;			HDDcmdNames[0x8E] = "SCE security control";



	// Set the number of sectors
	nbSectors = (((config.hddSize * 1024) / 512) * 1024 * 1024);
	hddInfo[60] = (nbSectors & 0xFFFF);
	hddInfo[61] = (nbSectors >> 16);

	hddImage = fopen(config.hddFile, "r+b");											// Open HDD image file
	if(!hddImage)																		// It doesn't exist yet ?
	{
		hddImage = fopen(config.hddFile, "w+b");										// OK, let's create it

		DialogBox(hInst, MAKEINTRESOURCE(IDD_ZF_PROGRESS), NULL, zfProgressDlgProc);	// Show "Please wait..." dialog which handles the zero-filling
	}

	setvbuf(hddImage, NULL, _IONBF, 0);
	
	return 1;
}

void HDDshutdown()
{
	fclose(hddImage);
}

static u32 HDDgetLBA()
{
	if(dev9.ata_regs.select & 0x40)
	{
		return ((dev9.ata_regs.sector) |
				(dev9.ata_regs.lcyl << 8) |
				(dev9.ata_regs.hcyl << 16) |
				((dev9.ata_regs.select & 0x0F) << 24));
	}
	else
	{
		dev9.ata_regs.status |= ATA_STAT_ERR;
		dev9.ata_regs.error |= ATA_ERR_ABORT;

		printf("DEV9 ERROR : tried to get LBA address while LBA mode disabled\n");

		return -1;
	}

	return -1;
}

static int HDDseek()
{
	u32 lba;
	u64 pos;

	lba = HDDgetLBA();
	if(lba == -1)
		return -1;

	pos = (lba * 512);
	fsetpos(hddImage, &pos);

	return 0;
}

void HDDreadDMA8Mem(u32 *pMem, int size)
{
	static int transferred;

	printf("DEV9 : DMA read, size %i, transferred %i, total size %i\n", size, transferred, (dev9.ata_regs.nsector * 512));

	if(HDDseek() == 0)
		fread(pMem, size, 1, hddImage);

	transferred += size;
	if(transferred >= (dev9.ata_regs.nsector * 512))
	{
		triggerIRQ(3, 0x6C);
		transferred = 0;
	}
}

void HDDwriteDMA8Mem(u32 *pMem, int size)
{
	static int transferred;

	printf("DEV9 : DMA write, size %i, transferred %i, total size %i\n", size, transferred, (dev9.ata_regs.nsector * 512));

	if(HDDseek() == 0)
		fwrite(pMem, size, 1, hddImage);

	transferred += size;
	if(transferred >= (dev9.ata_regs.nsector * 512))
	{
		triggerIRQ(3, 0x6C);
		transferred = 0;
	}
}