﻿using System;
using System.Windows.Input;
using System.Windows.Media;
using System.ComponentModel;
using System.IO.IsolatedStorage;
using System.IO;
using System.Linq;
using System.Collections.ObjectModel;
using Emulation.Computers.ZxSpectrum;
using System.Windows.Threading;
using System.Threading;
using SpecNiX;
using System.Windows;

namespace SilverlightSpecNix
{
    public class SpecNixFileInfo
    {
        string mPath;

        public string Path
        {
            get { return mPath; }
        }

        public string Name
        {
            get { return System.IO.Path.GetFileNameWithoutExtension(mPath); }
        }

        public string Extension
        {
            get { return System.IO.Path.GetExtension(mPath); }
        }

        public SpecNixFileInfo(string path)
        {
            mPath = path;
        }
    }

    public class Model : INotifyPropertyChanged
    {
        EditableBitmapImage mEditableBitmapImage;
        readonly ObservableCollection<SpecNixFileInfo> mSpecNixFiles;
        //
        readonly DispatcherTimer mTimer;
        //
        Spectrum mSpectrum;
        RzxPlayer mRzxPlayer;
        //
        bool _IsRunning;
        public event Action IsRunningChangedAsync;

        readonly object mSyncObject = new object();
        Thread mSpectrumThread;

        public ImageSource DisplayBitmap
        {
            get { return mEditableBitmapImage.BitmapImage; }
        }

        public ObservableCollection<SpecNixFileInfo> SpecNixFiles
        {
            get { return mSpecNixFiles; }
        }

        public bool IsRunning
        {
            get { return _IsRunning; }
            private set { _IsRunning = value; if (IsRunningChangedAsync != null) IsRunningChangedAsync(); }
        }

        public Model()
        {
            mSpecNixFiles = new ObservableCollection<SpecNixFileInfo>();
            mEditableBitmapImage = new EditableBitmapImage(256, 192);
            mTimer = new DispatcherTimer() { Interval = TimeSpan.FromMilliseconds(50) };
            mTimer.Tick += mTimer_Tick;
        }

        public void Initialize()
        {
            //
            RefreshFiles();
        }

        public void RefreshFiles()
        {
            while (mSpecNixFiles.Count > 0)
                mSpecNixFiles.RemoveAt(0);
            //mSpecNixFiles.Clear();
            IsolatedStorageFile store = IsolatedStorageFile.GetUserStoreForApplication();
            var files = (from file in store.GetFileNames()
                         let ext = Path.GetExtension(file)
                         where ((ext == ".z80") || (ext == ".sna") || (ext == ".rzx"))
                         select new SpecNixFileInfo(file)).ToArray();
            foreach (var file in files)
                mSpecNixFiles.Add(file);
        }

        void mTimer_Tick(object sender, EventArgs e)
        {
            var spectrum = mSpectrum;             
            if (spectrum != null)
            {
                spectrum.Display.UpdateScreen();
                mEditableBitmapImage.WriteData(spectrum.Display.ScreenBuffer);
            }
            else
            {
                mTimer.Stop();
            }
        }

        public void Run(SpecNixFileInfo fileInfo)
        {
            if (fileInfo != null)
            {
                IsolatedStorageFile store = IsolatedStorageFile.GetUserStoreForApplication();
                //Check if file exist - for the case in wich isolated storage is deleted by user
                if (store.FileExists(fileInfo.Path) == false)
                {
                    MessageBox.Show("File does not exists", "Error", MessageBoxButton.OK);
                    RefreshFiles();
                    return;
                }
                //Load and run the file
                using (IsolatedStorageFileStream stream = store.OpenFile(fileInfo.Path, 
                    FileMode.Open, FileAccess.Read))
                {
                    try
                    {
                        switch (fileInfo.Extension)
                        {
                            case ".z80":
                                Spectrum spectrum = (Spectrum)new Z80Loader().Load(stream);
                                RunGame(spectrum);
                                break;
                            case ".sna":
                                spectrum = new Spectrum48();
                                SnaLoader.Load48(spectrum.Memory, spectrum.Z80, stream);
                                RunGame(spectrum);
                                break;
                            case ".rzx":
                                RzxPlayer player = new RzxPlayer(stream);
                                RunRzx(player);
                                break;
                            default:
                                throw new InvalidOperationException();
                        }
                    }
                    catch (LoaderException ex)
                    {
                        MessageBox.Show(string.Format("Error loading file: {0}", ex.Message), "Error", MessageBoxButton.OK);
                    }
                }
            }
            else
            {
                RunGame(new Spectrum48());
            }
        }

        public void Stop()
        {
            if (IsRunning)
            {
                if (mRzxPlayer != null)
                    mRzxPlayer.Stop();
                else if (mSpectrum != null)
                    mSpectrum.Stop();
            }
        }

        void RunRzx(RzxPlayer player)
        {
            if (IsRunning || player == null)
                throw new InvalidOperationException();
            lock (mSyncObject) { IsRunning = true; mRzxPlayer = player; mSpectrum = player.Spectrum; }
            mTimer.Start();
            mSpectrumThread = new Thread(() =>
            {
                try
                {
                    player.Run();
                }
                finally
                {
                    //App.Current.RootVisual.Dispatcher.BeginInvoke(() => { mTimer.Stop(); });
                    lock (mSyncObject) { IsRunning = false; mRzxPlayer = null; mSpectrum = null; }
                }
            }) { IsBackground = true };
            mSpectrumThread.Start();
        }

        void RunGame(Spectrum spectrum)
        {
            if (IsRunning||spectrum==null)
                throw new InvalidOperationException();
            lock (mSyncObject) { IsRunning = true; mSpectrum = spectrum; }
            mTimer.Start();
            mSpectrumThread = new Thread(() =>
                {
                    try
                    {
                        spectrum.Run();
                    }
                    finally
                    {                        
                        //App.Current.RootVisual.Dispatcher.BeginInvoke(() => { mTimer.Stop(); });
                        lock (mSyncObject) { IsRunning = false; mSpectrum = null; }
                    }
                }) { IsBackground = true };
            mSpectrumThread.Start();
        }

        public void KeyDown(Key key)
        {
            var spectrum = mSpectrum;
            if (spectrum != null)
                spectrum.Keyboard.OnKeyDown(key);
        }

        public void KeyUp(Key key)
        {
            var spectrum = mSpectrum;
            if (spectrum != null)
                spectrum.Keyboard.OnKeyUp(key);
        }

        #region INotifyPropertyChanged Members

        public event PropertyChangedEventHandler PropertyChanged;

        void SignalPropertyChanged(string propertyName)
        {
            if (PropertyChanged != null)
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
        }

        #endregion
    }
}
