/* ////////////////////////////////////////////////////////////////////////////
 * This file implements the z80 file format decoding logic
 * 
 * Written by: Pedro Anuarbe Corts
 * ////////////////////////////////////////////////////////////////////////////
 * CREDITS:
 * 
 * This code file is ["inspired by"|"based on"|"a port of"|"posible due"] the 
 * following resources:
 *      - Z80 file format specification (http://www.worldofspectrum.org/faq/reference/z80format.htm)
 * 
 * ////////////////////////////////////////////////////////////////////////////
 * LICENSE: Microsoft Permissive License (Ms-PL)
 *
 *     
 * This license governs use of the accompanying software. If you use the software,
 * you accept this license. If you do not accept the license, do not use the software.
 *
 * 1. Definitions
 *
 * The terms reproduce, reproduction, derivative works, and distribution 
 * have the same meaning here as under U.S. copyright law.
 *
 * A contribution is the original software, or any additions or changes to the software.
 *
 * A contributor is any person that distributes its contribution under this license.
 * 
 * Licensed patents are a contributors patent claims that read directly on its contribution.
 *
 *
 * 2. Grant of Rights
 *
 * (A) Copyright Grant- Subject to the terms of this license, including the license conditions
 * and limitations in section 3, each contributor grants you a non-exclusive,
 * worldwide, royalty-free copyright license to reproduce its contribution, prepare derivative
 * works of its contribution, and distribute its contribution or any derivative works that you create.
 *
 * (B) Patent Grant- Subject to the terms of this license, including the license conditions and
 * limitations in section 3, each contributor grants you a non-exclusive, worldwide, 
 * royalty-free license under its licensed patents to make, have made, use, sell, offer for sale,
 * import, and/or otherwise dispose of its contribution in the software or derivative works of the
 * contribution in the software.
 *
 *
 * 3. Conditions and Limitations
 *
 * (A) No Trademark License- This license does not grant you rights to use any contributors name,
 * logo, or trademarks.
 *
 * (B) If you bring a patent claim against any contributor over patents that you claim are infringed
 * by the software, your patent license from such contributor to the software ends automatically.
 *
 * (C) If you distribute any portion of the software, you must retain all copyright, patent, trademark,
 * and attribution notices that are present in the software.
 *
 * (D) If you distribute any portion of the software in source code form, you may do so only under this
 * license by including a complete copy of this license with your distribution. If you distribute any portion
 * of the software in compiled or object code form, you may only do so under a license that complies with this license.
 *
 *(E) The software is licensed as-is. You bear the risk of using it. The contributors give no express warranties,
 * guarantees or conditions. You may have additional consumer rights under your local laws which this license 
 * cannot change. To the extent permitted under your local laws, the contributors exclude the implied warranties
 * of merchantability, fitness for a particular purpose and non-infringement.
 * ////////////////////////////////////////////////////////////////////////////
 */


using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using Emulation.Cpu.Z80;

namespace Emulation.Computers.ZxSpectrum
{
    public class Z80Loader
    {
        public Z80Loader()
        {
        }

        public ISpectrum Load(string filePath)
        {
            using (FileStream inputStream = File.Open(filePath, FileMode.Open))
            {
                return Load(inputStream);
            }
        }

        public ISpectrum Load(Stream inputStream)
        {
            try
            {
                int version;
                ISpectrum spectrum = null;
                BinaryReader reader = new BinaryReader(inputStream);
                //Test file version
                int hardwareMode = 0;
                inputStream.Position = 6;
                int pc = reader.ReadUInt16();
                if (pc == 0)
                {
                    //Version 2 or 3
                    inputStream.Position = 30;
                    int v23 = reader.ReadUInt16();
                    if (v23 == 23)
                        version = 2;
                    else if ((v23 == 54) || (v23 == 55))
                        version = 3;
                    else
                        throw new Exception("Version not supported");
                    //Buscamos la version
                    inputStream.Position = 34;
                    hardwareMode = reader.ReadByte();
                    switch (hardwareMode)
                    {
                        case 0:
                            spectrum = new Spectrum48();
                            break;
                        case 3:
                            spectrum = new Spectrum128();
                            break;
                        default:
                            throw new Exception("z80 format not supported");
                    }

                }
                else
                {
                    version = 1;
                    spectrum = new Spectrum48();
                }
                //Rebobinamos y cargamos
                inputStream.Position = 0;

                spectrum.Z80.SetRegister(Z80Registers.A, reader.ReadByte());
                spectrum.Z80.SetRegister(Z80Registers.F, reader.ReadByte());
                spectrum.Z80.SetRegister(Z80Registers.BC, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.HL, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.PC, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.SP, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.I, reader.ReadByte());
                spectrum.Z80.SetRegister(Z80Registers.R, 0x7F & reader.ReadByte());
                //
                byte b = reader.ReadByte();
                if (b == 255) b = 1;
                spectrum.Z80.SetRegister(Z80Registers.R, spectrum.Z80.GetRegister(Z80Registers.R) | ((b & 0x01) << 7));
                byte borderColour = (byte)((b & 0x0C) >> 1);
                bool compressed = (b & 0x020) != 0;
                //
                spectrum.Z80.SetRegister(Z80Registers.DE, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.AltBC, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.AltDE, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.AltHL, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.AltA, reader.ReadByte());
                spectrum.Z80.SetRegister(Z80Registers.AltF, reader.ReadByte());
                spectrum.Z80.SetRegister(Z80Registers.IY, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.IX, reader.ReadUInt16());
                spectrum.Z80.SetRegister(Z80Registers.IFF1, reader.ReadByte());
                spectrum.Z80.SetRegister(Z80Registers.IFF2, reader.ReadByte());
                //
                b = reader.ReadByte();
                spectrum.Z80.SetRegister(Z80Registers.InterruptMode, (byte)(b & 0x03));
#warning falta aadir cosas
                //
                //Si la version es la 1 vamos a cargar la memoria
                if (version == 1)
                {
                    byte[] memoryContent = reader.ReadBytes((int)(inputStream.Length - inputStream.Position - 4));
                    if (compressed)
                        memoryContent = Decompress(memoryContent);
                    //Copiamos la memoria
                    Memory48 memory = (spectrum as Spectrum48).Memory;
                    for (int i = 0; i < memoryContent.Length; i++)
                    {
                        memory.WriteByte(i + 0x4000, memoryContent[i]);
                    }
                }
                else
                {
                    //Tenemos version 2 o 3
                    int secondHeaderLength = reader.ReadUInt16();
                    spectrum.Z80.SetRegister(Z80Registers.PC, reader.ReadUInt16());
                    hardwareMode = reader.ReadByte();
                    int last07fdPortWrite = reader.ReadByte();
                    reader.ReadBytes(secondHeaderLength - 4);
                    if (hardwareMode == 0 || hardwareMode == 1)
                    {
                        Memory48 memory = (spectrum as Spectrum48).Memory;

                        while (reader.BaseStream.Position < reader.BaseStream.Length)
                        {
                            int length = reader.ReadUInt16();
                            int page = reader.ReadByte();
                            byte[] data = reader.ReadBytes(length);
                            data = Decompress(data);
                            int pos = 0;
                            switch (page)
                            {
                                case 0:
                                    pos = 0;
                                    break;
                                case 4:
                                    pos = 0x8000;
                                    break;
                                case 5:
                                    pos = 0xC000;
                                    break;
                                case 8:
                                    pos = 0x4000;
                                    break;
                                default:
                                    throw new Exception();
                            }
                            for (int i = 0; i < data.Length; ++i)
                                memory.WriteByte(i + pos, data[i]);
                        }
                    }
                    else if (hardwareMode == 3 || hardwareMode == 4)
                    {
                        Memory128 memory = (spectrum as Spectrum128).Memory;
                        memory.SetupBankConfig(last07fdPortWrite);

                        while (reader.BaseStream.Position < reader.BaseStream.Length)
                        {
                            int length = reader.ReadUInt16();
                            int page = reader.ReadByte();
                            byte[] data = reader.ReadBytes(length);
                            data = Decompress(data);

                            int[] targetBank = null;
                            switch (page)
                            {
                                case 0:
                                    targetBank = memory.Rom0;
                                    break;
                                case 2:
                                    targetBank = memory.Rom1;
                                    break;
                                case 3:
                                    targetBank = memory.Bank0;
                                    break;
                                case 4:
                                    targetBank = memory.Bank1;
                                    break;
                                case 5:
                                    targetBank = memory.Bank2;
                                    break;
                                case 6:
                                    targetBank = memory.Bank3;
                                    break;
                                case 7:
                                    targetBank = memory.Bank4;
                                    break;
                                case 8:
                                    targetBank = memory.Bank5;
                                    break;
                                case 9:
                                    targetBank = memory.Bank6;
                                    break;
                                case 10:
                                    targetBank = memory.Bank7;
                                    break;
                                default:
                                    System.Diagnostics.Debug.WriteLine("Unexpected z80 file data");
                                    break;
                            }
                            for (int i = 0; i < data.Length; i++)
                            {
                                targetBank[i] = data[i];
                            }
                        }
                    }
                    else
                        throw new NotSupportedException("Hardware mode not supported");
                }
                return spectrum;

            }
            catch (Exception ex)
            {
                throw new LoaderException("Z80 loader error", ex);
            }
        }           
        

        public byte[] Decompress(byte[] input)
        {
            MemoryStream ms = new MemoryStream(input);
            MemoryStream output = new MemoryStream();

            string state = "Unprefixed";
            int b;
            while (ms.Position < ms.Length)
            {
                b = ms.ReadByte();

                switch (state)
                {
                    case "Unprefixed":
                        if (b != 0xED)
                        {
                            output.WriteByte((byte)b);
                        }
                        else
                        {
                            state = "EdPrefix";
                        }
                        break;
                    case "EdPrefix":
                        if (b != 0xED)
                        {
                            output.WriteByte(0xED);
                            output.WriteByte((byte)b);
                            state = "Unprefixed";
                        }
                        else
                        {
                            state = "EdEdPrefix";
                        }
                        break;
                    case "EdEdPrefix":                        
                        int val = ms.ReadByte();
                        for (int i = 0; i < b; ++i)
                            output.WriteByte((byte)val);
                        state = "Unprefixed";
                        break;
                    default:
                        throw new InvalidOperationException();
                }               
            }
            return output.ToArray();
        }
    }
}
