/* ////////////////////////////////////////////////////////////////////////////
 * This file implements the display decoder of the spectrum
 * 
 * Written by: Pedro Anuarbe Corts
 * ////////////////////////////////////////////////////////////////////////////
 * CREDITS:
 * 
 * This file is based on the following article: 
 *  ZX Display by Alvin Albrecht (http://www.speccy.org/sromero/spectrum/prog/video/zx_display.pdf)
 * 
 * ////////////////////////////////////////////////////////////////////////////
 * LICENSE: Microsoft Permissive License (Ms-PL)
 *
 *     
 * This license governs use of the accompanying software. If you use the software,
 * you accept this license. If you do not accept the license, do not use the software.
 *
 * 1. Definitions
 *
 * The terms reproduce, reproduction, derivative works, and distribution 
 * have the same meaning here as under U.S. copyright law.
 *
 * A contribution is the original software, or any additions or changes to the software.
 *
 * A contributor is any person that distributes its contribution under this license.
 * 
 * Licensed patents are a contributors patent claims that read directly on its contribution.
 *
 *
 * 2. Grant of Rights
 *
 * (A) Copyright Grant- Subject to the terms of this license, including the license conditions
 * and limitations in section 3, each contributor grants you a non-exclusive,
 * worldwide, royalty-free copyright license to reproduce its contribution, prepare derivative
 * works of its contribution, and distribute its contribution or any derivative works that you create.
 *
 * (B) Patent Grant- Subject to the terms of this license, including the license conditions and
 * limitations in section 3, each contributor grants you a non-exclusive, worldwide, 
 * royalty-free license under its licensed patents to make, have made, use, sell, offer for sale,
 * import, and/or otherwise dispose of its contribution in the software or derivative works of the
 * contribution in the software.
 *
 *
 * 3. Conditions and Limitations
 *
 * (A) No Trademark License- This license does not grant you rights to use any contributors name,
 * logo, or trademarks.
 *
 * (B) If you bring a patent claim against any contributor over patents that you claim are infringed
 * by the software, your patent license from such contributor to the software ends automatically.
 *
 * (C) If you distribute any portion of the software, you must retain all copyright, patent, trademark,
 * and attribution notices that are present in the software.
 *
 * (D) If you distribute any portion of the software in source code form, you may do so only under this
 * license by including a complete copy of this license with your distribution. If you distribute any portion
 * of the software in compiled or object code form, you may only do so under a license that complies with this license.
 *
 *(E) The software is licensed as-is. You bear the risk of using it. The contributors give no express warranties,
 * guarantees or conditions. You may have additional consumer rights under your local laws which this license 
 * cannot change. To the extent permitted under your local laws, the contributors exclude the implied warranties
 * of merchantability, fitness for a particular purpose and non-infringement.
 * ////////////////////////////////////////////////////////////////////////////
 */

using System;
using System.Collections.Generic;
using System.Text;
using Emulation.Infrastructure;
using System.Windows.Media;
using System.Linq;

namespace Emulation.Computers.ZxSpectrum
{
    public class Display
    {
        #region Constants
        /// <summary>
        /// Number of rows
        /// </summary>
        private const int RowsCount = 24;
        /// <summary>
        /// Number of columns
        /// </summary>
        private const int ColumnsCount = 32;
        private const int ClientWidth = ColumnsCount * 8;
        private const int ClientHeight = RowsCount * 8;
        private const int ScreenStartAddress = 0x4000;
        private const int AttributeStartAddress = ScreenStartAddress + (ClientWidth * ClientHeight / 8);
        /// <summary>
        /// Border size, in pixels
        /// </summary>
        private const int BorderWidth = 20;
        private const int ScreenWidth = ClientWidth + 2 * BorderWidth;
        private const int ScreenHeight = ClientHeight + 2 * BorderWidth;
        private const int ScreenSize = ScreenWidth * ScreenHeight;

        #endregion

        #region Fields

        protected IMemory memory;
        private bool flashPhase = true;
        //private Bitmap videoBitmap;
        protected byte[] screenData = new byte[6912];
        protected UInt32[] screenBuffer = new UInt32[256 * 192];
        int[] _screenBufferColorIndexes = new int[256*192];

        //Color palette in argb format
        private UInt32[] colorPalette = new UInt32[]
        {
            0xFF000000, //black
            0xFF0000BF, //blue
            0xFFBF0000, //red
            0xFFBF00BF, //magenta
            0xFF00BF00, //green
            0xFF00BFBF, //cyan
            0xFFBFBF00, //yellow
            0xFFBFBFBF, //white
            0xFF000000, //bright black
            0xFF0000FF, //bright blue
            0xFFFF0000, //bright red
            0xFFFF00FF, //bright mangenta
            0xFF00FF00, //bright green
            0xFF00FFFF, //bright cyan
            0xFFFFFF00, //bright yellow
            0xFFFFFFFF, //bright white
        };

        private SolidColorBrush[] _paletteBrushes;

        #endregion

        #region Properties

        public Display(IMemory memory)
        {
            this.memory = memory;

           }

        public UInt32[] ScreenBuffer
        {
            get { return screenBuffer; }
        }

        public int[] ScreenBufferColorIndexes
        {
            get { return _screenBufferColorIndexes; }
        }

        public UInt32[] ColorPalette
        {
            get { return colorPalette; }
        }

        public SolidColorBrush[] ColorPaletteBrushes
        {
            get { return _paletteBrushes; }
        }

        #endregion

        public  void InvertFlashFase()
        {
            flashPhase = !flashPhase;
        }

        public virtual void UpdateScreen()
        {
            for (int i = 0; i < screenData.Length; i++)
            {
                screenData[i] = (byte)memory.ReadByte(0x4000 + i);
            }
            UpdateScreen(screenData);
        }

        protected void UpdateScreen(byte[] screenData)
        {
            //Vamos a obtener el color de cada pixel            
            for (int y = 0; y < ClientHeight; ++y)
            {
                for (int x = 0; x < ClientWidth; ++x)
                {
                    int ccccc = (x >> 3);
                    int lll = (y & 0x38) << 2;
                    int sss = (y & 0x07) << 8;
                    int bb = (y & 0xC0) << 5;
                    UInt16 pixelAddress = (UInt16)(/*0x4000 |*/ bb | sss | lll | ccccc);
                    bool pixelActivated = (screenData[pixelAddress] & (0x80 >> (x % 0x08))) != 0;
                    UInt16 attributeAddress = (UInt16)(/*AttributeStartAddress*/6144 + (((y >> 3) * 32) + (x >> 3)));
                    int pixelAttribute = screenData[attributeAddress];
                    bool flash = (pixelAttribute & 0x80) != 0;
                    if (flashPhase && ((pixelAttribute & 0x80) != 0))
                        pixelAttribute = (pixelAttribute & 0xC0) | (~pixelAttribute & 0x3F);
                    bool bright = (pixelAttribute & 0x40) != 0;
                    int inkIndex = (pixelAttribute & 0x07);
                    if (bright) inkIndex += 8;
                    int paperIndex = (pixelAttribute & 0x38)>>3;
                    if (bright) paperIndex += 8;

                    int colorIndex = pixelActivated ? inkIndex : paperIndex;
                    
                    UInt32 pixelValue = pixelActivated ? colorPalette[inkIndex] : colorPalette[paperIndex];
                    //videoBitmap.SetPixel(x, y, Color.FromArgb((int)pixelValue));
                    screenBuffer[y * ClientWidth + x] = pixelValue;                    
                    //_screenBufferColorIndexes[y * ClientWidth + x] = colorIndex;
                }
            }                        
        }
    }
}
