/* ////////////////////////////////////////////////////////////////////////////
 * This file implements rzx decoding logic
 * 
 * Written by: Pedro Anuarbe Corts
 * ////////////////////////////////////////////////////////////////////////////
 * CREDITS:
 * 
 * This code file is ["inspired by"|"based on"|"a port of"|"posible due"] the 
 * following resources:
 *      - RZX specification (http://www.ramsoft.bbk.org/rzx.html)
 * 
 * ////////////////////////////////////////////////////////////////////////////
 * LICENSE: Microsoft Permissive License (Ms-PL)
 *
 *     
 * This license governs use of the accompanying software. If you use the software,
 * you accept this license. If you do not accept the license, do not use the software.
 *
 * 1. Definitions
 *
 * The terms reproduce, reproduction, derivative works, and distribution 
 * have the same meaning here as under U.S. copyright law.
 *
 * A contribution is the original software, or any additions or changes to the software.
 *
 * A contributor is any person that distributes its contribution under this license.
 * 
 * Licensed patents are a contributors patent claims that read directly on its contribution.
 *
 *
 * 2. Grant of Rights
 *
 * (A) Copyright Grant- Subject to the terms of this license, including the license conditions
 * and limitations in section 3, each contributor grants you a non-exclusive,
 * worldwide, royalty-free copyright license to reproduce its contribution, prepare derivative
 * works of its contribution, and distribute its contribution or any derivative works that you create.
 *
 * (B) Patent Grant- Subject to the terms of this license, including the license conditions and
 * limitations in section 3, each contributor grants you a non-exclusive, worldwide, 
 * royalty-free license under its licensed patents to make, have made, use, sell, offer for sale,
 * import, and/or otherwise dispose of its contribution in the software or derivative works of the
 * contribution in the software.
 *
 *
 * 3. Conditions and Limitations
 *
 * (A) No Trademark License- This license does not grant you rights to use any contributors name,
 * logo, or trademarks.
 *
 * (B) If you bring a patent claim against any contributor over patents that you claim are infringed
 * by the software, your patent license from such contributor to the software ends automatically.
 *
 * (C) If you distribute any portion of the software, you must retain all copyright, patent, trademark,
 * and attribution notices that are present in the software.
 *
 * (D) If you distribute any portion of the software in source code form, you may do so only under this
 * license by including a complete copy of this license with your distribution. If you distribute any portion
 * of the software in compiled or object code form, you may only do so under a license that complies with this license.
 *
 *(E) The software is licensed as-is. You bear the risk of using it. The contributors give no express warranties,
 * guarantees or conditions. You may have additional consumer rights under your local laws which this license 
 * cannot change. To the extent permitted under your local laws, the contributors exclude the implied warranties
 * of merchantability, fitness for a particular purpose and non-infringement.
 * ////////////////////////////////////////////////////////////////////////////
 */


using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using ICSharpCode.SharpZipLib.Zip;
using ICSharpCode.SharpZipLib.GZip;
using ICSharpCode.SharpZipLib.Zip.Compression;
using Emulation.Computers.ZxSpectrum;

namespace SpecNiX
{
    public class RzxRecordingFrame
    {
        private int fetchCounter;
        private int inCounter;
        private byte[] inValues;        

        public int FetchCounter
        {
            get { return fetchCounter; }
            set { fetchCounter = value; }
        }

        public int InCounter
        {
            get { return inCounter; }
            set { inCounter = value; }
        }

        public byte[] InValues
        {
            get { return inValues; }
            set { inValues = value; }
        }
    }

    public class RzxLoader
    {
        private Queue<RzxRecordingFrame> frameQueue;

        private int majorRevisionNumber;
        private int minorRevisionNumber;
        private string creator;
        private int creatorMinorVersion;
        private int creatorMajorVersion;
        private Spectrum spectrum;

        public RzxLoader()
        {
        }

        public int FrameCount
        {
            get
            {
                if (frameQueue == null)
                    throw new InvalidOperationException();
                else
                    return frameQueue.Count;
            }
        }

        public Spectrum Load(string fileName)
        {   
            using (FileStream inputStream = File.Open(fileName, FileMode.Open))
            {
                return Load(inputStream);
            }
        }

        public Spectrum Load(Stream inputStream)
        {
            try
            {
                BinaryReader reader = new BinaryReader(inputStream, Encoding.UTF8);

                if (new string(reader.ReadChars(4)) != "RZX!")
                    throw new LoaderException("File is not in rzx format");
                majorRevisionNumber = reader.ReadByte();
                minorRevisionNumber = reader.ReadByte();
                bool signed = (reader.ReadUInt32() & 0x1) != 0;
                if (signed)
                    throw new LoaderException("Signed rzx not supported");

                while (reader.BaseStream.Position < reader.BaseStream.Length)
                {
                    byte blockId = reader.ReadByte();
                    switch (blockId)
                    {
                        case 0x10:
                            int creatorBlockLength = reader.ReadInt32();
                            creator = new string(reader.ReadChars(20));
                            creatorMajorVersion = reader.ReadUInt16();
                            creatorMinorVersion = reader.ReadUInt16();
                            byte[] customData = reader.ReadBytes(creatorBlockLength - 29);
                            break;
                        case 0x30:
                            int snapshotBlockLength = reader.ReadInt32();
                            uint snapshotFlags = reader.ReadUInt32();
                            if ((snapshotFlags & 0x1) != 0)
                                throw new LoaderException("External snapshots not supported");
                            bool compressedData = (snapshotFlags & 0x2) != 0;
                            string snapshotExt = new string(reader.ReadChars(4));
                            int snapshotLength = reader.ReadInt32();
                            byte[] snapshotData = reader.ReadBytes(snapshotBlockLength - 17);
                            snapshotData = Inflate(snapshotData);
                            if (snapshotData.Length != snapshotLength)
                                throw new LoaderException("Inflater error");

                            switch (snapshotExt.Substring(0, 3).ToUpper())
                            {
                                case "SNA":
                                    //new SnaLoader(spectrum).Load(new MemoryStream(snapshotData));
                                    throw new NotImplementedException("sna based rzx files not supported");
                                case "Z80":
                                    spectrum = (Spectrum)new Z80Loader().Load(new MemoryStream(snapshotData));
                                    break;
                                default:
                                    throw new LoaderException("Snapshot format not supported");
                            }
                            break;
                        case 0x80:
                            int inputRecordingBlockLength = reader.ReadInt32();
                            int numberOfFrames = reader.ReadInt32();
                            reader.ReadByte();
                            int tStatesCounter = reader.ReadInt32();
                            spectrum.Z80.SetRegister(Emulation.Cpu.Z80.Z80Registers.TStates, tStatesCounter);
                            int flags = reader.ReadInt32();
                            bool compressed = (flags & 0x02) != 0;
                            bool encrypted = (flags & 0x01) != 0;
                            byte[] irbData = reader.ReadBytes(inputRecordingBlockLength - 18);
                            irbData = Inflate(irbData);
                            frameQueue = GetRecordingFrames(irbData);
                            if (frameQueue.Count != numberOfFrames)
                                throw new LoaderException("Invalid frame count");
                            break;
                        default:
                            goto ScanFinished;
                    }
                }
            ScanFinished:
                return spectrum;

            }
            catch (Exception ex)
            {
                
                throw new LoaderException("rzx loader error", ex);
            }
        }

        public RzxRecordingFrame GetNextFrame()
        {
            if (frameQueue.Count > 0)
                return frameQueue.Dequeue();
            else
                return null;
        }

        private Queue<RzxRecordingFrame> GetRecordingFrames(byte[] inputRecordingData)
        {
            Queue<RzxRecordingFrame> queue = new Queue<RzxRecordingFrame>();
            byte[] lastFrameInput = null;
            MemoryStream ms = new MemoryStream(inputRecordingData);

            using (BinaryReader reader = new BinaryReader(ms))
            {
                while (reader.BaseStream.Position < reader.BaseStream.Length)
                {
                    RzxRecordingFrame frame = new RzxRecordingFrame();
                    frame.FetchCounter = reader.ReadUInt16();                    
                    frame.InCounter = reader.ReadUInt16();
                    if (frame.InCounter == 0xFFFF)
                    {
                        if (lastFrameInput == null)
                            throw new InvalidOperationException();
                        
                        frame.InCounter = lastFrameInput.Length;
                        frame.InValues = (byte[])lastFrameInput.Clone();
                    }
                    else
                    {                        
                        lastFrameInput = reader.ReadBytes(frame.InCounter);
                        frame.InValues = (byte[])lastFrameInput.Clone();
                    }                    
                    queue.Enqueue(frame);
                }
            }
            return queue;
        }

        private byte[] Inflate(byte[] input)
        {
            MemoryStream inflatedStream = new MemoryStream();

            Inflater inflater = new Inflater(false);
            inflater.SetInput(input);
            byte[] buffer = new byte[4096];
            while (!inflater.IsFinished)
            {
                int readed = inflater.Inflate(buffer, 0, buffer.Length);
                inflatedStream.Write(buffer, 0, readed);
            }
            return inflatedStream.ToArray();
        }
    }
}
