using System;
using System.Collections.Generic;
using System.Text;
using Emulation.Computers.ZxSpectrum;
using System.Diagnostics;
using System.Threading;
using Infrastructure;
using System.IO;

namespace SpecNiX
{
    public class RzxPlayer
    {
        bool _stopping;
        bool _isRunning;
        Int32 _progressPercentage;
        private int frameInterval = 20;
        private RzxLoader loader;
        private RzxRecordingFrame currentFrame;
        private Queue<byte> currentInQueue;
        private Spectrum spectrum;
        private IProgressReport progressReport = new NullProgressReport();
        

        public event Action InterruptRaised;
        
        public RzxPlayer(string fileName)
        {
            this.loader = new RzxLoader();
            this.spectrum = loader.Load(fileName);
            this.spectrum.Ula.PortReadHandler = PortReadHandler;
        }

        public RzxPlayer(Stream stream)
        {
            this.loader = new RzxLoader();
            this.spectrum = loader.Load(stream);
            this.spectrum.Ula.PortReadHandler = PortReadHandler;
        }



        /// <summary>
        /// The time elapsed between interrupt frames
        /// </summary>
        public int FrameInterval
        {
            get { return frameInterval; }
            set { frameInterval = value; }
        }

        public IProgressReport ProgressReport
        {
            get { return progressReport; }
            set { progressReport = value; }
        }

        public Spectrum Spectrum
        {
            get { return spectrum; }
            set { spectrum = value; }
        }

        public RzxRecordingFrame CurrentFrame
        {
            get { return currentFrame; }
        }

        public bool IsRunning
        {
            get { return _isRunning; }
        }

        public Int32 ProgressPercentage
        {
            get { return _progressPercentage; }
        }

        public void Stop()
        {
            _stopping = true;
        }

        public void Run()
        {
            int frameFetchCount = 0;
            int totalFrames = loader.FrameCount;
            //Tenemos que sacar 100 puntos de porcentaje
            int mod = totalFrames / 100;
            int nFrame = 0;
            _progressPercentage = 0;
            _isRunning = true;

            while ((currentFrame = loader.GetNextFrame()) != null)
            {
                if (_stopping == true)
                {
                    _stopping = false;
                    _isRunning = false;
                    return;
                }
                frameFetchCount = currentFrame.FetchCounter;
                //Stopwatch timer = new Stopwatch();
                //timer.Start();
                DateTime startedOn = DateTime.Now;
                currentInQueue = new Queue<byte>(currentFrame.InValues);
                spectrum.Z80.FetchCounter = 0;
                while (spectrum.Z80.FetchCounter<frameFetchCount)
                {
                    spectrum.Z80.Step();
                }
                if (spectrum.Z80.FetchCounter != frameFetchCount)
                {

                }
                if (currentInQueue.Count != 0)
                    throw new RzxPlayerException("The input queue is not empty.");
                spectrum.Z80.MaskableInterrupt();

                if (InterruptRaised != null)
                    InterruptRaised();

                int elapsedTime = (int)(DateTime.Now - startedOn).TotalMilliseconds;
                int waitTime = (int)(frameInterval - elapsedTime);
                if (waitTime > 0)
                    Thread.Sleep(waitTime);
                
                //Calculo del progreso
                if (nFrame % mod == 0)
                {
                    _progressPercentage = (int)(100 * (1f * nFrame / totalFrames));
                    progressReport.ReportProgress(_progressPercentage);
                }
                nFrame++;                
            }
        }

        public int PortReadHandler(int port)
        {
            if (currentInQueue.Count > 0)
                return currentInQueue.Dequeue();
            throw new RzxPlayerException("the input queue is empty");
        }
    }
}
