/* ////////////////////////////////////////////////////////////////////////////
 * This file implements a Z80 microprocessor.
 * 
 * Written by: Pedro Anuarbe Corts
 * ////////////////////////////////////////////////////////////////////////////
 * CREDITS:
 * 
 * This code file is ["inspired by"|"based on"|"a port of"|"posible due"] the 
 * following resources:
 *      - Portable Z80 emulator v3.5 by Juergen Buchmueller
 *      - Emulador en Java de un Sinclair ZX Spectrum 48K v1.0B by Alberto 
 *          Snchez Terrn
 *      - http://www.worldofspectrum.org/
 *      - http://www.speccy.org/portal/
 *      - http://fuse-emulator.sourceforge.net/
 *      - All the people contributing with documentation, source code or whatever
 * 
 * ////////////////////////////////////////////////////////////////////////////
 * LICENSE: Microsoft Permissive License (Ms-PL)
 *
 *     
 * This license governs use of the accompanying software. If you use the software,
 * you accept this license. If you do not accept the license, do not use the software.
 *
 * 1. Definitions
 *
 * The terms reproduce, reproduction, derivative works, and distribution 
 * have the same meaning here as under U.S. copyright law.
 *
 * A contribution is the original software, or any additions or changes to the software.
 *
 * A contributor is any person that distributes its contribution under this license.
 * 
 * Licensed patents are a contributors patent claims that read directly on its contribution.
 *
 *
 * 2. Grant of Rights
 *
 * (A) Copyright Grant- Subject to the terms of this license, including the license conditions
 * and limitations in section 3, each contributor grants you a non-exclusive,
 * worldwide, royalty-free copyright license to reproduce its contribution, prepare derivative
 * works of its contribution, and distribute its contribution or any derivative works that you create.
 *
 * (B) Patent Grant- Subject to the terms of this license, including the license conditions and
 * limitations in section 3, each contributor grants you a non-exclusive, worldwide, 
 * royalty-free license under its licensed patents to make, have made, use, sell, offer for sale,
 * import, and/or otherwise dispose of its contribution in the software or derivative works of the
 * contribution in the software.
 *
 *
 * 3. Conditions and Limitations
 *
 * (A) No Trademark License- This license does not grant you rights to use any contributors name,
 * logo, or trademarks.
 *
 * (B) If you bring a patent claim against any contributor over patents that you claim are infringed
 * by the software, your patent license from such contributor to the software ends automatically.
 *
 * (C) If you distribute any portion of the software, you must retain all copyright, patent, trademark,
 * and attribution notices that are present in the software.
 *
 * (D) If you distribute any portion of the software in source code form, you may do so only under this
 * license by including a complete copy of this license with your distribution. If you distribute any portion
 * of the software in compiled or object code form, you may only do so under a license that complies with this license.
 *
 *(E) The software is licensed as-is. You bear the risk of using it. The contributors give no express warranties,
 * guarantees or conditions. You may have additional consumer rights under your local laws which this license 
 * cannot change. To the extent permitted under your local laws, the contributors exclude the implied warranties
 * of merchantability, fitness for a particular purpose and non-infringement.
 * ////////////////////////////////////////////////////////////////////////////
 */

using System;
using System.Collections.Generic;
using System.Text;
using Emulation.Infrastructure;

namespace Emulation.Cpu.Z80
{
    public sealed partial class Z80
    {
        private IMemory memory;
        private IInputOutput io;
        private int tStates;
        private bool isHalted;
        public long FetchCounter;
        public int IrqVector = 0xFF;
        
        public Z80(IMemory memory, IInputOutput io)
        {
            this.memory = memory;
            this.io = io;
            InitializeTables();
            Reset();
        }

        public Z80(): this(null, null)
        {

        }

        public IMemory Memory
        {
            get { return memory; }
            set { memory = value; }
        }

        public IInputOutput IO
        { 
            get { return io; }
            set { io = value; }
        }

        public void Step()
        {
            int opcode = memory.ReadByte(PC);
            PC = ++PC;
            R = ++R & 0x7F;
            FetchCounter++;

            #region Main switch
            switch (opcode)
            {
                case 0x00:
                    //NOP
                    tStates += 4;
                    break;
                case 0x01:
                    //LD BC, nn
                    tStates += 10;
                    C = memory.ReadByte(PC++);
                    B = memory.ReadByte(PC++);
                    break;
                case 0x02:
                    //LD (BC), A
                    tStates += 7;
                    memory.WriteByte(BC, A);
                    break;
                case 0x03:
                    {
                        //INC BC
                        tStates += 6;
                        BC = (BC+1) & 0xFFFF;
                        break;
                    }
                case 0x04:
                    {
                        //INC B
                        tStates += 4;
                        B = Inc(B);
                        break;
                    }
                case 0x05:
                    {
                        ///DEC B
                        tStates += 4;
                        B = Dec(B);
                        break;
                    }
                case 0x06:
                    {
                        //LD B,n
                        tStates += 7;
                        B = memory.ReadByte(PC++);
                        break;
                    }
                case 0x07:
                    {
                        //RLCA
                        tStates += 4;
                        Rlca();
                        break;
                    }
                case 0x08:
                    {
                        //EX AF,AF'
                        tStates += 4;
                        int aux = A;
                        A = AltA;
                        AltA = aux;
                        aux = F;
                        F = AltF;
                        AltF = aux;
                        break;
                    }
                case 0x09:
                    {
                        //ADD HL,BC
                        tStates += 11;
                        HL = Add16(HL, BC);
                        break;
                    }
                case 0x0A:
                    {
                        //LD A,(BC)
                        tStates += 7;
                        A = memory.ReadByte(BC);
                        break;
                    }
                case 0x0B:
                    {
                        //DEC BC
                        tStates += 6;
                        BC = (BC-1) & 0xFFFF;
                        break;
                    }
                case 0x0C:
                    {
                        //INC C
                        tStates += 4;
                        C = Inc(C);
                        break;
                    }
                case 0x0D:
                    {
                        //DEC C
                        tStates += 4;
                        C = Dec(C);
                        break;
                    }
                case 0x0E:
                    {
                        //LD C,n
                        tStates += 7;
                        C = memory.ReadByte(PC++);
                        break;
                    }
                case 0x0F:
                    {
                        //RRCA
                        tStates += 4;
                        Rrca();
                        break;
                    }
                case 0x10:
                    {
                        //DJNZ e
                        B = --B & 0xFF;
                        if (B != 0)
                        {
                            tStates += 13;
                            int d = (sbyte)memory.ReadByte(PC++);
                            PC += d;
                        }
                        else
                        {
                            tStates += 8;
                            PC++;
                        }
                        break;
                    }
                case 0x11:
                    {
                        //LD DE,nn
                        tStates += 10;
                        E = memory.ReadByte(PC++);
                        D = memory.ReadByte(PC++);
                        break;
                    }
                case 0x12:
                    {
                        //LD (DE),A
                        tStates += 7;
                        memory.WriteByte(DE, A);
                        break;
                    }
                case 0x13:
                    {
                        //INC DE
                        tStates += 6;
                        DE = (DE+1) & 0xFFFF;
                        break;
                    }
                case 0x14:
                    {
                        //INC D
                        tStates += 4;
                        D = Inc(D);
                        break;
                    }
                case 0x15:
                    {
                        //DEC D
                        tStates += 4;
                        D = Dec(D);
                        break;
                    }
                case 0x16:
                    {
                        //LD D,n
                        tStates += 7;
                        D = memory.ReadByte(PC++);
                        break;
                    }
                case 0x17:
                    {
                        //RLA
                        tStates += 4;
                        Rla();
                        break;
                    }
                case 0x18:
                    {
                        //JR e
                        tStates += 12;
                        int d = (sbyte)memory.ReadByte(PC++);
                        PC += d;
                        break;
                    }
                case 0x19:
                    {
                        //ADD HL,DE
                        tStates += 11;
                        HL = Add16(HL, DE);
                        break;
                    }
                case 0x1A:
                    {
                        //LD A,(DE)
                        tStates += 7;
                        A = memory.ReadByte(DE);
                        break;
                    }
                case 0x1B:
                    {
                        //DEC DE
                        tStates += 6;
                        DE = (DE-1) & 0xFFFF;
                        break;
                    }
                case 0x1C:
                    {
                        //INC E
                        tStates += 4;
                        E = Inc(E);
                        break;
                    }
                case 0x1D:
                    {
                        //DEC E
                        tStates += 4;
                        E = Dec(E);
                        break;
                    }
                case 0x1E:
                    {
                        //LD E,n
                        tStates += 7;
                        E = memory.ReadByte(PC++);
                        break;
                    }
                case 0x1F:
                    {
                        //RRA
                        tStates += 4;
                        Rra();
                        break;
                    }
                case 0x20:
                    {
                        //JR NZ,e
                        if ((F & ZF) == 0)
                        {
                            tStates += 12;
                            int d = (sbyte)memory.ReadByte(PC++);
                            PC += d;
                        }
                        else
                        {
                            tStates += 7;
                            PC++;
                        }
                        break;
                    }
                case 0x21:
                    {
                        //LD HL,nn
                        tStates += 10;
                        L = memory.ReadByte(PC++);
                        H = memory.ReadByte(PC++);
                        break;
                    }
                case 0x22:
                    {
                        //LD (nn),HL
                        tStates += 16;
                        memory.WriteWord(memory.ReadWord(PC), HL);
                        PC += 2;
                        break;
                    }
                case 0x23:
                    {
                        //INC HL
                        tStates += 6;
                        HL = (HL+1) & 0xFFFF;
                        break;
                    }
                case 0x24:
                    {
                        //INC H
                        tStates += 4;
                        H = Inc(H);
                        break;
                    }
                case 0x25:
                    {
                        //DEC H
                        tStates += 4;
                        H = Dec(H);
                        break;
                    }
                case 0x26:
                    {
                        //LD H,n
                        tStates += 7;
                        H = memory.ReadByte(PC++);
                        break;
                    }
                case 0x27:
                    {
                        //DAA
                        tStates += 4;
                        Daa();
                        break;
                    }
                case 0x28:
                    {
                        ///JR Z,e
                        if ((F & ZF) != 0)
                        {
                            tStates += 12;
                            int d = (sbyte)memory.ReadByte(PC++);
                            PC += d;
                        }
                        else
                        {
                            tStates += 7;
                            PC++;
                        }
                        break;
                    }
                case 0x29:
                    {
                        //ADD HL,HL
                        tStates += 11;
                        HL = Add16(HL, HL);
                        break;
                    }
                case 0x2A:
                    {
                        //LD HL,(nn)
                        tStates += 16;
                        HL = memory.ReadWord(memory.ReadWord(PC));
                        PC += 2;
                        break;
                    }
                case 0x2B:
                    {
                        //DEC HL
                        tStates += 6;
                        HL = (HL-1) & 0xFFFF;
                        break;
                    }
                case 0x2C:
                    {
                        //INC L
                        tStates += 4;
                        L = Inc(L);
                        break;
                    }
                case 0x2D:
                    {
                        //DEC L
                        tStates += 4;
                        L = Dec(L);
                        break;
                    }
                case 0x2E:
                    {
                        //LD L,n
                        tStates += 7;
                        L = memory.ReadByte(PC++);
                        break;
                    }
                case 0x2F:
                    {
                        //CPL
                        tStates += 4;
                        A ^= 0xFF;
                        F = (F & (SF | ZF | PF | CF)) | HF | NF | (A & (YF | XF));
                        break;
                    }
                case 0x30:
                    {
                        ///JR NC,e
                        if ((F & CF) == 0)
                        {
                            tStates += 12;
                            int d = (sbyte)memory.ReadByte(PC++);
                            PC += d;
                        }
                        else
                        {
                            tStates += 7;
                            PC++;
                        }
                        break;
                    }
                case 0x31:
                    {
                        //LD SP,nn
                        tStates += 10;
                        SP = memory.ReadWord(PC);
                        PC += 2;
                        break;
                    }
                case 0x32:
                    {
                        //LD (nn),A
                        tStates += 13;
                        memory.WriteByte(memory.ReadWord(PC), A);
                        PC += 2;
                        break;
                    }
                case 0x33:
                    {
                        //INC SP
                        tStates += 6;
                        SP = ++SP & 0xFFFF;
                        break;
                    }
                case 0x34:
                    {
                        //INC (HL)
                        tStates += 11;
                        memory.WriteByte(HL, Inc(memory.ReadByte(HL)));
                        break;
                    }
                case 0x35:
                    {
                        //DEC (HL)
                        tStates += 11;
                        memory.WriteByte(HL, Dec(memory.ReadByte(HL)));
                        break;
                    }
                case 0x36:
                    {
                        //LD (HL),n
                        tStates += 10;
                        memory.WriteByte(HL, memory.ReadByte(PC++));
                        break;
                    }
                case 0x37:
                    {
                        //SCF
                        tStates += 4;
                        F = (F & (SF | ZF | PF)) | CF | (A & (YF | XF));
                        break;
                    }
                case 0x38:
                    {
                        //JR C,e
                        if ((F & CF) != 0)
                        {
                            tStates += 12;
                            int d = (sbyte)memory.ReadByte(PC++);
                            PC += d;
                        }
                        else
                        {
                            tStates += 7;
                            PC++;
                        }
                        break;
                    }
                case 0x39:
                    {
                        //ADD HL,SP
                        tStates += 11;
                        HL = Add16(HL, SP);
                        break;
                    }
                case 0x3A:
                    {
                        //LD A,(nn)
                        tStates += 13;
                        A = memory.ReadByte(memory.ReadWord(PC));
                        PC += 2;
                        break;
                    }
                case 0x3B:
                    {
                        //DEC SP
                        tStates += 6;
                        SP = --SP & 0xFFFF;
                        break;
                    }
                case 0x3C:
                    {
                        //INC A
                        tStates += 4;
                        A = Inc(A);
                        break;
                    }
                case 0x3D:
                    {
                        //DEC A
                        tStates += 4;
                        A = Dec(A);
                        break;
                    }
                case 0x3E:
                    {
                        //LD A,n
                        tStates += 7;
                        A = memory.ReadByte(PC++);
                        break;
                    }
                case 0x3F:
                    {
                        ///CCF
                        tStates += 4;
                        F = ((F & (SF | ZF | PF | CF)) | ((F & CF) << 4) | (A & (YF | XF))) ^ CF;
                        break;
                    }
                case 0x40:
                    {
                        //LD B,B
                        tStates += 4;
                        break;
                    }
                case 0x41:
                    {
                        //LD B,C
                        tStates += 4;
                        B = C;
                        break;
                    }
                case 0x42:
                    {
                        //LD B,D
                        tStates += 4;
                        B = D;
                        break;
                    }
                case 0x43:
                    {
                        //LD B,E
                        tStates += 4;
                        B = E;
                        break;
                    }
                case 0x44:
                    {
                        //LD B,H
                        tStates += 4;
                        B = H;
                        break;
                    }
                case 0x45:
                    {
                        //LD B,L
                        tStates += 4;
                        B = L;
                        break;
                    }
                case 0x46:
                    {
                        //LD B,(HL)
                        tStates += 7;
                        B = memory.ReadByte(HL);
                        break;
                    }
                case 0x47:
                    {
                        //LD B,A
                        tStates += 4;
                        B = A;
                        break;
                    }
                case 0x48:
                    {
                        //LD C,B
                        tStates += 4;
                        C = B;
                        break;
                    }
                case 0x49:
                    {
                        ///LD C,C
                        tStates += 4;
                        break;
                    }
                case 0x4A:
                    {
                        //LD C,D
                        tStates += 4;
                        C = D;
                        break;
                    }
                case 0x4B:
                    {
                        //LD C,E
                        tStates += 4;
                        C = E; break;
                    }
                case 0x4C:
                    {
                        //LD C,H
                        tStates += 4;
                        C = H;
                        break;
                    }
                case 0x4D:
                    {
                        //LD C,L
                        tStates += 4;
                        C = L;
                        break;
                    }
                case 0x4E:
                    {
                        //LD C,(HL)
                        tStates += 7;
                        C = memory.ReadByte(HL);
                        break;
                    }
                case 0x4F:
                    {
                        //LD C,A
                        tStates += 4;
                        C = A;
                        break;
                    }
                case 0x50:
                    {
                        //LD D,B
                        tStates += 4;
                        D = B;
                        break;
                    }
                case 0x51:
                    {
                        //LD D,C
                        tStates += 4;
                        D = C;
                        break;
                    }
                case 0x52:
                    {
                        //LD D,D
                        tStates += 4;
                        break;
                    }
                case 0x53:
                    {
                        //LD D,E
                        tStates += 4;
                        D = E;
                        break;
                    }
                case 0x54:
                    {
                        //LD D,H
                        tStates += 4;
                        D = H;
                        break;
                    }
                case 0x55:
                    {
                        //LD D,L
                        tStates += 4;
                        D = L;
                        break;
                    }
                case 0x56:
                    {
                        //LD D,(HL)
                        tStates += 7;
                        D = memory.ReadByte(HL);
                        break;
                    }
                case 0x57:
                    {
                        //LD D,A
                        tStates += 4;
                        D = A;
                        break;
                    }
                case 0x58:
                    {
                        //LD E,B
                        tStates += 4;
                        E = B;
                        break;
                    }
                case 0x59:
                    {
                        //LD E,C
                        tStates += 4;
                        E = C;
                        break;
                    }
                case 0x5A:
                    {
                        //LD E,D
                        tStates += 4;
                        E = D;
                        break;
                    }
                case 0x5B:
                    {
                        //LD E,E
                        tStates += 4;
                        break;
                    }
                case 0x5C:
                    {
                        //LD E,H
                        tStates += 4;
                        E = H;
                        break;
                    }
                case 0x5D:
                    {
                        //LD E,L
                        tStates += 4;
                        E = L;
                        break;
                    }
                case 0x5E:
                    {
                        //LD E, (HL)
                        tStates += 7;
                        E = memory.ReadByte(HL);
                        break;
                    }
                case 0x5F:
                    {
                        //LD E,A
                        tStates += 4;
                        E = A;
                        break;
                    }
                case 0x60:
                    {
                        //LD H,B
                        tStates += 4;
                        H = B;
                        break;
                    }
                case 0x61:
                    {
                        //LD H,C
                        tStates += 4;
                        H = C;
                        break;
                    }
                case 0x62:
                    {
                        //LD H,D
                        tStates += 4;
                        H = D;
                        break;
                    }
                case 0x63:
                    {
                        //LD H,E
                        tStates += 4;
                        H = E;
                        break;
                    }
                case 0x64:
                    {
                        //LD H,H
                        tStates += 4;
                        break;
                    }
                case 0x65:
                    {
                        //LD H,L
                        tStates += 4;
                        H = L;
                        break;
                    }
                case 0x66:
                    {
                        //LD H,(HL)
                        tStates += 7;
                        H = memory.ReadByte(HL);
                        break;
                    }
                case 0x67:
                    {
                        //LD H,A
                        tStates += 4;
                        H = A;
                        break;
                    }
                case 0x68:
                    {
                        //LD L,B
                        tStates += 4;
                        L = B;
                        break;
                    }
                case 0x69:
                    {
                        //LD L,C
                        tStates += 4;
                        L = C;
                        break;
                    }
                case 0x6A:
                    {
                        //LD L,D
                        tStates += 4;
                        L = D;
                        break;
                    }
                case 0x6B:
                    {
                        //LD L,E
                        tStates += 4;
                        L = E;
                        break;
                    }
                case 0x6C:
                    {
                        //LD L,H
                        tStates += 4;
                        L = H;
                        break;
                    }
                case 0x6D:
                    {
                        //LD L,L
                        tStates += 4;
                        break;
                    }
                case 0x6E:
                    {
                        //LD L,(HL)
                        tStates += 7;
                        L = memory.ReadByte(HL);
                        break;
                    }
                case 0x6F:
                    {
                        //LD L,A
                        tStates += 4;
                        L = A;
                        break;
                    }
                case 0x70:
                    {
                        //LD (HL),B
                        tStates += 7;
                        memory.WriteByte(HL, B);
                        break;
                    }
                case 0x71:
                    {
                        //LD (HL),C
                        tStates += 7;
                        memory.WriteByte(HL, C);
                        break;
                    }
                case 0x72:
                    {
                        //LD (HL),D
                        tStates += 7;
                        memory.WriteByte(HL, D);
                        break;
                    }
                case 0x73:
                    {
                        //LD (HL),E
                        tStates += 7;
                        memory.WriteByte(HL, E);
                        break;
                    }
                case 0x74:
                    {
                        //LD (HL),H
                        tStates += 7;
                        memory.WriteByte(HL, H);
                        break;
                    }
                case 0x75:
                    {
                        //LD (HL),L
                        tStates += 7;
                        memory.WriteByte(HL, L);
                        break;
                    }
                case 0x76:
                    {
                        //HALT
                        tStates += 4;
                        isHalted = true;
                        PC--;
                        break;
                    }
                case 0x77:
                    {
                        //LD (HL),A
                        tStates += 7;
                        memory.WriteByte(HL, A);
                        break;
                    }
                case 0x78:
                    {
                        //LD A,B
                        tStates += 4;
                        A = B;
                        break;
                    }
                case 0x79:
                    {
                        //LD A,C
                        tStates += 4;
                        A = C;
                        break;
                    }
                case 0x7A:
                    {
                        //LD A,D
                        tStates += 4;
                        A = D;
                        break;
                    }
                case 0x7B:
                    {
                        //LD A,E
                        tStates += 4;
                        A = E;
                        break;
                    }
                case 0x7C:
                    {
                        //LD A,H
                        tStates += 4;
                        A = H;
                        break;
                    }
                case 0x7D:
                    {
                        //LD A,L
                        tStates += 4;
                        A = L;
                        break;
                    }
                case 0x7E:
                    {
                        //LD A,(HL)
                        tStates += 7;
                        A = memory.ReadByte(HL);
                        break;
                    }
                case 0x7F:
                    {
                        //LD A,A
                        tStates += 4;
                        break;
                    }
                case 0x80:
                    {
                        //ADD A,B
                        tStates += 4;
                        Add(B);
                        break;
                    }
                case 0x81:
                    {
                        //ADD A,C
                        tStates += 4;
                        Add(C);
                        break;
                    }
                case 0x82:
                    {
                        //ADD A,D
                        tStates += 4;
                        Add(D);
                        break;
                    }
                case 0x83:
                    {
                        //ADD A,E
                        tStates += 4;
                        Add(E);
                        break;
                    }
                case 0x84:
                    {
                        //ADD A,H
                        tStates += 4;
                        Add(H);
                        break;
                    }
                case 0x85:
                    {
                        //ADD A,L
                        tStates += 4;
                        Add(L);
                        break;
                    }
                case 0x86:
                    {
                        //ADD A,(HL)
                        tStates += 7;
                        Add(memory.ReadByte(HL));
                        break;
                    }
                case 0x87:
                    {
                        //ADD A,A
                        tStates = 4;
                        Add(A);
                        break;
                    }
                case 0x88:
                    {
                        //ADC A,B
                        tStates += 4;
                        Adc(B);
                        break;
                    }
                case 0x89:
                    {
                        //ADC A,C
                        tStates += 4;
                        Adc(C);
                        break;
                    }
                case 0x8A:
                    {
                        //ADC A,D
                        tStates += 4;
                        Adc(D);
                        break;
                    }
                case 0x8B:
                    {
                        //ADC A,E
                        tStates += 4;
                        Adc(E);
                        break;
                    }
                case 0x8C:
                    {
                        //ADC A,H
                        tStates += 4;
                        Adc(H);
                        break;
                    }
                case 0x8D:
                    {
                        //ADC A,L
                        tStates += 4;
                        Adc(L);
                        break;
                    }
                case 0x8E:
                    {
                        //ADC A,(HL)
                        tStates += 7;
                        Adc(memory.ReadByte(HL));
                        break;
                    }
                case 0x8F:
                    {
                        //ADC A,A
                        tStates += 4;
                        Adc(A);
                        break;
                    }
                case 0x90:
                    {
                        //SUB B
                        tStates += 4;
                        Sub(B);
                        break;
                    }
                case 0x91:
                    {
                        //SUB C
                        tStates += 4;
                        Sub(C);
                        break;
                    }
                case 0x92:
                    {
                        //SUB D
                        tStates += 4;
                        Sub(D);
                        break;
                    }
                case 0x93:
                    {
                        //SUB E
                        tStates += 4;
                        Sub(E);
                        break;
                    }
                case 0x94:
                    {
                        //SUB H
                        tStates += 4;
                        Sub(H);
                        break;
                    }
                case 0x95:
                    {
                        //SUB L
                        tStates += 4;
                        Sub(L);
                        break;
                    }
                case 0x96:
                    {
                        //SUB (HL)
                        tStates += 7;
                        Sub(memory.ReadByte(HL));
                        break;
                    }
                case 0x97:
                    {
                        //SUB A
                        tStates += 4;
                        Sub(A);
                        break;
                    }
                case 0x98:
                    {
                        //SBC A,B
                        tStates += 4;
                        Sbc(B);
                        break;
                    }
                case 0x99:
                    {
                        //SBC A,C
                        tStates += 4;
                        Sbc(C);
                        break;
                    }
                case 0x9A:
                    {
                        //SBC A,D
                        tStates += 4;
                        Sbc(D);
                        break;
                    }
                case 0x9B:
                    {
                        //SBC A,E
                        tStates += 4;
                        Sbc(E);
                        break;
                    }
                case 0x9C:
                    {
                        //SBC A,H
                        tStates += 4;
                        Sbc(H);
                        break;
                    }
                case 0x9D:
                    {
                        //SBC A,L
                        tStates += 4;
                        Sbc(L);
                        break;
                    }
                case 0x9E:
                    {
                        //SBC A,(HL)
                        tStates += 7;
                        Sbc(memory.ReadByte(HL));
                        break;
                    }
                case 0x9F:
                    {
                        //SBC A,A
                        tStates += 4;
                        Sbc(A);
                        break;
                    }
                case 0xA0:
                    {
                        //AND B
                        tStates += 4;
                        And(B);
                        break;
                    }
                case 0xA1:
                    {
                        //AND C
                        tStates += 4;
                        And(C);
                        break;
                    }
                case 0xA2:
                    {
                        //AND D
                        tStates += 4;
                        And(D);
                        break;
                    }
                case 0xA3:
                    {
                        //AND E
                        tStates += 4;
                        And(E);
                        break;
                    }
                case 0xA4:
                    {
                        //AND H
                        tStates += 4;
                        And(H);
                        break;
                    }
                case 0xA5:
                    {
                        //AND L
                        tStates += 4;
                        And(L);
                        break;
                    }
                case 0xA6:
                    {
                        //AND (HL)
                        tStates += 7;
                        And(memory.ReadByte(HL));
                        break;
                    }
                case 0xA7:
                    {
                        //AND A
                        tStates += 4;
                        And(A);
                        break;
                    }
                case 0xA8:
                    {
                        //XOR B
                        tStates += 4;
                        Xor(B);
                        break;
                    }
                case 0xA9:
                    {
                        //XOR C
                        tStates += 4;
                        Xor(C);
                        break;
                    }
                case 0xAA:
                    {
                        //XOR D
                        tStates += 4;
                        Xor(D);
                        break;
                    }
                case 0xAB:
                    {
                        //XOR E
                        tStates += 4;
                        Xor(E);
                        break;
                    }
                case 0xAC:
                    {
                        //XOR H
                        tStates += 4;
                        Xor(H);
                        break;
                    }
                case 0xAD:
                    {
                        //XOR L
                        tStates += 4;
                        Xor(L);
                        break;
                    }
                case 0xAE:
                    {
                        //XOR (HL)
                        tStates += 7;
                        Xor(memory.ReadByte(HL));
                        break;
                    }
                case 0xAF:
                    {
                        //XOR A
                        tStates += 4;
                        Xor(A);
                        break;
                    }
                case 0xB0:
                    {
                        //OR B
                        tStates += 4;
                        Or(B);
                        break;
                    }
                case 0xB1:
                    {
                        //OR C
                        tStates += 4;
                        Or(C);
                        break;
                    }
                case 0xB2:
                    {
                        //OR D
                        tStates += 4;
                        Or(D);
                        break;
                    }
                case 0xB3:
                    {
                        //OR E
                        tStates += 4;
                        Or(E);
                        break;
                    }
                case 0xB4:
                    {
                        //OR H
                        tStates += 4;
                        Or(H);
                        break;
                    }
                case 0xB5:
                    {
                        //OR L
                        tStates += 4;
                        Or(L);
                        break;
                    }
                case 0xB6:
                    {
                        //OR (HL)
                        tStates += 7;
                        Or(memory.ReadByte(HL));
                        break;
                    }
                case 0xB7:
                    {
                        //OR A
                        tStates += 4;
                        Or(A);
                        break;
                    }
                case 0xB8:
                    {
                        //CP B
                        tStates += 4;
                        Cp(B);
                        break;
                    }
                case 0xB9:
                    {
                        //CP C
                        tStates += 4;
                        Cp(C);
                        break;
                    }
                case 0xBA:
                    {
                        //CP D
                        tStates += 4;
                        Cp(D);
                        break;
                    }
                case 0xBB:
                    {
                        //CP E
                        tStates += 4;
                        Cp(E);
                        break;
                    }
                case 0xBC:
                    {
                        //CP H
                        tStates += 4;
                        Cp(H);
                        break;
                    }
                case 0xBD:
                    {
                        //CP L
                        tStates += 4;
                        Cp(L);
                        break;
                    }
                case 0xBE:
                    {
                        //CP (HL)
                        tStates += 7;
                        Cp(memory.ReadByte(HL));
                        break;
                    }
                case 0xBF:
                    {
                        //CP A
                        tStates += 4;
                        Cp(A);
                        break;
                    }
                case 0xC0:
                    {
                        //RET NZ
                        if ((F & ZF) == 0)
                        {
                            tStates += 11;
                            PC = memory.ReadWord(SP);
                            SP = (SP + 2) & 0xFFFF;
                        }
                        else
                        {
                            tStates += 5;
                        }
                        break;
                    }
                case 0xC1:
                    {
                        //POP BC
                        tStates += 10;
                        BC = memory.ReadWord(SP);
                        SP = (SP + 2) & 0xFFFF;
                        break;
                    }
                case 0xC2:
                    {
                        //JP NZ,nn
                        tStates += 10;
                        if ((F & ZF) == 0)
                        {
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            PC += 2;
                        }
                        break;
                    }
                case 0xC3:
                    {
                        //JP nn
                        tStates += 10;
                        PC = memory.ReadWord(PC);
                        break;
                    }
                case 0xC4:
                    {
                        //CALL NZ,nn
                        if ((F & ZF) == 0)
                        {
                            tStates += 17;
                            SP = (SP - 2) & 0xFFFF;
                            memory.WriteWord(SP, PC + 2);
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            tStates += 10;
                            PC += 2;
                        }
                        break;
                    }
                case 0xC5:
                    {
                        //PUSH BC
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, BC);
                        break;
                    }
                case 0xC6:
                    {
                        //ADD A,n
                        tStates += 7;
                        Add(memory.ReadByte(PC++));
                        break;
                    }
                case 0xC7:
                    {
                        //RST 00H
                        tStates += 11;
                        SP -= 2;
                        memory.WriteWord(SP, PC);
                        PC = 0;
                        break;
                    }
                case 0xC8:
                    {
                        //RET Z
                        if ((F & ZF) != 0)
                        {
                            tStates += 11;
                            PC = memory.ReadWord(SP);
                            SP = (SP+2)&0xFFFF;
                        }
                        else
                        {
                            tStates += 5;
                        }
                        break;
                    }
                case 0xC9:
                    {
                        //RET
                        tStates += 10;
                        PC = memory.ReadWord(SP);
                        SP = (SP + 2) & 0xFFFF;
                        break;
                    }
                case 0xCA:
                    {
                        //JP Z,nn
                        tStates += 10;
                        if ((F & ZF) != 0)
                        {
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            PC += 2;
                        }
                        break;
                    }
                case 0xCB:
                    {
                        //CB prefix
                        ExecuteCB();
                        break;
                    }
                case 0xCC:
                    {
                        //CALL Z,nn
                        if ((F & ZF) != 0)
                        {
                            tStates += 17;
                            SP = (SP - 2) & 0xFFFF;
                            memory.WriteWord(SP, PC + 2);
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            tStates += 10;
                            PC += 2;
                        }
                        break;
                    }
                case 0xCD:
                    {
                        //CALL nn
                        tStates += 17;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC + 2);
                        PC = memory.ReadWord(PC);
                        break;
                    }
                case 0xCE:
                    {
                        //ADC A,n
                        tStates += 7;
                        Adc(memory.ReadByte(PC++));
                        break;
                    }
                case 0xCF:
                    {
                        //RST 08H
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC);
                        PC = 0x08;
                        break;
                    }
                case 0xD0:
                    {
                        //RET NC
                        if ((F & CF) == 0)
                        {
                            tStates += 11;
                            PC = memory.ReadWord(SP);
                            SP = (SP + 2) & 0xFFFF;
                        }
                        else
                        {
                            tStates += 5;
                        }
                        break;
                    }
                case 0xD1:
                    {
                        //POP DE
                        tStates += 10;
                        DE = memory.ReadWord(SP);
                        SP = (SP + 2) & 0xFFFF;
                        break;
                    }
                case 0xD2:
                    {
                        //JP NC,nn
                        tStates += 10;
                        if ((F & CF) == 0)
                        {
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            PC += 2;
                        }
                        break;
                    }
                case 0xD3:
                    {
                        //OUT (n),A
                        tStates += 11;
                        io.WritePort((A << 8) | memory.ReadByte(PC++), A);
                        break;
                    }
                case 0xD4:
                    {
                        //CALL NC,nn
                        if ((F & CF) == 0)
                        {
                            tStates += 17;
                            SP = (SP - 2) & 0xFFFF;
                            memory.WriteWord(SP, PC + 2);
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            tStates += 10;
                            PC += 2;
                        }
                        break;
                    }
                case 0xD5:
                    {
                        //PUSH DE
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, DE);
                        break;
                    }
                case 0xD6:
                    {
                        //SUB n
                        tStates += 7;
                        Sub(memory.ReadByte(PC++));
                        break;
                    }
                case 0xD7:
                    {
                        //RST 10H
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC);
                        PC = 0x10;
                        break;
                    }
                case 0xD8:
                    {
                        //RET C
                        if ((F & CF) != 0)
                        {
                            tStates += 11;
                            PC = memory.ReadWord(SP);
                            SP = (SP + 2) & 0xFFFF;
                        }
                        else
                        {
                            tStates += 5;
                        }
                        break;
                    }
                case 0xD9:
                    {
                        //EXX
                        tStates += 4;
                        EXX();
                        break;
                    }
                case 0xDA:
                    {
                        //JP C,nn
                        tStates += 10;
                        if ((F & CF) != 0)
                        {
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            PC += 2;
                        }
                        break;
                    }
                case 0xDB:
                    {
                        //IN A,(n)
                        tStates += 11;
                        A = io.ReadPort((A << 8) | memory.ReadByte(PC++));
                        break;
                    }
                case 0xDC:
                    {
                        //CALL C,nn
                        if ((F & CF) != 0)
                        {
                            tStates += 17;
                            SP = (SP - 2) & 0xFFFF;
                            memory.WriteWord(SP, PC + 2);
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            tStates += 10;
                            PC += 2;
                        }
                        break;
                    }
                case 0xDD:
                    {
                        //DD prefix
                        ExecuteDD();
                        break;
                    }
                case 0xDE:
                    {
                        //SBC A,n
                        tStates += 7;
                        Sbc(memory.ReadByte(PC++));
                        break;
                    }
                case 0xDF:
                    {
                        //RST 18 H
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC);
                        PC = 0x18;
                        break;
                    }
                case 0xE0:
                    {
                        //RET PO
                        if ((F & PF) == 0)
                        {
                            tStates += 11;
                            PC = memory.ReadWord(SP);
                            SP = (SP + 2) & 0xFFFF;
                        }
                        else
                        {
                            tStates += 5;
                        }
                        break;
                    }
                case 0xE1:
                    {
                        //POP HL
                        tStates += 10;
                        L = memory.ReadByte(SP++);
                        H = memory.ReadByte(SP++);
                        break;
                    }
                case 0xE2:
                    {
                        //JP PO,nn
                        tStates += 10;
                        if ((F & PF) == 0)
                        {
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            PC += 2;
                        }
                        break;
                    }
                case 0xE3:
                    {
                        //EX (SP),HL
                        tStates += 19;
                        int aux = HL;
                        HL = memory.ReadWord(SP);
                        memory.WriteWord(SP, aux);
                        break;
                    }
                case 0xE4:
                    {
                        //CALL PO,nn
                        if ((F & PF) == 0)
                        {
                            tStates += 17;
                            SP = (SP - 2) & 0xFFFF;
                            memory.WriteWord(SP, PC + 2);
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            tStates += 10;
                            PC += 2;
                        }
                        break;
                    }
                case 0xE5:
                    {
                        //PUSH HL
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, HL);
                        break;
                    }
                case 0xE6:
                    {
                        //AND n
                        tStates += 7;
                        And(memory.ReadByte(PC++));
                        break;
                    }
                case 0xE7:
                    {
                        //RST 20H
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC);
                        PC = 0x20;
                        break;
                    }
                case 0xE8:
                    {
                        //RET PE
                        if ((F & PF) != 0)
                        {
                            tStates += 11;
                            PC = memory.ReadWord(SP);
                            SP = (SP + 2) & 0xFFFF;
                        }
                        else
                        {
                            tStates += 5;
                        }
                        break;
                    }
                case 0xE9:
                    {
                        //JP (HL)
                        tStates += 4;
                        PC = HL;
                        break;
                    }
                case 0xEA:
                    {
                        //JP PE,nn
                        tStates += 10;
                        if ((F & PF) != 0)
                        {
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            PC += 2;
                        }
                        break;
                    }
                case 0xEB:
                    {
                        //EX DE,HL
                        tStates += 4;
                        int aux;
                        aux = H;
                        H = D;
                        D = aux;
                        aux = L;
                        L = E;
                        E = aux;
                        break;
                    }
                case 0xEC:
                    {
                        //CALL PE,nn
                        if ((F & PF) != 0)
                        {
                            tStates += 17;
                            SP = (SP - 2) & 0xFFFF;
                            memory.WriteWord(SP, PC + 2);
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            tStates += 10;
                            PC += 2;
                        }
                        break;
                    }
                case 0xED:
                    {
                        //ED prefix
                        ExecuteED();
                        break;
                    }
                case 0xEE:
                    {
                        //XOR n
                        tStates += 7;
                        Xor(memory.ReadByte(PC++));
                        break;
                    }
                case 0xEF:
                    {
                        //RST 28H
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC);
                        PC = 0x28;
                        break;
                    }
                case 0xF0:
                    {
                        //RET P
                        if ((F & SF) == 0)
                        {
                            tStates += 11;
                            PC = memory.ReadWord(SP);
                            SP = (SP + 2) & 0xFFFF;
                        }
                        else
                        {
                            tStates += 5;
                        }
                        break;
                    }
                case 0xF1:
                    {
                        //POP AF
                        tStates += 10;
                        AF = memory.ReadWord(SP);
                        SP = (SP + 2) & 0xFFFF;
                        break;
                    }
                case 0xF2:
                    {
                        //JP P,nn
                        tStates += 10;
                        if ((F & SF) == 0)
                        {
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            PC += 2;
                        }
                        break;
                    }
                case 0xF3:
                    {
                        //DI
                        tStates += 4;
                        IFF1 = false;
                        IFF2 = false;
                        break;
                    }
                case 0xF4:
                    {
                        //CALL P,nn
                        if ((F & SF) == 0)
                        {
                            tStates += 17;
                            SP = (SP - 2) & 0xFFFF;
                            memory.WriteWord(SP, PC + 2);
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            tStates += 10;
                            PC += 2;
                        }
                        break;
                    }
                case 0xF5:
                    {
                        //PUSH AF
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, AF);
                        break;
                    }
                case 0xF6:
                    {
                        //OR n
                        tStates += 7;
                        Or(memory.ReadByte(PC++));
                        break;
                    }
                case 0xF7:
                    {
                        //RST 30H
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC);
                        PC = 0x30;
                        break;
                    }
                case 0xF8:
                    {
                        //RET M
                        if ((F & SF) != 0)
                        {
                            tStates += 11;
                            PC = memory.ReadWord(SP);
                            SP = (SP + 2) & 0xFFFF;
                        }
                        else
                        {
                            tStates += 5;
                        }
                        break;
                    }
                case 0xF9:
                    {
                        //LD SP,HL
                        tStates += 6;
                        SP = HL;
                        break;
                    }
                case 0xFA:
                    {
                        //JP M,nn
                        tStates += 10;
                        if ((F & SF) != 0)
                        {
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            PC += 2;
                        }
                        break;
                    }
                case 0xFB:
                    {
                        //EI
                        tStates += 4;
                        IFF1 = true;
                        IFF2 = true;
                        break;
                    }
                case 0xFC:
                    {
                        //CALL M,nn
                        if ((F & SF) != 0)
                        {
                            tStates += 17;
                            SP = (SP - 2) & 0xFFFF;
                            memory.WriteWord(SP, PC + 2);
                            PC = memory.ReadWord(PC);
                        }
                        else
                        {
                            tStates += 10;
                            PC += 2;
                        }
                        break;
                    }
                case 0xFD:
                    {
                        //FD prefix
                        ExecuteFD();
                        break;
                    }
                case 0xFE:
                    {
                        //CP n
                        tStates += 7;
                        Cp(memory.ReadByte(PC++));
                        break;
                    }
                case 0xFF:
                    {
                        //RST 38H
                        tStates += 11;
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC);
                        PC = 0x38;
                        break;
                    }
                default:
                    {
                        throw new NotImplementedException();
                    }

            }
            #endregion

            PC &= 0xFFFF;
        }

        public void MaskableInterrupt()
        {
            if (IFF1 == true)
            {
                R = ++R & 0x7F;
                FetchCounter++;                

                // Check if processor was halted
                if (isHalted)
                {
                    isHalted = false;
                    PC++;
                }

                // Clear both interrupt flip flops
                IFF1 = IFF2 = false;

                // Interrupt mode 2. Call [Z80.i:databyte]
                if (InterruptMode == 2)
                {
                    tStates += 19;
                    IrqVector = (IrqVector & 0xff) | ((I & 0xFF) << 8);
                    SP = (SP - 2) & 0xFFFF;
                    memory.WriteWord(SP, PC);
                    PC = memory.ReadWord(IrqVector&0xFFFF);
                }
                else
                    /* Interrupt mode 1. RST 38h */
                    if (InterruptMode == 1)
                    {
                        SP = (SP - 2) & 0xFFFF;
                        memory.WriteWord(SP, PC);
                        PC = 0x0038;
                    }
                    else
                    {
                        /* Interrupt mode 0. We check for CALL and JP instructions, */
                        /* if neither of these were found we assume a 1 byte opcode */
                        /* was placed on the databus                                */
                        //LOG(("Z80 #%d IM0 $%04x\n",cpu_getactivecpu() , irq_vector));
                        switch (IrqVector & 0xff0000)
                        {
                            case 0xcd0000:	/* call */
                                SP = (SP - 2) & 0xFFFF;
                                memory.WriteWord(SP, PC);
                                PC = IrqVector & 0xffff;
                                break;
                            case 0xc30000:	/* jump */
                                PC = IrqVector & 0xffff;
                                break;
                            default:		/* rst (or other opcodes?) */
                                SP = (SP - 2) & 0xFFFF;
                                memory.WriteWord(SP, PC);
                                PC = IrqVector | 0x0038;
                                break;
                        }
                    }
            }
        }

        #region CB prefix

        private void ExecuteCB()
        {
            FetchCounter++;
            R = ++R & 0x7F;
            int opcode = memory.ReadByte(PC++);
            switch (opcode)
            {
                case 0x00:
                    //RLC B
                    tStates += 8;
                    B = Rlc(B);
                    break;

                case 0x01:
                    {
                        //RLC C
                        tStates += 8;
                        C = Rlc(C);
                        break;
                    }
                case 0x02:
                    {
                        //RLC D
                        tStates += 8;
                        D = Rlc(D);
                        break;
                    }
                case 0x03:
                    {     
                        //RLC E
                        tStates += 8;
                        E = Rlc(E);
                        break;
                    }
                case 0x04:
                    {     
                        //RLC H
                        tStates += 8;
                        H = Rlc(H);
                        break;
                    }
                case 0x05:
                    {
                        //RLC L
                        tStates += 8;
                        L = Rlc(L);
                        break;
                    }
                case 0x06:
                    {     
                        //RLC (HL)
                        tStates += 15;
                        memory.WriteByte(HL, Rlc(memory.ReadByte(HL)));
                        break;
                    }
                case 0x07:
                    {     
                        //RLC A
                        tStates += 8;
                        A = Rlc(A);
                        break;
                    }
                case 0x08:
                    {     
                        //RRC B
                        tStates += 8;
                        B = Rrc(B);
                        break;
                    }
                case 0x09:
                    {     
                        //RRC C
                        tStates += 8;
                        C = Rrc(C);
                        break;
                    }
                case 0x0A:
                    {     
                        //RRC D
                        tStates += 8;
                        D = Rrc(D);
                        break;
                    }
                case 0x0B:
                    {     
                        //RRC E
                        tStates += 8;
                        E = Rrc(E);
                        break;
                    }
                case 0x0C:
                    {     
                        //RRC H
                        tStates += 8;
                        H = Rrc(H);
                        break;
                    }
                case 0x0D:
                    {     
                        //RRC L
                        tStates += 8;
                        L = Rrc(L);
                        break;
                    }
                case 0x0E:
                    {     
                        //RRC (HL)
                        tStates += 15;
                        memory.WriteByte(HL, Rrc(memory.ReadByte(HL)));
                        break;
                    }
                case 0x0F:
                    {     
                        //RRC A
                        tStates += 8;
                        A = Rrc(A);
                        break;
                    }
                case 0x10:
                    {   
                        //RL B
                        tStates += 8;
                        B = Rl(B);
                        break;
                    }
                case 0x11:
                    {     
                        //RL C
                        tStates += 8;
                        C = Rl(C);
                        break;
                    }
                case 0x12:
                    {     
                        //RL D
                        tStates += 8;
                        D = Rl(D);
                        break;
                    }
                case 0x13:
                    {     
                        //RL E
                        tStates += 8;
                        E = Rl(E);
                        break;
                    }
                case 0x14:
                    {     
                        //RL H
                        tStates += 8;
                        H = Rl(H);
                        break;
                    }
                case 0x15:
                    {     
                        //RL L
                        tStates += 8;
                        L = Rl(L);
                        break;
                    }
                case 0x16:
                    {     
                        //RL (HL)
                        tStates += 15;
                        memory.WriteByte(HL, Rl(memory.ReadByte(HL)));
                        break;
                    }
                case 0x17:
                    {     
                        //RL A
                        tStates += 8;
                        A = Rl(A);
                        break;
                    }
                case 0x18:
                    {     
                        //RR B
                        tStates += 8;
                        B = Rr(B);
                        break;
                    }
                case 0x19:
                    {     
                        //RR C
                        tStates += 8;
                        C = Rr(C);
                        break;
                    }
                case 0x1A:
                    {     
                        //RR D
                        tStates += 8;
                        D = Rr(D);
                        break;
                    }
                case 0x1B:
                    {     
                        //RR E
                        tStates += 8;
                        E = Rr(E);
                        break;
                    }
                case 0x1C:
                    {     
                        //RR H
                        tStates += 8;
                        H = Rr(H);
                        break;
                    }
                case 0x1D:
                    {   
                        //RR L
                        tStates += 8;
                        L = Rr(L);
                        break;
                    }
                case 0x1E:
                    {     
                        //RR (HL)
                        tStates += 15;
                        memory.WriteByte(HL, Rr(memory.ReadByte(HL)));
                        break;
                    }
                case 0x1F:
                    {     
                        //RR A
                        tStates += 8;
                        A = Rr(A);
                        break;
                    }
                case 0x20:
                    {     
                        //SLA B
                        tStates += 8;
                        B = Sla(B);
                        break;
                    }
                case 0x21:
                    {     
                        //SLA C
                        tStates += 8;
                        C = Sla(C);
                        break;
                    }
                case 0x22:
                    {     
                        //SLA D
                        tStates += 8;
                        D = Sla(D);
                        break;
                    }
                case 0x23:
                    {     
                        //SLA E
                        tStates += 8;
                        E = Sla(E);
                        break;
                    }
                case 0x24:
                    {     
                        //SLA H
                        tStates += 8;
                        H = Sla(H);
                        break;
                    }
                case 0x25:
                    {     
                        //SLA L
                        tStates += 8;
                        L = Sla(L);
                        break;
                    }
                case 0x26:
                    {     
                        //SLA (HL)
                        tStates += 15;
                        memory.WriteByte(HL, Sla(memory.ReadByte(HL)));
                        break;
                    }
                case 0x27:
                    {     
                        //SLA A
                        tStates += 8;
                        A = Sla(A);
                        break;
                    }
                case 0x28:
                    {     
                        //SRA B
                        tStates += 8;
                        B=Sra(B);
                        break;
                    }
                case 0x29:
                    {     
                        //SRA C
                        tStates += 8;
                        C = Sra(C);
                        break;
                    }
                case 0x2A:
                    {     
                        //SRA D
                        tStates += 8;
                        D = Sra(D);
                        break;
                    }
                case 0x2B:
                    {     
                        //SRA E
                        tStates += 8;
                        E = Sra(E);
                        break;
                    }
                case 0x2C:
                    {     
                        //SRA H
                        tStates += 8;
                        H = Sra(H);
                        break;
                    }
                case 0x2D:
                    {     
                        //SRA L
                        tStates += 8;
                        L = Sra(L);
                        break;
                    }
                case 0x2E:
                    {     
                        //SRA (HL)
                        tStates += 15;
                        memory.WriteByte(HL, Sra(memory.ReadByte(HL)));
                        break;
                    }
                case 0x2F:
                    {     
                        //SRA A
                        tStates += 8;
                        A = Sra(A);
                        break;
                    }
                case 0x30:
                    {     
                        //SLL B
                        tStates += 8;
                        B = Sll(B);
                        break;
                    }
                case 0x31:
                    {     
                        //SLL C
                        tStates += 8;
                        C = Sll(C);
                        break;
                    }
                case 0x32:
                    {     
                        //SLL D
                        tStates += 8;
                        D = Sll(D);
                        break;
                    }
                case 0x33:
                    {     
                        //SLL E
                        tStates += 8;
                        E = Sll(E);
                        break;
                    }
                case 0x34:
                    {     
                        //SLL H
                        tStates += 8;
                        H = Sll(H);
                        break;
                    }
                case 0x35:
                    {     
                        //SLL L
                        tStates += 8;
                        L = Sll(L);
                        break;
                    }
                case 0x36:
                    {     
                        //SLL (HL)
                        tStates += 15;
                        memory.WriteByte(HL, Sll(memory.ReadByte(HL)));
                        break;
                    }
                case 0x37:
                    {     
                        //SLL A
                        tStates += 8;
                        A = Sll(A);
                        break;
                    }
                case 0x38:
                    {     
                        //SRL B
                        tStates += 8;
                        B = Srl(B);
                        break;
                    }
                case 0x39:
                    {     
                        //SRL C
                        tStates += 8;
                        C = Srl(C);
                        break;
                    }
                case 0x3A:
                    {     
                        //SRL D
                        tStates += 8;
                        D = Srl(D);
                        break;
                    }
                case 0x3B:
                    {     
                        //SRL E
                        tStates += 8;
                        E = Srl(E);
                        break;
                    }
                case 0x3C:
                    {     
                        //SRL H
                        tStates += 8;
                        H = Srl(H);
                        break;
                    }
                case 0x3D:
                    {     
                        //SRL L
                        tStates += 8;
                        L = Srl(L);
                        break;
                    }
                case 0x3E:
                    {     
                        //SRL (HL)
                        tStates += 15;
                        memory.WriteByte(HL, Srl(memory.ReadByte(HL)));
                        break;
                    }
                case 0x3F:
                    {     
                        //SRL A
                        tStates += 8;
                        A = Srl(A);
                        break;
                    }
                case 0x40:
                    {     
                        //BIT 0,B
                        tStates += 8;
                        Bit(0, B);
                        break;
                    }
                case 0x41:
                    {     
                        //BIT 0,C
                        tStates += 8;
                        Bit(0, C);
                        break;
                    }
                case 0x42:
                    {     
                        //BIT 0,D
                        tStates += 8;
                        Bit(0, D);
                        break;
                    }
                case 0x43:
                    {     
                        //BIT 0,E
                        tStates += 8;
                        Bit(0, E);
                        break;
                    }
                case 0x44:
                    {     
                        //BIT 0,H
                        tStates += 8;
                        Bit(0, H);
                        break;
                    }
                case 0x45:
                    {     
                        //BIT 0,L
                        tStates += 8;
                        Bit(0, L);
                        break;
                    }
                case 0x46:
                    {     
                        //BIT 0,(HL)
                        tStates += 12;
                        Bit(0, memory.ReadByte(HL));
                        break;
                    }
                case 0x47:
                    {     
                        //BIT 0,A
                        tStates += 8;
                        Bit(0, A);
                        break;
                    }
                case 0x48:
                    {     
                        //BIT 1,B
                        tStates += 8;
                        Bit(1, B);
                        break;
                    }
                case 0x49:
                    {     
                        //BIT 1,C
                        tStates += 8;
                        Bit(1, C);
                        break;
                    }
                case 0x4A:
                    {     
                        //BIT 1,D
                        tStates += 8;
                        Bit(1, D);
                        break;
                    }
                case 0x4B:
                    {     
                        //BIT 1,E
                        tStates += 8;
                        Bit(1, E);
                        break;
                    }
                case 0x4C:
                    {     
                        //BIT 1,H
                        tStates += 8;
                        Bit(1, H);
                        break;
                    }
                case 0x4D:
                    {
                        //BIT 1,L
                        tStates += 8;
                        Bit(1, L);
                        break;
                    }
                case 0x4E:
                    {     
                        //BIT 1,(HL)
                        tStates += 12;
                        Bit(1, memory.ReadByte(HL));
                        break;
                    }
                case 0x4F:
                    {     
                        //BIT 1,A
                        tStates += 8;
                        Bit(1, A);
                        break;
                    }
                case 0x50:
                    {     
                        //BIT 2,B
                        tStates += 8;
                        Bit(2, B);
                        break;
                    }
                case 0x51:
                    {     
                        //BIT 2,C
                        tStates += 8;
                        Bit(2, C);
                        break;
                    }
                case 0x52:
                    {     
                        //BIT 2,D
                        tStates += 8;
                        Bit(2, D);
                        break;
                    }
                case 0x53:
                    {     
                        //BIT 2,E
                        tStates += 8;
                        Bit(2, E);
                        break;
                    }
                case 0x54:
                    {     
                        //BIT 2,H
                        tStates += 8;
                        Bit(2, H);
                        break;
                    }
                case 0x55:
                    {     
                        //BIT 2,L
                        tStates += 8;
                        Bit(2, L);
                        break;
                    }
                case 0x56:
                    {     
                        //BIT 2,(HL)
                        tStates += 12;
                        Bit(2, memory.ReadByte(HL));
                        break;
                    }
                case 0x57:
                    {     
                        //BIT 2,A
                        tStates += 8;
                        Bit(2, A);
                        break;
                    }
                case 0x58:
                    {     
                        //BIT 3,B
                        tStates += 8;
                        Bit(3, B);
                        break;
                    }
                case 0x59:
                    {     
                        //BIT 3,C
                        tStates += 8;
                        Bit(3, C);
                        break;
                    }
                case 0x5A:
                    {     
                        //BIT 3,D
                        tStates += 8;
                        Bit(3, D);
                        break;
                    }
                case 0x5B:
                    {     
                        //BIT 3,E
                        tStates += 8;
                        Bit(3, E);
                        break;
                    }
                case 0x5C:
                    {     
                        //BIT 3,H
                        tStates += 8;
                        Bit(3, H);
                        break;
                    }
                case 0x5D:
                    {     
                        //BIT 3,L
                        tStates += 8;
                        Bit(3, L);
                        break;
                    }
                case 0x5E:
                    {     
                        //BIT 3,(HL)
                        tStates += 12;
                        Bit(3, memory.ReadByte(HL));
                        break;
                    }
                case 0x5F:
                    {     
                        //BIT 3,A
                        tStates+=8;
                        Bit(3,A);
                        break;
                    }
                case 0x60:
                    {     
                        //BIT 4,B
                        tStates+=8;
                        Bit(4,B);
                        break;
                    }
                case 0x61:
                    {     
                        //BIT 4,C
                        tStates+=8;
                        Bit(4,C);
                        break;
                    }
                case 0x62:
                    {     
                        //BIT 4,D
                        tStates+=8;
                        Bit(4,D);
                        break;
                    }
                case 0x63:
                    {     
                        //BIT 4,E
                        tStates+=8;
                        Bit(4,E);
                        break;
                    }
                case 0x64:
                    {     
                        //BIT 4,H
                        tStates+=8;
                        Bit(4,H);
                        break;
                    }
                case 0x65:
                    {     
                        //BIT 4,L
                        tStates+=8;
                        Bit(4,L);
                        break;
                    }
                case 0x66:
                    {     
                        //BIT 4,(HL)
                        tStates+=12;
                        Bit(4,memory.ReadByte(HL));
                        break;
                    }
                case 0x67:
                    {     
                        //BIT 4,A
                        tStates+=8;
                        Bit(4,A);
                        break;
                    }
                case 0x68:
                    {     
                        //BIT 5,B
                        tStates+=8;
                        Bit(5,B);
                        break;
                    }
                case 0x69:
                    {     
                        //BIT 5,C
                        tStates+=8;
                        Bit(5,C);
                        break;
                    }
                case 0x6A:
                    {     
                        //BIT 5,D
                        tStates+=8;
                        Bit(5,D);
                        break;
                    }
                case 0x6B:
                    {     
                        //BIT 5,E
                        tStates+=8;
                        Bit(5,E);
                        break;
                    }
                case 0x6C:
                    {     
                        //BIT 5,H
                        tStates+=8;
                        Bit(5,H);
                        break;
                    }
                case 0x6D:
                    {     
                        //BIT 5,L
                        tStates+=8;
                        Bit(5,L);
                        break;
                    }
                case 0x6E:
                    {     
                        //BIT 5,(HL)
                        tStates+=12;
                        Bit(5,memory.ReadByte(HL));
                        break;
                    }
                case 0x6F:
                    {     
                        //BIT 5,A
                        tStates+=8;
                        Bit(5,A);
                        break;
                    }
                case 0x70:
                    {     
                        //BIT 6,B
                        tStates+=8;
                        Bit(6,B);
                        break;
                    }
                case 0x71:
                    {     
                        //BIT 6,C
                        tStates+=8;
                        Bit(6,C);
                        break;
                    }
                case 0x72:
                    {     
                        //BIT 6,D
                        tStates+=8;
                        Bit(6,D);
                        break;
                    }
                case 0x73:
                    {     
                        //BIT 6,E
                        tStates+=8;
                        Bit(6,E);
                        break;
                    }
                case 0x74:
                    {     
                        //BIT 6,H
                        tStates+=8;
                        Bit(6,H);
                        break;
                    }
                case 0x75:
                    {     
                        //BIT 6,L
                        tStates+=8;
                        Bit(6,L);
                        break;
                    }
                case 0x76:
                    {     
                        //BIT 6,(HL)
                        tStates+=12;
                        Bit(6, memory.ReadByte(HL));
                        break;
                    }
                case 0x77:
                    {     
                        //BIT 6,A
                        tStates+=8;
                        Bit(6,A);
                        break;
                    }
                case 0x78:
                    {     
                        //BIT 7,B
                        tStates+=8;
                        Bit(7,B);
                        break;
                    }
                case 0x79:
                    {     
                        //BIT 7,C
                        tStates+=8;
                        Bit(7,C);
                        break;
                    }
                case 0x7A:
                    {     
                        //BIT 7,D
                        tStates+=8;
                        Bit(7,D);
                        break;
                    }
                case 0x7B:
                    {     
                        //BIT 7,E
                        tStates+=8;
                        Bit(7,E);
                        break;
                    }
                case 0x7C:
                    {     
                        //BIT 7,H
                        tStates+=8;
                        Bit(7,H);
                        break;
                    }
                case 0x7D:
                    {     
                        //BIT 7,L
                        tStates+=8;
                        Bit(7,L);
                        break;
                    }
                case 0x7E:
                    {   
                        //BIT 7,(HL)
                        tStates+=12;
                        Bit(7, memory.ReadByte(HL));
                        break;
                    }
                case 0x7F:
                    {     
                        //BIT 7,A
                        tStates+=8;
                        Bit(7,A);
                        break;
                    }
                case 0x80:
                    {     
                        //RES 0,B
                        tStates+=8;
                        B &= 0xFE;
                        break;
                    }
                case 0x81:
                    {     
                        //RES 0,C
                        tStates+=8;
                        C &=0xFE;
                        break;
                    }
                case 0x82:
                    {     
                        //RES 0,D
                        tStates += 8;
                        D &= 0xFE;
                        break;
                    }
                case 0x83:
                    {     
                        //RES 0,E
                        tStates += 8;
                        E &= 0xFE;
                        break;
                    }
                case 0x84:
                    {     
                        //RES 0,H
                        tStates += 8;
                        H &= 0xFE;
                        break;
                    }
                case 0x85:
                    {     
                        //RES 0,L
                        tStates += 8;
                        L &= 0xFE;
                        break;
                    }
                case 0x86:
                    {     
                        //RES 0,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0xFE);
                        break;
                    }
                case 0x87:
                    {     
                        //RES 0,A
                        tStates += 8;
                        A &= 0xFE;
                        break;
                    }
                case 0x88:
                    {     
                        //RES 1,B
                        tStates += 8;
                        B &= 0xFD;
                        break;
                    }
                case 0x89:
                    {     
                        //RES 1,C
                        tStates += 8;
                        C &= 0xFD;
                        break;
                    }
                case 0x8A:
                    {     
                        //RES 1,D
                        tStates += 8;
                        D &= 0xFD;
                        break;
                    }
                case 0x8B:
                    {     
                        //RES 1,E
                        tStates += 8;
                        E &= 0xFD;
                        break;
                    }
                case 0x8C:
                    {     
                        //RES 1,H
                        tStates += 8;
                        H &= 0xFD;
                        break;
                    }
                case 0x8D:
                    {     
                        //RES 1,L
                        tStates += 8;
                        L &= 0xFD;
                        break;
                    }
                case 0x8E:
                    {     
                        //RES 1,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0xFD);
                        break;
                    }
                case 0x8F:
                    {     
                        //RES 1,A
                        tStates += 8;
                        A &= 0xFD;
                        break;
                    }
                case 0x90:
                    {     
                        //RES 2,B
                        tStates += 8;
                        B &= 0xFB;
                        break;
                    }
                case 0x91:
                    {     
                        //RES 2,C
                        tStates += 8;
                        C &= 0xFB;
                        break;
                    }
                case 0x92:
                    {     
                        //RES 2,D
                        tStates += 8;
                        D &= 0xFB;
                        break;
                    }
                case 0x93:
                    {     
                        //RES 2,E
                        tStates += 8;
                        E &= 0xFB;
                        break;
                    }
                case 0x94:
                    {   
                        //RES 2,H
                        tStates += 8;
                        H &= 0xFB;
                        break;
                    }
                case 0x95:
                    {     
                        //RES 2,L
                        tStates += 8;
                        L &= 0xFB;
                        break;
                    }
                case 0x96:
                    {     
                        //RES 2,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0xFB);
                        break;
                    }
                case 0x97:
                    {     
                        //RES 2,A
                        tStates += 8;
                        A &= 0xFB;
                        break;
                    }
                case 0x98:
                    {     
                        //RES 3,B
                        tStates += 8;
                        B &= 0xF7;
                        break;
                    }
                case 0x99:
                    {     
                        //RES 3,C
                        tStates += 8;
                        C &= 0xF7;
                        break;
                    }
                case 0x9A:
                    {     
                        //RES 3,D
                        tStates += 8;
                        D &= 0xF7;
                        break;
                    }
                case 0x9B:
                    {     
                        //RES 3,E
                        tStates += 8;
                        E &= 0xF7;
                        break;
                    }
                case 0x9C:
                    {     
                        //RES 3,H
                        tStates += 8;
                        H &= 0xF7;
                        break;
                    }
                case 0x9D:
                    {     
                        //RES 3,L
                        tStates += 8;
                        L &= 0xF7;
                        break;
                    }
                case 0x9E:
                    {     
                        //RES 3,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0xF7);
                        break;
                    }
                case 0x9F:
                    {     
                        //RES 3,A
                        tStates += 8;
                        A &= 0xF7;
                        break;
                    }
                case 0xA0:
                    {     
                        //RES 4,B
                        tStates += 8;
                        B &= 0xEF;
                        break;
                    }
                case 0xA1:
                    {     
                        //RES 4,C
                        tStates += 8;
                        C &= 0xEF;
                        break;
                    }
                case 0xA2:
                    {     
                        //RES 4,D
                        tStates += 8;
                        D &= 0xEF;
                        break;
                    }
                case 0xA3:
                    {     
                        //RES 4,E
                        tStates += 8;
                        E &= 0xEF;
                        break;
                    }
                case 0xA4:
                    {     
                        //RES 4,H
                        tStates += 8;
                        H &= 0xEF;
                        break;
                    }
                case 0xA5:
                    {     
                        //RES 4,L
                        tStates += 8;
                        L &= 0xEF;
                        break;
                    }
                case 0xA6:
                    {     
                        //RES 4,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0xEF);
                        break;
                    }
                case 0xA7:
                    {     
                        //RES 4,A
                        tStates += 8;
                        A &= 0xEF;
                        break;
                    }
                case 0xA8:
                    {     
                        //RES 5,B
                        tStates += 8;
                        B &= 0xDF;
                        break;
                    }
                case 0xA9:
                    {     
                        //RES 5,C
                        tStates += 8;
                        C &= 0xDF;
                        break;
                    }
                case 0xAA:
                    {     
                        //RES 5,D
                        tStates += 8;
                        D &= 0xDF;
                        break;
                    }
                case 0xAB:
                    {     
                        //RES 5,E
                        tStates += 8;
                        E &= 0xDF;
                        break;
                    }
                case 0xAC:
                    {     
                        //RES 5,H
                        tStates += 8;
                        H &= 0xDF;
                        break;
                    }
                case 0xAD:
                    {     
                        //RES 5,L
                        tStates += 8;
                        L &= 0xDF;
                        break;
                    }
                case 0xAE:
                    {     
                        //RES 5,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0xDF);
                        break;
                    }
                case 0xAF:
                    {     
                        //RES 5,A
                        tStates += 8;
                        A &= 0xDF;
                        break;
                    }
                case 0xB0:
                    {     
                        //RES 6,B
                        tStates += 8;
                        B &= 0xBF;
                        break;
                    }
                case 0xB1:
                    {   
                        //RES 6,C
                        tStates += 8;
                        C &= 0xBF;
                        break;
                    }
                case 0xB2:
                    {     
                        //RES 6,D
                        tStates += 8;
                        D &= 0xBF;
                        break;
                    }
                case 0xB3:
                    {     
                        //RES 6,E
                        tStates += 8;
                        E &= 0xBF;
                        break;
                    }
                case 0xB4:
                    {     
                        //RES 6,H
                        tStates += 8;
                        H &= 0xBF;
                        break;
                    }
                case 0xB5:
                    {     
                        //RES 6,L
                        tStates += 8;
                        L &= 0xBF;
                        break;
                    }
                case 0xB6:
                    {     
                        //RES 6,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0xBF);
                        break;
                    }
                case 0xB7:
                    {     
                        //RES 6,A
                        tStates += 8;
                        A &= 0xBF;
                        break;
                    }
                case 0xB8:
                    {     
                        //RES 7,B
                        tStates += 8;
                        B &= 0x7F;
                        break;
                    }
                case 0xB9:
                    {     
                        //RES 7,C
                        tStates += 8;
                        C &= 0x7F;
                        break;
                    }
                case 0xBA:
                    {     
                        //RES 7,D
                        tStates += 8;
                        D &= 0x7F;
                        break;
                    }
                case 0xBB:
                    {     
                        //RES 7,E
                        tStates += 8;
                        E &= 0x7F;
                        break;
                    }
                case 0xBC:
                    {     
                        //RES 7,H
                        tStates += 8;
                        H &= 0x7F;
                        break;
                    }
                case 0xBD:
                    {     
                        //RES 7,L
                        tStates += 8;
                        L &= 0x7F;
                        break;
                    }
                case 0xBE:
                    {     
                        //RES 7,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0x7F);
                        break;
                    }
                case 0xBF:
                    {     
                        //RES 7,A
                        tStates += 8;
                        A &= 0x7F;
                        break;
                    }
                case 0xC0:
                    {     
                        //SET 0,B
                        tStates += 8;
                        B |= 0x01;
                        break;
                    }
                case 0xC1:
                    {     
                        //SET 0,C
                        tStates += 8;
                        C |= 0x01;
                        break;
                    }
                case 0xC2:
                    {     
                        //SET 0,D
                        tStates += 8;
                        D |= 0x01;
                        break;
                    }
                case 0xC3:
                    {     
                        //SET 0,E
                        tStates += 8;
                        E |= 0x01;
                        break;
                    }
                case 0xC4:
                    {     
                        //SET 0,H
                        tStates += 8;
                        H |= 0x01;
                        break;
                    }
                case 0xC5:
                    {     
                        //SET 0,L
                        tStates += 8;
                        L |= 0x01;
                        break;
                    }
                case 0xC6:
                    {     
                        //SET 0,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) | 0x01);
                        break;
                    }
                case 0xC7:
                    {     
                        //SET 0,A
                        tStates += 8;
                        A |= 0x01;
                        break;
                    }
                case 0xC8:
                    {     
                        //SET 1,B
                        tStates += 8;
                        B |= 0x02;
                        break;
                    }
                case 0xC9:
                    {     
                        //SET 1,C
                        tStates += 8;
                        C |= 0x02;
                        break;
                    }
                case 0xCA:
                    {     
                        //SET 1,D
                        tStates += 8;
                        D |= 0x02;
                        break;
                    }
                case 0xCB:
                    {     
                        //SET 1,E
                        tStates += 8;
                        E |= 0x02;
                        break;
                    }
                case 0xCC:
                    {     
                        //SET 1,H
                        tStates += 8;
                        H |= 0x02;
                        break;
                    }
                case 0xCD:
                    {     
                        //SET 1,L
                        tStates += 8;
                        L |= 0x02;
                        break;
                    }
                case 0xCE:
                    {     
                        //SET 1,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) | 0x02);
                        break;
                    }
                case 0xCF:
                    {     
                        //SET 1,A
                        tStates += 8;
                        A |= 0x02;
                        break;
                    }
                case 0xD0:
                    {     
                        //SET 2,B
                        tStates += 8;
                        B |= 0x04;
                        break;
                    }
                case 0xD1:
                    {     
                        //SET 2,C
                        tStates += 8;
                        C |= 0x04;
                        break;
                    }
                case 0xD2:
                    {     
                        //SET 2,D
                        tStates += 8;
                        D |= 0x04;
                        break;
                    }
                case 0xD3:
                    {     
                        //SET 2,E
                        tStates += 8;
                        E |= 0x04;
                        break;
                    }
                case 0xD4:
                    {     
                        //SET 2,H
                        tStates += 8;
                        H |= 0x04;
                        break;
                    }
                case 0xD5:
                    {     
                        //SET 2,L
                        tStates += 8;
                        L |= 0x04;
                        break;
                    }
                case 0xD6:
                    {     
                        //SET 2,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) | 0x04);
                        break;
                    }
                case 0xD7:
                    {     
                        //SET 2,A
                        tStates += 8;
                        A |= 0x04;
                        break;
                    }
                case 0xD8:
                    {     
                        //SET 3,B
                        tStates += 8;
                        B |= 0x08;
                        break;
                    }
                case 0xD9:
                    {    
                        //SET 3,C
                        tStates += 8;
                        C |= 0x08;
                        break;
                    }
                case 0xDA:
                    {     
                        //SET 3,D
                        tStates += 8;
                        D |= 0x08;
                        break;
                    }
                case 0xDB:
                    {     
                        //SET 3,E
                        tStates += 8;
                        E |= 0x08;
                        break;
                    }
                case 0xDC:
                    {     
                        //SET 3,H
                        tStates += 8;
                        H |= 0x08;
                        break;
                    }
                case 0xDD:
                    {     
                        //SET 3,L
                        tStates += 8;
                        L |= 0x08;
                        break;
                    }
                case 0xDE:
                    {     
                        //SET 3,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) | 0x08);
                        break;
                    }
                case 0xDF:
                    {     
                        //SET 3,A
                        tStates += 8;
                        A |= 0x08;
                        break;
                    }
                case 0xE0:
                    {     
                        //SET 4,B
                        tStates += 8;
                        B |= 0x10;
                        break;
                    }
                case 0xE1:
                    {     
                        //SET 4,C
                        tStates += 8;
                        C |= 0x10;
                        break;
                    }
                case 0xE2:
                    {     
                        //SET 4,D
                        tStates += 8;
                        D |= 0x10;
                        break;
                    }
                case 0xE3:
                    {     
                        //SET 4,E
                        tStates += 8;
                        E |= 0x10;
                        break;
                    }
                case 0xE4:
                    {     
                        //SET 4,H
                        tStates += 8;
                        H |= 0x10;
                        break;
                    }
                case 0xE5:
                    {     
                        //SET 4,L
                        tStates += 8;
                        L |= 0x10;
                        break;
                    }
                case 0xE6:
                    {     
                        //SET 4,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) | 0x10);
                        break;
                    }
                case 0xE7:
                    {     
                        //SET 4,A
                        tStates += 8;
                        A |= 0x10;
                        break;
                    }
                case 0xE8:
                    {     
                        //SET 5,B
                        tStates += 8;
                        B |= 0x20;
                        break;
                    }
                case 0xE9:
                    {     
                        //SET 5,C
                        tStates += 8;
                        C |= 0x20;
                        break;
                    }
                case 0xEA:
                    {     
                        //SET 5,D
                        tStates += 8;
                        D |= 0x20;
                        break;
                    }
                case 0xEB:
                    {     
                        //SET 5,E
                        tStates += 8;
                        E |= 0x20;
                        break;
                    }
                case 0xEC:
                    {     
                        //SET 5,H
                        tStates += 8;
                        H |= 0x20;
                        break;
                    }
                case 0xED:
                    {     
                        //SET 5,L
                        tStates += 8;
                        L |= 0x20;
                        break;
                    }
                case 0xEE:
                    {     
                        //SET 5,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) & 0x20);
                        break;
                    }
                case 0xEF:
                    {     
                        //SET 5,A
                        tStates += 8;
                        A |= 0x20;
                        break;
                    }
                case 0xF0:
                    {     
                        //SET 6,B
                        tStates += 8;
                        B |= 0x40;
                        break;
                    }
                case 0xF1:
                    {     
                        //SET 6,C
                        tStates += 8;
                        C |= 0x40;
                        break;
                    }
                case 0xF2:
                    {     
                        //SET 6,D
                        tStates += 8;
                        D |= 0x40;
                        break;
                    }
                case 0xF3:
                    {     
                        //SET 6,E
                        tStates += 8;
                        E |= 0x40;
                        break;
                    }
                case 0xF4:
                    {     
                        //SET 6,H
                        tStates += 8;
                        H |= 0x40;
                        break;
                    }
                case 0xF5:
                    {     
                        //SET 6,L
                        tStates += 8;
                        L |= 0x40;
                        break;
                    }
                case 0xF6:
                    {     
                        //SET 6,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) | 0x40);
                        break;
                    }
                case 0xF7:
                    {     
                        //SET 6,A
                        tStates += 8;
                        A |= 0x40;
                        break;
                    }
                case 0xF8:
                    {     
                        //SET 7,B
                        tStates += 8;
                        B |= 0x80;
                        break;
                    }
                case 0xF9:
                    {     
                        //SET 7,C
                        tStates += 8;
                        C |= 0x80;
                        break;
                    }
                case 0xFA:
                    {     
                        //SET 7,D
                        tStates += 8;
                        D |= 0x80;
                        break;
                    }
                case 0xFB:
                    {     
                        //SET 7,E
                        tStates += 8;
                        E |= 0x80;
                        break;
                    }
                case 0xFC:
                    {     
                        //SET 7,H
                        tStates += 8;
                        H |= 0x80;
                        break;
                    }
                case 0xFD:
                    {     
                        //SET 7,L
                        tStates += 8;
                        L |= 0x80;
                        break;
                    }
                case 0xFE:
                    {
                        //SET 7,(HL)
                        tStates += 15;
                        memory.WriteByte(HL, memory.ReadByte(HL) | 0x80);
                        break;
                    }
                case 0xFF:
                    {     
                        //SET 7,A
                        tStates += 8;
                        A |= 0x80;
                        break;
                    }
                default:
                    {
                        tStates += 8;
                        break;
                    }
            }
        }


        #endregion

        #region DD prefix
        
        //Subconjunto de instrucciones 0xDD
        private void ExecuteDD()
        {
            FetchCounter++;
            R = ++R & 0x7F;
            int opcode = memory.ReadByte(PC++);
            switch (opcode)
            {
                case 0x21:
                    {     
                        //LD IX,nn
                        tStates += 14;
                        IX = memory.ReadWord(PC);
                        PC += 2;
                        break;
                    }
                case 0x22:
                    {     
                        //LD (nn),IX
                        tStates += 20;
                        memory.WriteWord(memory.ReadWord(PC), IX);
                        PC += 2;
                        break;
                    }
                case 0x23:
                    {     
                        //INC IX
                        tStates += 10;
                        IX = ++IX & 0xFFFF;
                        break;
                    }
                case 0x24:
                    {     
                        //INC IXh
                        tStates += 8;
                        IX = ((Inc(IX >> 8) & 0xFF) << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x25:
                    {     
                        //DEC IXh
                        tStates += 8;
                        IX = ((Dec(IX >> 8) & 0xFF) << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x26:
                    {     
                        //LD IXh,n
                        tStates += 11;
                        IX = (memory.ReadByte(PC++) << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x29:
                    {     
                        //ADD IX,IX
                        tStates += 15;
                        IX = Add16(IX, IX);
                        break;
                    }
                case 0x2A:
                    {     
                        //LD IX,(nn)
                        tStates += 20;
                        IX = memory.ReadWord(memory.ReadWord(PC));
                        PC += 2;
                        break;
                    }
                case 0x2B:
                    {     
                        //DEC IX
                        tStates += 10;
                        IX = --IX & 0xFFFF;
                        break;
                    }
                case 0x2C:
                    {     
                        //INC IXl
                        tStates += 8;
                        IX = (IX & 0xFF00) | (Inc(IX & 0xFF));
                        break;
                    }
                case 0x2D:
                    {     
                        //DEC IXl
                        tStates += 8;
                        IX = (IX & 0xFF00) | (Dec(IX & 0xFF));
                        break;
                    }
                case 0x2E:
                    {     
                        //LD IXl,n
                        tStates += 11;
                        IX = (IX & 0xFF00) | (memory.ReadByte(PC++));
                        break;
                    }
                case 0x34:
                    {     
                        //INC (IX+d)
                        tStates += 23;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, Inc(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x35:
                    {     
                        //DEC (IX+d)
                        tStates += 23;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, Dec(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x36:
                    {     
                        //LD (IX+d),n
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, memory.ReadByte(PC++));
                        break;
                    }
                case 0x39:
                    {     
                        //ADD IX,SP
                        tStates += 15;
                        IX = Add16(IX, SP);
                        break;
                    }
                case 0x44:
                    {     
                        //LD B,IXh
                        tStates += 8;
                        B = (IX & 0xFF00) >> 8;
                        break;
                    }
                case 0x45:
                    {     
                        //LD B,IXl
                        tStates += 8;
                        B = IX & 0xFF;
                        break;
                    }
                case 0x46:
                    {     
                        //LD B,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        B = memory.ReadByte(ixd);
                        break;
                    }
                case 0x09:
                case 0x49:
                    {     
                        //ADD IX,BC
                        tStates += 15;
                        IX = Add16(IX, BC);
                        break;
                    }
                case 0x4C:
                    {     
                        //LD C,IXh
                        tStates += 8;
                        C = (IX & 0xFF00) >> 8;
                        break;
                    }
                case 0x4D:
                    {     
                        //LD C,IXl
                        tStates += 8;
                        C = IX & 0xFF;
                        break;
                    }
                case 0x4E:
                    {     
                        //LD C,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        C = memory.ReadByte(ixd);
                        break;
                    }
                case 0x54:
                    {     
                        //LD D,IXh
                        tStates += 8;
                        D = (IX & 0xFF00) >> 8;
                        break;
                    }
                case 0x55:
                    {     
                        //LD D,IXl
                        tStates += 8;
                        D = (IX & 0xFF);
                        break;
                    }
                case 0x56:
                    {     
                        //LD D,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        D = memory.ReadByte(ixd);
                        break;
                    }
                case 0x19:
                case 0x59:
                    {     
                        //ADD IX,DE
                        tStates += 15;
                        IX = Add16(IX, DE);
                        break;
                    }
                case 0x5C:
                    {     
                        //LD E,IXh
                        tStates += 8;
                        E = (IX & 0xFF00) >> 8;
                        break;
                    }
                case 0x5D:
                    {     
                        //LD E,IXl
                        tStates += 8;
                        E = IX & 0xFF;
                        break;
                    }
                case 0x5E:
                    {     
                        //LD E,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        E = memory.ReadByte(ixd);
                        break;
                    }
                case 0x60:
                    {     
                        //LD IXh,B
                        tStates += 8;
                        IX = (B << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x61:
                    {     
                        //LD IXh,C
                        tStates += 8;
                        IX = (C << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x62:
                    {     
                        //LD IXh,D
                        tStates += 8;
                        IX = (D << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x63:
                    {     
                        //LD IXh,E
                        tStates += 8;
                        IX = (E << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x64:
                    {     
                        //LD IXh,IXh
                        tStates += 8;
                        break;
                    }
                case 0x65:
                    {     
                        //LD IXh,IXl
                        tStates += 8;
                        IX = ((IX & 0xFF) << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x66:
                    {     
                        //LD H,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        H = memory.ReadByte(ixd);
                        break;
                    }
                case 0x67:
                    {     
                        //LD IXh,A
                        tStates += 8;
                        IX = (A << 8) | (IX & 0xFF);
                        break;
                    }
                case 0x68:
                    {     
                        //LD IXl,B
                        tStates += 8;
                        IX = (IX & 0xFF00) | (B & 0xFF);
                        break;
                    }
                case 0x69:
                    {     
                        //LD IXl,C
                        tStates += 8;
                        IX = (IX & 0xFF00) | (C & 0xFF);
                        break;
                    }
                case 0x6A:
                    {     
                        //LD IXl,D
                        tStates += 8;
                        IX = (IX & 0xFF00) | (D & 0xFF);
                        break;
                    }
                case 0x6B:
                    {     
                        //LD IXl,E
                        tStates += 8;
                        IX = (IX & 0xFF00) | (E & 0xFF);
                        break;
                    }
                case 0x6C:
                    {     
                        //LD IXl,IXh
                        tStates += 8;
                        IX = (IX & 0xFF00) | (IX >> 8);
                        break;
                    }
                case 0x6D:
                    {     
                        //LD IXl,IXl
                        tStates += 8;
                        break;
                    }
                case 0x6E:
                    {     
                        //LD L,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        L = memory.ReadByte(ixd);
                        break;
                    }
                case 0x6F:
                    {     
                        //LD IXl,A
                        tStates += 8;
                        IX = (IX & 0xFF00) | (A & 0xFF);
                        break;
                    }
                case 0x70:
                    {     
                        //LD (IX+d),B
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x71:
                    {     
                        //LD (IX+d),C
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x72:
                    {
                        //LD (IX+d),D
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x73:
                    {     
                        //LD (IX+d),E
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x74:
                    {     
                        //LD (IX+d),H
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x75:
                    {     
                        //LD (IX+d),L
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x77:
                    {     
                        //LD (IX+d),A
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x79:
                    {
                        //ADD IX,SP
                        tStates += 15;
                        IX = Add16(IX, SP);
                        break;
                    }
                case 0x7C:
                    {     
                        //LD A,IXh
                        tStates += 8;
                        A = (IX & 0xFF00) >> 8;
                        break;
                    }
                case 0x7D:
                    {     
                        //LD A,IXl
                        tStates += 8;
                        A = IX & 0xFF;
                        break;
                    }
                case 0x7E:
                    {     
                        //LD A,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        A = memory.ReadByte(ixd);
                        break;
                    }
                case 0x84:
                    {     
                        //ADD A,IXh
                        tStates += 8;
                        Add((IX & 0xFF00) >> 8);
                        break;
                    }
                case 0x85:
                    {     
                        //ADD A,IXl
                        tStates += 8;
                        Add(IX & 0xFF);
                        break;
                    }
                case 0x86:
                    {     
                        //ADD A,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        Add(memory.ReadByte(ixd));
                        break;
                    }
                case 0x8C:
                    {     
                        //ADC A,IXh
                        tStates += 8;
                        Adc((IX & 0xFF00) >> 8);
                        break;
                    }
                case 0x8D:
                    {     
                        //ADC A,IXl
                        tStates += 8;
                        Adc(IX & 0xFF);
                        break;
                    }
                case 0x8E:
                    {     
                        //ADC A,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        Adc(memory.ReadByte(ixd));
                        break;
                    }
                case 0x94:
                    {     
                        //SUB IXh
                        tStates += 8;
                        Sub((IX & 0xFF00) >> 8);
                        break;
                    }
                case 0x95:
                    {     
                        //SUB IXl
                        tStates += 8;
                        Sub(IX & 0xFF);
                        break;
                    }
                case 0x96:
                    {     
                        //SUB (IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        Sub(memory.ReadByte(ixd));
                        break;
                    }
                case 0x9C:
                    {     
                        //SBC A,IXh
                        tStates += 8;
                        Sbc((IX & 0xFF00) >> 8);
                        break;
                    }
                case 0x9D:
                    {     
                        //SBC A,IXl
                        tStates += 8;
                        Sbc(IX & 0xFF);
                        break;
                    }
                case 0x9E:
                    {     
                        //SBC A,(IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        Sbc(memory.ReadByte(ixd));
                        break;
                    }
                case 0xA4:
                    {     
                        //AND IXh
                        tStates += 8;
                        And((IX & 0xFF00) >> 8);
                        break;
                    }
                case 0xA5:
                    {     
                        //AND IXl
                        tStates += 8;
                        And(IX & 0xFF);
                        break;
                    }
                case 0xA6:
                    {     
                        //AND (IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        And(memory.ReadByte(ixd));
                        break;
                    }
                case 0xAC:
                    {     
                        //XOR IXh
                        tStates += 8;
                        Xor((IX & 0xFF00) >> 8);
                        break;
                    }
                case 0xAD:
                    {     
                        //XOR IXl
                        tStates += 8;
                        Xor(IX & 0xFF);
                        break;
                    }
                case 0xAE:
                    {     
                        //XOR (IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        Xor(memory.ReadByte(ixd));
                        break;
                    }
                case 0xB4:
                    {     
                        //OR IXh
                        tStates += 8;
                        Or((IX & 0xFF00) >> 8);
                        break;
                    }
                case 0xB5:
                    {     
                        //OR IXl
                        tStates += 8;
                        Or(IX & 0xFF);
                        break;
                    }
                case 0xB6:
                    {     
                        //OR (IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        Or(memory.ReadByte(ixd));
                        break;
                    }
                case 0xBC:
                    {     
                        //CP IXh
                        tStates += 8;
                        Cp((IX & 0xFF00) >> 8);
                        break;
                    }
                case 0xBD:
                    {     
                        //CP IXl
                        tStates += 8;
                        Cp(IX & 0xFF);
                        break;
                    }
                case 0xBE:
                    {     
                        //CP (IX+d)
                        tStates += 19;
                        int ixd = IX + (sbyte)memory.ReadByte(PC++);
                        Cp(memory.ReadByte(ixd));
                        break;
                    }
                case 0xCB:
                    {     
                        //DDCB Prefix
                        ExecuteDDCB();
                        break;
                    }
                case 0xE1:
                    {     
                        //POP IX
                        tStates += 14;
                        IX = memory.ReadByte(SP++) & 0xFF;
                        IX |= (memory.ReadByte(SP++) << 8);
                        break;
                    }
                case 0xE3:
                    {     
                        //EX (SP),IX
                        tStates += 23;
                        int aux = IX;
                        IX = memory.ReadWord(SP);
                        memory.WriteWord(SP, aux);
                        break;
                    }
                case 0xE5:
                    {     
                        //PUSH IX
                        tStates += 15;
                        SP -= 2;
                        memory.WriteWord(SP, IX);
                        break;
                    }
                case 0xE9:
                    {     
                        //JP (IX)
                        tStates += 8;
                        PC = IX;
                        break;
                    }
                case 0xF9:
                    {     
                        //LD SP,IX
                        tStates += 10;
                        SP = IX;
                        break;
                    }
                default:
                    {
                        tStates += 8;
                        break;
                    }
            }
        }

        #endregion

        #region FD prefix

        //Subconjunto de instrucciones 0xDD
        private void ExecuteFD()
        {
            FetchCounter++;
            R = ++R & 0x7F;
            int opcode = memory.ReadByte(PC++);
            switch (opcode)
            {
                case 0x21:
                    {
                        //LD IY,nn
                        tStates += 14;
                        IY = memory.ReadWord(PC);
                        PC += 2;
                        break;
                    }
                case 0x22:
                    {
                        //LD (nn),IY
                        tStates += 20;
                        memory.WriteWord(memory.ReadWord(PC), IY);
                        PC += 2;
                        break;
                    }
                case 0x23:
                    {
                        //INC IY
                        tStates += 10;
                        IY = ++IY & 0xFFFF;
                        break;
                    }
                case 0x24:
                    {
                        //INC IYh
                        tStates += 8;
                        IY = ((Inc(IY >> 8) & 0xFF) << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x25:
                    {
                        //DEC IYh
                        tStates += 8;
                        IY = ((Dec(IY >> 8) & 0xFF) << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x26:
                    {
                        //LD IYh,n
                        tStates += 11;
                        IY = (memory.ReadByte(PC++) << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x29:
                    {
                        //ADD IY,IY
                        tStates += 15;
                        IY = Add16(IY, IY);
                        break;
                    }
                case 0x2A:
                    {
                        //LD IY,(nn)
                        tStates += 20;
                        IY = memory.ReadWord(memory.ReadWord(PC));
                        PC += 2;
                        break;
                    }
                case 0x2B:
                    {
                        //DEC IY
                        tStates += 10;
                        IY = --IY & 0xFFFF;
                        break;
                    }
                case 0x2C:
                    {
                        //INC IYl
                        tStates += 8;
                        IY = (IY & 0xFF00) | (Inc(IY & 0xFF));
                        break;
                    }
                case 0x2D:
                    {
                        //DEC IYl
                        tStates += 8;
                        IY = (IY & 0xFF00) | (Dec(IY & 0xFF));
                        break;
                    }
                case 0x2E:
                    {
                        //LD IYl,n
                        tStates += 11;
                        IY = (IY & 0xFF00) | (memory.ReadByte(PC++));
                        break;
                    }
                case 0x34:
                    {
                        //INC (IY+d)
                        tStates += 23;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, Inc(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x35:
                    {
                        //DEC (IY+d)
                        tStates += 23;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, Dec(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x36:
                    {
                        //LD (IY+d),n
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, memory.ReadByte(PC++));
                        break;
                    }
                case 0x39:
                    {
                        //ADD IY,SP
                        tStates += 15;
                        IY = Add16(IY, SP);
                        break;
                    }
                case 0x44:
                    {
                        //LD B,IYh
                        tStates += 8;
                        B = (IY & 0xFF00) >> 8;
                        break;
                    }
                case 0x45:
                    {
                        //LD B,IYl
                        tStates += 8;
                        B = IY & 0xFF;
                        break;
                    }
                case 0x46:
                    {
                        //LD B,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        B = memory.ReadByte(iyd);
                        break;
                    }
                case 0x09:
                case 0x49:
                    {
                        //ADD IY,BC
                        tStates += 15;
                        IY = Add16(IY, BC);
                        break;
                    }
                case 0x4C:
                    {
                        //LD C,IYh
                        tStates += 8;
                        C = (IY & 0xFF00) >> 8;
                        break;
                    }
                case 0x4D:
                    {
                        //LD C,IYl
                        tStates += 8;
                        C = IY & 0xFF;
                        break;
                    }
                case 0x4E:
                    {
                        //LD C,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        C = memory.ReadByte(iyd);
                        break;
                    }
                case 0x54:
                    {
                        //LD D,IYh
                        tStates += 8;
                        D = (IY & 0xFF00) >> 8;
                        break;
                    }
                case 0x55:
                    {
                        //LD D,IYl
                        tStates += 8;
                        D = (IY & 0xFF);
                        break;
                    }
                case 0x56:
                    {
                        //LD D,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        D = memory.ReadByte(iyd);
                        break;
                    }
                case 0x19:
                case 0x59:
                    {
                        //ADD IY,DE
                        tStates += 15;
                        IY = Add16(IY, DE);
                        break;
                    }
                case 0x5C:
                    {
                        //LD E,IYh
                        tStates += 8;
                        E = (IY & 0xFF00) >> 8;
                        break;
                    }
                case 0x5D:
                    {
                        //LD E,IYl
                        tStates += 8;
                        E = IY & 0xFF;
                        break;
                    }
                case 0x5E:
                    {
                        //LD E,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        E = memory.ReadByte(iyd);
                        break;
                    }
                case 0x60:
                    {
                        //LD IYh,B
                        tStates += 8;
                        IY = (B << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x61:
                    {
                        //LD IYh,C
                        tStates += 8;
                        IY = (C << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x62:
                    {
                        //LD IYh,D
                        tStates += 8;
                        IY = (D << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x63:
                    {
                        //LD IYh,E
                        tStates += 8;
                        IY = (E << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x64:
                    {
                        //LD IYh,IYh
                        tStates += 8;
                        break;
                    }
                case 0x65:
                    {
                        //LD IYh,IYl
                        tStates += 8;
                        IY = ((IY & 0xFF) << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x66:
                    {
                        //LD H,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        H = memory.ReadByte(iyd);
                        break;
                    }
                case 0x67:
                    {
                        //LD IYh,A
                        tStates += 8;
                        IY = (A << 8) | (IY & 0xFF);
                        break;
                    }
                case 0x68:
                    {
                        //LD IYl,B
                        tStates += 8;
                        IY = (IY & 0xFF00) | (B & 0xFF);
                        break;
                    }
                case 0x69:
                    {
                        //LD IYl,C
                        tStates += 8;
                        IY = (IY & 0xFF00) | (C & 0xFF);
                        break;
                    }
                case 0x6A:
                    {
                        //LD IYl,D
                        tStates += 8;
                        IY = (IY & 0xFF00) | (D & 0xFF);
                        break;
                    }
                case 0x6B:
                    {
                        //LD IYl,E
                        tStates += 8;
                        IY = (IY & 0xFF00) | (E & 0xFF);
                        break;
                    }
                case 0x6C:
                    {
                        //LD IYl,IYh
                        tStates += 8;
                        IY = (IY & 0xFF00) | (IY >> 8);
                        break;
                    }
                case 0x6D:
                    {
                        //LD IYl,IYl
                        tStates += 8;
                        break;
                    }
                case 0x6E:
                    {
                        //LD L,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        L = memory.ReadByte(iyd);
                        break;
                    }
                case 0x6F:
                    {
                        //LD IYl,A
                        tStates += 8;
                        IY = (IY & 0xFF00) | (A & 0xFF);
                        break;
                    }
                case 0x70:
                    {
                        //LD (IY+d),B
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x71:
                    {
                        //LD (IY+d),C
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x72:
                    {
                        //LD (IY+d),D
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x73:
                    {
                        //LD (IY+d),E
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x74:
                    {
                        //LD (IY+d),H
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x75:
                    {
                        //LD (IY+d),L
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x77:
                    {
                        //LD (IY+d),A
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x79:
                    {
                        //ADD IY,SP
                        tStates += 15;
                        IY = Add16(IY, SP);
                        break;
                    }
                case 0x7C:
                    {
                        //LD A,IYh
                        tStates += 8;
                        A = (IY & 0xFF00) >> 8;
                        break;
                    }
                case 0x7D:
                    {
                        //LD A,IYl
                        tStates += 8;
                        A = IY & 0xFF;
                        break;
                    }
                case 0x7E:
                    {
                        //LD A,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        A = memory.ReadByte(iyd);
                        break;
                    }
                case 0x84:
                    {
                        //ADD A,IYh
                        tStates += 8;
                        Add((IY & 0xFF00) >> 8);
                        break;
                    }
                case 0x85:
                    {
                        //ADD A,IYl
                        tStates += 8;
                        Add(IY & 0xFF);
                        break;
                    }
                case 0x86:
                    {
                        //ADD A,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        Add(memory.ReadByte(iyd));
                        break;
                    }
                case 0x8C:
                    {
                        //ADC A,IYh
                        tStates += 8;
                        Adc((IY & 0xFF00) >> 8);
                        break;
                    }
                case 0x8D:
                    {
                        //ADC A,IYl
                        tStates += 8;
                        Adc(IY & 0xFF);
                        break;
                    }
                case 0x8E:
                    {
                        //ADC A,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        Adc(memory.ReadByte(iyd));
                        break;
                    }
                case 0x94:
                    {
                        //SUB IYh
                        tStates += 8;
                        Sub((IY & 0xFF00) >> 8);
                        break;
                    }
                case 0x95:
                    {
                        //SUB IYl
                        tStates += 8;
                        Sub(IY & 0xFF);
                        break;
                    }
                case 0x96:
                    {
                        //SUB (IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        Sub(memory.ReadByte(iyd));
                        break;
                    }
                case 0x9C:
                    {
                        //SBC A,IYh
                        tStates += 8;
                        Sbc((IY & 0xFF00) >> 8);
                        break;
                    }
                case 0x9D:
                    {
                        //SBC A,IYl
                        tStates += 8;
                        Sbc(IY & 0xFF);
                        break;
                    }
                case 0x9E:
                    {
                        //SBC A,(IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        Sbc(memory.ReadByte(iyd));
                        break;
                    }
                case 0xA4:
                    {
                        //AND IYh
                        tStates += 8;
                        And((IY & 0xFF00) >> 8);
                        break;
                    }
                case 0xA5:
                    {
                        //AND IYl
                        tStates += 8;
                        And(IY & 0xFF);
                        break;
                    }
                case 0xA6:
                    {
                        //AND (IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        And(memory.ReadByte(iyd));
                        break;
                    }
                case 0xAC:
                    {
                        //XOR IYh
                        tStates += 8;
                        Xor((IY & 0xFF00) >> 8);
                        break;
                    }
                case 0xAD:
                    {
                        //XOR IYl
                        tStates += 8;
                        Xor(IY & 0xFF);
                        break;
                    }
                case 0xAE:
                    {
                        //XOR (IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        Xor(memory.ReadByte(iyd));
                        break;
                    }
                case 0xB4:
                    {
                        //OR IYh
                        tStates += 8;
                        Or((IY & 0xFF00) >> 8);
                        break;
                    }
                case 0xB5:
                    {
                        //OR IYl
                        tStates += 8;
                        Or(IY & 0xFF);
                        break;
                    }
                case 0xB6:
                    {
                        //OR (IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        Or(memory.ReadByte(iyd));
                        break;
                    }
                case 0xBC:
                    {
                        //CP IYh
                        tStates += 8;
                        Cp((IY & 0xFF00) >> 8);
                        break;
                    }
                case 0xBD:
                    {
                        //CP IYl
                        tStates += 8;
                        Cp(IY & 0xFF);
                        break;
                    }
                case 0xBE:
                    {
                        //CP (IY+d)
                        tStates += 19;
                        int iyd = IY + (sbyte)memory.ReadByte(PC++);
                        Cp(memory.ReadByte(iyd));
                        break;
                    }
                case 0xCB:
                    {
                        //FDCB Prefx
                        ExecuteFDCB();
                        break;
                    }
                case 0xE1:
                    {
                        //POP IY
                        tStates += 14;
                        IY = memory.ReadByte(SP++) & 0xFF;
                        IY |= (memory.ReadByte(SP++) << 8);
                        break;
                    }
                case 0xE3:
                    {
                        //EX (SP),IY
                        tStates += 23;
                        int aux = IY;
                        IY = memory.ReadWord(SP);
                        memory.WriteWord(SP, aux);
                        break;
                    }
                case 0xE5:
                    {
                        //PUSH IY
                        tStates += 15;
                        SP -= 2;
                        memory.WriteWord(SP, IY);
                        break;
                    }
                case 0xE9:
                    {
                        //JP (IY)
                        tStates += 8;
                        PC = IY;
                        break;
                    }
                case 0xF9:
                    {
                        //LD SP,IY
                        tStates += 10;
                        SP = IY;
                        break;
                    }
                default:
                    {
                        tStates += 8;
                        break;
                    }
            }
        }

        #endregion

        #region DDCB prefix

        private void ExecuteDDCB()
        {
            int ixd = IX + (sbyte)memory.ReadByte(PC++);
            int opcode = memory.ReadByte(PC++);
            switch (opcode)
            {
                case 0x00:
                    {     
                        //RLC (IX+d),B
                        tStates += 23;
                        B = Rlc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x01:
                    {     
                        //RLC (IX+d),C
                        tStates += 23;
                        C = Rlc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x02:
                    {     
                        //RLC (IX+d),D
                        tStates += 23;
                        D = Rlc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x03:
                    {     
                        //RLC (IX+d),E
                        tStates += 23;
                        E = Rlc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x04:
                    {     
                        //RLC (IX+d),H
                        tStates += 23;
                        H = Rlc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x05:
                    {     
                        //RLC (IX+d),L
                        tStates += 23;
                        L = Rlc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x06:
                    {     
                        //RLC (IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, Rlc(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x07:
                    {     
                        //RLC (IX+d),A
                        tStates += 23;
                        A = Rlc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x08:
                    {     
                        //RRC (IX+d),B
                        tStates += 23;
                        B = Rrc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x09:
                    {     
                        //RRC (IX+d),C
                        tStates += 23;
                        C = Rrc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x0A:
                    {     
                        //RRC (IX+d),D
                        tStates += 23;
                        D = Rrc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x0B:
                    {     
                        //RRC (IX+d),E
                        tStates += 23;
                        E = Rrc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x0C:
                    {     
                        //RRC (IX+d),H
                        tStates += 23;
                        H = Rrc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x0D:
                    {     
                        //RRC (IX+d),L
                        tStates += 23;
                        L = Rrc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x0E:
                    {     
                        //RRC (IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, Rrc(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x0F:
                    {     
                        //RRC (IX+d),A
                        tStates += 23;
                        A = Rrc(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x10:
                    {     
                        //RL (IX+d),B
                        tStates += 23;
                        B = Rl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x11:
                    {     
                        //RL (IX+d),C
                        tStates += 23;
                        C = Rl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x12:
                    {
                        //RL (IX+d),D
                        tStates += 23;
                        D = Rl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x13:
                    {     
                        //RL (IX+d),E
                        tStates += 23;
                        E = Rl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x14:
                    {
                        //RL (IX+d),H
                        tStates += 23;
                        H = Rl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x15:
                    {     
                        //RL (IX+d),L
                        tStates += 23;
                        L = Rl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x16:
                    {     
                        //RL (IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, Rl(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x17:
                    {     
                        //RL (IX+d),A
                        tStates += 23;
                        A = Rl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x18:
                    {     
                        //RR (IX+d),B
                        tStates += 23;
                        B = Rr(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x19:
                    {     
                        //RR (IX+d),C
                        tStates += 23;
                        C = Rr(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x1A:
                    {     
                        //RR (IX+d),D
                        tStates += 23;
                        D = Rr(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x1B:
                    {     
                        //RR (IX+d),E
                        tStates += 23;
                        E = Rr(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x1C:
                    {     
                        //RR (IX+d),H
                        tStates += 23;
                        H = Rr(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x1D:
                    {     
                        //RR (IX+d),L
                        tStates += 23;
                        L = Rr(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x1E:
                    {     
                        //RR (IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, Rr(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x1F:
                    {     
                        //RR (IX+d),A
                        tStates += 23;
                        A = Rr(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x20:
                    {     
                        //SLA (IX+d),B
                        tStates += 23;
                        B = Sla(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x21:
                    {     
                        //SLA (IX+d),C
                        tStates += 23;
                        C = Sla(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x22:
                    {     
                        //SLA (IX+d),D
                        tStates += 23;
                        D = Sla(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x23:
                    {     
                        //SLA (IX+d),E
                        tStates += 23;
                        E = Sla(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x24:
                    {     
                        //SLA (IX+d),H
                        tStates += 23;
                        H = Sla(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x25:
                    {     
                        //SLA (IX+d),L
                        tStates += 23;
                        L = Sla(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x26:
                    {     
                        //SLA (IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, Sla(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x27:
                    {     
                        //SLA (IX+d),A
                        tStates += 23;
                        A = Sla(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x28:
                    {     
                        //SRA (IX+d),B
                        tStates += 23;
                        B = Sra(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x29:
                    {
                        //SRA (IX+d),C
                        tStates += 23;
                        C = Sra(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x2A:
                    {     
                        //SRA (IX+d),D
                        tStates += 23;
                        D = Sra(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x2B:
                    {     
                        //SRA (IX+d),E
                        tStates += 23;
                        E = Sra(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x2C:
                    {     
                        //SRA (IX+d),H
                        tStates += 23;
                        H = Sra(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x2D:
                    {     
                        //SRA (IX+d),L
                        tStates += 23;
                        L = Sra(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x2E:
                    {     
                        //SRA (IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, Sra(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x2F:
                    {     
                        //SRA (IX+d),A
                        tStates += 23;
                        A = Sra(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x30:
                    {     
                        //SLL (IX+d),B
                        tStates += 23;
                        B = Sll(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x31:
                    {     
                        //SLL (IX+d),C
                        tStates += 23;
                        C = Sll(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x32:
                    {     
                        //SLL (IX+d),D
                        tStates += 23;
                        D = Sll(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x33:
                    {     
                        //SLL (IX+d),E
                        tStates += 23;
                        E = Sll(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x34:
                    {     
                        //SLL (IX+d),H
                        tStates += 23;
                        H = Sll(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x35:
                    {     
                        //SLL (IX+d),L
                        tStates += 23;
                        L = Sll(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x36:
                    {     
                        //SLL (IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, Sll(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x37:
                    {     
                        //SLL (IX+d),A
                        tStates += 23;
                        A = Sll(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x38:
                    {     
                        //SRL (IX+d),B
                        tStates += 23;
                        B = Srl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x39:
                    {
                        //SRL (IX+d),C
                        tStates += 23;
                        C = Srl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x3A:
                    {     
                        //SRL (IX+d),D
                        tStates += 23;
                        D = Srl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x3B:
                    {     
                        //SRL (IX+d),E
                        tStates += 23;
                        E = Srl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x3C:
                    {     
                        //SRL (IX+d),H
                        tStates += 23;
                        H = Srl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x3D:
                    {     
                        //SRL (IX+d),L
                        tStates += 23;
                        L = Srl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x3E:
                    {     
                        //SRL (IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, Srl(memory.ReadByte(ixd)));
                        break;
                    }
                case 0x3F:
                    {     
                        //SRL (IX+d),A
                        tStates += 23;
                        A = Srl(memory.ReadByte(ixd));
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x40:
                case 0x41:
                case 0x42:
                case 0x43:
                case 0x44:
                case 0x45:
                case 0x46:
                case 0x47:
                    {     
                        //BIT 0,(IX+d)
                        tStates += 20;
                        Bit(0, memory.ReadByte(ixd));
                        break;
                    }
                case 0x48:
                case 0x49:
                case 0x4A:
                case 0x4B:
                case 0x4C:
                case 0x4D:
                case 0x4E:
                case 0x4F:
                    {     
                        //BIT 1,(IX+d)
                        tStates += 20;
                        Bit(1, memory.ReadByte(ixd));
                        break;
                    }
                case 0x50:
                case 0x51:
                case 0x52:
                case 0x53:
                case 0x54:
                case 0x55:
                case 0x56:
                case 0x57:
                    {     
                        //BIT 2,(IX+d)
                        tStates += 20;
                        Bit(2, memory.ReadByte(ixd));
                        break;
                    }
                case 0x58:
                case 0x59:
                case 0x5A:
                case 0x5B:
                case 0x5C:
                case 0x5D:
                case 0x5E:
                case 0x5F:
                    {     
                        //BIT 3,(IX+d)
                        tStates += 20;
                        Bit(3, memory.ReadByte(ixd));
                        break;
                    }
                case 0x60:
                case 0x61:
                case 0x62:
                case 0x63:
                case 0x64:
                case 0x65:
                case 0x66:
                case 0x67:
                    {     
                        //BIT 4,(IX+d)
                        tStates += 20;
                        Bit(4, memory.ReadByte(ixd));
                        break;
                    }
                case 0x68:
                case 0x69:
                case 0x6A:
                case 0x6B:
                case 0x6C:
                case 0x6D:
                case 0x6E:
                case 0x6F:
                    {     
                        //BIT 5,(IX+d)
                        tStates += 20;
                        Bit(5, memory.ReadByte(ixd));
                        break;
                    }
                case 0x70:
                case 0x71:
                case 0x72:
                case 0x73:
                case 0x74:
                case 0x75:
                case 0x76:
                case 0x77:
                    {     
                        //BIT 6,(IX+d)
                        tStates += 20;
                        Bit(6, memory.ReadByte(ixd));
                        break;
                    }
                case 0x78:
                case 0x79:
                case 0x7A:
                case 0x7B:
                case 0x7C:
                case 0x7D:
                case 0x7E:
                case 0x7F:
                    {     
                        //BIT 7,(IX+d)
                        tStates += 20;
                        Bit(7, memory.ReadByte(ixd));
                        break;
                    }
                case 0x80:
                    {     
                        //RES 0,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) & 0xFE;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x81:
                    {     
                        //RES 0,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) & 0xFE;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x82:
                    {     
                        //RES 0,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) & 0xFE;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x83:
                    {     
                        //RES 0,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) & 0xFE;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x84:
                    {     
                        //RES 0,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) & 0xFE;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x85:
                    {     
                        //RES 0,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) & 0xFE;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x86:
                    {     
                        //RES 0,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) & 0xFE);
                        break;
                    }
                case 0x87:
                    {     
                        //RES 0,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) & 0xFE;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x88:
                    {     
                        //RES 1,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) & 0xFD;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x89:
                    {     
                        //RES 1,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) & 0xFD;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x8A:
                    {     
                        //RES 1,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) & 0xFD;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x8B:
                    {     
                        //RES 1,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) & 0xFD;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x8C:
                    {     
                        //RES 1,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) & 0xFD;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x8D:
                    {     
                        //RES 1,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) & 0xFD;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x8E:
                    {     
                        //RES 1,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) & 0xFD);
                        break;
                    }
                case 0x8F:
                    {     
                        //RES 1,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) & 0xFD;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x90:
                    {     
                        //RES 2,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) & 0xFB;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x91:
                    {     
                        //RES 2,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) & 0xFB;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x92:
                    {     
                        //RES 2,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) & 0xFB;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x93:
                    {     
                        //RES 2,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) & 0xFB;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x94:
                    {     
                        //RES 2,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) & 0xFB;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x95:
                    {     
                        //RES 2,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) & 0xFB;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x96:
                    {     
                        //RES 2,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) & 0xFB);
                        break;
                    }
                case 0x97:
                    {     
                        //RES 2,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) & 0xFB;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0x98:
                    {     
                        //RES 3,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) & 0xF7;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0x99:
                    {
                        //RES 3,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) & 0xF7;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0x9A:
                    {     
                        //RES 3,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) & 0xF7;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0x9B:
                    {     
                        //RES 3,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) & 0xF7;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0x9C:
                    {     
                        //RES 3,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) & 0xF7;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0x9D:
                    {     
                        //RES 3,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) & 0xF7;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0x9E:
                    {     
                        //RES 3,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) & 0xF7);
                        break;
                    }
                case 0x9F:
                    {     
                        //RES 3,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) & 0xF7;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xA0:
                    {
                        //RES 4,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) & 0xEF;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xA1:
                    {     
                        //RES 4,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) & 0xEF;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xA2:
                    {     
                        //RES 4,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) & 0xEF;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xA3:
                    {     
                        //RES 4,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) & 0xEF;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xA4:
                    {     
                        //RES 4,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) & 0xEF;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xA5:
                    {     
                        //RES 4,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) & 0xEF;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xA6:
                    {     
                        //RES 4,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) & 0xEF);
                        break;
                    }
                case 0xA7:
                    {     
                        //RES 4,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) & 0xEF;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xA8:
                    {     
                        //RES 5,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) & 0xDF;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xA9:
                    {     
                        //RES 5,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) & 0xDF;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xAA:
                    {     
                        //RES 5,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) & 0xDF;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xAB:
                    {     
                        //RES 5,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) & 0xDF;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xAC:
                    {     
                        //RES 5,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) & 0xDF;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xAD:
                    {     
                        //RES 5,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) & 0xDF;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xAE:
                    {     
                        //RES 5,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) & 0xDF);
                        break;
                    }
                case 0xAF:
                    {     
                        //RES 5,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) & 0xDF;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xB0:
                    {     
                        //RES 6,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) & 0xBF;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xB1:
                    {     
                        //RES 6,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) & 0xBF;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xB2:
                    {     
                        //RES 6,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) & 0xBF;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xB3:
                    {     
                        //RES 6,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) & 0xBF;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xB4:
                    {     
                        //RES 6,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) & 0xBF;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xB5:
                    {     
                        //RES 6,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) & 0xBF;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xB6:
                    {     
                        //RES 6,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) & 0xBF);
                        break;
                    }
                case 0xB7:
                    {     
                        //RES 6,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) & 0xBF;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xB8:
                    {     
                        //RES 7,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) & 0x7F;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xB9:
                    {     
                        //RES 7,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) & 0x7F;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xBA:
                    {     
                        //RES 7,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) & 0x7F;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xBB:
                    {     
                        //RES 7,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) & 0x7F;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xBC:
                    {     
                        //RES 7,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) & 0x7F;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xBD:
                    {     
                        //RES 7,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) & 0x7F;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xBE:
                    {     
                        //RES 7,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) & 0x7F);
                        break;
                    }
                case 0xBF:
                    {     
                        //RES 7,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) & 0x7F;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xC0:
                    {     
                        //SET 0,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) | 0x01;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xC1:
                    {     
                        //SET 0,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) | 0x01;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xC2:
                    {     
                        //SET 0,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) | 0x01;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xC3:
                    {     
                        //SET 0,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) | 0x01;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xC4:
                    {     
                        //SET 0,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) | 0x01;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xC5:
                    {     
                        //SET 0,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) | 0x01;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xC6:
                    {     
                        //SET 0,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) | 0x01);
                        break;
                    }
                case 0xC7:
                    {     
                        //SET 0,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) | 0x01;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xC8:
                    {     
                        //SET 1,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) | 0x02;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xC9:
                    {
                        //SET 1,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) | 0x02;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xCA:
                    {     
                        //SET 1,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) | 0x02;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xCB:
                    {     
                        //SET 1,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) | 0x02;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xCC:
                    {     
                        //SET 1,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) | 0x02;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xCD:
                    {     
                        //SET 1,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) | 0x02;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xCE:
                    {     
                        //SET 1,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) | 0x02);
                        break;
                    }
                case 0xCF:
                    {     
                        //SET 1,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) | 0x02;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xD0:
                    {
                        //SET 2,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) | 0x04;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xD1:
                    {     
                        //SET 2,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) | 0x04;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xD2:
                    {     
                        //SET 2,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) | 0x04;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xD3:
                    {     
                        //SET 2,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) | 0x04;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xD4:
                    {     
                        //SET 2,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) | 0x04;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xD5:
                    {     
                        //SET 2,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) | 0x04;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xD6:
                    {     
                        //SET 2,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) | 0x04);
                        break;
                    }
                case 0xD7:
                    {     
                        //SET 2,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) | 0x04;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xD8:
                    {     
                        //SET 3,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) | 0x08;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xD9:
                    {
                        //SET 3,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) | 0x08;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xDA:
                    {     
                        //SET 3,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) | 0x08;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xDB:
                    {     
                        //SET 3,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) | 0x08;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xDC:
                    {     
                        //SET 3,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) | 0x08;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xDD:
                    {     
                        //SET 3,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) | 0x08;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xDE:
                    {     
                        //SET 3,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) | 0x08);
                        break;
                    }
                case 0xDF:
                    {     
                        //SET 3,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) | 0x08;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xE0:
                    {     
                        //SET 4,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) | 0x10;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xE1:
                    {
                        //SET 4,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) | 0x10;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xE2:
                    {     
                        //SET 4,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) | 0x10;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xE3:
                    {     
                        //SET 4,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) | 0x10;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xE4:
                    {     
                        //SET 4,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) | 0x10;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xE5:
                    {     
                        //SET 4,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) | 0x10;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xE6:
                    {     
                        //SET 4,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) | 0x10);
                        break;
                    }
                case 0xE7:
                    {     
                        //SET 4,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) | 0x10;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xE8:
                    {
                        //SET 5,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) | 0x20;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xE9:
                    {
                        //SET 5,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) | 0x20;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xEA:
                    {     
                        //SET 5,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) | 0x20;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xEB:
                    {     
                        //SET 5,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) | 0x20;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xEC:
                    {     
                        //SET 5,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) | 0x20;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xED:
                    {     
                        //SET 5,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) | 0x20;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xEE:
                    {     
                        //SET 5,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) | 0x20);
                        break;
                    }
                case 0xEF:
                    {     
                        //SET 5,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) | 0x20;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xF0:
                    {     
                        //SET 6,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) | 0x40;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xF1:
                    {
                        //SET 6,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) | 0x40;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xF2:
                    {     
                        //SET 6,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) | 0x40;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xF3:
                    {     
                        //SET 6,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) | 0x40;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xF4:
                    {     
                        //SET 6,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) | 0x40;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xF5:
                    {     
                        //SET 6,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) | 0x40;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xF6:
                    {     
                        //SET 6,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) | 0x40);
                        break;
                    }
                case 0xF7:
                    {     
                        //SET 6,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) | 0x40;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                case 0xF8:
                    {
                        //SET 7,(IX+d),B
                        tStates += 23;
                        B = memory.ReadByte(ixd) | 0x80;
                        memory.WriteByte(ixd, B);
                        break;
                    }
                case 0xF9:
                    {
                        //SET 7,(IX+d),C
                        tStates += 23;
                        C = memory.ReadByte(ixd) | 0x80;
                        memory.WriteByte(ixd, C);
                        break;
                    }
                case 0xFA:
                    {     
                        //SET 7,(IX+d),D
                        tStates += 23;
                        D = memory.ReadByte(ixd) | 0x80;
                        memory.WriteByte(ixd, D);
                        break;
                    }
                case 0xFB:
                    {     
                        //SET 7,(IX+d),E
                        tStates += 23;
                        E = memory.ReadByte(ixd) | 0x80;
                        memory.WriteByte(ixd, E);
                        break;
                    }
                case 0xFC:
                    {     
                        //SET 7,(IX+d),H
                        tStates += 23;
                        H = memory.ReadByte(ixd) | 0x80;
                        memory.WriteByte(ixd, H);
                        break;
                    }
                case 0xFD:
                    {     
                        //SET 7,(IX+d),L
                        tStates += 23;
                        L = memory.ReadByte(ixd) | 0x80;
                        memory.WriteByte(ixd, L);
                        break;
                    }
                case 0xFE:
                    {     
                        //SET 7,(IX+d)
                        tStates += 23;
                        memory.WriteByte(ixd, memory.ReadByte(ixd) | 0x80);
                        break;
                    }
                case 0xFF:
                    {
                        //SET 7,(IX+d),A
                        tStates += 23;
                        A = memory.ReadByte(ixd) | 0x80;
                        memory.WriteByte(ixd, A);
                        break;
                    }
                default:
                    {
                        tStates += 12;
                        break;
                    }
            }  
        }

        #endregion

        #region FDCB prefix

        private void ExecuteFDCB()
        {
            int iyd = IY + (sbyte)memory.ReadByte(PC++);
            int opcode = memory.ReadByte(PC++);
            switch (opcode)
            {
                case 0x00:
                    {
                        //RLC (IY+d),B
                        tStates += 23;
                        B = Rlc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x01:
                    {
                        //RLC (IY+d),C
                        tStates += 23;
                        C = Rlc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x02:
                    {
                        //RLC (IY+d),D
                        tStates += 23;
                        D = Rlc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x03:
                    {
                        //RLC (IY+d),E
                        tStates += 23;
                        E = Rlc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x04:
                    {
                        //RLC (IY+d),H
                        tStates += 23;
                        H = Rlc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x05:
                    {
                        //RLC (IY+d),L
                        tStates += 23;
                        L = Rlc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x06:
                    {
                        //RLC (IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, Rlc(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x07:
                    {
                        //RLC (IY+d),A
                        tStates += 23;
                        A = Rlc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x08:
                    {
                        //RRC (IY+d),B
                        tStates += 23;
                        B = Rrc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x09:
                    {
                        //RRC (IY+d),C
                        tStates += 23;
                        C = Rrc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x0A:
                    {
                        //RRC (IY+d),D
                        tStates += 23;
                        D = Rrc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x0B:
                    {
                        //RRC (IY+d),E
                        tStates += 23;
                        E = Rrc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x0C:
                    {
                        //RRC (IY+d),H
                        tStates += 23;
                        H = Rrc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x0D:
                    {
                        //RRC (IY+d),L
                        tStates += 23;
                        L = Rrc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x0E:
                    {
                        //RRC (IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, Rrc(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x0F:
                    {
                        //RRC (IY+d),A
                        tStates+= 23;
                        A = Rrc(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x10:
                    {
                        //RL (IY+d),B
                        tStates += 23;
                        B = Rl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x11:
                    {
                        //RL (IY+d),C
                        tStates += 23;
                        C = Rl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x12:
                    {
                        //RL (IY+d),D
                        tStates += 23;
                        D = Rl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x13:
                    {
                        //RL (IY+d),E
                        tStates += 23;
                        E = Rl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x14:
                    {
                        //RL (IY+d),H
                        tStates += 23;
                        H = Rl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x15:
                    {
                        //RL (IY+d),L
                        tStates += 23;
                        L = Rl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x16:
                    {
                        //RL (IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, Rl(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x17:
                    {
                        //RL (IY+d),A
                        tStates += 23;
                        A = Rl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x18:
                    {
                        //RR (IY+d),B
                        tStates += 23;
                        B = Rr(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x19:
                    {
                        //RR (IY+d),C
                        tStates += 23;
                        C = Rr(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x1A:
                    {
                        //RR (IY+d),D
                        tStates += 23;
                        D = Rr(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x1B:
                    {
                        //RR (IY+d),E
                        tStates += 23;
                        E = Rr(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x1C:
                    {
                        //RR (IY+d),H
                        tStates += 23;
                        H = Rr(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x1D:
                    {
                        //RR (IY+d),L
                        tStates += 23;
                        L = Rr(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x1E:
                    {
                        //RR (IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, Rr(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x1F:
                    {
                        //RR (IY+d),A
                        tStates += 23;
                        A = Rr(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x20:
                    {
                        //SLA (IY+d),B
                        tStates += 23;
                        B = Sla(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x21:
                    {
                        //SLA (IY+d),C
                        tStates += 23;
                        C = Sla(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x22:
                    {
                        //SLA (IY+d),D
                        tStates += 23;
                        D = Sla(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x23:
                    {
                        //SLA (IY+d),E
                        tStates += 23;
                        E = Sla(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x24:
                    {
                        //SLA (IY+d),H
                        tStates += 23;
                        H = Sla(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x25:
                    {
                        //SLA (IY+d),L
                        tStates += 23;
                        L = Sla(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x26:
                    {
                        //SLA (IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, Sla(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x27:
                    {
                        //SLA (IY+d),A
                        tStates += 23;
                        A = Sla(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x28:
                    {
                        //SRA (IY+d),B
                        tStates += 23;
                        B = Sra(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x29:
                    {
                        //SRA (IY+d),C
                        tStates += 23;
                        C = Sra(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x2A:
                    {
                        //SRA (IY+d),D
                        tStates += 23;
                        D = Sra(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x2B:
                    {
                        //SRA (IY+d),E
                        tStates += 23;
                        E = Sra(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x2C:
                    {
                        //SRA (IY+d),H
                        tStates += 23;
                        H = Sra(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x2D:
                    {
                        //SRA (IY+d),L
                        tStates += 23;
                        L = Sra(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x2E:
                    {
                        //SRA (IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, Sra(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x2F:
                    {
                        //SRA (IY+d),A
                        tStates += 23;
                        A = Sra(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x30:
                    {
                        //SLL (IY+d),B
                        tStates += 23;
                        B = Sll(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x31:
                    {
                        //SLL (IY+d),C
                        tStates += 23;
                        C = Sll(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x32:
                    {
                        //SLL (IY+d),D
                        tStates += 23;
                        D = Sll(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x33:
                    {
                        //SLL (IY+d),E
                        tStates += 23;
                        E = Sll(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x34:
                    {
                        //SLL (IY+d),H
                        tStates += 23;
                        H = Sll(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x35:
                    {
                        //SLL (IY+d),L
                        tStates += 23;
                        L = Sll(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x36:
                    {
                        //SLL (IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, Sll(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x37:
                    {
                        //SLL (IY+d),A
                        tStates += 23;
                        A = Sll(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x38:
                    {
                        //SRL (IY+d),B
                        tStates += 23;
                        B = Srl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x39:
                    {
                        //SRL (IY+d),C
                        tStates += 23;
                        C = Srl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x3A:
                    {
                        //SRL (IY+d),D
                        tStates += 23;
                        D = Srl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x3B:
                    {
                        //SRL (IY+d),E
                        tStates += 23;
                        E = Srl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x3C:
                    {
                        //SRL (IY+d),H
                        tStates += 23;
                        H = Srl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x3D:
                    {
                        //SRL (IY+d),L
                        tStates += 23;
                        L = Srl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x3E:
                    {
                        //SRL (IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, Srl(memory.ReadByte(iyd)));
                        break;
                    }
                case 0x3F:
                    {
                        //SRL (IY+d),A
                        tStates += 23;
                        A = Srl(memory.ReadByte(iyd));
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x40:
                case 0x41:
                case 0x42:
                case 0x43:
                case 0x44:
                case 0x45:
                case 0x46:
                case 0x47:
                    {
                        //BIT 0,(IY+d)
                        tStates += 20;
                        Bit(0, memory.ReadByte(iyd));
                        break;
                    }
                case 0x48:
                case 0x49:
                case 0x4A:
                case 0x4B:
                case 0x4C:
                case 0x4D:
                case 0x4E:
                case 0x4F:
                    {
                        //BIT 1,(IY+d)
                        tStates += 20;
                        Bit(1, memory.ReadByte(iyd));
                        break;
                    }
                case 0x50:
                case 0x51:
                case 0x52:
                case 0x53:
                case 0x54:
                case 0x55:
                case 0x56:
                case 0x57:
                    {
                        //BIT 2,(IY+d)
                        tStates += 20;
                        Bit(2, memory.ReadByte(iyd));
                        break;
                    }
                case 0x58:
                case 0x59:
                case 0x5A:
                case 0x5B:
                case 0x5C:
                case 0x5D:
                case 0x5E:
                case 0x5F:
                    {
                        //BIT 3,(IY+d)
                        tStates += 20;
                        Bit(3, memory.ReadByte(iyd));
                        break;
                    }
                case 0x60:
                case 0x61:
                case 0x62:
                case 0x63:
                case 0x64:
                case 0x65:
                case 0x66:
                case 0x67:
                    {
                        //BIT 4,(IY+d)
                        tStates += 20;
                        Bit(4, memory.ReadByte(iyd));
                        break;
                    }
                case 0x68:
                case 0x69:
                case 0x6A:
                case 0x6B:
                case 0x6C:
                case 0x6D:
                case 0x6E:
                case 0x6F:
                    {
                        //BIT 5,(IY+d)
                        tStates += 20;
                        Bit(5, memory.ReadByte(iyd));
                        break;
                    }
                case 0x70:
                case 0x71:
                case 0x72:
                case 0x73:
                case 0x74:
                case 0x75:
                case 0x76:
                case 0x77:
                    {
                        //BIT 6,(IY+d)
                        tStates += 20;
                        Bit(6, memory.ReadByte(iyd));
                        break;
                    }
                case 0x78:
                case 0x79:
                case 0x7A:
                case 0x7B:
                case 0x7C:
                case 0x7D:
                case 0x7E:
                case 0x7F:
                    {
                        //BIT 7,(IY+d)
                        tStates += 20;
                        Bit(7, memory.ReadByte(iyd));
                        break;
                    }
                case 0x80:
                    {
                        //RES 0,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) & 0xFE;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x81:
                    {
                        //RES 0,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) & 0xFE;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x82:
                    {
                        //RES 0,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) & 0xFE;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x83:
                    {
                        //RES 0,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) & 0xFE;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x84:
                    {
                        //RES 0,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) & 0xFE;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x85:
                    {
                        //RES 0,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) & 0xFE;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x86:
                    {
                        //RES 0,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) & 0xFE);
                        break;
                    }
                case 0x87:
                    {
                        //RES 0,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) & 0xFE;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x88:
                    {
                        //RES 1,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) & 0xFD;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x89:
                    {
                        //RES 1,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) & 0xFD;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x8A:
                    {
                        //RES 1,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) & 0xFD;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x8B:
                    {
                        //RES 1,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) & 0xFD;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x8C:
                    {
                        //RES 1,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) & 0xFD;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x8D:
                    {
                        //RES 1,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) & 0xFD;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x8E:
                    {
                        //RES 1,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) & 0xFD);
                        break;
                    }
                case 0x8F:
                    {
                        //RES 1,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) & 0xFD;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x90:
                    {
                        //RES 2,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) & 0xFB;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x91:
                    {
                        //RES 2,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) & 0xFB;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x92:
                    {
                        //RES 2,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) & 0xFB;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x93:
                    {
                        //RES 2,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) & 0xFB;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x94:
                    {
                        //RES 2,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) & 0xFB;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x95:
                    {
                        //RES 2,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) & 0xFB;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x96:
                    {
                        //RES 2,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) & 0xFB);
                        break;
                    }
                case 0x97:
                    {
                        //RES 2,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) & 0xFB;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0x98:
                    {
                        //RES 3,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) & 0xF7;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0x99:
                    {
                        //RES 3,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) & 0xF7;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0x9A:
                    {
                        //RES 3,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) & 0xF7;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0x9B:
                    {
                        //RES 3,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) & 0xF7;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0x9C:
                    {
                        //RES 3,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) & 0xF7;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0x9D:
                    {
                        //RES 3,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) & 0xF7;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0x9E:
                    {
                        //RES 3,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) & 0xF7);
                        break;
                    }
                case 0x9F:
                    {
                        //RES 3,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) & 0xF7;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xA0:
                    {
                        //RES 4,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) & 0xEF;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xA1:
                    {
                        //RES 4,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) & 0xEF;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xA2:
                    {
                        //RES 4,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) & 0xEF;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xA3:
                    {
                        //RES 4,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) & 0xEF;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xA4:
                    {
                        //RES 4,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) & 0xEF;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xA5:
                    {
                        //RES 4,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) & 0xEF;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xA6:
                    {
                        //RES 4,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) & 0xEF);
                        break;
                    }
                case 0xA7:
                    {
                        //RES 4,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) & 0xEF;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xA8:
                    {
                        //RES 5,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) & 0xDF;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xA9:
                    {
                        //RES 5,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) & 0xDF;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xAA:
                    {
                        //RES 5,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) & 0xDF;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xAB:
                    {
                        //RES 5,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) & 0xDF;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xAC:
                    {
                        //RES 5,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) & 0xDF;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xAD:
                    {
                        //RES 5,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) & 0xDF;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xAE:
                    {
                        //RES 5,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) & 0xDF);
                        break;
                    }
                case 0xAF:
                    {
                        //RES 5,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) & 0xDF;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xB0:
                    {
                        //RES 6,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) & 0xBF;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xB1:
                    {
                        //RES 6,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) & 0xBF;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xB2:
                    {
                        //RES 6,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) & 0xBF;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xB3:
                    {
                        //RES 6,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) & 0xBF;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xB4:
                    {
                        //RES 6,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) & 0xBF;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xB5:
                    {
                        //RES 6,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) & 0xBF;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xB6:
                    {
                        //RES 6,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) & 0xBF);
                        break;
                    }
                case 0xB7:
                    {
                        //RES 6,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) & 0xBF;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xB8:
                    {
                        //RES 7,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) & 0x7F;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xB9:
                    {
                        //RES 7,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) & 0x7F;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xBA:
                    {
                        //RES 7,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) & 0x7F;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xBB:
                    {
                        //RES 7,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) & 0x7F;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xBC:
                    {
                        //RES 7,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) & 0x7F;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xBD:
                    {
                        //RES 7,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) & 0x7F;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xBE:
                    {
                        //RES 7,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) & 0x7F);
                        break;
                    }
                case 0xBF:
                    {
                        //RES 7,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) & 0x7F;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xC0:
                    {
                        //SET 0,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) | 0x01;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xC1:
                    {
                        //SET 0,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) | 0x01;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xC2:
                    {
                        //SET 0,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) | 0x01;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xC3:
                    {
                        //SET 0,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) | 0x01;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xC4:
                    {
                        //SET 0,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) | 0x01;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xC5:
                    {
                        //SET 0,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) | 0x01;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xC6:
                    {
                        //SET 0,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) | 0x01);
                        break;
                    }
                case 0xC7:
                    {
                        //SET 0,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) | 0x01;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xC8:
                    {
                        //SET 1,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) | 0x02;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xC9:
                    {
                        //SET 1,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) | 0x02;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xCA:
                    {
                        //SET 1,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) | 0x02;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xCB:
                    {
                        //SET 1,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) | 0x02;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xCC:
                    {
                        //SET 1,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) | 0x02;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xCD:
                    {
                        //SET 1,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) | 0x02;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xCE:
                    {
                        //SET 1,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) | 0x02);
                        break;
                    }
                case 0xCF:
                    {
                        //SET 1,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) | 0x02;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xD0:
                    {
                        //SET 2,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) | 0x04;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xD1:
                    {
                        //SET 2,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) | 0x04;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xD2:
                    {
                        //SET 2,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) | 0x04;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xD3:
                    {
                        //SET 2,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) | 0x04;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xD4:
                    {
                        //SET 2,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) | 0x04;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xD5:
                    {
                        //SET 2,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) | 0x04;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xD6:
                    {
                        //SET 2,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) | 0x04);
                        break;
                    }
                case 0xD7:
                    {
                        //SET 2,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) | 0x04;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xD8:
                    {
                        //SET 3,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) | 0x08;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xD9:
                    {
                        //SET 3,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) | 0x08;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xDA:
                    {
                        //SET 3,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) | 0x08;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xDB:
                    {
                        //SET 3,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) | 0x08;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xDC:
                    {
                        //SET 3,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) | 0x08;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xDD:
                    {
                        //SET 3,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) | 0x08;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xDE:
                    {
                        //SET 3,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) | 0x08);
                        break;
                    }
                case 0xDF:
                    {
                        //SET 3,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) | 0x08;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xE0:
                    {
                        //SET 4,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) | 0x10;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xE1:
                    {
                        //SET 4,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) | 0x10;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xE2:
                    {
                        //SET 4,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) | 0x10;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xE3:
                    {
                        //SET 4,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) | 0x10;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xE4:
                    {
                        //SET 4,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) | 0x10;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xE5:
                    {
                        //SET 4,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) | 0x10;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xE6:
                    {
                        //SET 4,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) | 0x10);
                        break;
                    }
                case 0xE7:
                    {
                        //SET 4,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) | 0x10;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xE8:
                    {
                        //SET 5,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) | 0x20;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xE9:
                    {
                        //SET 5,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) | 0x20;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xEA:
                    {
                        //SET 5,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) | 0x20;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xEB:
                    {
                        //SET 5,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) | 0x20;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xEC:
                    {
                        //SET 5,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) | 0x20;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xED:
                    {
                        //SET 5,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) | 0x20;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xEE:
                    {
                        //SET 5,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) | 0x20);
                        break;
                    }
                case 0xEF:
                    {
                        //SET 5,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) | 0x20;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xF0:
                    {
                        //SET 6,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) | 0x40;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xF1:
                    {
                        //SET 6,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) | 0x40;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xF2:
                    {
                        //SET 6,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) | 0x40;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xF3:
                    {
                        //SET 6,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) | 0x40;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xF4:
                    {
                        //SET 6,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) | 0x40;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xF5:
                    {
                        //SET 6,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) | 0x40;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xF6:
                    {
                        //SET 6,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) | 0x40);
                        break;
                    }
                case 0xF7:
                    {
                        //SET 6,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) | 0x40;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                case 0xF8:
                    {
                        //SET 7,(IY+d),B
                        tStates += 23;
                        B = memory.ReadByte(iyd) | 0x80;
                        memory.WriteByte(iyd, B);
                        break;
                    }
                case 0xF9:
                    {
                        //SET 7,(IY+d),C
                        tStates += 23;
                        C = memory.ReadByte(iyd) | 0x80;
                        memory.WriteByte(iyd, C);
                        break;
                    }
                case 0xFA:
                    {
                        //SET 7,(IY+d),D
                        tStates += 23;
                        D = memory.ReadByte(iyd) | 0x80;
                        memory.WriteByte(iyd, D);
                        break;
                    }
                case 0xFB:
                    {
                        //SET 7,(IY+d),E
                        tStates += 23;
                        E = memory.ReadByte(iyd) | 0x80;
                        memory.WriteByte(iyd, E);
                        break;
                    }
                case 0xFC:
                    {
                        //SET 7,(IY+d),H
                        tStates += 23;
                        H = memory.ReadByte(iyd) | 0x80;
                        memory.WriteByte(iyd, H);
                        break;
                    }
                case 0xFD:
                    {
                        //SET 7,(IY+d),L
                        tStates += 23;
                        L = memory.ReadByte(iyd) | 0x80;
                        memory.WriteByte(iyd, L);
                        break;
                    }
                case 0xFE:
                    {
                        //SET 7,(IY+d)
                        tStates += 23;
                        memory.WriteByte(iyd, memory.ReadByte(iyd) | 0x80);
                        break;
                    }
                case 0xFF:
                    {
                        //SET 7,(IY+d),A
                        tStates += 23;
                        A = memory.ReadByte(iyd) | 0x80;
                        memory.WriteByte(iyd, A);
                        break;
                    }
                default:
                    {
                        tStates += 12;
                        break;
                    }
            }
        }

        #endregion

        #region ED prefix

        private void ExecuteED()
        {
            FetchCounter++;
            R = ++R &0x7F;
            int opcode = memory.ReadByte(PC++);
            switch (opcode)
            {
                case 0x40:
                    {
                        //IN B,(C)
                        tStates += 12;
                        B = io.ReadPort(BC);
                        F = (F & CF) | SZP[B];
                        break;
                    }
                case 0x41:
                    {     
                        //OUT (C),B
                        tStates += 12;
                        io.WritePort(BC, B);
                        break;
                    }
                case 0x42:
                    {
                        //SBC HL,BC
                        tStates += 15;
                        Sbc16(BC);
                        break;
                    }
                case 0x43:
                    {     
                        //LD (nn),BC
                        tStates += 20;                        
                        memory.WriteWord(memory.ReadWord(PC), BC);
                        PC += 2;
                        break;
                    }
                case 0x44:
                case 0x4C:
                case 0x54:
                case 0x5C:
                case 0x64:
                case 0x6C:
                case 0x74:
                case 0x7C:
                    {     
                        //NEG
                        tStates += 8;
                        int aux = A;
                        A = 0;
                        Sub(aux);
                        break;
                    }
                case 0x45:
                case 0x55:
                case 0x5D:
                case 0x65:
                case 0x6D:
                case 0x75:
                case 0x7D:
                    {     
                        //RETN
                        tStates += 14;
                        IFF1 = IFF2;
                        PC = memory.ReadByte(SP++) & 0xFF;
                        PC |= (memory.ReadByte(SP++) << 8);
                        break;
                    }
                case 0x46:
                case 0x4E:
                case 0x66:
                case 0x6E:
                    {     
                        //IM 0
                        tStates += 8;
                        InterruptMode = 0;
                        break;
                    }
                case 0x47:
                    {     
                        //LD I,A
                        tStates += 9;
                        I = A;
                        break;
                    }
                case 0x48:
                    {     
                        //IN C,(C)
                        tStates += 12;
                        C = io.ReadPort(BC);
                        F = (F & CF) | SZP[C];
                        break;
                    }
                case 0x49:
                    {     
                        //OUT (C),C
                        tStates += 12;
                        io.WritePort(BC, C);
                        break;
                    }
                case 0x4A:
                    {     
                        //ADC HL,BC
                        tStates += 15;
                        Adc16(BC);
                        break;
                    }
                case 0x4B:
                    {     
                        //LD BC,(nn)
                        tStates += 20;
                        BC = memory.ReadWord(memory.ReadWord(PC));
                        PC += 2;
                        break;
                    }
                case 0x4D:
                    {     
                        //RETI
                        tStates += 14;
                        PC = memory.ReadByte(SP++) & 0xFF;
                        PC |= (memory.ReadByte(SP++) << 8);
                        break;
                    }
                case 0x4F:
                    {     
                        //LD R,A
                        tStates += 9;
                        R = A &0x7F;
                        R7 = A & 0x80;
                        break;
                    }
                case 0x50:
                    {     
                        //IN D,(C)
                        tStates += 12;
                        D = io.ReadPort(BC);
                        F = (F & CF) | SZP[D];
                        break;
                    }
                case 0x51:
                    {     
                        //OUT (C),D
                        tStates += 12;
                        io.WritePort(BC, D);
                        break;
                    }
                case 0x52:
                    {     
                        //SBC HL,DE
                        tStates += 15;
                        Sbc16(DE);
                        break;
                    }
                case 0x53:
                    {     
                        //LD (nn),DE
                        tStates += 20;
                        memory.WriteWord(memory.ReadWord(PC), DE);
                        PC += 2;
                        break;
                    }
                case 0x56:
                case 0x76:
                    {     
                        //IM 1
                        tStates += 8;
                        InterruptMode = 1;
                        break;
                    }
                case 0x57:
                    {     
                        //LD A,I
                        tStates += 9;
                        A = I;
                        F = (F & CF) | SZ[A] | ((IFF2 == true) ? 1 : 0 << 2);
                        break;
                    }
                case 0x58:
                    {     
                        //IN E,(C)
                        tStates += 12;
                        E = io.ReadPort(BC);
                        F = (F & CF) | SZP[E];
                        break;
                    }
                case 0x59:
                    {     
                        //OUT (C),E
                        tStates += 12;
                        io.WritePort(BC, E);
                        break;
                    }
                case 0x5A:
                    {     
                        //ADC HL,DE
                        tStates += 15;
                        Adc16(DE);
                        break;
                    }
                case 0x5B:
                    {     
                        //LD DE,(nn)
                        tStates += 20;
                        DE = memory.ReadWord(memory.ReadWord(PC));
                        PC += 2;
                        break;
                    }
                case 0x5E:
                case 0x7E:
                    {     
                        //IM 2
                        tStates += 8;
                        InterruptMode = 2;
                        break;
                    }
                case 0x5F:
                    {     
                        //LD A,R
                        tStates += 9;
                        A = (R & 0x7f) | R7;
	                    F = (F & CF) | SZ[A] | ( IFF2?1:0 << 2 );						
                        break;
                    }
                case 0x60:
                    {     
                        //IN H,(C)
                        tStates += 12;
                        H = io.ReadPort(BC);
                        F = (F & CF) | SZP[H];
                        break;
                    }
                case 0x61:
                    {     
                        //OUT (C),H
                        tStates += 12;
                        io.WritePort(BC, H);
                        break;
                    }
                case 0x62:
                    {     
                        //SBC HL,HL
                        tStates += 15;
                        Sbc16(HL);
                        break;
                    }
                case 0x63:
                    {     
                        //LD (nn),HL
                        tStates += 20;
                        memory.WriteWord(memory.ReadWord(PC), HL);
                        PC += 2;
                        break;
                    }
                case 0x67:
                    {     
                        //RRD
                        tStates += 18;
                        Rrd();
                        break;
                    }
                case 0x68:
                    {     
                        //IN L,(C)
                        tStates += 12;
                        L = io.ReadPort(BC);
                        F = (F & CF) | SZP[L];
                        break;
                    }
                case 0x69:
                    {     
                        //OUT (C),L
                        tStates += 12;
                        io.WritePort(BC, L);
                        break;
                    }
                case 0x6A:
                    {     
                        //ADC HL,HL
                        tStates += 15;
                        Adc16(HL);
                        break;
                    }
                case 0x6B:
                    {     
                        //LD HL,(nn)
                        tStates += 20;
                        HL = memory.ReadWord(memory.ReadWord(PC));
                        PC += 2;
                        break;
                    }
                case 0x6F:
                    {     
                        //RLD
                        tStates += 18;
                        Rld();
                        break;
                    }
                case 0x70:
                    {     
                        //IN (C)
                        tStates += 12;
                        int res = io.ReadPort(BC);
                        F = (F & CF) | SZP[res];
                        break;
                    }
                case 0x71:
                    {     
                        //OUT (C),0
                        tStates += 12;
                        io.WritePort(BC, 0);
                        break;
                    }
                case 0x72:
                    {     
                        //SBC HL,SP
                        tStates += 15;
                        Sbc16(SP);
                        break;
                    }
                case 0x73:
                    {     
                        //LD (nn),SP
                        tStates += 20;
                        memory.WriteWord(memory.ReadWord(PC), SP);
                        PC += 2;
                        break;
                    }
                case 0x78:
                    {     
                        //IN A,(C)
                        tStates += 12;
                        A = io.ReadPort(BC);
                        F = (F & CF) | SZP[A];
                        break;
                    }
                case 0x79:
                    {     
                        //OUT (C),A
                        tStates += 12;
                        io.WritePort(BC, A);
                        break;
                    }
                case 0x7A:
                    {     
                        //ADC HL,SP
                        tStates += 15;
                        Adc16(SP);
                        break;
                    }
                case 0x7B:
                    {     
                        //LD SP,(nn)
                        tStates += 20;
                        SP = memory.ReadWord(memory.ReadWord(PC));
                        PC += 2;
                        break;
                    }
                case 0xA0:
                    {     
                        //LDI
                        tStates += 16;
                        int io = memory.ReadByte(HL);
                        memory.WriteByte(DE, io);
	                    F &= SF | ZF | CF;
                        if (((A + io) & 0x02) != 0) F |= YF; /* bit 1 -> flag 5 */
                        if (((A + io) & 0x08) != 0) F |= XF; /* bit 3 -> flag 3 */		
	                    HL = ++HL&0xFFFF;
                        DE = ++DE&0xFFFF;
                        BC = --BC&0xFFFF;
                        if (BC != 0) F |= VF;
                        break;
                    }
                case 0xA1:
                    {     
                        //CPI
                        tStates += 16;
                        int val = memory.ReadByte(HL);
                        int res = (byte)(A - val);
                        HL = ++HL & 0xFFFF;
                        BC = --BC & 0xFFFF;
                        F = (F & CF) | (SZ[res] & ~(YF | XF)) | ((A ^ val ^ res) & HF) | NF;
                        if ((F & HF) != 0) res -= 1;
                        if ((res & 0x02) != 0) F |= YF; /* bit 1 -> flag 5 */
                        if ((res & 0x08) != 0) F |= XF; /* bit 3 -> flag 3 */
                        if (BC != 0) F |= VF;
                        break;
                    }
                case 0xA2:
                    {     
                        //INI
                        tStates += 16;
                        uint t;													
	                    int io = this.io.ReadPort(BC);
                        B = --B & 0xFF;
	                    memory.WriteByte( HL, io );
                        HL = ++HL & 0xFFFF;
	                    F = SZ[B];
                        t =(uint)((byte)((C + 1) & 0xff) + (byte)io);				
	                    if( (io & SF)!=0 ) F |= NF;
                        if ((t & 0x100) != 0) F |= (HF | CF);								
	                    F |= SZP[(byte)((t & 0x07) ^ B)] & PF;						
                        break;
                    }
                case 0xA3:
                    {     
                        //OUTI
                        tStates += 16;
                        uint t;													
	                    int io = memory.ReadByte(HL);
                        B = --B & 0xFF;
	                    this.io.WritePort( BC, io );
                        HL = ++HL & 0xFFFF;
	                    F = SZ[B];													
                        t = (uint)L + (uint)io;								
	                    if( (io & SF)!=0 ) F |= NF;										
	                    if( (t & 0x100)!=0 ) F |= HF | CF;							
	                    F |= SZP[(byte)(t & 0x07) ^ B] & PF;						
                        break;
                    }
                case 0xA8:
                    {
                        //LDD
                        tStates += 16;
                        int io = memory.ReadByte(HL);											
	                    memory.WriteByte( DE, io );												
	                    F &= SF | ZF | CF;											
	                    if( ((A + io) & 0x02)!=0 ) F |= YF; /* bit 1 -> flag 5 */	
	                    if( ((A + io) & 0x08)!=0 ) F |= XF; /* bit 3 -> flag 3 */
                        HL = --HL & 0xFFFF;
                        DE = --DE & 0xFFFF;
                        BC = --BC & 0xFFFF;                        											
	                    if( BC!=0 ) F |= VF;											
                        break;
                    }
                case 0xA9:
                    {     
                        //CPD
                        tStates += 16;
                        int val = memory.ReadByte(HL);											
	                    int res = (byte)(A - val);
                        HL = --HL & 0xFFFF;
                        BC = --BC & 0xFFFF;				
	                    F = (F & CF) | (SZ[res]&~(YF|XF)) | ((A^val^res)&HF) | NF;
                        if ((F & HF) != 0) res = (byte)--res;										
	                    if( (res & 0x02)!=0 ) F |= YF; /* bit 1 -> flag 5 */				
	                    if( (res & 0x08)!=0 ) F |= XF; /* bit 3 -> flag 3 */				
	                    if( BC!=0 ) F |= VF;	
                        break;
                    }
                case 0xAA:
                    {     
                        //IND
                        tStates += 16;
                        uint t;													
	                    int io = this.io.ReadPort(BC);
                        B = --B & 0xFF;											
	                    memory.WriteByte( HL, io );
                        HL = --HL & 0xFFFF;
	                    F = SZ[B];													
	                    t = ((uint)(C - 1) & 0xff) + (uint)io;				
	                    if( (io & SF)!=0 ) F |= NF;										
	                    if( (t & 0x100)!=0 ) F |= HF | CF;								
	                    F |= SZP[(t & 0x07) ^ B] & PF;		
                        break;
                    }
                case 0xAB:
                    {     
                        //OUTD
                        tStates+=16;
                        uint t;													
	                    int io = memory.ReadByte(HL);
                        B = --B & 0xFF;											
	                    this.io.WritePort( BC, io );
                        HL = --HL & 0xFFFF;												
	                    F = SZ[B];													
	                    t = (uint)L + (uint)io;								
	                    if( (io & SF)!=0 ) F |= NF;										
	                    if( (t & 0x100)!=0 ) F |= HF | CF;								
	                    F |= SZP[(t & 0x07) ^ B] & PF;						
                        break;
                    }
                case 0xB0:
                    {     
                        //LDIR
                        tStates += 16;
                        int io = memory.ReadByte(HL);
                        memory.WriteByte(DE, io);
                        F &= SF | ZF | CF;
                        if (((A + io) & 0x02) != 0) F |= YF; /* bit 1 -> flag 5 */
                        if (((A + io) & 0x08) != 0) F |= XF; /* bit 3 -> flag 3 */
                        HL = ++HL & 0xFFFF;
                        DE = ++DE & 0xFFFF;
                        BC = --BC & 0xFFFF;
                        if (BC != 0) F |= VF;
                        if (BC != 0)
                        {
                            PC -= 2;
                            tStates += 5;
                        }
                        break;
                    }
                case 0xB1:
                    {     
                        //CPIR
                        tStates += 16;
                        int val = memory.ReadByte(HL);
                        int res = (byte)(A - val);
                        HL = ++HL & 0xFFFF;
                        BC = --BC & 0xFFFF;
                        F = (F & CF) | (SZ[res] & ~(YF | XF)) | ((A ^ val ^ res) & HF) | NF;
                        if ((F & HF) != 0) res = (byte)--res;
                        if ((res & 0x02) != 0) F |= YF; /* bit 1 -> flag 5 */
                        if ((res & 0x08) != 0) F |= XF; /* bit 3 -> flag 3 */
                        if (BC != 0) F |= VF;
                        if ((BC != 0) && (!((F & ZF) != 0)))
                        {
                            PC -= 2;
                            tStates += 5;
                        }                        
                        break;
                    }
                case 0xB2:
                    {     
                        //INIR
                        tStates += 16;
                        uint t;
                        int io = this.io.ReadPort(BC);
                        B = --B & 0xFF;
                        memory.WriteByte(HL, io);
                        HL = ++HL & 0xFFFF;
                        F = SZ[B];
                        t = (uint)((C + 1) & 0xff) + (uint)io;
                        if ((io & SF) != 0) F |= NF;
                        if ((t & 0x100) != 0) F |= HF | CF;
                        F |= SZP[(t & 0x07) ^ B] & PF;
                        if (B != 0)
                        {
                            PC -= 2;
                            tStates += 5;
                        }
                        break;
                    }
                case 0xB3:
                    {     
                        //OTIR
                        tStates += 16;
                        uint t;
                        int io = memory.ReadByte(HL);
                        B = --B & 0xFF;
                        this.io.WritePort(BC, io);
                        HL = ++HL & 0xFFFF;
                        F = SZ[B];
                        t = (uint)L + (uint)io;
                        if ((io & SF) != 0) F |= NF;
                        if ((t & 0x100) != 0) F |= HF | CF;
                        F |= SZP[(byte)(t & 0x07) ^ B] & PF;
                        if (B!=0)
                        {
                            PC -= 2;
                            tStates += 5;
                        }
                        break;
                    }
                case 0xB8:
                    {     
                        //LDDR
                        tStates += 16;
                        int io = memory.ReadByte(HL);
                        memory.WriteByte(DE, io);
                        F &= SF | ZF | CF;
                        if (((A + io) & 0x02) != 0) F |= YF; /* bit 1 -> flag 5 */
                        if (((A + io) & 0x08) != 0) F |= XF; /* bit 3 -> flag 3 */
                        HL = --HL & 0xFFFF;
                        DE = --DE & 0xFFFF;
                        BC = --BC & 0xFFFF;
                        if (BC != 0) F |= VF;
                        if (BC != 0)
                        {
                            PC -= 2;
                            tStates += 5;
                        }
                        break;
                    }
                case 0xB9:
                    {     
                        //CPDR
                        tStates += 16;
                        int val = memory.ReadByte(HL);
                        int res = (byte)(A - val);
                        HL = --HL & 0xFFFF;
                        BC = --BC & 0xFFFF;
                        F = (F & CF) | (SZ[res] & ~(YF | XF)) | ((A ^ val ^ res) & HF) | NF;
                        if ((F & HF) != 0) res = (byte)--res;
                        if ((res & 0x02) != 0) F |= YF; /* bit 1 -> flag 5 */
                        if ((res & 0x08) != 0) F |= XF; /* bit 3 -> flag 3 */
                        if (BC != 0) F |= VF;
                        if ((BC != 0) && !((F & ZF) != 0))
                        {
                            PC -= 2;
                            tStates += 5;
                        }
                        break;
                    }
                case 0xBA:
                    {     
                        //INDR
                        tStates += 16;
                        uint t;
                        int io = this.io.ReadPort(BC);
                        B = --B & 0xFF;
                        memory.WriteByte(HL, io);
                        HL = --HL & 0xFFFF;
                        F = SZ[B];
                        t = ((uint)(C - 1) & 0xff) + (uint)io;
                        if ((io & SF) != 0) F |= NF;
                        if ((t & 0x100) != 0) F |= HF | CF;
                        F |= SZP[(t & 0x07) ^ B] & PF;
                        if (B != 0)
                        {
                            PC -= 2;
                            tStates += 5;
                        }
                        break;
                    }
                case 0xBB:
                    {     
                        //OTDR
                        tStates += 16;
                        uint t;
                        int io = memory.ReadByte(HL);
                        B = --B & 0xFF;
                        this.io.WritePort(BC, io);
                        HL = --HL & 0xFFFF;
                        F = SZ[B];
                        t = (uint)L + (uint)io;
                        if ((io & SF) != 0) F |= NF;
                        if ((t & 0x100) != 0) F |= HF | CF;
                        F |= SZP[(t & 0x07) ^ B] & PF;
                        if (B!=0)
                        {
                            PC -= 2;
                            tStates += 5;
                        }
                        break;
                    }
                default:
                    {
                        tStates += 8;
                        break;
                    }
            }
        }

        #endregion
    }
}
