/* my_text.h - Alexander Shabarshin (shaos@mail.ru)

   Classes Text and Line for representation of text as list of lines
   
   29 Nov 1996 - First C++ version (http://home.ural.ru/~shabun/public/TEXT.ZIP)
   31 Oct 2000 - Modified for ANSI-C (includable file text.hpp without classes)
   28 Nov 2005 - Divided to my_text.h and my_text.c files (with fixes and comments)
*/   

#ifndef __MY_TEXT_H
#define __MY_TEXT_H

#define MY_MAXLEN 1024 /* Limit for line length */

/* Enumeration for Text errors */

typedef enum myTextErr 
{ 
  TextErrMem = 1, /* Out of Memory */
  TextErrIdx = 2, /* Index Error */
  TextErrFil = 3, /* File Error */
  TextErrFld = 4  /* Wrong Field */
} TextErr;

/* Enumeration for Text fields to sort and to search */

typedef enum myTextFld
{
  TextFldNul, /* do not search */
  TextFldStr, /* line->str */
  TextFldTyp, /* line->type */
  TextFldId1, /* line->id */
  TextFldId2, /* line->id2 */
  TextFldAdr, /* line->adr */
  TextFldLen  /* line->len */
} TextFld;

/* Structure for Line of text representation */

typedef struct myLine
{
  void *next;           /* pointer to next line */
  char *str;            /* pointer to string */
  /* below fields may be used for your purposes: */
  int type;             /* type of the string */
  int id;               /* first identifier of the string */
  int id2;              /* second identifier of the string */
  unsigned short adr;   /* 16-bit word for address */
  unsigned short len;   /* 16-bit word for length */
} Line;

/* Structure for Text representation */

typedef struct myText
{
  long num;             /* number of lines */
  Line *first;          /* first line of text */ 
  Line *last;           /* last line of text */ 
  /* below fields are used for searching: */
  Line *find;           /* last found line */
  TextFld find_fld;     /* search by what field */  
  TextFld find_fld2;    /* search by what second field */  
  char *find_str;       /* searching string */ 
  int find_type;        /* searching type */
  int find_id;          /* searching identifier */ 
  int find_id2;         /* searching additional identifier */
  unsigned short find_adr; /* searching adr field */
  unsigned short find_len; /* searching len field */
} Text;

#ifdef __cplusplus
extern "C" {
#endif

/* Functions for Line object: */

/* Constructor from string. Return pointer to new Line object. */
Line* LineNew(char *s);
/* Destructor. Delete Line object by pointer. */
void LineDel(Line *l);
/* Get string from Line object. */
char* LineGetStr(Line *l);
/* Set type field of Line object. */
void LineSetTyp(Line *l, int t);
/* Get type field of Line object. */
int LineGetTyp(Line *l);
/* Set 1st identifier of Line object. */
void LineSetId1(Line *l, int i);
/* Get 1st identifier of Line object. */
int LineGetId1(Line *l);
/* Set 2nd identifier of Line object. */
void LineSetId2(Line *l, int i);
/* Get 2nd identifier of Line object. */
int LineGetId2(Line *l);
/* Set adr field of Line object. */
void LineSetAdr(Line *l, unsigned short u);
/* Get adr field of Line object. */
unsigned short LineGetAdr(Line *l);
/* Set len field of Line object. */
void LineSetLen(Line *l, unsigned short u);
/* Get len field of Line object. */
unsigned short LineGetLen(Line *l);

/* Functions for Text object: */

/* Constructor without parameters. Return pointer to new Text object. */
Text* TextNew(void);
/* Destructor. Delete Text object by pointer. */
void TextDel(Text *t);
/* Write error with specified code to stdout and exit program. */
void TextError(TextErr err);
/* Return number of lines in Text object. */
long TextGetNumber(Text *t);
/* Add string as line to end of text. Return pointer to new Line object. */
Line* TextAdd(Text *t, char *s);
/* Insert string as line to be with specified index. Return pointer to new Line object. */
Line* TextInsert(Text *t, long n, char *s);
/* Delete line with specified index. */
void TextDelete(Text *t, long n);
/* Delete all lines in Text object. */
void TextAllDelete(Text *t);
/* Return pointer to Line object by index (0 for first). */
Line* TextGet(Text *t, long n);
/* Return index of Line object (0 for first, -1 if not found). */
long TextIndex(Text *t, Line *l);
/* Swap Line object and next Line object. Return 2 if first/last, 1 if not, 0 if error. */
int TextReplace(Text *t, Line *l1);
/* Slow sorting for Text object by field identifier. Return 1 if Ok or 0 if no lines. */
int TextSort(Text *t, TextFld field);
/* Save Text object to text file. */
void TextSave(Text *t, char *filename);
/* Load Text object from text file. */
void TextLoad(Text *t, char *filename);
/* Write list of lines to stdout */
void TextList(Text *t);
/* Search Line object by string. Return pointer to first one or NULL. */
Line* TextFindFirstStr(Text *t, char *s);
/* Search Line object by type. Return pointer to first one or NULL. */
Line* TextFindFirstTyp(Text *t, int y);
/* Search Line object by 1st identifier. Return pointer to first one or NULL. */
Line* TextFindFirstId1(Text *t, int i);
/* Search Line object by 2nd identifier. Return pointer to first one or NULL. */
Line* TextFindFirstId2(Text *t, int i);
/* Search Line object by adr. Return pointer to first one or NULL. */
Line* TextFindFirstAdr(Text *t, unsigned short u);
/* Search Line object by len. Return pointer to first one or NULL. */
Line* TextFindFirstLen(Text *t, unsigned short u);
/* Search Line object by string and type. Return pointer to first one or NULL. */
Line* TextFindFirstStrTyp(Text *t, char *s, int y);
/* Search Line object by type and id. Return pointer to first one or NULL. */
Line* TextFindFirstTypId(Text *t, int y, int i);
/* Return pointer to next Line object or NULL if no more search results. */
Line* TextFindNext(Text *t);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif

