
/************************************

  PicchioEngine

  Copyright(c)2008 Emanuele Bettidi

************************************/

/* Config.cpp */

/* Config */

#include <cstring>
#include <cctype>
#include <stdlib.h>
#include <sstream>
#include <fstream>
#include <ios>
#include <iostream>
#include <SDL/SDL.h>
#include "Types.h"
#include "Input.h"
#include "Config.h"

namespace Config
{
 void to_lower_case(std::string& str);
 int key_to_int(std::string& word);
 bool str_to_int(std::string& word, int32& value);

 const int OPTIONS = 18;
 const int FUNC_KEYS = 72;

 enum {BOOLEAN, INTEGER, KEY, STRING};

 struct option_type
 {
  const char* name;
  int value_type;
  int value;
  int min_value;
  int max_value;
 };

 option_type option[OPTIONS] =
 {
  { "console"         , STRING ,    0,    0,    2 },  // auto
  { "timer_phase"     , INTEGER,    0,    0,    2 },
  { "rgb_output"      , BOOLEAN,    0,    0,    1 },
  { "controller_port" , STRING ,    4,    0,    4 },  // multitap
  { "multitap.port_1" , STRING ,    2,    0,    3 },  // turbopad
  { "multitap.port_2" , STRING ,    2,    0,    3 },  // turbopad
  { "multitap.port_3" , STRING ,    2,    0,    3 },  // turbopad
  { "multitap.port_4" , STRING ,    2,    0,    3 },  // turbopad
  { "multitap.port_5" , STRING ,    2,    0,    3 },  // turbopad
  { "audio_buffer"    , INTEGER, 2048,    1, 3072 },
  { "scale"           , INTEGER,    2,    1,    6 },
  { "overscan"        , BOOLEAN,    0,    0,    1 },
  { "fullscreen"      , BOOLEAN,    0,    0,    1 },
  { "stretch"         , BOOLEAN,    0,    0,    1 },
  { "bilinear"        , BOOLEAN,    0,    0,    1 },
  { "pause_key"       , KEY    ,  112,    0,    0 },  // SDLK_p
  { "exit_key"        , KEY    ,   27,    0,    0 },  // SDLK_ESCAPE
  { "analog_threshold", INTEGER,   75,    1,  100 }
 };

 struct func_key_type
 {
  const char* name;
  const int code;
 };

 func_key_type func_key[FUNC_KEYS] =
 {
  { "backspace", SDLK_BACKSPACE   },
  { "tab"      , SDLK_TAB         },
  { "clear"    , SDLK_CLEAR       },
  { "return"   , SDLK_RETURN      },
  { "pause"    , SDLK_PAUSE       },
  { "escape"   , SDLK_ESCAPE      },
  { "space"    , SDLK_SPACE       },
  { "delete"   , SDLK_DELETE      },
  { "kp_0"     , SDLK_KP0         },
  { "kp_1"     , SDLK_KP1         },
  { "kp_2"     , SDLK_KP2         },
  { "kp_3"     , SDLK_KP3         },
  { "kp_4"     , SDLK_KP4         },
  { "kp_5"     , SDLK_KP5         },
  { "kp_6"     , SDLK_KP6         },
  { "kp_7"     , SDLK_KP7         },
  { "kp_8"     , SDLK_KP8         },
  { "kp_9"     , SDLK_KP9         },
  { "kp_."     , SDLK_KP_PERIOD   },
  { "kp_/"     , SDLK_KP_DIVIDE   },
  { "kp_*"     , SDLK_KP_MULTIPLY },
  { "kp_-"     , SDLK_KP_MINUS    },
  { "kp_+"     , SDLK_KP_PLUS     },
  { "kp_enter" , SDLK_KP_ENTER    },
  { "kp_="     , SDLK_KP_EQUALS   },
  { "up"       , SDLK_UP          },
  { "down"     , SDLK_DOWN        },
  { "right"    , SDLK_RIGHT       },
  { "left"     , SDLK_LEFT        },
  { "insert"   , SDLK_INSERT      },
  { "home"     , SDLK_HOME        },
  { "end"      , SDLK_END         },
  { "pageup"   , SDLK_PAGEUP      },
  { "pagedown" , SDLK_PAGEDOWN    },
  { "f1"       , SDLK_F1          },
  { "f2"       , SDLK_F2          },
  { "f3"       , SDLK_F3          },
  { "f4"       , SDLK_F4          },
  { "f5"       , SDLK_F5          },
  { "f6"       , SDLK_F6          },
  { "f7"       , SDLK_F7          },
  { "f8"       , SDLK_F8          },
  { "f9"       , SDLK_F9          },
  { "f10"      , SDLK_F10         },
  { "f11"      , SDLK_F11         },
  { "f12"      , SDLK_F12         },
  { "f13"      , SDLK_F13         },
  { "f14"      , SDLK_F14         },
  { "f15"      , SDLK_F15         },
  { "numlock"  , SDLK_NUMLOCK     },
  { "capslock" , SDLK_CAPSLOCK    },
  { "scrollock", SDLK_SCROLLOCK   },
  { "rshift"   , SDLK_RSHIFT      },
  { "lshift"   , SDLK_LSHIFT      },
  { "rctrl"    , SDLK_RCTRL       },
  { "lctrl"    , SDLK_LCTRL       },
  { "ralt"     , SDLK_RALT        },
  { "lalt"     , SDLK_LALT        },
  { "rmeta"    , SDLK_RMETA       },
  { "lmeta"    , SDLK_LMETA       },
  { "lsuper"   , SDLK_LSUPER      },
  { "rsuper"   , SDLK_RSUPER      },
  { "mode"     , SDLK_MODE        },
  { "compose"  , SDLK_COMPOSE     },
  { "help"     , SDLK_HELP        },
  { "print"    , SDLK_PRINT       },
  { "sysreq"   , SDLK_SYSREQ      },
  { "break"    , SDLK_BREAK       },
  { "menu"     , SDLK_MENU        },
  { "power"    , SDLK_POWER       },
  { "euro"     , SDLK_EURO        },
  { "undo"     , SDLK_UNDO        }
 };

 #define ERROR_UNKNOWN_SETTING \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": unknown setting" << std::endl; \
   break; \
  }
 #define ERROR_MISSING_VALUE \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": missing value" << std::endl; \
   break; \
  }
 #define ERROR_NOT_VALID_BOOLEAN \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": not valid boolean" << std::endl; \
   break; \
  }
 #define ERROR_NOT_VALID_INTEGER \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": not valid integer" << std::endl; \
   break; \
  }
 #define ERROR_NOT_VALID_KEY \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": not valid key" << std::endl; \
   break; \
  }
 #define ERROR_NOT_VALID_STRING \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": not valid string" << std::endl; \
   break; \
  }
 #define ERROR_OUT_OF_RANGE \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": integer out of range" << std::endl; \
   break; \
  }
 #define ERROR_WRONG_NUMBER_OF_VALUES \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": wrong number of values" << std::endl; \
   break; \
  }
 #define ERROR_UNKNOWN_INPUT_KEYWORD \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": unknown input keyword" << std::endl; \
   break; \
  }
 #define ERROR_INCOMPL_INPUT_DEF \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": incomplete input definition" << std::endl; \
   break; \
  }
 #define ERROR_INPUT_REDEFINITION \
  { \
   rcode = false; \
   std::cout << "  error at line " << line_number << ": input redefinition" << std::endl; \
   break; \
  }

 bool load()
 {
  Input::to_undefined_values();
  bool rcode = true;
  std::ifstream config_file_in("PicchioEngine.cfg");
  if (!config_file_in)
  {
   std::cout << "  warning: unable to open the file 'PicchioEngine.cfg'" << std::endl;
   std::cout << " Creating a new configuration file..." << std::endl;
   std::ofstream config_file_out("PicchioEngine.cfg");
   if (!config_file_out)
   {
    std::cout << "  error: unable to create a new configuration file" << std::endl;
    return false;
   }
   Input::to_default_values();
   config_file_out << ";PicchioEngine Configuration File" << "\n"
                   << "\n"
                   << "console          auto      ;auto/pce/tg16" << "\n"
                   << "timer_phase      0         ;0~2" << "\n"
                   << "rgb_output       no        ;no/yes" << "\n"
                   << "controller_port  multitap  ;none/pcepad/turbopad/avenuepad6/multitap" << "\n"
                   << "multitap.port_1  turbopad  ;none/pcepad/turbopad/avenuepad6" << "\n"
                   << "multitap.port_2  turbopad  ;none/pcepad/turbopad/avenuepad6" << "\n"
                   << "multitap.port_3  turbopad  ;none/pcepad/turbopad/avenuepad6" << "\n"
                   << "multitap.port_4  turbopad  ;none/pcepad/turbopad/avenuepad6" << "\n"
                   << "multitap.port_5  turbopad  ;none/pcepad/turbopad/avenuepad6" << "\n"
                   << "audio_buffer     2048      ;1~3072" << "\n"
                   << "scale            2         ;1~6" << "\n"
                   << "overscan         no        ;no/yes" << "\n"
                   << "fullscreen       no        ;no/yes" << "\n"
                   << "stretch          no        ;no/yes" << "\n"
                   << "bilinear         no        ;no/yes" << "\n"
                   << "pause_key        p         ;key" << "\n"
                   << "exit_key         escape    ;key" << "\n"
                   << "analog_threshold 75        ;1~100" << "\n"
                   << std::endl;
   for (int i = 1; i <= 5; i++)
   {
    config_file_out << "controller " << i << "  pcepad      joystick " << (i - 1) << "\n"
                    << "controller " << i << "  turbopad    joystick " << (i - 1) << "\n"
                    << "controller " << i << "  avenuepad6  joystick " << (i - 1) << "\n"
                    << std::endl;
    if (i == 1)
    {
     config_file_out << "controller 1  pcepad.left    key left    axis 0 -" << "\n"
                     << "controller 1  pcepad.right   key right   axis 0 +" << "\n"
                     << "controller 1  pcepad.up      key up      axis 1 -" << "\n"
                     << "controller 1  pcepad.down    key down    axis 1 +" << "\n"
                     << "controller 1  pcepad.select  key space   button 2" << "\n"
                     << "controller 1  pcepad.run     key return  button 3" << "\n"
                     << "controller 1  pcepad.i       key d       button 1" << "\n"
                     << "controller 1  pcepad.ii      key s       button 0" << "\n"
                     << "\n"
                     << "controller 1  turbopad.left           key left    axis 0 -" << "\n"
                     << "controller 1  turbopad.right          key right   axis 0 +" << "\n"
                     << "controller 1  turbopad.up             key up      axis 1 -" << "\n"
                     << "controller 1  turbopad.down           key down    axis 1 +" << "\n"
                     << "controller 1  turbopad.select         key space   button 2" << "\n"
                     << "controller 1  turbopad.run            key return  button 3" << "\n"
                     << "controller 1  turbopad.i              key d       button 1" << "\n"
                     << "controller 1  turbopad.ii             key s       button 0" << "\n"
                     << "controller 1  turbopad.i_turbo_slow   key 4       button 7" << "\n"
                     << "controller 1  turbopad.i_turbo_fast   key 2       button 5" << "\n"
                     << "controller 1  turbopad.ii_turbo_slow  key 3       button 6" << "\n"
                     << "controller 1  turbopad.ii_turbo_fast  key 1       button 4" << "\n"
                     << "\n"
                     << "controller 1  avenuepad6.left      key left    axis 0 -" << "\n"
                     << "controller 1  avenuepad6.right     key right   axis 0 +" << "\n"
                     << "controller 1  avenuepad6.up        key up      axis 1 -" << "\n"
                     << "controller 1  avenuepad6.down      key down    axis 1 +" << "\n"
                     << "controller 1  avenuepad6.select    key space   button 6" << "\n"
                     << "controller 1  avenuepad6.run       key return  button 7" << "\n"
                     << "controller 1  avenuepad6.i         key d       button 1" << "\n"
                     << "controller 1  avenuepad6.ii        key s       button 0" << "\n"
                     << "controller 1  avenuepad6.iii       key a       button 2" << "\n"
                     << "controller 1  avenuepad6.iv        key q       button 3" << "\n"
                     << "controller 1  avenuepad6.v         key w       button 4" << "\n"
                     << "controller 1  avenuepad6.vi        key e       button 5" << "\n"
                     << "controller 1  avenuepad6.i_turbo   key 2       button 9" << "\n"
                     << "controller 1  avenuepad6.ii_turbo  key 1       button 8" << "\n"
                     << "controller 1  avenuepad6.mode      key tab     button 10" << "\n"
                     << "controller 1  avenuepad6.slow      key kp_0    button 11" << "\n"
                     << std::endl;
    }
    else
    {
     config_file_out << "controller " << i << "  pcepad.left    axis 0 -" << "\n"
                     << "controller " << i << "  pcepad.right   axis 0 +" << "\n"
                     << "controller " << i << "  pcepad.up      axis 1 -" << "\n"
                     << "controller " << i << "  pcepad.down    axis 1 +" << "\n"
                     << "controller " << i << "  pcepad.select  button 2" << "\n"
                     << "controller " << i << "  pcepad.run     button 3" << "\n"
                     << "controller " << i << "  pcepad.i       button 1" << "\n"
                     << "controller " << i << "  pcepad.ii      button 0" << "\n"
                     << "\n"
                     << "controller " << i << "  turbopad.left           axis 0 -" << "\n"
                     << "controller " << i << "  turbopad.right          axis 0 +" << "\n"
                     << "controller " << i << "  turbopad.up             axis 1 -" << "\n"
                     << "controller " << i << "  turbopad.down           axis 1 +" << "\n"
                     << "controller " << i << "  turbopad.select         button 2" << "\n"
                     << "controller " << i << "  turbopad.run            button 3" << "\n"
                     << "controller " << i << "  turbopad.i              button 1" << "\n"
                     << "controller " << i << "  turbopad.ii             button 0" << "\n"
                     << "controller " << i << "  turbopad.i_turbo_slow   button 7" << "\n"
                     << "controller " << i << "  turbopad.i_turbo_fast   button 5" << "\n"
                     << "controller " << i << "  turbopad.ii_turbo_slow  button 6" << "\n"
                     << "controller " << i << "  turbopad.ii_turbo_fast  button 4" << "\n"
                     << "\n"
                     << "controller " << i << "  avenuepad6.left      axis 0 -" << "\n"
                     << "controller " << i << "  avenuepad6.right     axis 0 +" << "\n"
                     << "controller " << i << "  avenuepad6.up        axis 1 -" << "\n"
                     << "controller " << i << "  avenuepad6.down      axis 1 +" << "\n"
                     << "controller " << i << "  avenuepad6.select    button 6" << "\n"
                     << "controller " << i << "  avenuepad6.run       button 7" << "\n"
                     << "controller " << i << "  avenuepad6.i         button 1" << "\n"
                     << "controller " << i << "  avenuepad6.ii        button 0" << "\n"
                     << "controller " << i << "  avenuepad6.iii       button 2" << "\n"
                     << "controller " << i << "  avenuepad6.iv        button 3" << "\n"
                     << "controller " << i << "  avenuepad6.v         button 4" << "\n"
                     << "controller " << i << "  avenuepad6.vi        button 5" << "\n"
                     << "controller " << i << "  avenuepad6.i_turbo   button 9" << "\n"
                     << "controller " << i << "  avenuepad6.ii_turbo  button 8" << "\n"
                     << "controller " << i << "  avenuepad6.mode      button 10" << "\n"
                     << "controller " << i << "  avenuepad6.slow      button 11" << "\n"
                     << std::endl;
    }
   }
   return true;
  }

  uint32 line_number = 0;
  std::string temp_str;
  std::string word;

  while (getline(config_file_in, temp_str) != 0)
  {
   line_number++;
   std::istringstream line(temp_str);
   if ((line >> word) == 0) continue;
   if (word[0] == ';') continue;

   to_lower_case(word);
   for (int i = 0; i < OPTIONS; i++)
   {
    if (word.compare(option[i].name) == 0)
    {
     switch (option[i].value_type)
     {
      case BOOLEAN:
      {
       if ((line >> word) == 0) ERROR_MISSING_VALUE
       if (word[0] == ';') ERROR_MISSING_VALUE
       to_lower_case(word);
            if (word.compare("no"   ) == 0) option[i].value = 0;
       else if (word.compare("yes"  ) == 0) option[i].value = 1;
       else if (word.compare("n"    ) == 0) option[i].value = 0;
       else if (word.compare("y"    ) == 0) option[i].value = 1;
       else if (word.compare("false") == 0) option[i].value = 0;
       else if (word.compare("true" ) == 0) option[i].value = 1;
       else if (word.compare("f"    ) == 0) option[i].value = 0;
       else if (word.compare("t"    ) == 0) option[i].value = 1;
       else ERROR_NOT_VALID_BOOLEAN
       break;
      }
      case INTEGER:
      {
       if ((line >> word) == 0) ERROR_MISSING_VALUE
       if (word[0] == ';') ERROR_MISSING_VALUE
       int32 value;
       if (str_to_int(word, value) == false) ERROR_NOT_VALID_INTEGER
       if ((value < option[i].min_value) || (value > option[i].max_value)) ERROR_OUT_OF_RANGE
       option[i].value = value;
       break;
      }
      case KEY:
      {
       if ((line >> word) == 0) ERROR_MISSING_VALUE
       if ((word.length() != 1) && (word[0] == ';')) ERROR_MISSING_VALUE
       to_lower_case(word);
       int value = key_to_int(word);
       if (value == 0) ERROR_NOT_VALID_KEY
       option[i].value = value;
       break;
      }
      case STRING:
      {
       if (word.compare("console") == 0)
       {
        if ((line >> word) == 0) ERROR_MISSING_VALUE
        if (word[0] == ';') ERROR_MISSING_VALUE
        to_lower_case(word);
             if (word.compare("auto") == 0) option[i].value = 0;
        else if (word.compare("pce" ) == 0) option[i].value = 1;
        else if (word.compare("tg16") == 0) option[i].value = 2;
        else ERROR_NOT_VALID_STRING
        break;
       }
       if (word.compare("controller_port") == 0)
       {
        if ((line >> word) == 0) ERROR_MISSING_VALUE
        if (word[0] == ';') ERROR_MISSING_VALUE
        to_lower_case(word);
             if (word.compare("none"      ) == 0) option[i].value = 0;
        else if (word.compare("pcepad"    ) == 0) option[i].value = 1;
        else if (word.compare("turbopad"  ) == 0) option[i].value = 2;
        else if (word.compare("avenuepad6") == 0) option[i].value = 3;
        else if (word.compare("multitap"  ) == 0) option[i].value = 4;
        else ERROR_NOT_VALID_STRING
        break;
       }
       if (word.compare("multitap.port_1") == 0)
       {
        if ((line >> word) == 0) ERROR_MISSING_VALUE
        if (word[0] == ';') ERROR_MISSING_VALUE
        to_lower_case(word);
             if (word.compare("none"      ) == 0) option[i].value = 0;
        else if (word.compare("pcepad"    ) == 0) option[i].value = 1;
        else if (word.compare("turbopad"  ) == 0) option[i].value = 2;
        else if (word.compare("avenuepad6") == 0) option[i].value = 3;
        else ERROR_NOT_VALID_STRING
        break;
       }
       if (word.compare("multitap.port_2") == 0)
       {
        if ((line >> word) == 0) ERROR_MISSING_VALUE
        if (word[0] == ';') ERROR_MISSING_VALUE
        to_lower_case(word);
             if (word.compare("none"      ) == 0) option[i].value = 0;
        else if (word.compare("pcepad"    ) == 0) option[i].value = 1;
        else if (word.compare("turbopad"  ) == 0) option[i].value = 2;
        else if (word.compare("avenuepad6") == 0) option[i].value = 3;
        else ERROR_NOT_VALID_STRING
        break;
       }
       if (word.compare("multitap.port_3") == 0)
       {
        if ((line >> word) == 0) ERROR_MISSING_VALUE
        if (word[0] == ';') ERROR_MISSING_VALUE
        to_lower_case(word);
             if (word.compare("none"      ) == 0) option[i].value = 0;
        else if (word.compare("pcepad"    ) == 0) option[i].value = 1;
        else if (word.compare("turbopad"  ) == 0) option[i].value = 2;
        else if (word.compare("avenuepad6") == 0) option[i].value = 3;
        else ERROR_NOT_VALID_STRING
        break;
       }
       if (word.compare("multitap.port_4") == 0)
       {
        if ((line >> word) == 0) ERROR_MISSING_VALUE
        if (word[0] == ';') ERROR_MISSING_VALUE
        to_lower_case(word);
             if (word.compare("none"      ) == 0) option[i].value = 0;
        else if (word.compare("pcepad"    ) == 0) option[i].value = 1;
        else if (word.compare("turbopad"  ) == 0) option[i].value = 2;
        else if (word.compare("avenuepad6") == 0) option[i].value = 3;
        else ERROR_NOT_VALID_STRING
        break;
       }
       if (word.compare("multitap.port_5") == 0)
       {
        if ((line >> word) == 0) ERROR_MISSING_VALUE
        if (word[0] == ';') ERROR_MISSING_VALUE
        to_lower_case(word);
             if (word.compare("none"      ) == 0) option[i].value = 0;
        else if (word.compare("pcepad"    ) == 0) option[i].value = 1;
        else if (word.compare("turbopad"  ) == 0) option[i].value = 2;
        else if (word.compare("avenuepad6") == 0) option[i].value = 3;
        else ERROR_NOT_VALID_STRING
        break;
       }
       break;
      }
     }
     if ((line >> word) != 0)
     {
      if (word[0] != ';') ERROR_WRONG_NUMBER_OF_VALUES
     }
     break;
    }
    if (i == (OPTIONS - 1))
    {
     if (word.compare("controller") != 0) ERROR_UNKNOWN_SETTING
     if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
     if (word[0] == ';') ERROR_INCOMPL_INPUT_DEF
     int32 port;
     if (str_to_int(word, port) == false) ERROR_NOT_VALID_INTEGER
     if ((port < 1) || (port > 5)) ERROR_OUT_OF_RANGE
     port--;
     if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
     if (word[0] == ';') ERROR_INCOMPL_INPUT_DEF
     to_lower_case(word);
     int pad_type;
     int index;
          if (word.compare("pcepad"       ) == 0) { pad_type = 0; index = -1; }
     else if (word.compare("pcepad.i"     ) == 0) { pad_type = 0; index =  0; }
     else if (word.compare("pcepad.ii"    ) == 0) { pad_type = 0; index =  1; }
     else if (word.compare("pcepad.select") == 0) { pad_type = 0; index =  2; }
     else if (word.compare("pcepad.run"   ) == 0) { pad_type = 0; index =  3; }
     else if (word.compare("pcepad.up"    ) == 0) { pad_type = 0; index =  4; }
     else if (word.compare("pcepad.right" ) == 0) { pad_type = 0; index =  5; }
     else if (word.compare("pcepad.down"  ) == 0) { pad_type = 0; index =  6; }
     else if (word.compare("pcepad.left"  ) == 0) { pad_type = 0; index =  7; }
     else if (word.compare("turbopad"              ) == 0) { pad_type = 1; index = -1; }
     else if (word.compare("turbopad.i"            ) == 0) { pad_type = 1; index =  0; }
     else if (word.compare("turbopad.ii"           ) == 0) { pad_type = 1; index =  1; }
     else if (word.compare("turbopad.select"       ) == 0) { pad_type = 1; index =  2; }
     else if (word.compare("turbopad.run"          ) == 0) { pad_type = 1; index =  3; }
     else if (word.compare("turbopad.up"           ) == 0) { pad_type = 1; index =  4; }
     else if (word.compare("turbopad.right"        ) == 0) { pad_type = 1; index =  5; }
     else if (word.compare("turbopad.down"         ) == 0) { pad_type = 1; index =  6; }
     else if (word.compare("turbopad.left"         ) == 0) { pad_type = 1; index =  7; }
     else if (word.compare("turbopad.i_turbo_slow" ) == 0) { pad_type = 1; index =  8; }
     else if (word.compare("turbopad.i_turbo_fast" ) == 0) { pad_type = 1; index =  9; }
     else if (word.compare("turbopad.ii_turbo_slow") == 0) { pad_type = 1; index = 10; }
     else if (word.compare("turbopad.ii_turbo_fast") == 0) { pad_type = 1; index = 11; }
     else if (word.compare("avenuepad6"         ) == 0) { pad_type = 2; index = -1; }
     else if (word.compare("avenuepad6.i"       ) == 0) { pad_type = 2; index =  0; }
     else if (word.compare("avenuepad6.ii"      ) == 0) { pad_type = 2; index =  1; }
     else if (word.compare("avenuepad6.select"  ) == 0) { pad_type = 2; index =  2; }
     else if (word.compare("avenuepad6.run"     ) == 0) { pad_type = 2; index =  3; }
     else if (word.compare("avenuepad6.up"      ) == 0) { pad_type = 2; index =  4; }
     else if (word.compare("avenuepad6.right"   ) == 0) { pad_type = 2; index =  5; }
     else if (word.compare("avenuepad6.down"    ) == 0) { pad_type = 2; index =  6; }
     else if (word.compare("avenuepad6.left"    ) == 0) { pad_type = 2; index =  7; }
     else if (word.compare("avenuepad6.iii"     ) == 0) { pad_type = 2; index =  8; }
     else if (word.compare("avenuepad6.iv"      ) == 0) { pad_type = 2; index =  9; }
     else if (word.compare("avenuepad6.v"       ) == 0) { pad_type = 2; index = 10; }
     else if (word.compare("avenuepad6.vi"      ) == 0) { pad_type = 2; index = 11; }
     else if (word.compare("avenuepad6.i_turbo" ) == 0) { pad_type = 2; index = 12; }
     else if (word.compare("avenuepad6.ii_turbo") == 0) { pad_type = 2; index = 13; }
     else if (word.compare("avenuepad6.mode"    ) == 0) { pad_type = 2; index = 14; }
     else if (word.compare("avenuepad6.slow"    ) == 0) { pad_type = 2; index = 15; }
     else ERROR_UNKNOWN_INPUT_KEYWORD
     if (index == -1)
     {
      if ((line >> word) == 0) break;
      if (word[0] == ';') break;
      to_lower_case(word);
      if (word.compare("joystick") != 0) ERROR_UNKNOWN_INPUT_KEYWORD
      if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
      if (word[0] == ';') ERROR_INCOMPL_INPUT_DEF
      int32 value;
      if (str_to_int(word, value) == false) ERROR_NOT_VALID_INTEGER
      if ((value < 0) || (value > 255)) ERROR_OUT_OF_RANGE
      switch (pad_type)
      {
       case 0: Input::PCEPad[port].joy_index = value; break;
       case 1: Input::TurboPad[port].joy_index = value; break;
       case 2: Input::AvenuePad6[port].joy_index = value; break;
      }
     }
     else
     {
      bool input_key = false;
      bool input_button = false;
      while (true)
      {
       if ((line >> word) == 0) break;
       if (word[0] == ';') break;
       to_lower_case(word);
       if (word.compare("key") == 0)
       {
        if (input_key) ERROR_INPUT_REDEFINITION
        input_key = true;
        if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
        if ((word.length() != 1) && (word[0] == ';')) ERROR_INCOMPL_INPUT_DEF
        to_lower_case(word);
        int value = key_to_int(word);
        if (value == 0) ERROR_NOT_VALID_KEY
        switch (pad_type)
        {
         case 0: Input::PCEPad[port].key[index] = value; break;
         case 1: Input::TurboPad[port].key[index] = value; break;
         case 2: Input::AvenuePad6[port].key[index] = value; break;
        }
       }
       else if (word.compare("button") == 0)
       {
        if (input_button) ERROR_INPUT_REDEFINITION
        input_button = true;
        if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
        if (word[0] == ';') ERROR_INCOMPL_INPUT_DEF
        int32 value;
        if (str_to_int(word, value) == false) ERROR_NOT_VALID_INTEGER
        if ((value < 0) || (value > 255)) ERROR_OUT_OF_RANGE
        switch (pad_type)
        {
         case 0: Input::PCEPad[port].joy_button[index] = Input::BUTTON | value; break;
         case 1: Input::TurboPad[port].joy_button[index] = Input::BUTTON | value; break;
         case 2: Input::AvenuePad6[port].joy_button[index] = Input::BUTTON | value; break;
        }
       }
       else if (word.compare("axis") == 0)
       {
        if (input_button) ERROR_INPUT_REDEFINITION
        input_button = true;
        int32 temp = Input::AXIS;
        if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
        if (word[0] == ';') ERROR_INCOMPL_INPUT_DEF
        int32 value;
        if (str_to_int(word, value) == false) ERROR_NOT_VALID_INTEGER
        if ((value < 0) || (value > 255)) ERROR_OUT_OF_RANGE
        temp |= (value << 8);
        if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
        if (word[0] == ';') ERROR_INCOMPL_INPUT_DEF
             if (word.compare("+"  ) == 0) temp |= Input::POS;
        else if (word.compare("-"  ) == 0) temp |= Input::NEG;
        else if (word.compare("p"  ) == 0) temp |= Input::POS;
        else if (word.compare("n"  ) == 0) temp |= Input::NEG;
        else if (word.compare("pos") == 0) temp |= Input::POS;
        else if (word.compare("neg") == 0) temp |= Input::NEG;
        else ERROR_NOT_VALID_STRING
        switch (pad_type)
        {
         case 0: Input::PCEPad[port].joy_button[index] = temp; break;
         case 1: Input::TurboPad[port].joy_button[index] = temp; break;
         case 2: Input::AvenuePad6[port].joy_button[index] = temp; break;
        }
       }
       else if (word.compare("hat") == 0)
       {
        if (input_button) ERROR_INPUT_REDEFINITION
        input_button = true;
        int32 temp = Input::HAT;
        if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
        if (word[0] == ';') ERROR_INCOMPL_INPUT_DEF
        int32 value;
        if (str_to_int(word, value) == false) ERROR_NOT_VALID_INTEGER
        if ((value < 0) || (value > 255)) ERROR_OUT_OF_RANGE
        temp |= (value << 8);
        if ((line >> word) == 0) ERROR_INCOMPL_INPUT_DEF
        if (word[0] == ';') ERROR_INCOMPL_INPUT_DEF
        to_lower_case(word);
             if (word.compare("up"   ) == 0) temp |= SDL_HAT_UP;
        else if (word.compare("right") == 0) temp |= SDL_HAT_RIGHT;
        else if (word.compare("down" ) == 0) temp |= SDL_HAT_DOWN;
        else if (word.compare("left" ) == 0) temp |= SDL_HAT_LEFT;
        else if (word.compare("u"    ) == 0) temp |= SDL_HAT_UP;
        else if (word.compare("r"    ) == 0) temp |= SDL_HAT_RIGHT;
        else if (word.compare("d"    ) == 0) temp |= SDL_HAT_DOWN;
        else if (word.compare("l"    ) == 0) temp |= SDL_HAT_LEFT;
        else ERROR_NOT_VALID_STRING
        switch (pad_type)
        {
         case 0: Input::PCEPad[port].joy_button[index] = temp; break;
         case 1: Input::TurboPad[port].joy_button[index] = temp; break;
         case 2: Input::AvenuePad6[port].joy_button[index] = temp; break;
        }
       }
       else ERROR_UNKNOWN_INPUT_KEYWORD
       if (input_key && input_button) break;
      }
     }
     if ((line >> word) != 0)
     {
      if (word[0] != ';') ERROR_WRONG_NUMBER_OF_VALUES
     }
    }
   }
  }
  return rcode;
 }

 int read_setting(const char* name)
 {
  for (int i = 0; i < OPTIONS; i++)
  {
   if (strcmp(option[i].name, name) == 0) return option[i].value;
  }
  std::cout << "INTERNAL ERROR: THE SETTING NAME '" << name << "' IS UNKNOWN" << std::endl;
  return 0;
 }

 void to_lower_case(std::string& str)
 {
  for (uint i = 0; i < str.length(); i++)
  {
   char c = str[i];
   str.replace(i, 1, 1, (char)tolower(c));
  }
 }

 int key_to_int(std::string& word)
 {
  if (word.length() == 1) return (uint8)word[0];
  for (int i = 0; i < FUNC_KEYS; i++)
  {
   if (word.compare(func_key[i].name) == 0) return func_key[i].code;
  }
  return 0;
 }

 bool str_to_int(std::string& word, int32& value)
 {
  errno = 0;
  char *endptr = 0;
  value = (int32)strtol(word.c_str(), &endptr, 10);
  if ((errno != 0) || (endptr[0] != 0) || (endptr == word.c_str()))
  {
   errno = 0;
   return false;
  }
  return true;
 }
}
