
/************************************

  PicchioEngine

  Copyright(c)2008 Emanuele Bettidi

************************************/

/* PSG.cpp */

/* HuC6280 - PSG (Programmable Sound Generator) */

#include <cmath>
#include "Types.h"
#include "Audio.h"
#include "CPU.h"
#include "PSG.h"

namespace PSG
{
 void reset();

 void fill_level_lut();  // fill the level look-up table
 void fill_noise_lut();  // fill the noise look-up table

 int32 clock;

 /* registers */
  // R0 - channel selecting register
  // R1 - main sound volume adjusting register
  // R2 - fine frequency adjusting register
  // R3 - rough frequency adjusting register
  // R4 - channel ON / sound volume register
  // R5 - left and right sound volume register
  // R6 - waveform / DDA register
  // R7 - noise enable / noise frequency register
  // R8 - low frequency oscillator (LFO) frequency register
  // R9 - low frequency oscillator (LFO) control register

 struct channel
 {
  uint16 freq;         // channel frequency
  int32 count;         // internal frequency counter
  uint8 ctrl;          // channel control
  uint8 on;            // channel enable
  uint8 dda;           // direct D/A mode
  uint8 wave[32];      // waveform data
  uint8 wave_index;    // waveform data index
  uint8 noise_enable;  // noise enable (channels 4 and 5 only)
  uint8 noise_freq;    // noise frequency (channels 4 and 5 only)
  int32 noise_count;   // internal noise frequency counter (channels 4 and 5 only)
  uint32 lfsr;         // linear feedback shift register (channels 4 and 5 only)
  uint8 dac;           // last value written in the DAC
  uint8 al;            // attenuator level
  uint8 lal;           // left attenuator level
  uint8 ral;           // right attenuator level
  uint8 L_vol;         // left volume
  uint8 R_vol;         // right volume
  int16 L_out;         // left output level
  int16 R_out;         // right output level
 };

 /* registers */
 uint8 sel;         // selected channel (0-5)
 uint8 lmal;        // left main attenuator level
 uint8 rmal;        // right main attenuator level
 // uint16 base_freq;  // ... (channel 0 only)
 // uint8 lfo_freq;    // LFO (low frequency ocillator) frequency
 // uint8 lfo_trig;    // LFO (low frequency ocillator) trigger
 // uint8 lfo_ctrl;    // LFO (low frequency ocillator) control
 // uint8 lfo_count;   // ...
 channel ch[6];     // ...

 int16 level_lut[92][32];
 uint8 noise_lut[0x1FFFF];  // 131071 elements

 void fill_level_lut()
 {
  // the operating level of every PSG channel is -20dBfs
  for (int32 i = 0; i < 92; i++)
  {
   for (int32 j = 0; j < 32; j++)
   {
    level_lut[91 - i][j] = (int16)floor((pow(2.0, -0.25 * i) * (j / 31.0) * 6553.5) + 0.5);
   }
  }
 }

 void fill_noise_lut()
 {
  // the noise generator is a 17-stage linear feedback shift register with taps 6(11) and 17(0)
  uint32 lfsr = 0x10000;  // NOT TESTED
  for (int32 i = 0; i < 131071; i++)
  {
   noise_lut[i] = (bool)(lfsr & 1) ? 0x1F : 0;
   lfsr = (lfsr >> 1) ^ (-(int32)(lfsr & 1) & 0x10020);
  }
 }

 void init()
 {
  clock = 0;
  fill_level_lut();
  fill_noise_lut();
  /* typical values at the power-up */
  sel = 3;
  lmal = 0;
  rmal = 0;
  // lfo_freq = 0;  // NOT TESTED
  // lfo_trig = 0;  // NOT TESTED
  // lfo_ctrl = 0;  // NOT TESTED
  // lfo_count = 1;  // ...
  for (int32 i = 0; i < 6; i++)
  {
   ch[i].freq = 0;
   ch[i].count = 0x1000;  // ...
   ch[i].ctrl = 0;  // DDA BIT TESTED BUT DOUBTS REMAIN
   ch[i].dda = 0;  // TESTED BUT DOUBTS REMAIN
   for (int n = 0; n < 32; n++) ch[i].wave[n] = 0;  // NOT TESTED
   ch[i].wave_index = 0;  // NOT TESTED
   ch[i].noise_enable = 0;
   ch[i].noise_freq = 0;  // NOT TESTED
   ch[i].noise_count = 1;  // ...
   ch[i].lfsr = 0x10000;  // NOT TESTED
   ch[i].dac = 0;  // NOT TESTED
   ch[i].al = 0;  // NOT TESTED BUT PROBABLE
   ch[i].lal = 0;
   ch[i].ral = 0;
   ch[i].L_vol = 0;
   ch[i].R_vol = 0;
   ch[i].L_out = 0;
   ch[i].R_out = 0;
  }
  reset();
 }

 void reset()
 {
  for (int i = 0; i < 6; i++) ch[i].on = 0;
 }

 void resync()
 {
  int32 run_time;
  run_time = clock - ((CPU::clock >> 1) | (CPU::clock & 0x80000000));
  // ALT: run_time = ((clock << 1) - (CPU::clock & ~1L)) >> 1;
  if (run_time != 0)
  {
   clock -= run_time;
   for (int i = 0; i < 6; i++)
   {
    int16 L_out;
    int16 R_out;
    if (ch[i].on == 0)
    {
     if ((ch[i].L_out != 0) && (ch[i].R_out != 0))
     {
      /* zero -> output */
      L_out = 0;
      R_out = 0;
      Audio::add_blep(clock + run_time - 1, L_out - ch[i].L_out, R_out - ch[i].R_out);
      ch[i].L_out = L_out;
      ch[i].R_out = R_out;
     }
     if (i >= 4)
     {
      /* noise generator -> no output */
      int32 noise_freq = 0x1F - ch[i].noise_freq;
      if (noise_freq == 0) { noise_freq = 0x20; } else { noise_freq <<= 6; }
      // noise_freq <<= 1;
      ch[i].noise_count -= run_time;
      while(ch[i].noise_count <= 0)
      {
       ch[i].lfsr = (ch[i].lfsr >> 1) ^ (-(int32)(ch[i].lfsr & 1) & 0x10020);
       ch[i].noise_count += noise_freq;
      }
     }
    }
    else
    {
     /* dac -> output */
     L_out = level_lut[ch[i].L_vol][ch[i].dac];
     R_out = level_lut[ch[i].R_vol][ch[i].dac];
     Audio::add_blep(clock + run_time - 1, L_out - ch[i].L_out, R_out - ch[i].R_out);
     ch[i].L_out = L_out;
     ch[i].R_out = R_out;
     if (ch[i].noise_enable == 0)
     {
      if (ch[i].dda == 0)
      {
       /* wave generator -> output */
       int32 freq;
       if (ch[i].freq != 0) { freq = ch[i].freq; } else { freq = 0x1000; }
       // freq <<= 1;
       ch[i].count -= run_time;
       while(ch[i].count <= 0)
       {
        ch[i].wave_index++;
        ch[i].wave_index &= 0x1F;
        ch[i].dac = ch[i].wave[ch[i].wave_index];  // prima o dopo dell'incremento dell'indice?
        L_out = level_lut[ch[i].L_vol][ch[i].dac];
        R_out = level_lut[ch[i].R_vol][ch[i].dac];
        Audio::add_blep(clock - ch[i].count - 1, L_out - ch[i].L_out, R_out - ch[i].R_out);
        ch[i].L_out = L_out;
        ch[i].R_out = R_out;
        ch[i].count += freq;
       }
      }
      else
      { 
       /* dac -> output */
       //...
      }
      if (i >= 4)
      {
       /* noise generator -> no output */
       int32 noise_freq = 0x1F - ch[i].noise_freq;
       if (noise_freq == 0) { noise_freq = 0x20; } else { noise_freq <<= 6; }
       // noise_freq <<= 1;
       ch[i].noise_count -= run_time;
       while(ch[i].noise_count <= 0)
       {
        ch[i].lfsr = (ch[i].lfsr >> 1) ^ (-(int32)(ch[i].lfsr & 1) & 0x10020);
        ch[i].noise_count += noise_freq;
       }
      }
     }
     else
     {
      /* noise generator -> output */
      int32 noise_freq = 0x1F - ch[i].noise_freq;
      if (noise_freq == 0) { noise_freq = 0x20; } else { noise_freq <<= 6; }
      // noise_freq <<= 1;
      ch[i].noise_count -= run_time;
      while(ch[i].noise_count <= 0)
      {
       ch[i].lfsr = (ch[i].lfsr >> 1) ^ (-(int32)(ch[i].lfsr & 1) & 0x10020);
       ch[i].dac = (bool)(ch[i].lfsr & 1) ? 0x1F : 0x00;
       L_out = level_lut[ch[i].L_vol][ch[i].dac];
       R_out = level_lut[ch[i].R_vol][ch[i].dac];
       Audio::add_blep(clock - ch[i].noise_count - 1, L_out - ch[i].L_out, R_out - ch[i].R_out);
       ch[i].L_out = L_out;
       ch[i].R_out = R_out;
       ch[i].noise_count += noise_freq;
      }
      /* wave generator -> no output */
      int32 freq;
      if (ch[i].freq != 0) { freq = ch[i].freq; } else { freq = 0x1000; }
      // freq <<= 1;
      ch[i].count -= run_time;
      while(ch[i].count <= 0)
      {
       ch[i].wave_index++;
       ch[i].wave_index &= 0x1F;
       ch[i].count += freq;
      }
     }
    }
   }
  }
 }

 void write(uint8 reg, uint8 data)
 {
  resync();
  switch (reg & 0x0F)
  {
   case 0x00: /* ch_SEL */
              sel = data & 0x07;
              break;
   case 0x01: /* LMAL, RMAL */
              lmal = data >> 4;
              rmal = data & 0x0F;
              for (int32 i = 0; i < 6; i++)
              {
               ch[i].L_vol = ch[i].al + ((ch[i].lal + lmal) << 1);
               ch[i].R_vol = ch[i].al + ((ch[i].ral + rmal) << 1);
              }
              break;
   case 0x02: /* FRQ_LOW */
              if (sel < 6)
              {
               ch[sel].freq &= 0x0F00;
               ch[sel].freq |= data;
              }
              break;
   case 0x03: /* FRQ_HIGH */
              if (sel < 6)
              {
               ch[sel].freq &= 0x00FF;
               ch[sel].freq |= ((data & 0x0F) << 8);
              }
              break;
   case 0x04: /* ch_ON, DDA, AL */
              if (sel < 6)
              {
               ch[sel].al = data & 0x1F;
               ch[sel].L_vol = ch[sel].al + ((ch[sel].lal + lmal) << 1);
               ch[sel].R_vol = ch[sel].al + ((ch[sel].ral + rmal) << 1);
               ch[sel].ctrl = data >> 6;
               ch[sel].on = data >> 7;
               ch[sel].dda = (data & 0x40) >> 6;
               if (ch[sel].dda == 0)
               {
                ch[sel].wave_index++;  // OK?
                ch[sel].wave_index &= 0x1F;  // OK?
                ch[sel].dac = ch[sel].wave[ch[sel].wave_index];  // OK?
                // ATTENZIONE: Possibile doppio avanzamento dell'indice!
               }
               else
               {
                ch[sel].wave_index = 0x1F;
               }
               if (ch[sel].freq != 0) ch[sel].count = ch[sel].freq; else ch[sel].count = 0x1000;
              }
              break;
   case 0x05: /* LAL, RAL */
              if (sel < 6)
              {
               ch[sel].lal = data >> 4;
               ch[sel].ral = data & 0x0F;
               ch[sel].L_vol = ch[sel].al + ((ch[sel].lal + lmal) << 1);
               ch[sel].R_vol = ch[sel].al + ((ch[sel].ral + rmal) << 1);
              }
              break;
   case 0x06: /* WAVE_DATA */
              if (sel < 6)
              {
               switch (ch[sel].ctrl)
               {
                case 0: /* ch_ON = 0, DDA = 0 */
                        ch[sel].wave_index++;
                        ch[sel].wave_index &= 0x1F;
                        ch[sel].wave[ch[sel].wave_index] = data & 0x1F;
                        ch[sel].dac = data & 0x1F;  // OK?
                        if (ch[sel].freq != 0) ch[sel].count = ch[sel].freq; else ch[sel].count = 0x1000;
                        break;
                case 1: /* ch_ON = 0, DDA = 1 */
                        ch[sel].dac = data & 0x1F;  // TESTED
                        break;
                case 2: /* ch_ON = 1, DDA = 0 */
                        ch[sel].wave_index++;  // OK?
                        ch[sel].wave_index &= 0x1F;  // OK?
                        ch[sel].wave[ch[sel].wave_index] = data & 0x1F;  // OK?
                        ch[sel].dac = data & 0x1F;  // OK?
                        if (ch[sel].freq != 0) ch[sel].count = ch[sel].freq; else ch[sel].count = 0x1000;
                        // ATTENZIONE: Possibile doppio avanzamento dell'indice!
                        break;
                case 3: /* ch_ON = 1, DDA = 1 */
                        ch[sel].dac = data & 0x1F;
                        break;
               }
              }
              break;
   case 0x07: /* NE, NOISE_FRQ */
              if ((sel == 4) || (sel == 5))
              {
               ch[sel].noise_enable = data >> 7;
               ch[sel].noise_freq = data & 0x1F;
              }
              break;
   case 0x08: /* LFO_FRQ */
              // lfo_freq = data;
              break;
   case 0x09: /* LF_TRG, LF_CTL */
              // lfo_trig = data >> 7;
              // lfo_ctrl = data & 0x03;
              // if (lfo_trig == 1) ch[1].wave_index = 0;
              break;
   case 0x0A: break;
   case 0x0B: break;
   case 0x0C: break;
   case 0x0D: break;
   case 0x0E: break;
   case 0x0F: break;
  }
 }
}
