
/************************************

  PicchioEngine

  Copyright(c)2008 Emanuele Bettidi

************************************/

/* Input.cpp */

/* Input */

#include <ios>
#include <iostream>
#include <SDL/SDL.h>
#include "Types.h"
#include "Config.h"
#include "IOPort.h"
#include "Video.h"
#include "MultiTap.h"
#include "Gamepads.h"
#include "Input.h"

namespace Input
{
 struct joy_t
 {
  int index;
  SDL_Joystick* pointer;
 };

 void translate_key(int code, int action);
 void translate_button(int joy_index, int32 code, int action);

 extern const int32 BUTTON = 0x00000;
 extern const int32 AXIS   = 0x10000;
 extern const int32 HAT    = 0x20000;

 extern const int32 POS = 0;
 extern const int32 NEG = 1;

 const int RELEASE = 1;
 const int PRESS   = 0;

 SDL_Event event;
 bool exit = false;
 bool pause = false;
 int exit_key;
 int pause_key;
 int32 analog_threshold;

 joy_t Joy[5];

 pcepad_t PCEPad[5];
 turbopad_t TurboPad[5];
 avenuepad6_t AvenuePad6[5];

 void init()
 {
  exit_key = Config::read_setting("exit_key");
  pause_key = Config::read_setting("pause_key");
  analog_threshold = (32767 * Config::read_setting("analog_threshold")) / 100;
  for (int i = 0; i < 5; i++)
  {
   Joy[i].index = -1;
   Joy[i].pointer = 0;
  }
  switch (IOPort::controller_port)
  {
   case 0: /* none       */                                         break;
   case 1: /* pcepad     */ Joy[0].index = PCEPad[0].joy_index;     break;
   case 2: /* turbopad   */ Joy[0].index = TurboPad[0].joy_index;   break;
   case 3: /* avenuepad6 */ Joy[0].index = AvenuePad6[0].joy_index; break;
   case 4: /* multitap   */
    for (int i = 0; i < 5; i++)
    {
     switch (MultiTap::Port[i])
     {
      case 0: /* none       */                                         break;
      case 1: /* pcepad     */ Joy[i].index = PCEPad[i].joy_index;      break;
      case 2: /* turbopad   */ Joy[i].index = TurboPad[i].joy_index;   break;
      case 3: /* avenuepad6 */ Joy[i].index = AvenuePad6[i].joy_index; break;
     }
     for (int j = 0; j < i; j++)
     {
      if (Joy[i].index == Joy[j].index) Joy[i].index = -1;
     }
    }
    break;
  }
  std::cout << " Initializing joysticks..." << std::endl;
  for (int i = 0; i < 5; i++)
  {
   if (Joy[i].index != -1)
   {
    if (SDL_NumJoysticks() > Joy[i].index)
    {
     Joy[i].pointer = SDL_JoystickOpen(Joy[i].index);
     if (Joy[i].pointer != 0)
     {
      std::cout << "  Joystick " << Joy[i].index << ": " << std::endl;
      std::cout << "   name: " << SDL_JoystickName(Joy[i].index) << std::endl;
      std::cout << "   number of axes: " << SDL_JoystickNumAxes(Joy[i].pointer) << std::endl;
      std::cout << "   number of hats: " << SDL_JoystickNumHats(Joy[i].pointer) << std::endl;
      std::cout << "   number of trackballs: " << SDL_JoystickNumBalls(Joy[i].pointer) << std::endl;
      std::cout << "   number of buttons: " << SDL_JoystickNumButtons(Joy[i].pointer) << std::endl;
     }
     else
     {
      std::cout << "   warning: couldn't open joystick " << Joy[i].index << std::endl;
     }
    }
    else
    {
     std::cout << "  Joystick " << Joy[i].index << " not detected" << std::endl;
    }
   }
  }
  std::cout << " Done" << "\n" << std::endl;
 }

 void deinit()
 {
  for (int i = 0; i < 5; i++)
  {
   if ((Joy[i].index != -1) && (Joy[i].pointer != 0)) SDL_JoystickClose(Joy[i].pointer);
  }
 }

 void to_undefined_values()
 {
  for (int i = 0; i < 5; i++)
  {
   for (int j = 0; j < 8; j++)
   {
    PCEPad[i].joy_index = -1;
    PCEPad[i].key[j] = SDLK_UNKNOWN;
    PCEPad[i].joy_button[j] = -1;
   }
   for (int j = 0; j < 12; j++)
   {
    TurboPad[i].joy_index = -1;
    TurboPad[i].key[j] = SDLK_UNKNOWN;
    TurboPad[i].joy_button[j] = -1;
   }
   for (int j = 0; j < 16; j++)
   {
    AvenuePad6[i].joy_index = -1;
    AvenuePad6[i].key[j] = SDLK_UNKNOWN;
    AvenuePad6[i].joy_button[j] = -1;
   }
  }
 }

 void to_default_values()
 {
  PCEPad[0].key[0] = SDLK_d;       // I
  PCEPad[0].key[1] = SDLK_s;       // II
  PCEPad[0].key[2] = SDLK_SPACE;   // Select
  PCEPad[0].key[3] = SDLK_RETURN;  // Run
  PCEPad[0].key[4] = SDLK_UP;      // Up
  PCEPad[0].key[5] = SDLK_RIGHT;   // Right
  PCEPad[0].key[6] = SDLK_DOWN;    // Down
  PCEPad[0].key[7] = SDLK_LEFT;    // Left

  TurboPad[0].key[ 0] = SDLK_d;       // I
  TurboPad[0].key[ 1] = SDLK_s;       // II
  TurboPad[0].key[ 2] = SDLK_SPACE;   // Select
  TurboPad[0].key[ 3] = SDLK_RETURN;  // Run
  TurboPad[0].key[ 4] = SDLK_UP;      // Up
  TurboPad[0].key[ 5] = SDLK_RIGHT;   // Right
  TurboPad[0].key[ 6] = SDLK_DOWN;    // Down
  TurboPad[0].key[ 7] = SDLK_LEFT;    // Left
  TurboPad[0].key[ 8] = SDLK_4;       // I_Turbo_Slow
  TurboPad[0].key[ 9] = SDLK_2;       // I_Turbo_Fast
  TurboPad[0].key[10] = SDLK_3;       // II_Turbo_Slow
  TurboPad[0].key[11] = SDLK_1;       // II_Turbo_Fast

  AvenuePad6[0].key[ 0] = SDLK_d;       // I
  AvenuePad6[0].key[ 1] = SDLK_s;       // II
  AvenuePad6[0].key[ 2] = SDLK_SPACE;   // Select
  AvenuePad6[0].key[ 3] = SDLK_RETURN;  // Run
  AvenuePad6[0].key[ 4] = SDLK_UP;      // Up
  AvenuePad6[0].key[ 5] = SDLK_RIGHT;   // Right
  AvenuePad6[0].key[ 6] = SDLK_DOWN;    // Down
  AvenuePad6[0].key[ 7] = SDLK_LEFT;    // Left
  AvenuePad6[0].key[ 8] = SDLK_a;       // III
  AvenuePad6[0].key[ 9] = SDLK_q;       // IV
  AvenuePad6[0].key[10] = SDLK_w;       // V
  AvenuePad6[0].key[11] = SDLK_e;       // VI
  AvenuePad6[0].key[12] = SDLK_2;       // I_Turbo
  AvenuePad6[0].key[13] = SDLK_1;       // II_Turbo
  AvenuePad6[0].key[14] = SDLK_TAB;     // Mode
  AvenuePad6[0].key[15] = SDLK_KP0;     // Slow

  for (int i = 0; i < 5; i++)
  {
   PCEPad[i].joy_index = i;
   PCEPad[i].joy_button[0] = BUTTON|1;        // I
   PCEPad[i].joy_button[1] = BUTTON|0;        // II
   PCEPad[i].joy_button[2] = BUTTON|2;        // Select
   PCEPad[i].joy_button[3] = BUTTON|3;        // Run
   PCEPad[i].joy_button[4] = AXIS|0x100|NEG;  // Up
   PCEPad[i].joy_button[5] = AXIS|0x000|POS;  // Right
   PCEPad[i].joy_button[6] = AXIS|0x100|POS;  // Down
   PCEPad[i].joy_button[7] = AXIS|0x000|NEG;  // Left

   TurboPad[i].joy_index = i;
   TurboPad[i].joy_button[ 0] = BUTTON|1;        // I
   TurboPad[i].joy_button[ 1] = BUTTON|0;        // II
   TurboPad[i].joy_button[ 2] = BUTTON|2;        // Select
   TurboPad[i].joy_button[ 3] = BUTTON|3;        // Run
   TurboPad[i].joy_button[ 4] = AXIS|0x100|NEG;  // Up
   TurboPad[i].joy_button[ 5] = AXIS|0x000|POS;  // Right
   TurboPad[i].joy_button[ 6] = AXIS|0x100|POS;  // Down
   TurboPad[i].joy_button[ 7] = AXIS|0x000|NEG;  // Left
   TurboPad[i].joy_button[ 8] = BUTTON|7;        // I_Turbo_Slow
   TurboPad[i].joy_button[ 9] = BUTTON|5;        // I_Turbo_Fast
   TurboPad[i].joy_button[10] = BUTTON|6;        // II_Turbo_Slow
   TurboPad[i].joy_button[11] = BUTTON|4;        // II_Turbo_Fast

   AvenuePad6[i].joy_index = i;
   AvenuePad6[i].joy_button[ 0] = BUTTON|1;        // I
   AvenuePad6[i].joy_button[ 1] = BUTTON|0;        // II
   AvenuePad6[i].joy_button[ 2] = BUTTON|6;        // Select
   AvenuePad6[i].joy_button[ 3] = BUTTON|7;        // Run
   AvenuePad6[i].joy_button[ 4] = AXIS|0x100|NEG;  // Up
   AvenuePad6[i].joy_button[ 5] = AXIS|0x000|POS;  // Right
   AvenuePad6[i].joy_button[ 6] = AXIS|0x100|POS;  // Down
   AvenuePad6[i].joy_button[ 7] = AXIS|0x000|NEG;  // Left
   AvenuePad6[i].joy_button[ 8] = BUTTON|2;        // III
   AvenuePad6[i].joy_button[ 9] = BUTTON|3;        // IV
   AvenuePad6[i].joy_button[10] = BUTTON|4;        // V
   AvenuePad6[i].joy_button[11] = BUTTON|5;        // VI
   AvenuePad6[i].joy_button[12] = BUTTON|9;        // I_Turbo
   AvenuePad6[i].joy_button[13] = BUTTON|8;        // II_Turbo
   AvenuePad6[i].joy_button[14] = BUTTON|10;       // Mode
   AvenuePad6[i].joy_button[15] = BUTTON|11;       // Slow
  }
 }

 void pause_loop()
 {
  SDL_WaitEvent(&event);
  if (event.type == SDL_QUIT) exit = true;
  if (event.type == SDL_KEYDOWN)
  {
   if (event.key.keysym.sym == exit_key) exit = true;
   if (event.key.keysym.sym == pause_key) pause = !pause;
  }
  if (event.type == SDL_VIDEOEXPOSE) Video::redraw();
 }

 void update_events()
 {
  while(SDL_PollEvent(&event))
  {
   switch (event.type)
   {
    case SDL_ACTIVEEVENT: break;
    case SDL_VIDEORESIZE: break;
    case SDL_VIDEOEXPOSE: break;
    case SDL_KEYDOWN:
     if (event.key.keysym.sym == exit_key) { exit = true; break; }
     if (event.key.keysym.sym == pause_key) { pause = !pause; break; }
     translate_key(event.key.keysym.sym, PRESS);
     break;
    case SDL_KEYUP:
     translate_key(event.key.keysym.sym, RELEASE);
     break;
    case SDL_JOYBUTTONDOWN:
     translate_button(event.jbutton.which, BUTTON | event.jbutton.button, PRESS);
     break;
    case SDL_JOYBUTTONUP:
     translate_button(event.jbutton.which, BUTTON | event.jbutton.button, RELEASE);
     break;
    case SDL_JOYAXISMOTION:
     if (event.jaxis.value >= analog_threshold)
     {
      translate_button(event.jaxis.which, AXIS | (event.jaxis.axis << 8) | NEG, RELEASE);
      translate_button(event.jaxis.which, AXIS | (event.jaxis.axis << 8) | POS, PRESS);
      break;
     }
     if (event.jaxis.value <= -analog_threshold)
     {
      translate_button(event.jaxis.which, AXIS | (event.jaxis.axis << 8) | POS, RELEASE);
      translate_button(event.jaxis.which, AXIS | (event.jaxis.axis << 8) | NEG, PRESS);
      break;
     }
     translate_button(event.jaxis.which, AXIS | (event.jaxis.axis << 8) | POS, RELEASE);
     translate_button(event.jaxis.which, AXIS | (event.jaxis.axis << 8) | NEG, RELEASE);
     break;
    case SDL_JOYHATMOTION:
     if (event.jhat.value == SDL_HAT_CENTERED)
     {
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_UP,    RELEASE);
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_RIGHT, RELEASE);
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_DOWN,  RELEASE);
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_LEFT,  RELEASE);
      break;
     }
     if ((event.jhat.value & SDL_HAT_UP) != 0)
     {
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_DOWN, RELEASE);
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_UP,   PRESS);
     }
     if ((event.jhat.value & SDL_HAT_RIGHT) != 0)
     {
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_LEFT,  RELEASE);
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_RIGHT, PRESS);
     }
     if ((event.jhat.value & SDL_HAT_DOWN) != 0)
     {
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_UP,   RELEASE);
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_DOWN, PRESS);
     }
     if ((event.jhat.value & SDL_HAT_LEFT) != 0)
     {
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_RIGHT, RELEASE);
      translate_button(event.jhat.which, HAT | (event.jhat.hat << 8) | SDL_HAT_LEFT,  PRESS);
     }
     break;
    case SDL_QUIT: exit = true; break;
   }
  }
 }

 void translate_key(int code, int action)
 {
  if (IOPort::controller_port == 0) return;
  for (int i = 0; i < 5; i++)
  {
   int pad_type;
   if (IOPort::controller_port != 4)
    pad_type = IOPort::controller_port;
   else
    pad_type = MultiTap::Port[i];
   switch (pad_type)
   {
    case 0: /* none */
     break;
    case 1: /* pcepad */
     if (action == RELEASE)
     {
      if (PCEPad[i].key[0] == code) { Gamepad[i].buttons |= 0x01; return; }  // I
      if (PCEPad[i].key[1] == code) { Gamepad[i].buttons |= 0x02; return; }  // II
      if (PCEPad[i].key[2] == code) { Gamepad[i].buttons |= 0x04; return; }  // Select
      if (PCEPad[i].key[3] == code) { Gamepad[i].buttons |= 0x08; return; }  // Run
      if (PCEPad[i].key[4] == code) { Gamepad[i].buttons |= 0x10; return; }  // Up
      if (PCEPad[i].key[5] == code) { Gamepad[i].buttons |= 0x20; return; }  // Right
      if (PCEPad[i].key[6] == code) { Gamepad[i].buttons |= 0x40; return; }  // Down
      if (PCEPad[i].key[7] == code) { Gamepad[i].buttons |= 0x80; return; }  // Left
     }
     else  // PRESS
     {
      if (PCEPad[i].key[0] == code) { Gamepad[i].buttons &= ~0x01;                             return; }  // I
      if (PCEPad[i].key[1] == code) { Gamepad[i].buttons &= ~0x02;                             return; }  // II
      if (PCEPad[i].key[2] == code) { Gamepad[i].buttons &= ~0x04;                             return; }  // Select
      if (PCEPad[i].key[3] == code) { Gamepad[i].buttons &= ~0x08;                             return; }  // Run
      if (PCEPad[i].key[4] == code) { Gamepad[i].buttons &= ~0x10; Gamepad[i].buttons |= 0x40; return; }  // Up
      if (PCEPad[i].key[5] == code) { Gamepad[i].buttons &= ~0x20; Gamepad[i].buttons |= 0x80; return; }  // Right
      if (PCEPad[i].key[6] == code) { Gamepad[i].buttons &= ~0x40; Gamepad[i].buttons |= 0x10; return; }  // Down
      if (PCEPad[i].key[7] == code) { Gamepad[i].buttons &= ~0x80; Gamepad[i].buttons |= 0x20; return; }  // Left
     }
     break;
    case 2: /* turbopad */
     if (action == RELEASE)
     {
      if (TurboPad[i].key[0] == code) { Gamepad[i].buttons |= 0x01; return; }  // I
      if (TurboPad[i].key[1] == code) { Gamepad[i].buttons |= 0x02; return; }  // II
      if (TurboPad[i].key[2] == code) { Gamepad[i].buttons |= 0x04; return; }  // Select
      if (TurboPad[i].key[3] == code) { Gamepad[i].buttons |= 0x08; return; }  // Run
      if (TurboPad[i].key[4] == code) { Gamepad[i].buttons |= 0x10; return; }  // Up
      if (TurboPad[i].key[5] == code) { Gamepad[i].buttons |= 0x20; return; }  // Right
      if (TurboPad[i].key[6] == code) { Gamepad[i].buttons |= 0x40; return; }  // Down
      if (TurboPad[i].key[7] == code) { Gamepad[i].buttons |= 0x80; return; }  // Left
     }
     else  // PRESS
     {
      if (TurboPad[i].key[ 0] == code) { Gamepad[i].buttons &= ~0x01;                              return; }  // I
      if (TurboPad[i].key[ 1] == code) { Gamepad[i].buttons &= ~0x02;                              return; }  // II
      if (TurboPad[i].key[ 2] == code) { Gamepad[i].buttons &= ~0x04;                              return; }  // Select
      if (TurboPad[i].key[ 3] == code) { Gamepad[i].buttons &= ~0x08;                              return; }  // Run
      if (TurboPad[i].key[ 4] == code) { Gamepad[i].buttons &= ~0x10; Gamepad[i].buttons |= 0x40;  return; }  // Up
      if (TurboPad[i].key[ 5] == code) { Gamepad[i].buttons &= ~0x20; Gamepad[i].buttons |= 0x80;  return; }  // Right
      if (TurboPad[i].key[ 6] == code) { Gamepad[i].buttons &= ~0x40; Gamepad[i].buttons |= 0x10;  return; }  // Down
      if (TurboPad[i].key[ 7] == code) { Gamepad[i].buttons &= ~0x80; Gamepad[i].buttons |= 0x20;  return; }  // Left
      if (TurboPad[i].key[ 8] == code) { Gamepad[i].i_turbo  = (Gamepad[i].i_turbo  != 4) ? 4 : 0; return; }  // I Turbo Slow
      if (TurboPad[i].key[ 9] == code) { Gamepad[i].i_turbo  = (Gamepad[i].i_turbo  != 2) ? 2 : 0; return; }  // I Turbo Fast
      if (TurboPad[i].key[10] == code) { Gamepad[i].ii_turbo = (Gamepad[i].ii_turbo != 4) ? 4 : 0; return; }  // II Turbo Slow
      if (TurboPad[i].key[11] == code) { Gamepad[i].ii_turbo = (Gamepad[i].ii_turbo != 2) ? 2 : 0; return; }  // II Turbo Fast
     }
     break;
    case 3: /* avenuepad6 */
     if (action == RELEASE)
     {
      if (AvenuePad6[i].key[ 0] == code) { Gamepad[i].buttons |= 0x001; return; }  // I
      if (AvenuePad6[i].key[ 1] == code) { Gamepad[i].buttons |= 0x002; return; }  // II
      if (AvenuePad6[i].key[ 2] == code) { Gamepad[i].buttons |= 0x004; return; }  // Select
      if (AvenuePad6[i].key[ 3] == code) { Gamepad[i].buttons |= 0x008; return; }  // Run
      if (AvenuePad6[i].key[ 4] == code) { Gamepad[i].buttons |= 0x010; return; }  // Up
      if (AvenuePad6[i].key[ 5] == code) { Gamepad[i].buttons |= 0x020; return; }  // Right
      if (AvenuePad6[i].key[ 6] == code) { Gamepad[i].buttons |= 0x040; return; }  // Down
      if (AvenuePad6[i].key[ 7] == code) { Gamepad[i].buttons |= 0x080; return; }  // Left
      if (AvenuePad6[i].key[ 8] == code) { Gamepad[i].buttons |= 0x100; return; }  // III
      if (AvenuePad6[i].key[ 9] == code) { Gamepad[i].buttons |= 0x200; return; }  // IV
      if (AvenuePad6[i].key[10] == code) { Gamepad[i].buttons |= 0x400; return; }  // V
      if (AvenuePad6[i].key[11] == code) { Gamepad[i].buttons |= 0x800; return; }  // VI
     }
     else  // PRESS
     {
      if (AvenuePad6[i].key[ 0] == code) { Gamepad[i].buttons &= ~0x001;                              return; }  // I
      if (AvenuePad6[i].key[ 1] == code) { Gamepad[i].buttons &= ~0x002;                              return; }  // II
      if (AvenuePad6[i].key[ 2] == code) { Gamepad[i].buttons &= ~0x004;                              return; }  // Select
      if (AvenuePad6[i].key[ 3] == code) { Gamepad[i].buttons &= ~0x008;                              return; }  // Run
      if (AvenuePad6[i].key[ 4] == code) { Gamepad[i].buttons &= ~0x010; Gamepad[i].buttons |= 0x040; return; }  // Up
      if (AvenuePad6[i].key[ 5] == code) { Gamepad[i].buttons &= ~0x020; Gamepad[i].buttons |= 0x080; return; }  // Right
      if (AvenuePad6[i].key[ 6] == code) { Gamepad[i].buttons &= ~0x040; Gamepad[i].buttons |= 0x010; return; }  // Down
      if (AvenuePad6[i].key[ 7] == code) { Gamepad[i].buttons &= ~0x080; Gamepad[i].buttons |= 0x020; return; }  // Left
      if (AvenuePad6[i].key[ 8] == code) { Gamepad[i].buttons &= ~0x100;                              return; }  // III
      if (AvenuePad6[i].key[ 9] == code) { Gamepad[i].buttons &= ~0x200;                              return; }  // IV
      if (AvenuePad6[i].key[10] == code) { Gamepad[i].buttons &= ~0x400;                              return; }  // V
      if (AvenuePad6[i].key[11] == code) { Gamepad[i].buttons &= ~0x800;                              return; }  // VI
      if (AvenuePad6[i].key[12] == code) { Gamepad[i].i_turbo  = (Gamepad[i].i_turbo  != 2) ? 2 : 0;  return; }  // I Turbo
      if (AvenuePad6[i].key[13] == code) { Gamepad[i].ii_turbo = (Gamepad[i].ii_turbo != 2) ? 2 : 0;  return; }  // II Turbo
      if (AvenuePad6[i].key[14] == code) { Gamepad[i].mode = !Gamepad[i].mode;                        return; }  // Mode
      if (AvenuePad6[i].key[15] == code) { Gamepad[i].slow = !Gamepad[i].slow;                        return; }  // Slow

     }
     break;
   }
   if (IOPort::controller_port != 4) return;
  }
 }

 void translate_button(int joy_index, int32 code, int action)
 {
  if (IOPort::controller_port == 0) return;
  for (int i = 0; i < 5; i++)
  {
   int pad_type;
   if (IOPort::controller_port != 4)
    pad_type = IOPort::controller_port;
   else
    pad_type = MultiTap::Port[i];
   switch (pad_type)
   {
    case 0: /* none */
     break;
    case 1: /* pcepad */
     if (PCEPad[i].joy_index != joy_index) break;
     if (action == RELEASE)
     {
      if (PCEPad[i].joy_button[0] == code) { Gamepad[i].buttons |= 0x01; return; }  // I
      if (PCEPad[i].joy_button[1] == code) { Gamepad[i].buttons |= 0x02; return; }  // II
      if (PCEPad[i].joy_button[2] == code) { Gamepad[i].buttons |= 0x04; return; }  // Select
      if (PCEPad[i].joy_button[3] == code) { Gamepad[i].buttons |= 0x08; return; }  // Run
      if (PCEPad[i].joy_button[4] == code) { Gamepad[i].buttons |= 0x10; return; }  // Up
      if (PCEPad[i].joy_button[5] == code) { Gamepad[i].buttons |= 0x20; return; }  // Right
      if (PCEPad[i].joy_button[6] == code) { Gamepad[i].buttons |= 0x40; return; }  // Down
      if (PCEPad[i].joy_button[7] == code) { Gamepad[i].buttons |= 0x80; return; }  // Left
     }
     else  // PRESS
     {
      if (PCEPad[i].joy_button[0] == code) { Gamepad[i].buttons &= ~0x01;                             return; }  // I
      if (PCEPad[i].joy_button[1] == code) { Gamepad[i].buttons &= ~0x02;                             return; }  // II
      if (PCEPad[i].joy_button[2] == code) { Gamepad[i].buttons &= ~0x04;                             return; }  // Select
      if (PCEPad[i].joy_button[3] == code) { Gamepad[i].buttons &= ~0x08;                             return; }  // Run
      if (PCEPad[i].joy_button[4] == code) { Gamepad[i].buttons &= ~0x10; Gamepad[i].buttons |= 0x40; return; }  // Up
      if (PCEPad[i].joy_button[5] == code) { Gamepad[i].buttons &= ~0x20; Gamepad[i].buttons |= 0x80; return; }  // Right
      if (PCEPad[i].joy_button[6] == code) { Gamepad[i].buttons &= ~0x40; Gamepad[i].buttons |= 0x10; return; }  // Down
      if (PCEPad[i].joy_button[7] == code) { Gamepad[i].buttons &= ~0x80; Gamepad[i].buttons |= 0x20; return; }  // Left
     }
     break;
    case 2: /* turbopad */
     if (TurboPad[i].joy_index != joy_index) break;
     if (action == RELEASE)
     {
      if (TurboPad[i].joy_button[0] == code) { Gamepad[i].buttons |= 0x01; return; }  // I
      if (TurboPad[i].joy_button[1] == code) { Gamepad[i].buttons |= 0x02; return; }  // II
      if (TurboPad[i].joy_button[2] == code) { Gamepad[i].buttons |= 0x04; return; }  // Select
      if (TurboPad[i].joy_button[3] == code) { Gamepad[i].buttons |= 0x08; return; }  // Run
      if (TurboPad[i].joy_button[4] == code) { Gamepad[i].buttons |= 0x10; return; }  // Up
      if (TurboPad[i].joy_button[5] == code) { Gamepad[i].buttons |= 0x20; return; }  // Right
      if (TurboPad[i].joy_button[6] == code) { Gamepad[i].buttons |= 0x40; return; }  // Down
      if (TurboPad[i].joy_button[7] == code) { Gamepad[i].buttons |= 0x80; return; }  // Left
     }
     else  // PRESS
     {
      if (TurboPad[i].joy_button[ 0] == code) { Gamepad[i].buttons &= ~0x01;                              return; }  // I
      if (TurboPad[i].joy_button[ 1] == code) { Gamepad[i].buttons &= ~0x02;                              return; }  // II
      if (TurboPad[i].joy_button[ 2] == code) { Gamepad[i].buttons &= ~0x04;                              return; }  // Select
      if (TurboPad[i].joy_button[ 3] == code) { Gamepad[i].buttons &= ~0x08;                              return; }  // Run
      if (TurboPad[i].joy_button[ 4] == code) { Gamepad[i].buttons &= ~0x10; Gamepad[i].buttons |= 0x40;  return; }  // Up
      if (TurboPad[i].joy_button[ 5] == code) { Gamepad[i].buttons &= ~0x20; Gamepad[i].buttons |= 0x80;  return; }  // Right
      if (TurboPad[i].joy_button[ 6] == code) { Gamepad[i].buttons &= ~0x40; Gamepad[i].buttons |= 0x10;  return; }  // Down
      if (TurboPad[i].joy_button[ 7] == code) { Gamepad[i].buttons &= ~0x80; Gamepad[i].buttons |= 0x20;  return; }  // Left
      if (TurboPad[i].joy_button[ 8] == code) { Gamepad[i].i_turbo  = (Gamepad[i].i_turbo  != 4) ? 4 : 0; return; }  // I Turbo Slow
      if (TurboPad[i].joy_button[ 9] == code) { Gamepad[i].i_turbo  = (Gamepad[i].i_turbo  != 2) ? 2 : 0; return; }  // I Turbo Fast
      if (TurboPad[i].joy_button[10] == code) { Gamepad[i].ii_turbo = (Gamepad[i].ii_turbo != 4) ? 4 : 0; return; }  // II Turbo Slow
      if (TurboPad[i].joy_button[11] == code) { Gamepad[i].ii_turbo = (Gamepad[i].ii_turbo != 2) ? 2 : 0; return; }  // II Turbo Fast
     }
     break;
    case 3: /* avenuepad6 */
     if (AvenuePad6[i].joy_index != joy_index) break;
     if (action == RELEASE)
     {
      if (AvenuePad6[i].joy_button[ 0] == code) { Gamepad[i].buttons |= 0x001; return; }  // I
      if (AvenuePad6[i].joy_button[ 1] == code) { Gamepad[i].buttons |= 0x002; return; }  // II
      if (AvenuePad6[i].joy_button[ 2] == code) { Gamepad[i].buttons |= 0x004; return; }  // Select
      if (AvenuePad6[i].joy_button[ 3] == code) { Gamepad[i].buttons |= 0x008; return; }  // Run
      if (AvenuePad6[i].joy_button[ 4] == code) { Gamepad[i].buttons |= 0x010; return; }  // Up
      if (AvenuePad6[i].joy_button[ 5] == code) { Gamepad[i].buttons |= 0x020; return; }  // Right
      if (AvenuePad6[i].joy_button[ 6] == code) { Gamepad[i].buttons |= 0x040; return; }  // Down
      if (AvenuePad6[i].joy_button[ 7] == code) { Gamepad[i].buttons |= 0x080; return; }  // Left
      if (AvenuePad6[i].joy_button[ 8] == code) { Gamepad[i].buttons |= 0x100; return; }  // III
      if (AvenuePad6[i].joy_button[ 9] == code) { Gamepad[i].buttons |= 0x200; return; }  // IV
      if (AvenuePad6[i].joy_button[10] == code) { Gamepad[i].buttons |= 0x400; return; }  // V
      if (AvenuePad6[i].joy_button[11] == code) { Gamepad[i].buttons |= 0x800; return; }  // VI
     }
     else  // PRESS
     {
      if (AvenuePad6[i].joy_button[ 0] == code) { Gamepad[i].buttons &= ~0x001;                              return; }  // I
      if (AvenuePad6[i].joy_button[ 1] == code) { Gamepad[i].buttons &= ~0x002;                              return; }  // II
      if (AvenuePad6[i].joy_button[ 2] == code) { Gamepad[i].buttons &= ~0x004;                              return; }  // Select
      if (AvenuePad6[i].joy_button[ 3] == code) { Gamepad[i].buttons &= ~0x008;                              return; }  // Run
      if (AvenuePad6[i].joy_button[ 4] == code) { Gamepad[i].buttons &= ~0x010; Gamepad[i].buttons |= 0x040; return; }  // Up
      if (AvenuePad6[i].joy_button[ 5] == code) { Gamepad[i].buttons &= ~0x020; Gamepad[i].buttons |= 0x080; return; }  // Right
      if (AvenuePad6[i].joy_button[ 6] == code) { Gamepad[i].buttons &= ~0x040; Gamepad[i].buttons |= 0x010; return; }  // Down
      if (AvenuePad6[i].joy_button[ 7] == code) { Gamepad[i].buttons &= ~0x080; Gamepad[i].buttons |= 0x020; return; }  // Left
      if (AvenuePad6[i].joy_button[ 8] == code) { Gamepad[i].buttons &= ~0x100;                              return; }  // III
      if (AvenuePad6[i].joy_button[ 9] == code) { Gamepad[i].buttons &= ~0x200;                              return; }  // IV
      if (AvenuePad6[i].joy_button[10] == code) { Gamepad[i].buttons &= ~0x400;                              return; }  // V
      if (AvenuePad6[i].joy_button[11] == code) { Gamepad[i].buttons &= ~0x800;                              return; }  // VI
      if (AvenuePad6[i].joy_button[12] == code) { Gamepad[i].i_turbo  = (Gamepad[i].i_turbo  != 2) ? 2 : 0;  return; }  // I Turbo
      if (AvenuePad6[i].joy_button[13] == code) { Gamepad[i].ii_turbo = (Gamepad[i].ii_turbo != 2) ? 2 : 0;  return; }  // II Turbo
      if (AvenuePad6[i].joy_button[14] == code) { Gamepad[i].mode = !Gamepad[i].mode;                        return; }  // Mode
      if (AvenuePad6[i].joy_button[15] == code) { Gamepad[i].slow = !Gamepad[i].slow;                        return; }  // Slow
     }
     break;
   }
   if (IOPort::controller_port != 4) return;
  }
 }
}
