
/************************************

  PicchioEngine

  Copyright(c)2008 Emanuele Bettidi

************************************/

/* Main.cpp */

// #include <windows.h>
#include <cstring>
#include <fstream>
#include <ios>
#include <iostream>
// #include <cstdio>
// #include <cstdlib>
#include <SDL/SDL.h>
#include <SDL/SDL_opengl.h>
#include "Types.h"
#include "Config.h"
#include "Input.h"
#include "IOPort.h"
#include "MultiTap.h"
#include "Gamepads.h"
#include "Audio.h"
#include "Video.h"
#include "PCE.h"

char *filename = 0;
char VideoDriverName[256];
char AudioDriverName[256];

int main(int argc, char* argv[])
{
 std::cout << "Start PicchioEngine v0.0.8" << "\n" << std::endl;

 std::cout << " Loading configuration file..." << std::endl;
 if (Config::load() == false)
 {
  std::cout << "\n" << "->Exit" << std::endl;
  return 1;
 }
 std::cout << " Done" << "\n" << std::endl;

 std::cout << " Reading CLI arguments..." << std::endl;
 if (argc == 1)
 {
  std::cout << "  error: rom image file not specified" << std::endl;
  std::cout << "\n" << "->Exit" << std::endl;
  return 1;
 }
 if (argc != 2)
 {
  std::cout << "  error: wrong number of arguments" << std::endl;
  std::cout << "\n" << "->Exit" << std::endl;
  return 1;
 }
 filename = argv[1];
 if (Config::read_setting("console") == 0)
 {
  PCE::console = 1;
  if (strstr(filename, "(U)") != 0) PCE::console = 2;
 }
 std::cout << " Done" << "\n" << std::endl;

 /* load the rom image */
 std::cout << " Loading ROM image file..." << std::endl;
 std::ifstream HuCard(filename, std::ios::binary);
 if (!HuCard)
 {
  std::cout << "  error: unable to open the file '" << filename << "'" << std::endl;
  return 1;
 }
 for (int32 i = 0; i < 0x280000; i++) PCE::rom[i] = 0xFF;
 HuCard.seekg(0, std::ios::end);
 std::ifstream::pos_type HuCardSize = HuCard.tellg();
 std::ifstream::pos_type header_size = HuCardSize % 1024;
 HuCardSize -= header_size;
 if (HuCardSize < 0x400)
 {
  std::cout << "  error: the rom size is smaller than 1kB" << std::endl;
  return 1;
 }
 if (HuCardSize > 0x280000)
 {
  std::cout << "  error: the rom size is greater than 2.5MB" << std::endl;
  return 1;
 }
 HuCard.seekg(header_size, std::ios::beg);
 HuCard.read((char*)PCE::rom, HuCardSize);
 // HuCard.clear();
 HuCard.close();
 if (HuCardSize == (std::ifstream::pos_type)0x60000)
 {
  for (int32 i = 0; i < 0x20000; i++) PCE::rom[i + 0x80000] = PCE::rom[i + 0x40000];
 }
 if ((strstr(filename, "Populous (J)") != 0) ) PCE::pop_card = true;
 if (HuCardSize > (std::ifstream::pos_type)0x100000) PCE::sf2_card = true;
 std::cout << "  file name: " << filename << std::endl;
 std::cout << "  header:    " << (header_size ? "yes" : "no") << std::endl;
 std::cout << "  rom size:  " << (HuCardSize / 1024) << "kB" << std::endl;
 std::cout << " Done" << "\n" << std::endl;

 /* SDL SYSTEM INIT */
 std::cout << " Initializing SDL..." << std::endl;
 if (SDL_Init(SDL_INIT_TIMER | SDL_INIT_AUDIO | SDL_INIT_VIDEO | SDL_INIT_JOYSTICK) == -1)
 {
  std::cout << "  error: " << SDL_GetError() << std::endl;
  std::cout << "\n" << "->Exit" << std::endl;
  return 1;
 }
 SDL_version ver;
 SDL_VERSION(&ver);
 std::cout << "  compile-time version: "
           << (uint)ver.major << "." << (uint)ver.minor << "." << (uint)ver.patch << std::endl;
 ver = *SDL_Linked_Version();
 std::cout << "  link-time version:    "
           << (uint)ver.major << "." << (uint)ver.minor << "." << (uint)ver.patch << std::endl;
 std::cout << " Done" << "\n" << std::endl;

 /* VIDEO INIT */
 std::cout << " Initializing video..." << std::endl;
 if (SDL_VideoDriverName(VideoDriverName, 1024) != 0)
 {
  std::cout << "  video driver: " << VideoDriverName << std::endl;
 }
 else
 {
  std::cout << "  video driver: unknown" << std::endl;  // or default?
 }
 if (Video::init() == false)
 {
  std::cout << "  error: " << SDL_GetError() << std::endl;
  std::cout << "\n" << "->Exit" << std::endl;
  SDL_Quit();
  return 1;
 }
 std::cout << "  OpenGL implementation:" << std::endl;
 std::cout << "   vendor:   " << glGetString(GL_VENDOR) << std::endl;
 std::cout << "   renderer: " << glGetString(GL_RENDERER) << std::endl;
 std::cout << "   version:  " << glGetString(GL_VERSION) << std::endl;
 std::cout << " Done" << "\n" << std::endl;

 /* AUDIO INIT */
 std::cout << " Initializing audio..." << std::endl;
 if (SDL_AudioDriverName(AudioDriverName, 1024) != 0)
 {
  std::cout << "  audio driver: " << AudioDriverName << std::endl;
 }
 else
 {
  std::cout << "  audio driver: unknown" << std::endl;  // or default?
 }
 if (Audio::init() == false)
 {
  std::cout << "  error: " << SDL_GetError() << std::endl;
  std::cout << "\n" << "->Exit" << std::endl;
  SDL_Quit();
  return 1;
 }
 std::cout << " Done" << "\n" << std::endl;

 /* CONTROLLERS INIT */
 IOPort::controller_port = Config::read_setting("controller_port");
 Gamepad[0].init();
 Gamepad[1].init();
 Gamepad[2].init();
 Gamepad[3].init();
 Gamepad[4].init();
 MultiTap::init();
 if (IOPort::controller_port == 3)
 {
  Gamepad[0].mode = Gamepad[0].MODE_B;
 }
 else
 {
  if (IOPort::controller_port == 4)
  {
   for (int i = 0; i < 5; i++)
   {
    if (MultiTap::Port[i] == 3) Gamepad[i].mode = Gamepad[i].MODE_B;
   }
  }
 }

 /* INPUT INIT */
 Input::init();

 std::cout << " Initializing emulation core..." << std::endl;
 PCE::init();
 std::cout << "  emulated console: ";
 if (PCE::console == 1) std::cout << "PC Engine" << std::endl;
 if (PCE::console == 2) std::cout << "TurboGrafx 16" << std::endl;
 std::cout << " Done" << "\n" << std::endl;

 SDL_PauseAudio(0);

 std::cout << " Start emulation core" << std::endl;
 PCE::run();
 std::cout << " Stop" << "\n" << std::endl;

 Input::deinit();
 Audio::deinit();
 Video::deinit();

 /* statistics */
 int32 total_bufs = Audio::full_bufs + Audio::empty_bufs;
 if (total_bufs != 0)
 {
  float64 run_time = total_bufs * (Audio::buffer_len / 48000.0);
  float64 speed_factor = (float64)Audio::full_bufs / total_bufs;
  // std::cout.precision(6);
  std::cout << " Statistics" << std::endl;
  std::cout << "  run time: " << run_time << " seconds" << std::endl;
  std::cout << "  audio buffer length: " << Audio::buffer_len << " samples" << std::endl;
  std::cout << "  empty audio buffers: " << Audio::empty_bufs << " / " << total_bufs << std::endl;
  std::cout << "  emulation speed factor: " << speed_factor << std::endl;
 }

 std::cout << "\n" << "->Exit" << std::endl;
 SDL_Quit();
 return 0;
}
