
/************************************

  PicchioEngine

  Copyright(c)2008 Emanuele Bettidi

************************************/

/* PCE.cpp */

/* PCE */

#include "Types.h"
#include "Config.h"
#include "Input.h"
#include "Audio.h"
#include "Video.h"
#include "CPU.h"
#include "Timer.h"
#include "IntCtrl.h"
#include "IOPort.h"
#include "PSG.h"
#include "VDC.h"
#include "VCE.h"
#include "PCE.h"

namespace PCE
{
 uint8 read_null(uint32 addr21);
 uint8 read_ram(uint32 addr21);
 uint8 read_rom(uint32 addr21);
 uint8 read_sf2(uint32 addr21);
 uint8 read_io(uint32 addr21);

 void write_null(uint32 addr21, uint8 data);
 void write_ram(uint32 addr21, uint8 data);
 void write_pop(uint32 addr21, uint8 data);
 void write_sf2(uint32 addr21, uint8 data);
 void write_io(uint32 addr21, uint8 data);

 uint8 ram[0x2000];    /* Work RAM (8kB) */
 uint8 rom[0x280000];  /* HuCard ROM (2.5MB) */

 uint8 (*read_page[0x100])(uint32 addr21);
 void (*write_page[0x100])(uint32 addr21, uint8 data);

 int console = -1;
 bool pop_card = false;
 bool sf2_card = false;
 uint32 sf2_latch;

 void init()
 {
  if (console == -1) console = Config::read_setting("console");
  for (int32 i = 0; i < 0x2000; i++) ram[i] = 0xFF;  // NOT TESTED
  // for (int32 i = 0; i < 0x280000; i++) rom[i] = 0xFF;
  sf2_latch = 0;

  for (uint32 i = 0x00; i <= 0x7F; i++)
  {
   read_page[i] = read_rom;
  }
  if (sf2_card == true)
  {
   for (uint32 i = 0x40; i <= 0x7F; i++)
   {
    read_page[i] = read_sf2;
   }
  }
  for (uint32 i = 0x80; i <= 0xF7; i++)
  {
   read_page[i] = read_null;
  }
  read_page[0xF8] = read_ram;
  read_page[0xF9] = read_ram;
  read_page[0xFA] = read_ram;
  read_page[0xFB] = read_ram;
  read_page[0xFC] = read_null;
  read_page[0xFD] = read_null;
  read_page[0xFE] = read_null;
  read_page[0xFF] = read_io;

  for (uint32 i = 0x00; i <= 0x7F; i++)
  {
   write_page[i] = write_null;
  }
  if (pop_card == true)
  {
   for (uint32 i = 0x40; i <= 0x43; i++)
   {
    write_page[i] = write_pop;
   }
  }
  if (sf2_card == true)
  {
   write_page[0x00] = write_sf2;
  }
  for (uint32 i = 0x80; i <= 0xF7; i++)
  {
   write_page[i] = write_null;
  }
  write_page[0xF8] = write_ram;
  write_page[0xF9] = write_ram;
  write_page[0xFA] = write_ram;
  write_page[0xFB] = write_ram;
  write_page[0xFC] = write_null;
  write_page[0xFD] = write_null;
  write_page[0xFE] = write_null;
  write_page[0xFF] = write_io;

  VCE::init();
  VDC::init();
  PSG::init();
  CPU::init();
 }

 void run()
 {
  for(;;)
  {
   Audio::calc_psg_cycles();
   VDC::clock += (Audio::psg_cycles * 2);
   PSG::clock += (Audio::psg_cycles);
   Timer::clock += (Audio::psg_cycles * 2);
   CPU::run(Audio::psg_cycles * 2);
   VDC::resync();
   PSG::resync();
   Timer::resync();
   Audio::wait_callback();
   Input::update_events();
   if (Input::exit == true) return;
   while (Input::pause == true)
   {
    Input::pause_loop();
    if (Input::exit == true) return;
   }
  }
 }

 uint8 read_null(uint32 addr21)
 {
  return CPU::read_buffer = 0xFF;
 }

 uint8 read_ram(uint32 addr21)
 {
  return CPU::read_buffer = ram[addr21 & 0x1FFF];
 }

 uint8 read_rom(uint32 addr21)
 {
  return CPU::read_buffer = rom[addr21];
 }

 uint8 read_sf2(uint32 addr21)
 {
  return CPU::read_buffer = rom[addr21 + sf2_latch];
 }

 uint8 read_io(uint32 addr21)
 {
  switch ((addr21 & 0x1C00) >> 10)
  {
   case 0: return CPU::read_buffer = VDC::read(addr21);
   case 1: return CPU::read_buffer = VCE::read(addr21);
   case 2: return CPU::IO_buffer;
   case 3: return CPU::IO_buffer = (CPU::IO_buffer & 0x80) | (Timer::read() & 0x7F);
   case 4: return CPU::IO_buffer = IOPort::read();
   case 5: if (addr21 & 2) CPU::IO_buffer = (CPU::IO_buffer & 0xF8) | (IntCtrl::read(addr21) & 7);
           return CPU::IO_buffer ;
   case 6: return CPU::read_buffer = 0xFF;
   case 7: return CPU::read_buffer = 0xFF;
   default: return 0;  // to avoid the warning
  }
 }

 void write_null(uint32 addr21, uint8 data)
 {
 }

 void write_ram(uint32 addr21, uint8 data)
 {
  ram[addr21 & 0x1FFF] = data;
 }

 void write_pop(uint32 addr21, uint8 data)
 {
  rom[addr21] = data;
 }

 void write_sf2(uint32 addr21, uint8 data)
 {
  if ((addr21 & 0x1FFC) == 0x1FF0)
  {
   sf2_latch = (addr21 & 3) << 19;
  }
 }

 void write_io(uint32 addr21, uint8 data)
 {
  switch ((addr21 & 0x1C00) >> 10)
  {
   case 0: VDC::write(addr21, data);
           break;
   case 1: VCE::write(addr21, data);
           break;
   case 2: CPU::IO_buffer = data;
           PSG::write(addr21, data);
           break;
   case 3: CPU::IO_buffer = data;
           Timer::write(addr21, data);
           break;
   case 4: CPU::IO_buffer = data;
           IOPort::write(data);
           break;
   case 5: CPU::IO_buffer = data;
           IntCtrl::write(addr21, data);
           break;
   case 6: break;
   case 7: break;
  }
 }
}

