
/************************************

  PicchioEngine

  Copyright(c)2008 Emanuele Bettidi

************************************/

/* VCE.cpp */

/* HuC6260 - VCE (Video Color Encoder) */

#include <cmath>
#include "Types.h"
#include "Config.h"
#include "Video.h"
#include "CPU.h"
#include "VCE.h"

namespace VCE
{
 void reset();

 void fill_matrix_rom();
 void fill_color_lut();
 inline void update_palette_lut(uint16 addr);

 /* registers */
 uint8 CR;    // control register
 static uint16 CTA;  // color table address register
 // uint16 CTW;      // color table write register
 // uint16 CTR;      // color table read register
 static int8 matrix_rom[512][3];    // matrix ROM
 static uint16 color_table[0x200];  // color table RAM

 bool rgb_output;                 // ...
 static uint32 color_lut[0x200];  // RGBA8 colors look-up table
 uint32 palette_lut[0x200];       // RGBA8 palette look-up table

 void fill_matrix_rom()
 {
  // 0<=R<=7, 0<=G<=7, 0<=B<=7
  // 0<=Y<=31, -15<=R-Y<=15, -15<=B-Y<=15
  //   Y = nint(  1.33*R + 2.61*G + 0.49*B)
  // R-Y = nint(  2.14*R - 1.80*G - 0.34*B)
  // B-Y = nint(- 0.72*R - 1.42*G + 2.14*B)
  // where nint(x) is the nearest integer function(symmetric arithmetic rounding)
  // Note: the rounding method described in the patent US 5389949 is wrong,
  //       probably an error of translation from the Japanese.
  int32 R, G, B, Y, RY, BY;
  for (uint32 i = 0; i < 512; i++)
  {
   G = i >> 6;
   R = (i >> 3) & 7;
   B = i & 7;
   Y  =  (133 * R) + (261 * G) +  (49 * B);
   RY =  (214 * R) - (180 * G) -  (34 * B);
   BY = - (72 * R) - (142 * G) + (214 * B);
   Y += 50;
   if (RY < 0) RY -= 50; else RY += 50;
   if (BY < 0) BY -= 50; else BY += 50;
   Y /= 100;
   RY /= 100;
   BY /= 100;
   matrix_rom[i][0] = (int8)Y;
   matrix_rom[i][1] = (int8)RY;
   matrix_rom[i][2] = (int8)BY;
  }
 }

 void fill_color_lut()
 {
  uint32 R, G, B, A;
  if (rgb_output == true) // pseudo-rgb output colors
  {
   // GRB333 -> RGBA8
   for (uint32 i = 0; i < 512; i++)
   {
    G = (i >> 6) * 36;
    R = ((i >> 3) & 7) * 36;
    B = (i & 7) * 36;
    A = 255;
    color_lut[i] = (A << 24) | (B << 16) | (G << 8) | R;
   }
  }
  else // composite output colors
  {
   // YUV555 -> RGBA8
   for (uint32 i = 0; i < 512; i++)
   {
    float64 y, ry, by, r, g , b;
    y  = (matrix_rom[i][0] / 31.0);
    ry = (matrix_rom[i][1] / 15.0) * 0.701 * 0.75;
    by = (matrix_rom[i][2] / 15.0) * 0.886 * 0.75;
    r = y + ry;
    g = y - ((0.114/0.587) * by) - ((0.299/0.587) * ry);
    b = y + by;
    R = (int32)floor((255.0 * r) + 0.5);
    G = (int32)floor((255.0 * g) + 0.5);
    B = (int32)floor((255.0 * b) + 0.5);
    if ((int32)R < 0) R = 0; if (R > 255) R = 255;
    if ((int32)G < 0) G = 0; if (G > 255) G = 255;
    if ((int32)B < 0) B = 0; if (B > 255) B = 255;
    A = 255;
    color_lut[i] = (A << 24) | (B << 16) | (G << 8) | R;
   }
  }
 }

 inline void update_palette_lut(uint16 addr)
 {
  if (addr != 0)
  {
   if ((addr & 0x10F) != 0)
   {
    palette_lut[addr] = color_lut[color_table[addr]];
   }
  }
  else
  {
   uint32 temp = color_lut[color_table[0]];;
   for (uint i = 0; i < 16; i++)
   {
    palette_lut[i << 4] = temp;
   }
  }
 }

 void init()
 {
  rgb_output = Config::read_setting("rgb_output");
  CR = 0;  // NOT TESTED
  CTA = 0;  // NOT TESTED
  fill_matrix_rom();
  fill_color_lut();
  for (uint32 i = 0; i < 0x200; i++)
  {
   color_table[i] = 0x1FF;  // NOT FULLY TESTED
   update_palette_lut(i);
  }
  // ...
  reset();
 }

 void reset()
 {
  // ...
 }

 void resync()
 {
  // VDC::resync();
  // ...
 }

 uint8 read(uint32 reg)
 {
  CPU::adv_clk();
  resync();
  switch(reg & 0x07)
  {
   case 0: return 0xFF;
   case 1: return 0xFF;
   case 2: return 0xFF;
   case 3: return 0xFF;
   case 4: return (uint8)color_table[CTA];
   case 5: uint8 data = (uint8)(color_table[CTA] >> 8);
	     data |= 0xFE;
           CTA++;
           CTA &= 0x1FF;
           return data;
   case 6: return 0xFF;
   case 7: return 0xFF;
   default: return 0;  // to avoid the warning
  }
 }

 void write(uint32 reg, uint8 data)
 {
  CPU::adv_clk();
  resync();
  switch(reg & 0x07)
  {
   case 0: CR = data;
           break;
   case 1: break;
   case 2: CTA &= 0x100;
           CTA |= (uint16)data;
           break;
   case 3: CTA &= 0x0FF;
           CTA |= (uint16)(data & 0x01) << 8;
           break;
   case 4: color_table[CTA] &= 0x100;
	     color_table[CTA] |= (uint16)data;
           update_palette_lut(CTA);
           break;
   case 5: color_table[CTA] &= 0x0FF;
	     color_table[CTA] |= (uint16)(data & 0x01) << 8;
           update_palette_lut(CTA);
	     CTA++;
           CTA &= 0x1FF;
	     break;
   case 6: break;
   case 7: break;
  }
 }

}
