/****************************************\
|* MFZPSP LLE PSP Emulator              *|
|* Copyright (c) 2008 Mauro Frischherz  *|
|* See License.txt for details          *|
\****************************************/

#include "loader.h"
#include "memory.h"
#include "../tools/log.h"
#include <stdio.h>

/*							*/
/* ELF Defines and Structs	*/
/*							*/

// ELF Ident Indexes
#define EI_MAG0			0
#define EI_MAG1			1
#define EI_MAG2			2
#define EI_MAG3			3
#define EI_CLASS		4
#define EI_DATA			5
#define EI_VERSION		6
#define EI_PAD			7
#define EI_NIDENT		16

// ELF Types
#define ET_NONE			0
#define ET_REL			1
#define ET_EXEC			2
#define ET_DYN			3
#define ET_CORE			4
#define ET_LOPROC		0xFF00
#define ET_HIPROC		0xFFFF

// ELF Machine -> MIPS
#define EM_NONE			0
#define EM_MIPS			8

// ELF Classes
#define ELFCLASSNONE	0
#define ELFCLASS32		1
#define ELFCLASS64		2

// ELF Data Encoding
#define ELFDATANONE		0
#define ELFDATA2LSB		1
#define ELFDATA2MSB		2

// ELF Section Types
#define SHT_NULL		0
#define SHT_PROGBITS	1
#define SHT_SYMTAB		2
#define SHT_STRTAB		3
#define SHT_RELA		4
#define SHT_HASH		5
#define SHT_DYNAMIC		6
#define SHT_NOTE		7
#define SHT_NOBITS		8
#define SHT_REL			9
#define SHT_SHLIB		10
#define SHT_DYNSYM		11
#define SHT_LOPROC		0x70000000
#define SHT_HIPROC		0x7FFFFFFF
#define SHT_LOUSER		0x80000000
#define SHT_HIUSER		0xFFFFFFFF

// ELF Section Attribute Flags
#define SHT_WRITE		0x1
#define SHF_ALLOC		0x2
#define SHF_EXECINSTR	0x4
#define SHR_MASKPROC	0xF0000000

// ELF Relocation Types
#define R_MIPS_NONE			0
#define R_MIPS_16			1
#define R_MIPS_32			2
#define R_MIPS_REL32		3
#define R_MIPS_26			4
#define R_MIPS_HI16			5
#define R_MIPS_LO16			6
#define R_MIPS_GPREL16		7
#define R_MIPS_LITERAL		8
#define R_MIPS_GOT16		9
#define R_MIPS_PC16			10
#define R_MIPS_CALL16		11
#define R_MIPS_GPREL32		12

// ELF Segment Types
#define PT_NULL			0
#define PT_LOAD			1
#define PT_DYNAMIC		2
#define PT_INTERP		3
#define PT_NOTE			4
#define PT_SHLIB		5
#define PT_PHDR			6
#define PT_LOPROC		0x70000000
#define PT_HIPROC		0x7FFFFFFF

// ELF Magic Number 0x7F 'E' 'L' 'F'
#define ELFMAGNUM		0x464C457F

// Macros
#define ELF32_R_ADDR_BASE(i) 	(((i)> >16) & 0xFF)
#define ELF32_R_OFS_BASE(i) 	(((i)> >8) & 0xFF)
#define ELF32_R_TYPE(i)			(i & 0xFF)

// ELF Header
struct ELFFileInfo {
	// Header
	ubyte ident[EI_NIDENT];
	short type;
	short machine;
	uint  version;
	uint  entry;
	uint  phoff;
	uint  shoff;
	uint  flags;
	short ehsize;
	short phentsize;
	short phnum;
	short shentsize;
	short shnum;
	short shstrndx;
} ELFInfo;

// ELF Section Header
struct ELFSectionHeader {
	uint name;
	uint type;
	uint flags;
	uint addr;
	uint offset;
	uint size;
	uint link;
	uint info;
	uint addralign;
	uint entsize;
} ELFSectHead;

// ELF Symbol Table
struct ELFSymbolTable {
	uint name;
	uint value;
	uint size;
	ubyte info;
	ubyte other;
	short shndx;
} ELFSymTbl;

// ELF Relocation Entry
struct ELFRelocationEntry {
	uint offset;
	uint info;
} ELFRel;

// ELF Relocation Entry Addend
struct ELFRelocationEntryAddend {
	uint offset;
	uint info;
	uint addend;
} ELFRelA;

// ELF Program Header
struct ELFProgramHeader {
	uint type;
	uint offset;
	uint vaddr;
	uint paddr;
	uint filesz;
	uint memsz;
	uint flags;
	uint align;
} ELFPrgHdr;

/*							*/
/* PBP Defines and Structs	*/
/*							*/

// PBP Magic Number 0x00 'P' 'B' 'P'
#define PBPMAGNUM 		0x50425000

// PBP Header + FileInfo
struct PBPFileInfo {
	// Header
	uint ident;
	uint version;
	uint offsetparam;
	uint offseticon0;
	uint offseticon1;
	uint offsetuknpng;
	uint offsetpic1;
	uint offsetsnd0;
	uint offsetpsp;
	uint offsetpsar;
	
	// FileInfo
	uint sizeparam;
	uint sizeicon0;
	uint sizeicon1;
	uint sizeuknpng;
	uint sizepic1;
	uint sizesnd0;
	uint sizepsp;
	uint sizepsar;
} PBPInfo;

bool PrepareFile();
bool DecodeFile();
bool WriteToMemory();
bool IsValidFile();

bool FileLoaded = false;
int  FileType = 0;
FILE* pFile = NULL;

bool GetLoaded() { return FileLoaded; }


bool LoadAndDecodeFile(const char* path, int type)
{
	FileLoaded = false;
	FileType = type;
	pFile = NULL;
	
	char buffer[500];
	sprintf(buffer, "Opening: %s", path);
	printevent(buffer);
	
	pFile = fopen(path,"rb");
	
	if(pFile == NULL)
		return false;
	
	if(!PrepareFile())
	{
		fclose(pFile);
		printerror("Load cancelled: Invalid File");
		return false;
	}
	
	if(!IsValidFile())
	{
		fclose(pFile);
		printerror("Load cancelled: Invalid File");
		return false;
	}
	
	if(!DecodeFile())
	{
		fclose(pFile);
		printerror("Load cancelled: Invalid File");
		return false;
	}
	
	if(!WriteToMemory())
	{
		fclose(pFile);
		printerror("Load cancelled: Invalid File");
		return false;
	}
	
	FileLoaded = true;
	fclose(pFile);
	return true;
}

bool PrepareFile()
{
	switch(FileType)
	{
	case FILE_TYPE_PBP:
		{
			printevent("Preparing PBP-File");
			uint PBPHeader[10];
			fread(PBPHeader, sizeof(PBPHeader[0]), sizeof(PBPHeader)/sizeof(PBPHeader[0]), pFile);
			
			fseek(pFile, 0, SEEK_END);
			uint filesize = ftell(pFile);
			
			PBPInfo.ident 			= PBPHeader[0];
			PBPInfo.version 		= PBPHeader[1];
			PBPInfo.offsetparam 	= PBPHeader[2];
			PBPInfo.offseticon0 	= PBPHeader[3];
			PBPInfo.offseticon1 	= PBPHeader[4];
			PBPInfo.offsetuknpng  	= PBPHeader[5];
			PBPInfo.offsetpic1		= PBPHeader[6];
			PBPInfo.offsetsnd0		= PBPHeader[7];
			PBPInfo.offsetpsp		= PBPHeader[8];
			PBPInfo.offsetpsar		= PBPHeader[9];
			
			PBPInfo.sizeparam 		= PBPInfo.offseticon0 - PBPInfo.offsetparam;
			PBPInfo.sizeicon0 		= PBPInfo.offseticon1 - PBPInfo.offseticon0;
			PBPInfo.sizeuknpng 		= PBPInfo.offsetpic1 - PBPInfo.offsetuknpng;
			PBPInfo.sizepic1		= PBPInfo.offsetsnd0 - PBPInfo.offsetpic1;
			PBPInfo.sizesnd0		= PBPInfo.offsetpsp - PBPInfo.offsetsnd0;
			PBPInfo.sizepsp			= PBPInfo.offsetpsar - PBPInfo.offsetpsp;
			PBPInfo.sizepsar		= filesize - PBPInfo.offsetpsar;
			
			printevent("Successfully read PBP-Header");
			return true;
		}
		break;
	case FILE_TYPE_ELF:
		{
			printevent("Preparing ELF-File");
			int i;
			ubyte ELFIdent[16];
			fread(ELFIdent, sizeof(ELFIdent[0]), sizeof(ELFIdent)/sizeof(ELFIdent[0]), pFile);
			
			ushort ELFShortHdr[26];
			fseek(pFile, 0, SEEK_SET);
			fread(ELFShortHdr, sizeof(ELFShortHdr[0]), sizeof(ELFShortHdr)/sizeof(ELFShortHdr[0]), pFile);
			
			uint ELFWordHdr[13];
			fseek(pFile, 0, SEEK_SET);
			fread(ELFWordHdr, sizeof(ELFWordHdr[0]), sizeof(ELFWordHdr)/sizeof(ELFWordHdr[0]), pFile);
	
			for(i = 0; i < 16; i++)
			{
				ELFInfo.ident[i] 	= ELFIdent[i];
			}
			
			ELFInfo.type		= ELFShortHdr[8];
			ELFInfo.machine		= ELFShortHdr[9];
			
			ELFInfo.version		= ELFWordHdr[5];
			ELFInfo.entry		= ELFWordHdr[6];
			ELFInfo.phoff		= ELFWordHdr[7];
			ELFInfo.shoff		= ELFWordHdr[8];
			ELFInfo.flags		= ELFWordHdr[9];
			
			ELFInfo.ehsize		= ELFShortHdr[20];
			ELFInfo.phentsize	= ELFShortHdr[21];
			ELFInfo.phnum		= ELFShortHdr[22];
			ELFInfo.shentsize	= ELFShortHdr[23];
			ELFInfo.shnum		= ELFShortHdr[24];
			ELFInfo.shstrndx	= ELFShortHdr[25];
			
			printevent("Successfully read ELF-Header");
			return true;
		}
	}
	return false;
}

bool IsValidFile()
{
	switch(FileType)
	{
	case FILE_TYPE_PBP:
		{
			if(!(PBPInfo.ident == PBPMAGNUM && PBPInfo.sizepsp > 0))
				return false;
		}
		break;
	case FILE_TYPE_ELF:
		{
			if(!(ELFInfo.ident[0] == 0x7F && ELFInfo.ident[1] == 0x45 && ELFInfo.ident[2] == 0x4C && ELFInfo.ident[3] == 0x46))
				return false;
		}
	}
	printevent("File-Check result: Valid");
	return true;
}

bool DecodeFile()
{
	switch(FileType)
	{
	case FILE_TYPE_PBP:
		{
			printevent("Unpacking PBP");
			/* Unpack PBP */
			
			FILE* pFileTarget;
			
			/* PARAM.SFO */
			if(PBPInfo.sizeparam > 0)
			{
				printevent("--> PARAM.SFO");
				uint steps;
				uint Buffer;
				pFileTarget = fopen("PARAM.SFO", "wb");
				for(steps = 0; steps < PBPInfo.sizeparam; steps += 4)
				{
					fseek(pFile, PBPInfo.offsetparam + steps, SEEK_SET);
					fread(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer), pFile);
					fwrite(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer),pFileTarget);
				}
				fclose(pFileTarget);
			}
			
			/* ICON0.PNG */
			if(PBPInfo.sizeicon0 > 0)
			{
				printevent("--> ICON0.PNG");
				uint steps;
				uint Buffer;
				pFileTarget = fopen("ICON0.PNG", "wb");
				for(steps = 0; steps < PBPInfo.sizeicon0; steps += 4)
				{
					fseek(pFile, PBPInfo.offseticon0 + steps, SEEK_SET);
					fread(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer), pFile);
					fwrite(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer),pFileTarget);
				}
				fclose(pFileTarget);
			}
			
			/* ICON1.PMF */
			if(PBPInfo.sizeicon1 > 0)
			{
				printevent("--> ICON1.PMF");
				uint steps;
				uint Buffer;
				pFileTarget = fopen("ICON1.PMF", "wb");
				for(steps = 0; steps < PBPInfo.sizeicon1; steps += 4)
				{
					fseek(pFile, PBPInfo.offseticon1 + steps, SEEK_SET);
					fread(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer), pFile);
					fwrite(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer),pFileTarget);
				}
				fclose(pFileTarget);
			}
			
			/* UNKOWNPNG */
			if(PBPInfo.sizeuknpng > 0)
			{
				printevent("--> UNKNOWN.PNG");
				uint steps;
				uint Buffer;
				pFileTarget = fopen("UNKNOWN.PNG", "wb");
				for(steps = 0; steps < PBPInfo.sizeuknpng; steps += 4)
				{
					fseek(pFile, PBPInfo.offsetuknpng + steps, SEEK_SET);
					fread(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer), pFile);
					fwrite(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer),pFileTarget);
				}
				fclose(pFileTarget);
			}
			
			/* PIC1.PNG */
			if(PBPInfo.sizepic1 > 0)
			{
				printevent("--> PIC1.PNG");
				uint steps;
				uint Buffer;
				pFileTarget = fopen("PIC1.PNG", "wb");
				for(steps = 0; steps < PBPInfo.sizepic1; steps += 4)
				{
					fseek(pFile, PBPInfo.offsetpic1 + steps, SEEK_SET);
					fread(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer), pFile);
					fwrite(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer),pFileTarget);
				}
				fclose(pFileTarget);
			}
			
			/* SND0.AT3 */
			if(PBPInfo.sizesnd0 > 0)
			{
				printevent("--> SND0.AT3");
				uint steps;
				uint Buffer;
				pFileTarget = fopen("SND0.AT3", "wb");
				for(steps = 0; steps < PBPInfo.sizesnd0; steps += 4)
				{
					fseek(pFile, PBPInfo.offsetsnd0 + steps, SEEK_SET);
					fread(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer), pFile);
					fwrite(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer),pFileTarget);
				}
				fclose(pFileTarget);
			}
			
			/* PSP.ELF */
			if(PBPInfo.sizepsp > 0)
			{
				printevent("--> PSP.ELF");
				uint steps;
				uint Buffer;
				pFileTarget = fopen("PSP.ELF", "wb");
				
				for(steps = 0; steps < PBPInfo.sizepsp; steps += 4)
				{
					fseek(pFile, PBPInfo.offsetpsp + steps, SEEK_SET);
					fread(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer), pFile);
					fwrite(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer),pFileTarget);
				}
				fclose(pFileTarget);
			}
			
			/* PSAR.DATA */
			if(PBPInfo.sizepsar > 0)
			{
				printevent("--> PSAR.DATA");
				uint steps;
				uint Buffer;
				pFileTarget = fopen("PSAR.DATA", "wb");
				
				for(steps = 0; steps < PBPInfo.sizepsar; steps += 4)
				{
					fseek(pFile, PBPInfo.offsetpsar + steps, SEEK_SET);
					fread(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer), pFile);
					fwrite(&Buffer, sizeof(Buffer), sizeof(Buffer)/sizeof(Buffer),pFileTarget);
				}
				fclose(pFileTarget);
			}
			
			printevent("Unpacking completed");
			printevent("Opening unpacked ELF-File");
			LoadAndDecodeFile(".\\PSP.ELF", FILE_TYPE_ELF);
		}
		break;
	case FILE_TYPE_ELF:
		{
			printevent("Reading ELF program and section headers");
			// Read Program Header
			if(ELFInfo.phoff != 0)
			{
				fseek(pFile, ELFInfo.phoff, SEEK_SET);
			}
		}
	}
	return false;
}

bool WriteToMemory()
{
	
	return true;
}


