/************************************************************************

  CDDAData.hpp

  Copyright (C) 2007 Virus
  Copyright (C) 2002 mooby

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 ************************************************************************/

#ifndef CDDADATA_HPP
#define CDDADATA_HPP

#include "FileInterface.hpp"
#include "CDTime.hpp"
#include "TrackInfo.hpp"
#include "Utils.hpp"
#include "Preferences.hpp"
#include <iostream>

using namespace std;

//#include <vorbis/vorbisfile.h>
#include <SDL.h>

// CDDA data virtual base class
class CDDAData
{
    public:
        CDDAData() {}
        virtual ~CDDAData() {}
        virtual void openFile(const std::string& file) = 0;
        virtual int play(const CDTime& startTime) = 0;
        virtual int stop() = 0;
        virtual void setVolume(const std::string& volstr)
        {
            istringstream tmp(volstr);
            if (tmp >> volume)
            {
                volume = volume/100;
                if (volume < 0) volume = 0;
                else if (volume > 1) volume = 1;
            }
            else
            {
                volume = 1;
                //printf("CDDAData::setVolume(%s) failed\n",volstr.c_str());
            }
        }
        virtual void setRepeat(const std::string& repstr)
        {
            if (repstr == repeatAllString)
                repeat = 2;
            else if (repstr == repeatOneString)
                repeat = 1;
            else if (repstr == playOneString)
                repeat = 0;
            else
            {
                repeat = 2;
                //printf("CDDAData::setRepeat(%s) failed\n",repstr.c_str());
            }
        }
        virtual bool isPlaying(void) {return false;}
        virtual CDTime playTime(void) {return CDTime();}

        int repeat;
        float volume;
};

// if there's no CDDA data, just return sensible values
class NoCDDAData : public CDDAData
{
    public:
        NoCDDAData() {}
        virtual ~NoCDDAData() {}
        virtual void openFile(const std::string& file) {}
        virtual int play(const CDTime& startTime) {return -1;}
        virtual int stop() {return -1;}
};

// for really playing CDDA data.
class PlayCDDAData : public CDDAData
{
    public:
        // It is the list of track lengths.
        // sets the volume from whatever is stored in the preferences
        PlayCDDAData(const std::vector<TrackInfo> ti);

        // cleans up
        virtual ~PlayCDDAData() {if (playing) stop(); delete theCD; SDL_Quit();}

        // opens the file and readies the plugin for playing
        virtual void openFile(const std::string& file);
        // plays data starting at startTime
        virtual int play(const CDTime& startTime);
        // stops playing
        virtual int stop();

        virtual bool isPlaying(void) {return playing;}
        virtual CDTime playTime(void) {return CDDAPos;}

        // All the data members are public so they can be accessed by the PortAudio
        // callback

        // the current position of playing
        CDTime CDDAPos;
        // the end time of this track
        CDTime CDDAEnd;
        // the start time taken from play()
        CDTime CDDAStart;
        CDTime InitialTime;
        // the offset into the frame that it's currently playing from
        unsigned long frameOffset;
        // a FileInterface for getting the data from
        FileInterface* theCD;
        // the track list of the CD for calculating times n such
        std::vector<TrackInfo> trackList;
        // if true, it's playing.
        bool playing;
        // a buffer of null audio if repeat is off
        char nullAudio[bytesPerFrame];
        // if true and repeating one track, 
        // this is at the end of the currently playing track
        bool endOfTrack;
};
/*
// for playing CDDA data compressed as OGG
class PlayOGGData : public CDDAData
{
    public:
        PlayOGGData(const std::vector<TrackInfo> ti);
        virtual ~PlayOGGData() { if (playing) stop(); SDL_Quit(); }
        virtual void openFile(const std::string& file);
        virtual int play(const CDTime& startTime);
        virtual int stop();
        virtual CDTime playTime(void) {return CDDAPos;}
        // Tries to load current track
        bool loadTrack();
        int track;

        // Opens audio with freq and channels
        bool openAudio();
        int freq;
        int channels;

        CDTime initialTime;
        CDTime CDDAPos;
        std::vector<TrackInfo> trackList;
        bool playing;
        bool endOfTrack;
        bool oggLoaded;
        bool audioOpen;

        std::string fileName;
        OggVorbis_File vf;
        int current_section;
};

*/

#endif
