/************************************************************************

PlayCDDAData.cpp

Copyright (C) 2008 Micket
Copyright (C) 2007 Virus
Copyright (C) 2002 mooby

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

************************************************************************/
#pragma warning(disable:4786)

#include "CDDAData.hpp"

using namespace std;

extern Preferences prefs;

// this callback repeats one track over and over
void CDDACallback(void * userData, Uint8 * outputBuffer, int len)
{
    // Cast data passed through stream to our structure type.
    PlayCDDAData* data = (PlayCDDAData*)userData;
    short* out = (short*)outputBuffer;

    data->theCD->seek(data->CDDAPos);
    short* buffer = ((short*)data->theCD->getBuffer()) + data->frameOffset/2;

    double volume = data->volume;
    int repeat = data->repeat;
    for(int i=0; i < len/4; i++ ) // len = bytes, short*stereo = 4 bytes
    {
        // at the end of a frame, get the next one and fill the buffer
        if (data->frameOffset == bytesPerFrame)
        {
            data->CDDAPos += CDTime(0,0,1);
            // if not at the end, then carry on
            if (data->CDDAPos < data->CDDAEnd)
            {
                data->theCD->seek(data->CDDAPos);
                data->frameOffset = 0;
                buffer = (short*)data->theCD->getBuffer();
            }
            // when at the end of this track, use null audio (playOne)
            else if (repeat == 0)
            {
                data->endOfTrack = true;
                buffer = (short*)data->nullAudio;
                data->CDDAPos -= CDTime(0,0,1);
                data->frameOffset = 0;
            }
            // when at the end of this track, loop to the start of this track
            else //if (repeat == 1)
            {
                data->CDDAPos = data->CDDAStart;
                data->theCD->seek(data->CDDAPos);
                data->frameOffset = 0;
                buffer = (short*)data->theCD->getBuffer();
            }
            // I think one should handle the repeat-things here. More dynamic.
        }
        // Stereo channels are interleaved.
        *out++ = (short)((double)(*buffer++) * volume);  // left 
        *out++ = (short)((double)(*buffer++) * volume);  // right
        data->frameOffset += 4;
    }
}

PlayCDDAData::PlayCDDAData(const std::vector<TrackInfo> ti) 
    : frameOffset(0), theCD(NULL), trackList(ti), playing(false),
    endOfTrack(false)
{
    printf("PlayCDDAData::PlayCDDAData\n");
    memset(nullAudio, 0, sizeof(nullAudio));
    setVolume(prefs.prefsMap[volumeString]);
    setRepeat(prefs.prefsMap[repeatString]);
}

// initialize the CDDA file data and initalize the audio stream
void PlayCDDAData::openFile(const std::string& file) 
{
    printf("PlayCDDAData::openFile(%s)\n",file.c_str());
    std::string extension;
    theCD = FileInterfaceFactory(file, extension);
    //theCD->setPregap(pregapLength, trackList[2].trackStart);
    if(SDL_Init(SDL_INIT_AUDIO)==-1)
    {
        Exception e(string("SDL_Init: ") + string(SDL_GetError()));
        THROW(e);
    }
    // disable extra caching on the file interface
    theCD->setCacheMode(FileInterface::oldMode);
}

// start playing the data
int PlayCDDAData::play(const CDTime& startTime)
{
    printf("PlayCDDAData::play\n");
    // if play was called with the same time as the previous call,
    // dont restart it.  this fixes a problem with FPSE's play call.
    // of course, if play is called with a different track, 
    // stop playing the current stream.
    if (playing)
    {
        if (startTime == InitialTime)
            return 0;
        else
            SDL_LockAudio();
    }
    InitialTime = startTime;
    cout << "Requested time:" << startTime;
    // figure out which track to play to set the end time
    int i = 2;
    while ( i < trackList.size()-1 && startTime > trackList[i+1].trackStart )
        i++;

    CDTime adjustedStartTime = startTime;
    // These 2 seconds have *nothing* to do with pregap
    if (adjustedStartTime < (trackList[i].trackStart + CDTime(0,2,0)))
        adjustedStartTime = trackList[i].trackStart;
    // If very close to the next track, adjust to that instead
    // I'm not sure i *want*  to do this though? Maybe the game wants to have a seconds
    // gap (Wipeout consistenly requests times that are around a second 
    // less then the track start )
    else if (i+1 < trackList.size() && 
        adjustedStartTime > trackList[i+1].trackStart - CDTime(0,2,0))
    {
        i++;
        adjustedStartTime = trackList[i].trackStart;
    }
    cout << "Adjusted time:" << adjustedStartTime;

    if ( repeat == 1 || repeat == 0)
    {
        CDDAStart = adjustedStartTime;
        CDDAEnd = trackList[i].trackStart + trackList[i].trackLength;
    }
    else //if (repeat == 2)
    {
        CDDAEnd = trackList[trackList.size() - 1].trackStart +
                  trackList[trackList.size() - 1].trackLength;
        CDDAStart = trackList[2].trackStart;
        if (adjustedStartTime > CDDAEnd)
            adjustedStartTime = CDDAStart;
    }
    // set the cdda position, start and end times
    CDDAPos = adjustedStartTime;
    endOfTrack = false;

    if (!playing)
    {
        // open a stream - pass in this CDDA object as the user data.
        SDL_AudioSpec *desired = (SDL_AudioSpec*)malloc(sizeof(SDL_AudioSpec));
        desired->freq     = 44100;
        desired->format   = AUDIO_S16SYS;
        desired->channels = 2;
        desired->samples  = bytesPerFrame; // Why 5880 in Pa? Any number should be fine.
                                           // also, it seems to be ignored completely.
        desired->userdata = NULL;
        desired->callback = CDDACallback;
        desired->userdata = (void*)this;

        if ( SDL_OpenAudio(desired,NULL) < 0 ){
            fprintf(stderr, "Couldn't open audio: %s\n", SDL_GetError());
            free(desired);
            return -1;
        }
        printf("(%d) (%d)\n",desired->samples,bytesPerFrame);
        free(desired);
        SDL_PauseAudio(0);
    }
    else
        SDL_UnlockAudio();
    playing = true;
    return 0;
}

// close the stream - nice and simple
int PlayCDDAData::stop()
{
    printf("PlayCDDAData::stop\n");
    if (playing)
    {
        SDL_CloseAudio();
        playing = false;
    }
    return 0;
}