/*
 * Decompiled with CFR 0.152.
 */
package jpcsp.graphics.capture;

import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.Buffer;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.ShortBuffer;
import java.util.HashMap;
import java.util.zip.CRC32;
import java.util.zip.Deflater;
import javax.imageio.ImageIO;
import jpcsp.graphics.capture.CaptureManager;
import jpcsp.memory.IMemoryReader;
import jpcsp.memory.ImageReader;
import jpcsp.memory.MemoryReader;
import jpcsp.util.Utilities;
import org.apache.log4j.Logger;

public class CaptureImage {
    private static Logger log = CaptureManager.log;
    private static final String bmpFileFormat = "bmp";
    private static final String pngFileFormat = "png";
    private int imageaddr;
    private int level;
    private Buffer buffer;
    private IMemoryReader imageReader;
    private int width;
    private int height;
    private int bufferWidth;
    private int bufferStorage;
    private boolean compressedImage;
    private int compressedImageSize;
    private boolean invert;
    private boolean overwriteFile;
    private String fileNamePrefix;
    private String directory = "tmp/";
    private String fileFormat = "bmp";
    private String fileName;
    private String fileNameSuffix = "";
    private static HashMap<Integer, Integer> lastFileIndex = new HashMap();

    public CaptureImage(int imageaddr, int level, Buffer buffer, int width, int height, int bufferWidth, int bufferStorage, boolean compressedImage, int compressedImageSize, boolean invert, boolean overwriteFile, String fileNamePrefix) {
        this.imageaddr = imageaddr;
        this.level = level;
        this.buffer = buffer;
        this.width = width;
        this.height = height;
        this.bufferWidth = bufferWidth;
        this.bufferStorage = bufferStorage;
        this.compressedImage = compressedImage;
        this.compressedImageSize = compressedImageSize;
        this.invert = invert;
        this.overwriteFile = overwriteFile;
        this.fileNamePrefix = fileNamePrefix == null ? "Image" : fileNamePrefix;
    }

    public CaptureImage(int imageaddr, int level, IMemoryReader imageReader, int width, int height, int bufferWidth, boolean invert, boolean overwriteFile, String fileNamePrefix) {
        this.imageaddr = imageaddr;
        this.level = level;
        this.imageReader = imageReader;
        this.width = width;
        this.height = height;
        this.bufferWidth = bufferWidth;
        this.bufferStorage = 3;
        this.compressedImage = false;
        this.compressedImageSize = 0;
        this.invert = invert;
        this.overwriteFile = overwriteFile;
        this.fileNamePrefix = fileNamePrefix == null ? "Image" : fileNamePrefix;
    }

    public void setFileFormat(String fileFormat) {
        this.fileFormat = fileFormat;
    }

    public void setDirectory(String directory) {
        this.directory = directory;
    }

    public String getFileName() {
        if (this.fileName == null) {
            String levelName = "";
            if (this.level > 0) {
                levelName = "_" + this.level;
            }
            int scanIndex = 0;
            Integer lastIndex = lastFileIndex.get(this.imageaddr);
            if (lastIndex != null) {
                scanIndex = lastIndex + 1;
            }
            int i = scanIndex;
            while (true) {
                String id = i == 0 ? "" : "-" + i;
                this.fileName = String.format("%s%s%08X%s%s%s.%s", this.directory, this.fileNamePrefix, this.imageaddr, this.fileNameSuffix, levelName, id, this.fileFormat);
                if (this.overwriteFile) break;
                File file = new File(this.fileName);
                if (!file.exists()) {
                    lastFileIndex.put(this.imageaddr, i);
                    break;
                }
                ++i;
            }
        }
        return this.fileName;
    }

    public void setFileName(String fileName) {
        this.fileName = fileName;
    }

    public boolean fileExists() {
        return new File(this.getFileName()).exists();
    }

    public void setFileNameSuffix(String fileNameSuffix) {
        this.fileNameSuffix = fileNameSuffix;
    }

    private ShortBuffer getShortBuffer() {
        if (this.buffer instanceof ShortBuffer) {
            return (ShortBuffer)this.buffer;
        }
        if (this.buffer instanceof ByteBuffer) {
            return ((ByteBuffer)this.buffer).asShortBuffer();
        }
        return null;
    }

    private IntBuffer getIntBuffer() {
        if (this.buffer instanceof IntBuffer) {
            return (IntBuffer)this.buffer;
        }
        if (this.buffer instanceof ByteBuffer) {
            return ((ByteBuffer)this.buffer).asIntBuffer();
        }
        return null;
    }

    private boolean isShortBuffer() {
        return this.getShortBuffer() != null;
    }

    private boolean isIntBuffer() {
        return this.getIntBuffer() != null;
    }

    public void write() throws IOException {
        boolean imageType32Bit;
        if (this.bufferStorage >= 4 && this.bufferStorage <= 7 && this.bufferStorage != 6) {
            return;
        }
        if (this.compressedImage) {
            this.decompressImage();
        }
        boolean imageInvert = this.invert;
        int readWidth = Math.min(this.width, this.bufferWidth);
        byte[] pixelBytes = new byte[4];
        byte[] blackPixelBytes = new byte[pixelBytes.length];
        AbstractCaptureImage captureImage = bmpFileFormat.equals(this.fileFormat) ? new CaptureImageBMP() : (pngFileFormat.equals(this.fileFormat) ? new CaptureImagePNG() : new CaptureImageImageIO());
        captureImage.writeHeader(this.getFileName(), this.fileFormat, this.width, this.height, readWidth);
        if (captureImage.isInverted()) {
            imageInvert = !imageInvert;
        }
        boolean bl = imageType32Bit = this.bufferStorage == 3 || this.bufferStorage == 17;
        if (this.imageReader != null) {
            for (int y = 0; y < this.height; ++y) {
                captureImage.startLine(imageInvert ? this.height - y - 1 : y);
                for (int x = 0; x < readWidth; ++x) {
                    int pixel = this.imageReader.readNext();
                    captureImage.writePixel(pixel);
                }
                captureImage.endLine();
            }
            captureImage.writeEnd();
        } else if (this.isIntBuffer() && imageType32Bit) {
            IntBuffer intBuffer = this.getIntBuffer();
            for (int y = 0; y < this.height; ++y) {
                intBuffer.position((imageInvert ? this.height - y - 1 : y) * this.bufferWidth);
                captureImage.startLine(imageInvert ? this.height - y - 1 : y);
                for (int x = 0; x < readWidth; ++x) {
                    try {
                        int pixel = intBuffer.get();
                        captureImage.writePixel(pixel);
                        continue;
                    }
                    catch (BufferUnderflowException e) {
                        captureImage.writePixel(blackPixelBytes);
                    }
                }
                captureImage.endLine();
            }
        } else if (this.isShortBuffer() && !imageType32Bit) {
            ShortBuffer shortBuffer = this.getShortBuffer();
            for (int y = 0; y < this.height; ++y) {
                shortBuffer.position((imageInvert ? this.height - y - 1 : y) * this.bufferWidth);
                captureImage.startLine(imageInvert ? this.height - y - 1 : y);
                for (int x = 0; x < readWidth; ++x) {
                    short pixel = shortBuffer.get();
                    this.getPixelBytes(pixel, this.bufferStorage, pixelBytes);
                    captureImage.writePixel(pixelBytes);
                }
                captureImage.endLine();
            }
        } else if (this.isIntBuffer() && !imageType32Bit) {
            IntBuffer intBuffer = this.getIntBuffer();
            for (int y = 0; y < this.height; ++y) {
                intBuffer.position((imageInvert ? this.height - y - 1 : y) * this.bufferWidth / 2);
                captureImage.startLine(imageInvert ? this.height - y - 1 : y);
                for (int x = 0; x < readWidth; x += 2) {
                    try {
                        int twoPixels = intBuffer.get();
                        this.getPixelBytes((short)twoPixels, this.bufferStorage, pixelBytes);
                        captureImage.writePixel(pixelBytes);
                        if (x + 1 >= readWidth) continue;
                        this.getPixelBytes((short)(twoPixels >>> 16), this.bufferStorage, pixelBytes);
                        captureImage.writePixel(pixelBytes);
                        continue;
                    }
                    catch (BufferUnderflowException e) {
                        captureImage.writePixel(blackPixelBytes);
                        captureImage.writePixel(blackPixelBytes);
                    }
                }
                captureImage.endLine();
            }
        } else if (imageType32Bit) {
            for (int y = 0; y < this.height; ++y) {
                IMemoryReader memoryReader = MemoryReader.getMemoryReader(this.imageaddr + (imageInvert ? this.height - y - 1 : y) * this.bufferWidth * 4, this.bufferWidth * 4, 4);
                captureImage.startLine(imageInvert ? this.height - y - 1 : y);
                for (int x = 0; x < readWidth; ++x) {
                    int pixel = memoryReader.readNext();
                    captureImage.writePixel(pixel);
                }
                captureImage.endLine();
            }
        } else {
            for (int y = 0; y < this.height; ++y) {
                IMemoryReader memoryReader = MemoryReader.getMemoryReader(this.imageaddr + (imageInvert ? this.height - y - 1 : y) * this.bufferWidth * 2, this.bufferWidth * 2, 2);
                captureImage.startLine(imageInvert ? this.height - y - 1 : y);
                for (int x = 0; x < readWidth; ++x) {
                    short pixel = (short)memoryReader.readNext();
                    this.getPixelBytes(pixel, this.bufferStorage, pixelBytes);
                    captureImage.writePixel(pixelBytes);
                }
                captureImage.endLine();
            }
        }
        if (this.buffer != null) {
            this.buffer.rewind();
        }
        captureImage.writeEnd();
        if (log.isDebugEnabled()) {
            log.debug((Object)String.format("Saved image to %s", this.getFileName()));
        }
    }

    private static void storeLittleEndianInt(byte[] buffer, int offset, int value) {
        buffer[offset] = (byte)value;
        buffer[offset + 1] = (byte)(value >> 8);
        buffer[offset + 2] = (byte)(value >> 16);
        buffer[offset + 3] = (byte)(value >> 24);
    }

    private static void storeBigEndianInt(byte[] buffer, int offset, int value) {
        buffer[offset] = (byte)(value >> 24);
        buffer[offset + 1] = (byte)(value >> 16);
        buffer[offset + 2] = (byte)(value >> 8);
        buffer[offset + 3] = (byte)value;
    }

    private static void storeChunkType(byte[] buffer, int offset, char c1, char c2, char c3, char c4) {
        buffer[offset] = (byte)c1;
        buffer[offset + 1] = (byte)c2;
        buffer[offset + 2] = (byte)c3;
        buffer[offset + 3] = (byte)c4;
    }

    private static void storeCRC(byte[] buffer, int offset) {
        CRC32 crc32 = new CRC32();
        crc32.update(buffer, 4, offset - 4);
        CaptureImage.storeBigEndianInt(buffer, offset, (int)crc32.getValue());
    }

    private static void storeLittleEndianShort(byte[] buffer, int offset, int value) {
        buffer[offset] = (byte)value;
        buffer[offset + 1] = (byte)(value >> 8);
    }

    private void getPixelBytes(short pixel, int imageType, byte[] pixelBytes) {
        switch (imageType) {
            case 0: {
                pixelBytes[0] = (byte)(pixel >> 8 & 0xF8);
                pixelBytes[1] = (byte)(pixel >> 3 & 0xFC);
                pixelBytes[2] = (byte)(pixel << 3 & 0xF8);
                pixelBytes[3] = 0;
                break;
            }
            case 1: {
                pixelBytes[0] = (byte)(pixel >> 7 & 0xF8);
                pixelBytes[1] = (byte)(pixel >> 2 & 0xF8);
                pixelBytes[2] = (byte)(pixel << 3 & 0xF8);
                pixelBytes[3] = (byte)(pixel >> 15 != 0 ? 255 : 0);
                break;
            }
            case 2: {
                pixelBytes[0] = (byte)(pixel >> 4 & 0xF0);
                pixelBytes[1] = (byte)(pixel & 0xF0);
                pixelBytes[2] = (byte)(pixel << 4 & 0xF0);
                pixelBytes[3] = (byte)(pixel >> 8 & 0xF0);
                break;
            }
            case 6: 
            case 15: {
                pixelBytes[0] = (byte)(pixel >> 8);
                pixelBytes[1] = pixelBytes[0];
                pixelBytes[2] = pixelBytes[0];
                pixelBytes[3] = pixelBytes[0];
                break;
            }
            default: {
                pixelBytes[0] = 0;
                pixelBytes[1] = 0;
                pixelBytes[2] = 0;
                pixelBytes[3] = 0;
            }
        }
    }

    private static int getARGB(byte[] pixelBytes) {
        return (pixelBytes[3] & 0xFF) << 24 | (pixelBytes[2] & 0xFF) << 16 | (pixelBytes[1] & 0xFF) << 8 | pixelBytes[0] & 0xFF;
    }

    private void storePixel(IntBuffer buffer, int x, int y, int color) {
        buffer.put(y * this.width + x, color);
    }

    private int round4(int n) {
        return Utilities.alignUp(n, 3);
    }

    private int getInt32(Buffer buffer) {
        if (buffer instanceof IntBuffer) {
            return ((IntBuffer)buffer).get();
        }
        if (buffer instanceof ShortBuffer) {
            ShortBuffer shortBuffer = (ShortBuffer)buffer;
            int n0 = shortBuffer.get() & 0xFFFF;
            int n1 = shortBuffer.get() & 0xFFFF;
            return n1 << 16 | n0;
        }
        if (buffer instanceof ByteBuffer) {
            return ((ByteBuffer)buffer).getInt();
        }
        return 0;
    }

    private void decompressImageDXT(int dxtLevel) {
        IntBuffer decompressedBuffer = IntBuffer.allocate(this.round4(this.width) * this.round4(this.height));
        int strideX = 0;
        int strideY = 0;
        int[] colors = new int[4];
        int strideSize = dxtLevel == 1 ? 8 : 16;
        int[] alphas = new int[16];
        int[] alphasLookup = new int[8];
        for (int i = 0; i < this.compressedImageSize; i += strideSize) {
            int b3;
            int g3;
            int r3;
            int b2;
            int g2;
            int r2;
            if (dxtLevel > 1) {
                if (dxtLevel <= 3) {
                    int alphaBits = 0;
                    int j = 0;
                    while (j < 16) {
                        if (j % 8 == 0) {
                            alphaBits = this.getInt32(this.buffer);
                        }
                        int alpha = alphaBits & 0xF;
                        alphas[j] = alpha << 4;
                        ++j;
                        alphaBits >>>= 4;
                    }
                } else {
                    int bits0 = this.getInt32(this.buffer);
                    int bits1 = this.getInt32(this.buffer);
                    int alpha0 = bits0 & 0xFF;
                    int alpha1 = bits0 >> 8 & 0xFF;
                    alphasLookup[0] = alpha0;
                    alphasLookup[1] = alpha1;
                    if (alpha0 > alpha1) {
                        alphasLookup[2] = (6 * alpha0 + 1 * alpha1) / 7;
                        alphasLookup[3] = (5 * alpha0 + 2 * alpha1) / 7;
                        alphasLookup[4] = (4 * alpha0 + 3 * alpha1) / 7;
                        alphasLookup[5] = (3 * alpha0 + 4 * alpha1) / 7;
                        alphasLookup[6] = (2 * alpha0 + 5 * alpha1) / 7;
                        alphasLookup[7] = (1 * alpha0 + 6 * alpha1) / 7;
                    } else {
                        alphasLookup[2] = (4 * alpha0 + 1 * alpha1) / 5;
                        alphasLookup[3] = (3 * alpha0 + 2 * alpha1) / 5;
                        alphasLookup[4] = (2 * alpha0 + 3 * alpha1) / 5;
                        alphasLookup[5] = (1 * alpha0 + 4 * alpha1) / 5;
                        alphasLookup[6] = 0;
                        alphasLookup[7] = 255;
                    }
                    int bits = bits0 >> 16;
                    for (int j = 0; j < 16; ++j) {
                        int lookup;
                        if (j == 5) {
                            lookup = (bits & 1) << 2 | bits1 & 3;
                            bits = bits1 >>> 2;
                        } else {
                            lookup = bits & 7;
                            bits >>>= 3;
                        }
                        alphas[j] = alphasLookup[lookup];
                    }
                }
            }
            int color = this.getInt32(this.buffer);
            int color0 = color >> 0 & 0xFFFF;
            int color1 = color >> 16 & 0xFFFF;
            int r0 = color0 >> 8 & 0xF8;
            int g0 = color0 >> 3 & 0xFC;
            int b0 = color0 << 3 & 0xF8;
            int r1 = color1 >> 8 & 0xF8;
            int g1 = color1 >> 3 & 0xFC;
            int b1 = color1 << 3 & 0xF8;
            if (color0 > color1 || dxtLevel > 1) {
                r2 = (r0 * 2 + r1) / 3;
                g2 = (g0 * 2 + g1) / 3;
                b2 = (b0 * 2 + b1) / 3;
            } else {
                r2 = (r0 + r1) / 2;
                g2 = (g0 + g1) / 2;
                b2 = (b0 + b1) / 2;
            }
            if (color0 > color1 || dxtLevel > 1) {
                r3 = (r0 + r1 * 2) / 3;
                g3 = (g0 + g1 * 2) / 3;
                b3 = (b0 + b1 * 2) / 3;
            } else {
                r3 = 0;
                g3 = 0;
                b3 = 0;
            }
            colors[0] = (b0 & 0xFF) << 16 | (g0 & 0xFF) << 8 | r0 & 0xFF;
            colors[1] = (b1 & 0xFF) << 16 | (g1 & 0xFF) << 8 | r1 & 0xFF;
            colors[2] = (b2 & 0xFF) << 16 | (g2 & 0xFF) << 8 | r2 & 0xFF;
            colors[3] = (b3 & 0xFF) << 16 | (g3 & 0xFF) << 8 | r3 & 0xFF;
            int bits = this.getInt32(this.buffer);
            int alphaIndex = 0;
            for (int y = 0; y < 4; ++y) {
                int x = 0;
                while (x < 4) {
                    int bgr = colors[bits & 3];
                    int alpha = alphas[alphaIndex] << 24;
                    this.storePixel(decompressedBuffer, strideX + x, strideY + y, bgr | alpha);
                    ++x;
                    bits >>>= 2;
                    ++alphaIndex;
                }
            }
            if ((strideX += 4) < this.width) continue;
            strideX = 0;
            strideY += 4;
        }
        this.buffer.rewind();
        this.compressedImage = false;
        this.buffer = decompressedBuffer;
        this.bufferWidth = this.width;
        this.bufferStorage = 3;
    }

    private void decompressImage() {
        switch (this.bufferStorage) {
            case 8: {
                this.decompressImageDXT(1);
                break;
            }
            case 9: {
                this.decompressImageDXT(3);
                break;
            }
            case 10: {
                this.decompressImageDXT(5);
                break;
            }
            default: {
                log.warn((Object)String.format("Unsupported compressed buffer storage %d", this.bufferStorage));
            }
        }
    }

    private static class CaptureImagePNG
    extends AbstractCaptureImage {
        private int width;
        private int height;
        private String fileName;
        private byte[] buffer;
        private int pixelIndex;

        private CaptureImagePNG() {
        }

        @Override
        public void writeHeader(String fileName, String fileFormat, int width, int height, int readWidth) throws IOException {
            this.fileName = fileName;
            this.width = width;
            this.height = height;
            this.buffer = new byte[width * height * 4 + height];
        }

        @Override
        public void startLine(int y) {
            this.pixelIndex = this.width * y * 4 + y;
            this.buffer[this.pixelIndex] = 0;
            ++this.pixelIndex;
        }

        @Override
        public void writePixel(int pixel) throws IOException {
            this.buffer[this.pixelIndex + 0] = (byte)pixel;
            this.buffer[this.pixelIndex + 1] = (byte)(pixel >> 8);
            this.buffer[this.pixelIndex + 2] = (byte)(pixel >> 16);
            this.buffer[this.pixelIndex + 3] = (byte)(pixel >> 24);
            this.pixelIndex += 4;
        }

        @Override
        public void writeEnd() throws IOException {
            byte[] fileHeader = new byte[]{-119, 80, 78, 71, 13, 10, 26, 10};
            byte[] ihdr = new byte[25];
            CaptureImage.storeBigEndianInt(ihdr, 0, 13);
            CaptureImage.storeChunkType(ihdr, 4, 'I', 'H', 'D', 'R');
            CaptureImage.storeBigEndianInt(ihdr, 8, this.width);
            CaptureImage.storeBigEndianInt(ihdr, 12, this.height);
            ihdr[16] = 8;
            ihdr[17] = 6;
            ihdr[18] = 0;
            ihdr[19] = 0;
            ihdr[20] = 0;
            CaptureImage.storeCRC(ihdr, 21);
            Deflater deflater = new Deflater();
            deflater.setInput(this.buffer);
            deflater.finish();
            byte[] data = new byte[this.buffer.length];
            int dataLength = deflater.deflate(data);
            byte[] idat = new byte[8 + dataLength + 4];
            CaptureImage.storeBigEndianInt(idat, 0, dataLength);
            CaptureImage.storeChunkType(idat, 4, 'I', 'D', 'A', 'T');
            System.arraycopy(data, 0, idat, 8, dataLength);
            CaptureImage.storeCRC(idat, 8 + dataLength);
            byte[] iend = new byte[12];
            CaptureImage.storeBigEndianInt(iend, 0, 0);
            CaptureImage.storeChunkType(iend, 4, 'I', 'E', 'N', 'D');
            CaptureImage.storeCRC(iend, 8);
            FileOutputStream outStream = new FileOutputStream(this.fileName);
            ((OutputStream)outStream).write(fileHeader);
            ((OutputStream)outStream).write(ihdr);
            ((OutputStream)outStream).write(idat);
            ((OutputStream)outStream).write(iend);
            ((OutputStream)outStream).close();
        }
    }

    private static class CaptureImageImageIO
    extends AbstractCaptureImage {
        private BufferedImage im;
        private int[] lineARGB;
        private int x;
        private int y;
        private int width;
        private String fileName;
        private String fileFormat;

        private CaptureImageImageIO() {
        }

        @Override
        public void writeHeader(String fileName, String fileFormat, int width, int height, int readWidth) throws IOException {
            this.fileName = fileName;
            this.fileFormat = fileFormat;
            this.width = width;
            this.im = new BufferedImage(width, height, 5);
            this.lineARGB = new int[width];
        }

        @Override
        public void startLine(int y) {
            this.y = y;
            this.x = 0;
        }

        @Override
        public void writePixel(int pixel) throws IOException {
            this.lineARGB[this.x] = ImageReader.colorABGRtoARGB(pixel);
            ++this.x;
        }

        @Override
        public void endLine() {
            this.im.setRGB(0, this.y, this.width, 1, this.lineARGB, 0, this.width);
        }

        @Override
        public void writeEnd() throws IOException {
            if (!ImageIO.write((RenderedImage)this.im, this.fileFormat, new File(this.fileName))) {
                log.error((Object)String.format("Cannot save image in format %s using ImageIO: %s", this.fileFormat, this.fileName));
            }
        }
    }

    private static class CaptureImageBMP
    extends AbstractCaptureImage {
        private int imageRawBytes;
        private byte[] completeImageBytes;
        private int pixelIndex;
        private int width;
        private int height;
        private String fileName;

        private CaptureImageBMP() {
        }

        @Override
        public void writeHeader(String fileName, String fileFormat, int width, int height, int readWidth) throws IOException {
            this.fileName = fileName;
            this.width = width;
            this.height = height;
            int rowPad = 4 - (width * 4 & 3) & 3;
            this.imageRawBytes = width * 4 + rowPad;
            this.completeImageBytes = new byte[height * this.imageRawBytes];
        }

        @Override
        public void startLine(int y) {
            this.pixelIndex = y * this.imageRawBytes;
        }

        @Override
        public void writePixel(int pixel) {
            this.completeImageBytes[this.pixelIndex + 0] = (byte)(pixel >> 16);
            this.completeImageBytes[this.pixelIndex + 1] = (byte)(pixel >> 8);
            this.completeImageBytes[this.pixelIndex + 2] = (byte)pixel;
            this.completeImageBytes[this.pixelIndex + 3] = (byte)(pixel >> 24);
            this.pixelIndex += 4;
        }

        @Override
        public void writeEnd() throws IOException {
            byte[] fileHeader = new byte[14];
            byte[] dibHeader = new byte[56];
            int fileSize = fileHeader.length + dibHeader.length + this.completeImageBytes.length;
            fileHeader[0] = 66;
            fileHeader[1] = 77;
            CaptureImage.storeLittleEndianInt(fileHeader, 2, fileSize);
            CaptureImage.storeLittleEndianInt(fileHeader, 10, fileHeader.length + dibHeader.length);
            CaptureImage.storeLittleEndianInt(dibHeader, 0, dibHeader.length);
            CaptureImage.storeLittleEndianInt(dibHeader, 4, this.width);
            CaptureImage.storeLittleEndianInt(dibHeader, 8, this.height);
            CaptureImage.storeLittleEndianShort(dibHeader, 12, 1);
            CaptureImage.storeLittleEndianShort(dibHeader, 14, 32);
            CaptureImage.storeLittleEndianInt(dibHeader, 16, 0);
            CaptureImage.storeLittleEndianInt(dibHeader, 20, this.completeImageBytes.length);
            CaptureImage.storeLittleEndianInt(dibHeader, 24, 2835);
            CaptureImage.storeLittleEndianInt(dibHeader, 28, 2835);
            CaptureImage.storeLittleEndianInt(dibHeader, 32, 0);
            CaptureImage.storeLittleEndianInt(dibHeader, 36, 0);
            CaptureImage.storeLittleEndianInt(dibHeader, 40, 0xFF0000);
            CaptureImage.storeLittleEndianInt(dibHeader, 44, 65280);
            CaptureImage.storeLittleEndianInt(dibHeader, 48, 255);
            CaptureImage.storeLittleEndianInt(dibHeader, 52, -16777216);
            FileOutputStream outStream = new FileOutputStream(this.fileName);
            ((OutputStream)outStream).write(fileHeader);
            ((OutputStream)outStream).write(dibHeader);
            ((OutputStream)outStream).write(this.completeImageBytes);
            ((OutputStream)outStream).close();
        }

        @Override
        public boolean isInverted() {
            return true;
        }
    }

    private static abstract class AbstractCaptureImage {
        private AbstractCaptureImage() {
        }

        public abstract void writeHeader(String var1, String var2, int var3, int var4, int var5) throws IOException;

        public abstract void startLine(int var1);

        public abstract void writePixel(int var1) throws IOException;

        public abstract void writeEnd() throws IOException;

        public void writePixel(byte[] pixel) throws IOException {
            this.writePixel(CaptureImage.getARGB(pixel));
        }

        public void endLine() throws IOException {
        }

        public boolean isInverted() {
            return false;
        }
    }
}

