/*
 * Decompiled with CFR 0.152.
 */
package libchdr;

import java.io.FileNotFoundException;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jpcsp.HLE.VFS.IVirtualFile;
import jpcsp.HLE.VFS.local.LocalVirtualFile;
import jpcsp.filesystems.SeekableRandomFile;
import jpcsp.util.Utilities;
import libchdr.Bitstream;
import libchdr.ChdHeader;
import libchdr.HuffmanDecoder;
import libchdr.codec.CdFlac;
import libchdr.codec.CdLzma;
import libchdr.codec.CdZlib;
import libchdr.codec.Flac;
import libchdr.codec.ICodecInterface;
import libchdr.codec.Lzma;
import libchdr.codec.Zlib;
import org.apache.log4j.Logger;

public class Chd {
    public static Logger log = Logger.getLogger((String)"libchdr");
    public static final int MAP_STACK_ENTRIES = 512;
    public static final int MAP_ENTRY_SIZE = 16;
    public static final int OLD_MAP_ENTRY_SIZE = 8;
    public static final int METADATA_HEADER_SIZE = 16;
    public static final int MAP_ENTRY_FLAG_TYPE_MASK = 15;
    public static final int MAP_ENTRY_FLAG_NO_CRC = 16;
    public static final int CHD_V1_SECTOR_SIZE = 512;
    public static final int COOKIE_VALUE = -1163005939;
    public static final byte[] nullmd5 = new byte[16];
    public static final byte[] nullsha1 = new byte[20];
    private static final CodecInterface[] codecInterfaces = new CodecInterface[]{new CodecInterface(0, "none", null), new CodecInterface(1, "zlib", new Zlib()), new CodecInterface(2, "zlib+", new Zlib()), new CodecInterface(ChdHeader.CHD_CODEC_ZLIB, "zlib (Deflate)", new Zlib()), new CodecInterface(ChdHeader.CHD_CODEC_LZMA, "lzma (LZMA)", new Lzma()), new CodecInterface(ChdHeader.CHD_CODEC_FLAC, "flac (FLAC)", new Flac()), new CodecInterface(ChdHeader.CHD_CODEC_CD_ZLIB, "cdzl (CD Deflate)", new CdZlib()), new CodecInterface(ChdHeader.CHD_CODEC_CD_LZMA, "cdlz (CD LZMA)", new CdLzma()), new CodecInterface(ChdHeader.CHD_CODEC_CD_FLAC, "cdfl (CD FLAC)", new CdFlac())};

    private static long get_bigendian_uint64(byte[] base, int baseOffset) {
        return ((long)Chd.get_bigendian_uint32(base, baseOffset) & 0xFFFFFFFFL) << 32 | (long)Chd.get_bigendian_uint32(base, baseOffset + 4) & 0xFFFFFFFFL;
    }

    private static void put_bigendian_uint64(byte[] base, int baseOffset, long value) {
        Chd.put_bigendian_uint32(base, baseOffset, (int)(value >> 32));
        Chd.put_bigendian_uint32(base, baseOffset + 4, (int)value);
    }

    private static long get_bigendian_uint48(byte[] base, int baseOffset) {
        return ((long)Chd.get_bigendian_uint16(base, baseOffset) & 0xFFFFFFFFL) << 32 | (long)Chd.get_bigendian_uint32(base, baseOffset + 2) & 0xFFFFFFFFL;
    }

    private static void put_bigendian_uint48(byte[] base, int baseOffset, long value) {
        Chd.put_bigendian_uint16(base, baseOffset, (int)(value >> 32));
        Chd.put_bigendian_uint32(base, baseOffset + 2, (int)value);
    }

    private static int get_bigendian_uint32(byte[] base, int baseOffset) {
        return Utilities.read8(base, baseOffset) << 24 | Utilities.read8(base, baseOffset + 1) << 16 | Utilities.read8(base, baseOffset + 2) << 8 | Utilities.read8(base, baseOffset + 3);
    }

    private static void put_bigendian_uint32(byte[] base, int baseOffset, int value) {
        base[baseOffset + 0] = (byte)(value >> 24);
        base[baseOffset + 1] = (byte)(value >> 16);
        base[baseOffset + 2] = (byte)(value >> 8);
        base[baseOffset + 3] = (byte)value;
    }

    private static int get_bigendian_uint24(byte[] base, int baseOffset) {
        return Utilities.read8(base, baseOffset) << 16 | Utilities.read8(base, baseOffset + 1) << 8 | Utilities.read8(base, baseOffset + 2);
    }

    private static void put_bigendian_uint24(byte[] base, int baseOffset, int value) {
        base[baseOffset + 0] = (byte)(value >> 16);
        base[baseOffset + 1] = (byte)(value >> 8);
        base[baseOffset + 2] = (byte)value;
    }

    private static int get_bigendian_uint16(byte[] base, int baseOffset) {
        return Utilities.read8(base, baseOffset) << 8 | Utilities.read8(base, baseOffset + 1);
    }

    private static void put_bigendian_uint16(byte[] base, int baseOffset, int value) {
        base[baseOffset + 0] = (byte)(value >> 8);
        base[baseOffset + 1] = (byte)value;
    }

    private void map_extract(byte[] base, int baseOffset, MapEntry entry) {
        entry.offset = Chd.get_bigendian_uint64(base, baseOffset + 0);
        entry.crc = Chd.get_bigendian_uint32(base, baseOffset + 8);
        entry.length = Chd.get_bigendian_uint16(base, baseOffset + 12) | Utilities.read8(base, baseOffset + 14) << 16;
        entry.flags = Utilities.read8(base, baseOffset + 15);
    }

    private int map_size_v5(ChdHeader header) {
        return header.hunkcount * header.mapentrybytes;
    }

    private int crc16(byte[] data, int dataOffset, int length) {
        int crc = 65535;
        int[] s_table = new int[]{0, 4129, 8258, 12387, 16516, 20645, 24774, 28903, 33032, 37161, 41290, 45419, 49548, 53677, 57806, 61935, 4657, 528, 12915, 8786, 21173, 17044, 29431, 25302, 37689, 33560, 45947, 41818, 54205, 50076, 62463, 58334, 9314, 13379, 1056, 5121, 25830, 29895, 17572, 21637, 42346, 46411, 34088, 38153, 58862, 62927, 50604, 54669, 13907, 9842, 5649, 1584, 30423, 26358, 22165, 18100, 46939, 42874, 38681, 34616, 63455, 59390, 55197, 51132, 18628, 22757, 26758, 30887, 2112, 6241, 10242, 14371, 51660, 55789, 59790, 63919, 35144, 39273, 43274, 47403, 23285, 19156, 31415, 27286, 6769, 2640, 14899, 10770, 56317, 52188, 64447, 60318, 39801, 35672, 47931, 43802, 27814, 31879, 19684, 23749, 11298, 15363, 3168, 7233, 60846, 64911, 52716, 56781, 44330, 48395, 36200, 40265, 32407, 28342, 24277, 20212, 15891, 11826, 7761, 3696, 65439, 61374, 57309, 53244, 48923, 44858, 40793, 36728, 37256, 33193, 45514, 41451, 53516, 49453, 61774, 57711, 4224, 161, 12482, 8419, 20484, 16421, 28742, 24679, 33721, 37784, 41979, 46042, 49981, 54044, 58239, 62302, 689, 4752, 8947, 13010, 16949, 21012, 25207, 29270, 46570, 42443, 38312, 34185, 62830, 58703, 54572, 50445, 13538, 9411, 5280, 1153, 29798, 25671, 21540, 17413, 42971, 47098, 34713, 38840, 59231, 63358, 50973, 55100, 9939, 14066, 1681, 5808, 26199, 30326, 17941, 22068, 55628, 51565, 63758, 59695, 39368, 35305, 47498, 43435, 22596, 18533, 30726, 26663, 6336, 2273, 14466, 10403, 52093, 56156, 60223, 64286, 35833, 39896, 43963, 48026, 19061, 23124, 27191, 31254, 2801, 6864, 10931, 14994, 64814, 60687, 56684, 52557, 48554, 44427, 40424, 36297, 31782, 27655, 23652, 19525, 15522, 11395, 7392, 3265, 61215, 65342, 53085, 57212, 44955, 49082, 36825, 40952, 28183, 32310, 20053, 24180, 11923, 16050, 3793, 7920};
        while (length-- != 0) {
            crc = crc << 8 & 0xFFFF ^ s_table[crc >> 8 ^ Utilities.read8(data, dataOffset++)];
        }
        return crc & 0xFFFF;
    }

    private boolean chd_compressed(ChdHeader header) {
        return header.compression[0] != 0;
    }

    private ChdHeader.ChdError decompress_v5_map(ChdFile chd, ChdHeader header) {
        int rawmapOffset;
        byte[] rawmap;
        int hunknum;
        int result = 0;
        int repcount = 0;
        int lastcomp = 0;
        int last_self = 0;
        long last_parent = 0L;
        byte[] rawbuf = new byte[16];
        int rawmapsize = this.map_size_v5(header);
        if (!this.chd_compressed(header)) {
            header.rawmap = new byte[rawmapsize];
            chd.file.ioLseek(header.mapoffset);
            result = chd.file.ioRead(header.rawmap, 0, rawmapsize);
            if (result != rawmapsize) {
                return ChdHeader.ChdError.CHDERR_READ_ERROR;
            }
            return ChdHeader.ChdError.CHDERR_NONE;
        }
        chd.file.ioLseek(header.mapoffset);
        result = chd.file.ioRead(rawbuf, 0, rawbuf.length);
        if (result != rawbuf.length) {
            return ChdHeader.ChdError.CHDERR_READ_ERROR;
        }
        int mapbytes = Chd.get_bigendian_uint32(rawbuf, 0);
        long firstoffs = Chd.get_bigendian_uint48(rawbuf, 4);
        int mapcrc = Chd.get_bigendian_uint16(rawbuf, 10);
        int lengthbits = Utilities.read8(rawbuf, 12);
        int selfbits = Utilities.read8(rawbuf, 13);
        int parentbits = Utilities.read8(rawbuf, 14);
        byte[] compressed_ptr = new byte[mapbytes];
        chd.file.ioLseek(header.mapoffset + 16L);
        result = chd.file.ioRead(compressed_ptr, 0, mapbytes);
        if (result != mapbytes) {
            return ChdHeader.ChdError.CHDERR_READ_ERROR;
        }
        Bitstream bitbuf = new Bitstream(compressed_ptr, 0, mapbytes);
        header.rawmap = new byte[rawmapsize];
        HuffmanDecoder decoder = new HuffmanDecoder(16, 8);
        if (decoder == null) {
            return ChdHeader.ChdError.CHDERR_OUT_OF_MEMORY;
        }
        HuffmanDecoder.HuffmanError err = decoder.import_tree_rle(bitbuf);
        if (err != HuffmanDecoder.HuffmanError.HUFFERR_NONE) {
            decoder.delete();
            return ChdHeader.ChdError.CHDERR_DECOMPRESSION_ERROR;
        }
        for (hunknum = 0; hunknum < header.hunkcount; ++hunknum) {
            rawmap = header.rawmap;
            rawmapOffset = 0 + hunknum * 12;
            if (repcount > 0) {
                Utilities.write8(rawmap, rawmapOffset + 0, lastcomp);
                --repcount;
                continue;
            }
            int val = decoder.decode_one(bitbuf);
            if (val == V5CompressionType.COMPRESSION_RLE_SMALL.ordinal()) {
                Utilities.write8(rawmap, rawmapOffset + 0, lastcomp);
                repcount = 2 + decoder.decode_one(bitbuf);
                continue;
            }
            if (val == V5CompressionType.COMPRESSION_RLE_LARGE.ordinal()) {
                Utilities.write8(rawmap, rawmapOffset + 0, lastcomp);
                repcount = 18 + (decoder.decode_one(bitbuf) << 4);
                repcount += decoder.decode_one(bitbuf);
                continue;
            }
            lastcomp = val;
            Utilities.write8(rawmap, rawmapOffset + 0, lastcomp);
        }
        long curoffset = firstoffs;
        for (hunknum = 0; hunknum < header.hunkcount; ++hunknum) {
            rawmap = header.rawmap;
            rawmapOffset = 0 + hunknum * 12;
            long offset = curoffset;
            int length = 0;
            int crc = 0;
            switch (V5CompressionType.values()[Utilities.read8(rawmap, rawmapOffset + 0)]) {
                case COMPRESSION_TYPE_0: 
                case COMPRESSION_TYPE_1: 
                case COMPRESSION_TYPE_2: 
                case COMPRESSION_TYPE_3: {
                    length = bitbuf.read(lengthbits);
                    curoffset += (long)length;
                    crc = bitbuf.read(16);
                    break;
                }
                case COMPRESSION_NONE: {
                    length = header.hunkbytes;
                    curoffset += (long)length;
                    crc = bitbuf.read(16);
                    break;
                }
                case COMPRESSION_SELF: {
                    offset = bitbuf.read(selfbits);
                    last_self = (int)offset;
                    break;
                }
                case COMPRESSION_PARENT: {
                    last_parent = offset = (long)bitbuf.read(parentbits);
                    break;
                }
                case COMPRESSION_SELF_1: {
                    ++last_self;
                }
                case COMPRESSION_SELF_0: {
                    Utilities.write8(rawmap, rawmapOffset + 0, V5CompressionType.COMPRESSION_SELF.ordinal());
                    offset = last_self;
                    break;
                }
                case COMPRESSION_PARENT_SELF: {
                    Utilities.write8(rawmap, rawmapOffset + 0, V5CompressionType.COMPRESSION_PARENT.ordinal());
                    last_parent = offset = (long)hunknum * (long)header.hunkbytes / (long)header.unitbytes;
                    break;
                }
                case COMPRESSION_PARENT_1: {
                    last_parent += (long)(header.hunkbytes / header.unitbytes);
                }
                case COMPRESSION_PARENT_0: {
                    Utilities.write8(rawmap, rawmapOffset + 0, V5CompressionType.COMPRESSION_PARENT.ordinal());
                    offset = last_parent;
                    break;
                }
            }
            Chd.put_bigendian_uint24(rawmap, rawmapOffset + 1, length);
            Chd.put_bigendian_uint48(rawmap, rawmapOffset + 4, offset);
            Chd.put_bigendian_uint16(rawmap, rawmapOffset + 10, crc);
        }
        decoder.delete();
        if (this.crc16(header.rawmap, 0, header.hunkcount * 12) != mapcrc) {
            return ChdHeader.ChdError.CHDERR_DECOMPRESSION_ERROR;
        }
        return ChdHeader.ChdError.CHDERR_NONE;
    }

    private void map_extract_old(byte[] base, int baseOffset, MapEntry entry, int hunkbytes) {
        entry.offset = Chd.get_bigendian_uint64(base, baseOffset + 0);
        entry.crc = 0;
        entry.length = (int)(entry.offset >> 44);
        entry.flags = 0x10 | (entry.length == hunkbytes ? V34MapEntryType.V34_MAP_ENTRY_TYPE_UNCOMPRESSED.ordinal() : V34MapEntryType.V34_MAP_ENTRY_TYPE_COMPRESSED.ordinal());
        entry.offset &= 0xFFFFFFFFFFFL;
    }

    public ChdHeader.ChdError chd_open_file(String fileName, int mode, ChdFile parent, ChdFile[] chd) {
        LocalVirtualFile vFile;
        if (fileName == null) {
            return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
        }
        switch (mode) {
            case 1: {
                break;
            }
            default: {
                return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
            }
        }
        try {
            vFile = new LocalVirtualFile(new SeekableRandomFile(fileName, "r"));
        }
        catch (FileNotFoundException e) {
            return ChdHeader.ChdError.CHDERR_FILE_NOT_FOUND;
        }
        ChdHeader.ChdError err = this.chd_open_core_file(vFile, mode, parent, chd);
        if (err != ChdHeader.ChdError.CHDERR_NONE && vFile != null) {
            vFile.ioClose();
        }
        return err;
    }

    public void chd_close(ChdFile chd) {
        if (chd == null || chd.cookie != -1163005939) {
            return;
        }
        if (chd.header.version < 5) {
            if (chd.codecintf[0] != null && chd.codecintf[0].codec != null) {
                chd.codecintf[0].codec.free(chd.zlibCodecData);
            }
        } else {
            for (int i = 0; i < chd.codecintf.length; ++i) {
                CodecData codec = null;
                if (chd.codecintf[i] == null) continue;
                int compression = chd.codecintf[i].compression;
                if (compression == ChdHeader.CHD_CODEC_ZLIB) {
                    codec = chd.zlibCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_LZMA) {
                    codec = chd.lzmaCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_FLAC) {
                    codec = chd.flacCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_ZLIB) {
                    codec = chd.cdzlCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_LZMA) {
                    codec = chd.cdlzCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_FLAC) {
                    codec = chd.cdflCodecData;
                }
                if (codec == null || chd.codecintf[i].codec == null) continue;
                chd.codecintf[i].codec.free(codec);
            }
            if (chd.header.rawmap != null) {
                chd.header.rawmap = null;
            }
        }
        if (chd.compressed != null) {
            chd.compressed = null;
        }
        if (chd.map != null) {
            chd.map = null;
        }
        if (chd.file != null) {
            chd.file.ioClose();
        }
        if (chd.parent != null) {
            this.chd_close(chd.parent);
        }
    }

    public ChdHeader chd_get_header(ChdFile chd) {
        if (chd == null || chd.cookie != -1163005939) {
            return null;
        }
        return chd.header;
    }

    public ChdHeader.ChdError chd_read(ChdFile chd, int hunknum, byte[] buffer, int bufferOffset) {
        if (chd == null || chd.cookie != -1163005939) {
            return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
        }
        if (hunknum >= chd.header.totalhunks) {
            return ChdHeader.ChdError.CHDERR_HUNK_OUT_OF_RANGE;
        }
        return this.hunk_read_into_memory(chd, hunknum, buffer, bufferOffset);
    }

    private ChdHeader.ChdError chd_open_core_file(IVirtualFile file, int mode, ChdFile parent, ChdFile[] chd) {
        if (file == null) {
            return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
        }
        if (parent != null && parent.cookie != -1163005939) {
            return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
        }
        ChdFile newChd = new ChdFile();
        newChd.cookie = -1163005939;
        newChd.parent = parent;
        newChd.file = file;
        newChd.header = new ChdHeader();
        ChdHeader.ChdError err = this.header_read(newChd, newChd.header);
        if (err != ChdHeader.ChdError.CHDERR_NONE) {
            return err;
        }
        err = this.header_validate(newChd.header);
        if (err != ChdHeader.ChdError.CHDERR_NONE) {
            return err;
        }
        if (mode == 2 && !Utilities.hasFlag(newChd.header.flags, 2)) {
            return ChdHeader.ChdError.CHDERR_FILE_NOT_WRITEABLE;
        }
        if (mode == 2 && newChd.header.version < 5) {
            return ChdHeader.ChdError.CHDERR_UNSUPPORTED_VERSION;
        }
        if (parent == null) {
            if (newChd.header.version < 5 && Utilities.hasFlag(newChd.header.flags, 1)) {
                return ChdHeader.ChdError.CHDERR_REQUIRES_PARENT;
            }
            if (newChd.header.version >= 5 && !Arrays.equals(nullsha1, newChd.header.parentsha1)) {
                return ChdHeader.ChdError.CHDERR_REQUIRES_PARENT;
            }
        }
        if (parent != null) {
            if (!(Arrays.equals(nullmd5, newChd.header.parentmd5) || Arrays.equals(nullmd5, newChd.parent.header.md5) || Arrays.equals(newChd.parent.header.md5, newChd.header.parentmd5))) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARENT;
            }
            if (!(Arrays.equals(nullsha1, newChd.header.parentsha1) || Arrays.equals(nullsha1, newChd.parent.header.sha1) || Arrays.equals(newChd.parent.header.sha1, newChd.header.parentsha1))) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARENT;
            }
        }
        if (newChd.header.version < 5) {
            err = this.map_read(newChd);
            if (err != ChdHeader.ChdError.CHDERR_NONE) {
                return err;
            }
        } else {
            err = this.decompress_v5_map(newChd, newChd.header);
        }
        if (err != ChdHeader.ChdError.CHDERR_NONE) {
            return err;
        }
        newChd.compressed = new byte[newChd.header.hunkbytes];
        if (newChd.compressed == null) {
            return ChdHeader.ChdError.CHDERR_OUT_OF_MEMORY;
        }
        if (newChd.header.version < 5) {
            int intfnum;
            for (intfnum = 0; intfnum < codecInterfaces.length; ++intfnum) {
                if (Chd.codecInterfaces[intfnum].compression != newChd.header.compression[0]) continue;
                newChd.codecintf[0] = codecInterfaces[intfnum];
                break;
            }
            if (intfnum == codecInterfaces.length) {
                return ChdHeader.ChdError.CHDERR_UNSUPPORTED_FORMAT;
            }
            if (newChd.codecintf[0].codec != null && (err = newChd.codecintf[0].codec.init(newChd.zlibCodecData, newChd.header.hunkbytes)) != ChdHeader.ChdError.CHDERR_NONE) {
                return err;
            }
        } else {
            for (int decompnum = 0; decompnum < newChd.header.compression.length; ++decompnum) {
                for (int i = 0; i < codecInterfaces.length; ++i) {
                    if (Chd.codecInterfaces[i].compression != newChd.header.compression[decompnum]) continue;
                    newChd.codecintf[decompnum] = codecInterfaces[i];
                    break;
                }
                if (newChd.codecintf[decompnum] == null && newChd.header.compression[decompnum] != 0) {
                    return ChdHeader.ChdError.CHDERR_UNSUPPORTED_FORMAT;
                }
                if (newChd.codecintf[decompnum].codec == null) continue;
                CodecData codec = null;
                int compression = newChd.header.compression[decompnum];
                if (compression == ChdHeader.CHD_CODEC_ZLIB) {
                    codec = newChd.zlibCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_LZMA) {
                    codec = newChd.lzmaCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_FLAC) {
                    codec = newChd.flacCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_ZLIB) {
                    codec = newChd.cdzlCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_LZMA) {
                    codec = newChd.cdlzCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_FLAC) {
                    codec = newChd.cdflCodecData;
                }
                if (codec == null) {
                    return ChdHeader.ChdError.CHDERR_UNSUPPORTED_FORMAT;
                }
                err = newChd.codecintf[decompnum].codec.init(codec, newChd.header.hunkbytes);
                if (err == ChdHeader.ChdError.CHDERR_NONE) continue;
                return err;
            }
        }
        chd[0] = newChd;
        return ChdHeader.ChdError.CHDERR_NONE;
    }

    private int header_guess_unitbytes(ChdFile chd) {
        Pattern p;
        Matcher m;
        byte[] metadata = new byte[512];
        if (this.chd_get_metadata(chd, ChdHeader.HARD_DISK_METADATA_TAG, 0, metadata, metadata.length, null, null, null) == ChdHeader.ChdError.CHDERR_NONE && (m = (p = Pattern.compile("CYLS:(\\d+),HEADS:(\\\\d+),SECS:(\\\\d+),BPS:(\\\\d+)")).matcher(new String(metadata))).find()) {
            int i0 = Integer.parseInt(m.group(1));
            int i1 = Integer.parseInt(m.group(2));
            int i2 = Integer.parseInt(m.group(3));
            int i3 = Integer.parseInt(m.group(4));
            if (log.isDebugEnabled()) {
                log.debug((Object)String.format("CD Header i0=%d, i1=%d, i2=%d, i3=%d", i0, i1, i2, i3));
            }
            return i3;
        }
        if (this.chd_get_metadata(chd, ChdHeader.CDROM_OLD_METADATA_TAG, 0, metadata, metadata.length, null, null, null) == ChdHeader.ChdError.CHDERR_NONE || this.chd_get_metadata(chd, ChdHeader.CDROM_TRACK_METADATA_TAG, 0, metadata, metadata.length, null, null, null) == ChdHeader.ChdError.CHDERR_NONE || this.chd_get_metadata(chd, ChdHeader.CDROM_TRACK_METADATA2_TAG, 0, metadata, metadata.length, null, null, null) == ChdHeader.ChdError.CHDERR_NONE || this.chd_get_metadata(chd, ChdHeader.GDROM_OLD_METADATA_TAG, 0, metadata, metadata.length, null, null, null) == ChdHeader.ChdError.CHDERR_NONE || this.chd_get_metadata(chd, ChdHeader.GDROM_TRACK_METADATA_TAG, 0, metadata, metadata.length, null, null, null) == ChdHeader.ChdError.CHDERR_NONE) {
            return 2448;
        }
        return chd.header.hunkbytes;
    }

    private ChdHeader.ChdError header_validate(ChdHeader header) {
        if (header.version == 0 || header.version > 5) {
            return ChdHeader.ChdError.CHDERR_UNSUPPORTED_VERSION;
        }
        if (header.version == 1 && header.length != 76 || header.version == 2 && header.length != 80 || header.version == 3 && header.length != 120 || header.version == 4 && header.length != 108 || header.version == 5 && header.length != 124) {
            return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
        }
        if (header.version <= 4) {
            int intfnum;
            if (Utilities.hasFlag(header.flags, -4)) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
            }
            for (intfnum = 0; intfnum < codecInterfaces.length && Chd.codecInterfaces[intfnum].compression != header.compression[0]; ++intfnum) {
            }
            if (intfnum == codecInterfaces.length) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
            }
            if (header.hunkbytes == 0 || header.hunkbytes >= 0x1000000) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
            }
            if (header.totalhunks == 0) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
            }
            if (Utilities.hasFlag(header.flags, 1) && Arrays.equals(header.parentmd5, nullmd5) && Arrays.equals(header.parentsha1, nullsha1)) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
            }
            if (header.version >= 3 && (header.obsolete_cylinders != 0 || header.obsolete_sectors != 0 || header.obsolete_heads != 0 || header.obsolete_hunksize != 0)) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
            }
            if (header.version < 3 && (header.obsolete_cylinders == 0 || header.obsolete_sectors == 0 || header.obsolete_heads == 0 || header.obsolete_hunksize == 0)) {
                return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
            }
        }
        return ChdHeader.ChdError.CHDERR_NONE;
    }

    private ChdHeader.ChdError header_read(ChdFile chd, ChdHeader header) {
        byte[] rawHeader = new byte[124];
        if (header == null) {
            return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
        }
        if (chd.file == null) {
            return ChdHeader.ChdError.CHDERR_INVALID_FILE;
        }
        chd.file.ioLseek(0L);
        int count = chd.file.ioRead(rawHeader, 0, rawHeader.length);
        if (count != rawHeader.length) {
            return ChdHeader.ChdError.CHDERR_READ_ERROR;
        }
        if (!"MComprHD".equals(new String(rawHeader, 0, 8))) {
            return ChdHeader.ChdError.CHDERR_INVALID_DATA;
        }
        header.length = Chd.get_bigendian_uint32(rawHeader, 8);
        header.version = Chd.get_bigendian_uint32(rawHeader, 12);
        if (header.version == 0 || header.version > 5) {
            return ChdHeader.ChdError.CHDERR_UNSUPPORTED_VERSION;
        }
        if (header.version == 1 && header.length != 76 || header.version == 2 && header.length != 80 || header.version == 3 && header.length != 120 || header.version == 4 && header.length != 108 || header.version == 5 && header.length != 124) {
            return ChdHeader.ChdError.CHDERR_INVALID_DATA;
        }
        header.flags = Chd.get_bigendian_uint32(rawHeader, 16);
        header.compression[0] = Chd.get_bigendian_uint32(rawHeader, 20);
        header.compression[1] = 0;
        header.compression[2] = 0;
        header.compression[3] = 0;
        if (header.version < 3) {
            int seclen = header.version == 1 ? 512 : Chd.get_bigendian_uint32(rawHeader, 76);
            header.obsolete_hunksize = Chd.get_bigendian_uint32(rawHeader, 24);
            header.totalhunks = Chd.get_bigendian_uint32(rawHeader, 28);
            header.obsolete_cylinders = Chd.get_bigendian_uint32(rawHeader, 32);
            header.obsolete_heads = Chd.get_bigendian_uint32(rawHeader, 36);
            header.obsolete_sectors = Chd.get_bigendian_uint32(rawHeader, 40);
            System.arraycopy(rawHeader, 44, header.md5, 0, 16);
            System.arraycopy(rawHeader, 60, header.parentmd5, 0, 16);
            header.logicalbytes = (long)header.obsolete_cylinders * (long)header.obsolete_heads * (long)header.obsolete_sectors * (long)seclen;
            header.hunkbytes = seclen * header.obsolete_hunksize;
            header.unitbytes = this.header_guess_unitbytes(chd);
            header.unitcount = (header.logicalbytes + (long)header.unitbytes - 1L) / (long)header.unitbytes;
            header.metaoffset = 0L;
        } else if (header.version == 3) {
            header.totalhunks = Chd.get_bigendian_uint32(rawHeader, 24);
            header.logicalbytes = Chd.get_bigendian_uint64(rawHeader, 28);
            header.metaoffset = Chd.get_bigendian_uint64(rawHeader, 36);
            System.arraycopy(rawHeader, 44, header.md5, 0, 16);
            System.arraycopy(rawHeader, 60, header.parentmd5, 0, 16);
            header.hunkbytes = Chd.get_bigendian_uint32(rawHeader, 76);
            header.unitbytes = this.header_guess_unitbytes(chd);
            header.unitcount = (header.logicalbytes + (long)header.unitbytes - 1L) / (long)header.unitbytes;
            System.arraycopy(rawHeader, 80, header.sha1, 0, 20);
            System.arraycopy(rawHeader, 100, header.parentsha1, 0, 20);
        } else if (header.version == 4) {
            header.totalhunks = Chd.get_bigendian_uint32(rawHeader, 24);
            header.logicalbytes = Chd.get_bigendian_uint64(rawHeader, 28);
            header.metaoffset = Chd.get_bigendian_uint64(rawHeader, 36);
            header.hunkbytes = Chd.get_bigendian_uint32(rawHeader, 44);
            header.unitbytes = this.header_guess_unitbytes(chd);
            header.unitcount = (header.logicalbytes + (long)header.unitbytes - 1L) / (long)header.unitbytes;
            System.arraycopy(rawHeader, 48, header.sha1, 0, 20);
            System.arraycopy(rawHeader, 68, header.parentsha1, 0, 20);
            System.arraycopy(rawHeader, 88, header.rawsha1, 0, 20);
        } else if (header.version == 5) {
            header.compression[0] = Chd.get_bigendian_uint32(rawHeader, 16);
            header.compression[1] = Chd.get_bigendian_uint32(rawHeader, 20);
            header.compression[2] = Chd.get_bigendian_uint32(rawHeader, 24);
            header.compression[3] = Chd.get_bigendian_uint32(rawHeader, 28);
            header.logicalbytes = Chd.get_bigendian_uint64(rawHeader, 32);
            header.mapoffset = Chd.get_bigendian_uint64(rawHeader, 40);
            header.metaoffset = Chd.get_bigendian_uint64(rawHeader, 48);
            header.hunkbytes = Chd.get_bigendian_uint32(rawHeader, 56);
            header.hunkcount = (int)((header.logicalbytes + (long)header.hunkbytes - 1L) / (long)header.hunkbytes);
            header.unitbytes = Chd.get_bigendian_uint32(rawHeader, 60);
            header.unitcount = (header.logicalbytes + (long)header.unitbytes - 1L) / (long)header.unitbytes;
            System.arraycopy(rawHeader, 84, header.sha1, 0, 20);
            System.arraycopy(rawHeader, 104, header.parentsha1, 0, 20);
            System.arraycopy(rawHeader, 64, header.rawsha1, 0, 20);
            header.mapentrybytes = this.chd_compressed(header) ? 12 : 4;
            header.totalhunks = header.hunkcount;
        } else {
            log.error((Object)String.format("Unknown version %d", header.version));
        }
        return ChdHeader.ChdError.CHDERR_NONE;
    }

    private byte[] hunk_read_compressed(ChdFile chd, long offset, int size) {
        chd.file.ioLseek(offset);
        int bytes = chd.file.ioRead(chd.compressed, 0, size);
        if (bytes != size) {
            return null;
        }
        return chd.compressed;
    }

    private ChdHeader.ChdError hunk_read_uncompressed(ChdFile chd, long offset, int size, byte[] dest, int destOffset) {
        chd.file.ioLseek(offset);
        int bytes = chd.file.ioRead(dest, destOffset, size);
        if (bytes != size) {
            return ChdHeader.ChdError.CHDERR_READ_ERROR;
        }
        return ChdHeader.ChdError.CHDERR_NONE;
    }

    private ChdHeader.ChdError hunk_read_into_memory(ChdFile chd, int hunknum, byte[] dest, int destOffset) {
        if (chd.file == null) {
            return ChdHeader.ChdError.CHDERR_INVALID_FILE;
        }
        if (hunknum >= chd.header.totalhunks) {
            return ChdHeader.ChdError.CHDERR_HUNK_OUT_OF_RANGE;
        }
        if (dest == null) {
            return ChdHeader.ChdError.CHDERR_INVALID_PARAMETER;
        }
        if (chd.header.version < 5) {
            MapEntry entry = chd.map[hunknum];
            switch (V34MapEntryType.values()[entry.flags & 0xF]) {
                case V34_MAP_ENTRY_TYPE_COMPRESSED: {
                    ZlibCodecData codec = null;
                    byte[] compressed_bytes = this.hunk_read_compressed(chd, entry.offset, entry.length);
                    if (compressed_bytes == null) {
                        return ChdHeader.ChdError.CHDERR_READ_ERROR;
                    }
                    ChdHeader.ChdError err = ChdHeader.ChdError.CHDERR_NONE;
                    codec = chd.zlibCodecData;
                    if (chd.codecintf[0].codec != null) {
                        err = chd.codecintf[0].codec.decompress(codec, compressed_bytes, 0, entry.length, dest, destOffset, chd.header.hunkbytes);
                    }
                    if (err == ChdHeader.ChdError.CHDERR_NONE) break;
                    return err;
                }
                case V34_MAP_ENTRY_TYPE_UNCOMPRESSED: {
                    ChdHeader.ChdError err = this.hunk_read_uncompressed(chd, entry.offset, chd.header.hunkbytes, dest, destOffset);
                    if (err == ChdHeader.ChdError.CHDERR_NONE) break;
                    return err;
                }
                case V34_MAP_ENTRY_TYPE_MINI: {
                    Chd.put_bigendian_uint64(dest, destOffset + 0, entry.offset);
                    for (int bytes = 8; bytes < chd.header.hunkbytes; ++bytes) {
                        dest[destOffset + bytes] = dest[destOffset + bytes - 8];
                    }
                    break;
                }
                case V34_MAP_ENTRY_TYPE_SELF_HUNK: {
                    return this.hunk_read_into_memory(chd, (int)entry.offset, dest, destOffset);
                }
                case V34_MAP_ENTRY_TYPE_PARENT_HUNK: {
                    ChdHeader.ChdError err = this.hunk_read_into_memory(chd.parent, (int)entry.offset, dest, destOffset);
                    if (err == ChdHeader.ChdError.CHDERR_NONE) break;
                    return err;
                }
            }
            return ChdHeader.ChdError.CHDERR_NONE;
        }
        CodecData codec = null;
        byte[] rawmap = chd.header.rawmap;
        int rawmapOffset = chd.header.mapentrybytes * hunknum;
        if (!this.chd_compressed(chd.header)) {
            long blockoffs = ((long)Chd.get_bigendian_uint32(rawmap, rawmapOffset) & 0xFFFFFFFFL) * (long)chd.header.hunkbytes;
            if (blockoffs != 0L) {
                chd.file.ioLseek(blockoffs);
                int result = chd.file.ioRead(dest, destOffset, chd.header.hunkbytes);
                if (result != chd.header.hunkbytes) {
                    return ChdHeader.ChdError.CHDERR_READ_ERROR;
                }
            } else if (chd.parent != null) {
                ChdHeader.ChdError err = this.hunk_read_into_memory(chd.parent, hunknum, dest, destOffset);
                if (err != ChdHeader.ChdError.CHDERR_NONE) {
                    return err;
                }
            } else {
                Arrays.fill(dest, destOffset, destOffset + chd.header.hunkbytes, (byte)0);
            }
            return ChdHeader.ChdError.CHDERR_NONE;
        }
        int blocklen = Chd.get_bigendian_uint24(rawmap, rawmapOffset + 1);
        long blockoffs = Chd.get_bigendian_uint48(rawmap, rawmapOffset + 4);
        int blockcrc = Chd.get_bigendian_uint16(rawmap, rawmapOffset + 10);
        int compressionType = Utilities.read8(rawmap, rawmapOffset + 0);
        switch (V5CompressionType.values()[compressionType]) {
            case COMPRESSION_TYPE_0: 
            case COMPRESSION_TYPE_1: 
            case COMPRESSION_TYPE_2: 
            case COMPRESSION_TYPE_3: {
                ChdHeader.ChdError err;
                byte[] compressed_bytes = this.hunk_read_compressed(chd, blockoffs, blocklen);
                if (compressed_bytes == null) {
                    return ChdHeader.ChdError.CHDERR_READ_ERROR;
                }
                int compression = chd.codecintf[compressionType].compression;
                if (compression == ChdHeader.CHD_CODEC_ZLIB) {
                    codec = chd.zlibCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_LZMA) {
                    codec = chd.lzmaCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_FLAC) {
                    codec = chd.flacCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_ZLIB) {
                    codec = chd.cdzlCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_LZMA) {
                    codec = chd.cdlzCodecData;
                } else if (compression == ChdHeader.CHD_CODEC_CD_FLAC) {
                    codec = chd.cdflCodecData;
                }
                if (codec == null) {
                    return ChdHeader.ChdError.CHDERR_CODEC_ERROR;
                }
                if (log.isDebugEnabled()) {
                    log.debug((Object)String.format("V5 hunk#%d compressed with %s at 0x%X, size=0x%X, decompressed size=0x%X%s", hunknum, chd.codecintf[compressionType], blockoffs, blocklen, chd.header.hunkbytes, chd.parent != null ? ", from child" : ""));
                }
                if ((err = chd.codecintf[compressionType].codec.decompress(codec, compressed_bytes, 0, blocklen, dest, destOffset, chd.header.hunkbytes)) != ChdHeader.ChdError.CHDERR_NONE) {
                    return err;
                }
                if (this.crc16(dest, destOffset, chd.header.hunkbytes) != blockcrc) {
                    return ChdHeader.ChdError.CHDERR_DECOMPRESSION_ERROR;
                }
                return ChdHeader.ChdError.CHDERR_NONE;
            }
            case COMPRESSION_NONE: {
                ChdHeader.ChdError err;
                if (log.isDebugEnabled()) {
                    log.debug((Object)String.format("V5 hunk#%d uncompressed 0x%X, size=0x%X", hunknum, blockoffs, blocklen));
                }
                if ((err = this.hunk_read_uncompressed(chd, blockoffs, blocklen, dest, destOffset)) != ChdHeader.ChdError.CHDERR_NONE) {
                    return err;
                }
                if (this.crc16(dest, destOffset, chd.header.hunkbytes) != blockcrc) {
                    return ChdHeader.ChdError.CHDERR_DECOMPRESSION_ERROR;
                }
                return ChdHeader.ChdError.CHDERR_NONE;
            }
            case COMPRESSION_SELF: {
                if (log.isDebugEnabled()) {
                    log.debug((Object)String.format("V5 hunk#%d compression self #%d", hunknum, blockoffs));
                }
                return this.hunk_read_into_memory(chd, (int)blockoffs, dest, destOffset);
            }
            case COMPRESSION_PARENT: {
                if (chd.parent == null) {
                    return ChdHeader.ChdError.CHDERR_REQUIRES_PARENT;
                }
                int units_in_hunk = chd.header.hunkbytes / chd.header.unitbytes;
                if (log.isDebugEnabled()) {
                    log.debug((Object)String.format("V5 hunk#%d compression parent #%d", hunknum, blockoffs / (long)units_in_hunk));
                }
                if (blockoffs % (long)units_in_hunk == 0L) {
                    return this.hunk_read_into_memory(chd.parent, (int)(blockoffs / (long)units_in_hunk), dest, destOffset);
                }
                int unit_in_hunk = (int)(blockoffs % (long)units_in_hunk);
                byte[] buf = new byte[chd.header.hunkbytes];
                ChdHeader.ChdError err = this.hunk_read_into_memory(chd.parent, (int)(blockoffs / (long)units_in_hunk), buf, 0);
                if (err != ChdHeader.ChdError.CHDERR_NONE) {
                    return err;
                }
                System.arraycopy(buf, unit_in_hunk * chd.header.unitbytes, dest, destOffset, (units_in_hunk - unit_in_hunk) * chd.header.unitbytes);
                err = this.hunk_read_into_memory(chd.parent, (int)(blockoffs / (long)units_in_hunk + 1L), buf, 0);
                if (err != ChdHeader.ChdError.CHDERR_NONE) {
                    return err;
                }
                System.arraycopy(buf, 0, dest, destOffset + (units_in_hunk - unit_in_hunk) * chd.header.unitbytes, unit_in_hunk * chd.header.unitbytes);
                break;
            }
        }
        return ChdHeader.ChdError.CHDERR_NONE;
    }

    public ChdHeader.ChdError chd_get_metadata(ChdFile chd, int searchtag, int searchindex, byte[] output, int outputlen, int[] resultlen, int[] resulttag, int[] resultflags) {
        MetaDataEntry metaentry = new MetaDataEntry();
        ChdHeader.ChdError err = this.metadata_find_entry(chd, searchtag, searchindex, metaentry);
        if (err != ChdHeader.ChdError.CHDERR_NONE) {
            if (chd.header.version < 3 && (searchtag == ChdHeader.HARD_DISK_METADATA_TAG || searchtag == 0) && searchindex == 0) {
                String s = String.format("CYLS:%d,HEADS:%d,SECS:%d,BPS:%d", chd.header.obsolete_cylinders, chd.header.obsolete_heads, chd.header.obsolete_sectors, chd.header.hunkbytes / chd.header.obsolete_hunksize);
                byte[] faux_metadata = s.getBytes();
                faux_metadata = Utilities.add(faux_metadata, (byte)0);
                int faux_length = faux_metadata.length;
                System.arraycopy(faux_metadata, 0, output, 0, Math.min(outputlen, faux_length));
                if (resultlen != null) {
                    resultlen[0] = faux_length;
                }
                if (resulttag != null) {
                    resulttag[0] = ChdHeader.HARD_DISK_METADATA_TAG;
                }
                return ChdHeader.ChdError.CHDERR_NONE;
            }
            return err;
        }
        outputlen = Math.min(outputlen, metaentry.length);
        chd.file.ioLseek(metaentry.offset + 16L);
        int count = chd.file.ioRead(output, 0, outputlen);
        if (count != outputlen) {
            return ChdHeader.ChdError.CHDERR_READ_ERROR;
        }
        if (resultlen != null) {
            resultlen[0] = metaentry.length;
        }
        if (resulttag != null) {
            resulttag[0] = metaentry.metatag;
        }
        if (resultflags != null) {
            resultflags[0] = metaentry.flags;
        }
        return ChdHeader.ChdError.CHDERR_NONE;
    }

    private ChdHeader.ChdError metadata_find_entry(ChdFile chd, int metatag, int metaindex, MetaDataEntry metaentry) {
        metaentry.offset = chd.header.metaoffset;
        metaentry.prev = 0L;
        while (metaentry.offset != 0L) {
            byte[] raw_meta_header = new byte[16];
            chd.file.ioLseek(metaentry.offset);
            int count = chd.file.ioRead(raw_meta_header, 0, raw_meta_header.length);
            if (count != raw_meta_header.length) break;
            metaentry.metatag = Chd.get_bigendian_uint32(raw_meta_header, 0);
            metaentry.length = Chd.get_bigendian_uint32(raw_meta_header, 4);
            metaentry.next = Chd.get_bigendian_uint64(raw_meta_header, 8);
            metaentry.flags = metaentry.length >> 24 & 0xFF;
            metaentry.length &= 0xFFFFFF;
            if ((metatag == 0 || metaentry.metatag == metatag) && metaindex-- == 0) {
                return ChdHeader.ChdError.CHDERR_NONE;
            }
            metaentry.prev = metaentry.offset;
            metaentry.offset = metaentry.next;
        }
        return ChdHeader.ChdError.CHDERR_METADATA_NOT_FOUND;
    }

    private ChdHeader.ChdError map_read(ChdFile chd) {
        int count;
        int entrysize = chd.header.version < 3 ? 8 : 16;
        byte[] raw_map_entries = new byte[8192];
        long maxoffset = 0L;
        byte[] cookie = new byte[16];
        chd.map = new MapEntry[chd.header.totalhunks];
        if (chd.map == null) {
            return ChdHeader.ChdError.CHDERR_OUT_OF_MEMORY;
        }
        long fileoffset = chd.header.length;
        for (int i = 0; i < chd.header.totalhunks; i += 512) {
            int j;
            int entries = chd.header.totalhunks - i;
            if (entries > 512) {
                entries = 512;
            }
            chd.file.ioLseek(fileoffset);
            count = chd.file.ioRead(raw_map_entries, 0, entries * entrysize);
            if (count != entries * entrysize) {
                return ChdHeader.ChdError.CHDERR_READ_ERROR;
            }
            fileoffset += (long)(entries * entrysize);
            if (entrysize == 16) {
                for (j = 0; j < entries; ++j) {
                    this.map_extract(raw_map_entries, j * 16, chd.map[i + j]);
                }
            } else {
                for (j = 0; j < entries; ++j) {
                    this.map_extract_old(raw_map_entries, j * 8, chd.map[i + j], chd.header.hunkbytes);
                }
            }
            for (j = 0; j < entries; ++j) {
                if ((chd.map[i + j].flags & 0xF) != V34MapEntryType.V34_MAP_ENTRY_TYPE_COMPRESSED.ordinal() && (chd.map[i + j].flags & 0xF) != V34MapEntryType.V34_MAP_ENTRY_TYPE_UNCOMPRESSED.ordinal()) continue;
                maxoffset = Math.max(maxoffset, chd.map[i + j].offset + (long)chd.map[i + j].length);
            }
        }
        chd.file.ioLseek(fileoffset);
        count = chd.file.ioRead(cookie, 0, entrysize);
        if (count != entrysize || !"EndOfListCookie".equals(new String(cookie, 0, entrysize))) {
            return ChdHeader.ChdError.CHDERR_INVALID_FILE;
        }
        if (maxoffset > chd.file.length()) {
            return ChdHeader.ChdError.CHDERR_INVALID_FILE;
        }
        return ChdHeader.ChdError.CHDERR_NONE;
    }

    public static class ChdFile {
        public int cookie;
        public IVirtualFile file;
        public ChdHeader header;
        public ChdFile parent;
        public MapEntry[] map;
        public byte[] compressed;
        public final CodecInterface[] codecintf = new CodecInterface[4];
        public ZlibCodecData zlibCodecData = new ZlibCodecData();
        public LzmaCodecData lzmaCodecData = new LzmaCodecData();
        public FlacCodecData flacCodecData = new FlacCodecData();
        public CdzlCodecData cdzlCodecData = new CdzlCodecData();
        public CdlzCodecData cdlzCodecData = new CdlzCodecData();
        public CdflCodecData cdflCodecData = new CdflCodecData();

        public String toString() {
            return String.format("ChdFile[file=%s, header=%s, parent=%s]", this.file, this.header, this.parent);
        }
    }

    public static class CdflCodecData
    extends CodecData {
        public boolean swapEndian;
    }

    public static class CdlzCodecData
    extends CodecData {
    }

    public static class CdzlCodecData
    extends CodecData {
    }

    public static class FlacCodecData
    extends CodecData {
        public boolean nativeEndian;
    }

    public static class LzmaCodecData
    extends CodecData {
    }

    public static class ZlibCodecData
    extends CodecData {
    }

    public static class CodecData {
        public byte[] buffer;
    }

    protected static class CodecInterface {
        public final int compression;
        public final String compname;
        public final boolean lossy;
        public final ICodecInterface codec;

        public CodecInterface(int compression, String compname, ICodecInterface codec) {
            this.compression = compression;
            this.compname = compname;
            this.lossy = false;
            this.codec = codec;
        }

        public String toString() {
            return this.compname;
        }
    }

    protected static class MetaDataEntry {
        public long offset;
        public long next;
        public long prev;
        public int length;
        public int metatag;
        public int flags;

        protected MetaDataEntry() {
        }
    }

    protected static class MapEntry {
        public long offset;
        public int crc;
        public int length;
        public int flags;

        protected MapEntry() {
        }
    }

    static enum V5CompressionType {
        COMPRESSION_TYPE_0,
        COMPRESSION_TYPE_1,
        COMPRESSION_TYPE_2,
        COMPRESSION_TYPE_3,
        COMPRESSION_NONE,
        COMPRESSION_SELF,
        COMPRESSION_PARENT,
        COMPRESSION_RLE_SMALL,
        COMPRESSION_RLE_LARGE,
        COMPRESSION_SELF_0,
        COMPRESSION_SELF_1,
        COMPRESSION_PARENT_SELF,
        COMPRESSION_PARENT_0,
        COMPRESSION_PARENT_1;

    }

    static enum V34MapEntryType {
        V34_MAP_ENTRY_TYPE_INVALID,
        V34_MAP_ENTRY_TYPE_COMPRESSED,
        V34_MAP_ENTRY_TYPE_UNCOMPRESSED,
        V34_MAP_ENTRY_TYPE_MINI,
        V34_MAP_ENTRY_TYPE_SELF_HUNK,
        V34_MAP_ENTRY_TYPE_PARENT_HUNK,
        V34_MAP_ENTRY_TYPE_2ND_COMPRESSED;

    }
}

