#ifndef MBC_H
#define MBC_H

#include "common.h"

typedef struct {
	u8 rom_bank, bank_mode, ram_bank, ram_enable;
} mbc_params_t;

/** Initialise un contrleur multibanque.
	La ROM doit avoir t charge d'abord! La fonction va notamment accder
	 l'adresse $147 pour obtenir le type de cartouche.
	\param loaded_size taille de la cartouche charge
*/
void mbc_init(u32 loaded_size);

/** Ralise une lecture dans la zone I/O de la cartouche ($4000-$7FFF).
	\param address adresse  lire
	\return valeur de retour par le bus de donnes
	\note les adresses entre $0000 et $3FFF peuvent tre retournes directement
		via mem_rom car les premiers 16k sont toujours mapps  la page 0
*/
extern u8 (*mbc_read)(u16 address);

/** Ralise une criture dans la zone I/O de la cartouche ($0000-$7FFF).
	Voir #mbc_read pour plus d'informations.
	\param address adresse  crire
	\param value valeur  crire (via le bus de donnes)
*/
extern void (*mbc_write)(u16 address, u8 value);

/** Ralise une lecture dans la zone SRAM de la cartouche ($A000-$BFFF).
	Voir #mbc_read pour plus d'informations.
	\param address adresse  crire
	\param value valeur  crire (via le bus de donnes)
*/
u8 mbc_sram_read(u16 address);

/** Ralise une criture dans la zone SRAM de la cartouche ($A000-$BFFF).
	Voir #mbc_read pour plus d'informations.
	\param address adresse  crire
	\param value valeur  crire (via le bus de donnes)
*/
void mbc_sram_write(u16 address, u8 value);

/** Retourne les paramtres du MBC. Utile pour raliser une sauvegarde de
	l'tat de la machine.
	\return une structure #mbc_params_t avec les paramtres courants.
	\note L'criture dans la structure retourne est permise et va affecter
	le MBC.
*/
mbc_params_t *mbc_get_params();

/** Lit les donnes de la SRAM. Utile pour la sauvegarde d'tat.
	\param buffer tampon de destination des donnes
	\param max_size taille maximale du tampon
	\return nombre d'octets lus (taille de la SRAM)
*/
unsigned mbc_get_sram_data(u8 *buffer, u32 max_size);

/** Ecrit les donnes dans la SRAM. Utile pour la restauration d'tat.
	\param buffer donnes  crire
	\param size nombre d'octets  crire
*/
void mbc_set_sram_data(const u8 *buffer, unsigned size);

#endif
