#ifndef CPU_H
#define CPU_H

#include "common.h"

/*	De la synchronisation des composants.

	La Game Boy contient plusieurs composants qui fonctionnent  leur rythme,
	mais qui sont tous synchroniss sur une horloge: l'oscillateur (OSC clock).
	Cette horloge est cadence  4 MHz et tous les timings peuvent tre
	exprims en fonction de cette horloge, notamment:
	- 4 cycles pour 1 cycle machine du processeur
	- 456 cycles pour le balayage d'une ligne de l'cran LCD
	- 32 cycles pour la gnration d'une sample audio
	- ...
	Ainsi une faon d'muler le paralllisme des composants sans utiliser un
	systme de threads trs complexe et trs lent  l'excution, est de tout
	simplement muler la synchronisation par l'oscillateur!
	
	On a donc une boucle principale qui fait avancer l'horloge d'un cycle, et
	le transmet  tous les composants. Les composants vont alors raliser la
	part de travail qui leur incombe durant ce laps de temps.
	Par exemple le CPU excutera une instruction ou finira d'excuter celle
	qu'il a en cours (une instruction durant plusieurs cycles); le contrleur
	son gnrera une "sample" audio ou attendra un peu pour gnrer la
	prochaine.
	On a alors une machine d'tat avec un systme vnementiel pour chaque
	composant, dont le schma en pseudo-code objet est ainsi:

	Composant:
		temps_dodo: entier := 0
		mthode tick():			// Un cycle d'horloge s'est pass
			si temps_dodo > 0:
				temps_dodo--	// Rien  faire
			sinon:
				traite un vnement (ralise une action)
				temps_dodo := temps que dure cette action

		mthode idle_cycles():	// cycles avant le prochain vnement
			retourne temps_dodo

	CPU, LCD, Son: Composants
	Boucle principale:
		CPU.tick()
		LCD.tick()
		Son.tick()
		...

	Notez le raccourci! On n'a pas divis l'action en le nombre de cycles
	qu'elle prendrait sur le matriel. En revanche on l'excute d'un seul coup
	puis on met le composant en pause pour simuler l'excution inacheve de
	celle-ci.

	Ce modle est joli mais il peut toutefois comporter un problme: celui de
	la performance du fait du nombre lev d'excutions de la boucle principale
	(4 MHz = 4 millions de fois par seconde). Surtout quand on sait que les
	instructions du CPU prennent au moins 4 cycles, et peuvent mme en durer
	24; on n'aurait pas forcment besoin d'tre aussi prcis.

	Une optimisation possible est alors de n'envoyer des ticks que
	priodiquement, en indiquant au composant le nombre de cycles qui se sont
	couls depuis la dernire fois.
	Et comme les vnements se produisant le plus souvent sont l'excution
	d'une instruction, pourquoi ne pas le prendre en rfrence? La boucle
	principale ressemblerait alors  ceci:

	CPU, LCD, Son: Composants
	Boucle:
		// execInstruction retourne le # de cycles requis par l'instruction
		couls: entier := CPU.execInstruction()
		LCD.tick(couls)
		Son.tick(couls)
		...

	D'o...
	Les composants ont tous une architecture similaire:
	void ***_tick(int cycles);
	int ***_idle_cycles();
	La premire fait "avancer" le temps d'un certain nombre de cycles. Par
	exemple dans le cas du timer cela le fera avancer s'il est activ.
	La deuxime retourne le nombre de cycles pendant lequel le composant
	est inactif (temps avant le prochain vnement le concernant).

	Les composants,  part le CPU qui est toujours actif, ralisent
	priodiquement des tches puis "s'endorment" pendant un moment (par exemple
	l'cran dessine une ligne puis attend un signal de synchronisation avant la
	prochaine).

	Ainsi la fonction "tick" doit tre appele aussi souvent que possible pour
	mettre  jour l'tat des composants (si le timer est mis  jour trop peu
	souvent et affiche des valeurs telles que 1, 4, 8, 11, etc. l'mulation
	sera probablement peu raliste), mais toutefois il n'est pas non plus
	ncessaire de la rappeler avant que "idle_cycles" soit zro, ainsi on
	pourrait imaginer rduire la boucle principale en utilisant cette notion
	d'vnements:

	Boucle:
		// Calcule l'vnement le plus proche
		cycles_libres := min(sound_idle_cycles, lcd_idle_cycles, ...)
		// Ensuite excute le CPU jusqu' cet vnement
		cycles_excuts := 0
		Tant que cycles_excuts < cycles_libres:
			cycles_excuts += cpu_exec_instruction()
		// Met  jour les composants (il y en a au moins un qui a quelque
		// chose  faire).
		lcd_tick(cycles_excuts)
		sound_tick(cycles_excuts)
		...

	Le fait que le CPU soit "matre" suppose que rien ne se passe pendant
	l'excution de l'instruction qui ait un effet sur celle-ci. En fait mme si
	cette assomption est fausse, il serait en fait impossible de dterminer
	exactement comment le matriel ragirait, ainsi on considrera que c'est le
	cas.
*/

/** Met le CPU dans l'tat initial (au dmarrage de la Game Boy) */
void cpu_init();

/** Excute une instruction CPU et donne le nombre de cycles "mangs". Il
	faudrait faire avancer le reste du hardware du mme nombre d'units et
	vrifier si des vnements ont lieu avant de continuer  excuter la
	prochaine instruction.
	\return le nombre de cycles utiliss par l'instruction
*/
unsigned cpu_exec_instruction();

/** Types d'interruptions possibles. */
typedef enum {
	INT_VBLANK = 0,		// priode de VBLANK du LCD
	INT_STAT = 1,			// statut du LCD
	INT_TIMER = 2,			// a se comprend tout seul
	INT_SERIAL = 3,		// pas mul
	INT_JOYPAD = 4,		// combinaison de touches
	INT_LAST = 5
} cpu_interrupt_t;

/** Dclenche une interruption logicielle, qui va ventuellement s'excuter sur
	le CPU si l'utilisateur ne l'a pas masque.
	\note le bit correspondant de IF est mis  jour
*/
void cpu_trigger_irq(cpu_interrupt_t int_no);

/** Rcupre les valeurs des registres du CPU dans un tampon (y.c statut).
	\param buffer tampon d'au moins 14 octets (8 registres + SP, HL, IME, HALT)
	\note Utilis pour la sauvegarde d'tat.
*/
unsigned cpu_get_state(u8 *buffer);

/** Dfinit les valeurs des registres du CPU depuis un tampon rcupr avec
	#cpu_get_regs (y.c. statut, IME, etc.).
	\param buffer tampon de 14 octets (voir #cpu_get_state)
*/
void cpu_set_state(const u8 *buffer);

/** Decode (dsassemble) une instruction.
	\param address adresse du bus o lire l'instruction
	\param name [out] chane contenant l'instruction dcode
	\param length [out] nombre d'octets utiliss par l'instruction. PC devrait
		ensuite tre avanc de ce nombre d'units.
	\param cycles [out] nombre de cycles machine (1 m-cycle = 4 t-cycles sur
		Game Boy) requis pour l'excution de l'instruction
*/
void cpu_disassemble(u16 address, char *name, int *length, int *cycles);

/** Affiche une instruction (formate) */
void cpu_print_instruction(u16 address);

/** Activer ceci sur un processeur little endian (comme l'Intel x86/x64 cible) */
#define LITTLE_ENDIAN

#endif
