//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ MFC MIDI ]
//
//---------------------------------------------------------------------------

#if defined(_WIN32)

#include "os.h"
#include "xm6.h"
#include "vm.h"
#include "midi.h"
#include "config.h"
#include "mfc_com.h"
#include "mfc_sch.h"
#include "mfc_cfg.h"
#include "mfc_midi.h"

//===========================================================================
//
//	MIDI
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	MIDIbZ[W`
//
//---------------------------------------------------------------------------
#define MIDI_NOTEOFF		0x80		// m[gIt
#define MIDI_NOTEON			0x90		// m[gI(ꕔm[gIt)
#define MIDI_PKEYPRS		0xa0		// |tHjbNL[vbV[
#define MIDI_CTRLCHG		0xb0		// Rg[`FW
#define MIDI_PROGCHG		0xc0		// vO`FW
#define MIDI_CHPRS			0xd0		// `lvbV[
#define MIDI_PITCHCHG		0xe0		// sb`xh`FW
#define MIDI_EXSTART		0xf0		// GNXN[VuJn
#define MIDI_QFRAME			0xf1		// ^CR[hNH[^[t[
#define MIDI_SONGPTR		0xf2		// \O|WV|C^
#define MIDI_SONGSEL		0xf3		// \OZNg
#define MIDI_TUNEREQ		0xf6		// `[ZNg
#define MIDI_EXEND			0xf7		// GNXN[VuI
#define MIDI_TIMINGCLK		0xf8		// ^C~ONbN
#define MIDI_START			0xfa		// X^[g
#define MIDI_CONTINUE		0xfb		// ReBj[
#define MIDI_STOP			0xfc		// Xgbv
#define MIDI_ACTSENSE		0xfe		// ANeBuZVO
#define MIDI_SYSRESET		0xff		// VXeZbg

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
CMIDI::CMIDI(CFrmWnd *pWnd) : CComponent(pWnd)
{
	// R|[lgp[^
	m_dwID = MAKEID('M', 'I', 'D', 'I');
	m_strDesc = _T("MIDI Driver");

	// IuWFNg
	m_pMIDI = NULL;
	m_pScheduler = NULL;
	m_pSch = NULL;

	// Out
	m_dwOutDevice = 0;
	m_hOut = NULL;
	m_pOutThread = NULL;
	m_bOutThread = FALSE;
	m_dwOutDevs = 0;
	m_pOutCaps = NULL;
	m_dwOutDelay = (84 * 2000);
	m_nOutReset = 0;
	m_dwShortSend = 0;
	m_dwExSend = 0;
	m_dwUnSend = 0;

	// In
	m_dwInDevice = 0;
	m_hIn = NULL;
	m_pInThread = NULL;
	m_bInThread = FALSE;
	m_dwInDevs = 0;
	m_pInCaps = NULL;
	m_InState[0] = InNotUsed;
	m_InState[1] = InNotUsed;
	m_dwShortRecv = 0;
	m_dwExRecv = 0;
	m_dwUnRecv = 0;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL CMIDI::Init()
{
	ASSERT(this);

	// {NX
	if (!CComponent::Init()) {
		return FALSE;
	}

	// MIDI擾
	ASSERT(!m_pMIDI);
	m_pMIDI = (MIDI*)::GetVM()->SearchDevice(MAKEID('M', 'I', 'D', 'I'));
	ASSERT(m_pMIDI);

	// XPW[擾
	ASSERT(!m_pScheduler);
	m_pScheduler = (Scheduler*)::GetVM()->SearchDevice(MAKEID('S', 'C', 'H', 'E'));
	ASSERT(m_pScheduler);

	// XPW[(Win32)擾
	ASSERT(!m_pSch);
	m_pSch = (CScheduler*)SearchComponent(MAKEID('S', 'C', 'H', 'E'));
	ASSERT(m_pSch);

	// OutfoCX
	m_dwOutDevs = ::midiOutGetNumDevs();
	if (m_dwOutDevs > 0) {
		// foCXACAPS\̂m
		try {
			m_pOutCaps = new MIDIOUTCAPS[m_dwOutDevs];
		}
		catch (...) {
			return FALSE;
		}
		if (!m_pOutCaps) {
			return FALSE;
		}

		// CAPS擾
		GetOutCaps();
	}

	// InfoCX
	m_dwInDevs = ::midiInGetNumDevs();
	if (m_dwInDevs > 0) {
		// foCXACAPS\̂m
		try {
			m_pInCaps = new MIDIINCAPS[m_dwInDevs];
		}
		catch (...) {
			return FALSE;
		}
		if (!m_pInCaps) {
			return FALSE;
		}

		// CAPS擾
		GetInCaps();
	}

	// InL[
	if (!m_InQueue.Init(InBufMax)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::Cleanup()
{
	ASSERT(this);

	// N[Y
	CloseOut();
	CloseIn();

	// Caps폜
	if (m_pOutCaps) {
		delete[] m_pOutCaps;
		m_pOutCaps = NULL;
	}
	if (m_pInCaps) {
		delete[] m_pInCaps;
		m_pInCaps = NULL;
	}

	// {NX
	CComponent::Cleanup();
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::ApplyCfg(const Config* pConfig)
{
	CConfig *pComponent;

	ASSERT(this);
	ASSERT(pConfig);
	ASSERT_VALID(this);

	// RtBOR|[lg擾
	pComponent = (CConfig*)SearchComponent(MAKEID('C', 'F', 'G', ' '));
	ASSERT(pComponent);

	// ZbgR}h
	m_nOutReset = pConfig->midi_reset;

	// OutfBC
	m_dwOutDelay = (DWORD)(pConfig->midiout_delay * 2000);

	// OutfoCX
	if (pConfig->midiout_device != (int)m_dwOutDevice) {
		// U
		CloseOut();

		// wfoCXŃgC
		if (pConfig->midiout_device > 0) {
			OpenOut(pConfig->midiout_device);
		}

		// ΁ARtBOɔf
		if (m_dwOutDevice != 0) {
			pComponent->SetMIDIDevice((int)m_dwOutDevice, FALSE);
		}
	}

	// InfBC
	SetInDelay(pConfig->midiin_delay * 2000);

	// InfoCX
	if (pConfig->midiin_device != (int)m_dwInDevice) {
		// U
		CloseIn();

		// wfoCXŃgC
		if (pConfig->midiin_device > 0) {
			OpenIn(pConfig->midiin_device);
		}

		// ΁ARtBOɔf
		if (m_dwInDevice > 0) {
			pComponent->SetMIDIDevice((int)m_dwInDevice, TRUE);
		}
	}
}

#if !defined(NDEBUG)
//---------------------------------------------------------------------------
//
//	ff
//
//---------------------------------------------------------------------------
void CMIDI::AssertValid() const
{
	// {NX
	CComponent::AssertValid();

	ASSERT(this);
	ASSERT(m_pMIDI);
	ASSERT(m_pMIDI->GetID() == MAKEID('M', 'I', 'D', 'I'));
	ASSERT(m_pScheduler);
	ASSERT(m_pScheduler->GetID() == MAKEID('S', 'C', 'H', 'E'));
	ASSERT(m_pSch);
	ASSERT(m_pSch->GetID() == MAKEID('S', 'C', 'H', 'E'));
	ASSERT((m_dwOutDevs == 0) || m_pOutCaps);
	ASSERT((m_dwInDevs == 0) || m_pInCaps);
	ASSERT((m_nOutReset >= 0) && (m_nOutReset <= 3));
}
#endif	// NDEBUG

//===========================================================================
//
//	OUT
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	OutI[v
//	dwDevice=0͊蓖ĂȂAdwDevice=1MIDI}bp[
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::OpenOut(DWORD dwDevice)
{
	MMRESULT mmResult;
	UINT uDeviceID;

	ASSERT(this);
	ASSERT(m_dwOutDevice == 0);
	ASSERT(!m_pOutThread);
	ASSERT_VALID(this);

	// 蓖ĂȂ̂ł΃^[
	if (dwDevice == 0) {
		return;
	}

	// foCXȂ΃^[
	if (m_dwOutDevs == 0) {
		return;
	}

	// foCX𒴂ĂMIDI}bp[
	if (dwDevice > (m_dwOutDevs + 1)) {
		dwDevice = 1;
	}

	// foCX
	if (dwDevice == 1) {
		uDeviceID = MIDI_MAPPER;
	}
	else {
		uDeviceID = (UINT)(dwDevice - 2);
	}

	// I[v
#if _MFC_VER >= 0x700
	mmResult = ::midiOutOpen(&m_hOut, uDeviceID,
							(DWORD_PTR)OutProc, (DWORD_PTR)this, CALLBACK_FUNCTION);
#else
	mmResult = ::midiOutOpen(&m_hOut, uDeviceID,
							(DWORD)OutProc, (DWORD)this, CALLBACK_FUNCTION);
#endif

	// sȂI
	if (mmResult != MMSYSERR_NOERROR) {
		return;
	}

	// foCXԍXV
	m_dwOutDevice = dwDevice;

	// ]obt@NA
	m_pMIDI->ClrTransData();

	// Xbh𗧂Ă
	m_bOutThread = FALSE;
	m_pOutThread = AfxBeginThread(OutThread, this);
}

//---------------------------------------------------------------------------
//
//	OutN[Y
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::CloseOut()
{
	ASSERT(this);
	ASSERT_VALID(this);

	// Xbh~߂
	if (m_pOutThread) {
		// ItOグ
		m_bOutThread = TRUE;

		// I҂()
		::WaitForSingleObject(m_pOutThread->m_hThread, INFINITE);

		// XbhȂ
		m_pOutThread = NULL;
	}

	// foCX
	if (m_dwOutDevice != 0) {
		// SẴm[gL[ItāAN[Y
		SendAllNoteOff();
		::midiOutReset(m_hOut);
		::midiOutClose(m_hOut);

		// I[vfoCXȂ
		m_dwOutDevice = 0;
	}
}

//---------------------------------------------------------------------------
//
//	OutXbh
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::RunOut()
{
	BOOL bEnable;
	BOOL bActive;
	OutState outState;
	DWORD dwTime;
	const MIDI::mididata_t *pData;
	DWORD dwLen;
	DWORD dwCnt;
	DWORD dwCmd;
	DWORD dwShortMsg;

	ASSERT(this);
	ASSERT(m_hOut);
	ASSERT_VALID(this);

	// XPW[̏ԂL
	bEnable = m_pSch->IsEnable();

	// GNXN[VuMȂAwb_gpȂ
	m_bSendEx = FALSE;
	m_bSendExHdr = FALSE;

	// JE^
	m_dwShortSend = 0;
	m_dwExSend = 0;
	m_dwUnSend = 0;

	// f[^ȂAR}hAXe[gfB
	pData = NULL;
	dwCmd = 0;
	dwLen = 0;
	dwCnt = 0;
	dwShortMsg = 0;
	outState = OutReady;

	// Zbg
	SendReset();

	// ANeBu(Lȃf[^MIDI OUT֑)
	bActive = FALSE;

	// [v
	while (!m_bOutThread) {
		// GNXN[VuMȂ҂
		if (m_bSendEx) {
			::Sleep(1);
			continue;
		}

		// GNXN[VuMwb_gȂAЕt
		if (m_bSendExHdr) {
			// bNčs
			m_OutSection.Lock();
			::midiOutUnprepareHeader(m_hOut, &m_ExHdr, sizeof(MIDIHDR));
			m_OutSection.Unlock();

			// GNXN[Vuwb_gptO낷
			m_bSendExHdr = FALSE;
			pData = NULL;
			outState = OutReady;

			// JEgAbv
			m_dwUnSend++;
		}

		// Ȃ
		if (!m_bEnable) {
			// ANeBuȂASm[gItƃZbg
			if (bActive) {
				SendAllNoteOff();
				SendReset();
				bActive = FALSE;

				// Lf[^Ȃ
				outState = OutReady;
				pData = NULL;
			}

			::Sleep(10);
			continue;
		}

		// XPW[Ȃ
		if (!m_pSch->IsEnable()) {
			// ܂ŗLłȂ
			if (bEnable) {
				// Sm[g[It
				SendAllNoteOff();

				// ȊȌԂ̓L[v
				bEnable = FALSE;
			}

			// ܂ł
			::Sleep(10);
			continue;
		}

		// XPW[͗L
		bEnable = TRUE;

		// MIDIZbgꂽ
		if (m_pMIDI->IsReset()) {
			// tOƂ
			m_pMIDI->ClrReset();

			// Sm[gItƃZbg
			SendAllNoteOff();
			SendReset();
			pData = NULL;
			outState = OutReady;
		}

		// MIDIANeBułȂΉȂ
		if (!m_pMIDI->IsActive()) {
			::Sleep(10);
			continue;
		}

		// MIDIMobt@̌擾B0ȂSleep
		if (!pData) {
			if (m_pMIDI->GetTransNum() == 0) {
				// 15msȏfBĈŁA10msEFCg
				if (m_dwOutDelay > 30000) {
					::Sleep(10);
					continue;
				}
				// fBC14msȉB1msEFCg
				::Sleep(1);
				continue;
			}

			// Mf[^̐擪擾
			pData = m_pMIDI->GetTransData(0);
		}

		// 1oCgȏf[^̂ŁAANeBuƂ
		bActive = TRUE;

		// fBŁAGNXN[VuR}hłȂ΁Aԑ҂
		if ((outState == OutReady) && (pData->data != MIDI_EXSTART)) {
			dwTime = m_pScheduler->GetTotalTime() - pData->vtime;
			if (dwTime < m_dwOutDelay) {
				if (dwTime > 30000) {
					// 15msȏ󂢂Ă΁A1.5ms҂
					::Sleep(10);
					continue;
				}
				if (dwTime > 3000) {
					// 1.5msȏ󂢂Ă΁A1.5ms҂
					::Sleep(1);
					continue;
				}
				// 1.5ms̑҂Sleep(0)ŋz
				::Sleep(0);
				continue;
			}
		}

		// f[^()
		if (outState == OutReady) {
			// GNXN[VuMȂAŏo(ł̌)
			if ((pData->data & 0x80) && (pData->data != MIDI_EXEND)) {
				if (outState == OutEx) {
					m_ExBuf[dwCnt] = MIDI_EXEND;
					dwCnt++;
					SendEx(m_ExBuf);
					continue;
				}
			}

			// f[^
			if ((pData->data >= MIDI_NOTEOFF) && (pData->data < MIDI_EXSTART)) {
				// 80-EF
				outState = OutShort;
				dwCmd = pData->data;
				dwCnt = 1;
				dwShortMsg = dwCmd;
				dwLen = 3;

				if ((pData->data >= MIDI_PROGCHG) && (pData->data < MIDI_PITCHCHG)) {
					// C0-DF
					dwLen = 2;
				}
			}

			// RbZ[WAA^CbZ[W
			switch (pData->data) {
				// GNXN[VuJn
				case MIDI_EXSTART:
					outState = OutEx;
					m_ExBuf[0] = (BYTE)pData->data;
					dwCnt = 1;
					break;
				// ^CR[hNH[^[t[
				case MIDI_QFRAME:
				// \OZNg
				case MIDI_SONGSEL:
					outState = OutShort;
					dwCnt = 1;
					dwShortMsg = pData->data;
					dwLen = 2;
					break;
				// \O|WV|C^
				case MIDI_SONGPTR:
					outState = OutShort;
					dwCnt = 1;
					dwShortMsg = pData->data;
					dwLen = 3;
					break;
				// `[NGXgAA^CbZ[W
				case MIDI_TUNEREQ:
				case MIDI_TIMINGCLK:
				case MIDI_START:
				case MIDI_CONTINUE:
				case MIDI_STOP:
				case MIDI_ACTSENSE:
				case MIDI_SYSRESET:
					// A^CbZ[W̗ނ͏o͂Ȃ(喂E)
					m_pMIDI->DelTransData(1);
					pData = NULL;
					continue;
			}

			// jOXe[^X
			if (pData->data < MIDI_NOTEOFF) {
				dwCnt = 2;
				dwShortMsg = (pData->data << 8) | dwCmd;
				outState = OutShort;
			}

#if defined(_DEBUG)
			// f[^`FbN(DEBUĜ)
			if (outState == OutReady) {
				TRACE("MIDIُf[^o $%02X\n", pData->data);
			}
#endif	// _DEBUG

			// ̃f[^폜
			m_pMIDI->DelTransData(1);
			pData = NULL;

			// őMł̂ɌAo
			if ((outState == OutShort) && (dwCnt == dwLen)) {
				m_OutSection.Lock();
				::midiOutShortMsg(m_hOut, dwShortMsg);
				m_OutSection.Unlock();

				// JEgAbv
				m_dwShortSend++;
				outState = OutReady;
			}
			continue;
		}

		// f[^(2oCgڈȍ~)
		if (pData->data & 0x80) {
			// GNXN[VuMȂAŏo(ł̌)
			if (pData->data != MIDI_EXEND) {
				if (outState == OutEx) {
					m_ExBuf[dwCnt] = MIDI_EXEND;
					dwCnt++;
					SendEx(m_ExBuf);
					continue;
				}
			}

			if ((pData->data >= MIDI_NOTEOFF) && (pData->data < MIDI_EXSTART)) {
				// 80-EF
				outState = OutShort;
				dwCmd = pData->data;
				dwCnt = 1;
				dwShortMsg = dwCmd;
				dwLen = 3;

				if ((pData->data >= MIDI_PROGCHG) && (pData->data < MIDI_PITCHCHG)) {
					// C0-DF
					dwLen = 2;
				}
			}

			// RbZ[WAA^CbZ[W
			switch (pData->data) {
				// GNXN[VuJn
				case MIDI_EXSTART:
					outState = OutEx;
					m_ExBuf[0] = (BYTE)pData->data;
					dwCnt = 1;
					break;
				// ^CR[hNH[^[t[
				case MIDI_QFRAME:
				// \OZNg
				case MIDI_SONGSEL:
					outState = OutShort;
					dwCnt = 1;
					dwShortMsg = pData->data;
					dwLen = 2;
					break;
				// \O|WV|C^
				case MIDI_SONGPTR:
					outState = OutShort;
					dwCnt = 1;
					dwShortMsg = pData->data;
					dwLen = 3;
					break;
				// `[NGXgAA^CbZ[W
				case MIDI_TUNEREQ:
				case MIDI_TIMINGCLK:
				case MIDI_START:
				case MIDI_CONTINUE:
				case MIDI_STOP:
				case MIDI_ACTSENSE:
				case MIDI_SYSRESET:
					// A^CbZ[W̗ނ͏o͂Ȃ(喂E)
					break;
				// GNXN[VuI
				case MIDI_EXEND:
					if (outState == OutEx) {
						// GNXN[VuMȂAŏo
						m_ExBuf[dwCnt] = (BYTE)pData->data;
						dwCnt++;
						SendEx(m_ExBuf);
						outState = OutReady;

						// ̃f[^폜
						m_pMIDI->DelTransData(1);
						pData = NULL;
						continue;
					}
			}
		}
		else {
			// 00-7F
			if (outState == OutEx) {
				// GNXN[Vu
				m_ExBuf[dwCnt] = (BYTE)pData->data;
				dwCnt++;
				if (dwCnt >= sizeof(m_ExBuf)) {
					outState = OutReady;
				}
			}
			if (outState == OutShort) {
				// V[gbZ[W
				dwShortMsg |= (pData->data << (dwCnt << 3));
				dwCnt++;
			}
		}

		// ̃f[^폜
		m_pMIDI->DelTransData(1);
		pData = NULL;

		// őMł̂ɌAo
		if ((outState == OutShort) && (dwCnt == dwLen)) {
			m_OutSection.Lock();
			::midiOutShortMsg(m_hOut, dwShortMsg);
			m_OutSection.Unlock();

			// JEgAbv
			m_dwShortSend++;
			outState = OutReady;
		}
	}

	// GNXN[VuM̃EFCg
	SendExWait();

	// Sm[gIt
	SendAllNoteOff();
}

//---------------------------------------------------------------------------
//
//	OutR[obN
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::CallbackOut(UINT wMsg, DWORD /*dwParam1*/, DWORD /*dwParam2*/)
{
	ASSERT(this);
	ASSERT_VALID(this);

	// GNXN[VuM̂݃nhO
	if (wMsg == MM_MOM_DONE) {

		// GNXN[VuMtOOFF(BOOLf[^̂߁A͎Ȃ)
		m_bSendEx = FALSE;
	}
}

//---------------------------------------------------------------------------
//
//	OutfoCXCaps擾
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::GetOutCaps()
{
	MMRESULT mmResult;
#if _MFC_VER >= 0x700
	UINT_PTR uDeviceID;
#else
	UINT uDeviceID;
#endif	// MFC_VER

	// foCXID
	uDeviceID = 0;

	// [v
	while ((DWORD)uDeviceID < m_dwOutDevs) {
		// midiOutGetDevCapsŎ擾
		mmResult = ::midiOutGetDevCaps( uDeviceID,
										&m_pOutCaps[uDeviceID],
										sizeof(MIDIOUTCAPS));

		// ʂL^
		if (mmResult == MMSYSERR_NOERROR) {
			m_pOutCaps[uDeviceID].dwSupport = 0;
		}
		else {
			m_pOutCaps[uDeviceID].dwSupport = 1;
		}

		// foCXID
		uDeviceID++;
	}
}

//---------------------------------------------------------------------------
//
//	OutXbh
//
//---------------------------------------------------------------------------
UINT CMIDI::OutThread(LPVOID pParam)
{
	CMIDI *pMIDI;

	// p[^󂯎
	pMIDI = (CMIDI*)pParam;

	// s
	pMIDI->RunOut();

	// I
	return 0;
}

//---------------------------------------------------------------------------
//
//	OutR[obN
//
//---------------------------------------------------------------------------
#if _MFC_VER >= 0x700
void CALLBACK CMIDI::OutProc(HMIDIOUT hOut, UINT wMsg, DWORD_PTR dwInstance,
							DWORD dwParam1, DWORD dwParam2)
#else
void CALLBACK CMIDI::OutProc(HMIDIOUT hOut, UINT wMsg, DWORD dwInstance,
							DWORD dwParam1, DWORD dwParam2)
#endif
{
	CMIDI *pMIDI;

	// p[^󂯎
	pMIDI = (CMIDI*)dwInstance;

	// Ăяo
	if (hOut == pMIDI->GetOutHandle()) {
		pMIDI->CallbackOut(wMsg, dwParam1, dwParam2);
	}
}

//---------------------------------------------------------------------------
//
//	GNXN[VuM
//	ȎMAwb_̌ЕtIĂĂяo
//
//---------------------------------------------------------------------------
BOOL FASTCALL CMIDI::SendEx(const BYTE *pExData)
{
	MMRESULT mmResult;
	DWORD dwLength;

	ASSERT(this);
	ASSERT(pExData);
	ASSERT(!m_bSendEx);
	ASSERT(!m_bSendExHdr);
	ASSERT_VALID(this);

	// 𐔂
	dwLength = 1;
	while (pExData[dwLength - 1] != 0xf7) {
		ASSERT(dwLength <= sizeof(m_ExBuf));
		dwLength++;
	}

	// ߂ăf[^
	ASSERT(pExData[0] == 0xf0);
	ASSERT(pExData[dwLength - 1] == 0xf7);

	// wb_
	memset(&m_ExHdr, 0, sizeof(m_ExHdr));
	m_ExHdr.lpData = (LPSTR)pExData;
	m_ExHdr.dwBufferLength = dwLength;
	m_ExHdr.dwBytesRecorded = dwLength;
	m_OutSection.Lock();
	mmResult = ::midiOutPrepareHeader(m_hOut, &m_ExHdr, sizeof(MIDIHDR));
	m_OutSection.Unlock();
	if (mmResult != MMSYSERR_NOERROR) {
		return FALSE;
	}

	// GNXN[Vuwb_gptOON
	m_bSendExHdr = TRUE;

	// GNXN[VuMtO𗧂Ă
	m_bSendEx = TRUE;

	// LongMsgM
	m_OutSection.Lock();
	mmResult = ::midiOutLongMsg(m_hOut, &m_ExHdr, sizeof(MIDIHDR));
	m_OutSection.Unlock();
	if (mmResult != MMSYSERR_NOERROR) {
		// stOƂ
		m_bSendEx = FALSE;
		return FALSE;
	}

	// 
	m_dwExSend++;
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	GNXN[VuM҂
//
//---------------------------------------------------------------------------
BOOL FASTCALL CMIDI::SendExWait()
{
	DWORD dwTime;
	DWORD dwDiff;
	BOOL bResult;

	ASSERT(this);
	ASSERT_VALID(this);

	// tO͐
	bResult = TRUE;

	// GNXN[VuMȂ
	if (m_bSendEx) {
		// ݎԂ擾
		dwTime = ::timeGetTime();

		// M܂ŃEFCg(^CAEg2000ms)
		while (m_bSendEx) {
			// oߎԂZo
			dwDiff = ::timeGetTime();
			dwDiff -= dwTime;

			// ^CAEgȂ玸sƂĔ
			if (dwDiff >= 2000) {
				bResult = FALSE;
				break;
			}

			// X[v
			::Sleep(1);
		}
	}

	// GNXN[VuMwb_gȂAЕt
	if (m_bSendExHdr) {
		m_OutSection.Lock();
		::midiOutUnprepareHeader(m_hOut, &m_ExHdr, sizeof(MIDIHDR));
		m_OutSection.Unlock();
		m_bSendExHdr = FALSE;
	}

	return bResult;
}

//---------------------------------------------------------------------------
//
//	Sm[gIt
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::SendAllNoteOff()
{
	int nCh;
	DWORD dwMsg;

	ASSERT(this);
	ASSERT_VALID(this);

	// [hbZ[WAll Sound Off𑗐M
	m_OutSection.Lock();
	for (nCh=0; nCh<16; nCh++) {
		dwMsg = (DWORD)(0x78b0 + nCh);
		::midiOutShortMsg(m_hOut, dwMsg);
	}
	m_OutSection.Unlock();

	// m[gIt(MT-32΍)
	m_OutSection.Lock();
	for (nCh=0; nCh<16; nCh++) {
		::midiOutShortMsg(m_hOut, (DWORD)(0x7bb0 + nCh));
		::midiOutShortMsg(m_hOut, (DWORD)(0x7db0 + nCh));
		::midiOutShortMsg(m_hOut, (DWORD)(0x7fb0 + nCh));
	}
	m_OutSection.Unlock();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
BOOL FASTCALL CMIDI::SendReset()
{
	int nCh;

	ASSERT(this);
	ASSERT_VALID(this);

	// Reset All ControllerR}h
	m_OutSection.Lock();
	for (nCh=0; nCh<16; nCh++) {
		::midiOutShortMsg(m_hOut, (DWORD)(0x79b0 + nCh));
	}
	m_OutSection.Unlock();

	// Zbg^Cv
	switch (m_nOutReset) {
		// GM
		case 0:
			SendEx(ResetGM);
			break;

		// GS
		case 1:
			// GM->GS̏őo
			SendEx(ResetGM);
			SendExWait();
			SendEx(ResetGS);
			break;

		// XG
		case 2:
			// GM->XG̏őo
			SendEx(ResetGM);
			SendExWait();
			SendEx(ResetXG);
			break;

		// LA
		case 3:
			SendEx(ResetLA);
			break;

		// ̑(蓾Ȃ)
		default:
			ASSERT(FALSE);
			break;
	};

	// GNXN[VuM҂
	SendExWait();

	// JEgNA
	m_dwShortSend = 0;
	m_dwExSend = 0;
	m_dwUnSend = 0;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	OutfoCX擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL CMIDI::GetOutDevs() const
{
	ASSERT(this);
	ASSERT_VALID(this);

	return m_dwOutDevs;
}

//---------------------------------------------------------------------------
//
//	OutfoCX̂擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL CMIDI::GetOutDevDesc(int nDevice, CString& strDesc) const
{
	LPCTSTR lpstrDesc;

	ASSERT(this);
	ASSERT(nDevice >= 0);
	ASSERT_VALID(this);

	// CfbNX`FbN
	if (nDevice >= (int)m_dwOutDevs) {
		strDesc.Empty();
		return FALSE;
	}

	// LPCTSTR֕ϊ
	lpstrDesc = A2T(m_pOutCaps[nDevice].szPname);
	strDesc = lpstrDesc;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	OutfBCݒ
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::SetOutDelay(int nDelay)
{
	ASSERT(this);
	ASSERT(nDelay >= 0);
	ASSERT_VALID(this);

	m_dwOutDelay = (DWORD)(nDelay * 2000);
}

//---------------------------------------------------------------------------
//
//	Outʎ擾
//
//---------------------------------------------------------------------------
int FASTCALL CMIDI::GetOutVolume()
{
	MMRESULT mmResult;
	DWORD dwVolume;
	int nVolume;

	ASSERT(this);
	ASSERT_VALID(this);

	// nhLŁAXbhĂ邱
	if ((m_dwOutDevice > 0) && m_pOutThread) {
		// NeBJZNVŁAʂ擾
		m_OutSection.Lock();
		mmResult = ::midiOutGetVolume(m_hOut, &dwVolume);
		m_OutSection.Unlock();

		// ꍇALOWORD()D悷
		if (mmResult == MMSYSERR_NOERROR) {
			// l16bit
			nVolume = LOWORD(dwVolume);

			// 
			return nVolume;
		}
	}

	// MIDI~LT͖
	return -1;
}

//---------------------------------------------------------------------------
//
//	Outʐݒ
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::SetOutVolume(int nVolume)
{
	DWORD dwVolume;

	ASSERT(this);
	ASSERT((nVolume >= 0) && (nVolume < 0x10000));
	ASSERT_VALID(this);

	// nhLŁAXbhĂ邱
	if ((m_dwOutDevice > 0) && m_pOutThread) {
		// l쐬
		dwVolume = (DWORD)nVolume;
		dwVolume <<= 16;
		dwVolume |= (DWORD)nVolume;

		// NeBJZNVŁAʂݒ
		m_OutSection.Lock();
		::midiOutSetVolume(m_hOut, dwVolume);
		m_OutSection.Unlock();
	}
}

//---------------------------------------------------------------------------
//
//	Out擾
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::GetOutInfo(LPMIDIINFO pInfo) const
{
	ASSERT(this);
	ASSERT(pInfo);
	ASSERT_VALID(this);

	// foCX
	pInfo->dwDevices = m_dwOutDevs;
	pInfo->dwDevice = m_dwOutDevice;

	// JE^
	pInfo->dwShort = m_dwShortSend;
	pInfo->dwLong = m_dwExSend;
	pInfo->dwUnprepare = m_dwUnSend;

	// obt@VMfoCX̂ŁAł͐ݒ肵Ȃ
}

//---------------------------------------------------------------------------
//
//	GMZbgR}h
//
//---------------------------------------------------------------------------
const BYTE CMIDI::ResetGM[] = {
	0xf0, 0x7e, 0x7f, 0x09, 0x01, 0xf7
};

//---------------------------------------------------------------------------
//
//	GSZbgR}h
//
//---------------------------------------------------------------------------
const BYTE CMIDI::ResetGS[] = {
	0xf0, 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7f,
	0x00, 0x41, 0xf7
};

//---------------------------------------------------------------------------
//
//	XGZbgR}h
//
//---------------------------------------------------------------------------
const BYTE CMIDI::ResetXG[] = {
	0xf0, 0x43, 0x10, 0x4c, 0x00, 0x00, 0x7e, 0x00,
	0xf7
};

//---------------------------------------------------------------------------
//
//	LAZbgR}h
//
//---------------------------------------------------------------------------
const BYTE CMIDI::ResetLA[] = {
	0xf0, 0x41, 0x10, 0x16, 0x12, 0x7f, 0x00, 0x00,
	0x00, 0x01, 0xf7
};

//===========================================================================
//
//	IN
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	InI[v
//	dwDevice=0͊蓖ĂȂ
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::OpenIn(DWORD dwDevice)
{
	MMRESULT mmResult;
	UINT uDeviceID;

	ASSERT(this);
	ASSERT(m_dwInDevice == 0);
	ASSERT(!m_pInThread);
	ASSERT_VALID(this);

	// 蓖ĂȂ̂ł΃^[
	if (dwDevice == 0) {
		return;
	}

	// foCXȂ΃^[
	if (m_dwInDevs == 0) {
		return;
	}

	// foCX𒴂Ă΃foCX0
	if (dwDevice > m_dwInDevs) {
		dwDevice = 1;
	}

	// foCX
	ASSERT(dwDevice >= 1);
	uDeviceID = (UINT)(dwDevice - 1);

	// I[v

#if _MFC_VER >= 0x700
	mmResult = ::midiInOpen(&m_hIn, uDeviceID,
							(DWORD_PTR)InProc, (DWORD_PTR)this, CALLBACK_FUNCTION);
#else
	mmResult = ::midiInOpen(&m_hIn, uDeviceID,
							(DWORD)InProc, (DWORD)this, CALLBACK_FUNCTION);
#endif

	// sȂI
	if (mmResult != MMSYSERR_NOERROR) {
		return;
	}

	// foCXԍXV
	m_dwInDevice = dwDevice;

	// Xbh𗧂Ă
	m_bInThread = FALSE;
	m_pInThread = AfxBeginThread(InThread, this);
}

//---------------------------------------------------------------------------
//
//	InN[Y
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::CloseIn()
{
	ASSERT(this);
	ASSERT_VALID(this);

	// Xbh~߂
	if (m_pInThread) {
		// ItOグ
		m_bInThread = TRUE;

		// I҂()
		::WaitForSingleObject(m_pInThread->m_hThread, INFINITE);

		// XbhȂ
		m_pInThread = NULL;
	}

	// foCX
	if (m_dwInDevice != 0) {
		// ZbgāAN[Y
		::midiInReset(m_hIn);
		::midiInClose(m_hIn);

		// I[vfoCXȂ
		m_dwInDevice = 0;
	}
}

//---------------------------------------------------------------------------
//
//	InXbh
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::RunIn()
{
	int i;
	BOOL bActive;
	BOOL bValid;
	BOOL bMsg;

	ASSERT(this);
	ASSERT(m_hIn);
	ASSERT_VALID(this);

	// ANeBu
	bActive = FALSE;

	// GNXN[Vuobt@gp
	m_InState[0] = InNotUsed;
	m_InState[1] = InNotUsed;

	// JE^Zbg
	m_dwShortRecv = 0;
	m_dwExRecv = 0;
	m_dwUnRecv = 0;

	// Zbg
	::midiInReset(m_hIn);

	// [v
	while (!m_bInThread) {
		// LtOݒ
		bValid = FALSE;
		if (m_pSch->IsEnable()) {
			// XPW[쒆
			if (m_pMIDI->IsActive()) {
				// MIDIANeBuȂ
				bValid = TRUE;
			}
		}

		// ȏꍇ
		if (!bValid) {
			// Sleep
			::Sleep(10);
			continue;
		}

		// LȂ̂ŁA܂X^[gĂȂ΃X^[g
		if (!bActive) {
			StartIn();
			bActive = TRUE;
		}

		// bZ[WȂ
		bMsg = FALSE;

		// ObZ[W̌
		for (i=0; i<2; i++) {
			if (m_InState[i] == InDone) {
				// ObZ[WMς
				LongIn(i);
				bMsg = TRUE;
			}
		}

		// V[gbZ[W̌
		m_InSection.Lock();
		if (!m_InQueue.IsEmpty()) {
			// V[gbZ[WMς
			ShortIn();
			bMsg = TRUE;
		}
		m_InSection.Unlock();

		// bZ[WȂSleep
		if (!bMsg) {
			::Sleep(1);
		}
	}

	// ~
	if (bActive) {
		bActive = FALSE;
		StopIn();
	}
}

//---------------------------------------------------------------------------
//
//	InJn
//
//---------------------------------------------------------------------------
BOOL FASTCALL CMIDI::StartIn()
{
	int i;
	MMRESULT mmResult;

	ASSERT(this);
	ASSERT(m_hIn);
	ASSERT_VALID(this);

	// GNXN[Vuobt@o^
	for (i=0; i<2; i++) {
		if (m_InState[i] == InNotUsed) {
			// MIDIwb_쐬
			memset(&m_InHdr[i], 0, sizeof(MIDIHDR));
			m_InHdr[i].lpData = (LPSTR)&m_InBuf[i][0];
			m_InHdr[i].dwBufferLength = InBufMax;
			m_InHdr[i].dwUser = (DWORD)i;

			// wb_
			mmResult = ::midiInPrepareHeader(m_hIn, &m_InHdr[i], sizeof(MIDIHDR));
			if (mmResult != MMSYSERR_NOERROR) {
				// wb_s
				continue;
			}

			// wb_ǉ
			mmResult = ::midiInAddBuffer(m_hIn, &m_InHdr[i], sizeof(MIDIHDR));
			if (mmResult != MMSYSERR_NOERROR) {
				// wb_ǉs
				::midiInUnprepareHeader(m_hIn, &m_InHdr[i], sizeof(MIDIHDR));
				continue;
			}

			// Xe[gݒ
			m_InState[i] = InReady;
		}
	}

	// InL[NA
	m_InQueue.Clear();

	// MIDI͊Jn
	mmResult = ::midiInStart(m_hIn);
	if (mmResult != MMSYSERR_NOERROR) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	In~(obt@̂)
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::StopIn()
{
	int i;

	ASSERT(this);
	ASSERT(m_hIn);
	ASSERT_VALID(this);

	// GNXN[Vuobt@폜
	for (i=0; i<2; i++) {
		// wb_폜
		if (m_InState[i] != InNotUsed) {
			::midiInUnprepareHeader(m_hIn, &m_InHdr[i], sizeof(MIDIHDR));
			m_InState[i] = InNotUsed;
		}
	}
}

//---------------------------------------------------------------------------
//
//	InR[obN
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::CallbackIn(UINT wMsg, DWORD dwParam1, DWORD /*dwParam2*/)
{
	MIDIHDR *lpMidiHdr;
	DWORD dwLength;
	BYTE msg[3];

	ASSERT(this);
	ASSERT_VALID(this);

	switch (wMsg) {
		// V[gbZ[W
		case MIM_DATA:
		case MIM_MOREDATA:
			// bZ[W쐬
			msg[0] = (BYTE)dwParam1;
			msg[1] = (BYTE)(dwParam1 >> 8);
			msg[2] = (BYTE)(dwParam1 >> 16);

			// Xe[^XoCgɉĒ߂
			if (msg[0] < MIDI_NOTEOFF) {
				// jOXe[^X͖̂ŁAُf[^
				break;
			}
			dwLength = 3;
			if ((msg[0] >= MIDI_PROGCHG) && (msg[0] < MIDI_PITCHCHG)){
				dwLength = 2;
			}
			if (msg[0] >= MIDI_EXSTART) {
				// F0ȏ̃bZ[W
				switch (msg[0]) {
					// \O|WV|C^
					case MIDI_SONGPTR:
						dwLength = 3;
						break;

					// ^CR[hNH[^[t[
					case MIDI_QFRAME:
					// \OZNg
					case MIDI_SONGSEL:
						dwLength = 2;
						break;

					// `[ZNg
					case MIDI_TUNEREQ:
					// ^C~ONbN
					case MIDI_TIMINGCLK:
					// X^[g
					case MIDI_START:
					// ReBj[
					case MIDI_CONTINUE:
					// Xgbv
					case MIDI_STOP:
					// ANeBuZVO
					case MIDI_ACTSENSE:
					// VXeZbg
					case MIDI_SYSRESET:
						dwLength = 1;

					// ͎̑󂯎Ȃ
					default:
						dwLength = 0;
				}
			}

			// f[^ǉ
			if (dwLength > 0) {
				// bN
				m_InSection.Lock();

				// }
				m_InQueue.Insert(msg, dwLength);

				// AbN
				m_InSection.Unlock();

				// JEgAbv
				m_dwShortRecv++;
			}
			break;

		// ObZ[W
		case MIM_LONGDATA:
		case MIM_LONGERROR:
			// tOύX
			lpMidiHdr = (MIDIHDR*)dwParam1;
			if (lpMidiHdr->dwUser < 2) {
				m_InState[lpMidiHdr->dwUser] = InDone;

				// JEgAbv
				m_dwExRecv++;
			}
			break;

		// ̑͏Ȃ
		default:
			break;
	}
}

//---------------------------------------------------------------------------
//
//	V[gbZ[WM
//	NeBJZNV̓bN
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::ShortIn()
{
	BYTE buf[InBufMax];
	DWORD dwReceived;

	ASSERT(this);
	ASSERT(m_pMIDI);
	ASSERT_VALID(this);

	// X^bNɂׂĎ擾
	dwReceived = m_InQueue.Get(buf);
	ASSERT(dwReceived > 0);

	// MIDIfoCX֑M
	m_pMIDI->SetRecvData(buf, (int)dwReceived);
}

//---------------------------------------------------------------------------
//
//	ObZ[WM
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::LongIn(int nHdr)
{
	int i;
	MMRESULT mmResult;

	ASSERT(this);
	ASSERT((nHdr == 0) || (nHdr == 1));
	ASSERT(m_hIn);
	ASSERT_VALID(this);

	// ɎMłƂmF
	ASSERT(m_InHdr[nHdr].dwFlags & MHDR_DONE);

	// MIDIfoCX֑M
	m_pMIDI->SetRecvData((const BYTE*)m_InHdr[nHdr].lpData,
						 (int)m_InHdr[nHdr].dwBytesRecorded);

	// wb_
	::midiInUnprepareHeader(m_hIn, &m_InHdr[nHdr], sizeof(MIDIHDR));
	m_InState[nHdr] = InNotUsed;
	m_dwUnRecv++;

	// ăX^[g
	for (i=0; i<2; i++) {
		if (m_InState[i] == InNotUsed) {
			// MIDIwb_쐬
			memset(&m_InHdr[i], 0, sizeof(MIDIHDR));
			m_InHdr[i].lpData = (LPSTR)&m_InBuf[i][0];
			m_InHdr[i].dwBufferLength = InBufMax;
			m_InHdr[i].dwUser = (DWORD)i;

			// wb_
			mmResult = ::midiInPrepareHeader(m_hIn, &m_InHdr[i], sizeof(MIDIHDR));
			if (mmResult != MMSYSERR_NOERROR) {
				// wb_s
				continue;
			}

			// wb_ǉ
			mmResult = ::midiInAddBuffer(m_hIn, &m_InHdr[i], sizeof(MIDIHDR));
			if (mmResult != MMSYSERR_NOERROR) {
				// wb_ǉs
				::midiInUnprepareHeader(m_hIn, &m_InHdr[i], sizeof(MIDIHDR));
				continue;
			}

			// Xe[gݒ
			m_InState[i] = InReady;
		}
	}
}

//---------------------------------------------------------------------------
//
//	InfoCXCaps擾
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::GetInCaps()
{
	MMRESULT mmResult;
#if _MFC_VER >= 0x700
	UINT_PTR uDeviceID;
#else
	UINT uDeviceID;
#endif	// MFC_VER

	// foCXID
	uDeviceID = 0;

	// [v
	while ((DWORD)uDeviceID < m_dwInDevs) {
		// midiInGetDevCapsŎ擾
		mmResult = ::midiInGetDevCaps(uDeviceID,
									  &m_pInCaps[uDeviceID],
									  sizeof(MIDIINCAPS));

		// ʂL^
		if (mmResult == MMSYSERR_NOERROR) {
			m_pInCaps[uDeviceID].dwSupport = 0;
		}
		else {
			m_pInCaps[uDeviceID].dwSupport = 1;
		}

		// foCXID
		uDeviceID++;
	}
}

//---------------------------------------------------------------------------
//
//	InXbh
//
//---------------------------------------------------------------------------
UINT CMIDI::InThread(LPVOID pParam)
{
	CMIDI *pMIDI;

	// p[^󂯎
	pMIDI = (CMIDI*)pParam;

	// s
	pMIDI->RunIn();

	// I
	return 0;
}

//---------------------------------------------------------------------------
//
//	InR[obN
//
//---------------------------------------------------------------------------
#if _MFC_VER >= 0x700
void CALLBACK CMIDI::InProc(HMIDIIN hIn, UINT wMsg, DWORD_PTR dwInstance,
							DWORD dwParam1, DWORD dwParam2)
#else
void CALLBACK CMIDI::InProc(HMIDIIN hIn, UINT wMsg, DWORD dwInstance,
							DWORD dwParam1, DWORD dwParam2)
#endif
{
	CMIDI *pMIDI;

	// p[^󂯎
	pMIDI = (CMIDI*)dwInstance;

	// Ăяo
	if (hIn == pMIDI->GetInHandle()) {
		pMIDI->CallbackIn(wMsg, dwParam1, dwParam2);
	}
}

//---------------------------------------------------------------------------
//
//	InfoCX擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL CMIDI::GetInDevs() const
{
	ASSERT(this);
	ASSERT_VALID(this);

	return m_dwInDevs;
}

//---------------------------------------------------------------------------
//
//	InfoCX̂擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL CMIDI::GetInDevDesc(int nDevice, CString& strDesc) const
{
	LPCTSTR lpstrDesc;

	ASSERT(this);
	ASSERT(nDevice >= 0);
	ASSERT_VALID(this);

	// CfbNX`FbN
	if (nDevice >= (int)m_dwInDevs) {
		strDesc.Empty();
		return FALSE;
	}

	// LPCTSTR֕ϊ
	lpstrDesc = A2T(m_pInCaps[nDevice].szPname);
	strDesc = lpstrDesc;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	InfBCݒ
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::SetInDelay(int nDelay)
{
	ASSERT(this);
	ASSERT(nDelay >= 0);
	ASSERT_VALID(this);

	// foCX֒ʒm
	m_pMIDI->SetRecvDelay(nDelay);
}

//---------------------------------------------------------------------------
//
//	In擾
//
//---------------------------------------------------------------------------
void FASTCALL CMIDI::GetInInfo(LPMIDIINFO pInfo) const
{
	ASSERT(this);
	ASSERT(pInfo);
	ASSERT_VALID(this);

	// foCX
	pInfo->dwDevices = m_dwInDevs;
	pInfo->dwDevice = m_dwInDevice;

	// JE^
	pInfo->dwShort = m_dwShortRecv;
	pInfo->dwLong = m_dwExRecv;
	pInfo->dwUnprepare = m_dwUnRecv;

	// obt@VMfoCX̂ŁAł͐ݒ肵Ȃ
}

#endif	// _WIN32
