//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ CPU(MC68000) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "iosc.h"
#include "mfp.h"
#include "vm.h"
#include "log.h"
#include "memory.h"
#include "dmac.h"
#include "scc.h"
#include "midi.h"
#include "scsi.h"
#include "fileio.h"
#include "cpu.h"

//---------------------------------------------------------------------------
//
//	AZuRAƂ̃C^tF[X
//
//---------------------------------------------------------------------------
#if defined(__cplusplus)
extern "C" {
#endif	// __cplusplus

//---------------------------------------------------------------------------
//
//	X^eBbN [N
//
//---------------------------------------------------------------------------
static CPU *cpu;
										// CPU

//---------------------------------------------------------------------------
//
//	O`
//
//---------------------------------------------------------------------------
DWORD s68000fbpc(void);
										// PCtB[hobN
void s68000buserr(DWORD addr, DWORD param);
										// oXG[

//---------------------------------------------------------------------------
//
//	RESET߃nh
//
//---------------------------------------------------------------------------
static void cpu_resethandler(void)
{
	cpu->ResetInst();
}

//---------------------------------------------------------------------------
//
//	荞ACK
//
//---------------------------------------------------------------------------
void s68000intack(void)
{
	int sr;

	sr = ::s68000context.sr;
	sr >>= 8;
	sr &= 0x0007;

	cpu->IntAck(sr);
}

//---------------------------------------------------------------------------
//
//	oXG[L^
//
//---------------------------------------------------------------------------
void s68000buserrlog(DWORD addr, DWORD stat)
{
	cpu->BusErrLog(addr, stat);
}

//---------------------------------------------------------------------------
//
//	AhXG[L^
//
//---------------------------------------------------------------------------
void s68000addrerrlog(DWORD addr, DWORD stat)
{
	cpu->AddrErrLog(addr, stat);
}

#if defined(__cplusplus)
}
#endif	// __cplusplus

//===========================================================================
//
//	CPU
//
//===========================================================================
//#define CPU_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
CPU::CPU(VM *p) : Device(p)
{
	// foCXID
	dev.id = MAKEID('C', 'P', 'U', ' ');
	dev.desc = "MPU (MC68000)";

	// |C^
	memory = NULL;
	dmac = NULL;
	mfp = NULL;
	iosc = NULL;
	scc = NULL;
	midi = NULL;
	scsi = NULL;
	scheduler = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL CPU::Init()
{
	ASSERT(this);

	// {NX
	if (!Device::Init()) {
		return FALSE;
	}

	// CPUL
	::cpu = this;

	// 擾
	memory = (Memory*)vm->SearchDevice(MAKEID('M', 'E', 'M', ' '));
	ASSERT(memory);

	// DMAC擾
	dmac = (DMAC*)vm->SearchDevice(MAKEID('D', 'M', 'A', 'C'));
	ASSERT(dmac);

	// MFP擾
	mfp = (MFP*)vm->SearchDevice(MAKEID('M', 'F', 'P', ' '));
	ASSERT(mfp);

	// IOSC擾
	iosc = (IOSC*)vm->SearchDevice(MAKEID('I', 'O', 'S', 'C'));
	ASSERT(iosc);

	// SCC擾
	scc = (SCC*)vm->SearchDevice(MAKEID('S', 'C', 'C', ' '));
	ASSERT(scc);

	// MIDI擾
	midi = (MIDI*)vm->SearchDevice(MAKEID('M', 'I', 'D', 'I'));
	ASSERT(midi);

	// SCSI擾
	scsi = (SCSI*)vm->SearchDevice(MAKEID('S', 'C', 'S', 'I'));
	ASSERT(scsi);

	// XPW[擾
	scheduler = (Scheduler*)vm->SearchDevice(MAKEID('S', 'C', 'H', 'E'));
	ASSERT(scheduler);

	// CPURÃWve[u쐬
	::s68000init();

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL CPU::Cleanup()
{
	ASSERT(this);

	// {NX
	Device::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL CPU::Reset()
{
	int i;
	S68000CONTEXT context;
	DWORD bit;

	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// G[AhXAG[ԃNA
	sub.erraddr = 0;
	sub.errtime = 0;

	// 荞݃JEgNA
	for (i=0; i<8; i++) {
		sub.intreq[i] = 0;
		sub.intack[i] = 0;
	}

	// ReLXg쐬(Zbgp)
	memory->MakeContext(TRUE);

	// Zbg
	::s68000reset();
	::s68000context.resethandler = cpu_resethandler;
	::s68000context.odometer = 0;

	// 荞ׂ݂Ď
	::s68000GetContext(&context);
	for (i=1; i<=7; i++) {
		bit = (1 << i);
		if (context.interrupts[0] & bit) {
			context.interrupts[0] &= (BYTE)(~bit);
			context.interrupts[i] = 0;
		}
	}
	::s68000SetContext(&context);

	// ReLXg쐬(ʏ)
	memory->MakeContext(FALSE);
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL CPU::Save(Fileio *fio, int /*ver*/)
{
	size_t sz;
	cpu_t cpu;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "Z[u");

	// ReLXg擾
	GetCPU(&cpu);

	// TCYZ[u
	sz = sizeof(cpu_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// ̂Z[u
	if (!fio->Write(&cpu, (int)sz)) {
		return FALSE;
	}

	// TCYZ[u(Tu)
	sz = sizeof(cpusub_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// ̂Z[u(Tu)
	if (!fio->Write(&sub, (int)sz)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL CPU::Load(Fileio *fio, int /*ver*/)
{
	cpu_t cpu;
	size_t sz;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "[h");

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(cpu_t)) {
		return FALSE;
	}

	// ̂[h
	if (!fio->Read(&cpu, (int)sz)) {
		return FALSE;
	}

	// Kp(ZbgĂs)
	memory->MakeContext(TRUE);
	::s68000reset();
	memory->MakeContext(FALSE);
	SetCPU(&cpu);

	// TCY[hAƍ(Tu)
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(cpusub_t)) {
		return FALSE;
	}

	// ̂[h(Tu)
	if (!fio->Read(&sub, (int)sz)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL CPU::ApplyCfg(const Config* /*config*/)
{
	ASSERT(this);

	LOG0(Log::Normal, "ݒKp");
}

//---------------------------------------------------------------------------
//
//	CPUWX^擾
//
//---------------------------------------------------------------------------
void FASTCALL CPU::GetCPU(cpu_t *buffer) const
{
	int i;

	ASSERT(this);
	ASSERT(buffer);

	// Dreg, Areg
	for (i=0; i<8; i++) {
		buffer->dreg[i] = ::s68000context.dreg[i];
		buffer->areg[i] = ::s68000context.areg[i];
	}

	// 荞
	for (i=0; i<8; i++) {
		buffer->intr[i] = (DWORD)::s68000context.interrupts[i];
		buffer->intreq[i] = sub.intreq[i];
		buffer->intack[i] = sub.intack[i];
	}

	// ̑
	buffer->sp = ::s68000context.asp;
	buffer->pc = ::s68000context.pc;
	buffer->sr = (DWORD)::s68000context.sr;
	buffer->odd = ::s68000context.odometer;
}

//---------------------------------------------------------------------------
//
//	CPUWX^ݒ
//
//---------------------------------------------------------------------------
void FASTCALL CPU::SetCPU(const cpu_t *buffer)
{
	int i;
	S68000CONTEXT context;

	ASSERT(this);
	ASSERT(buffer);

	// ReLXg擾
	::s68000GetContext(&context);

	// Dreg, Areg
	for (i=0; i<8; i++) {
		context.dreg[i] = buffer->dreg[i];
		context.areg[i] = buffer->areg[i];
	}

	// 荞
	for (i=0; i<8; i++) {
		context.interrupts[i] = (BYTE)buffer->intr[i];
		sub.intreq[i] = buffer->intreq[i];
		sub.intack[i] = buffer->intack[i];
	}

	// ̑
	context.asp = buffer->sp;
	context.pc = buffer->pc;
	context.sr = (WORD)buffer->sr;
	context.odometer = buffer->odd;

	// ReLXgݒ
	::s68000SetContext(&context);
}

//---------------------------------------------------------------------------
//
//	荞
//
//---------------------------------------------------------------------------
BOOL FASTCALL CPU::Interrupt(int level, int vector)
{
	int ret;

	// INTERRUPT SWITCHɂNMI荞݂̓xN^-1
	ASSERT(this);
	ASSERT((level >= 1) && (level <= 7));
	ASSERT(vector >= -1);

	// NGXg
	ret = ::s68000interrupt(level, vector);

	// ʕ]
	if (ret == 0) {
#if defined(CPU_LOG)
		LOG2(Log::Normal, "荞ݗv x%d xN^$%02X", level, vector);
#endif	// CPU_LOG
		sub.intreq[level]++;
		return TRUE;
	}

	return FALSE;
}

//---------------------------------------------------------------------------
//
//	荞ACK
//
//---------------------------------------------------------------------------
void FASTCALL CPU::IntAck(int level)
{
	ASSERT(this);
	ASSERT((level >= 1) && (level <= 7));

#if defined(CPU_LOG)
	LOG1(Log::Normal, "荞ݗvACK x%d", level);
#endif	// CPU_LOG

	// JEgAbv
	sub.intack[level]++;

	// 荞݃x
	switch (level) {
		// IOSC,SCSI()
		case 1:
			iosc->IntAck();
			scsi->IntAck(1);
			break;

		// MIDI,SCSI(x2)
		case 2:
			midi->IntAck(2);
			scsi->IntAck(2);
			break;

		// DMAC
		case 3:
			dmac->IntAck();
			break;

		// MIDI,SCSI(x4)
		case 4:
			midi->IntAck(4);
			scsi->IntAck(4);
			break;

		// SCC
		case 5:
			scc->IntAck();
			break;

		// MFP
		case 6:
			mfp->IntAck();
			break;

		// ̑
		default:
			break;
	}
}

//---------------------------------------------------------------------------
//
//	荞݃LZ
//
//---------------------------------------------------------------------------
void FASTCALL CPU::IntCancel(int level)
{
	S68000CONTEXT context;
	DWORD bit;

	ASSERT(this);
	ASSERT((level >= 1) && (level <= 7));

	// ReLXg𒼐ڏ
	::s68000GetContext(&context);

	// YrbgIȂ
	bit = (1 << level);
	if (context.interrupts[0] & bit) {
#if defined(CPU_LOG)
		LOG1(Log::Normal, "荞݃LZ x%d", level);
#endif	// CPU_LOG

		// rbg~낷
		context.interrupts[0] &= (BYTE)(~bit);

		// xN^0
		context.interrupts[level] = 0;

		// NGXg
		sub.intreq[level]--;
	}

	// ReLXg
	::s68000SetContext(&context);
}

//---------------------------------------------------------------------------
//
//	RESET
//
//---------------------------------------------------------------------------
void FASTCALL CPU::ResetInst()
{
	Device *device;

	ASSERT(this);
	LOG0(Log::Detail, "RESET");

	// 擾
	device = (Device*)vm->SearchDevice(MAKEID('M', 'E', 'M', ' '));
	ASSERT(device);

	// foCXɑ΂ĂׂăZbgĂ
	// mɂ́ACPURESETMǂ܂œ`Ă邩ɂ
	while (device) {
		device->Reset();
		device = device->GetNextDevice();
	}
}

//---------------------------------------------------------------------------
//
//	oXG[
//	DMA]ɂoXG[ɗ
//	CPURAŃoXG[Ɣ肵ꍇ́AoRȂ
//
//---------------------------------------------------------------------------
void FASTCALL CPU::BusErr(DWORD addr, BOOL read)
{
	DWORD pc;
	DWORD stat;

	ASSERT(this);
	ASSERT(addr <= 0xffffff);

	// DMACɓ]BDMACȂDMACɔC
	if (dmac->IsDMA()) {
		dmac->BusErr(addr, read);
		return;
	}

	// AhXÕAhX+2ŁAԂȂ疳(LONGANZX)
	if (addr == (sub.erraddr + 2)) {
		if (scheduler->GetTotalTime() == sub.errtime) {
			return;
		}
	}

	// AhXƎԂXV
	sub.erraddr = addr;
	sub.errtime = scheduler->GetTotalTime();

	// PC擾(Y߂̃IyR[hɈʒu)
	pc = GetPC();

	// ǂݏo(Word)
	stat = memory->ReadOnly(pc);
	stat <<= 8;
	stat |= memory->ReadOnly(pc + 1);
	stat <<= 16;

	// t@NVR[h쐬(Ƀf[^ANZXƂ݂Ȃ)
	stat |= 0x09;
	if (::s68000context.sr & 0x2000) {
		stat |= 0x04;
	}
	if (read) {
		stat |= 0x10;
	}

	// oXG[s
	::s68000buserr(addr, stat);
}

//---------------------------------------------------------------------------
//
//	AhXG[
//	DMA]ɂAhXG[ɗ
//	CPURAŃAhXG[Ɣ肵ꍇ́AoRȂ
//
//---------------------------------------------------------------------------
void FASTCALL CPU::AddrErr(DWORD addr, BOOL read)
{
	DWORD pc;
	DWORD stat;

	ASSERT(this);
	ASSERT(addr <= 0xffffff);
	ASSERT(addr & 1);

	// DMACɓ]BDMACȂDMACɔC
	if (dmac->IsDMA()) {
		dmac->AddrErr(addr, read);
		return;
	}

	// AhXÕAhX+2ŁAԂȂ疳(LONGANZX)
	if (addr == (sub.erraddr + 2)) {
		if (scheduler->GetTotalTime() == sub.errtime) {
			return;
		}
	}

	// AhXƎԂXV
	sub.erraddr = addr;
	sub.errtime = scheduler->GetTotalTime();

	// PC擾(Y߂̃IyR[hɈʒu)
	pc = GetPC();

	// ǂݏo(Word)
	stat = memory->ReadOnly(pc);
	stat <<= 8;
	stat |= memory->ReadOnly(pc + 1);
	stat <<= 16;

	// t@NVR[h쐬(Ƀf[^ANZXƂ݂Ȃ)
	stat |= 0x8009;
	if (::s68000context.sr & 0x2000) {
		stat |= 0x04;
	}
	if (read) {
		stat |= 0x10;
	}

	// oXG[s(ŃAhXG[֕)
	::s68000buserr(addr, stat);
}

//---------------------------------------------------------------------------
//
//	oXG[L^
//	CPURAŃoXG[Ɣ肵ꍇAʂ
//
//---------------------------------------------------------------------------
void FASTCALL CPU::BusErrLog(DWORD addr, DWORD stat)
{
	ASSERT(this);

	// K}XN(24bit𒴂ꍇ)
	addr &= 0xffffff;

	if (stat & 0x10) {
		LOG1(Log::Warning, "oXG[(ǂݍ) $%06X", addr);
	}
	else {
		LOG1(Log::Warning, "oXG[() $%06X", addr);
	}
}

//---------------------------------------------------------------------------
//
//	AhXG[L^
//	CPURAŃAhXG[Ɣ肵ꍇAʂ
//
//---------------------------------------------------------------------------
void FASTCALL CPU::AddrErrLog(DWORD addr, DWORD stat)
{
	ASSERT(this);

	// K}XN(24bit𒴂ꍇ)
	addr &= 0xffffff;

	if (stat & 0x10) {
		LOG1(Log::Warning, "AhXG[(ǂݍ) $%06X", addr);
	}
	else {
		LOG1(Log::Warning, "AhXG[() $%06X", addr);
	}
}
