//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ fBXN ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "fileio.h"
#include "vm.h"
#include "sasi.h"
#include "disk.h"

//===========================================================================
//
//	fBXNgbN
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
DiskTrack::DiskTrack(int track, int size, int sectors, BOOL raw)
{
	ASSERT(track >= 0);
	ASSERT((size == 8) || (size == 9) || (size == 11));
	ASSERT((sectors > 0) && (sectors <= 0x100));

	// p[^ݒ
	dt.track = track;
	dt.size = size;
	dt.sectors = sectors;
	dt.raw = raw;

	// ĂȂ([hKv)
	dt.init = FALSE;

	// ύXĂȂ
	dt.changed = FALSE;

	// I[N݂͑Ȃ
	dt.buffer = NULL;
	dt.changemap = NULL;
}

//---------------------------------------------------------------------------
//
//	fXgN^
//
//---------------------------------------------------------------------------
DiskTrack::~DiskTrack()
{
	// ͍sAZ[u͂Ȃ
	if (dt.buffer) {
		delete[] dt.buffer;
		dt.buffer = NULL;
	}
	if (dt.changemap) {
		delete[] dt.changemap;
		dt.changemap = NULL;
	}
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskTrack::Load(const Filepath& path)
{
	Fileio fio;
	DWORD offset;
	int i;
	int length;

	ASSERT(this);

	// Ƀ[hĂΕsv
	if (dt.init) {
		ASSERT(dt.buffer);
		ASSERT(dt.changemap);
		return TRUE;
	}

	ASSERT(!dt.buffer);
	ASSERT(!dt.changemap);

	// ItZbgvZ(ȑÕgbN256ZN^ێƂ݂Ȃ)
	offset = (dt.track << 8);
	if (dt.raw) {
		ASSERT(dt.size == 11);
		offset *= 0x930;
		offset += 0x10;
	}
	else {
		offset <<= dt.size;
	}

	// OXvZ(̃gbÑf[^TCY)
	length = dt.sectors << dt.size;

	// obt@̃m
	ASSERT((dt.size == 8) || (dt.size == 9) || (dt.size == 11));
	ASSERT((dt.sectors > 0) && (dt.sectors <= 0x100));
	try {
		dt.buffer = new BYTE[ length ];
	}
	catch (...) {
		dt.buffer = NULL;
		return FALSE;
	}
	if (!dt.buffer) {
		return FALSE;
	}

	// ύX}bṽm
	try {
		dt.changemap = new BOOL[dt.sectors];
	}
	catch (...) {
		dt.changemap = NULL;
		return FALSE;
	}
	if (!dt.changemap) {
		return FALSE;
	}

	// ύX}bvNA
	for (i=0; i<dt.sectors; i++) {
		dt.changemap[i] = FALSE;
	}

	// t@Cǂݍ
	if (!fio.Open(path, Fileio::ReadOnly)) {
		return FALSE;
	}
	if (dt.raw) {
		// ǂ
		for (i=0; i<dt.sectors; i++) {
			// V[N
			if (!fio.Seek(offset)) {
				fio.Close();
				return FALSE;
			}

			// ǂݍ
			if (!fio.Read(&dt.buffer[i << dt.size], 1 << dt.size)) {
				fio.Close();
				return FALSE;
			}

			// ̃ItZbg
			offset += 0x930;
		}
	}
	else {
		// Aǂ
		if (!fio.Seek(offset)) {
			fio.Close();
			return FALSE;
		}
		if (!fio.Read(dt.buffer, length)) {
			fio.Close();
			return FALSE;
		}
	}
	fio.Close();

	// tO𗧂āAI
	dt.init = TRUE;
	dt.changed = FALSE;
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskTrack::Save(const Filepath& path)
{
	DWORD offset;
	int i;
	Fileio fio;
	int length;

	ASSERT(this);

	// ĂȂΕsv
	if (!dt.init) {
		return TRUE;
	}

	// ύXĂȂΕsv
	if (!dt.changed) {
		return TRUE;
	}

	// ޕKv
	ASSERT(dt.buffer);
	ASSERT(dt.changemap);
	ASSERT((dt.size == 8) || (dt.size == 9) || (dt.size == 11));
	ASSERT((dt.sectors > 0) && (dt.sectors <= 0x100));

	// RAW[hł͏݂͂肦Ȃ
	ASSERT(!dt.raw);

	// ItZbgvZ(ȑÕgbN256ZN^ێƂ݂Ȃ)
	offset = (dt.track << 8);
	offset <<= dt.size;

	// ZN^̃OXvZ
	length = 1 << dt.size;

	// t@CI[v
	if (!fio.Open(path, Fileio::ReadWrite)) {
		return FALSE;
	}

	// ݃[v
	for (i=0; i<dt.sectors; i++) {
		// ύXĂ
		if (dt.changemap[i]) {
			// V[NA
			if (!fio.Seek(offset + (i << dt.size))) {
				fio.Close();
				return FALSE;
			}
			if (!fio.Write(&dt.buffer[i << dt.size], length)) {
				fio.Close();
				return FALSE;
			}

			// ύXtO𗎂Ƃ
			dt.changemap[i] = FALSE;
		}
	}

	// N[Y
	fio.Close();

	// ύXtO𗎂ƂAI
	dt.changed = FALSE;
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[hZN^
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskTrack::Read(BYTE *buf, int sec) const
{
	ASSERT(this);
	ASSERT(buf);
	ASSERT((sec >= 0) & (sec < 0x100));

	// ĂȂ΃G[
	if (!dt.init) {
		return FALSE;
	}

	// ZN^L𒴂Ă΃G[
	if (sec >= dt.sectors) {
		return FALSE;
	}

	// Rs[
	ASSERT(dt.buffer);
	ASSERT((dt.size == 8) || (dt.size == 9) || (dt.size == 11));
	ASSERT((dt.sectors > 0) && (dt.sectors <= 0x100));
	memcpy(buf, &dt.buffer[sec << dt.size], 1 << dt.size);

	// 
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	CgZN^
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskTrack::Write(const BYTE *buf, int sec)
{
	int offset;
	int length;

	ASSERT(this);
	ASSERT(buf);
	ASSERT((sec >= 0) & (sec < 0x100));
	ASSERT(!dt.raw);

	// ĂȂ΃G[
	if (!dt.init) {
		return FALSE;
	}

	// ZN^L𒴂Ă΃G[
	if (sec >= dt.sectors) {
		return FALSE;
	}

	// ItZbgAOXvZ
	offset = sec << dt.size;
	length = 1 << dt.size;

	// r
	ASSERT(dt.buffer);
	ASSERT((dt.size == 8) || (dt.size == 9) || (dt.size == 11));
	ASSERT((dt.sectors > 0) && (dt.sectors <= 0x100));
	if (memcmp(buf, &dt.buffer[offset], length) == 0) {
		// ̂ƂĂ̂ŁAI
		return TRUE;
	}

	// Rs[AύX
	memcpy(&dt.buffer[offset], buf, length);
	dt.changemap[sec] = TRUE;
	dt.changed = TRUE;

	// 
	return TRUE;
}

//===========================================================================
//
//	fBXNLbV
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
DiskCache::DiskCache(const Filepath& path, int size, int blocks)
{
	int i;

	ASSERT((size == 8) || (size == 9) || (size == 11));
	ASSERT(blocks > 0);

	// LbV[N
	for (i=0; i<CacheMax; i++) {
		cache[i].disktrk = NULL;
		cache[i].serial = 0;
	}

	// ̑
	serial = 0;
	sec_path = path;
	sec_size = size;
	sec_blocks = blocks;
	cd_raw = FALSE;
}

//---------------------------------------------------------------------------
//
//	fXgN^
//
//---------------------------------------------------------------------------
DiskCache::~DiskCache()
{
	// gbNNA
	Clear();
}

//---------------------------------------------------------------------------
//
//	RAW[hݒ
//
//---------------------------------------------------------------------------
void FASTCALL DiskCache::SetRawMode(BOOL raw)
{
	ASSERT(this);
	ASSERT(sec_size == 11);

	// ݒ
	cd_raw = raw;
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskCache::Save()
{
	int i;

	ASSERT(this);

	// gbNۑ
	for (i=0; i<CacheMax; i++) {
		// LȃgbN
		if (cache[i].disktrk) {
			// ۑ
			if (!cache[i].disktrk->Save(sec_path)) {
				return FALSE;
			}
		}
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	fBXNLbV擾
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskCache::GetCache(int index, int& track, DWORD& serial) const
{
	ASSERT(this);
	ASSERT((index >= 0) && (index < CacheMax));

	// gpȂFALSE
	if (!cache[index].disktrk) {
		return FALSE;
	}

	// gbNƃVAݒ
	track = cache[index].disktrk->GetTrack();
	serial = cache[index].serial;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	NA
//
//---------------------------------------------------------------------------
void FASTCALL DiskCache::Clear()
{
	int i;

	ASSERT(this);

	// LbV[N
	for (i=0; i<CacheMax; i++) {
		if (cache[i].disktrk) {
			delete cache[i].disktrk;
			cache[i].disktrk = NULL;
		}
	}
}

//---------------------------------------------------------------------------
//
//	ZN^[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskCache::Read(BYTE *buf, int block)
{
	int track;
	DiskTrack *disktrk;

	ASSERT(this);
	ASSERT(sec_size != 0);

	// ɍXV
	Update();

	// gbNZo(256ZN^/gbNɌŒ)
	track = block >> 8;

	// ̃gbNf[^𓾂
	disktrk = Assign(track);
	if (!disktrk) {
		return FALSE;
	}

	// gbNɔC
	return disktrk->Read(buf, (BYTE)block);
}

//---------------------------------------------------------------------------
//
//	ZN^Cg
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskCache::Write(const BYTE *buf, int block)
{
	int track;
	DiskTrack *disktrk;

	ASSERT(this);
	ASSERT(sec_size != 0);

	// ɍXV
	Update();

	// gbNZo(256ZN^/gbNɌŒ)
	track = block >> 8;

	// ̃gbNf[^𓾂
	disktrk = Assign(track);
	if (!disktrk) {
		return FALSE;
	}

	// gbNɔC
	return disktrk->Write(buf, (BYTE)block);
}

//---------------------------------------------------------------------------
//
//	gbN̊蓖
//
//---------------------------------------------------------------------------
DiskTrack* FASTCALL DiskCache::Assign(int track)
{
	int i;
	int c;
	DWORD s;

	ASSERT(this);
	ASSERT(sec_size != 0);
	ASSERT(track >= 0);

	// ܂AɊ蓖ĂĂȂׂ
	for (i=0; i<CacheMax; i++) {
		if (cache[i].disktrk) {
			if (cache[i].disktrk->GetTrack() == track) {
				// gbNv
				cache[i].serial = serial;
				return cache[i].disktrk;
			}
		}
	}

	// ɁA󂢂Ă̂Ȃׂ
	for (i=0; i<CacheMax; i++) {
		if (!cache[i].disktrk) {
			// [h݂
			if (Load(i, track)) {
				// [h
				cache[i].serial = serial;
				return cache[i].disktrk;
			}

			// [hs
			return NULL;
		}
	}

	// ŌɁAVAԍ̈ԎႢ̂TA폜

	// CfbNX0cƂ
	s = cache[0].serial;
	c = 0;

	// ƃVArA菬̂֍XV
	for (i=0; i<CacheMax; i++) {
		ASSERT(cache[i].disktrk);

		// ɑ݂VAƔrAXV
		if (cache[i].serial < s) {
			s = cache[i].serial;
			c = i;
		}
	}

	// ̃gbNۑ
	if (!cache[c].disktrk->Save(sec_path)) {
		return NULL;
	}

	// ̃gbN폜
	delete cache[c].disktrk;
	cache[c].disktrk = NULL;

	// [h
	if (Load(c, track)) {
		// [h
		cache[c].serial = serial;
		return cache[c].disktrk;
	}

	// [hs
	return NULL;
}

//---------------------------------------------------------------------------
//
//	gbÑ[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL DiskCache::Load(int index, int track)
{
	int sectors;
	DiskTrack *disktrk;

	ASSERT(this);
	ASSERT((index >= 0) && (index < CacheMax));
	ASSERT(track >= 0);
	ASSERT(!cache[index].disktrk);

	// ̃gbÑZN^擾
	sectors = sec_blocks - (track << 8);
	ASSERT(sectors > 0);
	if (sectors > 0x100) {
		sectors = 0x100;
	}

	// fBXNgbN쐬
	disktrk = new DiskTrack(track, sec_size, sectors, cd_raw);

	// [h݂
	if (!disktrk->Load(sec_path)) {
		// s
		delete disktrk;
		return FALSE;
	}

	// 蓖ĐA[Nݒ
	cache[index].disktrk = disktrk;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	VAԍ̍XV
//
//---------------------------------------------------------------------------
void FASTCALL DiskCache::Update()
{
	int i;

	ASSERT(this);

	// XVāA0ȊO͉Ȃ
	serial++;
	if (serial != 0) {
		return;
	}

	// SLbṼVANA(32bit[vĂ)
	for (i=0; i<CacheMax; i++) {
		cache[i].serial = 0;
	}
}

//===========================================================================
//
//	fBXN
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
Disk::Disk(Device *dev)
{
	// Rg[ƂȂfoCXL
	ctrl = dev;

	// [N
	disk.id = MAKEID('N', 'U', 'L', 'L');
	disk.ready = FALSE;
	disk.writep = FALSE;
	disk.readonly = FALSE;
	disk.removable = FALSE;
	disk.lock = FALSE;
	disk.attn = FALSE;
	disk.reset = FALSE;
	disk.size = 0;
	disk.blocks = 0;
	disk.lun = 0;
	disk.code = 0;
	disk.dcache = NULL;
}

//---------------------------------------------------------------------------
//
//	fXgN^
//
//---------------------------------------------------------------------------
Disk::~Disk()
{
	// fBXNLbV̕ۑ
	if (disk.ready) {
		// fB̏ꍇ̂
		ASSERT(disk.dcache);
		disk.dcache->Save();
	}

	// fBXNLbV̍폜
	if (disk.dcache) {
		delete disk.dcache;
		disk.dcache = NULL;
	}
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL Disk::Reset()
{
	ASSERT(this);

	// bNȂAAeVȂAZbg
	disk.lock = FALSE;
	disk.attn = FALSE;
	disk.reset = TRUE;
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Save(Fileio *fio, int ver)
{
	size_t sz;

	ASSERT(this);
	ASSERT(fio);

	// TCYZ[u
	sz = sizeof(disk_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// ̂Z[u
	if (!fio->Write(&disk, (int)sz)) {
		return FALSE;
	}

	// pXZ[u
	if (!diskpath.Save(fio, ver)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Load(Fileio *fio, int ver)
{
	size_t sz;
	disk_t buf;
	Filepath path;

	ASSERT(this);
	ASSERT(fio);

	// version2.03ÓAfBXN̓Z[uĂȂ
	if (ver <= 0x0202) {
		return TRUE;
	}

	// ݂̃fBXNLbV폜
	if (disk.dcache) {
		disk.dcache->Save();
		delete disk.dcache;
		disk.dcache = NULL;
	}

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(disk_t)) {
		return FALSE;
	}

	// obt@փ[h
	if (!fio->Read(&buf, (int)sz)) {
		return FALSE;
	}

	// pX[h
	if (!path.Load(fio, ver)) {
		return FALSE;
	}

	// IDvꍇ̂݁Aړ
	if (disk.id == buf.id) {
		// NULLȂ牽Ȃ
		if (IsNULL()) {
			return TRUE;
		}

		// Z[uƓނ̃foCX
		disk.ready = FALSE;
		if (Open(path)) {
			// OpenŃfBXNLbV͍쐬Ă
			// vpeB݈̂ړ
			if (!disk.readonly) {
				disk.writep = buf.writep;
			}
			disk.lock = buf.lock;
			disk.attn = buf.attn;
			disk.reset = buf.reset;
			disk.lun = buf.lun;
			disk.code = buf.code;

			// Ƀ[hł
			return TRUE;
		}
	}

	// fBXNLbVč쐬
	if (!IsReady()) {
		disk.dcache = NULL;
	}
	else {
		disk.dcache = new DiskCache(diskpath, disk.size, disk.blocks);
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	NULL`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::IsNULL() const
{
	ASSERT(this);

	if (disk.id == MAKEID('N', 'U', 'L', 'L')) {
		return TRUE;
	}
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	SASI`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::IsSASI() const
{
	ASSERT(this);

	if (disk.id == MAKEID('S', 'A', 'H', 'D')) {
		return TRUE;
	}
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	I[v
//	hNXŁAI[v̌㏈ƂČĂяo
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Open(const Filepath& path)
{
	Fileio fio;

	ASSERT(this);
	ASSERT((disk.size == 8) || (disk.size == 9) || (disk.size == 11));
	ASSERT(disk.blocks > 0);

	// fB
	disk.ready = TRUE;

	// LbV
	ASSERT(!disk.dcache);
	disk.dcache = new DiskCache(path, disk.size, disk.blocks);

	// ǂݏI[v\
	if (fio.Open(path, Fileio::ReadWrite)) {
		// ݋A[hI[łȂ
		disk.writep = FALSE;
		disk.readonly = FALSE;
		fio.Close();
	}
	else {
		// ݋֎~A[hI[
		disk.writep = TRUE;
		disk.readonly = TRUE;
	}

	// bNĂȂ
	disk.lock = FALSE;

	// pXۑ
	diskpath = path;

	// 
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	CWFNg
//
//---------------------------------------------------------------------------
void FASTCALL Disk::Eject(BOOL force)
{
	ASSERT(this);

	// [oułȂ΃CWFNgłȂ
	if (!disk.removable) {
		return;
	}

	// fBłȂ΃CWFNgKvȂ
	if (!disk.ready) {
		return;
	}

	// tOȂꍇAbNĂȂƂKv
	if (!force) {
		if (disk.lock) {
			return;
		}
	}

	// fBXNLbV폜
	disk.dcache->Save();
	delete disk.dcache;
	disk.dcache = NULL;

	// mbgfBAAeVȂ
	disk.ready = FALSE;
	disk.writep = FALSE;
	disk.readonly = FALSE;
	disk.attn = FALSE;
}

//---------------------------------------------------------------------------
//
//	݋֎~
//
//---------------------------------------------------------------------------
void FASTCALL Disk::WriteP(BOOL writep)
{
	ASSERT(this);

	// fBł邱
	if (!disk.ready) {
		return;
	}

	// Read Only̏ꍇ́AveNgԂ̂
	if (disk.readonly) {
		ASSERT(disk.writep);
		return;
	}

	// tOݒ
	disk.writep = writep;
}

//---------------------------------------------------------------------------
//
//	[N擾
//
//---------------------------------------------------------------------------
void FASTCALL Disk::GetDisk(disk_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);

	// [NRs[
	*buffer = disk;
}

//---------------------------------------------------------------------------
//
//	pX擾
//
//---------------------------------------------------------------------------
void FASTCALL Disk::GetPath(Filepath& path) const
{
	path = diskpath;
}

//---------------------------------------------------------------------------
//
//	tbV
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Flush()
{
	ASSERT(this);

	// LbVȂΉȂ
	if (!disk.dcache) {
		return TRUE;
	}

	// LbVۑ
	return disk.dcache->Save();
}

//---------------------------------------------------------------------------
//
//	fB`FbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::CheckReady()
{
	ASSERT(this);

	// ZbgȂAXe[^XԂ
	if (disk.reset) {
		disk.code = DISK_DEVRESET;
		disk.reset = FALSE;
		return FALSE;
	}

	// AeVȂAXe[^XԂ
	if (disk.attn) {
		disk.code = DISK_ATTENTION;
		disk.attn = FALSE;
		return FALSE;
	}

	// mbgfBȂAXe[^XԂ
	if (!disk.ready) {
		disk.code = DISK_NOTREADY;
		return FALSE;
	}

	// G[Ȃɏ
	disk.code = DISK_NOERROR;
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	INQUIRY
//	펞Kv
//
//---------------------------------------------------------------------------
int FASTCALL Disk::Inquiry(const DWORD* /*cdb*/, BYTE* /*buf*/)
{
	ASSERT(this);

	// ftHgINQUIRYs
	disk.code = DISK_INVALIDCMD;
	return 0;
}

//---------------------------------------------------------------------------
//
//	REQUEST SENSE
//	SASI͕ʏ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::RequestSense(const DWORD *cdb, BYTE *buf)
{
	int size;

	ASSERT(this);
	ASSERT(cdb);
	ASSERT(buf);

	// G[ȂꍇɌAmbgfB`FbN
	if (disk.code == DISK_NOERROR) {
		if (!disk.ready) {
			disk.code = DISK_NOTREADY;
		}
	}

	// TCY(AP[VOXɏ])
	size = (int)cdb[4];
	ASSERT((size >= 0) && (size < 0x100));

	// SCSI-1ł́ATCY0̂Ƃ4oCg](SCSI-2ł͂̎dl͍폜)
	if (size == 0) {
		size = 4;
	}

	// obt@NA
	memset(buf, 0, size);

	// gZXf[^܂߂A18oCgݒ
	buf[0] = 0x70;
	buf[2] = (BYTE)(disk.code >> 16);
	buf[7] = 10;
	buf[12] = (BYTE)(disk.code >> 8);
	buf[13] = (BYTE)disk.code;

	// R[hNA
	disk.code = 0x00;

	return size;
}

//---------------------------------------------------------------------------
//
//	MODE SELECT`FbN
//	disk.codẻe󂯂Ȃ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::SelectCheck(const DWORD *cdb)
{
	int length;

	ASSERT(this);
	ASSERT(cdb);

	// Z[up[^ݒ肳Ă΃G[
	if (cdb[1] & 0x01) {
		disk.code = DISK_INVALIDCDB;
		return 0;
	}

	// p[^OXŎw肳ꂽf[^󂯎
	length = (int)cdb[4];
	return length;
}

//---------------------------------------------------------------------------
//
//	MODE SELECT
//	disk.codẻe󂯂Ȃ
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::ModeSelect(const BYTE *buf, int size)
{
	ASSERT(this);
	ASSERT(buf);
	ASSERT(size >= 0);

	// ݒłȂ
	disk.code = DISK_INVALIDPRM;

	return FALSE;
}

//---------------------------------------------------------------------------
//
//	MODE SENSE
//	disk.codẻe󂯂Ȃ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::ModeSense(const DWORD *cdb, BYTE *buf)
{
	int page;
	int length;
	int size;
	BOOL valid;
	BOOL change;

	ASSERT(this);
	ASSERT(cdb);
	ASSERT(buf);
	ASSERT(cdb[0] == 0x1a);

	// OX擾Aobt@NA
	length = (int)cdb[4];
	ASSERT((length >= 0) && (length < 0x100));
	memset(buf, 0, length);

	// ύX\tO擾
	if ((cdb[2] & 0xc0) == 0x40) {
		change = TRUE;
	}
	else {
		change = FALSE;
	}

	// y[WR[h擾(0x00͍ŏvalid)
	page = cdb[2] & 0x3f;
	if (page == 0x00) {
		valid = TRUE;
	}
	else {
		valid = FALSE;
	}

	// {
	size = 4;
	if (disk.writep) {
		buf[2] = 0x80;
	}

	// DBD0ȂAubNfBXNv^ǉ
	if ((cdb[1] & 0x08) == 0) {
		// [hp[^wb_
		buf[ 3] = 0x08;

		// fB̏ꍇɌ
		if (disk.ready) {
			// ubNfBXNv^(ubN)
			buf[ 5] = (BYTE)(disk.blocks >> 16);
			buf[ 6] = (BYTE)(disk.blocks >> 8);
			buf[ 7] = (BYTE)disk.blocks;

			// ubNfBXNv^(ubNOX)
			size = 1 << disk.size;
			buf[ 9] = (BYTE)(size >> 16);
			buf[10] = (BYTE)(size >> 8);
			buf[11] = (BYTE)size;
		}

		// TCYĐݒ
		size = 12;
	}

	// y[WR[h1(read-write error recovery)
	if ((page == 0x01) || (page == 0x3f)) {
		size += AddError(change, &buf[size]);
		valid = TRUE;
	}

	// y[WR[h3(format device)
	if ((page == 0x03) || (page == 0x3f)) {
		size += AddFormat(change, &buf[size]);
		valid = TRUE;
	}

	// y[WR[h6(optical)
	if (disk.id == MAKEID('S', 'C', 'M', 'O')) {
		if ((page == 0x06) || (page == 0x3f)) {
			size += AddOpt(change, &buf[size]);
			valid = TRUE;
		}
	}

	// y[WR[h8(caching)
	if ((page == 0x08) || (page == 0x3f)) {
		size += AddCache(change, &buf[size]);
		valid = TRUE;
	}

	// y[WR[h13(CD-ROM)
	if (disk.id == MAKEID('S', 'C', 'C', 'D')) {
		if ((page == 0x0d) || (page == 0x3f)) {
			size += AddCDROM(change, &buf[size]);
			valid = TRUE;
		}
	}

	// y[WR[h14(CD-DA)
	if (disk.id == MAKEID('S', 'C', 'C', 'D')) {
		if ((page == 0x0e) || (page == 0x3f)) {
			size += AddCDDA(change, &buf[size]);
			valid = TRUE;
		}
	}

	// [hf[^OXŏIݒ
	buf[0] = (BYTE)(size - 1);

	// T|[gĂȂy[W
	if (!valid) {
		disk.code = DISK_INVALIDCDB;
		return 0;
	}

	// Saved values̓T|[gĂȂ
	if ((cdb[2] & 0xc0) == 0xc0) {
		disk.code = DISK_PARAMSAVE;
		return 0;
	}

	// MODE SENSE
	disk.code = DISK_NOERROR;
	return length;
}

//---------------------------------------------------------------------------
//
//	G[y[Wǉ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::AddError(BOOL change, BYTE *buf)
{
	ASSERT(this);
	ASSERT(buf);

	// R[hEOXݒ
	buf[0] = 0x01;
	buf[1] = 0x0a;

	// ύX\̈͂Ȃ
	if (change) {
		return 12;
	}

	// gCJEg0A~bg^C͑ũftHglgp
	return 12;
}

//---------------------------------------------------------------------------
//
//	tH[}bgy[Wǉ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::AddFormat(BOOL change, BYTE *buf)
{
	ASSERT(this);
	ASSERT(buf);

	// R[hEOXݒ
	buf[0] = 0x03;
	buf[1] = 0x16;

	// ύX\̈͂Ȃ
	if (change) {
		return 24;
	}

	// [ouݒ
	if (disk.removable) {
		buf[20] = 0x20;
	}

	return 24;
}

//---------------------------------------------------------------------------
//
//	IveBJy[Wǉ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::AddOpt(BOOL change, BYTE *buf)
{
	ASSERT(this);
	ASSERT(buf);

	// R[hEOXݒ
	buf[0] = 0x06;
	buf[1] = 0x02;

	// ύX\̈͂Ȃ
	if (change) {
		return 4;
	}

	// XVubÑ|[g͍sȂ
	return 4;
}

//---------------------------------------------------------------------------
//
//	LbVy[Wǉ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::AddCache(BOOL change, BYTE *buf)
{
	ASSERT(this);
	ASSERT(buf);

	// R[hEOXݒ
	buf[0] = 0x08;
	buf[1] = 0x0a;

	// ύX\̈͂Ȃ
	if (change) {
		return 12;
	}

	// ǂݍ݃LbV̂ݗLAvtFb`͍sȂ
	return 12;
}

//---------------------------------------------------------------------------
//
//	CD-ROMy[Wǉ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::AddCDROM(BOOL change, BYTE *buf)
{
	ASSERT(this);
	ASSERT(buf);

	// R[hEOXݒ
	buf[0] = 0x0d;
	buf[1] = 0x06;

	// ύX\̈͂Ȃ
	if (change) {
		return 8;
	}

	// CANeBu^C}2sec
	buf[3] = 0x05;

	// MSF{͂ꂼ60, 75
	buf[5] = 60;
	buf[7] = 75;

	return 8;
}

//---------------------------------------------------------------------------
//
//	CD-DAy[Wǉ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::AddCDDA(BOOL change, BYTE *buf)
{
	ASSERT(this);
	ASSERT(buf);

	// R[hEOXݒ
	buf[0] = 0x0e;
	buf[1] = 0x0e;

	// ύX\̈͂Ȃ
	if (change) {
		return 16;
	}

	// I[fBI͑슮҂A gbNɂ܂PLAY
	return 16;
}

//---------------------------------------------------------------------------
//
//	TEST UNIT READY
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::TestUnitReady(const DWORD* /*cdb*/)
{
	ASSERT(this);

	// ԃ`FbN
	if (!CheckReady()) {
		return FALSE;
	}

	// TEST UNIT READY
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	REZERO UNIT
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Rezero(const DWORD* /*cdb*/)
{
	ASSERT(this);

	// ԃ`FbN
	if (!CheckReady()) {
		return FALSE;
	}

	// REZERO
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	FORMAT UNIT
//	SASI̓IyR[h$06ASCSI̓IyR[h$04
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Format(const DWORD *cdb)
{
	ASSERT(this);

	// ԃ`FbN
	if (!CheckReady()) {
		return FALSE;
	}

	// FMTDATA=1̓T|[gȂ
	if (cdb[1] & 0x10) {
		disk.code = DISK_INVALIDCDB;
		return FALSE;
	}

	// FORMAT
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	REASSIGN BLOCKS
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Reassign(const DWORD* /*cdb*/)
{
	ASSERT(this);

	// ԃ`FbN
	if (!CheckReady()) {
		return FALSE;
	}

	// REASSIGN BLOCKS
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	READ
//
//---------------------------------------------------------------------------
int FASTCALL Disk::Read(BYTE *buf, int block)
{
	ASSERT(this);
	ASSERT(buf);
	ASSERT(block >= 0);

	// ԃ`FbN
	if (!CheckReady()) {
		return 0;
	}

	// g[^ubN𒴂Ă΃G[
	if (block >= disk.blocks) {
		disk.code = DISK_INVALIDLBA;
		return 0;
	}

	// LbVɔC
	if (!disk.dcache->Read(buf, block)) {
		disk.code = DISK_READFAULT;
		return 0;
	}

	// 
	return (1 << disk.size);
}

//---------------------------------------------------------------------------
//
//	WRITE`FbN
//
//---------------------------------------------------------------------------
int FASTCALL Disk::WriteCheck(int block)
{
	ASSERT(this);
	ASSERT(block >= 0);

	// ԃ`FbN
	if (!CheckReady()) {
		return 0;
	}

	// g[^ubN𒴂Ă΃G[
	if (block >= disk.blocks) {
		return 0;
	}

	// ݋֎~ȂG[
	if (disk.writep) {
		disk.code = DISK_WRITEPROTECT;
		return 0;
	}

	// 
	return (1 << disk.size);
}

//---------------------------------------------------------------------------
//
//	WRITE
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Write(const BYTE *buf, int block)
{
	ASSERT(this);
	ASSERT(buf);
	ASSERT(block >= 0);

	// fBłȂ΃G[
	if (!disk.ready) {
		disk.code = DISK_NOTREADY;
		return FALSE;
	}

	// g[^ubN𒴂Ă΃G[
	if (block >= disk.blocks) {
		disk.code = DISK_INVALIDLBA;
		return FALSE;
	}

	// ݋֎~ȂG[
	if (disk.writep) {
		disk.code = DISK_WRITEPROTECT;
		return FALSE;
	}

	// LbVɔC
	if (!disk.dcache->Write(buf, block)) {
		disk.code = DISK_WRITEFAULT;
		return FALSE;
	}

	// 
	disk.code = DISK_NOERROR;
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	SEEK
//	LBÃ`FbN͍sȂ(SASI IOCS)
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Seek(const DWORD* /*cdb*/)
{
	ASSERT(this);

	// ԃ`FbN
	if (!CheckReady()) {
		return FALSE;
	}

	// SEEK
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	START STOP UNIT
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::StartStop(const DWORD *cdb)
{
	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x1b);

	// CWFNgrbgāAKvȂCWFNg
	if (cdb[4] & 0x02) {
		if (disk.lock) {
			// bNĂ̂ŁACWFNgłȂ
			disk.code = DISK_PREVENT;
			return FALSE;
		}

		// CWFNg
		Eject(FALSE);
	}

	// OK
	disk.code = DISK_NOERROR;
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	SEND DIAGNOSTIC
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::SendDiag(const DWORD *cdb)
{
	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x1d);

	// PFrbg̓T|[gȂ
	if (cdb[1] & 0x10) {
		disk.code = DISK_INVALIDCDB;
		return FALSE;
	}

	// p[^Xg̓T|[gȂ
	if ((cdb[3] != 0) || (cdb[4] != 0)) {
		disk.code = DISK_INVALIDCDB;
		return FALSE;
	}

	// ɐ
	disk.code = DISK_NOERROR;
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	PREVENT/ALLOW MEDIUM REMOVAL
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Removal(const DWORD *cdb)
{
	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x1e);

	// ԃ`FbN
	if (!CheckReady()) {
		return FALSE;
	}

	// bNtOݒ
	if (cdb[4] & 0x01) {
		disk.lock = TRUE;
	}
	else {
		disk.lock = FALSE;
	}

	// REMOVAL
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	READ CAPACITY
//
//---------------------------------------------------------------------------
int FASTCALL Disk::ReadCapacity(const DWORD* /*cdb*/, BYTE *buf)
{
	DWORD blocks;
	DWORD length;

	ASSERT(this);
	ASSERT(buf);

	// obt@NA
	memset(buf, 0, 8);

	// ԃ`FbN
	if (!CheckReady()) {
		return 0;
	}

	// _ubNAhX̏I[(disk.blocks - 1)쐬
	ASSERT(disk.blocks > 0);
	blocks = disk.blocks - 1;
	buf[0] = (BYTE)(blocks >> 24);
	buf[1] = (BYTE)(blocks >> 16);
	buf[2] = (BYTE)(blocks >>  8);
	buf[3] = (BYTE)blocks;

	// ubNOX(1 << disk.size)쐬
	length = 1 << disk.size;
	buf[4] = (BYTE)(length >> 24);
	buf[5] = (BYTE)(length >> 16);
	buf[6] = (BYTE)(length >> 8);
	buf[7] = (BYTE)length;

	// ԑTCYԂ
	return 8;
}

//---------------------------------------------------------------------------
//
//	VERIFY
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::Verify(const DWORD *cdb)
{
	int record;
    int blocks;

	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x2f);

	// p[^擾
	record = cdb[2];
	record <<= 8;
	record |= cdb[3];
	record <<= 8;
	record |= cdb[4];
	record <<= 8;
	record |= cdb[5];
	blocks = cdb[7];
	blocks <<= 8;
	blocks |= cdb[8];

	// ԃ`FbN
	if (!CheckReady()) {
		return 0;
	}

	// p[^`FbN
	if (disk.blocks < (record + blocks)) {
		disk.code = DISK_INVALIDLBA;
		return FALSE;
	}

	// 
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	READ TOC
//
//---------------------------------------------------------------------------
int FASTCALL Disk::ReadToc(const DWORD *cdb, BYTE *buf)
{
	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x43);
	ASSERT(buf);

	// ̃R}h̓T|[gȂ
	disk.code = DISK_INVALIDCMD;
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::PlayAudio(const DWORD *cdb)
{
	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x45);

	// ̃R}h̓T|[gȂ
	disk.code = DISK_INVALIDCMD;
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO MSF
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::PlayAudioMSF(const DWORD *cdb)
{
	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x47);

	// ̃R}h̓T|[gȂ
	disk.code = DISK_INVALIDCMD;
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO TRACK
//
//---------------------------------------------------------------------------
BOOL FASTCALL Disk::PlayAudioTrack(const DWORD *cdb)
{
	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x48);

	// ̃R}h̓T|[gȂ
	disk.code = DISK_INVALIDCMD;
	return FALSE;
}

//===========================================================================
//
//	SASI n[hfBXN
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
SASIHD::SASIHD(Device *dev) : Disk(dev)
{
	// SASI n[hfBXN
	disk.id = MAKEID('S', 'A', 'H', 'D');
}

//---------------------------------------------------------------------------
//
//	I[v
//
//---------------------------------------------------------------------------
BOOL FASTCALL SASIHD::Open(const Filepath& path)
{
	Fileio fio;
	DWORD size;

	ASSERT(this);
	ASSERT(!disk.ready);

	// ǂݍ݃I[vKv
	if (!fio.Open(path, Fileio::ReadOnly)) {
		return FALSE;
	}

	// t@CTCY̎擾
	size = fio.GetFileSize();
	fio.Close();

	// 10MB, 20MB, 40MB̂
	switch (size) {
		// 10MB
		case 0x9f5400:
			break;

		// 20MB
		case 0x13c9800:
			break;

		// 40MB
		case 0x2793000:
			break;

		// ̑(T|[gȂ)
		default:
			return FALSE;
	}

	// ZN^TCYƃubN
	disk.size = 8;
	disk.blocks = size >> 8;

	// {NX
	return Disk::Open(path);
}

//---------------------------------------------------------------------------
//
//	foCXZbg
//
//---------------------------------------------------------------------------
void FASTCALL SASIHD::Reset()
{
	ASSERT(this);

	// bNԉAAeV
	disk.lock = FALSE;
	disk.attn = FALSE;

	// ZbgȂAR[hNA
	disk.reset = FALSE;
	disk.code = 0x00;
}

//---------------------------------------------------------------------------
//
//	REQUEST SENSE
//
//---------------------------------------------------------------------------
int FASTCALL SASIHD::RequestSense(const DWORD *cdb, BYTE *buf)
{
	int size;

	ASSERT(this);
	ASSERT(cdb);
	ASSERT(buf);

	// TCY
	size = (int)cdb[4];
	ASSERT((size >= 0) && (size < 0x100));

	// SASI͔gtH[}bgɌŒ
	memset(buf, 0, size);
	buf[0] = (BYTE)(disk.code >> 16);
	buf[1] = (BYTE)(disk.lun << 5);

	// R[hNA
	disk.code = 0x00;

	return size;
}

//===========================================================================
//
//	SCSI n[hfBXN
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
SCSIHD::SCSIHD(Device *dev) : Disk(dev)
{
	// SCSI n[hfBXN
	disk.id = MAKEID('S', 'C', 'H', 'D');
}

//---------------------------------------------------------------------------
//
//	I[v
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSIHD::Open(const Filepath& path)
{
	Fileio fio;
	DWORD size;

	ASSERT(this);
	ASSERT(!disk.ready);

	// ǂݍ݃I[vKv
	if (!fio.Open(path, Fileio::ReadOnly)) {
		return FALSE;
	}

	// t@CTCY̎擾
	size = fio.GetFileSize();
	fio.Close();

	// 512oCgPʂł邱
	if (size & 0x1ff) {
		return FALSE;
	}

	// 10MBȏ4GB
	if (size < 0x9f5400) {
		return FALSE;
	}
	if (size > 0xfff00000) {
		return FALSE;
	}

	// ZN^TCYƃubN
	disk.size = 9;
	disk.blocks = size >> 9;

	// {NX
	return Disk::Open(path);
}

//---------------------------------------------------------------------------
//
//	INQUIRY
//
//---------------------------------------------------------------------------
int FASTCALL SCSIHD::Inquiry(const DWORD *cdb, BYTE *buf)
{
	DWORD major;
	DWORD minor;
	char string[32];
	int size;
	int len;

	ASSERT(this);
	ASSERT(cdb);
	ASSERT(buf);
	ASSERT(cdb[0] == 0x12);

	// EVPD`FbN
	if (cdb[1] & 0x01) {
		disk.code = DISK_INVALIDCDB;
		return FALSE;
	}

	// fB`FbN(C[Wt@CȂꍇAG[Ƃ)
	if (!disk.ready) {
		disk.code = DISK_NOTREADY;
		return FALSE;
	}

	// {f[^
	// buf[0] ... Direct Access Device
	// buf[2] ... SCSI-2̃R}ȟn
	// buf[3] ... SCSI-2InquiryX|X
	// buf[4] ... Inquiryǉf[^
	memset(buf, 0, 8);
	buf[2] = 0x02;
	buf[3] = 0x02;
	buf[4] = 0x1f;

	// x_
	memset(&buf[8], 0x20, 28);
	memcpy(&buf[8], "XM6", 3);

	// i
	size = disk.blocks >> 11;
	if (size < 300)
		sprintf(string, "PRODRIVE LPS%dS", size);
	else if (size < 600)
		sprintf(string, "MAVERICK%dS", size);
	else if (size < 800)
		sprintf(string, "LIGHTNING%dS", size);
	else if (size < 1000)
		sprintf(string, "TRAILBRAZER%dS", size);
	else if (size < 2000)
		sprintf(string, "FIREBALL%dS", size);
	else
		sprintf(string, "FBSE%d.%dS", size / 1000, (size % 1000) / 100);
	memcpy(&buf[16], string, strlen(string));

	// rW(XM6̃o[WNo)
	ctrl->GetVM()->GetVersion(major, minor);
	sprintf(string, "0%01d%01d%01d",
				major, (minor >> 4), (minor & 0x0f));
	memcpy((char*)&buf[32], string, 4);

	// TCY36oCgAP[VOX̂AZœ]
	size = 36;
	len = (int)cdb[4];
	if (len < size) {
		size = len;
	}

	// 
	disk.code = DISK_NOERROR;
	return size;
}

//===========================================================================
//
//	SCSI CfBXN
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
SCSIMO::SCSIMO(Device *dev) : Disk(dev)
{
	// SCSI CfBXN
	disk.id = MAKEID('S', 'C', 'M', 'O');

	// [ou
	disk.removable = TRUE;
}

//---------------------------------------------------------------------------
//
//	I[v
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSIMO::Open(const Filepath& path, BOOL attn)
{
	Fileio fio;
	DWORD size;

	ASSERT(this);
	ASSERT(!disk.ready);

	// ǂݍ݃I[vKv
	if (!fio.Open(path, Fileio::ReadOnly)) {
		return FALSE;
	}

	// t@CTCY̎擾
	size = fio.GetFileSize();
	fio.Close();

	switch (size) {
		// 128MB
		case 0x797f400:
			disk.size = 9;
			disk.blocks = 248826;
			break;

		// 230MB
		case 0xd9eea00:
			disk.size = 9;
			disk.blocks = 446325;
			break;

		// 540MB
		case 0x1fc8b800:
			disk.size = 9;
			disk.blocks = 1041500;
			break;

		// 640MB
		case 0x25e28000:
			disk.size = 11;
			disk.blocks = 310352;
			break;

		// ̑(G[)
		default:
			return FALSE;
	}

	// {NX
	Disk::Open(path);

	// fBȂAeV
	if (disk.ready && attn) {
		disk.attn = TRUE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSIMO::Load(Fileio *fio, int ver)
{
	size_t sz;
	disk_t buf;
	Filepath path;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x0200);

	// version2.03ÓAfBXN̓Z[uĂȂ
	if (ver <= 0x0202) {
		return TRUE;
	}

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(disk_t)) {
		return FALSE;
	}

	// obt@փ[h
	if (!fio->Read(&buf, (int)sz)) {
		return FALSE;
	}

	// pX[h
	if (!path.Load(fio, ver)) {
		return FALSE;
	}

	// KCWFNg
	Eject(TRUE);

	// IDvꍇ̂݁Aړ
	if (disk.id != buf.id) {
		// Z[uMOłȂBCWFNgԂێ
		return TRUE;
	}

	// ăI[v݂
	if (!Open(path, FALSE)) {
		// ăI[vłȂBCWFNgԂێ
		return TRUE;
	}

	// OpenŃfBXNLbV͍쐬ĂBvpeB݈̂ړ
	if (!disk.readonly) {
		disk.writep = buf.writep;
	}
	disk.lock = buf.lock;
	disk.attn = buf.attn;
	disk.reset = buf.reset;
	disk.lun = buf.lun;
	disk.code = buf.code;

	// Ƀ[hł
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	INQUIRY
//
//---------------------------------------------------------------------------
int FASTCALL SCSIMO::Inquiry(const DWORD *cdb, BYTE *buf)
{
	DWORD major;
	DWORD minor;
	char string[32];
	int size;
	int len;

	ASSERT(this);
	ASSERT(cdb);
	ASSERT(buf);
	ASSERT(cdb[0] == 0x12);

	// EVPD`FbN
	if (cdb[1] & 0x01) {
		disk.code = DISK_INVALIDCDB;
		return FALSE;
	}

	// {f[^
	// buf[0] ... Optical Memory Device
	// buf[1] ... [ou
	// buf[2] ... SCSI-2̃R}ȟn
	// buf[3] ... SCSI-2InquiryX|X
	// buf[4] ... Inquiryǉf[^
	memset(buf, 0, 8);
	buf[0] = 0x07;
	buf[1] = 0x80;
	buf[2] = 0x02;
	buf[3] = 0x02;
	buf[4] = 0x1f;

	// x_
	memset(&buf[8], 0x20, 28);
	memcpy(&buf[8], "XM6", 3);

	// i
	memcpy(&buf[16], "M2513A", 6);

	// rW(XM6̃o[WNo)
	ctrl->GetVM()->GetVersion(major, minor);
	sprintf(string, "0%01d%01d%01d",
				major, (minor >> 4), (minor & 0x0f));
	memcpy((char*)&buf[32], string, 4);

	// TCY36oCgAP[VOX̂AZœ]
	size = 36;
	len = cdb[4];
	if (len < size) {
		size = len;
	}

	// 
	disk.code = DISK_NOERROR;
	return size;
}

//===========================================================================
//
//	CDgbN
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
CDTrack::CDTrack(SCSICD *scsicd)
{
	ASSERT(scsicd);

	// eƂȂCD-ROMfoCXݒ
	cdrom = scsicd;

	// gbN
	valid = FALSE;

	// ̑̃f[^
	track_no = -1;
	first_lba = 0;
	last_lba = 0;
	audio = FALSE;
	raw = FALSE;
}

//---------------------------------------------------------------------------
//
//	fXgN^
//
//---------------------------------------------------------------------------
CDTrack::~CDTrack()
{
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL CDTrack::Init(int track, DWORD first, DWORD last)
{
	ASSERT(this);
	ASSERT(!valid);
	ASSERT(track >= 1);
	ASSERT(first < last);

	// gbNԍݒAL
	track_no = track;
	valid = TRUE;

	// LBAL
	first_lba = first;
	last_lba = last;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	pXݒ
//
//---------------------------------------------------------------------------
void FASTCALL CDTrack::SetPath(BOOL cdda, const Filepath& path)
{
	ASSERT(this);
	ASSERT(valid);

	// CD-DAAf[^
	audio = cdda;

	// pXL
	imgpath = path;
}

//---------------------------------------------------------------------------
//
//	pX擾
//
//---------------------------------------------------------------------------
void FASTCALL CDTrack::GetPath(Filepath& path) const
{
	ASSERT(this);
	ASSERT(valid);

	// pXԂ
	path = imgpath;
}

//---------------------------------------------------------------------------
//
//	CfbNXǉ
//
//---------------------------------------------------------------------------
void FASTCALL CDTrack::AddIndex(int index, DWORD lba)
{
	ASSERT(this);
	ASSERT(valid);
	ASSERT(index > 0);
	ASSERT(first_lba <= lba);
	ASSERT(lba <= last_lba);

	// ݂̓CfbNX̓T|[gȂ
	ASSERT(FALSE);
}

//---------------------------------------------------------------------------
//
//	JnLBA擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL CDTrack::GetFirst() const
{
	ASSERT(this);
	ASSERT(valid);
	ASSERT(first_lba < last_lba);

	return first_lba;
}

//---------------------------------------------------------------------------
//
//	I[LBA擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL CDTrack::GetLast() const
{
	ASSERT(this);
	ASSERT(valid);
	ASSERT(first_lba < last_lba);

	return last_lba;
}

//---------------------------------------------------------------------------
//
//	ubN擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL CDTrack::GetBlocks() const
{
	ASSERT(this);
	ASSERT(valid);
	ASSERT(first_lba < last_lba);

	// JnLBAƍŏILBAZo
	return (DWORD)(last_lba - first_lba + 1);
}

//---------------------------------------------------------------------------
//
//	gbNԍ擾
//
//---------------------------------------------------------------------------
int FASTCALL CDTrack::GetTrackNo() const
{
	ASSERT(this);
	ASSERT(valid);
	ASSERT(track_no >= 1);

	return track_no;
}

//---------------------------------------------------------------------------
//
//	LȃubN
//
//---------------------------------------------------------------------------
BOOL FASTCALL CDTrack::IsValid(DWORD lba) const
{
	ASSERT(this);

	// gbN̂ȂAFALSE
	if (!valid) {
		return FALSE;
	}

	// firstOȂAFALSE
	if (lba < first_lba) {
		return FALSE;
	}

	// lastȂAFALSE
	if (last_lba < lba) {
		return FALSE;
	}

	// ̃gbN
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	I[fBIgbN
//
//---------------------------------------------------------------------------
BOOL FASTCALL CDTrack::IsAudio() const
{
	ASSERT(this);
	ASSERT(valid);

	return audio;
}

//===========================================================================
//
//	CD-DAobt@
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
CDDABuf::CDDABuf()
{
}

//---------------------------------------------------------------------------
//
//	fXgN^
//
//---------------------------------------------------------------------------
CDDABuf::~CDDABuf()
{
}

//===========================================================================
//
//	SCSI CD-ROM
//
//===========================================================================

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
SCSICD::SCSICD(Device *dev) : Disk(dev)
{
	int i;

	// SCSI CD-ROM
	disk.id = MAKEID('S', 'C', 'C', 'D');

	// [ouA݋֎~
	disk.removable = TRUE;
	disk.writep = TRUE;

	// RAW`łȂ
	rawfile = FALSE;

	// t[
	frame = 0;

	// gbN
	for (i=0; i<TrackMax; i++) {
		track[i] = NULL;
	}
	tracks = 0;
	dataindex = -1;
	audioindex = -1;
}

//---------------------------------------------------------------------------
//
//	fXgN^
//
//---------------------------------------------------------------------------
SCSICD::~SCSICD()
{
	// gbNNA
	ClearTrack();
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSICD::Load(Fileio *fio, int ver)
{
	size_t sz;
	disk_t buf;
	Filepath path;

	ASSERT(this);
	ASSERT(fio);
	ASSERT(ver >= 0x0200);

	// version2.03ÓAfBXN̓Z[uĂȂ
	if (ver <= 0x0202) {
		return TRUE;
	}

	// TCY[hAƍ
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(disk_t)) {
		return FALSE;
	}

	// obt@փ[h
	if (!fio->Read(&buf, (int)sz)) {
		return FALSE;
	}

	// pX[h
	if (!path.Load(fio, ver)) {
		return FALSE;
	}

	// KCWFNg
	Eject(TRUE);

	// IDvꍇ̂݁Aړ
	if (disk.id != buf.id) {
		// Z[uCD-ROMłȂBCWFNgԂێ
		return TRUE;
	}

	// ăI[v݂
	if (!Open(path, FALSE)) {
		// ăI[vłȂBCWFNgԂێ
		return TRUE;
	}

	// OpenŃfBXNLbV͍쐬ĂBvpeB݈̂ړ
	if (!disk.readonly) {
		disk.writep = buf.writep;
	}
	disk.lock = buf.lock;
	disk.attn = buf.attn;
	disk.reset = buf.reset;
	disk.lun = buf.lun;
	disk.code = buf.code;

	// ēxAfBXNLbVj
	if (disk.dcache) {
		delete disk.dcache;
		disk.dcache = NULL;
	}
	disk.dcache = NULL;

	// b
	disk.blocks = track[0]->GetBlocks();
	if (disk.blocks > 0) {
		// fBXNLbV蒼
		track[0]->GetPath(path);
		disk.dcache = new DiskCache(path, disk.size, disk.blocks);
		disk.dcache->SetRawMode(rawfile);

		// f[^CfbNXĐݒ
		dataindex = 0;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	I[v
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSICD::Open(const Filepath& path, BOOL attn)
{
	Fileio fio;
	DWORD size;
	char file[5];

	ASSERT(this);
	ASSERT(!disk.ready);

	// AgbNNA
	disk.blocks = 0;
	rawfile = FALSE;
	ClearTrack();

	// ǂݍ݃I[vKv
	if (!fio.Open(path, Fileio::ReadOnly)) {
		return FALSE;
	}

	// TCY擾
	size = fio.GetFileSize();
	if (size <= 4) {
		fio.Close();
		return FALSE;
	}

	// CUEV[gAISOt@C̔s
	fio.Read(file, 4);
	file[4] = '\0';
	fio.Close();

	// FILEŎn܂Ă΁ACUEV[gƂ݂Ȃ
	if (strnicmp(file, "FILE", 4) == 0) {
		// CUEƂăI[v
		if (!OpenCue(path)) {
			return FALSE;
		}
	}
	else {
		// ISOƂăI[v
		if (!OpenIso(path)) {
			return FALSE;
		}
	}

	// I[v
	ASSERT(disk.blocks > 0);
	disk.size = 11;

	// {NX
	Disk::Open(path);

	// RAWtOݒ
	ASSERT(disk.dcache);
	disk.dcache->SetRawMode(rawfile);

	// ROMfBAȂ̂ŁA݂͂łȂ
	disk.writep = TRUE;

	// fBȂAeV
	if (disk.ready && attn) {
		disk.attn = TRUE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	I[v(CUE)
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSICD::OpenCue(const Filepath& path)
{
	ASSERT(this);

	// Ɏs
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	I[v(ISO)
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSICD::OpenIso(const Filepath& path)
{
	Fileio fio;
	DWORD size;
	BYTE header[12];
	BYTE sync[12];

	ASSERT(this);

	// ǂݍ݃I[vKv
	if (!fio.Open(path, Fileio::ReadOnly)) {
		return FALSE;
	}

	// TCY擾
	size = fio.GetFileSize();
	if (size < 0x800) {
		fio.Close();
		return FALSE;
	}

	// ŏ12oCgǂݎāAN[Y
	if (!fio.Read(header, sizeof(header))) {
		fio.Close();
		return FALSE;
	}

	// RAW``FbN
	memset(sync, 0xff, sizeof(sync));
	sync[0] = 0x00;
	sync[11] = 0x00;
	rawfile = FALSE;
	if (memcmp(header, sync, sizeof(sync)) == 0) {
		// 00,FFx10,00Ȃ̂ŁARAW`Ɛ肳
		if (!fio.Read(header, 4)) {
			fio.Close();
			return FALSE;
		}

		// MODE1/2048܂MODE1/2352̂݃T|[g
		if (header[3] != 0x01) {
			// [hႤ
			fio.Close();
			return FALSE;
		}

		// RAWt@Cɐݒ
		rawfile = TRUE;
	}
	fio.Close();

	if (rawfile) {
		// TCY2536̔{ŁA700MBȉł邱
		if (size % 0x930) {
			return FALSE;
		}
		if (size > 912579600) {
			return FALSE;
		}

		// ubNݒ
		disk.blocks = size / 0x930;
	}
	else {
		// TCY2048̔{ŁA700MBȉł邱
		if (size & 0x7ff) {
			return FALSE;
		}
		if (size > 0x2bed5000) {
			return FALSE;
		}

		// ubNݒ
		disk.blocks = size >> 11;
	}

	// f[^gbN1̂ݍ쐬
	ASSERT(!track[0]);
	track[0] = new CDTrack(this);
	track[0]->Init(1, 0, disk.blocks - 1);
	track[0]->SetPath(FALSE, path);
	tracks = 1;
	dataindex = 0;

	// I[v
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	INQUIRY
//
//---------------------------------------------------------------------------
int FASTCALL SCSICD::Inquiry(const DWORD *cdb, BYTE *buf)
{
	DWORD major;
	DWORD minor;
	char string[32];
	int size;
	int len;

	ASSERT(this);
	ASSERT(cdb);
	ASSERT(buf);
	ASSERT(cdb[0] == 0x12);

	// EVPD`FbN
	if (cdb[1] & 0x01) {
		disk.code = DISK_INVALIDCDB;
		return FALSE;
	}

	// {f[^
	// buf[0] ... CD-ROM Device
	// buf[1] ... [ou
	// buf[2] ... SCSI-2̃R}ȟn
	// buf[3] ... SCSI-2InquiryX|X
	// buf[4] ... Inquiryǉf[^
	memset(buf, 0, 8);
	buf[0] = 0x05;
	buf[1] = 0x80;
	buf[2] = 0x02;
	buf[3] = 0x02;
	buf[4] = 0x1f;

	// x_
	memset(&buf[8], 0x20, 28);
	memcpy(&buf[8], "XM6", 3);

	// i
	memcpy(&buf[16], "CDU-55S", 7);

	// rW(XM6̃o[WNo)
	ctrl->GetVM()->GetVersion(major, minor);
	sprintf(string, "0%01d%01d%01d",
				major, (minor >> 4), (minor & 0x0f));
	memcpy((char*)&buf[32], string, 4);

	// TCY36oCgAP[VOX̂AZœ]
	size = 36;
	len = cdb[4];
	if (len < size) {
		size = len;
	}

	// 
	disk.code = DISK_NOERROR;
	return size;
}

//---------------------------------------------------------------------------
//
//	READ
//
//---------------------------------------------------------------------------
int FASTCALL SCSICD::Read(BYTE *buf, int block)
{
	int index;
	Filepath path;

	ASSERT(this);
	ASSERT(buf);
	ASSERT(block >= 0);

	// ԃ`FbN
	if (!CheckReady()) {
		return 0;
	}

	// gbN
	index = SearchTrack(block);

	// ȂA͈͊O
	if (index < 0) {
		disk.code = DISK_INVALIDLBA;
		return 0;
	}
	ASSERT(track[index]);

	// ݂̃f[^gbNƈقȂĂ
	if (dataindex != index) {
		// ݂̃fBXNLbV폜(Save̕Kv͂Ȃ)
		delete disk.dcache;
		disk.dcache = NULL;

		// ubNĐݒ
		disk.blocks = track[index]->GetBlocks();
		ASSERT(disk.blocks > 0);

		// fBXNLbV蒼
		track[index]->GetPath(path);
		disk.dcache = new DiskCache(path, disk.size, disk.blocks);
		disk.dcache->SetRawMode(rawfile);

		// f[^CfbNXĐݒ
		dataindex = index;
	}

	// {NX
	ASSERT(dataindex >= 0);
	return Disk::Read(buf, block);
}

//---------------------------------------------------------------------------
//
//	READ TOC
//
//---------------------------------------------------------------------------
int FASTCALL SCSICD::ReadToc(const DWORD *cdb, BYTE *buf)
{
	int last;
	int index;
	int length;
	int loop;
	int i;
	BOOL msf;
	DWORD lba;

	ASSERT(this);
	ASSERT(cdb);
	ASSERT(cdb[0] == 0x43);
	ASSERT(buf);

	// fB`FbN
	if (!CheckReady()) {
		return 0;
	}

	// fBłȂAgbNŒ1ȏ㑶݂
	ASSERT(tracks > 0);
	ASSERT(track[0]);

	// AP[VOX擾Aobt@NA
	length = cdb[7] << 8;
	length |= cdb[8];
	memset(buf, 0, length);

	// MSFtO擾
	if (cdb[1] & 0x02) {
		msf = TRUE;
	}
	else {
		msf = FALSE;
	}

	// ŏIgbNԍ擾A`FbN
	last = track[tracks - 1]->GetTrackNo();
	if ((int)cdb[6] > last) {
		// AA͏O
		if (cdb[6] != 0xaa) {
			disk.code = DISK_INVALIDCDB;
			return 0;
		}
	}

	// JnCfbNX`FbN
	index = 0;
	if (cdb[6] != 0x00) {
		// gbNԍv܂ŁAgbNi߂
		while (track[index]) {
			if ((int)cdb[6] == track[index]->GetTrackNo()) {
				break;
			}
			index++;
		}

		// ȂAAAG[
		if (!track[index]) {
			if (cdb[6] == 0xaa) {
				// AAȂ̂ŁAŏILBA+1Ԃ
				buf[0] = 0x00;
				buf[1] = 0x0a;
				buf[2] = (BYTE)track[0]->GetTrackNo();
				buf[3] = (BYTE)last;
				buf[6] = 0xaa;
				lba = track[tracks -1]->GetLast() + 1;
				if (msf) {
					LBAtoMSF(lba, &buf[8]);
				}
				else {
					buf[10] = (BYTE)(lba >> 8);
					buf[11] = (BYTE)lba;
				}
				return length;
			}

			// ȊO̓G[
			disk.code = DISK_INVALIDCDB;
			return 0;
		}
	}

	// ԂgbNfBXNv^̌([v)
	loop = last - track[index]->GetTrackNo() + 1;
	ASSERT(loop >= 1);

	// wb_쐬
	buf[0] = (BYTE)(((loop << 3) + 2) >> 8);
	buf[1] = (BYTE)((loop << 3) + 2);
	buf[2] = (BYTE)track[0]->GetTrackNo();
	buf[3] = (BYTE)last;
	buf += 4;

	// [v
	for (i=0; i<loop; i++) {
		// ADRControl
		if (track[index]->IsAudio()) {
			// I[fBIgbN
			buf[1] = 0x10;
		}
		else {
			// f[^gbN
			buf[1] = 0x14;
		}

		// gbNԍ
		buf[2] = (BYTE)track[index]->GetTrackNo();

		// gbNAhX
		if (msf) {
			LBAtoMSF(track[index]->GetFirst(), &buf[4]);
		}
		else {
			buf[6] = (BYTE)(track[index]->GetFirst() >> 8);
			buf[7] = (BYTE)(track[index]->GetFirst());
		}

		// obt@ƃCfbNXi߂
		buf += 8;
		index++;
	}

	// AP[VOXKԂ
	return length;
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSICD::PlayAudio(const DWORD *cdb)
{
	ASSERT(this);

	disk.code = DISK_INVALIDCDB;
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO MSF
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSICD::PlayAudioMSF(const DWORD *cdb)
{
	ASSERT(this);

	disk.code = DISK_INVALIDCDB;
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	PLAY AUDIO TRACK
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSICD::PlayAudioTrack(const DWORD *cdb)
{
	ASSERT(this);

	disk.code = DISK_INVALIDCDB;
	return FALSE;
}

//---------------------------------------------------------------------------
//
//	LBAMSFϊ
//
//---------------------------------------------------------------------------
void FASTCALL SCSICD::LBAtoMSF(DWORD lba, BYTE *msf) const
{
	DWORD m;
	DWORD s;
	DWORD f;

	ASSERT(this);

	// 75A75*60łꂼ]o
	m = lba / (75 * 60);
	s = lba % (75 * 60);
	f = s % 75;
	s /= 75;

	// _M=0,S=2,F=0
	s += 2;
	if (s >= 60) {
		s -= 60;
		m++;
	}

	// i[
	ASSERT(m < 0x100);
	ASSERT(s < 60);
	ASSERT(f < 75);
	msf[0] = 0x00;
	msf[1] = (BYTE)m;
	msf[2] = (BYTE)s;
	msf[3] = (BYTE)f;
}

//---------------------------------------------------------------------------
//
//	MSFLBAϊ
//
//---------------------------------------------------------------------------
DWORD FASTCALL SCSICD::MSFtoLBA(const BYTE *msf) const
{
	DWORD lba;

	ASSERT(this);
	ASSERT(msf[2] < 60);
	ASSERT(msf[3] < 75);

	// 1, 75, 75*60̔{ōZ
	lba = msf[1];
	lba *= 60;
	lba += msf[2];
	lba *= 75;
	lba += msf[3];

	// _M=0,S=2,F=0Ȃ̂ŁA150
	lba -= 150;

	return lba;
}

//---------------------------------------------------------------------------
//
//	gbNNA
//
//---------------------------------------------------------------------------
void FASTCALL SCSICD::ClearTrack()
{
	int i;

	ASSERT(this);

	// gbNIuWFNg폜
	for (i=0; i<TrackMax; i++) {
		if (track[i]) {
			delete track[i];
			track[i] = NULL;
		}
	}

	// gbN0
	tracks = 0;

	// f[^AI[fBIƂݒȂ
	dataindex = -1;
	audioindex = -1;
}

//---------------------------------------------------------------------------
//
//	gbN
//	Ȃ-1Ԃ
//
//---------------------------------------------------------------------------
int FASTCALL SCSICD::SearchTrack(DWORD lba) const
{
	int i;

	ASSERT(this);

	// gbN[v
	for (i=0; i<tracks; i++) {
		// gbNɕ
		ASSERT(track[i]);
		if (track[i]->IsValid(lba)) {
			return i;
		}
	}

	// Ȃ
	return -1;
}

//---------------------------------------------------------------------------
//
//	t[ʒm
//
//---------------------------------------------------------------------------
BOOL FASTCALL SCSICD::NextFrame()
{
	ASSERT(this);
	ASSERT((frame >= 0) && (frame < 75));

	// t[0`74͈̔͂Őݒ
	frame = (frame + 1) % 75;

	// 1FALSE
	if (frame != 0) {
		return TRUE;
	}
	else {
		return FALSE;
	}
}

//---------------------------------------------------------------------------
//
//	CD-DAobt@擾
//
//---------------------------------------------------------------------------
void FASTCALL SCSICD::GetBuf(DWORD *buffer, int samples, DWORD rate)
{
	ASSERT(this);
}
