//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ DMAC(HD63450) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "cpu.h"
#include "memory.h"
#include "vm.h"
#include "log.h"
#include "schedule.h"
#include "fdc.h"
#include "fileio.h"
#include "dmac.h"

//===========================================================================
//
//	DMAC
//
//===========================================================================
//#define DMAC_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
DMAC::DMAC(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('D', 'M', 'A', 'C');
	dev.desc = "DMAC (HD63450)";

	// JnAhXAIAhX
	memdev.first = 0xe84000;
	memdev.last = 0xe85fff;

	// ̑
	memory = NULL;
	fdc = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL DMAC::Init()
{
	int ch;

	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// 擾
	memory = (Memory*)vm->SearchDevice(MAKEID('M', 'E', 'M', ' '));
	ASSERT(memory);

	// FDC擾
	fdc = (FDC*)vm->SearchDevice(MAKEID('F', 'D', 'C', ' '));
	ASSERT(fdc);

	// `l[N
	for (ch=0; ch<4; ch++) {
		memset(&dma[ch], 0, sizeof(dma[ch]));
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::Cleanup()
{
	ASSERT(this);

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::Reset()
{
	int ch;

	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// O[o
	dmactrl.transfer = 0;
	dmactrl.load = 0;
	dmactrl.exec = FALSE;
	dmactrl.current_ch = 0;
	dmactrl.cpu_cycle = 0;
	dmactrl.vector = -1;

	// DMAC`l
	for (ch=0; ch<4; ch++) {
		ResetDMA(ch);
	}
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL DMAC::Save(Fileio *fio, int /*ver*/)
{
	size_t sz;
	int i;

	ASSERT(this);
	ASSERT(fio);
	LOG0(Log::Normal, "Z[u");

	// `l
	sz = sizeof(dma_t);
	for (i=0; i<4; i++) {
		if (!fio->Write(&sz, sizeof(sz))) {
			return FALSE;
		}
		if (!fio->Write(&dma[i], (int)sz)) {
			return FALSE;
		}
	}

	// O[o
	sz = sizeof(dmactrl_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (!fio->Write(&dmactrl, (int)sz)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL DMAC::Load(Fileio *fio, int /*ver*/)
{
	int i;
	size_t sz;

	ASSERT(this);
	ASSERT(fio);
	LOG0(Log::Normal, "[h");

	// `l
	for (i=0; i<4; i++) {
		// TCY[hAƍ
		if (!fio->Read(&sz, sizeof(sz))) {
			return FALSE;
		}
		if (sz != sizeof(dma_t)) {
			return FALSE;
		}

		// ̂[h
		if (!fio->Read(&dma[i], (int)sz)) {
			return FALSE;
		}
	}

	// O[o
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(dmactrl_t)) {
		return FALSE;
	}

	if (!fio->Read(&dmactrl, (int)sz)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::ApplyCfg(const Config* /*config*/)
{
	ASSERT(this);
//	ASSERT(config);
	LOG0(Log::Normal, "ݒKp");
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::ReadByte(DWORD addr)
{
	int ch;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// EFCg
	scheduler->Wait(7);

	// `lɊ蓖
	ch = (int)(addr >> 6);
	ch &= 3;
	addr &= 0x3f;

	// `lPʂōs
	return ReadDMA(ch, addr);
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::ReadWord(DWORD addr)
{
	int ch;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);

	// EFCg
	scheduler->Wait(7);

	// `lɊ蓖
	ch = (int)(addr >> 6);
	ch &= 3;
	addr &= 0x3f;

	// `lPʂōs
	return ((ReadDMA(ch, addr) << 8) | ReadDMA(ch, addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::WriteByte(DWORD addr, DWORD data)
{
	int ch;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT(data < 0x100);

	// EFCg
	scheduler->Wait(7);

	// `lɊ蓖
	ch = (int)(addr >> 6);
	ch &= 3;
	addr &= 0x3f;

	// `lPʂōs
	WriteDMA(ch, addr, data);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::WriteWord(DWORD addr, DWORD data)
{
	int ch;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);
	ASSERT(data < 0x10000);

	// EFCg
	scheduler->Wait(7);

	// `lɊ蓖
	ch = (int)(addr >> 6);
	ch &= 3;
	addr &= 0x3f;

	// `lPʂōs
	WriteDMA(ch, addr, (BYTE)(data >> 8));
	WriteDMA(ch, addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::ReadOnly(DWORD addr) const
{
	int ch;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// `lɊ蓖
	ch = (int)(addr >> 6);
	ch &= 3;
	addr &= 0x3f;

	// `lPʂōs
	return ReadDMA(ch, addr);
}

//---------------------------------------------------------------------------
//
//	DMAǂݍ
//	ʃ[ۏ
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::ReadDMA(int ch, DWORD addr) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(addr <= 0x3f);

	switch (addr) {
		// CSR
		case 0x00:
			return GetCSR(ch);

		// CER
		case 0x01:
			return dma[ch].ecode;

		// DCR
		case 0x04:
			return GetDCR(ch);

		// OCR
		case 0x05:
			return GetOCR(ch);

		// SCR
		case 0x06:
			return GetSCR(ch);

		// CCR
		case 0x07:
			return GetCCR(ch);

		// MTC
		case 0x0a:
			return (BYTE)(dma[ch].mtc >> 8);
		case 0x0b:
			return (BYTE)dma[ch].mtc;

		// MAR
		case 0x0c:
			return 0;
		case 0x0d:
			return (BYTE)(dma[ch].mar >> 16);
		case 0x0e:
			return (BYTE)(dma[ch].mar >> 8);
		case 0x0f:
			return (BYTE)dma[ch].mar;

		// DAR
		case 0x14:
			return 0;
		case 0x15:
			return (BYTE)(dma[ch].dar >> 16);
		case 0x16:
			return (BYTE)(dma[ch].dar >> 8);
		case 0x17:
			return (BYTE)dma[ch].dar;

		// BTC
		case 0x1a:
			return (BYTE)(dma[ch].btc >> 8);
		case 0x1b:
			return (BYTE)dma[ch].btc;

		// BAR
		case 0x1c:
			return 0;
		case 0x1d:
			return (BYTE)(dma[ch].bar >> 16);
		case 0x1e:
			return (BYTE)(dma[ch].bar >> 8);
		case 0x1f:
			return (BYTE)dma[ch].bar;

		// NIV
		case 0x25:
			return dma[ch].niv;

		// EIV
		case 0x27:
			return dma[ch].eiv;

		// MFC
		case 0x29:
			return dma[ch].mfc;

		// CPR
		case 0x2d:
			return dma[ch].cp;

		// DFC
		case 0x31:
			return dma[ch].dfc;

		// BFC
		case 0x39:
			return dma[ch].bfc;

		// GCR
		case 0x3f:
			if (ch == 3) {
				// `l3̂݃o[Xg]Ԃ
				ASSERT(dma[ch].bt <= 3);
				ASSERT(dma[ch].br <= 3);

				data = dma[ch].bt;
				data <<= 2;
				data |= dma[ch].br;
				return data;
			}
			return 0xff;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	DMA
//	ʃ[ۏ؂v
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::WriteDMA(int ch, DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(addr <= 0x3f);
	ASSERT(data < 0x100);

	switch (addr) {
		// CSR
		case 0x00:
			SetCSR(ch, data);
			return;

		// CER(Read Only)
		case 0x01:
			return;

		// DCR
		case 0x04:
			SetDCR(ch, data);
			return;

		// OCR
		case 0x05:
			SetOCR(ch, data);
			return;

		// SCR
		case 0x06:
			SetSCR(ch, data);
			return;

		// CCR
		case 0x07:
			SetCCR(ch, data);
			return;

		// MTC
		case 0x0a:
			dma[ch].mtc &= 0x00ff;
			dma[ch].mtc |= (data << 8);
			return;
		case 0x0b:
			dma[ch].mtc &= 0xff00;
			dma[ch].mtc |= data;
			return;

		// MAR
		case 0x0c:
			return;
		case 0x0d:
			dma[ch].mar &= 0x0000ffff;
			dma[ch].mar |= (data << 16);
			return;
		case 0x0e:
			dma[ch].mar &= 0x00ff00ff;
			dma[ch].mar |= (data << 8);
			return;
		case 0x0f:
			dma[ch].mar &= 0x00ffff00;
			dma[ch].mar |= data;
			return;

		// DAR
		case 0x14:
			return;
		case 0x15:
			dma[ch].dar &= 0x0000ffff;
			dma[ch].dar |= (data << 16);
			return;
		case 0x16:
			dma[ch].dar &= 0x00ff00ff;
			dma[ch].dar |= (data << 8);
			return;
		case 0x17:
			dma[ch].dar &= 0x00ffff00;
			dma[ch].dar |= data;
			return;

		// BTC
		case 0x1a:
			dma[ch].btc &= 0x00ff;
			dma[ch].btc |= (data << 8);
			return;
		case 0x1b:
			dma[ch].btc &= 0xff00;
			dma[ch].btc |= data;
			return;

		// BAR
		case 0x1c:
			return;
		case 0x1d:
			dma[ch].bar &= 0x0000ffff;
			dma[ch].bar |=(data << 16);
			return;
		case 0x1e:
			dma[ch].bar &= 0x00ff00ff;
			dma[ch].bar |= (data << 8);
			return;
		case 0x1f:
			dma[ch].bar &= 0x00ffff00;
			dma[ch].bar |= data;
			return;

		// NIV
		case 0x25:
			dma[ch].niv = data;
			return;

		// EIV
		case 0x27:
			dma[ch].eiv = data;
			return;

		// MFC
		case 0x29:
			dma[ch].mfc = data;
			return;

		// CPR
		case 0x2d:
			dma[ch].cp = data & 0x03;
			return;

		// DFC
		case 0x31:
			dma[ch].dfc = data;
			return;

		// BFC
		case 0x39:
			dma[ch].bfc = data;
			return;

		// GCR
		case 0x3f:
			if (ch == 3) {
				// `l3̂
				SetGCR(data);
			}
			return;
	}
}

//---------------------------------------------------------------------------
//
//	DCRZbg
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::SetDCR(int ch, DWORD data)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(data < 0x100);

	// ACTオĂ΃^C~OG[
	if (dma[ch].act) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d ^C~OG[(SetDCR)", ch);
#endif	// DMAC_LOG
		ErrorDMA(ch, 0x02);
		return;
	}

	// XRM
	dma[ch].xrm = data >> 6;

	// DTYP
	dma[ch].dtyp = (data >> 4) & 0x03;

	// DPS
	if (data & 0x08) {
		dma[ch].dps = TRUE;
	}
	else {
		dma[ch].dps = FALSE;
	}

	// PCL
	dma[ch].pcl = (data & 0x03);

	// 荞݃`FbN
	Interrupt();
}

//---------------------------------------------------------------------------
//
//	DCR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::GetDCR(int ch) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(dma[ch].xrm <= 3);
	ASSERT(dma[ch].dtyp <= 3);
	ASSERT(dma[ch].pcl <= 3);

	// f[^쐬
	data = dma[ch].xrm;
	data <<= 2;
	data |= dma[ch].dtyp;
	data <<= 1;
	if (dma[ch].dps) {
		data |= 0x01;
	}
	data <<= 3;
	data |= dma[ch].pcl;

	return data;
}

//---------------------------------------------------------------------------
//
//	OCRZbg
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::SetOCR(int ch, DWORD data)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(data < 0x100);

	// ACTオĂ΃^C~OG[
	if (dma[ch].act) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d ^C~OG[(SetOCR)", ch);
#endif	// DMAC_LOG
		ErrorDMA(ch, 0x02);
		return;
	}

	// DIR
	if (data & 0x80) {
		dma[ch].dir = TRUE;
	}
	else {
		dma[ch].dir = FALSE;
	}

	// BTD
	if (data & 0x40) {
		dma[ch].btd = TRUE;
	}
	else {
		dma[ch].btd = FALSE;
	}

	// SIZE
	dma[ch].size = (data >> 4) & 0x03;

	// CHAIN
	dma[ch].chain = (data >> 2) & 0x03;

	// REQG
	dma[ch].reqg = (data & 0x03);
}

//---------------------------------------------------------------------------
//
//	OCR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::GetOCR(int ch) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(dma[ch].size <= 3);
	ASSERT(dma[ch].chain <= 3);
	ASSERT(dma[ch].reqg <= 3);

	// f[^쐬
	data = 0;
	if (dma[ch].dir) {
		data |= 0x02;
	}
	if (dma[ch].btd) {
		data |= 0x01;
	}
	data <<= 2;
	data |= dma[ch].size;
	data <<= 2;
	data |= dma[ch].chain;
	data <<= 2;
	data |= dma[ch].reqg;

	return data;
}

//---------------------------------------------------------------------------
//
//	SCRZbg
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::SetSCR(int ch, DWORD data)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(data < 0x100);

	// ACTオĂ΃^C~OG[
	if (dma[ch].act) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d ^C~OG[(SetSCR)", ch);
#endif	// DMAC_LOG
		ErrorDMA(ch, 0x02);
		return;
	}

	dma[ch].mac = (data >> 2) & 0x03;
	dma[ch].dac = (data & 0x03);
}

//---------------------------------------------------------------------------
//
//	SCR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::GetSCR(int ch) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(dma[ch].mac <= 3);
	ASSERT(dma[ch].dac <= 3);

	// f[^쐬
	data = dma[ch].mac;
	data <<= 2;
	data |= dma[ch].dac;

	return data;
}

//---------------------------------------------------------------------------
//
//	CCRZbg
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::SetCCR(int ch, DWORD data)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(data < 0x100);

	// INT
	if (data & 0x08) {
		dma[ch].intr = TRUE;
	}
	else {
		dma[ch].intr = FALSE;
	}

	// HLT
	if (data & 0x20) {
		dma[ch].hlt = TRUE;
	}
	else {
		dma[ch].hlt = FALSE;
	}

	// STR
	if (data & 0x80) {
		dma[ch].str = TRUE;
		StartDMA(ch);
	}

	// CNT
	if (data & 0x40) {
		dma[ch].cnt = TRUE;
		ContDMA(ch);
	}

	// SAB
	if (data & 0x10) {
		dma[ch].sab = TRUE;
		AbortDMA(ch);
	}
}

//---------------------------------------------------------------------------
//
//	CCR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::GetCCR(int ch) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));

	// INT,HLT,STR,CNTԂ
	data = 0;
	if (dma[ch].intr) {
		data |= 0x08;
	}
	if (dma[ch].hlt) {
		data |= 0x20;
	}
	if (dma[ch].str) {
		data |= 0x80;
	}
	if (dma[ch].cnt) {
		data |= 0x40;
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	CSRZbg
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::SetCSR(int ch, DWORD data)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(data < 0x100);

	// ACT,PCSȊO1ނƂɂNAł
	if (data & 0x80) {
		dma[ch].coc = FALSE;
	}
	if (data & 0x40) {
		dma[ch].boc = FALSE;
	}
	if (data & 0x20) {
		dma[ch].ndt = FALSE;
	}
	if (data & 0x10) {
		dma[ch].err = FALSE;
	}
	if (data & 0x04) {
		dma[ch].dit = FALSE;
	}
	if (data & 0x02) {
		dma[ch].pct = FALSE;
	}

	// 荞ݏ
	Interrupt();
}

//---------------------------------------------------------------------------
//
//	CSR擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::GetCSR(int ch) const
{
	DWORD data;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));

	// f[^쐬
	data = 0;
	if (dma[ch].coc) {
		data |= 0x80;
	}
	if (dma[ch].boc) {
		data |= 0x40;
	}
	if (dma[ch].ndt) {
		data |= 0x20;
	}
	if (dma[ch].err) {
		data |= 0x10;
	}
	if (dma[ch].act) {
		data |= 0x08;
	}
	if (dma[ch].dit) {
		data |= 0x04;
	}
	if (dma[ch].pct) {
		data |= 0x02;
	}
	if (dma[ch].pcs) {
		data |= 0x01;
	}

	return data;
}

//---------------------------------------------------------------------------
//
//	GCRݒ
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::SetGCR(DWORD data)
{
	int ch;
	DWORD bt;
	DWORD br;

	ASSERT(this);
	ASSERT(data < 0x100);

	// f[^
	bt = (data >> 2) & 0x03;
	br = data & 0x03;

	// S`lɐݒ
	for (ch=0; ch<4; ch++) {
		dma[ch].bt = bt;
		dma[ch].br = br;
	}
}

//---------------------------------------------------------------------------
//
//	DMAZbg
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::ResetDMA(int ch)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));

	// GCR
	dma[ch].bt = 0;
	dma[ch].br = 0;

	// DCR
	dma[ch].xrm = 0;
	dma[ch].dtyp = 0;
	dma[ch].dps = FALSE;
	dma[ch].pcl = 0;

	// OCR
	dma[ch].dir = FALSE;
	dma[ch].btd = FALSE;
	dma[ch].size = 0;
	dma[ch].chain = 0;
	dma[ch].reqg = 0;

	// SCR
	dma[ch].mac = 0;
	dma[ch].dac = 0;

	// CCR
	dma[ch].str = FALSE;
	dma[ch].cnt = FALSE;
	dma[ch].hlt = FALSE;
	dma[ch].sab = FALSE;
	dma[ch].intr = FALSE;

	// CSR
	dma[ch].coc = FALSE;
	dma[ch].boc = FALSE;
	dma[ch].ndt = FALSE;
	dma[ch].err = FALSE;
	dma[ch].act = FALSE;
	dma[ch].dit = FALSE;
	dma[ch].pct = FALSE;
	if (ch == 0) {
		// FDC'L'
		dma[ch].pcs = FALSE;
	}
	else {
		// ȊO'H'
		dma[ch].pcs = TRUE;
	}

	// CPR
	dma[ch].cp = 0;

	// CER
	dma[ch].ecode = 0;

	// 荞݃xN^
	dma[ch].niv = 0x0f;
	dma[ch].eiv = 0x0f;

	// AhXyуJE^̓ZbgȂ(f[^V[gɂ)
	dma[ch].mar &= 0x00ffffff;
	dma[ch].dar &= 0x00ffffff;
	dma[ch].bar &= 0x00ffffff;
	dma[ch].mtc &= 0x0000ffff;
	dma[ch].btc &= 0x0000ffff;

	// ]^CvAJE^
	dma[ch].type = 0;
	dma[ch].startcnt = 0;
	dma[ch].errorcnt = 0;
}

//---------------------------------------------------------------------------
//
//	DMAX^[g
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::StartDMA(int ch)
{
	int c;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));

#if defined(DMAC_LOG)
	LOG1(Log::Normal, "`l%d X^[g", ch);
#endif	// DMAC_LOG

	// ACT,COC,BOC,NDT,ERRオĂ΃^C~OG[
	if (dma[ch].act || dma[ch].coc || dma[ch].boc || dma[ch].ndt || dma[ch].err) {
#if defined(DMAC_LOG)
		if (dma[ch].act) {
			LOG1(Log::Normal, "`l%d ^C~OG[ (ACT)", ch);
		}
		if (dma[ch].coc) {
			LOG1(Log::Normal, "`l%d ^C~OG[ (COC)", ch);
		}
		if (dma[ch].boc) {
			LOG1(Log::Normal, "`l%d ^C~OG[ (BOC)", ch);
		}
		if (dma[ch].ndt) {
			LOG1(Log::Normal, "`l%d ^C~OG[ (NDT)", ch);
		}
		if (dma[ch].err) {
			LOG1(Log::Normal, "`l%d ^C~OG[ (ERR)", ch);
		}
#endif	// DMAC_LOG
		ErrorDMA(ch, 0x02);
		return;
	}

	// `FCȂ̏ꍇ́AMTC=0Ȃ烁JEgG[
	if (dma[ch].chain == 0) {
		if (dma[ch].mtc == 0) {
#if defined(DMAC_LOG)
			LOG1(Log::Normal, "`l%d JEgG[", ch);
#endif	// DMAC_LOG
			ErrorDMA(ch, 0x0d);
			return;
		}
	}

	// AC`FC̏ꍇ́ABTC=0Ȃx[XJEgG[
	if (dma[ch].chain == 0x02) {
		if (dma[ch].btc == 0) {
#if defined(DMAC_LOG)
			LOG1(Log::Normal, "`l%d x[XJEgG[", ch);
#endif	// DMAC_LOG
			ErrorDMA(ch, 0x0f);
			return;
		}
	}

	// RtBM[VG[`FbN
	if ((dma[ch].xrm == 0x01) || (dma[ch].mac == 0x03) || (dma[ch].dac == 0x03)
			|| (dma[ch].chain == 0x01)) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d RtBOG[", ch);
#endif	// DMAC_LOG
		ErrorDMA(ch, 0x01);
		return;
	}

	// ]^Cv쐬
	dma[ch].type = 0;
	if (dma[ch].dps) {
		dma[ch].type += 4;
	}
	dma[ch].type += dma[ch].size;

	// [N
	dma[ch].str = FALSE;
	dma[ch].act = TRUE;
	dma[ch].cnt = FALSE;
	dma[ch].sab = FALSE;

	// JEgAbv
	dma[ch].startcnt++;

	// AC`FC܂̓NAC`FĆAŏ̃ubN[h
	if (dma[ch].chain != 0) {
		LoadDMA(ch);
		// [hɃAhXG[܂̓oXG[NAG[tOオ
		if (dma[ch].err) {
			return;
		}
	}

	// CPUTCNNAāA[h
	dmactrl.cpu_cycle = 0;
	switch (dma[ch].reqg) {
		// I[gNGXg
		case 0:
		// I[gNGXgő
		case 1:
			// ݂̎c肾DMA𓮂āACPU~߂
			dmactrl.current_ch = ch;
			dmactrl.cpu_cycle = 0;
			dmactrl.exec = TRUE;
			scheduler->dma_active = TRUE;
			c = AutoDMA(cpu->GetIOCycle());
			if (c == 0) {
				cpu->Release();
			}
			break;

		// Ov]
		case 2:
			break;

		// I[gNGXg{Ov]
		case 3:
			// ݂̎c肾DMA𓮂āACPU~߂
			dmactrl.current_ch = ch;
			dmactrl.cpu_cycle = 0;
			dmactrl.exec = TRUE;
			scheduler->dma_active= TRUE;
			dma[ch].reqg = 1;
			c = AutoDMA(cpu->GetIOCycle());
			if (c == 0) {
				cpu->Release();
			}
			dma[ch].reqg = 3;
			break;

		default:
			ASSERT(FALSE);
	}
}

//---------------------------------------------------------------------------
//
//	DMAReBj[
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::ContDMA(int ch)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));

#if defined(DMAC_LOG)
	LOG1(Log::Normal, "`l%d ReBj[", ch);
#endif	// DMAC_LOG

	// ACTオĂȂƓ^C~OG[
	if (!dma[ch].act) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d ^C~OG[(Cont)", ch);
#endif	// DMAC_LOG
		ErrorDMA(ch, 0x02);
		return;
}
	// `FC[h̏ꍇ̓RtBOG[
	if (dma[ch].chain != 0) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d RtBOG[", ch);
#endif	// DMAC_LOG
		ErrorDMA(ch, 0x01);
	}
}

//---------------------------------------------------------------------------
//
//	DMA\tgEFAA{[g
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::AbortDMA(int ch)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));

	// ANeBuȂG[sȂ(Marianne.pan)
	if (!dma[ch].act) {
		// COC𗎂Ƃ(of[N)
		dma[ch].coc = FALSE;
		return;
	}

#if defined(DMAC_LOG)
	LOG1(Log::Normal, "`l%d \tgEFAA{[g", ch);
#endif	// DMAC_LOG

	// ]AANeBu
	dma[ch].coc = TRUE;
	dma[ch].act = FALSE;

	// \tgEFAA{[gŃG[
	ErrorDMA(ch, 0x11);
}

//---------------------------------------------------------------------------
//
//	DMAubÑ[h
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::LoadDMA(int ch)
{
	DWORD base;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(dmactrl.load == 0);

	// [h(ReadWordł̃AhXG[AoXG[ɒ)
	dmactrl.load = (ch + 1);

	if (dma[ch].bar & 1) {
		// BARAhXG[
		AddrErr(dma[ch].bar, TRUE);

		dmactrl.load = 0;
		return;
	}

	// MARǂݍ
	dma[ch].bar &= 0xfffffe;
	dma[ch].mar = (memory->ReadWord(dma[ch].bar) & 0x00ff);
	dma[ch].bar += 2;
	dma[ch].mar <<= 16;
	dma[ch].bar &= 0xfffffe;
	dma[ch].mar |= (memory->ReadWord(dma[ch].bar) & 0xffff);
	dma[ch].bar += 2;

	// MTCǂݍ
	dma[ch].bar &= 0xfffffe;
	dma[ch].mtc = (memory->ReadWord(dma[ch].bar) & 0xffff);
	dma[ch].bar += 2;

	if (dma[ch].err) {
		// MAR,MTCǂݍ݃G[
		dmactrl.load = 0;
		return;
	}

	// AC`FCł͂܂
	if (dma[ch].chain == 0x02) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d AC`FCubN", ch);
#endif	// DMAC_LOG
		dma[ch].btc--;
		dmactrl.load = 0;
		return;
	}

	// NAC`FC(ł͎̃NAhXBARփ[h
#if defined(DMAC_LOG)
	LOG1(Log::Normal, "`l%d NAC`FCubN", ch);
#endif	// DMAC_LOG
	dma[ch].bar &= 0xfffffe;
	base = (memory->ReadWord(dma[ch].bar) & 0x00ff);
	dma[ch].bar += 2;
	base <<= 16;
	dma[ch].bar &= 0xfffffe;
	base |= (memory->ReadWord(dma[ch].bar) & 0xffff);
	dma[ch].bar = base;

	// [hI
	dmactrl.load = 0;
}

//---------------------------------------------------------------------------
//
//	DMAG[
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::ErrorDMA(int ch, DWORD code)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT((code >= 0x01) && (code <= 17));

#if defined(DMAC_LOG)
	LOG2(Log::Normal, "`l%d G[$%02X", ch, code);
#endif	// DMAC_LOG

	// ACT~낷(t@NX ADPCM)
	dma[ch].act = FALSE;

	// G[R[h
	dma[ch].ecode = code;

	// G[tO𗧂Ă
	dma[ch].err = TRUE;

	// JEgAbv
	dma[ch].errorcnt++;

	// 荞ݏ
	Interrupt();
}

//---------------------------------------------------------------------------
//
//	DMA荞
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::Interrupt()
{
	DWORD cp;
	int ch;

	ASSERT(this);

	// DMAƓDxŏ(f[^V[g)
	for (cp=0; cp<=3; cp++) {
		for (ch=0; ch<=3; ch++) {
			// CP`FbN
			if (cp != dma[ch].cp) {
				continue;
			}

			// C^vgCl[u`FbN
			if (!dma[ch].intr) {
				continue;
			}

			// ERRɂEIVŏo
			if (dma[ch].err) {
				if (dmactrl.vector != (int)dma[ch].eiv) {
					// ʂ̊荞݂vĂ΁AULZ
					if (dmactrl.vector >= 0) {
						cpu->IntCancel(3);
					}
#if defined(DMAC_LOG)
					LOG1(Log::Normal, "`l%d G[荞", ch);
#endif	// DMAC_LOG
					cpu->Interrupt(3, (BYTE)dma[ch].eiv);
					dmactrl.vector = (int)dma[ch].eiv;
				}
				return;
			}

			// COC,BOC,NDT,PCT(荞݃C̏ꍇ)NIVŏo
			if (dma[ch].coc || dma[ch].boc || dma[ch].ndt) {
				if (dmactrl.vector != (int)dma[ch].niv) {
					// ʂ̊荞݂vĂ΁AULZ
					if (dmactrl.vector >= 0) {
						cpu->IntCancel(3);
					}
#if defined(DMAC_LOG)
					LOG1(Log::Normal, "`l%d ʏ튄荞", ch);
#endif	// DMAC_LOG
					cpu->Interrupt(3, (BYTE)dma[ch].niv);
					dmactrl.vector = (int)dma[ch].niv;
				}
				return;
			}

			if ((dma[ch].pcl == 0x01) && dma[ch].pct) {
				if (dmactrl.vector != (int)dma[ch].niv) {
					// ʂ̊荞݂vĂ΁AULZ
					if (dmactrl.vector >= 0) {
						cpu->IntCancel(3);
					}
#if defined(DMAC_LOG)
					LOG1(Log::Normal, "`l%d PCL荞", ch);
#endif	// DMAC_LOG
					cpu->Interrupt(3, (BYTE)dma[ch].niv);
					dmactrl.vector = (int)dma[ch].niv;
				}
				return;
			}
		}
	}

	// v̊荞݂͂Ȃ
	if (dmactrl.vector >= 0) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "荞݃LZ xN^$%02X", dmactrl.vector);
#endif	// DMAC_LOG

		cpu->IntCancel(3);
		dmactrl.vector = -1;
	}
}

//---------------------------------------------------------------------------
//
//	荞ACK
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::IntAck()
{
	ASSERT(this);

	// ZbgɁACPU犄荞݂Ԉēꍇ
	if (dmactrl.vector < 0) {
		LOG0(Log::Warning, "vĂȂ荞");
		return;
	}

#if defined(DMAC_LOG)
	LOG1(Log::Normal, "荞݉ xN^$%02X", dmactrl.vector);
#endif	// DMAC_LOG

	// vxN^Ȃ
	dmactrl.vector = -1;
}

//---------------------------------------------------------------------------
//
//	DMA擾
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::GetDMA(int ch, dma_t *buffer) const
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(buffer);

	// `l[NRs[
	*buffer = dma[ch];
}

//---------------------------------------------------------------------------
//
//	DMA擾
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::GetDMACtrl(dmactrl_t *buffer) const
{
	ASSERT(this);
	ASSERT(buffer);

	// 䃏[NRs[
	*buffer = dmactrl;
}

//---------------------------------------------------------------------------
//
//	DMA]
//
//---------------------------------------------------------------------------
BOOL FASTCALL DMAC::IsDMA() const
{
	ASSERT(this);

	// ]tO(`lp)ƁA[htO
	if ((dmactrl.transfer == 0) && (dmactrl.load == 0)) {
		return FALSE;
	}

	// ǂ炩Ă
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	DMA]\
//
//---------------------------------------------------------------------------
BOOL FASTCALL DMAC::IsAct(int ch) const
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));

	// ACTłȂAERRAHLTȂ]łȂ
	if (!dma[ch].act || dma[ch].err || dma[ch].hlt) {
		return FALSE;
	}

	// ]ł
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	DMAoXG[
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::BusErr(DWORD addr, BOOL read)
{
	ASSERT(this);
	ASSERT(addr <= 0xffffff);

	if (read) {
		LOG1(Log::Warning, "DMAoXG[(ǂݍ) $%06X", addr);
	}
	else {
		LOG1(Log::Warning, "DMAoXG[() $%06X", addr);
	}

	// [h̃G[
	if (dmactrl.load != 0) {
		// EfoCXEx[X̋ʂ͍lĂȂ
		ASSERT((dmactrl.load >= 1) && (dmactrl.load <= 4));
		ErrorDMA(dmactrl.load - 1, 0x08);
		return;
	}

	// EfoCXEx[X̋ʂ͍lĂȂ
	ASSERT((dmactrl.transfer >= 1) && (dmactrl.transfer <= 4));
	ErrorDMA(dmactrl.transfer - 1, 0x08);
}

//---------------------------------------------------------------------------
//
//	DMAAhXG[
//
//---------------------------------------------------------------------------
void FASTCALL DMAC::AddrErr(DWORD addr, BOOL read)
{
	ASSERT(this);
	ASSERT(addr <= 0xffffff);

	if (read) {
		LOG1(Log::Warning, "DMAAhXG[(ǂݍ) $%06X", addr);
	}
	else {
		LOG1(Log::Warning, "DMAAhXG[() $%06X", addr);
	}

	// [h̃G[
	if (dmactrl.load != 0) {
		// EfoCXEx[X̋ʂ͍lĂȂ
		ASSERT((dmactrl.load >= 1) && (dmactrl.load <= 4));
		ErrorDMA(dmactrl.load - 1, 0x0c);
		return;
	}

	// EfoCXEx[X̋ʂ͍lĂȂ
	ASSERT((dmactrl.transfer >= 1) && (dmactrl.transfer <= 4));
	ErrorDMA(dmactrl.transfer - 1, 0x0c);
}

//---------------------------------------------------------------------------
//
//	xN^擾
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::GetVector(int type) const
{
	ASSERT(this);
	ASSERT((type >= 0) && (type < 8));

	// m[}EG[̃xN^݂ɏo
	if (type & 1) {
		// G[
		return dma[type >> 1].eiv;
	}
	else {
		// m[}
		return dma[type >> 1].niv;
	}
}

//---------------------------------------------------------------------------
//
//	DMAONGXg
//
//---------------------------------------------------------------------------
BOOL FASTCALL DMAC::ReqDMA(int ch)
{
	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));

	// ACTłȂAERRAHLTȂ牽Ȃ
	if (!dma[ch].act || dma[ch].err || dma[ch].hlt) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d ONGXgs", ch);
#endif	// DMAC_LOG
		return FALSE;
	}

	// DMA]
	TransDMA(ch);
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	I[gNGXg
//
//---------------------------------------------------------------------------
DWORD FASTCALL DMAC::AutoDMA(DWORD cycle)
{
	int i;
	int ch;
	int mul;
	int remain;
	int used;
	DWORD backup;
	BOOL flag;

	ASSERT(this);

	// p[^L
	remain = (int)cycle;

	// stOオĂȂ΃I[gNGXg͖
	if (!dmactrl.exec) {
		return cycle;
	}

	// sptOZbg
	flag = FALSE;

	// ő呬xI[gNGXg̃`lɏ
	for (i=0; i<4; i++) {
		// ׂ`l
		ch = (dmactrl.current_ch + i) & 3;

		// ACT, ERR, HLT̃`FbN
		if (!dma[ch].act || dma[ch].err || dma[ch].hlt) {
			continue;
		}

		// ő呬xI[gNGXg
		if (dma[ch].reqg != 1) {
			continue;
		}

		// ZāAŒł10TCN͕KvB
		dmactrl.cpu_cycle += cycle;
		if (dmactrl.cpu_cycle < 10) {
			// CPU͎słȂBDMAp
			return 0;
		}

		// 2ȏZȂAtOUP
		cycle = 0;
		flag = TRUE;

		// XPW[cycle(I[o[TCNvZ)ێAZbg
		backup = scheduler->GetCPUCycle();
		scheduler->SetCPUCycle(0);

		// cpu_cycle}CiXɂȂ܂ŎsBԂ̓XPW[蓾
		while (scheduler->GetCPUCycle() < dmactrl.cpu_cycle) {
			// ACT, ERR, HLT̃`FbN
			if (!dma[ch].act || dma[ch].err || dma[ch].hlt) {
				break;
			}

			// scheulder->GetCPUCycle()gADMACTCN擾
			TransDMA(ch);
		}

		// TCNA
		dmactrl.cpu_cycle -= scheduler->GetCPUCycle();
		remain -= scheduler->GetCPUCycle();
		scheduler->SetCPUCycle(backup);

		// `l(Ehr)
		dmactrl.current_ch = (dmactrl.current_ch + 1) & 3;

		// ׂĎԂg؂
		if (dmactrl.cpu_cycle <= 0) {
			// CPU͎słȂ
			// őS`lꍇAAudoDMAŃtOƂ
			return 0;
		}
	}

	// ő呬xI[gNGXgȂAĎԂ]
	// 葬xI[gNGXg̃`l
	for (i=0; i<4; i++) {
		// ׂ`l
		ch = (dmactrl.current_ch + i) & 3;

		// ACT, ERR, HLT̃`FbN
		if (!dma[ch].act || dma[ch].err || dma[ch].hlt) {
			continue;
		}

		// ő呬xI[gNGXg͂肦Ȃ(̕ŕK)
		ASSERT(dma[ch].reqg != 1);

		// 葬xI[gNGXg
		if (dma[ch].reqg != 0) {
			continue;
		}

		// ZāAŒł10TCN͕KvB
		dmactrl.cpu_cycle += cycle;
		if (dmactrl.cpu_cycle < 10) {
			// CPU͎słȂBDMAp
			return 0;
		}

		// 2ȏZȂAtOUP
		cycle = 0;
		flag = TRUE;

		// XPW[cycle(I[o[TCNvZ)ێAZbg
		backup = scheduler->GetCPUCycle();
		scheduler->SetCPUCycle(0);

		// oXL{vZ(BT=0Ȃ2{Ȃ)
		mul = (dma[ch].bt + 1);

		// cpu_cycleoXLll𒴂Ă邩
		while ((scheduler->GetCPUCycle() << mul) < dmactrl.cpu_cycle) {
			// ACT, ERR, HLT̃`FbN
			if (!dma[ch].act || dma[ch].err || dma[ch].hlt) {
				break;
			}

			// ]
			TransDMA(ch);
		}

		// gpTCNL(Ŏg)
		used = scheduler->GetCPUCycle();
		scheduler->SetCPUCycle(backup);

		// `l(Ehr)
		dmactrl.current_ch = (dmactrl.current_ch + 1) & 3;

		// ŏI
		if (dmactrl.cpu_cycle < (used << mul)) {
			// \肳ĂoXLg؂BcCPUɕԋp
			dmactrl.cpu_cycle -= used;
			if (used < remain) {
				// \]肪
				return (remain - used);
			}
			else {
				// ȂAgBCPU0
				return 0;
			}
		}

		// ܂oX̎gp̂ŁA`l܂
		remain -= used;
	}

	if (!flag) {
		// DMA͎gȂBdmactrl.exec~낷
		dmactrl.exec = FALSE;
		scheduler->dma_active = FALSE;
	}

	return cycle;
}

//---------------------------------------------------------------------------
//
//	DMA1]
//
//---------------------------------------------------------------------------
BOOL FASTCALL DMAC::TransDMA(int ch)
{
	DWORD data;

	ASSERT(this);
	ASSERT((ch >= 0) && (ch <= 3));
	ASSERT(dmactrl.transfer == 0);

	// ]tOON
	dmactrl.transfer = ch + 1;

	// ^CvAfBNVɉē]
	switch (dma[ch].type) {
		// 8bit, PackoCg, 8bit|[g
		case 0:
		// 8bit, UnpackoCg, 8bit|[g
		case 3:
		// 8bit, UnpackoCg, 16bit|[g
		case 7:
			// SCSIfBXN x`}[N(dskbench.x)
			if (dma[ch].dir) {
				memory->WriteByte(dma[ch].mar, (BYTE)(memory->ReadByte(dma[ch].dar)));
				scheduler->Wait(11);
			}
			else {
				memory->WriteByte(dma[ch].dar, (BYTE)(memory->ReadByte(dma[ch].mar)));
				scheduler->Wait(11);
			}
			break;

		// 8bit, PackoCg, 16bit|[g(Unpack葬:pfBEX!)
		// Wait12:pfBEX!AWait?:Moon Fighter
		case 4:
			if (dma[ch].dir) {
				memory->WriteByte(dma[ch].mar, (BYTE)(memory->ReadByte(dma[ch].dar)));
				scheduler->Wait(10);
			}
			else {
				memory->WriteByte(dma[ch].dar, (BYTE)(memory->ReadByte(dma[ch].mar)));
				scheduler->Wait(10);
			}
			break;

		// 8bit, [h
		case 1:
			if (dma[ch].dir) {
				data = (BYTE)(memory->ReadByte(dma[ch].dar));
				data <<= 8;
				data |= (BYTE)(memory->ReadByte(dma[ch].dar + 2));
				memory->WriteWord(dma[ch].mar, data);
				scheduler->Wait(19);
			}
			else {
				data = memory->ReadWord(dma[ch].mar);
				memory->WriteByte(dma[ch].dar, (BYTE)(data >> 8));
				memory->WriteByte(dma[ch].dar + 2, (BYTE)data);
				scheduler->Wait(19);
			}
			break;

		// 8bit, O[h
		case 2:
			if (dma[ch].dir) {
				data = (BYTE)(memory->ReadByte(dma[ch].dar));
				data <<= 8;
				data |= (BYTE)(memory->ReadByte(dma[ch].dar + 2));
				data <<= 8;
				data |= (BYTE)(memory->ReadByte(dma[ch].dar + 4));
				data <<= 8;
				data |= (BYTE)(memory->ReadByte(dma[ch].dar + 6));
				memory->WriteWord(dma[ch].mar, (WORD)(data >> 16));
				memory->WriteWord(dma[ch].mar + 2, (WORD)data);
				scheduler->Wait(38);
			}
			else {
				data = memory->ReadWord(dma[ch].mar);
				data <<= 16;
				data |= (WORD)(memory->ReadWord(dma[ch].mar + 2));
				memory->WriteByte(dma[ch].dar, (BYTE)(data >> 24));
				memory->WriteByte(dma[ch].dar + 2, (BYTE)(data >> 16));
				memory->WriteByte(dma[ch].dar + 4, (BYTE)(data >> 8));
				memory->WriteByte(dma[ch].dar + 6, (BYTE)data);
				scheduler->Wait(38);
			}
			break;

		// 16bit, [h
		case 5:
			// ܂xFM荞݂Ђ(OfBEXII)
			if (dma[ch].dir) {
				data = memory->ReadWord(dma[ch].dar);
				memory->WriteWord(dma[ch].mar, (WORD)data);
				scheduler->Wait(10);
			}
			else {
				data = memory->ReadWord(dma[ch].mar);
				memory->WriteWord(dma[ch].dar, (WORD)data);
				scheduler->Wait(10);
			}
			break;

		// 16bit, O[h
		case 6:
			if (dma[ch].dir) {
				data = memory->ReadWord(dma[ch].dar);
				data <<= 16;
				data |= (WORD)(memory->ReadWord(dma[ch].dar + 2));
				memory->WriteWord(dma[ch].mar, (WORD)(data >> 16));
				memory->WriteWord(dma[ch].mar + 2, (WORD)data);
				scheduler->Wait(20);
			}
			else {
				data = memory->ReadWord(dma[ch].mar);
				data <<= 16;
				data |= (WORD)memory->ReadWord(dma[ch].mar + 2);
				memory->WriteWord(dma[ch].dar, (WORD)(data >> 16));
				memory->WriteWord(dma[ch].dar + 2, (WORD)data);
				scheduler->Wait(20);
			}
			break;

		// ȊO
		default:
			ASSERT(FALSE);
	}

	// ]tOOFF
	dmactrl.transfer = 0;

	// ]G[̃`FbN(oXG[yуAhXG[)
	if (dma[ch].err) {
		// AhXXVOɔ(f[^V[gɂ)
		return FALSE;
	}

	// AhXXV(12bitɐ:Racing Champ)
	dma[ch].mar += MemDiffTable[ dma[ch].type ][ dma[ch].mac ];
	dma[ch].mar &= 0xffffff;
	dma[ch].dar += DevDiffTable[ dma[ch].type ][ dma[ch].dac ];
	dma[ch].dar &= 0xffffff;

	// JEgfNg
	dma[ch].mtc--;
	if (dma[ch].mtc > 0) {
		// I钼ODONEAT[gFDĈTCݒ(DCII)
		if ((ch == 0) && (dma[ch].mtc == 1)) {
			fdc->SetTC();
		}
		return TRUE;
	}

	// ReBj[̏
	if (dma[ch].cnt) {
#if defined(DMAC_LOG)
		LOG1(Log::Normal, "`l%d ReBj[ubN", ch);
#endif	// DMAC_LOG

		// BAR,BFC,BTCMAR,MFC,MTCɓ]
		dma[ch].mar = dma[ch].bar;
		dma[ch].mfc = dma[ch].bfc;
		dma[ch].mtc = dma[ch].btc;

		// BOCグ
		dma[ch].boc = TRUE;
		Interrupt();
		return TRUE;
	}

	// AC`FC̏
	if (dma[ch].chain == 0x02) {
		if (dma[ch].btc > 0) {
			// ̃ubN
			LoadDMA(ch);
			return TRUE;
		}
	}

	// NAC`FC̏
	if (dma[ch].chain == 0x03) {
		if (dma[ch].bar != 0) {
			// ̃ubN
			LoadDMA(ch);
			return TRUE;
		}
	}

	// DMA
#if defined(DMAC_LOG)
	LOG1(Log::Normal, "`l%d DMA", ch);
#endif	// DMAC_LOG

	// tOݒA荞
	dma[ch].act = FALSE;
	dma[ch].coc = TRUE;
	Interrupt();

	return FALSE;
}

//---------------------------------------------------------------------------
//
//	AhXXVe[u
//
//---------------------------------------------------------------------------
const int DMAC::MemDiffTable[8][4] = {
	{ 0, 1, -1, 0},		// 8bit, oCg
	{ 0, 2, -2, 0},		// 8bit, [h
	{ 0, 4, -4, 0},		// 8bit, O[h
	{ 0, 1, -1, 0},		// 8bit, pbNoCg
	{ 0, 1, -1, 0},		// 16bit, oCg
	{ 0, 2, -2, 0},		// 16bit, [h
	{ 0, 4, -4, 0},		// 16bit, O[h
	{ 0, 1, -1, 0}		// 16bit, pbNoCg
};

//---------------------------------------------------------------------------
//
//	foCXAhXXVe[u
//
//---------------------------------------------------------------------------
const int DMAC::DevDiffTable[8][4] = {
	{ 0, 2, -2, 0},		// 8bit, oCg
	{ 0, 4, -4, 0},		// 8bit, [h
	{ 0, 8, -8, 0},		// 8bit, O[h
	{ 0, 2, -2, 0},		// 8bit, pbNoCg
	{ 0, 1, -1, 0},		// 16bit, oCg
	{ 0, 2, -2, 0},		// 16bit, [h
	{ 0, 4, -4, 0},		// 16bit, O[h
	{ 0, 1, -1, 0}		// 16bit, pbNoCg
};
