//---------------------------------------------------------------------------
//
//	X68000 EMULATOR "XM6"
//
//	Copyright (C) 2001-2006 ohD(ytanaka@ipc-tokai.or.jp)
//	[ FDC(uPD72065) ]
//
//---------------------------------------------------------------------------

#include "os.h"
#include "xm6.h"
#include "fdd.h"
#include "iosc.h"
#include "dmac.h"
#include "schedule.h"
#include "vm.h"
#include "log.h"
#include "fileio.h"
#include "config.h"
#include "fdc.h"

//===========================================================================
//
//	FDC
//
//===========================================================================
//#define FDC_LOG

//---------------------------------------------------------------------------
//
//	RXgN^
//
//---------------------------------------------------------------------------
FDC::FDC(VM *p) : MemDevice(p)
{
	// foCXID
	dev.id = MAKEID('F', 'D', 'C', ' ');
	dev.desc = "FDC (uPD72065)";

	// JnAhXAIAhX
	memdev.first = 0xe94000;
	memdev.last = 0xe95fff;

	// IuWFNg
	iosc = NULL;
	dmac = NULL;
	fdd = NULL;
}

//---------------------------------------------------------------------------
//
//	
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::Init()
{
	ASSERT(this);

	// {NX
	if (!MemDevice::Init()) {
		return FALSE;
	}

	// IOSC擾
	iosc = (IOSC*)vm->SearchDevice(MAKEID('I', 'O', 'S', 'C'));
	ASSERT(iosc);

	// DMAC擾
	dmac = (DMAC*)vm->SearchDevice(MAKEID('D', 'M', 'A', 'C'));
	ASSERT(dmac);

	// FDD擾
	fdd = (FDD*)vm->SearchDevice(MAKEID('F', 'D', 'D', ' '));
	ASSERT(fdd);

	// Cxg
	event.SetDevice(this);
	event.SetDesc("Data Transfer");
	event.SetUser(0);
	event.SetTime(0);
	scheduler->AddEvent(&event);

	// [htOAfAhCutO(ApplyCfg)
	fdc.fast = FALSE;
	fdc.dual = FALSE;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	N[Abv
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Cleanup()
{
	ASSERT(this);

	// {NX
	MemDevice::Cleanup();
}

//---------------------------------------------------------------------------
//
//	Zbg
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Reset()
{
	ASSERT(this);
	LOG0(Log::Normal, "Zbg");

	// f[^WX^EXe[^XWX^
	fdc.dr = 0;
	fdc.sr = 0;
	fdc.sr |= sr_rqm;
	fdc.sr &= ~sr_dio;
	fdc.sr &= ~sr_ndm;
	fdc.sr &= ~sr_cb;

	// hCuZNgWX^EST0-ST3
	fdc.dcr = 0;
	fdc.dsr = 0;
	fdc.st[0] = 0;
	fdc.st[1] = 0;
	fdc.st[2] = 0;
	fdc.st[3] = 0;

	// R}hʃp[^
	fdc.srt = 1 * 2000;
	fdc.hut = 16 * 2000;
	fdc.hlt = 2 * 2000;
	fdc.hd = 0;
	fdc.us = 0;
	fdc.cyl[0] = 0;
	fdc.cyl[1] = 0;
	fdc.cyl[2] = 0;
	fdc.cyl[3] = 0;
	fdc.chrn[0] = 0;
	fdc.chrn[1] = 0;
	fdc.chrn[2] = 0;
	fdc.chrn[3] = 0;

	// ̑
	fdc.eot = 0;
	fdc.gsl = 0;
	fdc.dtl = 0;
	fdc.sc = 0;
	fdc.gpl = 0;
	fdc.d = 0;
	fdc.err = 0;
	fdc.seek = FALSE;
	fdc.ndm = FALSE;
	fdc.mfm = FALSE;
	fdc.mt = FALSE;
	fdc.sk = FALSE;
	fdc.tc = FALSE;
	fdc.load = FALSE;

	// ]n
	fdc.offset = 0;
	fdc.len = 0;
	memset(fdc.buffer, 0, sizeof(fdc.buffer));

	// tF[YAR}h
	fdc.phase = idle;
	fdc.cmd = no_cmd;

	// pPbgǗ
	fdc.in_len = 0;
	fdc.in_cnt = 0;
	memset(fdc.in_pkt, 0, sizeof(fdc.in_pkt));
	fdc.out_len = 0;
	fdc.out_cnt = 0;
	memset(fdc.out_pkt, 0, sizeof(fdc.out_pkt));

	// Cxg~
	event.SetTime(0);

	// ANZX~(FDDZbgfdd.selected=0ƂȂ)
	fdd->Access(FALSE);
}

//---------------------------------------------------------------------------
//
//	\tgEFAZbg
//
//---------------------------------------------------------------------------
void FASTCALL FDC::SoftReset()
{
	// WX^(FDC)
	fdc.dr = 0;
	fdc.sr = 0;
	fdc.sr |= sr_rqm;
	fdc.sr &= ~sr_dio;
	fdc.sr &= ~sr_ndm;
	fdc.sr &= ~sr_cb;

	fdc.st[0] = 0;
	fdc.st[1] = 0;
	fdc.st[2] = 0;
	fdc.st[3] = 0;

	fdc.srt = 1 * 2000;
	fdc.hut = 16 * 2000;
	fdc.hlt = 2 * 2000;
	fdc.hd = 0;
	fdc.us = 0;
	fdc.cyl[0] = 0;
	fdc.cyl[1] = 0;
	fdc.cyl[2] = 0;
	fdc.cyl[3] = 0;
	fdc.chrn[0] = 0;
	fdc.chrn[1] = 0;
	fdc.chrn[2] = 0;
	fdc.chrn[3] = 0;

	fdc.eot = 0;
	fdc.gsl = 0;
	fdc.dtl = 0;
	fdc.sc = 0;
	fdc.gpl = 0;
	fdc.d = 0;
	fdc.err = 0;
	fdc.seek = FALSE;
	fdc.ndm = FALSE;
	fdc.mfm = FALSE;
	fdc.mt = FALSE;
	fdc.sk = FALSE;
	fdc.tc = FALSE;
	fdc.load = FALSE;

	fdc.offset = 0;
	fdc.len = 0;

	// tF[YAR}h
	fdc.phase = idle;
	fdc.cmd = fdc_reset;

	// pPbgǗ
	fdc.in_len = 0;
	fdc.in_cnt = 0;
	memset(fdc.in_pkt, 0, sizeof(fdc.in_pkt));
	fdc.out_len = 0;
	fdc.out_cnt = 0;
	memset(fdc.out_pkt, 0, sizeof(fdc.out_pkt));

	// Cxg~
	event.SetTime(0);

	// ANZX~
	fdd->Access(FALSE);
}

//---------------------------------------------------------------------------
//
//	Z[u
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::Save(Fileio *fio, int ver)
{
	size_t sz;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "Z[u");

	// TCYZ[u
	sz = sizeof(fdc_t);
	if (!fio->Write(&sz, sizeof(sz))) {
		return FALSE;
	}

	// {̂Z[u
	if (!fio->Write(&fdc, (int)sz)) {
		return FALSE;
	}

	// CxgZ[u
	if (!event.Save(fio, ver)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::Load(Fileio *fio, int ver)
{
	size_t sz;

	ASSERT(this);
	ASSERT(fio);

	LOG0(Log::Normal, "[h");

	// TCY[hAr
	if (!fio->Read(&sz, sizeof(sz))) {
		return FALSE;
	}
	if (sz != sizeof(fdc_t)) {
		return FALSE;
	}

	// {̂[h
	if (!fio->Read(&fdc, (int)sz)) {
		return FALSE;
	}

	// Cxg[h
	if (!event.Load(fio, ver)) {
		return FALSE;
	}

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	ݒKp
//
//---------------------------------------------------------------------------
void FASTCALL FDC::ApplyCfg(const Config *config)
{
	ASSERT(this);
	ASSERT(config);

	LOG0(Log::Normal, "ݒKp");

	// [h
	fdc.fast = config->floppy_speed;
#if defined(FDC_LOG)
	if (fdc.fast) {
		LOG0(Log::Normal, "[h ON");
	}
	else {
		LOG0(Log::Normal, "[h OFF");
	}
#endif	// FDC_LOG

	// 2DD/2HDphCu
	fdc.dual = config->dual_fdd;
#if defined(FDC_LOG)
	if (fdc.dual) {
		LOG0(Log::Normal, "2DD/2HDphCu");
	}
	else {
		LOG0(Log::Normal, "2HDphCu");
	}
#endif	// FDC_LOG
}

//---------------------------------------------------------------------------
//
//	oCgǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL FDC::ReadByte(DWORD addr)
{
	int i;
	int status;
	DWORD bit;
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// AhX̂݃fR[hĂ
	if ((addr & 1) == 0) {
		return 0xff;
	}

	// 8oCgPʂŃ[v
	addr &= 0x07;
	addr >>= 1;

	// EFCg
	scheduler->Wait(1);

	switch (addr) {
		// Xe[^XWX^
		case 0:
			return fdc.sr;

		// f[^WX^
		case 1:
			// SEEK荞݂łȂ΁A荞݃lQ[g
			if (!fdc.seek) {
				Interrupt(FALSE);
			}

			switch (fdc.phase) {
				// stF[Y(ER)
				case read:
					fdc.sr &= ~sr_rqm;
					return Read();

				// UgtF[Y
				case result:
					ASSERT(fdc.out_cnt >= 0);
					ASSERT(fdc.out_cnt < 0x10);
					ASSERT(fdc.out_len > 0);

					// pPbgf[^o
					data = fdc.out_pkt[fdc.out_cnt];
					fdc.out_cnt++;
					fdc.out_len--;

					// c背OX0ɂȂAAChtF[Y
					if (fdc.out_len == 0) {
						Idle();
					}
					return data;
			}
			LOG0(Log::Warning, "FDCf[^WX^ǂݍݖ");
			return 0xff;

		// hCuXe[^XWX^
		case 2:
			data = 0;
			bit = 0x08;
			for (i=3; i>=0; i--) {
				// DCR̃rbgĂ邩
				if ((fdc.dcr & bit) != 0) {
					// YhCũXe[^XOR(b7,b6̂)
					status = fdd->GetStatus(i);
					data |= (DWORD)(status & 0xc0);
				}
				bit >>= 1;
			}

			// FDD荞݂𗎂Ƃ(FDC荞݂ł͂ȂA)
			iosc->IntFDD(FALSE);
			return data;

		// hCuZNgWX^
		case 3:
			LOG0(Log::Warning, "hCuZNgWX^ǂݍ");
			return 0xff;
	}

	// ʏAɂ͂Ȃ
	ASSERT(FALSE);
	return 0xff;
}

//---------------------------------------------------------------------------
//
//	[hǂݍ
//
//---------------------------------------------------------------------------
DWORD FASTCALL FDC::ReadWord(DWORD addr)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);

	return (0xff00 | ReadByte(addr + 1));
}

//---------------------------------------------------------------------------
//
//	oCg
//
//---------------------------------------------------------------------------
void FASTCALL FDC::WriteByte(DWORD addr, DWORD data)
{
	int i;
	DWORD bit;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// AhX̂݃fR[hĂ
	if ((addr & 1) == 0) {
		return;
	}

	// 8oCgPʂŃ[v
	addr &= 0x07;
	addr >>= 1;

	// EFCg
	scheduler->Wait(1);

	switch (addr) {
		// R}hWX^
		case 0:
			switch (data) {
				// RESET STANDBY
				case 0x34:
#if defined(FDC_LOG)
					LOG0(Log::Normal, "RESET STANDBYR}h");
#endif	// FDC_LOG
					fdc.cmd = reset_stdby;
					Result();
					return;
				// SET STANDBY
				case 0x35:
#if defined(FDC_LOG)
					LOG0(Log::Normal, "SET STANDBYR}h");
#endif	// FDC_LOG
					fdc.cmd = set_stdby;
					Idle();
					return;
				// SOFTWARE RESET
				case 0x36:
#if defined(FDC_LOG)
					LOG0(Log::Normal, "SOFTWARE RESETR}h");
#endif	// FDC_LOG
					SoftReset();
					return;
			}
			LOG1(Log::Warning, "ȓR}h %02X", data);
			return;

		// f[^WX^
		case 1:
			// SEEK荞݂łȂ΁A荞݃lQ[g
			if (!fdc.seek) {
				Interrupt(FALSE);
			}

			switch (fdc.phase) {
				// AChtF[Y
				case idle:
					Command(data);
					return;

				// R}htF[Y
				case command:
					ASSERT(fdc.in_cnt >= 0);
					ASSERT(fdc.in_cnt < 0x10);
					ASSERT(fdc.in_len > 0);

					// pPbgɃf[^Zbg
					fdc.in_pkt[fdc.in_cnt] = data;
					fdc.in_cnt++;
					fdc.in_len--;

					// c背OX0ɂȂAstF[Y
					if (fdc.in_len == 0) {
						Execute();
					}
					return;

				// stF[Y(EW)
				case write:
					fdc.sr &= ~sr_rqm;
					Write(data);
					return;
			}
			LOG1(Log::Warning, "FDCf[^WX^ݖ $%02X", data);
			return;

		// hCuRg[WX^
		case 2:
			// 4bit10ɂȂƂ𒲂ׂ
			bit = 0x01;
			for (i=0; i<4; i++) {
				if ((fdc.dcr & bit) != 0) {
					if ((data & bit) == 0) {
						// 10̃GbWŁADCȐ4rbgKp
						fdd->Control(i, fdc.dcr);
					}
				}
				bit <<= 1;
			}

			// lۑ
			fdc.dcr = data;
			return;

		// hCuZNgWX^
		case 3:
			// 2bitŃANZXhCuI
			fdc.dsr = (DWORD)(data & 0x03);

			// ŏʂŃ[^
			if (data & 0x80) {
				fdd->SetMotor(fdc.dsr, TRUE);
			}
			else {
				fdd->SetMotor(fdc.dsr, FALSE);
			}

			// 2HD/2DD؂ւ(hCu2DDΉłȂΖ)
			if (fdc.dual) {
				if (data & 0x10) {
					fdd->SetHD(FALSE);
				}
				else {
					fdd->SetHD(TRUE);
				}
			}
			else {
				fdd->SetHD(TRUE);
			}
			return;
	}

	// ʏAɂ͂Ȃ
	ASSERT(FALSE);
}

//---------------------------------------------------------------------------
//
//	[h
//
//---------------------------------------------------------------------------
void FASTCALL FDC::WriteWord(DWORD addr, DWORD data)
{
	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));
	ASSERT((addr & 1) == 0);

	WriteByte(addr + 1, (BYTE)data);
}

//---------------------------------------------------------------------------
//
//	ǂݍ݂̂
//
//---------------------------------------------------------------------------
DWORD FASTCALL FDC::ReadOnly(DWORD addr) const
{
	int i;
	int status;
	DWORD bit;
	DWORD data;

	ASSERT(this);
	ASSERT((addr >= memdev.first) && (addr <= memdev.last));

	// AhX̂݃fR[hĂ
	if ((addr & 1) == 0) {
		return 0xff;
	}

	// 8oCgPʂŃ[v
	addr &= 0x07;
	addr >>= 1;

	switch (addr) {
		// Xe[^XWX^
		case 0:
			return fdc.sr;

		// f[^WX^
		case 1:
			if (fdc.phase == result) {
				// pPbgf[^o(XVȂ);
				return fdc.out_pkt[fdc.out_cnt];
			}
			return 0xff;

		// hCuXe[^XWX^
		case 2:
			data = 0;
			bit = 0x08;
			for (i=3; i>=0; i--) {
				// DCR̃rbgĂ邩
				if ((fdc.dcr & bit) != 0) {
					// YhCũXe[^XOR(b7,b6̂)
					status = fdd->GetStatus(i);
					data |= (DWORD)(status & 0xc0);
				}
				bit >>= 1;
			}
			return data;

		// hCuZNgWX^
		case 3:
			return 0xff;
	}

	return 0xff;
}

//---------------------------------------------------------------------------
//
//	CxgR[obN
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::Callback(Event *ev)
{
	int i;
	int thres;

	ASSERT(this);
	ASSERT(ev);

	// AChtF[Y̓wbhA[h
	if (fdc.phase == idle) {
		fdc.load = FALSE;

		// P
		return FALSE;
	}

	// wbh[h
	fdc.load = TRUE;

	// stF[Y
	if (fdc.phase == execute) {
		// IDNO DATA܂ł̎
		Result();

		// P
		return FALSE;
	}

	// Write ID͐p
	if (fdc.cmd == write_id) {
		ASSERT(fdc.len > 0);
		ASSERT((fdc.len & 3) == 0);

		// ԍĐݒ
		if (fdc.fast) {
			ev->SetTime(32 * 4);
		}
		else {
			ev->SetTime(fdd->GetRotationTime() / fdc.sc);
		}

		fdc.sr |= sr_rqm;
		dmac->ReqDMA(0);
		fdc.sr |= sr_rqm;
		dmac->ReqDMA(0);
		fdc.sr |= sr_rqm;
		dmac->ReqDMA(0);
		fdc.sr |= sr_rqm;
		dmac->ReqDMA(0);
		return TRUE;
	}

	// Read(Del)Data/Write(Del)Data/Scan/ReadDiagBԍĐݒ
	EventRW();

	// f[^]NGXg
	fdc.sr |= sr_rqm;

	// f[^]
	if (!fdc.ndm) {
		// DMA[h(DMANGXg)B64oCg܂Ƃ߂čs
		if (fdc.fast) {
			// 1̃CxgŁA]CPUp[2/3]
			thres = (int)scheduler->GetCPUSpeed();
			thres <<= 1;
			thres /= 3;

			// UgtF[Yɓ܂ŌJԂ
			while (fdc.phase != result) {
				// CPUp[Ȃrőł؂
				if (scheduler->GetCPUCycle() > thres) {
					break;
				}

				// ]
				dmac->ReqDMA(0);
			}
		}
		else {
			// ʏB64oCg܂Ƃ߂
			for (i=0; i<64; i++) {
				if (fdc.phase == result) {
					break;
				}
				dmac->ReqDMA(0);
			}
		}
		return TRUE;
	}

	// Non-DMA[h(荞݃NGXg)
	Interrupt(TRUE);
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	[NAhX擾
//
//---------------------------------------------------------------------------
const FDC::fdc_t* FASTCALL FDC::GetWork() const
{
	ASSERT(this);

	// AhX^(fdc.buffer傫)
	return &fdc;
}

//---------------------------------------------------------------------------
//
//	V[N
//
//---------------------------------------------------------------------------
void FASTCALL FDC::CompleteSeek(int drive, BOOL status)
{
	ASSERT(this);
	ASSERT((drive >= 0) && (drive <= 3));

#if defined(FDC_LOG)
	if (status) {
		LOG2(Log::Normal, "V[N hCu%d V_%02X",
					drive, fdd->GetCylinder(drive));
	}
	else {
		LOG2(Log::Normal, "V[Ns hCu%d V_%02X",
					drive, fdd->GetCylinder(drive));
	}
#endif	// FDC_LOG

	// recalibrate܂seek̂ݗL
	if ((fdc.cmd == recalibrate) || (fdc.cmd == seek)) {
		// ST0쐬(UŜ)
		fdc.st[0] = fdc.us;

		// Xe[^X
		if (status) {
			// hCu2,3EC𗧂Ă
			if (drive <= 1) {
				// Seek End
				fdc.st[0] |= 0x20;
			}
			else {
				// Equipment Check, Attention Interrupt
				fdc.st[0] |= 0x10;
				fdc.st[0] |= 0xc0;
			}
		}
		else {
			if (drive <= 1) {
				// Seek End
				fdc.st[0] |= 0x20;
			}
			// Not Ready, Abnormal Terminate
			fdc.st[0] |= 0x08;
			fdc.st[0] |= 0x40;
		}

		// SEEK荞
		Interrupt(TRUE);
		fdc.seek = TRUE;
		Idle();
		return;
	}

	LOG1(Log::Warning, "ȃV[Nʒm hCu%d", drive);
}

//---------------------------------------------------------------------------
//
//	TCAT[g
//
//---------------------------------------------------------------------------
void FASTCALL FDC::SetTC()
{
	ASSERT(this);

	// AChtF[YŃNA邽߁AAChtF[YȊOȂ
	if (fdc.phase != idle) {
		fdc.tc = TRUE;
	}
}

//---------------------------------------------------------------------------
//
//	AChtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Idle()
{
	ASSERT(this);

	// tF[Yݒ
	fdc.phase = idle;
	fdc.err = 0;
	fdc.tc = FALSE;

	// CxgI
	event.SetTime(0);

	// wbh[hԂȂAA[ĥ߂̃Cxgݒ
	if (fdc.load) {
		// A[h̕Kv
		if (fdc.hut > 0) {
			event.SetTime(fdc.hut);
		}
	}

	// Xe[^XWX^̓R}h҂
	fdc.sr = sr_rqm;

	// ANZXI
	fdd->Access(FALSE);
}

//---------------------------------------------------------------------------
//
//	R}htF[Y
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Command(DWORD data)
{
	DWORD mask;

	ASSERT(this);
	ASSERT(data < 0x100);

	// R}htF[Y(FDC BUSY)
	fdc.phase = command;
	fdc.sr |= sr_cb;

	// ̓pPbg
	fdc.in_pkt[0] = data;
	fdc.in_cnt = 1;
	fdc.in_len = 0;

	// }XN(1)
	mask = data;

	// FDCZbg͂łsł
	switch (mask) {
		// RESET STANDBY
		case 0x34:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "RESET STANDBYR}h");
#endif	// FDC_LOG
			fdc.cmd = reset_stdby;
			Result();
			return;

		// SET STANDBY
		case 0x35:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "SET STANDBYR}h");
#endif	// FDC_LOG
			fdc.cmd = set_stdby;
			Idle();
			return;

		// SOFTWARE RESET
		case 0x36:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "SOFTWARE RESETR}h");
#endif	// FDC_LOG
			SoftReset();
			return;
	}

	// V[NnR}hśASENSE INTERRUPT STATUSȊOȂ
	if (fdc.seek) {
		// SENSE INTERRUPT STATUS
		if (mask == 0x08) {
#if defined(FDC_LOG)
			LOG0(Log::Normal, "SENSE INTERRUPT STATUSR}h");
#endif	// FDC_LOG
			fdc.cmd = sense_int_stat;

			// 荞݃lQ[g
			fdc.sr &= 0xf0;
			fdc.seek = FALSE;
			Interrupt(FALSE);

			// p[^ȂAstF[YȂ
			Result();
			return;
		}

		// ȊO͑SĖR}h
#if defined(FDC_LOG)
		LOG0(Log::Normal, "INVALIDR}h");
#endif	// FDC_LOG
		fdc.cmd = invalid;
		Result();
		return;
	}

	// SENSE INTERRUPT STATUS()
	if (mask == 0x08) {
#if defined(FDC_LOG)
		LOG0(Log::Normal, "INVALIDR}h");
#endif	// FDC_LOG
		fdc.cmd = invalid;
		Result();
		return;
	}

	// Xe[^XNA
	fdc.st[0] = 0;
	fdc.st[1] = 0;
	fdc.st[2] = 0;

	// ʏ
	switch (mask) {
		// READ DIAGNOSTIC(FM[h)
		case 0x02:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "READ DIAGNOSTICR}h(FM[h)");
#endif	// FDC_LOG
			CommandRW(read_diag, data);
			return;

		// SPECIFY
		case 0x03:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "SPECIFYR}h");
#endif	// FDC_LOG
			fdc.cmd = specify;
			fdc.in_len = 2;
			return;

		// SENSE DEVICE STATUS
		case 0x04:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "SENSE DEVICE STATUSR}h");
#endif	// FDC_LOG
			fdc.cmd = sense_dev_stat;
			fdc.in_len = 1;
			return;

		// RECALIBRATE
		case 0x07:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "RECALIBRATER}h");
#endif	// FDC_LOG
			fdc.cmd = recalibrate;
			fdc.in_len = 1;
			return;

		// READ ID(FM[h)
		case 0x0a:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "READ IDR}h(FM[h)");
#endif	// FDC_LOG
			fdc.cmd = read_id;
			fdc.mfm = FALSE;
			fdc.in_len = 1;
			return;

		// WRITE ID(FM[h)
		case 0x0d:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "WRITE IDR}h(FM[h)");
#endif	// FDC_LOG
			fdc.cmd = write_id;
			fdc.mfm = FALSE;
			fdc.in_len = 5;
			return;

		// SEEK
		case 0x0f:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "SEEKR}h");
#endif	// FDC_LOG
			fdc.cmd = seek;
			fdc.in_len = 2;
			return;

		// READ DIAGNOSTIC(MFM[h)
		case 0x42:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "READ DIAGNOSTICR}h(MFM[h)");
#endif	// FDC_LOG
			CommandRW(read_diag, data);
			return;

		// READ ID(MFM[h)
		case 0x4a:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "READ IDR}h(MFM[h)");
#endif	// FDC_LOG
			fdc.cmd = read_id;
			fdc.mfm = TRUE;
			fdc.in_len = 1;
			return;

		// WRITE ID(MFM[h)
		case 0x4d:
#if defined(FDC_LOG)
			LOG0(Log::Normal, "WRITE IDR}h(MFM[h)");
#endif	// FDC_LOG
			fdc.cmd = write_id;
			fdc.mfm = TRUE;
			fdc.in_len = 5;
			return;
	}

	// }XN(2)
	mask &= 0x3f;

	// WRITE DATA
	if (mask == 0x05) {
#if defined(FDC_LOG)
		LOG0(Log::Normal, "WRITE DATAR}h");
#endif	// FDC_LOG
		CommandRW(write_data, data);
		return;
	}

	// WRITE DELETED DATA
	if (mask == 0x09) {
#if defined(FDC_LOG)
		LOG0(Log::Normal, "WRITE DELETED DATAR}h");
#endif	// FDC_LOG
		CommandRW(write_del_data, data);
		return;
	}

	// }XN(3);
	mask &= 0x1f;

	// READ DATA
	if (mask == 0x06) {
#if defined(FDC_LOG)
		LOG0(Log::Normal, "READ DATAR}h");
#endif	// FDC_LOG
		CommandRW(read_data, data);
		return;
	}

	// READ DELETED DATA
	if (mask == 0x0c) {
#if defined(FDC_LOG)
		LOG0(Log::Normal, "READ DELETED DATAR}h");
#endif	// FDC_LOG
		CommandRW(read_data, data);
		return;
	}

	// SCAN EQUAL
	if (mask == 0x11) {
#if defined(FDC_LOG)
		LOG0(Log::Normal, "SCAN EQUALR}h");
#endif	// FDC_LOG
		CommandRW(scan_eq, data);
		return;
	}

	// SCAN LOW OR EQUAL
	if (mask == 0x19) {
#if defined(FDC_LOG)
		LOG0(Log::Normal, "SCAN LOW OR EQUALR}h");
#endif	// FDC_LOG
		CommandRW(scan_lo_eq, data);
		return;
	}

	// SCAN HIGH OR EQUAL
	if (mask == 0x1d) {
#if defined(FDC_LOG)
		LOG0(Log::Normal, "SCAN HIGH OR EQUALR}h");
#endif	// FDC_LOG
		CommandRW(scan_hi_eq, data);
		return;
	}

	// 
	LOG1(Log::Warning, "R}htF[YΉR}h $%02X", data);
	Idle();
}

//---------------------------------------------------------------------------
//
//	R}htF[Y(Read/Writen)
//
//---------------------------------------------------------------------------
void FASTCALL FDC::CommandRW(fdccmd cmd, DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	// R}h
	fdc.cmd = cmd;

	// MT
	if (data & 0x80) {
		fdc.mt = TRUE;
	}
	else {
		fdc.mt = FALSE;
	}

	// MFM
	if (data & 0x40) {
		fdc.mfm = TRUE;
	}
	else {
		fdc.mfm = FALSE;
	}

	// SK(READ/SCAN̂)
	if (data & 0x20) {
		fdc.sk = TRUE;
	}
	else {
		fdc.sk = FALSE;
	}

	// R}htF[Y̎coCg
	fdc.in_len = 8;
}

//---------------------------------------------------------------------------
//
//	stF[Y
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Execute()
{
	ASSERT(this);

	// stF[Y
	fdc.phase = execute;

	// ANZXJnACxg~
	fdd->Access(TRUE);
	event.SetTime(0);

	// R}h
	switch (fdc.cmd) {
		// SPECIFY
		case specify:
			// SRT
			fdc.srt = (fdc.in_pkt[1] >> 4) & 0x0f;
			fdc.srt = 16 - fdc.srt;
			fdc.srt <<= 11;

			// HUT (016ƓBi82078f[^V[gɂ)
			fdc.hut = fdc.in_pkt[1] & 0x0f;
			if (fdc.hut == 0) {
				fdc.hut = 16;
			}
			fdc.hut <<= 15;

			// HLT (0HUTƓl)
			fdc.hlt = (fdc.in_pkt[2] >> 1) & 0x7f;
			if (fdc.hlt == 0) {
				fdc.hlt = 0x80;
			}
			fdc.hlt <<= 12;

			// NDM
			if (fdc.in_pkt[2] & 1) {
				fdc.ndm = TRUE;
				LOG0(Log::Warning, "Non-DMA[hɐݒ");
			}
			else {
				fdc.ndm = FALSE;
			}

			// UgtF[Ysv
			Idle();
			return;

		// SENSE DEVICE STATUS
		case sense_dev_stat:
			fdc.us = fdc.in_pkt[1] & 0x03;
			fdc.hd = fdc.in_pkt[1] & 0x04;

			// UgtF[Y
			Result();
			return;

		// RECALIBRATE
		case recalibrate:
			// gbN0փV[N
			fdc.us = fdc.in_pkt[1] & 0x03;
			fdc.cyl[fdc.us] = 0;

			// SR쐬(SEEKnR}hsNon-Busy)
			fdc.sr &= 0xf0;
			fdc.sr &= ~sr_cb;
			fdc.sr &= ~sr_rqm;
			fdc.sr |= (1 << fdc.dsr);

			// ŌɎsĂ(CompleteSeekĂ΂邽)
			fdd->Recalibrate(fdc.srt);
			return;

		// SEEK
		case seek:
			fdc.us = fdc.in_pkt[1] & 0x03;

			// SR쐬(SEEKnR}hsNon-Busy)
			fdc.sr &= 0xf0;
			fdc.sr &= ~sr_cb;
			fdc.sr &= ~sr_rqm;
			fdc.sr |= (1 << fdc.dsr);

			// ŌɎsĂ(CompleteSeekĂ΂邽)
			if (fdc.cyl[fdc.us] < fdc.in_pkt[2]) {
				// XebvC
				fdd->StepIn(fdc.in_pkt[2] - fdc.cyl[fdc.us], fdc.srt);
			}
			else {
				// XebvAEg
				fdd->StepOut(fdc.cyl[fdc.us] - fdc.in_pkt[2], fdc.srt);
			}
			fdc.cyl[fdc.us] = fdc.in_pkt[2];
			return;

		// READ ID
		case read_id:
			ReadID();
			return;

		// WRITE ID
		case write_id:
			fdc.us = fdc.in_pkt[1] & 0x03;
			fdc.hd = fdc.in_pkt[1] & 0x04;
			fdc.st[0] = fdc.us;
			fdc.st[0] |= fdc.hd;
			fdc.chrn[3] = fdc.in_pkt[2];
			fdc.sc = fdc.in_pkt[3];
			fdc.gpl = fdc.in_pkt[4];
			fdc.d = fdc.in_pkt[5];
			if (!WriteID()) {
				Result();
			}
			return;

		// READ DIAGNOSTIC
		case read_diag:
			ExecuteRW();
			if (!ReadDiag()) {
				Result();
			}
			return;

		// READ DATA
		case read_data:
			ExecuteRW();
			if (!ReadData()) {
				Result();
			}
			return;

		// READ DELETED DATA
		case read_del_data:
			ExecuteRW();
			if (!ReadData()) {
				Result();
			}
			return;

		// WRITE DATA
		case write_data:
			ExecuteRW();
			if (!WriteData()) {
				Result();
			}
			return;

		// WRITE DELETED_DATA
		case write_del_data:
			ExecuteRW();
			if (!WriteData()) {
				Result();
			}
			return;

		// SCANn
		case scan_eq:
		case scan_lo_eq:
		case scan_hi_eq:
			ExecuteRW();
			if (!Scan()) {
				Result();
			}
			return;
	}

	LOG1(Log::Warning, "stF[YΉR}h $%02X", fdc.in_pkt[0]);
}

//---------------------------------------------------------------------------
//
//	stF[Y(ReadID)
//
//---------------------------------------------------------------------------
void FASTCALL FDC::ReadID()
{
	DWORD hus;

	ASSERT(this);

	// HD, USL
	fdc.us = fdc.in_pkt[1] & 0x03;
	fdc.hd = fdc.in_pkt[1] & 0x04;

	// FDDɎsBNOTREADY, NODATA, MAMl
	fdc.err = fdd->ReadID(&(fdc.out_pkt[3]), fdc.mfm, fdc.hd);

	// NOT READYȂ炷UgtF[Y
	if (fdc.err & FDD_NOTREADY) {
		Result();
		return;
	}

	// ɂ鎞Ԃݒ
	hus = fdd->GetSearch();
	event.SetTime(hus);
	fdc.sr &= ~sr_rqm;
}

//---------------------------------------------------------------------------
//
//	stF[Y(Read/Writen)
//
//---------------------------------------------------------------------------
void FASTCALL FDC::ExecuteRW()
{
	ASSERT(this);

	// 8oCg̃pPbg𕪊(ŏIoCg͏DTLɃZbg)
	fdc.us = fdc.in_pkt[1] & 0x03;
	fdc.hd = fdc.in_pkt[1] & 0x04;
	fdc.st[0] = fdc.us;
	fdc.st[0] |= fdc.hd;

	fdc.chrn[0] = fdc.in_pkt[2];
	fdc.chrn[1] = fdc.in_pkt[3];
	fdc.chrn[2] = fdc.in_pkt[4];
	fdc.chrn[3] = fdc.in_pkt[5];

	fdc.eot = fdc.in_pkt[6];
	fdc.gsl = fdc.in_pkt[7];
	fdc.dtl = fdc.in_pkt[8];
}

//---------------------------------------------------------------------------
//
//	stF[Y(Read)
//
//---------------------------------------------------------------------------
BYTE FASTCALL FDC::Read()
{
	BYTE data;

	ASSERT(fdc.len > 0);
	ASSERT(fdc.offset < 0x4000);

	// obt@f[^
	data = fdc.buffer[fdc.offset];
	fdc.offset++;
	fdc.len--;

	// ŌłȂ΂̂܂ܑ
	if (fdc.len > 0) {
		return data;
	}

	// READ DIAGNOSTIC̏ꍇ͂ŏI
	if (fdc.cmd == read_diag) {
		// IȂAZN^i߂
		if (fdc.err == FDD_NOERROR) {
			NextSector();
		}
		// Cxgł؂AUgtF[Y
		event.SetTime(0);
		Result();
		return data;
	}

	// ُIȂÃZN^őł؂
	if (fdc.err != FDD_NOERROR) {
		// Cxgł؂AUgtF[Y
		event.SetTime(0);
		Result();
		return data;
	}

	// }`ZN^
	if (!NextSector()) {
		// Cxgł؂AUgtF[Y
		event.SetTime(0);
		Result();
		return data;
	}

	// ̃ZN^̂ŁA
	if (!ReadData()) {
		// ZN^ǂݎs\
		event.SetTime(0);
		Result();
		return data;
	}

	// OKÃZN^
	return data;
}

//---------------------------------------------------------------------------
//
//	stF[Y(Write)
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Write(DWORD data)
{
	ASSERT(this);
	ASSERT(fdc.len > 0);
	ASSERT(fdc.offset < 0x4000);
	ASSERT(data < 0x100);

	// WRITE ID̏ꍇ̓obt@ɗ߂̂
	if (fdc.cmd == write_id) {
		fdc.buffer[fdc.offset] = (BYTE)data;
		fdc.offset++;
		fdc.len--;

		// I`FbN
		if (fdc.len == 0) {
			WriteBack();
			event.SetTime(0);
			Result();
		}
		return;
	}

	// XLn̏ꍇ͔r
	if ((fdc.cmd == scan_eq) || (fdc.cmd == scan_lo_eq) || (fdc.cmd == scan_hi_eq)) {
		Compare(data);
		return;
	}

	// obt@փf[^
	fdc.buffer[fdc.offset] = (BYTE)data;
	fdc.offset++;
	fdc.len--;

	// ŌłȂ΂̂܂ܑ
	if (fdc.len > 0) {
		return;
	}

	// ݏI
	WriteBack();
	if (fdc.err != FDD_NOERROR) {
		event.SetTime(0);
		Result();
		return;
	}

	// }`ZN^
	if (!NextSector()) {
		// Cxgł؂AUgtF[Y
		event.SetTime(0);
		Result();
		return;
	}

	// ̃ZN^̂ŁA
	if (!WriteData()) {
		event.SetTime(0);
		Result();
	}
}

//---------------------------------------------------------------------------
//
//	stF[Y(Compare)
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Compare(DWORD data)
{
	ASSERT(this);
	ASSERT(data < 0x100);

	if (data != 0xff) {
		// LoCgŁA܂oĂȂȂ
		if (!(fdc.err & FDD_SCANNOT)) {
			// rKv
			switch (fdc.cmd) {
				case scan_eq:
					if (fdc.buffer[fdc.offset] != (BYTE)data) {
						fdc.err |= FDD_SCANNOT;
					}
					break;
				case scan_lo_eq:
					if (fdc.buffer[fdc.offset] > (BYTE)data) {
						fdc.err |= FDD_SCANNOT;
					}
					break;
				case scan_hi_eq:
					if (fdc.buffer[fdc.offset] < (BYTE)data) {
						fdc.err |= FDD_SCANNOT;
					}
					break;
				default:
					ASSERT(FALSE);
					break;
			}

		}
	}

	// ̃f[^
	fdc.offset++;
	fdc.len--;

	// ŌłȂ΂̂܂ܑ
	if (fdc.len > 0) {
		return;
	}

	// ŌȂ̂ŁAʂ܂Ƃ
	if (!(fdc.err & FDD_SCANNOT)) {
		// ok!
		fdc.err |= FDD_SCANEQ;
		event.SetTime(0);
		Result();
	}

	// STP2̂Ƃ́A+1
	if (fdc.dtl == 0x02) {
		fdc.chrn[2]++;
	}

	// }`ZN^
	if (!NextSector()) {
		// SCAN NOT͏オ܂܂Ȃ̂œs悢
		event.SetTime(0);
		Result();
		return;
	}

	// ̃ZN^̂ŁA
	if (!Scan()) {
		// SCAN NOT͏オ܂܂Ȃ̂œs悢
		event.SetTime(0);
		Result();
	}

	// SCAN NOTĂPZN^
	fdc.err &= ~FDD_SCANNOT;
}

//---------------------------------------------------------------------------
//
//	UgtF[Y
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Result()
{
	ASSERT(this);

	// UgtF[Y
	fdc.phase = result;
	fdc.sr |= sr_rqm;
	fdc.sr |= sr_dio;
	fdc.sr &= ~sr_ndm;

	// R}h
	switch (fdc.cmd) {
		// SENSE DEVICE STATUS
		case sense_dev_stat:
			// ST3쐬Af[^]
			MakeST3();
			fdc.out_pkt[0] = fdc.st[3];
			fdc.out_len = 1;
			fdc.out_cnt = 0;
			return;

		// SENSE INTERRUPT STATUS
		case sense_int_stat:
			// ST0EV_ԂBf[^]
			fdc.out_pkt[0] = fdc.st[0];
			fdc.out_pkt[1] = fdc.cyl[fdc.us];
			fdc.out_len = 2;
			fdc.out_cnt = 0;
			return;

		// READ ID
		case read_id:
			// ST0,ST1,ST2쐬BNOTREADY, NODATA, MAMl
			fdc.st[0] = fdc.us;
			fdc.st[0] |= fdc.hd;
			if (fdc.err & FDD_NOTREADY) {
				// Not Ready
				fdc.st[0] |= 0x08;
				fdc.st[1] = 0;
				fdc.st[2] = 0;
			}
			else {
				if (fdc.err != FDD_NOERROR) {
					// Abnormal Teriminate
					fdc.st[0] |= 0x40;
				}
				fdc.st[1] = fdc.err >> 8;
				fdc.st[2] = fdc.err & 0xff;
			}

			// f[^]AUgtF[Y荞
			fdc.out_pkt[0] = fdc.st[0];
			fdc.out_pkt[1] = fdc.st[1];
			fdc.out_pkt[2] = fdc.st[2];
			fdc.out_len = 7;
			fdc.out_cnt = 0;
			Interrupt(TRUE);
			return;

		// INVALID, RESET STANDBY
		case invalid:
		case reset_stdby:
			fdc.out_pkt[0] = 0x80;
			fdc.out_len = 1;
			fdc.out_cnt = 0;
			return;

		// READ,WRITE,SCANn
		case read_data:
		case read_del_data:
		case write_data:
		case write_del_data:
		case scan_eq:
		case scan_lo_eq:
		case scan_hi_eq:
		case read_diag:
		case write_id:
			ResultRW();
			return;
	}

	LOG1(Log::Warning, "UgtF[YΉR}h $%02X", fdc.in_pkt[0]);
}

//---------------------------------------------------------------------------
//
//	UgtF[Y(Read/Writen)
//
//---------------------------------------------------------------------------
void FASTCALL FDC::ResultRW()
{
	ASSERT(this);

	// ST0,ST1,ST2쐬
	if (fdc.err & FDD_NOTREADY) {
		// Not Ready
		fdc.st[0] |= 0x08;
		fdc.st[1] = 0;
		fdc.st[2] = 0;
	}
	else {
		if ((fdc.err != FDD_NOERROR) && (fdc.err != FDD_SCANEQ)) {
			// Abnormal Teriminate
			fdc.st[0] |= 0x40;
		}
		fdc.st[1] = fdc.err >> 8;
		fdc.st[2] = fdc.err & 0xff;
	}

	// READ DIAGNOSTIC0x40oȂ
	if (fdc.cmd == read_diag) {
		if (fdc.st[0] & 0x40) {
			fdc.st[0] &= ~0x40;
		}
	}

	// UgpPbgݒ
	fdc.out_pkt[0] = fdc.st[0];
	fdc.out_pkt[1] = fdc.st[1];
	fdc.out_pkt[2] = fdc.st[2];
	fdc.out_pkt[3] = fdc.chrn[0];
	fdc.out_pkt[4] = fdc.chrn[1];
	fdc.out_pkt[5] = fdc.chrn[2];
	fdc.out_pkt[6] = fdc.chrn[3];
	fdc.out_len = 7;
	fdc.out_cnt = 0;

	// ʏ̓UgtF[Y荞
	Interrupt(TRUE);
}

//---------------------------------------------------------------------------
//
//	荞
//
//---------------------------------------------------------------------------
void FASTCALL FDC::Interrupt(BOOL flag)
{
	ASSERT(this);

	// IOSCɒʒm
	iosc->IntFDC(flag);
}

//---------------------------------------------------------------------------
//
//	ST3쐬
//
//---------------------------------------------------------------------------
void FASTCALL FDC::MakeST3()
{
	ASSERT(this);

	// HD,USZbg
	fdc.st[3] = fdc.hd;
	fdc.st[3] |= fdc.us;

	// fB
	if (fdd->IsReady(fdc.dsr)) {
		// fB
		fdc.st[3] |= 0x20;

		// CgveNg
		if (fdd->IsWriteP(fdc.dsr)) {
			fdc.st[3] |= 0x40;
		}
	}
	else {
		// fBłȂ
		fdc.st[3] = 0x40;
	}

	// TRACK0
	if (fdd->GetCylinder(fdc.dsr) == 0) {
		fdc.st[3] |= 0x10;
	}
}

//---------------------------------------------------------------------------
//
//	READ (DELETED) DATAR}h
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::ReadData()
{
	int len;
	DWORD hus;

	ASSERT(this);
	ASSERT((fdc.cmd == read_data) || (fdc.cmd == read_del_data));

	// SRݒ
	fdc.sr |= sr_cb;
	fdc.sr |= sr_dio;
	fdc.sr &= ~sr_d3b;
	fdc.sr &= ~sr_d2b;
	fdc.sr &= ~sr_d1b;
	fdc.sr &= ~sr_d0b;

	// hCuɔCBNOTREADY,NODATA,MAM,CYLn,CRCn,DDAM
#if defined(FDC_LOG)
	LOG4(Log::Normal, "(C:%02X H:%02X R:%02X N:%02X)",
		fdc.chrn[0], fdc.chrn[1], fdc.chrn[2], fdc.chrn[3]);
#endif
	fdc.err = fdd->ReadSector(fdc.buffer, &fdc.len,
									fdc.mfm, fdc.chrn, fdc.hd);

	// DDAM(Deleted Sector)̗LŁACM(Control Mark)߂
	if (fdc.cmd == read_data) {
		// Read Data (DDAM̓G[)
		if (fdc.err & FDD_DDAM) {
			fdc.err &= ~FDD_DDAM;
			fdc.err |= FDD_CM;
		}
	}
	else {
		// Read Deleted Data (DDAMłȂ΃G[)
		if (!(fdc.err & FDD_DDAM)) {
			fdc.err |= FDD_CM;
		}
		fdc.err &= ~FDD_DDAM;
	}

	// IDCRC܂DATACRCȂADATAERR悹
	if ((fdc.err & FDD_IDCRC) || (fdc.err & FDD_DATACRC)) {
		fdc.err &= ~FDD_IDCRC;
		fdc.err |= FDD_DATAERR;
	}

	// N=0łNƂDTLg
	if (fdc.chrn[3] == 0) {
		len = 1 << (fdc.dtl + 7);
		if (len < fdc.len) {
			fdc.len = len;
		}
	}
	else {
		// (MacG~[^)
		len = (1 << (fdc.chrn[3] + 7));
		if (len < fdc.len) {
			fdc.len = len;
		}
	}

	// Not Ready̓UgtF[Y
	if (fdc.err & FDD_NOTREADY) {
		return FALSE;
	}

	// CMSK=1Ȃ烊UgtF[Y(i82078f[^V[gɂ)
	if (fdc.err & FDD_CM) {
		if (fdc.sk) {
			return FALSE;
		}
	}

	// ԂvZ(wbh[h͍lȂ)
	hus = fdd->GetSearch();

	// No Data͂̎ԌAUgtF[Y
	if (fdc.err & FDD_NODATA) {
		EventErr(hus);
		return TRUE;
	}

	// ItZbgACxgX^[gAERtF[YJn
	fdc.offset = 0;
	EventRW();
	fdc.phase = read;

	// Ԃwbh[hԂZ΁AP҂
	if (!fdc.load) {
		if (hus < fdc.hlt) {
			hus += fdd->GetRotationTime();
		}
	}

	// ԂZ
	if (!fdc.fast) {
		hus += event.GetTime();
		event.SetTime(hus);
	}
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	WRITE (DELETED) DATAR}h
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::WriteData()
{
	int len;
	DWORD hus;
	BOOL deleted;

	ASSERT(this);
	ASSERT((fdc.cmd == write_data) || (fdc.cmd == write_del_data));

	// SRݒ
	fdc.sr |= sr_cb;
	fdc.sr &= ~sr_dio;
	fdc.sr &= ~sr_d3b;
	fdc.sr &= ~sr_d2b;
	fdc.sr &= ~sr_d1b;
	fdc.sr &= ~sr_d0b;

	// hCuɔCBNOTREADY,NOTWRITE,NODATA,MAM,CYLn,IDCRC,DDAM
	deleted = FALSE;
	if (fdc.cmd == write_del_data) {
		deleted = TRUE;
	}
#if defined(FDC_LOG)
	LOG4(Log::Normal, "(C:%02X H:%02X R:%02X N:%02X)",
		fdc.chrn[0], fdc.chrn[1], fdc.chrn[2], fdc.chrn[3]);
#endif
	fdc.err = fdd->WriteSector(NULL, &fdc.len,
									fdc.mfm, fdc.chrn, fdc.hd, deleted);
	fdc.err &= ~FDD_DDAM;

	// IDCRCȂADATAERR悹
	if (fdc.err & FDD_IDCRC) {
		fdc.err &= ~FDD_IDCRC;
		fdc.err |= FDD_DATAERR;
	}

	// N=0łNƂDTLg
	if (fdc.chrn[3] == 0) {
		len = 1 << (fdc.dtl + 7);
		if (len < fdc.len) {
			fdc.len = len;
		}
	}
	else {
		len = (1 << (fdc.chrn[3] + 7));
		if (len < fdc.len) {
			fdc.len = len;
		}
	}

	// Not Ready, Not Writable̓UgtF[Y
	if ((fdc.err & FDD_NOTREADY) || (fdc.err & FDD_NOTWRITE)) {
		return FALSE;
	}

	// ԂvZ(wbh[h͍lȂ)
	hus = fdd->GetSearch();

	// No Data͎sナUg
	if (fdc.err & FDD_NODATA) {
		EventErr(hus);
		return TRUE;
	}

	// ItZbgACxgݒAEWtF[YJn
	fdc.offset = 0;
	EventRW();
	fdc.phase = write;

	// Ԃwbh[hԂZ΁AP҂
	if (!fdc.load) {
		if (hus < fdc.hlt) {
			hus += fdd->GetRotationTime();
		}
	}

	// ԂZ
	if (!fdc.fast) {
		hus += event.GetTime();
		event.SetTime(hus);
	}
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	SCANnR}h
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::Scan()
{
	int len;
	DWORD hus;

	ASSERT(this);

	// SRݒ
	fdc.sr |= sr_cb;
	fdc.sr &= ~sr_dio;
	fdc.sr &= ~sr_d3b;
	fdc.sr &= ~sr_d2b;
	fdc.sr &= ~sr_d1b;
	fdc.sr &= ~sr_d0b;

	// hCuɔCBNOTREADY,NODATA,MAM,CYLn,CRCn,DDAM
#if defined(FDC_LOG)
	LOG4(Log::Normal, "(C:%02X H:%02X R:%02X N:%02X)",
		fdc.chrn[0], fdc.chrn[1], fdc.chrn[2], fdc.chrn[3]);
#endif
	fdc.err = fdd->ReadSector(fdc.buffer, &fdc.len,
									fdc.mfm, fdc.chrn, fdc.hd);

	// DDAM(Deleted Sector)̗LŁACM(Control Mark)߂
	if (fdc.err & FDD_DDAM) {
		fdc.err &= ~FDD_DDAM;
		fdc.err |= FDD_CM;
	}

	// IDCRC܂DATACRCȂADATAERR悹
	if ((fdc.err & FDD_IDCRC) || (fdc.err & FDD_DATACRC)) {
		fdc.err &= ~FDD_IDCRC;
		fdc.err |= FDD_DATAERR;
	}

	// N=0łNƂDTLg
	if (fdc.chrn[3] == 0) {
		len = 1 << (fdc.dtl + 7);
		if (len < fdc.len) {
			fdc.len = len;
		}
	}
	else {
		len = (1 << (fdc.chrn[3] + 7));
		if (len < fdc.len) {
			fdc.len = len;
		}
	}

	// Not Ready̓UgtF[Y
	if (fdc.err & FDD_NOTREADY) {
		return FALSE;
	}

	// CMSK=1Ȃ烊UgtF[Y(i82078f[^V[gɂ)
	if (fdc.err & FDD_CM) {
		if (fdc.sk) {
			return FALSE;
		}
	}

	// ԂvZ(wbh[h͍lȂ)
	hus = fdd->GetSearch();

	// No Data͂̎ԌAUgtF[Y
	if (fdc.err & FDD_NODATA) {
		EventErr(hus);
		return TRUE;
	}

	// ItZbgACxgX^[gAERtF[YJn
	fdc.offset = 0;
	EventRW();
	fdc.phase = write;

	// Ԃwbh[hԂZ΁AP҂
	if (!fdc.load) {
		if (hus < fdc.hlt) {
			hus += fdd->GetRotationTime();
		}
	}

	// ԂZ
	if (!fdc.fast) {
		hus += event.GetTime();
		event.SetTime(hus);
	}
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	READ DIAGNOSTICR}h
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::ReadDiag()
{
	DWORD hus;

	ASSERT(this);
	ASSERT(fdc.cmd == read_diag);

	// SRݒ
	fdc.sr |= sr_cb;
	fdc.sr |= sr_dio;
	fdc.sr &= ~sr_d3b;
	fdc.sr &= ~sr_d2b;
	fdc.sr &= ~sr_d1b;
	fdc.sr &= ~sr_d0b;

	// EOT=0̓UgtF[Y(NO DATA)
	if (fdc.eot == 0) {
		if (fdd->IsReady(fdc.dsr)) {
			fdc.err = FDD_NODATA;
		}
		else {
			fdc.err = FDD_NOTREADY;
		}
		return FALSE;
	}

	// hCuɔCBNOTREADY,NODATA,MAM,CRCn,DDAM
	fdc.err = fdd->ReadDiag(fdc.buffer, &fdc.len, fdc.mfm,
								fdc.chrn, fdc.hd);
	// Not Ready̓UgtF[Y
	if (fdc.err & FDD_NOTREADY) {
		return FALSE;
	}

	// ԂvZ(wbh[h͍lȂ)
	hus = fdd->GetSearch();

	// MAMȂ玞ԑ҂AUgtF[YցBNODATAł邽
	if (fdc.err & FDD_MAM) {
		EventErr(hus);
		return TRUE;
	}

	ASSERT(fdc.len > 0);

	// ItZbgACxgX^[gAERtF[YJn
	fdc.offset = 0;
	EventRW();
	fdc.phase = read;

	// ԂZ
	if (!fdc.fast) {
		hus += event.GetTime();
		event.SetTime(hus);
	}
	return TRUE;
}

//---------------------------------------------------------------------------
//
//	WRITE IDR}h
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::WriteID()
{
	DWORD hus;

	ASSERT(this);
	ASSERT(fdc.cmd == write_id);

	// SRݒ
	fdc.sr |= sr_cb;
	fdc.sr &= ~sr_dio;
	fdc.sr &= ~sr_d3b;
	fdc.sr &= ~sr_d2b;
	fdc.sr &= ~sr_d1b;
	fdc.sr &= ~sr_d0b;

	// SC=0`FbN
	if (fdc.sc == 0) {
		fdc.err = 0;
		return FALSE;
	}

	// hCuɔCBNOTREADY,NOTWRITE
	fdc.err = fdd->WriteID(NULL, fdc.d, fdc.sc, fdc.mfm, fdc.hd, fdc.gpl);
	// Not Ready, Not Writable̓UgtF[Y
	if ((fdc.err & FDD_NOTREADY) || (fdc.err & FDD_NOTWRITE)) {
		return FALSE;
	}

	// ItZbg
	fdc.offset = 0;
	fdc.len = fdc.sc * 4;

	// Cxgݒ
	if (fdc.ndm) {
		fdc.sr |= sr_ndm;
		LOG0(Log::Warning, "Non-DMA[hWrite ID");
	}
	else {
		fdc.sr &= ~sr_ndm;
	}
	// N7܂łɐ(N=716KB/sector, AtH[}bg)
	if (fdc.chrn[3] > 7) {
		fdc.chrn[3] = 7;
	}

	// ԂݒBPZN^Ŋ
	hus = fdd->GetSearch();
	hus += (fdd->GetRotationTime() / fdc.sc);
	if (fdc.fast) {
		hus = 32 * 4;
	}

	// CxgX^[gARQM𗎂Ƃ
	event.SetTime(hus);
	fdc.sr &= ~sr_rqm;
	fdc.phase = write;

	return TRUE;
}

//---------------------------------------------------------------------------
//
//	Cxg(Read/Writen)
//
//---------------------------------------------------------------------------
void FASTCALL FDC::EventRW()
{
	DWORD hus;

	// SRݒ(Non-DMA)
	if (fdc.ndm) {
		fdc.sr |= sr_ndm;
	}
	else {
		fdc.sr &= ~sr_ndm;
	}

	// Cxg쐬
	if (fdc.ndm) {
		// Non-DMAB16us/32us
		if (fdc.mfm) {
			hus = 32;
		}
		else {
			hus = 64;
		}
	}
	else {
		// DMA64oCg܂Ƃ߂čsB1024us/2048us
		if (fdc.mfm) {
			hus = 32 * 64;
		}
		else {
			hus = 64 * 64;
		}
	}

	// DD͂̔{
	if (!fdd->IsHD()) {
		hus <<= 1;
	}

	// fast64usŒ(DMAɌ)
	if (fdc.fast) {
		if (!fdc.ndm) {
			hus = 128;
		}
	}

	// CxgX^[gARQM𗎂Ƃ
	event.SetTime(hus);
	fdc.sr &= ~sr_rqm;
}

//---------------------------------------------------------------------------
//
//	Cxg(G[)
//
//---------------------------------------------------------------------------
void FASTCALL FDC::EventErr(DWORD hus)
{
	// SRݒ(Non-DMA)
	if (fdc.ndm) {
		fdc.sr |= sr_ndm;
	}
	else {
		fdc.sr &= ~sr_ndm;
	}

	// CxgX^[gARQM𗎂Ƃ
	event.SetTime(hus);
	fdc.sr &= ~sr_rqm;
	fdc.phase = execute;
}

//---------------------------------------------------------------------------
//
//	݊
//
//---------------------------------------------------------------------------
void FASTCALL FDC::WriteBack()
{
	switch (fdc.cmd) {
		// Write Data
		case write_data:
			fdc.err = fdd->WriteSector(fdc.buffer, &fdc.len,
							fdc.mfm, fdc.chrn, fdc.hd, FALSE);
			return;

		// Write Deleted Data
		case write_del_data:
			fdc.err = fdd->WriteSector(fdc.buffer, &fdc.len,
							fdc.mfm, fdc.chrn, fdc.hd, TRUE);
			return;

		// Write ID
		case write_id:
			fdc.err = fdd->WriteID(fdc.buffer, fdc.d, fdc.sc,
							fdc.mfm, fdc.hd, fdc.gpl);
			return;
	}

	// 肦Ȃ
	ASSERT(FALSE);
}

//---------------------------------------------------------------------------
//
//	ZN^
//
//---------------------------------------------------------------------------
BOOL FASTCALL FDC::NextSector()
{
	// TC`FbN
	if (fdc.tc) {
		// C,H,R,Nړ
		if (fdc.chrn[2] < fdc.eot) {
			fdc.chrn[2]++;
			return FALSE;
		}
		fdc.chrn[2] = 0x01;
		// MTɂĕ
		if (fdc.mt && (!(fdc.chrn[1] & 0x01))) {
			// TCh1
			fdc.chrn[1] |= 0x01;
			fdc.hd |= 0x04;
			return FALSE;
		}
		// C+1, R=1ŏI
		fdc.chrn[0]++;
		return FALSE;
	}

	// EOT`FbN
	if (fdc.chrn[2] < fdc.eot) {
		fdc.chrn[2]++;
		return TRUE;
	}

	// EOTBR=1
	fdc.err |= FDD_EOT;
	fdc.chrn[2] = 0x01;

	// MTɂĕ
	if (fdc.mt && (!(fdc.chrn[1] & 0x01))) {
		// TCh1
		fdc.chrn[1] |= 0x01;
		fdc.hd |= 0x04;
		return TRUE;
	}

	// C+1, R=1ŏI
	fdc.chrn[0]++;
	return FALSE;
}
